/*:
 * @plugindesc スキル学習プラグイン
 * @author スタジオVR
 *
 * @command LearnRandomSkill
 * @text ランダムスキル習得
 * @desc 指定したアクターが、別のアクターからランダムにスキルを学習します。
 *
 * @arg targetActorId
 * @type actor
 * @text 習得対象アクター
 * @desc スキルを習得するアクター
 *
 * @arg sourceActorId
 * @type variable
 * @text 学習相手アクター（変数）
 * @desc スキルを教えるアクターのIDが格納された変数
 *
 * @arg resultVariableId
 * @type variable
 * @text 結果格納変数
 * @desc 習得したスキル名を格納する変数のID（習得失敗時は0）
 *
 * @help
 * このプラグインは、あるアクターが別のアクターからランダムにスキルを学習する機能を提供します。
 *
 * プラグインコマンド:
 * ランダムスキル習得 - 指定したアクターが、別のアクターからランダムに1つのスキルを学習します。
 * 学習相手のアクターIDは変数で指定します。
 * 習得したスキル名（または0）が指定した変数に格納されます。
 */

(() => {
    const pluginName = "Svr_SkillLearningActor";

    const isAbilitySkill = (skill) => {
        return skill && skill.meta && skill.meta.AbilitySkill;
    };

    PluginManager.registerCommand(pluginName, "LearnRandomSkill", args => {
        const targetActorId = Number(args.targetActorId);
        const sourceActorId = $gameVariables.value(Number(args.sourceActorId));
        const resultVariableId = Number(args.resultVariableId);

        const targetActor = $gameActors.actor(targetActorId);
        const sourceActor = $gameActors.actor(sourceActorId);

        if (!targetActor || !sourceActor) {
            console.error("Invalid actor ID");
            $gameVariables.setValue(resultVariableId, 0);
            return;
        }

        // ソースアクターの全てのスキルを取得
        const allSourceSkills = sourceActor.skills().concat(
            sourceActor.learnedAbilitySkillIds().map(id => $dataSkills[id])
        );

        const learnableSkills = allSourceSkills.filter(skill => {
            if (isAbilitySkill(skill)) {
                return !targetActor.learnedAbilitySkillIds().includes(skill.id);
            } else {
                return !targetActor.isLearnedSkill(skill.id);
            }
        });

        if (learnableSkills.length === 0) {
//            console.log(`Actor ${targetActorId} has no new skills to learn from Actor ${sourceActorId}`);
            $gameVariables.setValue(resultVariableId, 0);
            return;
        }

        const randomSkill = learnableSkills[Math.floor(Math.random() * learnableSkills.length)];
        
        if (isAbilitySkill(randomSkill)) {
            if (typeof targetActor.addAbilitySkill === 'function') {
                targetActor.addAbilitySkill(randomSkill.id);
            } else {
                console.warn(`addAbilitySkill method not found. Learning as a regular skill.`);
                targetActor.learnSkill(randomSkill.id);
            }
        } else {
            targetActor.learnSkill(randomSkill.id);
        }

//        console.log(`Actor ${targetActorId} learned skill: ${randomSkill.name} from Actor ${sourceActorId}`);
        $gameVariables.setValue(resultVariableId, randomSkill.name);
    });

})();