/*:
 * @plugindesc 敵の敵視ステート数を数え、最も敵視されているパーティメンバーの職業を判定し、スイッチをONにするプラグイン
 * @target MZ
 * @author スタジオVR
 *
 * @param startState
 * @text 敵視ステートの開始ID
 * @desc 敵視ステートの範囲の開始値を指定します。
 * @type number
 * @default 101
 *
 * @param endState
 * @text 敵視ステートの終了ID
 * @desc 敵視ステートの範囲の終了値を指定します。
 * @type number
 * @default 108
 *
 * @param jobIds
 * @text 職業IDリスト
 * @desc 監視する職業IDをカンマ区切りで指定します（例: 1,3,5）。
 * @type string
 * @default 0
 *
 * @param switchId
 * @text スイッチID
 * @desc 指定した職業が敵視されていた場合、ONにするスイッチのIDを指定します。
 * @type switch
 * @default 1
 *
 * @param variableId
 * @text 変数ID
 * @desc 指定した職業が敵視されていた場合、敵視している数を格納する変数のIDを指定します。
 * @type variable
 * @default 1
 *
 * @help
 * このプラグインは、ターン開始時にパーティメンバーのステート数を数え、
 * 最も多いステートIDを持つパーティメンバーの職業が指定した範囲に該当する場合、
 * スイッチをONにするプラグインです。
 */

(() => {
  const pluginName = 'Svr_TargetClass';
  const parameters = PluginManager.parameters(pluginName);

  const startState = Number(parameters['startState']);
  const endState = Number(parameters['endState']);
  const jobIds = parameters['jobIds'].split(',').map(Number);
  const switchId = Number(parameters['switchId']);
  const variableId = Number(parameters['variableId']);

  function countStates() {
    const stateCounts = $gameTroop.members().map(enemy => {
      return enemy.states().filter(state => {
        return state && state.id >= startState && state.id <= endState;
      }).length;
    });

    const maxCount = Math.max(...stateCounts);

    if (maxCount <= 0) {
      return;
    }

    let stateCountsList = Array(endState - startState + 1).fill(0);
    for (const enemy of $gameTroop.members()) {
      const states = enemy.states().filter(state => {
        return state && state.id >= startState && state.id <= endState;
      });
      for (const state of states) {
        stateCountsList[state.id - startState] += 1;
      }
    }
    const maxCountList = Math.max(...stateCountsList);
    const maxStateId = stateCountsList.indexOf(maxCountList) + startState;

    let partyId = maxStateId - startState;

    if (partyId < 0) {
      return;
    }

    let job = $gameParty.members()[partyId].currentClass().id;

    if (jobIds.includes(job)) {
      $gameSwitches.setValue(switchId, true);
      $gameVariables.setValue(variableId, maxCountList);
    } else {
      $gameSwitches.setValue(switchId, false);
      $gameVariables.setValue(variableId, 0);
    }
  }

  const _BattleManager_endAction = BattleManager.endAction;
  BattleManager.endAction = function() {
    _BattleManager_endAction.call(this);
    countStates();
  };
})();
