/*:
 * @plugindesc Svr_VariableArithmeticDebug with Enhanced Conditional Display, Nested Conditions, Item Names and Descriptions
 * @author スタジオVR
 * @help
 * このプラグインは、メッセージウィンドウ内で変数の値を表示し、
 * 変数間または変数と固定値間で四則演算を行うことができます。
 * また、条件に基づいてアイテム名や文字列の表示を切り替えることができます。
 * スイッチ1003が有効な時、デバッグログを出力します。
 * 
 * 使用方法：
 * \v[X]         - 変数Xの値を表示します
 * \vc[expression] - 複雑な数式を評価して表示します
 * \cc[condition]{trueResult}{falseResult} - 条件に基づいて結果を表示します
 * \cn[condition]{trueResult}{falseResult} - ネストされた条件分岐を処理します
 * \ncn[condition]{trueResult}{falseResult} - 完全なネスト対応の条件分岐を処理します
 * \item[X] - アイテムIDがXのアイテム名を表示します
 * \item_desc[X] - アイテムIDがXのアイテム説明を表示します
 * 
 * 数式内で使用可能な関数：
 * ceil(X)  - Xの値を切り上げます (例: ceil(5.3) -> 6)
 * floor(X) - Xの値を切り下げます (例: floor(5.7) -> 5)
 * round(X) - Xの値を四捨五入します (例: round(5.5) -> 6)
 * 
 * 数式の使用例：
 * \vc[ceil(v1/2)]    - 変数1の値を2で割った後、切り上げ
 * \vc[floor(v1/3)]   - 変数1の値を3で割った後、切り下げ
 * \vc[round(v1/4)]   - 変数1の値を4で割った後、四捨五入
 * 
 * 条件式の使用例：
 * 基本的な使い方:
 * \cc[s1]{条件1がTRUEの場合}{条件1がFALSEの場合}
 * 
 * 通常のネスト:
 * \cn[s1]{\cn[s2]{条件1と2がTRUE}{条件1はTRUE,条件2はFALSE}}{条件1がFALSE}
 * 
 * 完全なネスト:
 * \ncn[s1]{\ncn[s2]{\ncn[s3]{全部TRUE}{s3がFALSE}}{s2がFALSE}}{s1がFALSE}
 * 
 * プレフィックス：
 * v - 変数
 * s - スイッチ
 * i - アイテムID
 * 
 * 条件の例：
 * s1 - スイッチ1がON
 * !s2 - スイッチ2がOFF
 * v1>100 - 変数1が100より大きい
 * v2<=50 - 変数2が50以下
 * v3==10 - 変数3が10と等しい
 * v4!=0 - 変数4が0でない
 * v1>v2 - 変数1が変数2より大きい
 * v3<=v4 - 変数3が変数4以下
 * 
 * 複合条件：
 * s1&&v1>100 - スイッチ1がONかつ変数1が100より大きい
 * s2||v2<50 - スイッチ2がONまたは変数2が50未満
 * v1>v2&&v3<v4 - 変数1が変数2より大きく、かつ変数3が変数4より小さい
 */

(function() {
    // ColorManager._colors を定義
    if (!ColorManager._colors) {
        ColorManager._colors = [
            [0, 0, 0],         // 0: Black
            [85, 85, 85],      // 1: Dark Gray
            [170, 170, 170],   // 2: Gray
            [255, 255, 255],   // 3: White
            [255, 0, 0],       // 4: Red
            [255, 128, 0],     // 5: Orange
            [255, 255, 0],     // 6: Yellow
            [0, 255, 0],       // 7: Green
            [0, 0, 255],       // 8: Blue
            [128, 0, 255],     // 9: Purple
            [128, 64, 0],      // 10: Brown
            [255, 0, 255],     // 11: Pink
            [0, 128, 0],       // 12: Dark Green
            [0, 0, 128],       // 13: Dark Blue
            [128, 0, 128],     // 14: Dark Purple
            [255, 165, 0],     // 15: Amber
        ];
    }

    // オリジナルの convertEscapeCharacters 関数を保存
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;

    // convertEscapeCharacters 関数を上書き
    Window_Base.prototype.convertEscapeCharacters = function(text) {
        text = text.replace(/\\/g, '\x1b');
        
        // 単純な変数表示 \v[X]
        text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
            const variableId = parseInt(arguments[1]);
            const value = $gameVariables.value(variableId);
            if ($gameSwitches.value(1003)) {
                console.log(`Debug: Variable ${variableId} = ${value}`);
            }
            return value;
        });

        // 複雑な数式の評価 \vc[expression]
        text = text.replace(/\x1bVC\[(.+?)\]/gi, (match, expression) => {
            try {
                const replacedExpression = expression.replace(/v(\d+)/gi, (m, p1) => {
                    return $gameVariables.value(parseInt(p1));
                });
                
                // 数学関数を使用可能な形式に変換
                const mathExpression = replacedExpression.replace(
                    /(ceil|floor|round)\((.*?)\)/g,
                    (match, func, args) => `Math.${func}(${args})`
                );
                
                const result = this.safeEval(mathExpression);
                
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Evaluating expression "${expression}" => ${result}`);
                }
                return result;
            } catch (e) {
                if ($gameSwitches.value(1003)) {
                    console.error(`Error evaluating expression "${expression}": ${e}`);
                }
                return '0';
            }
        });

        // 完全なネスト対応の新コマンド \ncn[condition]{trueResult}{falseResult}
        text = text.replace(/\x1bNCN\[([^\]]+)\]\{((?:[^{}]|\{(?:[^{}]|\{[^{}]*\})*\})*)\}\{((?:[^{}]|\{(?:[^{}]|\{[^{}]*\})*\})*)\}/gi, 
            (match, condition, trueResult, falseResult) => {
                const conditionResult = this.evaluateCondition(condition);
                let result = conditionResult ? trueResult : falseResult;
                
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Deep nested condition "${condition}" is ${conditionResult ? 'TRUE' : 'FALSE'}, processing: ${result}`);
                }
                
                // 再帰的にネストされた条件を処理
                return this.processDeepNestedConditions(result);
            });

        // ネストされた条件分岐の処理 \cn[condition]{trueResult}{falseResult}
        text = text.replace(/\x1bCN\[([^\]]+)\]\{([^}]*)\}\{([^}]*)\}/gi, (match, condition, trueResult, falseResult) => {
            const conditionResult = this.evaluateCondition(condition);
            let result = conditionResult ? trueResult : falseResult;
            
            if ($gameSwitches.value(1003)) {
                console.log(`Debug: Nested condition "${condition}" is ${conditionResult ? 'TRUE' : 'FALSE'}, processing: ${result}`);
            }
            
            // ネストされた条件分岐を再帰的に処理
            return this.processNestedConditions(result);
        });
        
        // 通常の条件付き表示 \cc[condition]{trueResult}{falseResult}
        text = text.replace(/\x1bCC\[([^\]]+)\]\{([^}]*)\}\{([^}]*)\}/gi, (match, condition, trueResult, falseResult) => {
            const conditionResult = this.evaluateCondition(condition);
            const result = conditionResult ? trueResult : falseResult;
            
            if ($gameSwitches.value(1003)) {
                console.log(`Debug: Condition "${condition}" is ${conditionResult ? 'TRUE' : 'FALSE'}, displaying: ${result}`);
            }
            
            // 基本的なエスケープシーケンスを処理
            return this.convertEscapeCharacters(result);
        });

        // アイテム名表示 \item[X]
        text = text.replace(/\x1bITEM\[(\d+)\]/gi, (match, itemId) => {
            const item = $dataItems[parseInt(itemId)];
            if (item) {
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Displaying item for ID ${itemId}: ${item.name} with iconIndex ${item.iconIndex}`);
                }
                if (item.iconIndex > 0) {
                    return `\x1bI[${item.iconIndex}]${item.name}`;
                } else {
                    return item.name;
                }
            } else {
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Item with ID ${itemId} not found`);
                }
                return '';
            }
        });

        // アイテム説明表示 \item_desc[X]
        text = text.replace(/\x1bITEM_DESC\[(\d+)\]/gi, (match, itemId) => {
            const item = $dataItems[parseInt(itemId)];
            if (item && item.description) {
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Displaying description for item ID ${itemId}: ${item.description}`);
                }
                return this.convertEscapeCharacters(item.description);
            } else {
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Description for item ID ${itemId} not found`);
                }
                return '';
            }
        });

        return _Window_Base_convertEscapeCharacters.call(this, text);
    };

    // ネストされた条件分岐を処理する関数
    Window_Base.prototype.processNestedConditions = function(text) {
        // 元のテキストを保存
        let processedText = text;
        
        // ネストされた条件分岐が存在する限り処理を続ける
        let hasNestedConditions = true;
        while (hasNestedConditions) {
            const beforeProcess = processedText;
            
            // 通常の条件分岐とネストされた条件分岐の両方を処理
            processedText = this.convertEscapeCharacters(processedText);
            
            // 処理前後でテキストが変化しなければ処理を終了
            if (beforeProcess === processedText) {
                hasNestedConditions = false;
            }
        }
        
        return processedText;
    };

    // 深いネストに対応した条件処理関数
    Window_Base.prototype.processDeepNestedConditions = function(text) {
        let processedText = text;
        
        // NCN, CN, CC のパターンを定義
        const patterns = [
            {
                regex: /\x1bNCN\[([^\]]+)\]\{((?:[^{}]|\{(?:[^{}]|\{[^{}]*\})*\})*)\}\{((?:[^{}]|\{(?:[^{}]|\{[^{}]*\})*\})*)\}/gi,
                type: 'NCN'
            },
            {
                regex: /\x1bCN\[([^\]]+)\]\{([^}]*)\}\{([^}]*)\}/gi,
                type: 'CN'
            },
            {
                regex: /\x1bCC\[([^\]]+)\]\{([^}]*)\}\{([^}]*)\}/gi,
                type: 'CC'
            }
        ];

        let hasChanges;
        do {
            hasChanges = false;
            let previousText = processedText;

            // 各パターンで処理
            for (const pattern of patterns) {
                processedText = processedText.replace(pattern.regex, (match, condition, trueResult, falseResult) => {
                    const conditionResult = this.evaluateCondition(condition);
                    let result = conditionResult ? trueResult : falseResult;

                    if ($gameSwitches.value(1003)) {
                        console.log(`Debug: Processing ${pattern.type} condition "${condition}" is ${conditionResult ? 'TRUE' : 'FALSE'}, result: ${result}`);
                    }

                    // NCNの場合は再帰的に処理
                    if (pattern.type === 'NCN') {
                        result = this.processDeepNestedConditions(result);
                    }

                    return this.convertEscapeCharacters(result);
                });
            }

            // 変更があったかチェック
            if (previousText !== processedText) {
                hasChanges = true;
            }

        } while (hasChanges);

        return processedText;
    };

// 条件を評価する関数（改良版）
Window_Base.prototype.evaluateCondition = function(condition) {
    try {
        // ORで分割して評価
        return condition.split('||').some(orPart => {
            // ANDで分割して評価
            return orPart.split('&&').every(andPart => {
                const trimmedPart = andPart.trim();
                const result = this.evaluateSingleCondition(trimmedPart);
                
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Evaluating condition part "${trimmedPart}" = ${result}`);
                }
                
                return result;
            });
        });
    } catch (e) {
        if ($gameSwitches.value(1003)) {
            console.error('Error in condition evaluation:', e);
            console.error('Condition:', condition);
        }
        return false;
    }
};

    // 単一の条件を評価する関数（変数同士の比較に対応）
    Window_Base.prototype.evaluateSingleCondition = function(condition) {
        // スイッチの否定
        if (condition.startsWith('!s')) {
            return !$gameSwitches.value(parseInt(condition.slice(2)));
        } 
        // スイッチの評価
        else if (condition.startsWith('s')) {
            return $gameSwitches.value(parseInt(condition.slice(1)));
        } 
        // 変数の評価
        else if (condition.startsWith('v')) {
            // 変数と変数の比較
            const varComparePattern = /v(\d+)(==|!=|>|<|>=|<=)v(\d+)/;
            const varParts = condition.match(varComparePattern);
            
            if (varParts) {
                const [, var1Id, operator, var2Id] = varParts;
                const value1 = $gameVariables.value(parseInt(var1Id));
                const value2 = $gameVariables.value(parseInt(var2Id));
                
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Comparing variables v${var1Id}(${value1}) ${operator} v${var2Id}(${value2})`);
                }
                
                switch (operator) {
                    case '==': return value1 == value2;
                    case '!=': return value1 != value2;
                    case '>': return value1 > value2;
                    case '<': return value1 < value2;
                    case '>=': return value1 >= value2;
                    case '<=': return value1 <= value2;
                    default: return false;
                }
            }
            
            // 変数と固定値の比較
            const fixedPattern = /v(\d+)(==|!=|>|<|>=|<=)(-?\d+)/;
            const fixedParts = condition.match(fixedPattern);
            
            if (fixedParts) {
                const [, variableId, operator, value] = fixedParts;
                const variableValue = $gameVariables.value(parseInt(variableId));
                const compareValue = parseInt(value);
                
                if ($gameSwitches.value(1003)) {
                    console.log(`Debug: Comparing variable v${variableId}(${variableValue}) ${operator} ${compareValue}`);
                }
                
                switch (operator) {
                    case '==': return variableValue == compareValue;
                    case '!=': return variableValue != compareValue;
                    case '>': return variableValue > compareValue;
                    case '<': return variableValue < compareValue;
                    case '>=': return variableValue >= compareValue;
                    case '<=': return variableValue <= compareValue;
                    default: return false;
                }
            }
        }
        return false;
    };

    // 安全な数式評価関数を更新
    Window_Base.prototype.safeEval = function(expression) {
        // 許可する文字のパターンを更新（Math関数を含む）
        if (/[^0-9+\-*/().\sMath\.,ceil,floor,round]/.test(expression)) {
            throw new Error('Invalid characters in expression');
        }
        
        // eslint-disable-next-line no-eval
        return eval(expression);
    };
})();