//=============================================================================
// TMPlugin - エネミー行動予測
// バージョン: 1.1.0
// 最終更新日: 2018/08/21
// 配布元    : http://hikimoki.sakura.ne.jp/
//-----------------------------------------------------------------------------
// Copyright (c) 2016 tomoaky
// Released under the MIT license.
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 敵キャラの次の行動のヒントなどをテキストで表示します。
 * より戦略的なターンバトルが実現できるかもしれません。
 *
 * @author tomoaky (http://hikimoki.sakura.ne.jp/)
 *
 * @param width
 * @type number
 * @desc 行動予測表示の横幅
 * 初期値: 240
 * @default 240
 *
 * @param maxLines
 * @type number
 * @desc 行動予測表示の最大行数
 * 初期値: 3
 * @default 3
 *
 * @param lineHeight
 * @type number
 * @desc 行動予測表示の 1 行の高さ
 * 初期値: 36
 * @default 36
 *
 * @param fontSize
 * @type number
 * @desc 行動予測表示のフォントサイズ
 * 初期値: 28
 * @default 28
 *
 * @param color
 * @desc 行動予測表示の文字色
 * 初期値: white
 * @default white
 *
 * @param backColor
 * @desc 行動予測表示の背景の色
 * 初期値: black
 * @default black
 *
 * @param backOpacity
 * @type number
 * @max 255
 * @desc 行動予測表示の背景の不透明度
 * 初期値: 128 ( 0 ～ 255 )
 * @default 128
 *
 * @param textAlign
 * @type select
 * @option left
 * @option center
 * @option right
 * @desc 行動予測表示の行揃え
 * @default center
 * 
 * @param showIcon
 * @type boolean
 * @desc スキル名の頭にアイコンも表示する
 * 初期値: ON（ ON = 表示 / OFF = 非表示 )
 * @default true
 * 
 * @param headerText
 * @desc 行動予測表示のヘッダーテキスト
 * 初期値: Next
 * @default Next
 *
 * @param headerHeight
 * @type number
 * @desc 行動予測表示のヘッダーの高さ
 * 初期値: 28
@default 28
@help
RPGツクールMZ用に修正されたスクリプトです。このプラグインを使うことで、敵キャラの
次の行動の予測をテキストで表示することができます。プラグインパラメータを設定すれば、
行動予測表示の横幅やフォントサイズ、背景色などをカスタマイズできます。
使い方:
プラグインをプラグイン管理に追加して、プラグインパラメータを設定します。
スクリプトは他のプラグインと競合しないように設定されていますが、
必要に応じて競合を回避するためにプラグインの順序を変更してください。
注意:
このプラグインはRPGツクールMZ専用です。RPGツクールMVでは使用できません。
更新履歴:
2021/09/xx - 初版作成
2021/10/xx - ヘッダーテキストの追加
*/
(function() {

var parameters = PluginManager.parameters('EnemyActionPrediction');
var width = parseInt(parameters['width'] || 240);
var maxLines = parseInt(parameters['maxLines'] || 3);
var lineHeight = parseInt(parameters['lineHeight'] || 36);
var fontSize = parseInt(parameters['fontSize'] || 28);
var color = parameters['color'] || 'white';
var backColor = parameters['backColor'] || 'black';
var backOpacity = parseInt(parameters['backOpacity'] || 128);
var textAlign = parameters['textAlign'] || 'center';
//var showIcon = parameters['showIcon'].toLowerCase() === 'true';
var showIcon = false;
var headerText = parameters['headerText'] || 'Next';
var headerHeight = parseInt(parameters['headerHeight'] || 28);

//-----------------------------------------------------------------------------
// Window_EnemyActionPrediction
//
// The window for displaying predicted enemy actions.

function Window_EnemyActionPrediction() {
    this.initialize.apply(this, arguments);
}

Window_EnemyActionPrediction.prototype = Object.create(Window_Base.prototype);
Window_EnemyActionPrediction.prototype.constructor = Window_EnemyActionPrediction;

Window_EnemyActionPrediction.prototype.initialize = function() {
    const rect = new Rectangle(0, 0, 0, 0); // 適切なx, y, width, heightの値を設定してください
    Window_Base.prototype.initialize.call(this, rect);
};

Window_EnemyActionPrediction.prototype.setEnemy = function(enemy) {
if (this._enemy !== enemy) {
this._enemy = enemy;
this.refresh();
}
};

Window_EnemyActionPrediction.prototype.refresh = function() {
this.contents.clear();
this.resetTextColor();
this.drawBackground(0, 0, width, headerHeight);
this.drawText(headerText, 0, 0, width, textAlign);
if (this._enemy) {
var actions = this._enemy.nextActions();
for (var i = 0; i < actions.length; i++) {
    var action = actions[i];
    var y = (i * lineHeight) + headerHeight;
    if (showIcon) {
      var iconIndex = action.skill().iconIndex;
      this.drawIcon(iconIndex, 4, y + 2);
    }
    this.resetTextColor();
    this.drawItemName(action.skill(), showIcon ? 40 : 0, y);
  }
}
};

Window_EnemyActionPrediction.prototype.drawBackground = function(x, y, width, height) {
//var color1 = this.textColor(backColor);
//var color2 = this.textColor(backColor);
//this.contents.gradientFillRect(x, y, width, height, color1, color2);
this.contents.gradientFillRect(x, y, width, height, 0, 0);
this.contents.paintOpacity = backOpacity;
this.contents.fillRect(x, y, width, height, this.gaugeBackColor());
this.contents.paintOpacity = 255;
};

//-----------------------------------------------------------------------------
// Window_BattleEnemy
//
// The window for selecting a target enemy on the battle screen.

var _Window_BattleEnemy_initialize = Window_BattleEnemy.prototype.initialize;
Window_BattleEnemy.prototype.initialize = function(x, y) {
_Window_BattleEnemy_initialize.call(this, x, y);
this._predictionWindow = null;
};

Window_BattleEnemy.prototype.setPredictionWindow = function(predictionWindow) {
this._predictionWindow = predictionWindow;
this.updatePrediction();
};

var _Window_BattleEnemy_select = Window_BattleEnemy.prototype.select;
Window_BattleEnemy.prototype.select = function(index) {
_Window_BattleEnemy_select.call(this, index);
this.updatePrediction();
};

Window_BattleEnemy.prototype.updatePrediction = function() {
if (this._predictionWindow) {
var enemy = this.enemy();
this._predictionWindow.setEnemy(enemy);
}
};

//-----------------------------------------------------------------------------
// Scene_Battle
//
// The scene class of the battle screen.

var _Scene_Battle_createEnemyWindow = Scene_Battle.prototype.createEnemyWindow;
Scene_Battle.prototype.createEnemyWindow = function() {
_Scene_Battle_createEnemyWindow.call(this);
this.createPredictionWindow();
};

Scene_Battle.prototype.createPredictionWindow = function() {
this._predictionWindow = new Window_EnemyActionPrediction(0, this._enemyWindow.y + this._enemyWindow.height);
this.addWindow(this._predictionWindow);
this._enemyWindow.setPredictionWindow(this._predictionWindow);
};

})();