// AWY_MotionMaker.js Ver.4.2.0
// (C) 2023 あわやまたな

/*:
* @target MZ
* @orderAfter FesSpin
* @plugindesc Provides a function that allows you to create your own motions separate from the movement route.
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/500579516.html
* @help Ver.4.2.0
* 【How to use】
* Convert "Set Movement Route" directly
* under the plugin command "Set Motion" to motion.
* To put it simply, it is "another movement route."
* You can add movement to your character at any time you like.
*
* 【Specification】
* Please note that motion has different specifications from the movement route.
* ・All commands are executed within one frame unless there is a wait time.
* ・Proceeds even while the character is moving.
*
* Therefore, the basic idea is to create movement while adding wait commands.
*
* 【Scripts (Event)】
* this.character(characterId).isMotionPlaying() //Each Character
* $gamePlayer.isMotionPlaying() //Player
* //Is the motion playing?
*
* 【Scripts (Movement Route)】
* this.pauseMotion();  //Pause the motion.
* this.replayMotion(); //Resume the motion.
* this.clearMotion();  //Erase motion.
* this.lockPattern(true/false/integer between 0 and 2);
* //Fix the animation pattern so that it does not change arbitrarily.
*   While it is valid, the pattern will not be updated at all regardless
*   of whether there is a stepping animation or walking animation.
*   Enable if true or blank, disable if false.
*   In the case of numbers, the pattern is immediately reflected and then locked.
* this.setAppDir(0/2/4/6/8);
* //Change the apparent direction.
*   You can see its usefulness by taking a "1 Step Forward" after this script.
*   Current direction without input. Cancel with 0.
*
* 【Scripts (Motion)】
* You can use everything that can be used in "Set Movement Route".
*
* ※The following commands are not required, but may be useful in some cases.
* //LoopStart   //This is the starting point of the loop.
* //LoopEnd     //This is the end point of the loop. Subsequent commands will be executed only once at the end.
* //Pause       //Pause. It will not proceed unless restarted from the plugin command.
*
* Be sure to include a double slash before the command.
*
* @param replaceDirCommands
* @text Replace Direction Commands
* @desc Use the "Set Motion" turn commands as the "apparent direction". (Including "Direction Fix ON/OFF")
* @type boolean
* @default true
*
* @param loopStart
* @text Loop Start Point Command
* @desc Entering this value after the double slash marks the start of the loop.
* @default LoopStart
*
* @param loopEnd
* @text Loop End Point Command
* @desc Entering this value after the double slash marks the end of the loop.
* @default LoopEnd
*
* @param pause
* @text Pause Command
* @desc Entering this value after a double slash will cause a pause.
* @default Pause
*
* @command setMotion
* @text Set Motion
* @desc Converts the movement route into motion.
*
* @arg characterId
* @text Character ID
* @desc If left blank, the movement route will be followed.
*
* @arg loop
* @text Number of Loops
* @desc If left blank, the movement route option will be followed.
* Infinite loop with either 無限, infinity, or -1
* @type combo
* @option infinity
*
* @arg speed
* @text Speed
* @desc Specify as a percentage. 100 is the same size.
* @default 100
*
* @arg script
* @text Script
* @desc This is a script that is always executed when the motion ends or is interrupted.
* @type multiline_string
*
* @command clearMotion
* @text Clear Motion
* @desc Erase motion.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @command pauseMotion
* @text Pause Motion
* @desc Pause the motion.
* Motion is not deleted.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @command replayMotion
* @text Repaly Motion
* @desc Resumes a paused motion.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @command setMotionLoop
* @text Set Motion Loop
* @desc Override loop settings.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @arg loop
* @text Number of Loops
* @desc Infinite loop with either 無限, infinity, or -1
* @type combo
* @option infinity
* @default 1
*
* @command setMotionSpeed
* @text Set Motion Speed
* @desc Change playback speed.
* This is the playback speed for each motion.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @arg speed
* @text Speed
* @desc Specify as a percentage. 100 is the same size.
* @default 100
*
* @command setMotionMasterSpeed
* @text Set Motion Master Speed
* @desc Change master speed.
* This is the playback speed common to all motions.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @arg speed
* @text Speed
* @desc Specify as a percentage. 100 is the same size.
* @default 100
*
* @command setMotionScript
* @text Set Motion Script
* @desc Overwrite the script.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
* @arg script
* @text Script
* @desc This is a script that is always executed when the motion ends or is interrupted.
* @type multiline_string
*
* @command waitForCompletion
* @text Wait for Completion
* @desc Wait until the motion ends.
*
* @arg characterId
* @text Chacacter ID
* @desc If it is left blank, it will be the last target.
*
*/

/*:ja
* @target MZ
* @orderAfter FesSpin
* @plugindesc 移動ルートの設定とは別に独自モーションを作れる機能を提供します。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/article/500579516.html
* @help
* 【使い方】
* プラグインコマンド「モーションの設定」直下の移動ルートの設定を
* モーションに変換します。
* 分かりやすく言えば「もう一つの移動ルート」です。
* 好きなタイミングでキャラクターに動きを付けることが出来ます。
*
* 【仕様】
* モーションは移動ルートと仕様が異なる為、注意が必要です。
* ・ウェイトを入れない限りコマンドは1フレーム内で全て実行される
* ・キャラクターの移動中にも進行する
*
* よって、ウェイトを入れつつ動きを作っていくのが基本となります。
*
* 【スクリプト（イベント）】
* this.character(キャラクターID).isMotionPlaying() //各キャラクター
* $gamePlayer.isMotionPlaying() //プレイヤー
* //モーションが再生中か。
*
* 【スクリプト（移動ルート）】
* this.pauseMotion();  //モーションの一時停止をします。
* this.replayMotion(); //モーションの再開をします。
* this.clearMotion();  //モーションを消去します。
* this.lockPattern(true/false/0～2の整数);
* //アニメーションパターンが勝手に変わらないように固定します。
*   有効である間は、足踏みアニメ、歩行アニメの有無に関わらずパターンの更新を一切行いません。
*   trueまたは未入力で有効化、falseで無効化。
*   数字の場合はそのパターンを即時反映してからロック。
* this.setAppDir(0/2/4/6/8);
* //見かけ上の向きを変更します。
*   このスクリプトで向きを変更した後「一歩前進」を行うと有用性が分かると思います。
*   未入力で現在の向き。0で解除。
*
* 【スクリプト（モーション）】
* 「移動ルートの設定」で使用可能なものが全て使えます。
*
* ※以下のコマンドは必須ではありませんが、場合によっては便利です。
* //LoopStart   //ループの開始地点となります。
* //LoopEnd     //ループの終了地点となります。最後の一回のみそれ以降のコマンドが実行されます。
* //Pause       //一時停止をします。プラグインコマンドから再開しない限り進みません。
*
* コマンドの前に必ずダブルスラッシュを入れる事。
*
* [更新履歴]
* 2023/09/01：Ver.0.9.0b　ベータ版公開。
* 2023/09/03：Ver.1.0.0　公開。コードはベータ版のままです。
* 2023/09/09：Ver.1.0.1　モーションが無い時に「モーションの再開」を行うとエラーが起こる不具合を修正。
* 2023/09/10：Ver.2.0.0　アニメーションパターンを固定する機能を追加。
* 2023/09/12：Ver.2.0.1　パターン固定時にsetPatternを無効化するように変更。
* 2023/10/07：Ver.2.0.2　パターン固定時にstraightenを無効化するように変更。
* 2023/10/24：Ver.3.0.0　モーション終了・中断時にスクリプトを実行可能にしました。
* 2023/10/28：Ver.3.1.0　ウェイトが入力値より1フレーム多くなる不具合を修正。
* 2023/11/10：Ver.4.0.0　モーションの速度を変更可能にしました。
* 2023/12/07：Ver.4.1.0　見かけ上の向きを設定可能にしました。
* 2024/10/15：Ver.4.2.0　スロー再生時にも最初の1フレーム目は即時反映されるようにしました。
*
* @param replaceDirCommands
* @text 方向コマンド置き換え
* @desc 「モーションの設定」の向きコマンドを「見かけの方向」として使用します。（「向き固定ON/OFF」も含む）
* @type boolean
* @default true
*
* @param loopStart
* @text ループ開始地点コマンド
* @desc ダブルスラッシュの後に設定値を入力するとループ開始地点となります。
* @default LoopStart
*
* @param loopEnd
* @text ループ終了地点コマンド
* @desc ダブルスラッシュの後に設定値を入力するとループ終了地点となります。
* @default LoopEnd
*
* @param pause
* @text 一時停止コマンド
* @desc ダブルスラッシュの後に設定値を入力すると一時停止になります。
* @default Pause
*
* @command setMotion
* @text モーションの設定
* @desc 直下の移動ルートの設定をモーションに変換します。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと移動ルートの設定に従います。
*
* @arg loop
* @text ループ回数
* @desc 未記入だと移動ルートのオプションに従います。
* 無限、infinity、-1のいずれかで無限ループ
* @type combo
* @option 無限
*
* @arg speed
* @text 再生速度
* @desc 百分率で指定します。100が等倍。
* @default 100
*
* @arg script
* @text スクリプト
* @desc モーションが終了・中断された時に必ず実行するスクリプトです。
* @type multiline_string
*
* @command clearMotion
* @text モーションの消去
* @desc モーションを消去します。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @command pauseMotion
* @text モーションの一時停止
* @desc モーションを一時停止します。
* モーションは消去されません。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @command replayMotion
* @text モーションの再開
* @desc 一時停止したモーションを再開します。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @command setMotionLoop
* @text ループの設定
* @desc ループ設定を上書きします。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @arg loop
* @text ループ回数
* @desc 無限、infinity、-1のいずれかで無限ループ
* @type combo
* @option 無限
* @default 1
*
* @command setMotionSpeed
* @text 再生速度の設定
* @desc 再生速度を変更します。
* モーション個別の再生速度です。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @arg speed
* @text 再生速度
* @desc 百分率で指定します。100が等倍。
* @default 100
*
* @command setMotionMasterSpeed
* @text マスター速度の設定
* @desc マスター速度を変更します。
* 全モーション共通の再生速度です。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @arg speed
* @text 再生速度
* @desc 百分率で指定します。100が等倍。
* @default 100
*
* @command setMotionScript
* @text スクリプトの設定
* @desc スクリプトを上書きします。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
* @arg script
* @text スクリプト
* @desc モーションが終了・中断された時に必ず実行するスクリプトです。
* @type multiline_string
*
* @command waitForCompletion
* @text 完了までウェイト
* @desc モーションが終了するまで待ちます。
*
* @arg characterId
* @text キャラクターID
* @desc 未記入だと最後の対象になります。
*
*/

'use strict';
{
	const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
	const parameters = PluginManager.parameters(pluginName);
	const hasPluginCommonBase = typeof PluginManagerEx === "function";
	const replaceDirCommands = parameters.replaceDirCommands === "true";
	const loopStart = "//" + parameters.loopStart.toUpperCase();
	const loopEnd = "//" + parameters.loopEnd.toUpperCase();
	const pause = "//" + parameters.pause.toUpperCase();

	//-----------------------------------------------------------------------------
	// PluginManager

	PluginManager.registerCommand(pluginName, "setMotion", function(args) {
		const index = args.index = args.index ?? findReplaceCommandIndex(this._index, this._list);
		if (index === -1) {
			return;
		}
		const command = this._list[index];
		const params = command.parameters;
		command.setMotion = true;
		if (args.characterId) {
			params[0] = +args.characterId;
		}
		let numOfLoop = args.loop;
		if (!numOfLoop) {
			numOfLoop = params[1].repeat ? -1 : 1;
		} else if (["無限", "infinity"].includes(numOfLoop)) {
			numOfLoop = -1;
		}
		params[1].loop = +numOfLoop;
		params[1].speed = Number(args.speed || 100);
		params[1].script = args.script;
	});

	//置換するコマンドを検索。見つかったら登録、無かったらこのコマンドのインデックスを登録。
	function findReplaceCommandIndex(index, list) {
		const scanEnd = list.length - 1;
		for (let i = index + 1; i <= scanEnd; i++) {
			const command = list[i];
			if (command.code === 205) {
				return i;
			}
		}
		return -1;
	};

	PluginManager.registerCommand(pluginName, "clearMotion", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.clearMotion();
		}
	});

	PluginManager.registerCommand(pluginName, "pauseMotion", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.pauseMotion();
		}
	});

	PluginManager.registerCommand(pluginName, "replayMotion", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.replayMotion();
		}
	});

	PluginManager.registerCommand(pluginName, "setMotionLoop", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		let numOfLoop = args.loop || 0;
		if (["無限", "infinity"].includes(numOfLoop)) {
			numOfLoop = -1;
		}
		if (character) {
			character.setMotionLoop(+numOfLoop);
		}
	});

	PluginManager.registerCommand(pluginName, "setMotionSpeed", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.setMotionSpeed(Number(args.speed || 100));
		}
	});

	PluginManager.registerCommand(pluginName, "setMotionMasterSpeed", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.setMotionMasterSpeed(Number(args.speed || 100));
		}
	});

	PluginManager.registerCommand(pluginName, "setMotionScript", function(args) {
		const characterId = Number(args.characterId || this._characterId);
		const character = this.character(characterId);
		if (character) {
			character.setMotionScript(args.script);
		}
	});

	function commandList(characterId) {
		return [{"code":112,"indent":0,"parameters":[]},
		{"code":357,"indent":1,"parameters":[pluginName,"updateWaitMode","",{"characterId":characterId}]},
		{"code":0,"indent":1,"parameters":[]},
		{"code":413,"indent":0,"parameters":[]},
		{"code":0,"indent":0,"parameters":[]}];
	}

	PluginManager.registerCommand(pluginName, "waitForCompletion", function(args) {
		const characterId = Number(args.characterId || this._characterId || this._eventId);
		if (this.isOnCurrentMap()) {
			this.waitForMotion(characterId);
		}
	});
	//完了までウェイトの監視用
	PluginManager.registerCommand(pluginName, "updateWaitMode", function(args) {
		const character = this.character(args.characterId);
		if (character && character.isMotionPlaying()) {
			this.wait(1);
		} else {
			this.command113();
		}
	});

	if (hasPluginCommonBase) {
		PluginManagerEx.registerCommand(document.currentScript, "setMotion", function(args) {
			const index = args.index = args.index ?? findReplaceCommandIndex(this._index, this._list);
			if (index === -1) {
				return;
			}
			const command = this._list[index];
			const params = command.parameters;
			command.setMotion = true;
			if (args.characterId !== "") {
				params[0] = args.characterId;
			}
			let numOfLoop = args.loop;
			if (numOfLoop === "") {
				numOfLoop = params[1].repeat ? -1 : 1;
			} else if (["無限", "infinity"].includes(numOfLoop)) {
				numOfLoop = -1;
			}
			params[1].loop = numOfLoop;
			params[1].speed = args.speed || 100;
			params[1].script = args.script;
		});

		PluginManagerEx.registerCommand(document.currentScript, "clearMotion", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.clearMotion();
			};
		});

		PluginManagerEx.registerCommand(document.currentScript, "pauseMotion", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.pauseMotion();
			};
		});

		PluginManagerEx.registerCommand(document.currentScript, "replayMotion", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.replayMotion();
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "setMotionLoop", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			let numOfLoop = args.loop || 0;
			if (["無限", "infinity"].includes(numOfLoop)) {
				numOfLoop = -1;
			}
			if (character) {
				character.setMotionLoop(numOfLoop);
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "setMotionSpeed", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.setMotionSpeed(args.speed || 100);
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "setMotionMasterSpeed", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.setMotionMasterSpeed(args.speed || 100);
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "setMotionScript", function(args) {
			const characterId = args.characterId === "" ? this._characterId : args.characterId;
			const character = this.character(characterId);
			if (character) {
				character.setMotionScript(args.script);
			}
		});

		PluginManagerEx.registerCommand(document.currentScript, "waitForCompletion", function(args) {
			const characterId = args.characterId === "" ? this._characterId || this._eventId : args.characterId;
			if (this.isOnCurrentMap()) {
				this.waitForMotion(characterId);
			}
		});
	}

	//-----------------------------------------------------------------------------
	// Game_Interpreter
	// 移動ルートの設定
	const _Game_Interpreter_command205 = Game_Interpreter.prototype.command205;
	Game_Interpreter.prototype.command205 = function(params) {
		if (!this.currentCommand().setMotion) {
			return _Game_Interpreter_command205.apply(this, arguments);
		}
		$gameMap.refreshIfNeeded();
		this._characterId = params[0];
		const character = this.character(this._characterId);
		if (character) {
			character.setMotion(params[1]);
			if (params[1].wait) {
				this.waitForMotion();
			}
		}
		return true;
	};

	Game_Interpreter.prototype.waitForMotion = function(characterId = this._characterId) {
		this.setupChild(commandList(characterId), this._eventId);
	};

	//-----------------------------------------------------------------------------
	// Game_CharacterBase

	const _Game_CharacterBase_initMembers = Game_CharacterBase.prototype.initMembers;
	Game_CharacterBase.prototype.initMembers = function() {
		_Game_CharacterBase_initMembers.call(this);
		this._patternLock = false;
		this._apparentDir = 0;
	};

	const _Game_CharacterBase_straighten = Game_CharacterBase.prototype.straighten;
	Game_CharacterBase.prototype.straighten = function() {
		if (!this.isPatternLocked()) {
			_Game_CharacterBase_straighten.call(this);
		}
	};

	Game_CharacterBase.prototype.isPatternLocked = function() {
		return this._patternLock;
	};

	const _Game_CharacterBase_updateAnimation = Game_CharacterBase.prototype.updateAnimation;
	Game_CharacterBase.prototype.updateAnimation = function() {
		if (!this.isPatternLocked()) {
			_Game_CharacterBase_updateAnimation.call(this);
		}
	};

	const _Game_CharacterBase_setPattern = Game_CharacterBase.prototype.setPattern;
	Game_CharacterBase.prototype.setPattern = function(pattern) {
		if (!this.isPatternLocked()) {
			_Game_CharacterBase_setPattern.call(this, pattern);
		}
	};

	Game_CharacterBase.prototype.lockPattern = function(value = true) {
		this._patternLock = false;
		if (value === false) {
			return;
		} else if (value !== true) {
			this.setPattern(value);
		}
		this._patternLock = true;
	};

	Game_CharacterBase.prototype.apparentDir = function() {
		return this._apparentDir;
	};

	Game_CharacterBase.prototype.setAppDir = function(d = true) {
		this._apparentDir = 0;
		if (d) {
			this._apparentDir = d === true ? this.direction() : d;
		}
	};

	let rd = 0;
	if (replaceDirCommands) {
		const _Game_CharacterBase_direction = Game_CharacterBase.prototype.direction;
		Game_CharacterBase.prototype.direction = function() {
			return rd || _Game_CharacterBase_direction.call(this);
		};

		const _Game_CharacterBase_setDirection = Game_CharacterBase.prototype.setDirection;
		Game_CharacterBase.prototype.setDirection = function(d) {
			if (rd) {
				this.setAppDir(d);
				return;
			}
			_Game_CharacterBase_setDirection.call(this, d);
		};
	}

	//-----------------------------------------------------------------------------
	// Game_Character

	const _Game_Character_initMembers = Game_Character.prototype.initMembers;
	Game_Character.prototype.initMembers = function() {
		_Game_Character_initMembers.call(this);
		this._motionMasterSpeed = 100;
		this.clearMotion();
	};

	Game_Character.prototype.clearMotion = function() {
		const script = this._motion && this._motion.script;
		this._motionPlaying = false;
		this._motion = null;
		this._motionIndex = 0;
		this._motionWaitCount = 0;
		this._motionLoopStart = -1;
		this._motionLoopCount = 1;
		this._motionSpeed = 100;
		this._motionDeltaTime = 0;
		if (script) {
			eval(script);
		}
	};

	Game_Character.prototype.isMotionPlaying = function() {
		return this._motionPlaying;
	};

	Game_Character.prototype.replayMotion = function() {
		if (this._motion) {
			this._motionPlaying = true;
		}
	};

	Game_Character.prototype.pauseMotion = function() {
		this._motionPlaying = false;
	};

	Game_Character.prototype.setMotion = function(motion) {
		this.clearMotion();
		this._motionPlaying = true;
		this._motion = motion;
		this._motionLoopCount = this._motion.loop;
		this._motionSpeed = this._motion.speed;
		const speed = this._motionSpeed * this._motionMasterSpeed;
		if (speed < 10000) {
			//スローでも初回は瞬時に実行する。
			this._motionDeltaTime = 10000 - Math.floor(speed);
		}
	};

	Game_Character.prototype.setMotionLoop = function(numOfLoop) {
		this._motionLoopCount = numOfLoop;
	};

	Game_Character.prototype.setMotionMasterSpeed = function(speed) {
		this._motionMasterSpeed = speed;
	};

	Game_Character.prototype.setMotionSpeed = function(speed) {
		this._motionSpeed = speed;
	};

	Game_Character.prototype.setMotionScript = function(script) {
		const motion = this._motion;
		if (motion) {
			motion.script = script;
		}
	};

	if (Game_CharacterBase.prototype.update === Game_Character.prototype.update) {
		Game_Character.prototype.update = function() {
			Game_CharacterBase.prototype.update.apply(this, arguments);
		};
	}

	const _Game_Character_update = Game_Character.prototype.update;
	Game_Character.prototype.update = function() {
		_Game_Character_update.call(this);
		if (this._motionPlaying) {
			this.updateMotion();
		}
	};

	Game_Character.prototype.updateMotion = function() {
		this._motionDeltaTime += this._motionSpeed * this._motionMasterSpeed;
		while (this._motionDeltaTime >= 10000) {
			this._motionDeltaTime -= 10000;
			while (this._motionPlaying) {
				if (this._motionWaitCount > 0) {
					this._motionWaitCount--;
				} else {
					const command = this._motion.list[this._motionIndex];
					if (command) {
						this.processMotionCommand(command);
						this.advanceMotionIndex();
					}
				}
				if (this._motionWaitCount > 0) {
					break;
				}
			}
		}
	};

	Game_Character.prototype.processMotionEnd = function() {
		const motion = this._motion;
		if (this._motionLoopCount) {
			if (this._motionLoopCount > 0) {
				this._motionLoopCount--;
			}
			this._motionIndex = this._motionLoopStart;
		} else if (this._motionPlaying) {
			this.clearMotion();
		}
	};

	Game_Character.prototype.advanceMotionIndex = function() {
		const motion = this._motion;
		if (motion) {
			let numCommands = motion.list.length - 1;
			this._motionIndex++;
			if (this._motionPlaying && this._motionLoopCount && this._motionIndex >= numCommands) {
				if (this._motionLoopCount > 0) {
					this._motionLoopCount--;
				}
				if (this._motionLoopCount) {
					this._motionIndex = this._motionLoopStart + 1;
				}
			}
		}
	};

	let processMotionCommand = false;
	Game_Character.prototype.processMotionCommand = function(command) {
		processMotionCommand = true;
		const gc = Game_Character;
		const params = command.parameters;
		switch (command.code) {
			case gc.ROUTE_END:
				this.processMotionEnd();
				break;
			case gc.ROUTE_WAIT:
				this._motionWaitCount = params[0];
				break;
			case gc.ROUTE_SCRIPT:
				const comment = params[0].toUpperCase();
				if (comment === loopStart) {
					this.motionLoopStart();
				} else if (comment === loopEnd) {
					this.motionLoopEnd();
				} else if (comment === pause) {
					this.pauseMotion();
				} else {
					this.processMoveCommand(command);
				}
				break;
			default:
				this.processMoveCommand(command);
				break;
		}
		processMotionCommand = false;
	};

	if (replaceDirCommands) {
		const _Game_Character_processMotionCommand = Game_Character.prototype.processMotionCommand;
		Game_Character.prototype.processMotionCommand = function(command) {
			processMotionCommand = true;
			const gc = Game_Character;
			if (command.code >= gc.ROUTE_TURN_DOWN && command.code <= gc.ROUTE_TURN_AWAY) {
				rd = this.apparentDir() || this.direction();
			}
			switch (command.code) {
				case gc.ROUTE_DIR_FIX_ON:
					this.setAppDir(true);
					break;
				case gc.ROUTE_DIR_FIX_OFF:
					this.setAppDir(false);
					break;
				default:
					_Game_Character_processMotionCommand.call(this, command);
					break;
			}
			rd = 0;
			processMotionCommand = false;
		};
	}
	//FesSpin
	const _Game_Character_tryHalfSpin = Game_Character.prototype.tryHalfSpin;
	Game_Character.prototype.tryHalfSpin = function() {
		if (processMotionCommand) return;
		_Game_Character_tryHalfSpin.call(this);
	};

	Game_Character.prototype.motionLoopStart = function() {
		this._motionLoopStart = this._motionIndex;
	};

	Game_Character.prototype.motionLoopEnd = function() {
		const loop = this._motionLoopCount;
		if (loop && loop !== 1) {
			this.processMotionEnd();
		}
	};

	//-----------------------------------------------------------------------------
	// Sprite_Character

	const _Sprite_Character_characterPatternY = Sprite_Character.prototype.characterPatternY;
	Sprite_Character.prototype.characterPatternY = function() {
		const d = this._character.direction();
		this._character._direction = this._character.apparentDir() || d;
		const patternY = _Sprite_Character_characterPatternY.call(this);
		this._character._direction = d;
		return patternY;
	};

}