//-----------------------------------------------------------------------------
//  Galv's Basic Event Shadows
//-----------------------------------------------------------------------------
//  For: RPGMAKER MZ
//  GALV_CharacterShadowsMZ.js
//-----------------------------------------------------------------------------
//  2020-09-11 - Version 1.0 - release
//-----------------------------------------------------------------------------
// Terms can be found at:
// galvs-scripts.com
//-----------------------------------------------------------------------------

var Imported = Imported || {};
Imported.Galv_BasicEventShadows = true;

var Galv = Galv || {};        // Galv's main object
Galv.BES = Galv.BES || {};    // Plugin object
Galv.BES.pluginName = "GALV_CharacterShadowsMZ";

//-----------------------------------------------------------------------------
/*:ja
 * @plugindesc (v.1.0) マップ上のプレーヤ/イベントの下に影を表示します。
 * @url http://galvs-scripts.com
 * @target MZ
 * @author Galv
 * 
 * @help
 * 翻訳:ムノクラ
 * https://fungamemake.com/
 * https://twitter.com/munokura/
 * 
 * 元プラグイン:
 * https://galvs-scripts.com/2020/09/11/mz-character-shadows/
 * 
 *   Galv's Character Shadows
 * ---------------------------------------------------------------------------
 * このプラグインを使用すると、
 * マップ上のイベントとプレーヤーキャラクターに影を追加できます。
 * これらの影は、次の方法で有効/無効にできます。
 * 
 * ---------------------------------------------------------------------------
 *  イベントの影
 * ---------------------------------------------------------------------------
 * 
 *      <shadow>
 * 
 * このタグをイベントのメモ欄に入力すると、
 * イベントページに画像が含まれている限り、
 * イベントが使用するページに関係なく、常に影が付きます。
 * 
 * また、このタグをイベントページの任意の場所の'注釈'に入力できます。
 * イベントページの条件が有効である限り、影が表示されます。
 * タグがページ上にない場合、影は表示されません。
 * 
 * ---------------------------------------------------------------------------
 *  プレイヤーの影 - スクリプト
 * ---------------------------------------------------------------------------
 * 以下のスクリプトで使用できるプラグインコマンドがあります。
 * どちらも同じ機能を実行します。
 * どちらを使用するかを選択できます。
 * 
 *       Galv.BES.pShadows(true);     // プレイヤーアクターの影をオン
 *       Galv.BES.pShadows(false);    // プレイヤーアクターの影をオフ
 * 
 *       Galv.BES.eShadows(true);     // 全てのイベントで影をオン
 *       Galv.BES.eShadows(false);    // 全てのイベントの影をオフ
 * 
 * イベントコマンド'スクリプト'で使用されます。 
 * 全てのプレイヤーキャラクター/イベントの影のオンとオフを切り替えます。
 * 
 * 
 * @param sImage
 * @text 影の画像
 * @desc 使用される画像ファイル(/img/system/フォルダ内)
 * @type file
 * @dir img/system
 * @default Shadow1
 * 
 * @param sOffset
 * @type number
 * @min -9007
 * @max 9007
 * @text 影オフセット
 * @desc 影のオフセット量(ピクセル)
 * @default 5
 * 
 * @command playerShadows
 * @text プレイヤーの影
 * @desc プレイヤーの影の表示/非表示を変更する
 * 
 * @arg pStatus
 * @text 表示切り替え
 * @type boolean
 * @on 表示
 * @off 非表示
 * @default true
 * @desc プレーヤーの影(表示:true / 非表示:false)
 * 
 * @command eventShadows
 * @text イベントの影
 * @desc イベントの影の表示/非表示を変更する
 * 
 * @arg eStatus
 * @text 表示切り替え
 * @type boolean
 * @on 表示
 * @off 非表示
 * @default true
 * @desc イベントの影(表示:true / 非表示:false)
 */

//-----------------------------------------------------------------------------
//  CODE STUFFS
//-----------------------------------------------------------------------------

Galv.BES.img = PluginManager.parameters(Galv.BES.pluginName)["sImage"];
Galv.BES.os = Number(PluginManager.parameters(Galv.BES.pluginName)["sOffset"]);

Galv.BES.pShadows = function(status) {
	$gameSystem._playerShadow = status;
	SceneManager._scene._spriteset.doActorShadows();
};

Galv.BES.eShadows = function(status) {
	$gameSystem._eventShadows = status;
	SceneManager._scene._spriteset.doEventShadows();
};

PluginManager.registerCommand(Galv.BES.pluginName, "playerShadows", args => {
	let status = args.pStatus == "true";
	Galv.BES.pShadows(status);
});

PluginManager.registerCommand(Galv.BES.pluginName, "eventShadows", args => {
	let status = args.eStatus == "true";
	Galv.BES.eShadows(status);
});


// Game_System
//-----------------------------------------------------------------------------

Galv.BES.Game_System_initialize = Game_System.prototype.initialize;
Game_System.prototype.initialize = function() {
	Galv.BES.Game_System_initialize.call(this);
	this._playerShadow = true;
	this._eventShadows = true;
};


// Scene and Spriteset Map
//-----------------------------------------------------------------------------

Galv.BES.Spriteset_Map_hideCharacters = Spriteset_Map.prototype.hideCharacters;
Spriteset_Map.prototype.hideCharacters = function() {
	for (let sprite in this._bshadowSprites) {
		this._bshadowSprites[sprite].hide();
	};
    Galv.BES.Spriteset_Map_hideCharacters.call(this);
};


Galv.BES.Spriteset_Map_createCharacters = Spriteset_Map.prototype.createCharacters;
Spriteset_Map.prototype.createCharacters = function() {
	Galv.BES.Spriteset_Map_createCharacters.call(this);
	this.createCharacterShadows();
};

Spriteset_Map.prototype.createCharacterShadows = function() {
	this._bshadowSprites = {};
	this.doEventShadows();
	this.doActorShadows();
};

Spriteset_Map.prototype.doEventShadows = function() {
	$gameMap.events().forEach(function(event) {
		if ($gameSystem._eventShadows) {
        	this.createBShadow(event._eventId,event);
		} else {
			this.destroyBShadow(event._eventId,event);
		};
    }, this);
};

Spriteset_Map.prototype.doActorShadows = function() {
	if ($gameSystem._playerShadow) { // add
		let fols = $gamePlayer.followers()._data;
		for (let i = 0; i < fols.length; i++) {
			fols[i]._shadow = true;
			this.createBShadow("f" + i + 1,fols[i]);
		};
		$gamePlayer._shadow = true;
		this.createBShadow("f0",$gamePlayer);
	} else { // Remove
		let fols = $gamePlayer.followers()._data;
		for (let i = 0; i < fols.length; i++) {
			fols[i]._shadow = false;
			this.destroyBShadow("f" + i + 1,fols[i]);
		};
		$gamePlayer._shadow = false;
		this.destroyBShadow("f0",$gamePlayer);
	};
};

Spriteset_Map.prototype.createBShadow = function(id,character) {
	if (!character) return;
	if (!this._bshadowSprites[id] && character._shadow) {
		this._bshadowSprites[id] = new Sprite_BasicShadow(character);
		this._tilemap.addChild(this._bshadowSprites[id]);
		character._shadSprite = true;
	};
};

Spriteset_Map.prototype.destroyBShadow = function(id,character) {
	if (!id) return;
	if (this._bshadowSprites[id]) {
		this._tilemap.removeChild(this._bshadowSprites[id]);
		delete(this._bshadowSprites[id]);
		if (character) character._shadSprite = false;
	};
};


Spriteset_Map.prototype.destroyAllBShadows = function() {
	let fols = $gamePlayer.followers()._data;
	for (let i = 0; i < fols.length; i++) {
		fols[i]._shadow = false;
		this.destroyBShadow("f" + i + 1,fols[i]);
	};
	$gamePlayer._shadow = false;
	this.destroyBShadow("f0",$gamePlayer);
	
	$gameMap.events().forEach(function(event) {
			this.destroyBShadow(event._eventId,event);
    }, this);
};


// Game_Event
//-----------------------------------------------------------------------------

Galv.BES.Game_Event_refresh = Game_Event.prototype.refresh;
Game_Event.prototype.refresh = function() {
	Galv.BES.Game_Event_refresh.call(this);
	this.doShadows();
};

Game_Event.prototype.doShadows = function() {
	if (!$gameSystem._eventShadows) {
		if (SceneManager._scene._spriteset) SceneManager._scene._spriteset.destroyBShadow(this._eventId);
		return;
	};
	if (this.event().note.contains("<shadow>") && this._characterName != "") {
		this._shadow = true;
	} else {
		// check if page has shadow
		const page = this.page();
		let shadow = false;
		if (page) {
			for (let i = 0; i < page.list.length; i++) {
				if (page.list[i].code == 108 && page.list[i].parameters[0].contains("<shadow>")) {
					shadow = true;
				};
			};
		};
		if (shadow) {
			this._shadow = true;
			if (SceneManager._scene._spriteset)	SceneManager._scene._spriteset.createBShadow(this._eventId,this);
		} else {
			this._shadow = false;
			if (SceneManager._scene._spriteset)	SceneManager._scene._spriteset.destroyBShadow(this._eventId);
			this._shadSprite = false;
		};
	};
};

Galv.BES.Game_Event_erase = Game_Event.prototype.erase;
Game_Event.prototype.erase = function() {
	this._shadow = false;
	if (SceneManager._scene._spriteset)	SceneManager._scene._spriteset.destroyBShadow(this._eventId);
	this._shadSprite = false;
    Galv.BES.Game_Event_erase.call(this);
};


// Sprite_BasicShadow
//-----------------------------------------------------------------------------

function Sprite_BasicShadow() {
    this.initialize.apply(this, arguments);
}

Sprite_BasicShadow.prototype = Object.create(Sprite.prototype);
Sprite_BasicShadow.prototype.constructor = Sprite_BasicShadow;

Sprite_BasicShadow.prototype.initialize = function(character) {
    Sprite.prototype.initialize.call(this);
    this.setCharacter(character);
};

Sprite_BasicShadow.prototype.setCharacter = function(character) {
	this._character = character;
	this.bitmap = ImageManager.loadSystem(Galv.BES.img);
	this.anchor.x = 0.5;
	this.anchor.y = 1;
};

Sprite_BasicShadow.prototype.update = function() {
    this.x = this._character.screenX();
   	this.y = this._character.screenY() + this._character.jumpHeight() + Galv.BES.os;
    this.z = this._character.screenZ() - 1;
	if (this._character._characterName == "" || this._character._transparent) {
		this.opacity = 0;
	} else {
		this.opacity = this._character._opacity
	};
};