/*:
 * @target MZ
 * @plugindesc ピクチャをびよ～んと表示します。
 * @author はら
 * @url https://note.com/haranta/n/na3ba29af4078
 *
 * @command set
 * @text びよ～んピクチャ設定
 * @desc ピクチャをびよ～んと表示します。びよ～ん表示するピクチャは不透明度を0にしておいてください。
 *
 * @arg pictureId
 * @text ピクチャID
 * @desc びよ～ん表示するピクチャのID
 * @default 1
 * @type number
 *
 * @arg duration
 * @text 表示時間
 * @desc びよ～ん表示にかかる時間(フレーム)
 * @default 60
 * @type number
 *
 * @help
 * ピクチャをびよ～んと伸ばしながら表示します。
 * ピクチャの表示後、プラグインコマンドを使用してください。
 * 公式プラグインのTextPicture.jsで表示した文字のピクチャに使う想定で
 * 制作していますが、普通のピクチャにも使用できます。
 *
 * びよ～ん処理中のピクチャの拡大縮小、
 * メニューの開閉(シーン移動、復帰)などは考慮していません。
 * びよ～んが終われば通常のピクチャと同じように扱えます。
 *
 *
 * 更新履歴
 * 　2024/02/13 ver 1.0 作成
 *
 * 利用規約
 * 　禁止事項はありません。
 */


(function() {
    'use strict';

    const pluginName = "HR_BiyoonPicture";

    PluginManager.registerCommand(pluginName, "set", args => {
        const pictureId     = args.pictureId > 0 ? args.pictureId : 1;
        const duration      = args.duration > 0 ? args.duration : 1;
        $gameScreen.biyoonPicture(pictureId, duration);
    });

    // Game_Screen
    Game_Screen.prototype.biyoonPicture = function(pictureId, duration) {
        const picture = this.picture(pictureId);
        if (picture) picture.biyoonMove(duration);
    };

    // Game_Picture
    Game_Picture.prototype.biyoonMove = function(duration) {
        this._biyoonDuration = duration;
        this._biyoonWholeDuration = duration;
        this._biyoonDummyOrigin = false;
        if (this._origin == 1) {
            this._origin = 0;
            this._biyoonDummyOrigin = true;
        }
        this._biyoonSetUp = true;
    };

    Game_Picture.prototype.biyoonClear = function() {
        delete this._biyoonDummyOrigin;
        delete this._biyoonDuration;
        delete this._biyoonWholeDuration;
        delete this._biyoonWidth;
        delete this._biyoonHeight;
        delete this._biyoonFrameWidth;
        delete this._biyoonSpriteX;
        delete this._biyoonSpriteScaleX;
        delete this._biyoonSetUp;
    };

    Game_Picture.prototype.biyoonDuration = function() {
        return this._biyoonDuration;
    };

    Game_Picture.prototype.biyoonWholeDuration = function() {
        return this._biyoonWholeDuration;
    };

    Game_Picture.prototype.biyoonWidth = function() {
        return this._biyoonWidth ? this._biyoonWidth : 0;
    };

    Game_Picture.prototype.setBiyoonWidth = function(width) {
        this._biyoonWidth = width;
    };

    Game_Picture.prototype.biyoonHeight = function() {
        return this._biyoonHeight ? this._biyoonHeight : 0;
    };

    Game_Picture.prototype.setBiyoonHeight = function(height) {
        this._biyoonHeight = height;
    };

    Game_Picture.prototype.biyoonFrameWidth = function() {
        return this._biyoonFrameWidth ? this._biyoonFrameWidth : 0;
    };

    Game_Picture.prototype.biyoonSpriteX = function() {
        return this._biyoonSpriteX ? this._biyoonSpriteX : 0;
    };

    Game_Picture.prototype.biyoonSpriteScaleX = function() {
        return this._biyoonSpriteScaleX ? this._biyoonSpriteScaleX : 0;
    };

    Game_Picture.prototype.biyoonSpriteAnchorY = function() {
        return this.origin() == 0 ? 0.0 : 0.5;
    };

    const _Game_Picture_x = Game_Picture.prototype.x;
    Game_Picture.prototype.x = function() {
        const adjX = this._biyoonAdjustX ? this._biyoonAdjustX : 0;
        return _Game_Picture_x.apply(this, arguments) - adjX;
    };

    const _Game_Picture_y = Game_Picture.prototype.y;
    Game_Picture.prototype.y = function() {
        const adjY = this._biyoonAdjustY ? this._biyoonAdjustY : 0;
        return _Game_Picture_y.apply(this, arguments) - adjY;
    };

    Game_Picture.prototype.biyoonSetUp = function() {
        return !!this._biyoonSetUp;
    };

    Game_Picture.prototype.setBiyoonSetUp = function(bool) {
        this._biyoonSetUp = bool;
    };

    const _Game_Picture_update = Game_Picture.prototype.update;
    Game_Picture.prototype.update = function() {
        _Game_Picture_update.apply(this, arguments);
        this.updateBiyoon();
    };

    Game_Picture.prototype.updateBiyoon = function() {
        const dur = this.biyoonDuration();
        const wholeDur = this.biyoonWholeDuration();
        if (dur > 0 && this.biyoonWidth() > 0) {
            const frameW = this.biyoonWidth() * (1 - ((dur - 1) / wholeDur));
            if (this._biyoonDummyOrigin) {
                // 中央原点時、見た目を維持しながら一時的に左上原点にする
                if (dur == 1) {
                    // 処理終了、原点を中央に戻す
                    this._origin = 1;
                    this._biyoonDummyOrigin = false;
                    delete this._biyoonAdjustX;
                    delete this._biyoonAdjustY;
                } else {
                    // 処理中、原点を中央に見せる調整
                    this._biyoonAdjustX = this.biyoonWidth() / 2;
                    this._biyoonAdjustY = this.biyoonHeight() / 2;
                }
            }
            this._biyoonSpriteX = frameW - 1;
            this._biyoonSpriteScaleX = this.biyoonWidth() - frameW;
            this._biyoonFrameWidth = frameW;
            this._biyoonDuration--;
        }
    };

    // Sprite_Picture
    const _Sprite_Picture_updateOther = Sprite_Picture.prototype.updateOther;
    Sprite_Picture.prototype.updateOther = function() {
        _Sprite_Picture_updateOther.apply(this, arguments);
        this.updateBiyoon();
    };

    Sprite_Picture.prototype.updateBiyoon = function() {
        this.setBiyoonSize();
        const gamePic = this.picture();
        const bmpHeight = gamePic.biyoonHeight();
        const frameWidth = gamePic.biyoonFrameWidth();
        if (frameWidth <= 0) return;
        this.createBiyoonSprite();
        if (gamePic.biyoonDuration() === 0 && gamePic.biyoonWholeDuration() > 0) {
            // ラストの一回だけ
            this.removeChild(this._biyoonSprite);
            gamePic.biyoonClear();
        }
        if (this._biyoonSprite) {
            this._biyoonSprite.x = gamePic.biyoonSpriteX();
            this._biyoonSprite.scale.x = gamePic.biyoonSpriteScaleX();
            this._biyoonSprite.setFrame(frameWidth - 1, 0, 1, bmpHeight);
        }
        this.setFrame(0, 0, frameWidth, bmpHeight);
    };

    Sprite_Picture.prototype.setBiyoonSize = function() {
        const gamePic = this.picture();
        if (gamePic.biyoonSetUp()) {
            gamePic.setBiyoonWidth(this.bitmap.width);
            gamePic.setBiyoonHeight(this.bitmap.height);
        }
    };

    Sprite_Picture.prototype.createBiyoonSprite = function() {
        const gamePic = this.picture();
        if (gamePic.biyoonSetUp()) {
            this._biyoonSprite = new Sprite();
            this._biyoonSprite.bitmap = this.bitmap;
            this._biyoonSprite.anchor.x = 0.0;
            this._biyoonSprite.anchor.y = gamePic.biyoonSpriteAnchorY();
            this.addChild(this._biyoonSprite);
            gamePic._opacity = 255;
            gamePic.setBiyoonSetUp(false);
        }
    };

})();