//=============================================================================
// MM_ErasePictureAnimation.js
//=============================================================================
// Copyright (c) 2024- 牛乳もなか
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
//
// Twitter
// https://x.com/milkmonaka_1
//=============================================================================
// Version
// 1.0.2 2025/1/11  一部のアニメーションにskewを追加しました。若干奥行のある飛び方をします。
//                  ピースの透明度減少を設定できるパラメータを追加しました。
//                  エネミーのスプライトをワールド座標で取得するように変更しました、
//                  ただこれでもカメラが動くと結局ズレます、擬似3Dやバトルカメラとは相性が悪いと思ってください。
// 1.0.1 2024/11/10 エネミーの消滅エフェクトに使用できる機能を追加。
//                  voronoi,delaunay分割時のレンダリング方法を変更。
//                  一部のアニメーションが意図した動きになっていない不具合を修正。
//                  Bounceなどの物理計算を調整、破片の高さなどに応じて自然に跳ねるように。
// 1.0.0 2024/11/03 テスト公開
//=============================================================================
/*:ja
* @target MZ
* @plugindesc ピクチャを様々なパターンで破壊するアニメーションを実装します。
* @url https://x.com/milkmonaka_1
* @author milkmonaka
* @help プラグインコマンドでピクチャIDを指定して実行します。
* このプラグインでアニメーションする画像はピクチャそのものではなく、
* 「全く同じ描画をしたダミースプライト」です、Sprite_Pictureクラスは
* オーバーライドしていません。
* ダミースプライト表示時にピクチャは勝手に消えるので破壊して消えるような演出が
* 実現できます。
* 各アニメーションの分割係数は使用する画像サイズやPCスペックに合わせて
* 慎重に設定してください、各々環境によりますので少しづつ上げ下げして試して下さい。
*
* 目安として：
*   - 四角形分割の分割点数は4以上。
*   - D3分割の分割点数は100～500程度を目安にして、PCの負荷を確認してください。
*   - 表示する画像のサイズや同時表示数によっても異なります「使用は自己責任です」。
*
* このプラグインは画像の分割にd3-delaunayライブラリを使用しています。
* https://unpkg.com/d3-delaunay よりd3-delaunay.min.jsをダウンロードして
* プロジェクト内のjs/libsフォルダに配置してください。
*
* 「消滅エフェクトに使用」をONに設定することで敵を倒した際の消滅エフェクト時に
* プラグインパラメータで設定した破壊アニメーションを実行できます。
* パラメータはエネミーのメモ欄で
* <CrushEffect: 分割数, フレーム, アニメーションタイプ, 分割方式>
* と記載して個別に指定することもでき、
* タグがない場合はプラグインパラメータの値を使用します。
* 例: <CrushEffect: 100, 60, gravity, voronoi>
* この場合voronoi分割でgravityのアニメーションを分割係数100の60フレームで実行します。
* 分割係数はsquare, voronoi, delaunay
* アニメーションタイプはプラグインパラメータの一覧から名称を確認してください。
* パラメータの設定値記載ミス、特に分割係数に注意してください。
*
* @param enableEffect
* @text 消滅エフェクトに使用
* @type boolean
* @default false
* @desc エネミー消滅時に破壊エフェクトを使用します。
*
* @param defaultPieceSize
* @text デフォルトの分割係数
* @type number
* @default 10
* @desc エネミー消滅時のデフォルトの分割係数（細かくするほど負荷が上がります）
*
* @param defaultDuration
* @text デフォルトのエフェクト時間
* @type number
* @default 60
* @desc エネミー消滅時のエフェクト完了までのフレーム数
*
* @param defaultAnimationType
* @text デフォルトのアニメーションタイプ
* @type select
* @option shatter
* @option rotate
* @option gravity
* @option evaporate
* @option explode
* @option explodeD3
* @option wave
* @option converge
* @option rebuild
* @option randomFade
* @option bounce
* @option peel
* @default shatter
* @desc エネミー消滅時のデフォルトのアニメーションタイプ
*
* @param defaultSplitType
* @text デフォルトの分割タイプ
* @type select
* @option square
* @option voronoi
* @option delaunay
* @default square
* @desc エネミー消滅時のデフォルトの分割方法
*
* @command applyPictureEffect
* @text 四角形分割を適用
* @desc 指定したピクチャに四角形分割を適用します。
*
* @arg pictureId
* @text ピクチャID
* @type number
* @min 1
* @desc エアニメーションを適用するピクチャのIDを指定します。
* @default 1
*
* @arg duration
* @text アニメーションの時間
* @desc アニメーションが完了するまでのフレーム数を指定します。
* @type number
* @min 1
* @default 60
*
* @arg alphaDecay
* @text 透明度減少速度
* @type number
* @min 0
* @desc アニメーション中の透明度減少速度0~1（1: 完了時に透明、0: 透明度を減少無し）'rebuild'など一部無効の物もあります。
* @default 1
*
* @arg pieceSize
* @text 分割係数
* @desc 値が「小さい」ほど細かく砕けますが負荷がかかります。1はかなり重いのでどんなに下げても2までを推奨します。
* @type number
* @min 1
* @default 10
*
* @arg animationType
* @text アニメーションタイプ
* @desc 破片が飛び散る方法を指定します。
* @type select
* @option オーソドックスに砕け散ります（shatter）
* @value shatter
* @option 砕けた後回転しながら消えます（rotate）
* @value rotate
* @option 砕けた後下に落ちます（gravity）
* @value gravity
* @option 砕けた後上方向に蒸発 (evaporate)
* @value evaporate
* @option 砕けた後爆発的に飛びちります（explode）
* @value explode
* @option 砕けた後波打ちながら消えます（wave）
* @value wave
* @option 砕けた後中心に向かって収束して消えます（converge）
* @value converge
* @option 砕けて飛散した後、元の画像に再構築します（rebuild）
* @value rebuild
* @option ピースがランダムにフェードアウトします（randomFade）
* @value randomFade
* @option 砕け落ちてバウンドします (bounce)
* @value bounce
* @default shatter
*
* @command applyPictureEffectD3
* @text D3分割を適用
* @desc 指定したピクチャにD3ライブラリを使用した分割エフェクトを適用します。
*
* @arg pictureId
* @text ピクチャID
* @type number
* @min 1
* @desc アニメーションを適用するピクチャのIDを指定します。
* @default 1
*
* @arg duration
* @text アニメーションの時間
* @desc アニメーションが完了するまでのフレーム数を指定します。
* @type number
* @min 1
* @default 60
*
* @arg alphaDecay
* @text 透明度減少速度
* @type number
* @min 0
* @desc アニメーション中の透明度減少速度0~1（1: 完了時に透明、0: 透明度を減少無し）'rebuild'など一部無効の物もあります。
* @default 1
*
* @arg pieceSize
* @text 分割点数
* @desc 値が「大きい」ほど細かく分割しますが負荷がかかります。スペックに自身の無い方は少しづつ試すことを推奨。
* @type number
* @min 1
* @max 1000
* @default 60
*
* @arg animationType
* @text アニメーションタイプ
* @desc 破片が飛び散る方法を指定します。
* @type select
* @option オーソドックスに砕け散ります（shatter）
* @value shatter
* @option 砕けた後回転しながら消えます（rotate）
* @value rotate
* @option 砕けた後下に落ちます（gravity）
* @value gravity
* @option 砕けた後上方向に蒸発 (evaporate)
* @value evaporate
* @option 砕けた後爆発的に飛びちります（explodeD3）
* @value explodeD3
* @option 砕けた後波打ちながら消えます（wave）
* @value wave
* @option 砕けて飛散した後、元の画像に再構築します（rebuild）
* @value rebuild
* @option ピースがランダムにフェードアウトします（randomFade）
* @value randomFade
* @option 砕け落ちてバウンドします (bounce)
* @value bounce
* @option ピースが一枚づつ下に落下します (peel)
* @value peel
* @default shatter
*
* @arg splitType
* @text 分割タイプ
* @desc ピクチャを分割する方法を選択します。
* @type select
* @option 多角形 (Voronoi)
* @value voronoi
* @option 三角形 (Delaunay)
* @value delaunay
* @default voronoi
*/

(() => {
  'use strict';

  function loadLocalLibrary(path, callback) {
    const script = document.createElement('script');
    script.src = path;
    script.onload = callback;
    script.onerror = () => {
      console.error(`Error: Failed to load the library from ${path}.`);
      alert(`ライブラリ ${path} の読み込みに失敗しました。`);
    };
    document.head.appendChild(script);
  }

  loadLocalLibrary('js/libs/d3-delaunay.min.js', () => {
    try {
      if (typeof d3 === 'undefined') {
        throw new Error('d3-delaunay library did not load correctly.');
      }
      console.log('d3-delaunay library loaded successfully.');
    } catch (error) {
      console.error(error.message);
      alert('d3-delaunay library is not loaded correctly.');
    }
  });

  const pluginName = decodeURIComponent(document.currentScript.src).match(/^.*\/js\/plugins\/(.+)\.js$/)[1];
  const parameters = PluginManager.parameters(pluginName);
  const enableEffect = parameters['enableEffect'] === 'true';
  const defaultPieceSize = Number(parameters['defaultPieceSize'] || 10);
  const defaultDuration = Number(parameters['defaultDuration'] || 60);
  const defaultAnimationType = parameters['defaultAnimationType'] || 'shatter';
  const defaultSplitType = parameters['defaultSplitType'] || 'square';

  let activeAnimations = [];
  let currentScene;

  PluginManager.registerCommand(pluginName, 'applyPictureEffect', args => {
      const pictureId = Number(args.pictureId);
      const picture = $gameScreen.picture(pictureId);
      if (!picture) return;

      const pieceSize = Number(args.pieceSize);
      const duration = Number(args.duration);
      const animationType = args.animationType || 'shatter';
      const alphaDecay = args.alphaDecay !== undefined ? Number(args.alphaDecay) : 1;
      const container = new PIXI.Container();
      container._animationId = Date.now(); // 一意のIDを付与

      const bitmap = ImageManager.loadPicture(picture._name);
      bitmap.addLoadListener(() => {
          if (!bitmap.isReady()) {
              console.error('Image did not load.');
              return;
          }
          const animation = new ErasePictureAnimation(picture, pieceSize, duration, animationType, container, 'square', alphaDecay);
          SceneManager._scene.addChild(container);
          $gameScreen.erasePicture(pictureId);
          animation.start(animationType, pictureId, picture, bitmap);
      });
  });

  PluginManager.registerCommand(pluginName, 'applyPictureEffectD3', args => {
    const pictureId = Number(args.pictureId);
    const picture = $gameScreen.picture(pictureId);
    if (!picture) return;

    const numPoints = Number(args.pieceSize);
    const duration = Number(args.duration);
    const animationType = args.animationType || 'shatter';
    const alphaDecay = args.alphaDecay !== undefined ? Number(args.alphaDecay) : 1;
    const splitType = args.splitType || 'voronoi';
    const container = new PIXI.Container();
    container._animationId = Date.now(); // 一意のIDを付与

    const bitmap = ImageManager.loadPicture(picture._name);
    bitmap.addLoadListener(() => {
      if (!bitmap.isReady()) {
        console.error('Image did not load.');
        return;
      }
      const animation = new ErasePictureAnimation(picture, numPoints, duration, animationType, container, splitType, alphaDecay);
      SceneManager._scene.addChild(container);
      $gameScreen.erasePicture(pictureId);
      animation.start(animationType, pictureId, picture, bitmap);
    });

  });

  //ピクチャをpieceSizeに応じて四角形分割する
  function pictureCreatePieces(bitmap, picture, pieceSize) {
    const pieces = [];
    const rows = Math.ceil(bitmap.height / pieceSize);
    const cols = Math.ceil(bitmap.width / pieceSize);
    // Sprite_PictureかSprite_Enemyかでプロパティを分岐
    const isSpriteEnemy = picture instanceof Sprite_Enemy;
    const scaleX = isSpriteEnemy ? picture.scale.x : picture.scaleX() / 100;
    const scaleY = isSpriteEnemy ? picture.scale.y : picture.scaleY() / 100;
    const origin = isSpriteEnemy ? 1 : (picture._origin !== undefined ? picture._origin : 0);
    const posX = isSpriteEnemy ? picture.worldTransform.tx : picture.x();
    const posY = isSpriteEnemy ? picture.worldTransform.ty : picture.y();

    for (let y = 0; y < rows; y++) {
      for (let x = 0; x < cols; x++) {
        const pieceWidth = Math.min(pieceSize, bitmap.width - x * pieceSize);
        const pieceHeight = Math.min(pieceSize, bitmap.height - y * pieceSize);

        const texture = new PIXI.Texture(
          bitmap._baseTexture,
          new PIXI.Rectangle(x * pieceSize, y * pieceSize, pieceWidth, pieceHeight)
        );

        // 作成したテクスチャを使って新しいスプライトを生成
        const sprite = new PIXI.Sprite(texture);
        sprite.scale.set(scaleX, scaleY);
        sprite.anchor.set(0.5, 0.5);
        sprite.originalX = posX;
        sprite.originalY = posY;
        sprite.originalWidth = bitmap.width;
        sprite.originalHeight = bitmap.height;

        // 原点に応じて位置を調整
        if (origin === 0) {
          sprite.x = posX + (x * pieceSize + pieceWidth / 2) * scaleX;
          sprite.y = posY + (y * pieceSize + pieceHeight / 2) * scaleY;
        } else {
          sprite.x = posX + (x * pieceSize + pieceWidth / 2) * scaleX - (bitmap.width * scaleX / 2);
          if (isSpriteEnemy) {
            sprite.y = posY + (y * pieceSize + pieceHeight / 2) * scaleY - (bitmap.height * scaleY);
          } else {
            sprite.y = posY + (y * pieceSize + pieceHeight / 2) * scaleY - (bitmap.height * scaleY / 2);
          }
        }

        if (isSpriteEnemy) {
          sprite.originalY = posY * scaleY - (bitmap.height * scaleY);
        }

        // インデックス情報を保持
        sprite.indexX = x;
        sprite.indexY = y;
        sprite.rows = rows;
        sprite.cols = cols;

        pieces.push(sprite);
      }
    }

    return pieces;
  }

  //ピクチャをnumPointsに応じて多角形分割する
  function pictureCreatePiecesVoronoi(bitmap, picture, numPoints) {
      const pieces = [];
      const isSpriteEnemy = picture instanceof Sprite_Enemy;
      const scaleX = isSpriteEnemy ? picture.scale.x : picture.scaleX() / 100;
      const scaleY = isSpriteEnemy ? picture.scale.y : picture.scaleY() / 100;
      const origin = isSpriteEnemy ? 1 : (picture._origin !== undefined ? picture._origin : 0);
      const posX = isSpriteEnemy ? picture.worldTransform.tx : picture.x();
      const posY = isSpriteEnemy ? picture.worldTransform.ty : picture.y();

      // Voronoi分割用の点を生成
      const points = [];
      for (let i = 0; i < numPoints; i++) {
          points.push([Math.random() * bitmap.width, Math.random() * bitmap.height]);
      }
      const delaunay = d3.Delaunay.from(points);
      const voronoi = delaunay.voronoi([0, 0, bitmap.width, bitmap.height]);

      // 各Voronoiセルをピースとして扱う
      for (let i = 0; i < numPoints; i++) {
          const cell = voronoi.cellPolygon(i);

          // セルの境界（最小・最大座標）を計算
          let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
          cell.forEach(([x, y]) => {
              minX = Math.min(minX, x);
              minY = Math.min(minY, y);
              maxX = Math.max(maxX, x);
              maxY = Math.max(maxY, y);
          });

          // セルの境界に基づいてテクスチャを作成
          const textureWidth = maxX - minX;
          const textureHeight = maxY - minY;
          const texture = PIXI.RenderTexture.create({ width: textureWidth, height: textureHeight });

          // マスクを使用してピースを切り取り、個別のテクスチャに描画
          const baseSprite = new PIXI.Sprite(new PIXI.Texture(bitmap._baseTexture));
          const mask = new PIXI.Graphics();
          mask.beginFill(0xffffff);
          mask.drawPolygon(cell.map(point => [point[0] - minX, point[1] - minY]).flat());
          mask.endFill();

          baseSprite.mask = mask;
          baseSprite.position.set(-minX, -minY);
          Graphics.app.renderer.render(baseSprite, texture);

          // 作成したテクスチャを使って新しいスプライトを生成
          const sprite = new PIXI.Sprite(texture);
          sprite.scale.set(scaleX, scaleY);
          sprite.anchor.set(0.5, 0.5);
          sprite.originalX = posX;
          sprite.originalY = posY;
          sprite.originalWidth = bitmap.width;
          sprite.originalHeight = bitmap.height;
          sprite.centerX = bitmap.width / 2
          sprite.centerY = bitmap.height / 2
          sprite.cell = cell;

          // 原点による位置補正
          if (origin === 0) {  // 左上
              sprite.x = posX + (minX + maxX) * scaleX / 2;
              sprite.y = posY + (minY + maxY) * scaleY / 2;
          } else {  // 中心
              sprite.x = posX + (minX + maxX - bitmap.width) * scaleX / 2;
                if (isSpriteEnemy) {
              sprite.y = posY + (minY + maxY - bitmap.height * 2) * scaleY / 2;
                } else {
              sprite.y = posY + (minY + maxY - bitmap.height) * scaleY / 2;
            }
          }

          if (isSpriteEnemy) {
            sprite.originalY = posY * scaleY - (bitmap.height * scaleY);
          }

          pieces.push(sprite);

          // リソース破棄
          mask.destroy(true);
          baseSprite.destroy({ children: true, texture: true });
      }

      return pieces;
  }

  //ピクチャをnumPointsに応じて三角形分割する
  function pictureCreatePiecesDelaunay(bitmap, picture, numPoints) {
      const pieces = [];
      const isSpriteEnemy = picture instanceof Sprite_Enemy;
      const scaleX = isSpriteEnemy ? picture.scale.x : picture.scaleX() / 100;
      const scaleY = isSpriteEnemy ? picture.scale.y : picture.scaleY() / 100;
      const origin = isSpriteEnemy ? 1 : (picture._origin !== undefined ? picture._origin : 0);
      const posX = isSpriteEnemy ? picture.worldTransform.tx : picture.x();
      const posY = isSpriteEnemy ? picture.worldTransform.ty : picture.y();

      // Delaunay分割用の点を生成
      const points = [];
      for (let i = 0; i < numPoints; i++) {
        points.push([Math.random() * bitmap.width, Math.random() * bitmap.height]);
        for (let x = 0; x <= bitmap.width; x += bitmap.width / 2) {
          points.push([x, 0], [x, bitmap.height]);
        }
        for (let y = 0; y <= bitmap.height; y += bitmap.height / 2) {
          points.push([0, y], [bitmap.width, y]);
        }
      }
      const delaunay = d3.Delaunay.from(points);

      // 各三角形セルをピースとして扱う
      for (let i = 0; i < delaunay.triangles.length; i += 3) {
          const vertices = [
              points[delaunay.triangles[i]],
              points[delaunay.triangles[i + 1]],
              points[delaunay.triangles[i + 2]]
          ];

          // 三角形セルの境界（最小・最大座標）を計算
          let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
          vertices.forEach(([x, y]) => {
              minX = Math.min(minX, x);
              minY = Math.min(minY, y);
              maxX = Math.max(maxX, x);
              maxY = Math.max(maxY, y);
          });

          // 境界に基づいてテクスチャを作成
          const textureWidth = maxX - minX;
          const textureHeight = maxY - minY;
          const texture = PIXI.RenderTexture.create({ width: textureWidth, height: textureHeight });

          // マスクを使用して三角形セルを切り取り、個別のテクスチャに描画
          const baseSprite = new PIXI.Sprite(new PIXI.Texture(bitmap._baseTexture));
          const mask = new PIXI.Graphics();
          mask.beginFill(0xffffff);
          mask.drawPolygon(vertices.map(([x, y]) => [x - minX, y - minY]).flat());
          mask.endFill();

          baseSprite.mask = mask;
          baseSprite.position.set(-minX, -minY);
          Graphics.app.renderer.render(baseSprite, texture);

          // 作成したテクスチャを使って新しいスプライトを生成
          const sprite = new PIXI.Sprite(texture);
          sprite.scale.set(scaleX, scaleY);
          sprite.anchor.set(0.5, 0.5);
          sprite.originalX = posX;
          sprite.originalY = posY;
          sprite.originalWidth = bitmap.width;
          sprite.originalHeight = bitmap.height;
          sprite.centerX = bitmap.width / 2
          sprite.centerY = bitmap.height / 2
          sprite.cell = vertices;

          // 原点による位置補正
          if (origin === 0) {  // 左上
              sprite.x = posX + (minX + maxX) * scaleX / 2;
              sprite.y = posY + (minY + maxY) * scaleY / 2;
          } else {  // 中心
              sprite.x = posX + (minX + maxX - bitmap.width) * scaleX / 2;
                if (isSpriteEnemy) {
              sprite.y = posY + (minY + maxY - bitmap.height * 2) * scaleY / 2;
                } else {
              sprite.y = posY + (minY + maxY - bitmap.height) * scaleY / 2;
            }
          }

          if (isSpriteEnemy) {
            sprite.originalY = posY * scaleY - (bitmap.height * scaleY);
          }

          pieces.push(sprite);

          // リソース破棄
          mask.destroy(true);
          baseSprite.destroy({ children: true, texture: true });
      }

      return pieces;
  }

  // 粉砕アニメーションの管理クラス
  class ErasePictureAnimation {
    constructor(picture, pieceSize, duration, animationType, container, splitType, alphaDecay = 1) {
        this.picture = picture;
        this.pieceSize = pieceSize;
        this.duration = duration;
        this.animationType = animationType;
        this.container = container;
        this.frameCount = 0;
        this.splitType = splitType;
        this.alphaDecay = alphaDecay;
        this.activeAnimations = [];
        this.currentScene = SceneManager._scene;
    }

    start(animationType, pictureId, picture, bitmap) {
      let pieces
      pieces = this.createPieces(bitmap);
      const updater = new AnimationUpdater(pieces, this.duration, this.alphaDecay, this.container._animationId, this.animationType);

      const animate = (animationId) => {
        if (SceneManager._scene !== this.currentScene) {
          this.stop(animationId);
          return;
        }

        if (updater.update()) {
          this.activeAnimations.push(requestAnimationFrame(() => animate(animationId)));
        } else {
          if (animationType === 'rebuild') {
           $gameScreen.showPicture(pictureId, picture._name, picture.origin(), picture.x(), picture.y(), picture.scaleX(), picture.scaleY(), picture.opacity(), picture.blendMode());
          }
          this.stop(animationId);
        }
      };

      this.activeAnimations.push(requestAnimationFrame(() => animate(Date.now())));
    }

    stop(animationId) {
      this.activeAnimations.forEach(id => cancelAnimationFrame(id));
      this.activeAnimations = [];
      SceneManager._scene.removeChild(this.container);
      this.container.destroy({ children: true });
    }

    createPieces(bitmap) {
      let pieces;
      if (this.splitType === 'delaunay') {
        pieces = pictureCreatePiecesDelaunay(bitmap, this.picture, this.pieceSize);
      } else if (this.splitType === 'voronoi') {
        pieces = pictureCreatePiecesVoronoi(bitmap, this.picture, this.pieceSize);
      } else {
        pieces = pictureCreatePieces(bitmap, this.picture, this.pieceSize);
      }


      pieces.forEach(piece => {
        this.createAnimation(this.animationType, piece, this.duration, this.duration);

        this.container.addChild(piece);
      });
      return pieces;
    }

    createAnimation(type, sprite, duration) {
      switch (type) {
        case 'shatter':
        return new ShatterAnimation(sprite);
        case 'rotate':
        return new RotateAnimation(sprite);
        case 'gravity':
        return new GravityAnimation(sprite);
        case 'evaporate':
        return new EvaporateAnimation(sprite);
        case 'explode':
        return new ExplodeAnimation(sprite);
        case 'explodeD3':
        return new ExplodeAnimationD3(sprite);
        case 'wave':
        return new WaveAnimation(sprite);
        case 'converge':
        return new ConvergeAnimation(sprite);
        case 'rebuild':
        return new RebuildAnimation(sprite);
        case 'randomFade':
        return new RandomFadeAnimation(sprite, duration);
        case 'bounce':
        return new BounceAnimation(sprite);
        case 'peel':
        return new PeelAnimation(sprite);
        default:
        console.warn(`Unknown animation type: ${type}`);
        break;
      }
    }
  }

  // 各アニメーションのアップデートを行う
  class AnimationUpdater {
    constructor(pieces, duration, alphaDecay, containerId, animationType) {
      this.duration = duration;
      this.frame = 0;
      this.animationInstances = pieces.map((piece, index) => {
        switch (animationType) {
          case 'shatter':
          return new ShatterAnimation(piece, alphaDecay);
          case 'rotate':
          return new RotateAnimation(piece, alphaDecay);
          case 'gravity':
          return new GravityAnimation(piece, alphaDecay);
          case 'evaporate':
          return new EvaporateAnimation(piece, alphaDecay);
          case 'explode':
          return new ExplodeAnimation(piece, alphaDecay);
          case 'explodeD3':
          return new ExplodeAnimationD3(piece, alphaDecay);
          case 'wave':
          return new WaveAnimation(piece, alphaDecay);
          case 'converge':
          return new ConvergeAnimation(piece);
          case 'rebuild':
          return new RebuildAnimation(piece, this.duration);
          case 'randomFade':
          return new RandomFadeAnimation(piece, this.duration);
          case 'bounce':
          return new BounceAnimation(piece);
          case 'peel':
          const delay = index * 3;
          return new PeelAnimation(piece, delay);
          default:
          console.warn(`Unknown animation type: ${animationType}`);
          return null;
        }
      }).filter(instance => instance !== null); // 無効なインスタンスを除外
    }

    update() {
      this.frame++;
      const progress = this.frame / this.duration;

      for (let instance of this.animationInstances) {
        // スプライトが有効かチェックし、シーン切替時などに停止させる
        if (!instance.sprite || !instance.sprite.parent) {
          return false; // アニメーション終了を指示
        }
        instance.update(progress, this.frame);
      }

      return this.frame < this.duration; // アニメーション終了でfalse
    }
  }

  // オプションをオブジェクトで設定
  class SpriteUpdateOptions {
      constructor({ vx = 0, vy = 0, rotationSpeed = 0, scaleSpeed = 0, gravity = 0, alphaDecay = 0, progress = 0, skewX = 0, skewY = 0 } = {}) {
          this.vx = vx;
          this.vy = vy;
          this.rotationSpeed = rotationSpeed;
          this.scaleSpeed = scaleSpeed;
          this.gravity = gravity;
          this.alphaDecay = alphaDecay;
          this.progress = progress;
          this.skewX = skewX;
          this.skewY = skewY;
      }
  }

  // ピースのアニメーションのヘルパーだが動きの都合上ここを使っていない物もあります
  class AnimationHelper {
      static updateSprite(sprite, options) {
          options = options || new SpriteUpdateOptions();

          sprite.x += options.vx;
          sprite.y += options.vy;

          if (options.rotationSpeed !== 0) {
              sprite.rotation += options.rotationSpeed;
          }

          if (options.scaleSpeed !== 0) {
              sprite.scale.x += options.scaleSpeed;
              sprite.scale.y += options.scaleSpeed;
          }

          sprite.vy += options.gravity;

          if (options.alphaDecay !== 0) {
              sprite.alpha = Math.max(0, 1 - options.progress * options.alphaDecay);
          }

          sprite.skew.x += options.skewX;
          sprite.skew.y += options.skewY;
      }
  }

  //-----------------------------------------------------------------------------
  // 以下動作アニメーションクラス
  //
  //

  // 粉砕アニメーション
  class ShatterAnimation {
    constructor(sprite, alphaDecay) {
      sprite.vx = (Math.random() - 0.5) * 2;
      sprite.vy = (Math.random() - 0.5) * 2;
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            alphaDecay: this.alphaDecay,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 回転アニメーション
  class RotateAnimation {
    constructor(sprite, alphaDecay) {
      sprite.vx = (Math.random() - 0.5) * 2;
      sprite.vy = (Math.random() - 0.5) * 2;
      sprite.rotationSpeed = 0.05;
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            rotationSpeed: this.sprite.rotationSpeed,
            alphaDecay: this.alphaDecay,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 重力アニメーション
  class GravityAnimation {
    constructor(sprite, alphaDecay) {
      sprite.vx = (Math.random() - 0.5) * 10;
      sprite.vy = (Math.random() - 1) * 5;
      sprite.skew.x = (Math.random() - 0.5) * 0.5;
      sprite.skew.y = (Math.random() - 0.5) * 0.5;
      sprite.gravity = 0.5
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            gravity: this.sprite.gravity,
            alphaDecay: this.alphaDecay,
            skewX: 0.01,
            skewY: -0.01,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 蒸発アニメーション
  class EvaporateAnimation {
    constructor(sprite, alphaDecay) {
      sprite.vx = (Math.random() - 0.5) * 5;
      sprite.vy = (Math.random() - 0.5) * 3;
      sprite.gravity = -0.1
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            gravity: this.sprite.gravity,
            alphaDecay: this.alphaDecay,
            skewX: 0.01,
            skewY: -0.01,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 爆散アニメーション
  class ExplodeAnimation {
    constructor(sprite, alphaDecay) {
      const angle = Math.atan2(sprite.indexY - sprite.rows / 2, sprite.indexX - sprite.cols / 2);
      const speed = Math.random() * 5 + 2;
      sprite.vx = Math.cos(angle) * speed;
      sprite.vy = Math.sin(angle) * speed;
      sprite.skew.x = (Math.random() - 0.5) * 0.5;
      sprite.skew.y = (Math.random() - 0.5) * 0.5;
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            alphaDecay: this.alphaDecay,
            skewX: 0.01,
            skewY: -0.01,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 爆散アニメーション(d3)
  class ExplodeAnimationD3 {
    constructor(sprite, alphaDecay) {
      // 重心計算
      const centroid = sprite.cell.reduce((acc, point) => {
        return [acc[0] + point[0], acc[1] + point[1]];
      }, [0, 0]).map(coord => coord / sprite.cell.length);

      const angle = Math.atan2(centroid[1] - sprite.centerY, centroid[0] - sprite.centerX);
      const speed = Math.random() * 10 + 2;
      sprite.vx = Math.cos(angle) * speed;
      sprite.vy = Math.sin(angle) * speed;
      sprite.skew.x = (Math.random() - 0.5) * 0.5;
      sprite.skew.y = (Math.random() - 0.5) * 0.5;
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            alphaDecay: this.alphaDecay,
            skewX: 0.01,
            skewY: -0.01,
            scaleSpeed:  0.01,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // ウェーブアニメーション
  class WaveAnimation {
    constructor(sprite, alphaDecay) {
      sprite.vx = (Math.random() - 0.5) * 10;
      sprite.amplitude = Math.random() * 10 + 5; // 波の振幅
      sprite.frequency = Math.random() * 0.1 + 0.05; // 波の周波数
      this.alphaDecay = alphaDecay;
      this.sprite = sprite;
    }

    update(progress, frame) {
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: Math.sin(frame * this.sprite.frequency) * this.sprite.amplitude,
            alphaDecay: this.alphaDecay,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // 収束アニメーション
  class ConvergeAnimation {
    constructor(sprite) {
      sprite.vx = (sprite.cols / 2 - sprite.indexX) * 0.05;
      sprite.vy = (sprite.rows / 2 - sprite.indexY) * 0.05;
      sprite.scaleSpeed = -0.02; // 徐々に縮小
      this.sprite = sprite;
    }

    update(progress) {

      if (this.sprite.scale.x <= 0.01 || this.sprite.scale.y <= 0.01) {
        this.sprite.scale.set(0, 0);
        return;
      }
        const options = new SpriteUpdateOptions({
            vx: this.sprite.vx,
            vy: this.sprite.vy,
            alphaDecay: 1,
            scaleSpeed: this.sprite.scaleSpeed,
            progress
        });
        AnimationHelper.updateSprite(this.sprite, options);
    }
  }

  // ピース復元アニメーション
  class RebuildAnimation {
    constructor(sprite, duration) {
      sprite.vx = (Math.random() - 0.5) * 10; // ランダムな速度
      sprite.vy = (Math.random() - 0.5) * 10; // ランダムな速度
      sprite.returnSpeed = 0.05; // 元の位置に戻る速度
      sprite.baseX = sprite.x
      sprite.baseY = sprite.y
      this.sprite = sprite;
      this.duration = duration;
    }

    update(progress, frame) {
      if (frame < this.duration / 1.5) {
        // アニメーションの前半はランダムに飛び散る
        this.sprite.x += this.sprite.vx;
        this.sprite.y += this.sprite.vy;
        this.sprite.alpha = 1 - progress;
      } else {
        // アニメーションの後半は元の位置に戻る
        this.sprite.alpha += 1 - progress;
        this.sprite.x += (this.sprite.baseX - this.sprite.x) * this.sprite.returnSpeed * 2;
        this.sprite.y += (this.sprite.baseY - this.sprite.y) * this.sprite.returnSpeed * 2;
      }
    }
  }

  // ピースがランダムな順序でフェードアウト
  class RandomFadeAnimation {
    constructor(sprite, duration) {
      sprite.fadeDelay = Math.random() * (duration / 2);
      this.sprite = sprite;
      this.duration = duration;
      this.fadeStarted = false;
    }

    update(progress, frame) {
      // フェード開始まで待機
      if (frame >= this.sprite.fadeDelay && !this.fadeStarted) {
        this.fadeStarted = true; // フェードを開始
        this.fadeStartFrame = frame;
      }

      // フェードアウト開始で進行
      if (this.fadeStarted) {
        const fadeProgress = (frame - this.fadeStartFrame) / (this.duration / 2);
        this.sprite.alpha = Math.max(1 - fadeProgress * 2, 0);
      }
    }
  }

  // 重力バウンドアニメーション
  class BounceAnimation {
    constructor(sprite, alphaDecay) {
          sprite.vx = (Math.random() - 0.5) * 5; // 横方向のランダムな動き
          sprite.vy = -5;                        // 上方向に跳ねる初速度
          this.sprite = sprite;
      }

      update(progress) {
          // 基本の重力と反発係数
          const baseGravity = 0.5;       // 基本重力加速度
          const baseBounceFactor = -0.6; // 基本反発係数
          const friction = 0.8;          // 摩擦係数
          const stopThreshold = 0.1;     // 完全制止の閾値

          // 高度に応じた重力の変動
          const heightFactor = (this.sprite.originalY + this.sprite.originalHeight - this.sprite.y) / this.sprite.originalHeight;
          const gravity = baseGravity + heightFactor * 0.3; // 高度に応じて重力を増加

          this.sprite.vy += gravity;

          // スプライトの位置を速度に応じて更新
          this.sprite.x += this.sprite.vx;
          this.sprite.y += this.sprite.vy;

          // バウンド処理
          if (this.sprite.y >= this.sprite.originalY + this.sprite.originalHeight && this.sprite.vy > 0) {
              this.sprite.y = this.sprite.originalY + this.sprite.originalHeight; // 地面位置に固定

              // 落下速度に応じて反発係数を増加させる
              const bounceFactor = baseBounceFactor * Math.min(Math.abs(this.sprite.vy) / 6, 1.2); // 最大1.2倍

              this.sprite.vy *= bounceFactor;  // Y方向速度の反発
              this.sprite.vx *= friction;      // 摩擦によるX方向速度の減少

              // バウンドが小さくなった場合に制止
              if (Math.abs(this.sprite.vy) < stopThreshold) {
                  this.sprite.vy = 0; // Y方向を停止
                  this.sprite.vx = 0; // X方向を停止
                  this.sprite.y = this.sprite.originalY + this.sprite.originalHeight; // 位置を地面に固定
              }
          }

          // 透明度の減少
          this.sprite.alpha = 1 - progress;
      }
  }

  // ピースが一枚ずつ時間差で落下
  class PeelAnimation {
    constructor(sprite, delay) {
      sprite.vx = 0;
      sprite.vy = 0;
      sprite.startDelay = delay || 0;
      this.sprite = sprite;
      this.elapsed = 0; // 経過フレーム数
    }

    update(progress, frame) {
      if (frame >= this.sprite.startDelay) {
        this.sprite.vy += 0.5; // 重力による加速
        this.sprite.y += this.sprite.vy;
        this.sprite.alpha = 1 - progress / 2;
        this.sprite.skew.x += 0.02
        this.sprite.skew.y -= 0.02
      }
    }
  }

  //-----------------------------------------------------------------------------
  // Sprite_Enemy
  //
  // エネミー消滅エフェクトを実行するために、各消滅エフェクトのstartをオーバーライドします。

  const _Sprite_Enemy_startCollapse = Sprite_Enemy.prototype.startCollapse;
  Sprite_Enemy.prototype.startCollapse = function() {
    this.applyCrushEffect();
    _Sprite_Enemy_startCollapse.call(this);
  };

  const _Sprite_Enemy_startBossCollapse = Sprite_Enemy.prototype.startBossCollapse;
  Sprite_Enemy.prototype.startBossCollapse = function() {
    this.applyCrushEffect();
    _Sprite_Enemy_startBossCollapse.call(this);
  };

  const _Sprite_Enemy_startInstantCollapse = Sprite_Enemy.prototype.startInstantCollapse;
  Sprite_Enemy.prototype.startInstantCollapse = function() {
    this.applyCrushEffect();
    _Sprite_Enemy_startInstantCollapse.call(this);
  };

  // エネミーの破砕エフェクトを適用する関数を追加
  Sprite_Enemy.prototype.applyCrushEffect = function() {
    if (!enableEffect) return;

    // 規定値の設定
    const meta = this._enemy.enemy().meta;
    const crushEffectParams = meta["CrushEffect"] ? meta["CrushEffect"].split(',').map(param => param.trim()) : [];

    // エネミーのメモタグからパラメータを取得、タグが存在しない場合は規定値を使用
    const pieceSize = Math.min(Number(crushEffectParams[0]) || defaultPieceSize, 1000);　// 最大1000タイプミス時などの保険　
    const duration = Math.max(Number(crushEffectParams[1]) || defaultDuration, 1); // 最低1フレーム
    const animationType = crushEffectParams[2] || defaultAnimationType;
    const splitType = crushEffectParams[3] || defaultSplitType;

    const container = new PIXI.Container();
    this.parent.addChild(container);

    // Bitmap内容を新しいBitmapに描画してコピー
    const bitmap = new Bitmap(this.bitmap.width, this.bitmap.height);
    bitmap.blt(this.bitmap, 0, 0, this.bitmap.width, this.bitmap.height, 0, 0);

    bitmap.addLoadListener(() => {
      if (!bitmap.isReady()) {
        console.error('Image did not load.');
        return;
      }
      const animation = new ErasePictureAnimation(this, pieceSize, duration, animationType, container, splitType);
      animation.start(animationType, null, this, bitmap);
    });
  };

})();
