//=============================================================================
// TRP_ParticleMZ_ExAttractMove.js
//=============================================================================
/*:
 * @author Thirop
 * @target MZ
 * @plugindesc パーティクルを指定したターゲットに向かわせる
 * @help
 * ※グループコマンド内で「attract」コマンドを使用するには
 * 本体プラグインのv1.25(MV版は2.25)へのアップデートが必要です。
 *
 * ◆MV形式コマンド
 * 「particle attract 管理ID ターゲット X座標 Y座標 緩急係数」
 * 管理ID:操作対象のパーティクル管理ID
 * ターゲット:パーティクルを向かわせる対象
 * X座標:X方向の座標調整(ピクセル単位)、ターゲットtilesetのみタイル単位
 * Y座標:Y方向の座標調整(ピクセル単位)、ターゲットtilesetのみタイル単位
 * 緩急係数:1.0で等速で対象に向かう、1以下で最初早く(イーズアウト)、1以上で最初遅く(イーズイン)
 *
 * ◆有効なターゲット
 * player → プレイヤー
 * follower:1~4 → フォロワー
 * event:イベントID → イベント
 * party:1~4 → 味方(戦闘中)
 * enemy:1~ → 敵(戦闘中)
 * picture:ピクチャID → ピクチャ
 * screen/battle → 画面固定(左上が0,0)
 * tilemap → マップ上(スクロール連動)
 *
 * ◆補足事項
 * ・tilemapのみX座標/Y座標はタイル単位。(細かな調整は小数で)
 * ・キャラクター対象の場合はY座標を-24とすると縦方向の中心
 * ・キャラから発生してピクチャの上などに向かわせたいときはパーティクル再生時に
 * 　Z値を「spriteset」などとすると◯
 * ・スクリプトから呼び出す際は
 * 「$gameScreen._particle.particleAttract(0,管理ID,ターゲット,X座標,Y座標,緩急係数);」
 * 
 *
 *
 * @command attract
 * @text attract
 * @desc 指定したターゲットにパーティクルを向かわせる
 *
 * @arg id
 * @text 管理ID
 * @desc 操作対象のパーティクル管理ID
 *
 * @arg target
 * @text ターゲット
 * @desc パーティクルを向かわせる対象。this,player,event:イベントIDなど(help参照)
 * 
 * @arg x
 * @text X座標(pixel)
 * @desc X方向の座標調整(ピクセル単位)、ターゲットtilesetのみタイル単位
 * @default 0
 *
 * @arg y
 * @text Y座標(pixel)
 * @desc Y方向の座標調整(ピクセル単位)、ターゲットtilesetのみタイル単位。キャラ対象は-24で中心
 * @default 0
 *
 * @arg coeff
 * @text 緩急係数
 * @desc 1.0で等速で対象に向かう、1以下で最初早く(イーズアウト)、1以上で最初遅く(イーズイン)
 * @default 1.0
 *
 * 
 */
//============================================================================= 

(function(){
'use strict';

var pluginName = 'TRP_ParticleMZ_ExAttractMove';

(()=>{
	const command = 'attract';
	PluginManager.registerCommand(pluginName, command, function(args){
		var argsArr = Object.values(args)
		argsArr.unshift(command);

		var eventId = this.eventId();
		$gameScreen._particle.pluginCommand(this,argsArr,eventId);
	});
})();


var _Game_Particle__pluginCommand = Game_Particle.prototype._pluginCommand;
Game_Particle.prototype._pluginCommand = function(interpreter,sub,args,eventId){
	if(sub === 'attract'){
		args.unshift(eventId);
		this.particleAttract.apply(this,args);
		args.shift();
	}else{
		_Game_Particle__pluginCommand.call(this,interpreter,sub,args,eventId);
	}
};

var TARGET_TYPES = ParticleEmitter.TARGET_TYPES;
Game_Particle.prototype.particleAttract = function(eventId,id,target,x=0,y=0,coeff=1.0){
	var targetType = TARGET_TYPES.character;
	var targetId = 0;
	var clearFlag = false;

	if(target==='clear'){
		clearFlag = true;
	}else if(target instanceof PIXI.Container){
		targetType = TARGET_TYPES.displayObject;
		targetId = id+'::attract';
		Game_Particle.displayObjects[targetId] = target;
	}else if(target === 'this'){
		targetId = eventId;
	}else if(!isNaN(target)){
		targetId = Number(target);
	}else if(target==='player'){
		targetId = 0;
	}else if(target==='tilemap'){
		targetType = TARGET_TYPES.tilemap;
	}else if(target==='screen'||target==='spriteset'){
		targetType = TARGET_TYPES.screen;
	}else if(target==='battle'){
		targetType = TARGET_TYPES.battle;
	}else if(target==='battleWeather'){
		targetType = TARGET_TYPES.battleWeather;
	}else{
		var targetElems = target.split(':');
		targetId = targetElems.length>=2 ? Number(targetElems[1]) : 0;
		switch(targetElems[0].toLowerCase()){
		case 'follower':
			targetType = TARGET_TYPES.character;
			targetId *= -1;
			break;
		case 'skit':
		case 'battleSkit':
			targetType = TARGET_TYPES[targetElems[0]];
			if(!this._skitIds.contains(targetElems[1])){
				this._skitIds.push(targetElems[1]);
			}
			targetId = this._skitIds.indexOf(targetElems[1]);
			break;
		case 'event':
			targetType = TARGET_TYPES.character;
			break;
		default:
			if(TARGET_TYPES[targetElems[0]]){
				targetType = TARGET_TYPES[targetElems[0]];
			}
		}
	}

	/* apply command
	===================================*/
	var targetIds = this.targetIds(id);
	var length = targetIds.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var id = targetIds[i];
        var data = this.dataWithId(id,true);
        if(data){
        	if(clearFlag){
        		data.attract = null;
        	}else{
	        	data.attract = {
	        		targetType,
	        		targetId,
	        		x:Number(x)||0,
	        		y:Number(y)||0,
	        		coeff:Number(coeff)||1.0,
	        	};
        	}
        }
    }
};

var _Game_Particle_particleData = Game_Particle.prototype.particleData;
Game_Particle.prototype.particleData = function(eventId,id,target,name,z,x,y,image,ignoreError=false){
	var data = _Game_Particle_particleData.call(this,eventId,id,target,name,z,x,y,image,ignoreError);
	data.attract = null;
	return data;
};




//=============================================================================
// ParticleEmitter
//=============================================================================
var _ParticleEmitter_initMembers = ParticleEmitter.prototype.initMembers;
ParticleEmitter.prototype.initMembers = function(){
	_ParticleEmitter_initMembers.call(this);
	this._attract = null;
	this._useMoveContainer = PluginManager._scripts.contains('TRP_ParticleMZ_ExMove');
};


var _ParticleEmitter_update = ParticleEmitter.prototype.update;
ParticleEmitter.prototype.update = function(){
	var ret = _ParticleEmitter_update.call(this);

	if(this._attract && this._attract.data!==this._data.attract){
		if(!this._data.attract){
			this._attract = null;
		}else{
			this.setupAttract(this._data.attract);
		}
	}else if(!this._attract && this._data.attract){
		this.setupAttract(this._data.attract);
	}

	return ret;
};

ParticleEmitter.prototype.setupAttract = function(attract){
	var target;
	var id = attract.targetId;

	var parent = SceneManager._scene;

	switch(attract.targetType){
	case TARGET_TYPES.displayObject:
		target = {
			x:attract.x,
			y:attract.y
		}

		if(Game_Particle.displayObjects[this._targetId]){
			parent = (Game_Particle.displayObjects[this._targetId].parent||parent);
		}
		break;
	case TARGET_TYPES.character:
		if(id === 0){
			target = $gamePlayer;
		}else if(id<0){
			target = $gamePlayer._followers.follower(id*-1-1);
			if(!target || !target.actor()){
				target = null;
			}
		}else{
			target = $gameMap.event(id);
		}
		if(SceneManager._scene._spriteset && SceneManager._scene._spriteset._tilemap){
			parent = SceneManager._scene._spriteset._tilemap;
		}
		break;
	case TARGET_TYPES.screen:
		break;
	case TARGET_TYPES.battle:
		break;
	case TARGET_TYPES.tilemap:
		target = {
			x:attract.x,
			y:attract.y
		};
		if(SceneManager._scene._spriteset && SceneManager._scene._spriteset._tilemap){
			parent = SceneManager._scene._spriteset._tilemap;
		}
		break;
	case TARGET_TYPES.actor:
		target = this._actorSpriteTarget($gameActors.actor(id));
		parent = target.parent;
		break;
	case TARGET_TYPES.party:
		target = this._actorSpriteTarget($gameParty.members()[id-1]);
		parent = target.parent;
		break;
	case TARGET_TYPES.enemy:
		target = this._enemySpriteTarget($gameTroop.members()[id-1]);
		parent = target.parent;
		break;
	
	case TARGET_TYPES.picture:
	case TARGET_TYPES.battlePicture:
		target = id;
		if(SceneManager._scene._spriteset && SceneManager._scene._spriteset._pictureContainer){
			parent = SceneManager._scene._spriteset._pictureContainer;
		}
		break;
	case TARGET_TYPES.skit:
	case TARGET_TYPES.battleSkit:
		target = id;
		if(SceneManager._scene._spriteset && SceneManager._scene._spriteset._bustPictureContainer){
			parent = SceneManager._scene._spriteset._bustPictureContainer;
		}
		break;
	default:
		if(type<-1){
			target = {
				x:Graphics.width/2,
				y:Graphics.height/2
			};
		}
	}


	//calc dx, dy
	var dx = attract.x;
	var dy = attract.y;
	var c = this._useMoveContainer ? (this._moveContainer||this._container) : this._container;

	var diffParent = (parent&&parent!==c.parent)?parent:null;
	this._attract = {
		data:attract,
		target,
		parent:diffParent,
		dx,
		dy,
		pos0:diffParent ? new Point(0,0) : null,
		pos1:diffParent ? new Point(0,0) : null,
		pos2:diffParent ? new Point(0,0) : null,
	};
};

var _ParticleEmitter__updateEmitter = ParticleEmitter.prototype._updateEmitter;
ParticleEmitter.prototype._updateEmitter = function(dt){
	_ParticleEmitter__updateEmitter.call(this,dt);

	if(dt && this._attract && this._emitter){
		this.updateAttract();
	}
};



ParticleEmitter.prototype.updateAttract = function(){
	var attract = this._attract;
	var coeff = attract.data.coeff;
	var target = attract.target;
	var x,y;

	switch(attract.data.targetType){
	case TARGET_TYPES.displayObject:
		target = Game_Particle.displayObjects[attract.data.targetId];
		if(!target || !target.parent){
			Game_Particle.displayObjects[attract.data.targetId] = null;
			this._attract = null;
			return;
		}else{
			x = target.x;
			y = target.y;
		}
		break;
	case TARGET_TYPES.character:
		if(!target){
			this._attract = null;
			return;
		}
		x = target.screenX();
		y = target.screenY();
		break;
	case TARGET_TYPES.tilemap:
		x = Math.round($gameMap.adjustX(target.x)*48+24);
		y = Math.round($gameMap.adjustY(target.y)*48+24);
		break;
	case TARGET_TYPES.actor:
	case TARGET_TYPES.party:
	case TARGET_TYPES.enemy:
		if(!target){
			this._attract = null;
			return;
		}
		x = target.x;
		y = target.y - this.battlerSpriteHeight(target)/2;
		break;

	case TARGET_TYPES.screen:
	case TARGET_TYPES.battle:
	case TARGET_TYPES.battleWeather:
		x = 0;
		y = 0;
		break;

	case TARGET_TYPES.picture:
	case TARGET_TYPES.battlePicture:
		target = $gameScreen.picture(target);
		if(!target){
			this._attract = null;
			return;
		}
		x = target.x();
		y = target.y();
		break;
	case TARGET_TYPES.skit:
	case TARGET_TYPES.battleSkit:
		target = $gameScreen.picture($gameScreen._particle.skitPictureId(target));
		if(!target){
			this._attract = null;
			return;
		}
		x = target.x();
		y = target.y();
		break;
	};


	x += attract.dx;
	y += attract.dy;

	if(attract.parent){
		if(this._useMoveContainer){
			this._moveContainer.worldTransform.applyInverse(attract.pos0,attract.pos1);
		}else{
			this._container.worldTransform.applyInverse(attract.pos0,attract.pos1);
		}
		attract.parent.worldTransform.applyInverse(attract.pos0,attract.pos2);

		x += attract.pos1.x-attract.pos2.x;
		y += attract.pos1.y-attract.pos2.y;
	}else{
		if(this._useMoveContainer){
			x -= this._moveContainer.x;
			y -= this._moveContainer.y;
		}else{
			x -= this._container.x;
			y -= this._container.y;
		}
	}

	var p = this._emitter._activeParticlesFirst;
	var rate;

	while(!!p){
		var d = Math.floor((p.maxLife-p.age)*1000/16);
		if(d===0){
			rate = 1;
		}else {
			rate = coeff === 1 ? 1/d : 1/Math.pow(d,coeff);
		}

		p.position.x = p.position.x+(x-p.position.x)*rate;
		p.position.y = p.position.y+(y-p.position.y)*rate;

		p = p.next;
	}
};







})();