//============================================================================
// SU_ExpandedPictureSpine.js - Version 1.0.4
//============================================================================
/*:
 * @plugindesc v1.0.4 - PictureSpine.js の機能を拡張する
 * @author Su
 *
    * @help
    * ============================================================================
    * 概要
    * ============================================================================
    * PictureSpine.js を機能拡張し、プラグインコマンドを追加します。
    * このプラグインは PictureSpine.js のすぐ下に置いてください
    *
    * ピクチャの表示とアニメーション指定を同時に行えるコマンドを追加しています。
    *
    * ============================================================================
    * 開発者向けに
    * ============================================================================
    * Game_Spineクラスにイベント登録・削除を行うメンバ関数を追加しました。
    * アニメーション中のイベント（Spine側で設定するものではなく）に関数を登録することができます。
    * - start
    * - interrupt
    * - dispose
    * - end
    * - complete
    * - event
    *
    * イベントリスナー の登録
    * addEventListener(eventName:string, listener:function, <trackId:number>, <options:object>)
    * イベントリスナー の削除
    * removeEventListener(eventName:string, listener:function, <trackId:number>)
    *
    * ============================================================================
    * プラグインコマンド
    * ============================================================================
    * 値は \v[変数番号] とすることで変数の中身に置き換えることができます。
    *
    * - S_setAnimation [pictureId] [skeletonId] (animation:[trackId]:[animationName] <mix:[mixValue]>)
    *  指定のピクチャ番号のスケルトンにアニメーションをセットします。
    *  指定番号のピクチャが表示されていない場合、ピクチャの表示を同時に行います。
    *  SkeletonId はピクチャの表示を同時に行う場合のみに有効な値で、プラグインパラメータで設定した
    *  リストの行番号と対応しており、該当のスケルトンが表示されます。
    *  値は [ ] で囲った部分です
    *  < > で囲った中身は省略可能
    *   pictureId : ピクチャ番号
    *   skeletonId : プラグインパラメータのスケルトンリスト内の番号
    *   trackId : アニメーションをセットしたいトラック番号
    *   animationName :  セットしたいアニメーション名、「,」でつなぐことで複数指定することができます。
    *                   複数選択した場合は順番にアニメーションが再生され、最後のものが繰り返し表示されます。
    *   mixValue : 手前で指定したアニメーション遷移に対するミックス値
    *   [ ]、< >, ( ) は無視してください。
    *
    * 例）１０番のピクチャに「走る」というアニメーションをトラック番号１に指定する場合
    *   S_setAnimation 10 1 animation:1:走る
    *
    * 例）上記に加え、ミックス値を 0.5 に設定する場合
    *   S_setAnimation 10 1 animation:1:走る mix:0.5
    *
    *  また ( ) で囲っている animation と mix の指定は繰り返し行うことができます
    * 例）上記の例に加え、「狙う」 というアニメーションをトラック番号３にミックス値０．３で指定する場合
    *   S_setAnimation 10 1 animation:1:走る mix:0.5 animation:3:狙う mix:0.3
    *
    * 例）トラック番号に変数１番、アニメーション名に変数２番の値を使用する場合
    *   S_setAnimation 10 1 animation:\v[1]:\v[2]
    *
    *
    * - S_setAnimationR [pictureId] [skeletonId] (animation:[trackId]:[animationName])
    *  指定のピクチャ番号のスケルトンにアニメーションをセットします。
    *  指定番号のピクチャが表示されていない場合、ピクチャの表示を同時に行います。
    *  S_setAnimation との違いは、アニメーションを複数指定した場合、
    *  順次実行ではなく、どれか一つがランダムに選択され、アニメーション終了毎に再選出を繰り返すことです。
    *  アニメーション名の後ろに /%[数字] とすることで、そのアニメーションの選択率を調整することができます。
    *  値は [ ] で囲った部分です
    *  < > で囲った中身は省略可能
    *   pictureId : ピクチャ番号
    *   skeletonId : プラグインパラメータのスケルトンリスト内の番号
    *   trackId : アニメーションをセットしたいトラック番号
    *   animationName :  セットしたいアニメーション名、「,」でつなぐことで複数指定することができます。
    *   [ ]、< >, ( ) は無視してください。
    *
    * 例）「走る」、「歩く」というアニメーションをランダムに再生する場合
    *   S_setAnimation 10 1 animation:1:走る,歩く
    *
    * 例）上記に加え、「走る」の選出率を3倍にする場合
    *   S_setAnimation 10 1 animation:1:走る/%3,歩く
    *
    *  - S_setAnimationP [pictureId] [skeletonId] [presetNum]
    *  指定のピクチャ番号のスケルトンにアニメーションプリセット番号のアニメーションをセットします。
    *  基本的な動作は S_setAnimation と同じですが、プラグインパラメータで予めアニメーションを指定して利用します。
    *   pictureId : ピクチャ番号
    *   skeletonId : プラグインパラメータのスケルトンリスト内の番号
    *   presetNum : プラグインパラメータのアニメーションプリセットの番号
    * 
    * 例）ピクチャ番号10に対してスケルトンリスト1番目のスケルトンを設定し、アニメーションプリセット番号1のアニメーションをセットする場合
    *   S_setAnimationP 10 1 1
    * 
    * - S_setSkin [pictureId] [skeletonId] ([skinName])
    *  指定のピクチャ番号のスケルトンにスキンをセットします。
    *  スキン名は空白を置いて複数指定することができます。
    *  値は [ ] で囲った部分です
    *  ( ) で囲った部分は複数指定することができます。
    *   pictureId : ピクチャ番号
    *   skeletonId : プラグインパラメータのスケルトンリスト内の番号
    *   skinName : スキン名
    *
    * 例）「スーツ」というスキンをセットする場合
    *   S_setSkin 10 1 スーツ
    *
    * 例）「スーツ」と「バッグ」という２つのスキンをセットする場合
    *   S_setSkin 10 1 スーツ バッグ
    * 
    * - S_setMix [pictureId] [skeletonId] [mixValue]
    *  指定のピクチャ番号のスケルトンの全てのアニメーションのミックス値を設定します。
    *   pictureId : ピクチャ番号
    *   mixValue : ミックス値
    * 
    * 
    * - S_setMix [pictureId] [skeletonId] ([beforeAnimationName,afterAnimationName:mixValue])
    *  指定のピクチャ番号のスケルトンのアニメーションのミックス値を設定します。
    *  ( ) で囲った部分は複数指定することができます。
    *   pictureId : ピクチャ番号
    *   beforeAnimationName : 遷移元のアニメーション名
    *   afterAnimationName : 遷移後のアニメーション名
    *   mixValue : ミックス値
    * 
    * 例）「idle」から「run」に切り替わる際のミックス値を0.5秒にする
    *   S_setMix 10 idle,run:0.5
    *  
    * 例）「idle」から「run」に切り替わる際のミックス値を0.5秒にし、「run」から「idle」に切り替わる際のミックス値を0.8秒にする
    *   S_setMix 10 idle,run:0.5 run,idle:0.8
    * 
    * - S_setMosaic [pictureId] [skeletonId] [size]
    *  指定のピクチャ番号のスケルトン全体にモザイクをかけます。
    *   pictureId : ピクチャ番号
    *   size : モザイクのサイズ(px)
    * 
    * 例）スケルトン全体に5pxのモザイクをかける
    *   S_setMosaic 10 5
    * 
    * 
    * - S_setMosaic [pictureId] [skeletonId] [image] [size]
    *  指定のピクチャ番号のスケルトンの画像にモザイクをかけます。
    *   pictureId : ピクチャ番号
    *   image : モザイクをかける画像名
    *   size : モザイクのサイズ(px)
    * 
    * 例）「モザイク」という名前の画像に8pxのモザイクをかける
    *   S_setMosaic 10 モザイク 8
    * 
    * 
    * - S_in [pictureId] [skeletonId] [frame]
    *  指定のピクチャ番号のスケルトンを徐々にフェードインさせます。
    *   pictureId : ピクチャ番号
    *   frame : フェードインにかかるフレーム数
    * 
    * 例）360フレームかけてフェードインさせる
    *   S_in 10 360
    * 
    * - S_showIn [pictureId] [skeletonId] [frame]
    *  指定のピクチャ番号のスケルトンを透明度最大状態から徐々にフェードインさせます。
    *   pictureId : ピクチャ番号
    *   frame : フェードインにかかるフレーム数
    * 
    * 例）360フレームかけてフェードインさせる
    *   S_showin 10 360
    * 
    * 
    * - S_out [pictureId] [skeletonId] [frame]
    *  指定のピクチャ番号のスケルトンを徐々にフェードアウトさせます。
    *  フェードアウト完了後に見えなくなってもスケルトンは存在しているので、完全に削除する場合は S_delete を利用してください。
    *   pictureId : ピクチャ番号
    *   frame : フェードアウトにかかるフレーム数
    * 
    * 例）360フレームかけてフェードアウトさせる
    *   S_out 10 360
    * 
    * - S_delete [pictureId] ([frame])
    *  指定のピクチャ番号のスケルトンを徐々にフェードアウトさせた後に削除します。
    *   pictureId : ピクチャ番号
    *   frame : フェードアウトにかかるフレーム数(省略することで即時削除)
    * 
    * 例）ピクチャ番号10のスケルトンを削除する
    *   S_delete 10
    * 
    * 例）360フレームかけてフェードアウトさせた後にスケルトンを削除する
    *   S_delete 10 360
    * 
    * 
    * - S_setA [pictureId] [skeletonId] [alpha] [frame]
    *  指定ピクチャ番号のスケルトンを徐々に指定した透明度に変化させます。
    *   pictureId : ピクチャ番号
    *   alpha : 目標透明度(0~1 の範囲で 0で完全に透明になります)
    *   frame : 変化にかかるフレーム数
    * 
    * 例）600フレームかけて透明度を半分の0.5にする
    *   S_setA 10 0.5 600
    * 
    * 
    * - S_setB [pictureId] [skeletonId] [bright] [frame]
    *  指定ピクチャ番号のスケルトンを徐々に暗く変化させます。
    *   pictureId : ピクチャ番号
    *   bright : 目標明度(0~1 の範囲で 0で完全に黒になります)
    *   frame : 変化にかかるフレーム数
    * 
    * 例）600フレームかけて完全に黒にする
    *   S_setB 10 0 600
    * 
    * 
    * - S_setC [pictureId] [skeletonId] [r] [g] [b] [frame]
    *  指定ピクチャ番号のスケルトンの色を徐々に変化させます。
    *   pictureId : ピクチャ番号
    *   r : 目標赤要素(0~1 の範囲で 1で赤要素を最大まで強めます)
    *   g : 目標緑要素(0~1 の範囲で 1で緑要素を最大まで強めます)
    *   b : 目標青要素(0~1 の範囲で 1で青要素を最大まで強めます)
    *   frame : 変化にかかるフレーム数
    * 
    * 例）600フレームかけて赤要素をなくします（青緑がかった色になる）
    *   S_setC 10 0 1 1 600
    * 
    * 
    * - S_waitOn
    * フェード処理が完了するまでwaitが掛かるようになります。
    *
    * - S_waitOff
    * フェード処理の完了を待たずに次のコマンドを実行するようになります。

 * ============================================================================
 * 変更履歴
 * ============================================================================
 * Version 1.0.4 - 2023/11/21 - moca
 * イベントリスナーにnullが指定された際に発生する e.listener is not function の修正
 * Version 1.0.3 - 2023/11/14 - moca
 * プラグインコマンド「S_setAnimation」を実行した際にアニメーションが再生されない問題を修正
 * Version 1.0.1 - 2023/10/31 - moca
 * ピクチャの表示がされないままコマンドを実行した際に、自動的にピクチャを表示するように
 *  SU_FadeSpine.js から以下のプラグインコマンドを移植
 *  - S_in      : フェードイン
 *  - S_showIn  : 透明状態からフェードイン
 *  - S_out     : フェードアウト
 *  - S_delete  : 削除
 *  - S_setA    : 特定の透明度に変化
 *  - S_waitOn  : フェード処理をwaitする
 *  - S_waitOff : フェード処理をwaitしない
 * 以下のプラグインコマンドを追加
 *  - S_setB          : 特定の明度に変化（内容はSU_FadeSpine.jsのS_setVと同じ動作）
 *  - S_setC          : 特定の色に変化
 *  - S_setMix        : ミックス値を設定
 *  - S_setMosaic     : モザイクを設定
 *  - S_setAnimationP : プリセットを用いてアニメーションを設定
 *  
 * Version 1.0.0 - 2023/07/04
 *  最初のバージョン
 *
 * @param Skeletons
 * @type struct<SpineData>[]
 * @default []
 * @desc ピクチャが表示されていない場合に表示するピクチャ、スケルトンについて設定します
 * @text スケルトン
 * 
 * @param FrameCount
 * @text 変化時間（フレーム数）
 * @desc 指定した値をフェードフレーム数のデフォルト値として設定します。
 * @type number
 * @default 24
 * @min 1
 */
/*~struct~Initialization:
 * @param PositionX
 * @type number
 * @default 0
 * @text 位置X
 *
 * @param PositionY
 * @type number
 * @default 0
 * @text 位置Y
 *
 * @param ScaleX
 * @type number
 * @default 100
 * @text 拡大率X
 *
 * @param ScaleY
 * @type number
 * @default 100
 * @text 拡大率Y
 */
/*~struct~Vector2:
 * @param x
 * @type number
 * @default 0
 * 
 * @param y
 * @type number
 * @default 0
 */
/*~struct~MosaicSetting:
 * @param reg
 * @type string
 * @default ""
 * @text モザイクをかけるスロットに含まれる文字列
 * @desc 「arm」と指定した場合、「left_arm」と「right_arm」、「armstrong」全てにモザイクがかかります。
 * 
 * @param size
 * @type number
 * @default 5
 * @text モザイクサイズ
 */
/*~struct~TrackAnimation:
 * @param track
 * @type number
 * @default 0
 * @desc アニメーションを表示するトラックの番号を指定します
 * @text トラック番号
 * 
 * @param animations
 * @type string[]
 * @default ""
 * @desc 再生するアニメーションを「,」区切りで指定します
 * @text アニメーション名/アニメーション名リスト
 * 
 * @param order
 * @type select
 * @default 0
 * @text 順序オプション
 * @desc 複数アニメーション指定時の表示順序を指定します
 * @option 指定した順番で表示(sequential)
 * @value sequential
 * @option 指定したアニメーションのいずれかを1つ表示(random)
 * @value random
 * @option 指定したアニメーションの順番を入れ替えて表示(shuffle)
 * @value shuffle
 * @default sequential
 * 
 * @param continuance
 * @type select
 * @default continue
 * @text 継続オプション
 * @desc 指定したアニメーションの表示が終わったあとの動作を指定します
 * @option 最後のアニメーションを繰り返し表示(continue)
 * @value continue
 * @option 最初に戻ってアニメーションを表示(reset)
 * @value reset
 * @option 最後のアニメーションが終わった状態で停止(none)
 * @value none
 * 
 * @param interrupt
 * @default true
 * @type boolean
 * @text 割り込みフラグ
 * @desc Spineが現在表示しているアニメーションの終了を待つか、中断させて表示するかを指定します
 */
/*~struct~AnimationPreset:
 * @param name
 * @type string
 * @default プリセット名
 * @text プリセット名
 * @desc プリセットを見やすくするためのもので、プラグインの動作には影響しません
 * 
 * @param TrackList
 * @type struct<TrackAnimation>[]
 * @text トラックごとのアニメーションリスト
 */
/*~struct~SpineData:
 * @param Skeleton
 * @type string
 * @text スケルトン名
 *
 * @param Skin
 * @type string[]
 * @text スキン名
 *
 * @param AnimationData
 * @type struct<AnimationPreset>[]
 * @text アニメーションプリセット
 * @desc プリセット番号で呼び出すアニメーションの内容を設定します
 * 
 * @param NumInitializeTrack
 * @type number
 * @default 10
 * @desc スケルトンをセットする際に指定個数のトラックを初期化します
 * @text トラック初期化数
 *
 * @param ResetAnimation
 * @type string
 * @default 000
 * @desc 前項の初期化に使用するアニメーション名を設定します
 * @text 初期化用アニメーション
 *
 * @param Mix
 * @type number
 * @decimals 2
 * @text 全体のミックス値
 * 
 * @param MixList
 * @type string
 * @default ""
 * @text アニメーション間のミックス値
 * @desc アニメーションの間のミックス値を指定します。例）idle,run:0.5 idle,jump:0.2
 * 
 * @param Mosaic
 * @type struct<MosaicSetting>
 * @text モザイク設定
 * 
 * @param Position
 * @type struct<Vector2>
 * @desc 表示座標
 * @text 表示座標
 * @default {"x": "0", "y": "0"}
 * 
 * @param Scale
 * @type struct<Vector2>
 * @desc 100で等倍サイズになります
 * @text 拡大率
 * @default {"x": "100", "y": "100"}
 *
 * @param InitialPicSetting
 * @type struct<Initialization>
 * @desc 予めピクチャの表示がされていない場合のピクチャ設定
 * @text ピクチャ初期化設定
 * @default {"PositionX":"0","PositionY":"0","ScaleX":"100","ScaleY":"100"}
 *
 */
/*~struct~Animation:
 * @param Track
 * @ytpe number
 *
 * @param Animation
 * @type string
 */

var SU = SU || {};
SU.FadeSpine = SU.FadeSpine || {};

(() => {
    "use strict";
    //============================================================================
    // パラメータ取得
    //============================================================================
    const parsedParams = (() => {
        var _a, _b;
        // const pluginName = document.currentScript.src.split('/').pop().replace(/\.js$/, '');
        const pluginName = (_b = (_a = document.currentScript.src
            .split("/")
            .pop()) === null || _a === void 0 ? void 0 : _a.replace(/\.js$/, "")) !== null && _b !== void 0 ? _b : "";
        const pluginParams = PluginManager.parameters(pluginName);
        return JSON.parse(JSON.stringify(pluginParams, (key, value) => {
            try {
                return JSON.parse(value);
            }
            catch (e) { }
            return value;
        }));
    })();
    const initSpineData = parsedParams["Skeletons"];
    //============================================================================
    // プラグイン本体
    //============================================================================
    ((_pluginCommand_) => {
        Game_Interpreter.prototype.pluginCommand = function (command, args) {
            _pluginCommand_.call(this, command, args);
            const [prefix, com] = command.toLowerCase().split('_');
            if (prefix !== 's') {
                return;
            }

            const convArgs = args.map((arg) => convert2Var(arg));
            const picId = Number(convArgs[0]);
            // args[1] には行番号が渡される
            const skeletonId = Number(convArgs[1]) - 1;

            // 引数なしの時picIdがnullになるので、そのときはshowPictureしない
            if (!picId) {
                switch (com) {
                    case 'waiton':
                        SU.FadeSpine.onWait = true;
                        break;
                    case 'waitoff':
                        SU.FadeSpine.onWait = false;
                        break;
                }
                return;
            }
            // showPicture(picId, initSpineData[skeletonId]);

            const frameCount = parsedParams['FrameCount'] || Game_Interpreter.prototype.fadeSpeed();

            if (com === "showin") {
                showPicture(picId, initSpineData[skeletonId], false);
                fadeSpine(picId, 1, Number(convArgs[1]) || frameCount);
            }
            else {
                showPicture(picId, initSpineData[skeletonId]);
            }

            switch (com) {
                case "setanimation":
                    setAnimationExpanded(picId, convArgs);
                    break;
                case "setanimationr":
                    setAnimationExpandedRandom(picId, convArgs);
                    break;
                case "setanimationp":
                    setAnimationExpandedPreset(picId, skeletonId, convArgs[2]);
                    break;
                case "setskin":
                    setSkinExpanded(convArgs);
                    break;
                case "setmix":
                    setMixSpine(picId, convArgs);
                    break;
                case "setmosaic":
                    setMosaicSpine(picId, convArgs.splice(0, 1));
                    break;
                // case "setmosaicd":
                //     setMosaicDefault(picId, skeletonId);
                //     break;

                // SU_FadeSpineからの命令
                case "in":
                    fadeSpine(picId, 1, Number(convArgs[1]) || frameCount);
                    break;
                case "out":
                    fadeSpine(picId, 0, Number(convArgs[1]) || frameCount);
                    break;
                case "seta":
                    fadeSpine(picId, Number(convArgs[1]), Number(convArgs[2]) || frameCount);
                    break;
                case "setb":
                    fadeColorSpine(picId, Number(convArgs[1]), Number(convArgs[1]), Number(convArgs[1]), Number(convArgs[2]) || frameCount);
                    break;
                case "setc":
                    fadeColorSpine(picId, Number(convArgs[1]), Number(convArgs[2]), Number(convArgs[3]), Number(convArgs[4]) || frameCount);
                    break;
                case "delete":
                    deleteSpine(picId, Number(convArgs[1]) || 0);
                    break;
            }
        };
    })(Game_Interpreter.prototype.pluginCommand);
    const convert2Var = (text) => {
        return text.replace(/\\v\[(\d+)\]/gi, (match, p1) => $gameVariables.value(Number(p1)).toString());
    };
    ((_init_) => {
        Game_Spine.prototype.init = function () {
            _init_.call(this);
            this.initListeners();
        };
    })(Game_Spine.prototype.init);
    const setAnimationExpanded = (picId, args) => {
        const spine = $gameScreen.spine(picId);
        let track = 0;
        let trackOptions = [];
        let animation = [];
        const currentTrackList = spine.track;

        // ループ開始前にトラック番号に紐づけた順序、継続、割り込みフラグを取得する
        for (let i = 2; i < args.length; i++) {
            let options = args[i].split(":");
            switch (options[0]) {
                case "track":
                    track = Number(options[1]);
                    trackOptions[track] = {
                        order: "sequential",
                        continuance: "continue",
                        interrupt: true
                    };
                    break;
                case "order":
                    trackOptions[track].order = options[1];
                    break;
                case "continuance":
                case "cont":
                    trackOptions[track].continuance = options[1];
                    break;
                case "interrupt":
                    trackOptions[track].interrupt = options[1] === "true" ? true : false;
                    break;
            }
        }
        track = 0;

        for (let i = 2; i < args.length; i++) {
            let options = args[i].split(":");
            switch (options[0]) {
                case "animation":
                    track = Number(options[1]);
                    animation = options[2].split(",");
                    let order = "sequential";
                    let cont = "continue";
                    let interrupt = true;
                    if (trackOptions[track]) {
                        order = trackOptions[track]["order"] || "sequential";
                        cont = trackOptions[track]["continuance"] || "continue";
                        interrupt = trackOptions[track]["interrupt"] || true;
                    }
                    spine.setAnimation(track, animation, order, cont, interrupt);
                    break;
                case "mix":
                    const value = Number(options[1]);
                    let currentAnimation = (() => {
                        if (!!currentTrackList[track]) {
                            const list = currentTrackList[track].list;
                            return list[list.length - 1].name;
                        }
                        else {
                            return "";
                        }
                    })();
                    const nextAnimation = animation[0];
                    const transition = `${currentAnimation}/${nextAnimation}`;
                    let currentMix = spine.mix[transition] || null;
                    if (currentAnimation) {
                        spine.setMix(currentAnimation, nextAnimation, value);
                        const createCallback = (spine, track, prevMix, transition) => {
                            const baseSpine = spine;
                            const targetTrack = track;
                            const callback = () => {
                                if (prevMix)
                                    baseSpine.mix[transition] = prevMix;
                                baseSpine.removeEventListener("end", callback, targetTrack);
                            };
                            return callback;
                        };
                        const callback = createCallback(spine, track, currentMix, transition);
                        spine.addEventListener("end", callback, track);
                    }
                    break;
            }
        }
    };
    const setAnimationExpandedRandom = (picId, args) => {
        const spine = $gameScreen.spine(picId);
        let track = 0;
        let animation = [];
        const currentTrackList = spine.track;
        for (let i = 0; i < args.length; i++) {
            let options = args[i].split(":");
            switch (options[0]) {
                case "animation":
                    track = Number(options[1]);
                    animation = options[2].split(",").map((a) => {
                        if (!a.match("/"))
                            return a;
                        return a.replace(/%(\d+)/, "times=$1");
                    });
                    spine.setAnimation(track, animation, "random", "reset", true);
                    break;
            }
        }
    };
    const setAnimationExpandedPreset = (picId, skeletonId, presetNum) => {
        // プリセットデータを従来のデータ形式に置換する
        let params = [picId, skeletonId + 1];
        // 各トラック毎のアニメーションとミックス値の追加
        let spineData = initSpineData[skeletonId];
        spineData.AnimationData[presetNum - 1].TrackList.forEach(data => {
            params.push(`track:${data.track}`);
            params.push(`animation:${data.track}:${data.animations}`);
            params.push(`order:${data.order}`);
            params.push(`cont:${data.continuance}`);
            params.push(`interrupt:${data.interrupt}`);
        });

        setAnimationExpanded(picId, params);
        setMixSpine(picId, ['dummy'].concat(spineData.MixList.split(' ')));
    };
    const setSkinExpanded = (args) => {
        const picId = Number(args[0]);
        // args[1] には行番号が渡される
        const spineData = initSpineData[Number(args[1]) - 1];
        if (!$gameScreen.picture(picId))
            return;
        const spine = $gameScreen.spine(picId);
        spine.setSkin(...args.slice(2));
    };
    const showSpine = (picId, spineData, visible = true) => {
        const pos = spineData["Position"];
        const scale = spineData["Scale"];
        $gameScreen.showPicture(picId, "", 0, pos.x, pos.y, scale.x, scale.y, 255, 0);
        const skeleton = spineData["Skeleton"];
        const spine = $gameScreen.spine(picId);
        if (visible !== true) {
            spine.setSkeleton(skeleton).setColor(1, 1, 1, 0);
            console.log('a');
        }
        else {
            spine.setSkeleton(skeleton);
        }
        const skin = spineData["Skin"];
        const mix = spineData["Mix"];
        if (skin)
            spine.setSkin(...skin);
        if (mix)
            spine.setMix(mix);
    };
    /**
     * Spineのミックス値を設定します
     * @param {Number} picId Spineに紐づいたピクチャ番号
     * @param {*} args 
     * @returns 
     */
    const setMixSpine = (picId, args) => {
        // setmix id value
        // setmix id anim1,anim2:value1 anim2,anim3:value2

        const spine = $gameScreen.spine(picId);
        if (args.length === 2) {
            spine.setMix(Number(args[1]));
        }
        else {
            for (let i = 1; i < args.length; i++) {
                // idle,run:0.5 idle,jump:0.2
                let options = args[i].split(":");
                if (isNaN(options[1])) {
                    console.warn(`"${options[1]}" is not a number. "${options[0]}" mix value is not set.`);
                    return;
                }

                let anims = options[0].split(',');
                if (anims.length < 2) {
                    console.error(`${options} is invalid value. example: jump,run:0.5`);
                    return;
                }
                spine.setMix(anims[0], anims[1], Number(options[1]));
            }
        }
    };

    const setMosaicSpine = (picId, args) => {
        // s_setmosaic id size
        // s_setmosaic id image size

        const spine = $gameScreen.spine(picId);

        spine.setMosaic(args);
    };

    const setMosaicDefault = (picId, skeletonId) => {
        // s_setmosaic id skeletonId

        const spine = $gameScreen.spine(picId);
        const spineData = initSpineData[skeletonId];

        spineData.MosaicList.forEach(mosaic => {
            const options = mosaic.split(' ');
            if (options.length < 1) {
                return;
            }
            spine.setMosaic(options[0], options[1]);
        });
    };
    /**
     * Spineの透明度を徐々に変化させます
     * @param {Number} picId Spineに紐づいたピクチャ番号
     * @param {Number} alpha 目標透明度
     * @param {Number} frame 変化にかかるフレーム数
     */
    const fadeSpine = (picId, alpha, frame) => {
        SU.FadeSpine.executions[picId] = SU.FadeSpine.setAlphaFromId(picId, alpha, frame);
        if (SU.FadeSpine.onWait) this.wait(frame);
    };

    /**
     * Spineをフェードアウトさせながら削除します
     * @param {Number} picId Spineに紐づいたピクチャ番号
     * @param {Number} frame 削除にかかるフレーム数
     */
    const deleteSpine = (picId, frame) => {
        if (frame <= 0) {
            $gameScreen.erasePicture(picId);
        }
        else {
            SU.FadeSpine.executions[picId] = SU.FadeSpine.setAlphaFromId(picId, 0, frame,
                () => $gameScreen.erasePicture(picId));
            if (SU.FadeSpine.onWait) this.wait(frame);
        }
    };

    /**
     * Spineの色を徐々に変化させます
     * @param {Number} picId Spineに紐づいたピクチャ番号
     * @param {Number} r 赤(0~1)
     * @param {Number} g 緑(0~1)
     * @param {Number} b 青(0~1)
     * @param {Number} frame 変化にかかるフレーム数
     */
    const fadeColorSpine = (picId, r, g, b, frame) => {
        const targetColor = new Array(r, g, b);
        SU.FadeSpine.executions[picId] = SU.FadeSpine.setColorFromId(picId, targetColor, frame);
        if (SU.FadeSpine.onWait) this.wait(frame);
    };

    /**
     * ピクチャが表示されていなければ表示する
     * @param {Number} picId 
     * @param {*} spineData 
     */
    const showPicture = (picId, spineData, visible = true) => {
        // ピクチャが表示されていなければ用意する
        if (!$gameScreen.picture(picId)) {
            if (spineData)
                showSpine(picId, spineData, visible);
            else {
                return;
            }
        }
    };

    //============================================================================
    // トラックを指定数初期化
    //============================================================================
    Game_Spine.prototype.initTracks = function () {
        const refData = initSpineData
            .filter((params) => params["Skeleton"] === this.skeleton.split("_")[0])
            .shift();
        if (!refData)
            return;
        const numInitializedTrack = refData["NumInitializeTrack"] || 0;
        this.resetAnimation = refData["ResetAnimation"];
        for (let i = 0; i < numInitializedTrack; i++) {
            this.setAnimation(i, this.resetAnimation, "none");
        }
    };
    Game_Spine.prototype.setDefaultMosaic = function () {
        const refData = initSpineData
            .filter((params) => params["Skeleton"] === this.skeleton.split("_")[0])
            .shift();
        if (!refData)
            return;
        const mosaicRegText = refData["Mosaic"]["reg"] || null;
        if (mosaicRegText === null) {
            return;
        }
        const mosaicRegExp = new RegExp(`${mosaicRegText}`);
        const mosaicSize = refData["Mosaic"]["size"] || 5;
        const data = Game_Spine.spineData()[refData.Skeleton];
        // 指定文字列が含まれたスロット全てにモザイクをかける
        data.slots.filter(slot => slot.name.match(mosaicRegExp)).forEach(slot => {
            this.setMosaic(slot.name, mosaicSize);
        });
    };
    // overwrite
    ((_setSkeleton_) => {
        Game_Spine.prototype.setSkeleton = function (name = "") {
            _setSkeleton_.call(this, name);
            // this.initTracks();
            this.setDefaultMosaic();
            return this;
        };
    })(Game_Spine.prototype.setSkeleton);
    //============================================================================
    // イベントを登録できるようにする
    //============================================================================
    Object.defineProperty(Game_Spine, "listeners", {
        get() {
            return [this._globalListener, ...this._trackListeners];
        },
    });
    Game_Spine.prototype.initListeners = function () {
        this._globalListener = {
            start: [],
            interrupt: [],
            dispose: [],
            end: [],
            complete: [],
            event: [],
        };
        this._trackListeners = [];
    };
    /**
     * アニメーション中の各イベント(Spine で設定するイベントとは異なる）が起こった時に呼び出される関数を設定します
     * @param {string} eventName イベント名
     * @param {Function} listener 呼び出される関数
     * @param {number} trackIndex 対象のトラック番号、指定しなければ全てのトラックを対象にする
     * @param {object} options 追加動作を指定する
     */
    Game_Spine.prototype.addEventListener = function (eventName, listener, trackIndex = -1, options = { listener, once: false }) {
        if (trackIndex >= 0) {
            if (!this._trackListeners[trackIndex])
                this._trackListeners[trackIndex] = {
                    start: [],
                    interrupt: [],
                    dispose: [],
                    end: [],
                    complete: [],
                    event: [],
                };
            const track = this._trackListeners[trackIndex];
            track[eventName].push(Object.assign(options, { listener: listener }));
        }
        else
            this._globalListener[eventName].push(Object.assign(options, { listener: listener }));
    };
    /**
     * 登録したイベントを削除します
     * @param {string} eventName イベント名
     * @param {Function} listener 登録した関数
     * @param {number} trackIndex 対象のトラック番号、指定しなければ全てのトラックを対象にする
     */
    Game_Spine.prototype.removeEventListener = function (eventName, listener, trackIndex) {
        const listeners = trackIndex >= 0
            ? this._trackListeners[trackIndex][eventName]
            : this._globalListener[eventName];
        const index = listeners.map((e) => e.listener).indexOf(listener);
        if (index !== -1) {
            listeners.splice(index, 1);
        }
    };
    /**
     * イベントに登録された関数を呼び出します
     * @param {string} eventName イベント名
     * @param {number} trackIndex トラック番号
     */
    Game_Spine.prototype.dispatchEventListener = function (eventName, entry) {
        const trackIndex = entry.trackIndex;
        const globalCallbacks = this._globalListener[eventName];
        if (globalCallbacks)
            globalCallbacks.forEach((e) => {
                if (!e.listener) {
                    this.removeEventListener(eventName, e.listener, trackIndex);
                }
                else {
                    e.listener(entry);
                }
                if (e.once)
                    this.removeEventListener(eventName, e.listener, trackIndex);
            });
        if (trackIndex >= 0 && this._trackListeners[trackIndex]) {
            const trackCallbacks = this._trackListeners[trackIndex][eventName];
            if (trackCallbacks) {
                trackCallbacks.forEach((e) => {
                    if (!e.listener) {
                        this.removeEventListener(eventName, e.listener, trackIndex);
                    }
                    else {
                        e.listener(entry);
                    }
                    if (e.once)
                        this.removeEventListener(eventName, e.listener, trackIndex);
                });
            }
        }
    };
    // overwrite
    // end と dispose のコメントアウトを外しただけです
    Sprite_Spine.prototype.updateSkeleton = function (spine) {
        if (spine) {
            if (spine.skeleton != this._skeleton) {
                this.init();
                let skeleton = spine.skeleton.replace(/_\d+$/, "");
                if (skeleton) {
                    let fullName = Game_Spine.fullName(skeleton);
                    let data = Game_Spine.spineData()[fullName];
                    if (data) {
                        if ("animations" in data) {
                            this._data = new PIXI.spine.Spine(data);
                            this._data.destroy = function () { };
                            this._data.state.addListener({
                                start: this.onStart.bind(this),
                                interrupt: this.onInterrupt.bind(this),
                                end: this.onEnd.bind(this),
                                dispose: this.onDispose.bind(this),
                                complete: this.onComplete.bind(this),
                                event: this.onEvent.bind(this),
                            });
                            this.addChild(this._data);
                            if (spine.playData.length > 0) {
                                this._isRestore = true;
                            }
                            this.registerCallback(spine);
                            // this.createNames();
                        }
                        else {
                            Game_Spine.loadSkeleton(skeleton);
                        }
                    }
                    else if (fullName in Game_Spine.spineData() == false) {
                        throw Error(`'${skeleton}' is unknown model.`);
                    }
                }
                if (!skeleton || this._data) {
                    this._skeleton = spine.skeleton;
                }
            }
        }
        else if (this._skeleton) {
            this.init();
        }
    };
    ((_onStart_) => {
        Sprite_Spine.prototype.onStart = function (entry) {
            this.spine().dispatchEventListener("start", entry);
            _onStart_.call(this, entry);
        };
    })(Sprite_Spine.prototype.onStart);
    ((_onInterrupt_) => {
        Sprite_Spine.prototype.onInterrupt = function (entry) {
            this.spine().dispatchEventListener("interrupt", entry);
            _onInterrupt_.call(this, entry);
        };
    })(Sprite_Spine.prototype.onInterrupt);
    ((_onEnd_) => {
        Sprite_Spine.prototype.onEnd = function (entry) {
            this.spine().dispatchEventListener("end", entry);
            _onEnd_.call(this, entry);
        };
    })(Sprite_Spine.prototype.onEnd);
    ((_onDispose_) => {
        Sprite_Spine.prototype.onDispose = function (entry) {
            this.spine().dispatchEventListener("dispose", entry);
            _onDispose_.call(this, entry);
        };
    })(Sprite_Spine.prototype.onDispose);
    ((_onComplete_) => {
        Sprite_Spine.prototype.onComplete = function (entry) {
            this.spine().dispatchEventListener("complete", entry);
            _onComplete_.call(this, entry);
        };
    })(Sprite_Spine.prototype.onComplete);
    ((_onEvent_) => {
        Sprite_Spine.prototype.onEvent = function (entry, event) {
            this.spine().dispatchEventListener("event", entry);
            _onEvent_.call(this, entry, event);
        };
    })(Sprite_Spine.prototype.onEvent);


    // 以下 SU_FadeSpine.jsより引用
    // SU.FadeSpine.FSFrameCount = parsedParams['FrameCount'] || Game_Interpreter.prototype.fadeSpeed();
    SU.FadeSpine.executions = {};
    SU.FadeSpine.onWait = false;
    //=============================================================================
    // Update
    //=============================================================================
    (_updateMain_ => {
        Scene_Map.prototype.updateMain = function () {
            _updateMain_.call(this);

            SU.FadeSpine.update();
        }
    })(Scene_Map.prototype.updateMain);

    (_update_ => {
        Scene_Battle.prototype.update = function () {
            _update_.call(this);

            SU.FadeSpine.update();
        }
    })(Scene_Battle.prototype.update);

    SU.FadeSpine.update = function () {
        let executions = this.executions;
        if (Object.keys(executions).length == 0) return;

        for (const key in executions) {
            let e = executions[key];
            e.update();
            if (e.duration() <= 0) {
                const callback = e.callback;
                if (callback || typeof callback === 'function') callback();
                delete executions[key];
            }
        }
    };

    SU.FadeSpine.getColor = (spine) => {
        let col = spine.color['/default/'];

        if (!col) return [1, 1, 1, 1];

        return [...col];
    };

    //=============================================================================
    // Fade
    //=============================================================================

    SU.FadeSpine.setColor = function (spine, argColor, frame, callback) {
        let duration = (frame > 0) ? frame : this.FSFrameCount;

        const getColor = () => this.getColor(spine);

        const currentColor = getColor();
        const targetColor = argColor;

        // アルファ値の指定が無ければ現在のアルファ値で補間する
        if (targetColor[3] == undefined) {
            targetColor[3] = currentColor[3];
        }

        // 変化させる必要のある色インデックスを抽出
        let difIndex = targetColor
            .map((color, i) => (color != currentColor[i]) ? i : -1)
            .filter(index => index != -1);

        function fadeUpdate() {
            if (duration <= 0) return;

            let c = getColor();

            for (const i of difIndex) {
                c[i] = (c[i] * (duration - 1) + targetColor[i]) / duration;
            }

            spine.setColor(...c);

            duration--;
        }

        return {
            update: () => fadeUpdate(),
            duration: () => duration,
            callback: callback,
        }

    };

    SU.FadeSpine.setColorFromId = function (id, argColor, frame, callback) {
        const spine = $gameScreen.spine(id)
        return this.setColor(spine, argColor, frame, callback);
    };

    SU.FadeSpine.setAlpha = function (spine, alpha, frame, callback) {
        let targetColor = this.getColor(spine);
        // アルファ値を設定
        targetColor[3] = alpha

        return this.setColor(spine, targetColor, frame, callback);
    };

    SU.FadeSpine.setAlphaFromId = function (id, alpha, frame, callback) {
        let spine = $gameScreen.spine(id)
        return this.setAlpha(spine, alpha, frame, callback);
    };
})();
