//=============================================================================
// StandPictureManager.js
//=============================================================================

/*:
 * @plugindesc 改造立ち絵マネージャー（ver1.01a）
 * @author Leeroynto（http://leeroynto.livedoor.blog/）
 *
 * @param UseNamePlate
 * @desc ネームプレートを使用するかどうか（true or false）
 * true の場合は下のプラグイン選択も行ってください
 * @type Boolean
 * @default false
 *
 * @param NamePlatePlugin
 * @text 名前表示プラグイン選択
 * @desc どのネームプレート表示プラグインを使用するかを指定します(UseNamePlateがTrueの場合に有効)
 * @type select
 * @option YEP_MessageCore
 * @value YEP
 * @option MPP_MessageEX_Op1
 * @value MPP
 * @default YEP
 * @parent UseNamePlate
 * 
 * @param UseSpine
 * @desc Spineを使用するかどうか指定します
 * (Trueの場合でもSpine以外のピクチャに対しても効くようにしています)
 * @text Spine を使用する
 * @type Boolean
 * @default false
 *
 * @param StandPictureBrightnessChangeWait
 * @desc 立ち絵を明るくor暗くするのにかかるフレーム（wait）数
 * (強調方法が明暗以外の場合でもこの値を適用します)
 * @default 5
 * @text 要フレーム数
 *
 * @param DisplayNamePlate
 * @desc ネームプレートを表示するかどうか（true or false）
 * ネームプレートを使用する場合に設定してください
 * @default true
 * @text 名札を表示する
 *
 * @param InitialLetter
 * @desc 名前につけてる頭文字 (ネームプレートを使用しない場合)
 * 例：【キャラクター名】の場合、【 が頭文字
 * @default 
 * @param LastLetter
 * @desc 名前につけてるお尻文字 (ネームプレートを使用しない場合)
 * 例：【キャラクター名】の場合、】がお尻文字
 * @default 
 *
 * @param SpotType
 * @text 強調方法
 * @desc 話者を強調する方法を指定します。
 * @type select
 * @option 明暗
 * @value brightness
 * @option 拡大
 * @value expansion
 * @default brightness
 *
 * @param BrightnessRate
 * @text 暗くしたときの明度
 * @desc 非話者を暗くした時の明度を指定します。
 * @type Number
 * @default 0.5
 * @min 0
 * @max 1
 * @decimals 2
 * @parent SpotType
 *
 * @param ExpansionRate
 * @text 拡大時スケール
 * @desc 話者の表示スケールを指定します。
 * @type Number
 * @default 1.1
 * @min 0
 * @decimals 2
 * @parent SpotType
 *
 * @help　
 * ============================================================================
 * 改造者追記
 * ============================================================================
 * * ゲーム公開時にプラグイン作成者様の著作権表記が必要です。
 * 
 * ・Spineを使用した立ち絵表示に対応
 * ・MPP_MessageEX_Op1.js による名前表示に合わせた名前取得に対応
 *   合わせて使用プラグインを選択式にしています。
 * ・ネームプレートプラグインを利用する使用する場合は、そのプラグインよりも下に配置してください。
 * ・話者強調の方法を追加しています。SpotTypeで選択してください。
 * 拡大はUseSpineに関わらずSpineピクチャに対しても機能します。
 * ・立ち絵の数に制限はありませんが、複数人が同時に発言している場面には現在対応していません。
 * 
 * ** 使い方
 * 作成者様のホームページでサンプルプロジェクトが配布されているので、そちらもご参照ください。
 * 
 * 1. プラグインコマンド SartStandPIcture を実行すると、その時点から会話の中で話者の強調表示が適用されるようになります。
 * 
 * 2. StandPictureManager コマンドによりピクチャ番号と名前が紐づけされます。この情報は EndStandPicture を実行するまで保持されます。
 *    ピクチャの表示はコマンドの実行前でも実行後でも問題ありません。
 * 
 * 3. 以降はテキスト中に以下の方法で埋め込まれた名前情報から話者を取得します。
 *    ・デフォルトの状態ではテキストの頭から改行までのテキストを名前として取得します。
 *      InitialLetter と LastLetter 指定することで名前部分を指定することができます。この場合でも最初の改行までが対象となります。
 *    ・対応する名前表示プラグインを使用する場合はそのプラグインで名前を指定する方法で同様に取得できますので、書き方を変える必要はありません。
 *    どちらの方法で取得するかはパラメータで指定してください。
 * 
 * 4. 会話を終了する際には EndStandPicture を実行してください。
 *    （使用する画像とピクチャ番号の割り振りが全て決まっている場合は必要ないかもしれませんが、
 *    予期せぬ箇所で機能しないとも限らないので、会話の度に StartStandPicture 及び EndStandPicture のどちらも実行することを推奨します。）
 *
 * ============================================================================
 * 作成者様注意書き
 * ============================================================================
 * 
 * ＊必ずYEP_MessageCore.jsより下に入れてください
 *
 *
 * ============================================================================
 * プラグインコマンド
 * ============================================================================
 *   StartStandPicture
 *     立ち絵による会話を開始する時に使用してください
 *   EndStandPicture
 *     立ち絵による会話を終了する時に使用してください
 *   StandPictureManager xx yy
 *     xxにピクチャー番号、yyにキャラクター名を入れてください
 *     yyを半角数値で指定すると、そのIDに対応するアクター名に置き換えられます
 *   ResetStandPictures
 *     登録した立ち絵の情報をリセットします
 *
 * ============================================================================
 * 更新履歴
 * ============================================================================
 *
 * 2018/09/01 ver1.01
 * ・戦闘のテストプレイでエラー落ちするバグを修正
 * ・テキストが一行の場合、立ち絵の明るさ操作に不具合がでるバグを修正
 *
 * 2018/08/14 ver1.00
 * ・公開
 *
 */


var $currentSpeakingActorName = null;
var $pastSpeakingActorName = null;
var $currentPictures = {};
var $standPictureSpeaking = false;


(function () {
	let getBoolParam = (str) => parameters[str] == 'true';
	let getNumParam = (str) => Number(parameters[str]);

	const pluginName = document.currentScript.src.split('/').pop().replace(/\.js$/, '');
	var parameters = PluginManager.parameters(pluginName);
	var useNamePlate = getBoolParam('UseNamePlate');
	var displayNamePlate = getBoolParam('DisplayNamePlate');
	var standPictureBrightnessChangeWait = getNumParam('StandPictureBrightnessChangeWait');
	var initialLetterStandPicture = parameters['InitialLetter'];
	var lastLetterStandPicture = parameters['LastLetter'];

	let usedPlugin = parameters['NamePlatePlugin'];
	let useSpine = getBoolParam('UseSpine');
	let spotType = parameters['SpotType'];
	let brightnessOnNotFocus = getNumParam('BrightnessRate');
	let expansionRate = getNumParam('ExpansionRate');

	let duration = 0;

	const pluginList = {
		YEP: 'YEP',
		MPP: 'MPP',
	}
	Object.freeze(pluginList);

	(_initialize_ => {
		Game_System.prototype.initialize = function () {
			_initialize_.call(this);
		}
	})(Game_System.prototype.initialize);

	(_extractSaveContents_ => {
		DataManager.extractSaveContents = function (contents) {
			_extractSaveContents_.call(this, contents);

			$currentPictures = $gameSystem.currentPictures;
			console.log($currentPictures);
		}
	})(DataManager.extractSaveContents);

	var _Scene_Map_prototype_initialize = Scene_Map.prototype.initialize;
	Scene_Map.prototype.initialize = function () {

		_Scene_Map_prototype_initialize.call(this);

		this._isTalking = false;

		//現在話している人の名前
		if ($currentSpeakingActorName == null) { $currentSpeakingActorName = ""; }
		if ($pastSpeakingActorName == null) { $pastSpeakingActorName = ""; }
		//key：ピクチャー番号、value：キャラ名
		// if ($currentPictures == null) {
		// 	$currentPictures = {};
		// }

		// if ($gameSystem.currentPictures) {
		// 	// $currentPictures = $gameSystem.currentPictures;
		// }
		$gameSystem.currentPictures = $currentPictures;
		if ($gameSystem.standPictureSpeaking) {
			$standPictureSpeaking = $gameSystem.standPictureSpeaking;
		}
	};

	(_clear_ => {
		Game_Interpreter.prototype.clear = function () {
			_clear_.call(this);
		}
	})(Game_Interpreter.prototype.clear);

	Game_Interpreter.prototype.startStandPicture = function () {
		$standPictureSpeaking = true;
		$gameSystem.standPictureSpeaking = $standPictureSpeaking;
		$pastSpeakingActorName = '';
	};

	Game_Interpreter.prototype.endStandPicture = function () {
		$standPictureSpeaking = false;
		$gameSystem.standPictureSpeaking = $standPictureSpeaking;
		if ($currentPictures == null) return;
		for (_key in $currentPictures) {
			$gameScreen.erasePicture(Number(_key));
		}
	};

	Game_Interpreter.prototype.setCurrentSpeakingActorNameFromPictureId = function (_picId) {
		if ($currentPictures[_picId]) {
			$currentSpeakingActorName = $currentPictures[_picId].name;
		}
	};

	Game_Interpreter.prototype.standPictureColorChange = function () {
		if ($currentPictures == null) return;
		if ($pastSpeakingActorName == $currentSpeakingActorName) return;

		for (_key in $currentPictures) {
			let picture = $currentPictures[_key];

			if (!picture.exists()) continue;

			switch (spotType) {
				case 'brightness':
					//su Spine用に改変
					if (useSpine && picture.hasSpine()) {
						picture.toneTarget = (picture.name == $currentSpeakingActorName) ?
							[1, 1, 1, 1]
							: [brightnessOnNotFocus, brightnessOnNotFocus, brightnessOnNotFocus, 1];
					} else {
						if (picture.name == $currentSpeakingActorName) {
							$gameScreen.tintPicture(Number(_key), [0, 0, 0, 0], standPictureBrightnessChangeWait);
						} else {
							let reducedBrightness = (255 - 255 * brightnessOnNotFocus) * -1;
							$gameScreen.tintPicture(Number(_key), [reducedBrightness, reducedBrightness, reducedBrightness, 0], standPictureBrightnessChangeWait);
						}
					}
					break;
				case 'expansion':
					let pic = picture.picture();
					let originalScale = picture.originalScale;

					// if (!originalScale.x && !originalScale.y) {
					// picture.originalScale.x = pic.scaleX();
					// picture.originalScale.y = pic.scaleY();
					if (!originalScale) {
						originalScale = picture.originalScale = { x: pic.scaleX(), y: pic.scaleY() };
					}
					if (picture.name == $currentSpeakingActorName) {
						setScale(pic, originalScale.x * expansionRate, originalScale.y * expansionRate,);
					} else {
						setScale(pic, originalScale.x, originalScale.y);
					}
					break;
			}

		}
		duration = standPictureBrightnessChangeWait;
		$pastSpeakingActorName = $currentSpeakingActorName;
	};

	// メッセージウィンドウの表示を会話イベントの開始とする
	(_command101_ => {
		Game_Interpreter.prototype.command101 = function () {
			if (!$gameMessage.isBusy()) this._isTalking = true;
			_command101_.call(this);
		}
	})(Game_Interpreter.prototype.command101);

	// イベントの終了時に表示をリセット
	(_terminate_ => {
		Game_Interpreter.prototype.terminate = function () {
			_terminate_.call(this);


			// 並列処理のイベントを除外
			if (!this._isTalking) {
				this._isTalking = false;
				return;
			}
			console.log('terminate');

			for (const _key in $currentPictures) {
				const data = $currentPictures[_key];
				if (!data.picture()) continue;

				if (spotType == 'brightness') {
					if (useSpine && data.hasSpine()) {
						data.toneTarget = [1, 1, 1, 1];
					} else {
						$gameScreen.tintPicture(Number(_key), [0, 0, 0, 0], standPictureBrightnessChangeWait);
					}
				}
				else if (spotType == 'expansion') {
					if (!data.originalScale) continue;

					setScale(data.picture(), data.originalScale.x, data.originalScale.y);
				}
			}
			// 会話のデータをリセット
			$currentSpeakingActorName = '';
			$pastSpeakingActorName = '';
		}
	})(Game_Interpreter.prototype.terminate);

	const setScale = (picture, x, y) => {
		picture.move(
			picture.origin(),
			picture.x(), picture.y(),
			x, y,
			picture.opacity(),
			picture.blendMode(),
			10);
	};

	//su 表示されているSpine画像を指定フレーム数かけて変化させる
	const updateColor = () => {
		if (duration > 0) {
			let d = duration;
			for (_key in $currentPictures) {
				let key = Number(_key);
				let picture = $currentPictures[key];
				if (!picture.exists() || !picture.hasSpine()) continue;

				let imColor = $gameScreen.spine(key).color;
				//su PictureSpineで定義されている、全体を選択する場合のキー
				let defaultKey = '/default/';
				//su 色調変化をしていない場合はcolorのプロパティが消される謎使用のため、プロパティの存在判定が必要
				let c = (defaultKey in imColor)
					? imColor[defaultKey]
					: [1, 1, 1, 1];
				let targetColor = picture.toneTarget;

				for (var i = 0; i < 4; i++) {
					c[i] = (c[i] * (d - 1) + targetColor[i]) / d;
				}
				$gameScreen.spine(key).setColor(...c);
			}
			duration--;
		}
	};

	var _Scene_Map_Prototype_Update = Scene_Map.prototype.update;
	Scene_Map.prototype.update = function () {

		_Scene_Map_Prototype_Update.call(this);

		if ($standPictureSpeaking) {
			$gameMap._interpreter.standPictureColorChange();
		}

		//su Spineスプラインの色調をアップデートする
		if (useSpine && spotType == 'brightness') updateColor();
	};

	var _Scene_Battle_Prototype_Update = Scene_Battle.prototype.update;
	Scene_Battle.prototype.update = function () {
		_Scene_Battle_Prototype_Update.call(this);

		if ($standPictureSpeaking) {
			$gameTroop._interpreter.standPictureColorChange();
		}
		if (useSpine && spotType == 'brightness') updateColor();
	};


	if (useNamePlate) {
		//ネームプレートを使う場合の名前情報取得
		//YEP_MessageCore.jsの上書き
		switch (usedPlugin) {
			case pluginList.YEP:
				console.log("get name from YEP_MessageCore")

				Window_Message.prototype.convertNameBox = function (text) {
					$currentSpeakingActorName = "";
					if (!displayNamePlate) this._nameWindow.hide();
					text = text.replace(/\x1bN\<(.*)\>/gi, function () {
						//数値だけの場合、それをアクターのIDだと判断して対応するIDのネームを入れる
						if (text.search(/\x1bN\<([0-9]\d*|0)\>/gi) == 0) {
							if (useNamePlate) $currentSpeakingActorName = $gameActors.actor(arguments[1]).name();
							return Yanfly.nameWindow.refresh($gameActors.actor(arguments[1]).name(), 1);
						} else {
							if (useNamePlate) $currentSpeakingActorName = arguments[1];
							return Yanfly.nameWindow.refresh(arguments[1], 1);
						}
					}, this);
					text = text.replace(/\x1bN1\<(.*)\>/gi, function () {
						return Yanfly.nameWindow.refresh(arguments[1], 1);
					}, this);
					text = text.replace(/\x1bN2\<(.*)\>/gi, function () {
						return Yanfly.nameWindow.refresh(arguments[1], 2);
					}, this);
					text = text.replace(/\x1bN3\<(.*)\>/gi, function () {
						return Yanfly.nameWindow.refresh(arguments[1], 3);
					}, this);
					text = text.replace(/\x1bNC\<(.*)\>/gi, function () {
						if (useNamePlate) $currentSpeakingActorName = arguments[1];
						return Yanfly.nameWindow.refresh(arguments[1], 3);
					}, this);
					text = text.replace(/\x1bN4\<(.*)\>/gi, function () {
						return Yanfly.nameWindow.refresh(arguments[1], 4);
					}, this);
					text = text.replace(/\x1bN5\<(.*)\>/gi, function () {
						return Yanfly.nameWindow.refresh(arguments[1], 5);
					}, this);
					text = text.replace(/\x1bNR\<(.*)\>/gi, function () {
						if (useNamePlate) $currentSpeakingActorName = arguments[1];
						return Yanfly.nameWindow.refresh(arguments[1], 5);
					}, this);
					return text;
				};

				break;
			case pluginList.MPP:
				console.log("get name from MPP_MessageEX_Op1")

				const _Window_Message_convertEscapeCharacters = Window_Message.prototype.convertEscapeCharacters;
				Window_Message.prototype.convertEscapeCharacters = function (text) {
					text = _Window_Message_convertEscapeCharacters.call(this, text);

					$currentSpeakingActorName = this._speakerName;

					return text;
				};

				break;
		}
	} else {
		//ネームプレートを使わない場合の名前情報取得
		if (!useNamePlate) {
			console.log("get name from head of message")

			var _Window_Message_prototype_startMessage = Window_Message.prototype.startMessage;

			Window_Message.prototype.startMessage = function () {
				_Window_Message_prototype_startMessage.call(this);

				if ($standPictureSpeaking) {
					this._nameCharacterPreserve = ""; var _countStart = false;
					if (initialLetterStandPicture == "") _countStart = true;
					if (this._textState != null) {
						while (!this.isEndOfText(this._textState)) {
							if (this._textState.text[this._textState.index] == '\n' || (lastLetterStandPicture != "" && this._textState.text[this._textState.index] == lastLetterStandPicture)) {
								//						$currectSpeakingActorName = this._nameCharacterPreserve;
								break;
							} else {
								if (this._textState.text[this._textState.index] != '\x1b') {
									if (_countStart) {
										this._nameCharacterPreserve += this._textState.text[this._textState.index];
									} else {
										if (initialLetterStandPicture != "" && this._textState.text[this._textState.index] == initialLetterStandPicture) _countStart = true;
									}
									this._textState.index++;
								} else {
									this.processEscapeCharacter(this.obtainEscapeCode(this._textState), this._textState);
								}
							}
						}
						$currentSpeakingActorName = this._nameCharacterPreserve;
						this._textState.index = 0;
					}
				}
			};
		}
	};

	//プラグインコマンド
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function (command, args) {
		_Game_Interpreter_pluginCommand.call(this, command, args);

		if ((command || '').toUpperCase() === 'STANDPICTUREMANAGER') {
			let picId = args[0];
			$currentPictures[picId] = (() => {
				let id = picId;
				let name = args[1];

				//名前が数値だけの場合はそれをIDにアクターデータから名前を取得
				if (name.search(/^[0-9]+$/) == 0) {
					name = $gameActors.actor(name.replace(/[^0-9]/g, "")).name();
				}

				return new StandingPicture(name, id);
			})();
		}
		if ((command || '').toUpperCase() === 'RESETSTANDPICTURE') {
			$currentSpeakingActorName = "";
			$pastSpeakingActorName = "";
			$currentPictures = {};
		}
		if ((command || '').toUpperCase() === 'STARTSTANDPICTURE') {
			this.startStandPicture();
		}
		if ((command || '').toUpperCase() === 'ENDSTANDPICTURE') {
			this.endStandPicture();
		}
	};


	// ピクチャと名前の結びつきを保持するobject
	function StandingPicture() {
		this.initialize.apply(this, arguments);
	};

	StandingPicture.prototype.initialize = function (name, id, originalScale) {
		this._name = name;
		this._id = Number(id);
		// this._originalScale = originalScale || { x: 0, y: 0 };
	};

	Object.defineProperty(StandingPicture.prototype, 'name', {
		get: function () {
			return this._name;
		},
		configurable: true
	});
	Object.defineProperty(StandingPicture.prototype, 'id', {
		get: function () {
			return this._id;
		},
		configurable: true
	});
	// Object.defineProperty(StandingPicture.prototype, 'originalScale', {
	// 	get: function () {
	// 		return this._originalScale;
	// 	},
	// 	configurable: true
	// });
	StandingPicture.prototype.picture = function () { return $gameScreen.picture(this._id); };

	StandingPicture.prototype.exists = function () {
		const picture = this.picture();
		return !!picture;
	};

	StandingPicture.prototype.hasSpine = function () {
		return ($gameScreen.spine(this._id) != undefined)
			? $gameScreen.spine(this._id).skeleton != ''
			: false;
	};

	window.StandingPicture = StandingPicture;
})();
