//=============================================================================
// ** RPG Maker MZ - Hakubox_CG_Effect.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 白箱雾气特效插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MZ
 * 
 * @help
 * 
 * 白箱雾气特效插件可以让你在游戏中循环播放雾气特效，需要配置对应特效列表。
 * 
 * 调用代码：
 * effectModule.playEffect("特效别名");
 * effectModule.playEffect("特效别名", { x: 0, y: 0 });
 * effectModule.stopEffect();
 * 
 * 
 * @command playEffect
 * @text 播放特效
 * @desc 播放特效。
 * 
 * @arg name
 * @text 特效名称
 * @desc 特效名称
 * @type text
 * 
 * @arg x
 * @parent config
 * @text X坐标
 * @desc X坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * @arg y
 * @parent config
 * @text Y坐标
 * @desc Y坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * 
 * @command stopEffect
 * @text 停止特效
 * @desc 停止特效。
 * 
 * @arg name
 * @text 特效名称
 * @desc 特效名称
 * @type text
 * 
 * 
 * 
 * @param effectList
 * @text 特效列表
 * @type struct<effectInfo>[]
 * @desc 特效列表，可以设置多个特效，每个特效可以设置多个属性。
 * @default []
 * 
 * 
 */
/*~struct~effectInfo:
 * 
 * @param path
 * @text 特效图片路径
 * @type file[]
 * @desc 输入特效图片的路径，用于在游戏中显示。游戏中会随机显示其中一个。
 * @dir img/pictures
 * @default []
 * 
 * @param alias
 * @text 别名
 * @desc 当前特效的别名，可以当路径使用。
 * @type text
 * 
 * @param visibleCondition
 * @text 显示条件
 * @desc 显示条件，内容为代码块，返回boolean值，true显示，false隐藏。
 * @type text
 * 
 * @param desc
 * @text 备注
 * @desc 当前特效的备注。
 * @type note
 * 
 * @param config
 * @text 配置
 * 
 * @param effectType
 * @text 特效类型
 * @desc 特效类型
 * @type select
 * @option 静态 - static
 * @value static
 * @option 雾气 - fog
 * @value fog
 * @default fog
 * 
 * @param loop
 * @parent config
 * @text 是否循环播放
 * @type boolean
 * @desc 是否循环播放特效，不循环则播放完成后自动销毁。
 * @on 循环
 * @off 不循环
 * @default true
 * 
 * @param x
 * @parent config
 * @text X坐标
 * @desc X坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * @param y
 * @parent config
 * @text Y坐标
 * @desc Y坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * @param scale
 * @parent config
 * @text 缩放
 * @desc 缩放，默认为1.0
 * @type text
 * @default 1.0
 * 
 * @param alpha
 * @parent config
 * @text 不透明度
 * @desc 不透明度，默认为1，最小为0则完全透明。
 * @type text
 * @default 1
 * 
 * @param blendMode
 * @parent config
 * @text 混合模式
 * @type select
 * @option NORMAL - 默认,0
 * @value NORMAL
 * @option ADD - 线性减淡 (添加),1
 * @value ADD
 * @option MULTIPLY - 正片叠底,2
 * @value MULTIPLY
 * @option SCREEN - 滤色,3
 * @value SCREEN
 * @option OVERLAY - 叠加,4
 * @value OVERLAY
 * @option DARKEN - 变暗,5
 * @value DARKEN
 * @option LIGHTEN - 变亮,6
 * @value LIGHTEN
 * @option COLOR_DODGE - 颜色减淡,7
 * @value COLOR_DODGE
 * @option COLOR_BURN - 颜色加深,8
 * @value COLOR_BURN
 * @option HARD_LIGHT - 强光,9
 * @value HARD_LIGHT
 * @option SOFT_LIGHT - 柔光,10
 * @value SOFT_LIGHT
 * @option DIFFERENCE - 差值,11
 * @value DIFFERENCE
 * @option EXCLUSION - 排除,12
 * @value EXCLUSION
 * @option HUE - 色相,13
 * @value HUE
 * @option SATURATION - 饱和度,14
 * @value SATURATION
 * @option COLOR - 颜色,15
 * @value COLOR
 * @option LUMINOSITY - 明度,16
 * @value LUMINOSITY
 * @default NORMAL
 * 
 * @param floor
 * @text 插入层级
 * @desc 特效精灵的插入层级，默认top为最上层，normal在地图上则会插入到对话框下方。
 * @type select
 * @option top - 最上层
 * @value top
 * @option normal - 普通
 * @value normal
 * @option custom - 自定义
 * @value custom
 * @default top
 * 
 * @param customFloor
 * @parent floor
 * @text 自定义层级
 * @desc 自定义层级，如果floor选择custom，则使用此层级，在层级中可编写代码。
 * @type text
 * 
 * @param fogEffect
 * @text ———— 雾气特效配置 ————
 * @default 特效类型选择雾气后生效。
 * 
 * @param fogDuration
 * @parent fogEffect
 * @text 雾气帧数
 * @desc 雾气帧数
 * @type number
 * @default 100
 * 
 * @param moveAngle
 * @parent fogEffect
 * @text 移动角度
 * @desc 移动角度，从0~360度，0为正东方向。
 * @type number[]
 * @default ["0","180"]
 * 
 * @param offsetAngle
 * @parent fogEffect
 * @text 偏移角度
 * @desc 偏移角度，从0~360度，0为正东方向。
 * @type number
 * @default 20
 * 
 * @param moveDistance
 * @parent fogEffect
 * @text 移动距离
 * @desc 移动距离
 * @type number
 * @default 50
 * 
 * @param skipDuration
 * @parent fogEffect
 * @text 跳过帧数
 * @desc 跳过帧数，在帧数范围内为隐藏状态
 * @type number
 * @default 0
 * 
 * @param delayFrame
 * @parent fogEffect
 * @text 间隔帧数
 * @desc 间隔帧数，在帧数范围内为隐藏状态
 * @type number
 * @default 50
 * 
 * @param offsetDistance
 * @parent fogEffect
 * @text 偏移距离
 * @desc 偏移距离
 * @type number
 * @default 20
 * 
 * @param maxCount
 * @parent fogEffect
 * @text 最大雾气数量
 * @desc 最大雾气数量
 * @type number
 * @default 2
 * 
 * @param reScale
 * @parent fogEffect
 * @text 最终放大倍数
 * @desc 最终放大倍数
 * @type text
 * @default 1.6
 * 
 * @param fadeInDuration
 * @parent useFade
 * @text 淡入时长（帧数）
 * @desc 特效开始播放显示的淡入时长（帧数）
 * @type number
 * @default 30
 * 
 * @param fadeOutDuration
 * @parent useFade
 * @text 淡出时长（帧数）
 * @desc 特效结束播放显示的淡出时长（帧数）
 * @type number
 * @default 50
 * 
 */
// #endregion
(() => {
  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_CG_Effect";
  
  // #region 插件参数解释器
  class PluginParamsParser {
    constructor(predictEnable = true) {
      this._predictEnable = predictEnable;
    }
    static parse(params, typeData, predictEnable = true) {
      return new PluginParamsParser(predictEnable).parse(params, typeData);
    }
    parse(params, typeData, loopCount = 0) {
      if (++loopCount > 255)
        throw new Error("endless loop error");
      const result = {};
      for (const name in typeData) {
        if (params[name] === "" || params[name] === undefined) {
          result[name] = null;
        }
        else {
          result[name] = this.convertParam(params[name], typeData[name], loopCount);
        }
      }
      if (!this._predictEnable)
        return result;
      if (typeof params === "object" && !(params instanceof Array)) {
        for (const name in params) {
          if (result[name])
            continue;
          const param = params[name];
          const type = this.predict(param);
          result[name] = this.convertParam(param, type, loopCount);
        }
      }
      return result;
    }
    convertParam(param, type, loopCount) {
      if (typeof type === "string") {
        let str = param;
        if (str[0] == '"' && str[str.length - 1] == '"') {
          str = str.substring(1, str.length - 1).replace(/\\n/g, '\n').replace(/\\"/g, '"')
        }
        return this.cast(str, type);
      }
      else if (typeof type === "object" && type instanceof Array) {
        const aryParam = JSON.parse(param);
        if (type[0] === "string") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else if (type[0] === "number") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else {
          if (!aryParam.length) return [];
          else return aryParam.map((strParam) => this.parse(JSON.parse(strParam), type[0]), loopCount);
        }
      }
      else if (typeof type === "object") {
        return this.parse(JSON.parse(param), type, loopCount);
      }
      else {
        throw new Error(`${type} is not string or object`);
      }
    }
    cast(param, type) {
      switch (type) {
        case "any":
          if (!this._predictEnable)
            throw new Error("Predict mode is disable");
          return this.cast(param, this.predict(param));
        case "string":
          return param;
        case "number":
          if (param.match(/^\-?\d+\.\d+$/))
            return parseFloat(param);
          return parseInt(param);
        case "boolean":
          return param === "true";
        default:
          throw new Error(`Unknow type: ${type}`);
      }
    }
    predict(param) {
      if (param.match(/^\-?\d+$/) || param.match(/^\-?\d+\.\d+$/)) {
        return "number";
      }
      else if (param === "true" || param === "false") {
        return "boolean";
      }
      else {
        return "string";
      }
    }
  }
  // #endregion

  const typeDefine = {
    effectList: [
      {
        path: ['string'],
        moveAngle: ['number'],
      }
    ]
  };

  const params = PluginParamsParser.parse(PluginManager.parameters(PluginName), typeDefine);

  const effectList = params.effectList;

  /** 特效表 */
  const effectMap = {
    /** 雾气特效 */
    fog: {
      duration(config) {
        return config.duration || 60;
      },
      cache: {},
      prepare(callback, container, config) {
        container.removeChildren();

        this.cache.paths = Array.isArray(config.path) ? config.path : [config.path];

        this.cache.delayFrame = 0;

        this.cache.list = [];

        Promise.all(this.cache.paths.map(path => effectModule.loadImg(path))).then(() => {
          callback();
        });
      },
      start(container, progress, frameIndex, config) {
        // 如果缓存中特效比最大值要小，切缓存时间过了则创建
        if (this.cache.list.length <= config.maxCount && this.cache.delayFrame <= 0) {
          const _randomPath = this.cache.paths[Math.floor(Math.random() * this.cache.paths.length)];
          const _sprite = new Sprite(ImageManager.loadPicture(_randomPath));
          _sprite.blendMode = PIXI.BLEND_MODES[config.blendMode || 'NORMAL'] || 0;
          _sprite.x = config.x || 0;
          _sprite.y = config.y || 0;
          _sprite.scale.x = config.scale || 1;
          _sprite.scale.y = config.scale || 1;
          _sprite.anchor.x = 0.5;
          _sprite.anchor.y = 0.5;
          _sprite.alpha = 0;

          const _offsetAngle = config.offsetAngle || 0;
          const _offsetDistance = config.offsetDistance || 0;
          const _delayFrame = config.delayFrame || 50;
          let _angle = config.moveAngle[Math.floor(Math.random() * config.moveAngle.length)];
          _angle += Math.random() * _offsetAngle * 2 - _offsetAngle;
          this.cache.list.push({
            sprite: _sprite,
            angle: _angle,
            frameIndex: 0,
            moveDistance: (config.moveDistance || 50) + (Math.random() * _offsetDistance * 2 - _offsetDistance),
          });
          container.addChild(_sprite);
          this.cache.delayFrame = Math.floor(Math.random() * _delayFrame + _delayFrame);
        }
        if (this.cache.delayFrame > 0) this.cache.delayFrame--;

        // 每帧判断是否要显示或隐藏
        if (config.visibleCondition) {
          try {
            const _re = new Function('', 'return ' + config.visibleCondition)();
            if (_re === true) {
              container.alpha = 1;
            } else if (_re === false) {
              container.alpha = 0;
            }
          } catch (error) {
            throw new Error(`显示条件代码执行出错: ${error.message}`);
          }
        }

        const _frameCount = config.fogDuration || 100;
        const _fadeInDuration = config.fadeInDuration || 30;
        const _fadeOutDuration = config.fadeOutDuration || 30;
        for (let i = this.cache.list.length - 1; i >= 0; i--) {
          const _item = this.cache.list[i];
          const _frameIndex = _item.frameIndex - (config.skipDuration || 0);
          const _scale = 1 + ((config.reScale || 1) - 1) * (_frameIndex / _frameCount);
          if (_frameIndex < _fadeInDuration) {
            _item.sprite.alpha = _frameIndex / _fadeInDuration;
          } else if (_frameIndex >= _frameCount - _fadeOutDuration) {
            _item.sprite.alpha = 1 - (_frameIndex - _frameCount + _fadeOutDuration) / _fadeOutDuration;
            if (_frameIndex >= _frameCount) {
              container.removeChild(_item.sprite);
              this.cache.list.splice(i, 1);
            }
          }
          _item.frameIndex++;
          const { x, y } = effectModule.calculateAnglePosition(
            _item.angle, config.x || 0, config.y || 0, 
            _item.frameIndex / _frameCount * _item.moveDistance,
          );
          _item.sprite.x = x;
          _item.sprite.y = y;
          _item.sprite.scale.x = _scale * (config.scale || 1);
          _item.sprite.scale.y = _scale * (config.scale || 1);
        }
      },
      end(container) {
      },
    }
  };

  /** 特效模块 */
  const effectModule = {
    containers: {},
    effectSprite: undefined,
    _tickers: [],
    on(callback) {
      for (let i = 0; i < this._tickers.length; i++) {
        if (this._tickers[i] === undefined) {
          this._tickers[i] = callback;
          return;
        }
      }
      this._tickers.push(callback);
    },
    off(callback) {
      const _index = this._tickers.indexOf(callback);
      if (_index >= 0) {
        this._tickers.splice(_index, 1);
      }
    },
    update() {
      for (let i = 0; i < this._tickers.length; i++) {
        if (this._tickers[i]) this._tickers[i]();
      }
    },
    /**
     * 使用延时功能
     * @param {() => void} callback 回调函数
     * @param {number} delay 等待帧数
     * @param {(progress: number) => void} update 更新函数
     * @param {{ easingType: string, inout: string }} info 动画信息
     */
    useTimeout(callback, delay, update, info = { easingType: 'Linear', inout: 'None' }) {
      const _timer = {
        frameIndex: 0,
        frameCount: delay,
        stop() {
          this.frameIndex = 0;
          effectModule.off(_timerUpdate);
        }
      };

      const _timerUpdate = () => {
        _timer.frameIndex++;

        if (update) {
          const progress = delay <= 0 ? 0 : _timer.frameIndex / _timer.frameCount;
          update(progress, _timer.frameIndex);
        }

        if (delay > 0 && _timer.frameIndex >= _timer.frameCount) {
          _timer.frameIndex = 0;
          effectModule.off(_timerUpdate);
          callback();
        }
      };

      effectModule.on(_timerUpdate);
      return _timer;
    },
    /**
     * 切换特效层级
     * @param {object} effectInfo 特效信息
     * @param {PIXI.Container} effectSprite 特效容器
     * @param {string | (() => void)} replaceFloor 重设层级
     */
    changeFloor(effectInfo, effectSprite, replaceFloor) {
      let _floor = replaceFloor || effectInfo.floor || 'top';

      if (typeof _floor === 'function') {
        _floor.call(SceneManager._scene, effectSprite);
      } else if (_floor === 'custom' && effectInfo.customFloor) {
        new Function('sprite', effectInfo.customFloor).call(SceneManager._scene, effectSprite);
      } else {
        switch (_floor) {
          case 'top':
            SceneManager._scene.addChild(effectSprite);
            break;
          case 'normal':
            if (this instanceof Scene_Map) {
              SceneManager._scene._spriteset.addChild(effectSprite);
            } else {
              SceneManager._scene.addChild(effectSprite);
            }
            break;
          default:
            SceneManager._scene.addChild(effectSprite);
            break;
        }
      }
    },
    /**
     * 根据角度、初始坐标和距离，计算新的坐标
     * @param {number} angle - 角度（以度为单位，0 到 360）
     * @param {number} x - 初始 x 坐标
     * @param {number} y - 初始 y 坐标
     * @param {number} distance - 距离
     * @returns {Object} - 返回计算后的新坐标 { x, y }
     */
    calculateAnglePosition(angle, x, y, distance) {
      // 将角度转换为弧度
      const radians = (angle * Math.PI) / 180;

      // 计算新的 x 和 y 坐标
      const newX = x + distance * Math.cos(radians);
      const newY = y + distance * Math.sin(radians);

      // 返回新坐标
      return { x: newX, y: newY };
    },
    /**
     * 加载指定路径的位图
     * @param {string} path 图片路径
     * @param {string} prefix 前缀路径，默认为 "img/pictures/"
     */
    loadImg(path, prefix = "img/pictures/") {
      return new Promise((resolve, reject) => {
        const _bitmap = ImageManager.loadBitmap(prefix, path);
        if (_bitmap.isReady()) {
          resolve(_bitmap);
        } else {
          _bitmap.addLoadListener(() => {
            resolve(_bitmap);
          });
        }
      });
    },
    /** 停止特效 */
    stopEffect(name) {
      if (!SceneManager._scene.__effectContainer) {
        return;
      }
      effectModule.useTimeout(() => {
        const _effect = effectMap[name];
        effectMap[name].cache.timer.stop();
        effectModule.containers[name].destroy();
        effectModule.containers[name] = undefined;
        _effect.cache.isStart = false;
        if (effectModule.effectSprite) {
          effectModule.effectSprite.destroy();
        }
      }, 20, (progress, frameIndex) => {
        effectModule.containers[name].alpha = 1 - progress;
      });
    },
    /**
     * 开始特效
     * @param {string} name 特效名称
     * @param {object} config 参数
     * @param {Function} callback 回调函数
     * @param {Game_Interpreter} interpreter 事件执行器
     */
    playEffect(name, config = {}, callback, interpreter) {
      if (config.isWait === undefined) config.isWait = true;

      const _effect = effectMap[name];
      if (_effect.cache.isStart) {
        console.warn(`特效 ${name} 已经开始播放`);
        return;
      }
      _effect.cache.isStart = true;

      const _info = { ...effectList.find(i => i.alias === name), ...config };

      if (!_info.duration) _info.duration = 100;
      if (!_info.fadeOutDuration) _info.fadeOutDuration = 0;


      // 判断是否有基础缓存容器，没有则创建
      effectModule.containers[name]
      if (!effectModule.containers[name]) {
        effectModule.containers[name] = new PIXI.Container();
        _effect.cache.container = effectModule.containers[name];
        effectModule.changeFloor(_info, effectModule.containers[name]);
      }
      effectModule.containers[name].alpha = 1;
      
      if (interpreter && _info.isWait && interpreter.wait) interpreter.wait(_effect.duration + _info.fadeOutDuration);
      
      const _start = () => {
        effectMap[name].cache.timer = effectModule.useTimeout(() => {
          effectModule.endEffect(name, _info, callback, interpreter);
        }, _info.loop ? 0 : _info.duration + _info.fadeOutDuration, (progress, frameIndex) => {
          _effect.start.call(_effect, effectModule.containers[name], progress, frameIndex, _info);
        });
      };

      if (_effect.prepare) {
        _effect.prepare.call(_effect, _start, effectModule.containers[name], _info);
      } else {
        _start();
      }
      
    },
    /**
     * 结束特效
     * @param {string} name 特效名称
     * @param {object} config 参数
     * @param {Function} callback 回调函数
     * @param {Game_Interpreter} interpreter 事件执行器
     */
    endEffect(name, config, callback, interpreter) {
      if (config.isWait === undefined) config.isWait = true;

      const _effect = effectMap[name];
      if (interpreter && config.isWait) interpreter.wait(_effect.duration);
      
      useTimeout(() => {
        _effect.cache.container.destroy();
        _effect.cache = {};
        _effect.cache.isStart = false;
        callback && callback();
      }, _effect.duration, (progress, frameIndex) => {
        _effect.end.call(_effect, _effect.cache.container, progress, frameIndex, config);
      });
    }
  }
  window.effectModule = effectModule;

  const _Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _Scene_Base_update.call(this);
    
    effectModule.update();
  };

  Game_Interpreter.prototype.command261 = function(p) {
    effectModule.playEffect(p[0], { x: p[1] || 0, y: p[2] || 0 });
    return true;
  };

  // MZ 指令
  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'playEffect', (args) => {
      effectModule.playEffect(args.name, {
        x: args.x,
        y: args.y,
      }, () => {
        effectModule.stopEffect(args.name, {
          floor: args.floor
        });
      }, this);
    });

    
    PluginManager.registerCommand(PluginName, 'stopEffect', (args) => {
      effectModule.stopEffect(args.name);
    });
  }
  
  // MV 指令
  const Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'effectplay') {
      effectModule.playEffect(args[0], { floor: args[1] });
    } else if (command === 'effectstop') {
      effectModule.stopeffect();
    }
  };

})();