//=============================================================================
// ** RPG Maker MZ - Hakubox_Module_Actor.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 定制模块 - 主角属性插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MZ
 * 
 * @help
 * 【与傲娇妹妹的治愈日常】主角属性相关插件。
 * 
 * 
 * @command addLive
 * @text 设置体力
 * 
 * @arg value
 * @text 体力值
 * @type number
 * @min -100
 * @max 100
 * @default 0
 * 
 * @arg operation
 * @text 操作类型
 * @desc 操作类型
 * @type select
 * @option 增加 - add
 * @value add
 * @option 减少 - minus
 * @value minus
 * @option 设为固定值 - set
 * @value set
 * @default add
 * 
 * 
 * @command addMaxLive
 * @text 设置体力上限
 * 
 * @arg value
 * @text 体力上限
 * @type number
 * @min -100
 * @max 100
 * @default 0
 * 
 * @arg operation
 * @text 操作类型
 * @desc 操作类型
 * @type select
 * @option 增加 - add
 * @value add
 * @option 减少 - minus
 * @value minus
 * @option 设为固定值 - set
 * @value set
 * @default add
 * 
 * 
 * @command addSan
 * @text 修改理智值
 * 
 * @arg value
 * @text 理智值
 * @type number
 * @min -100
 * @max 100
 * @default 0
 * 
 * @arg operation
 * @text 操作类型
 * @desc 操作类型
 * @type select
 * @option 增加 - add
 * @value add
 * @option 减少 - minus
 * @value minus
 * @option 设为固定值 - set
 * @value set
 * @default minus
 * 
 * 
 * @param live
 * @type variable
 * @text 当前体力值
 * @desc 体力值绑定变量
 * 
 * @param maxLive
 * @type variable
 * @text 最大体力值
 * @desc 最大体力值绑定变量
 * 
 * @param san
 * @type variable
 * @text 当前理智值
 * @desc 理智值绑定变量
 * 
 * @param makeMoneyRate
 * @type variable
 * @text 基础赚钱倍率
 * @desc 基础赚钱倍率绑定变量，默认为 1.0
 * 
 * @param makeLiveRate
 * @type variable
 * @text 基础锻炼倍率
 * @desc 基础锻炼倍率绑定变量，默认为 1.0
 * 
 */
// #endregion
(() => {

  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_Module_Actor";
  
  const typeDefine = {
  };
  // #endregion

  // #region 插件参数

  const params = PluginParamsParser.parse(PluginManager.parameters(PluginName), typeDefine);

  /** 主角信息类 */
  const actorConfig = {
    /** 体力值 @type {number} */
    get live() {
      return $gameVariables.value(params.live);
    },
    set live(value) {
      let _value = value;
      if (value < 0) _value = 0;
      else if (value >= actorConfig.maxLive) _value = actorConfig.maxLive;
      actorConfig.changeLive(_value);
      $gameVariables.setValue(params.live, _value);

      // 体力值归零
      // if (_value <= 0) {
      //   $gameData.isEventBusy = true;
      //   $gameTemp.execCommonEvent(265);
      // }
    },
    /** 最大体力值 @type {number} */
    get maxLive() {
      return $gameVariables.value(params.maxLive) || 100;
    },
    set maxLive(value) {
      $gameVariables.setValue(params.maxLive, value);
    },
    /** 理智值 @type {number} */
    get san() {
      return $gameVariables.value(params.san);
    },
    set san(value) {
      let _value = value;
      if (value < 0) _value = 0;
      else if (value >= actorConfig.maxSan) _value = actorConfig.maxSan;
      actorConfig.changeSan(_value);
      $gameVariables.setValue(params.san, _value);
    },
    /** 最大理智值 */
    get maxSan() {
      return 100;
    },
    /** 理智值等级 */
    get sanLevel() {
      const _san = actorConfig.san;
      if (_san >= 60) return 1;
      else return 0;
    },
    /** 基础赚钱倍率 */
    get makeMoneyRate() {
      return $gameVariables.value(params.makeMoneyRate) || 1;
    },
    set makeMoneyRate(value = 1) {
      $gameVariables.setValue(params.makeMoneyRate, value);
    },
    /** 基础锻炼倍率 */
    get makeLiveRate() {
      return $gameVariables.value(params.makeLiveRate) || 1;
    },
    set makeLiveRate(value = 1) {
      $gameVariables.setValue(params.makeLiveRate, value);
    },
    /** 初始化数据 */
    initGameData() {
      this.live = this.maxLive;
      this.san = this.maxSan;
      this.makeMoneyRate = 1;
      this.makeLiveRate = 1;
    },
    /**
     * 体力改变
     * @param {number} value 新体力
     */
    changeLive(value) {
      const _scene = SceneManager._scene;
      const _oldValue = actorConfig.live;
      if (_oldValue === value) return;
      useAnime({
        frameDelay: 0,
        frameCount: 30,
        autoUpdate: true,
        easingType: 'Quadratic',
        inout: 'InOut',
      }, progress => {
        if (!_scene.liveBitmap || !_scene.textLiveSprite) return;
        const _val = _oldValue + (value - _oldValue) * progress;
        const _width = Math.round(_scene.liveBitmap.width * _val / actorConfig.maxLive);
        _scene.textLiveSprite.bitmap.clear();
        _scene.textLiveSprite.bitmap.drawText(`${Math.round(_val)}/${Utils_Actor.maxLive}`, 0, 0, 80, 18, 'center');
        _scene.liveSprite.bitmap.clear();
        _scene.liveSprite.bitmap.blt(_scene.liveBitmap, 
          0, 0, _width, 20,
          0, 0, _width, 20
        );
      });
      if (_scene.liveContainer) {
        createFloatText({
          sprite: _scene.liveContainer,
          x: _scene.liveContainer.x + _scene.liveContainer.width - 15,
          y: _scene.liveContainer.y,
          text: `${_oldValue < value ? '+' : ''}${value - _oldValue}`
        });
      }
    },
    /**
     * SAN值改变
     * @param {number} value 新体力
     */
    changeSan(value) {
      const _scene = SceneManager._scene;
      const _oldValue = actorConfig.san;
      if (_oldValue === value) return;
      useAnime({
        frameDelay: 0,
        frameCount: 30,
        autoUpdate: true,
        easingType: 'Quadratic',
        inout: 'InOut',
      }, progress => {
        if (!_scene.sanBitmap || !_scene.textSanSprite) return;
        const _val = _oldValue + (value - _oldValue) * progress;
        const _width = Math.round(_scene.sanBitmap.width * _val / actorConfig.maxSan);
        _scene.textSanSprite.bitmap.clear();
        _scene.textSanSprite.bitmap.drawText(`${Math.round(_val)}/${Utils_Actor.maxSan}`, 0, 0, 80, 18, 'center');
        _scene.sanSprite.bitmap.clear();
        _scene.sanSprite.bitmap.blt(_scene.sanBitmap, 
          0, 0, _width, 20, 
          0, 0, _width, 20
        );
      });
      if (_scene.sanContainer) {
        createFloatText({
          sprite: _scene.sanContainer,
          x: _scene.sanContainer.x + _scene.sanContainer.width - 15,
          y: _scene.sanContainer.y,
          text: `${_oldValue < value ? '+' : ''}${value - _oldValue}`
        });
      }
    }
  };

  /** 主角信息工具类 */
  class Utils_Actor {
    /** 体力值 @type {number} */
    static get live() {
      return actorConfig.live;
    }
    static set live(value) {
      const _val = Math.max(0, Math.min(value, actorConfig.maxLive));
      actorConfig.live = _val;
    }
    /** 最大体力值 */
    static get maxLive() {
      return actorConfig.maxLive;
    }
    static set maxLive(value) {
      actorConfig.maxLive = value;
    }
    /** 理智值 @type {number} */
    static get san() {
      return actorConfig.san;
    }
    static set san(value) {
      const _val = Math.max(0, Math.min(value, 100));
      actorConfig.san = _val;
    }
    /** 最大理智值 */
    static get maxSan() {
      return 100;
    }
    /** 理智值等级 @type {number} */
    static get sanLevel() {
      return actorConfig.sanLevel;
    }
    /** 基础赚钱倍率 @type {number} */
    static get makeMoneyRate() {
      return actorConfig.makeMoneyRate;
    }
    static set makeMoneyRate(value) {
      actorConfig.makeMoneyRate = value;
    }
    /** 基础锻炼倍率 @type {number} */
    static get makeLiveRate() {
      return actorConfig.makeLiveRate;
    }
    static set makeLiveRate(value) {
      actorConfig.makeLiveRate = value;
    }
    /**
     * 修改好感度
     * @param {number} value 好感度值
     * @param {'add'|'minus'|'set'} operation 操作类型
     */
    static addLive(value, operation = 'add') {
      if (!value) return;
      if (operation === 'add') {
        actorConfig.live += value;
      } else if (operation === 'minus') {
        actorConfig.live -= value;
      } else if (operation === 'set') {
        actorConfig.live = value;
      }
    }
    /**
     * 修改体力上限
     * @param {number} value 好感度值
     * @param {'add'|'minus'|'set'} operation 操作类型
     */
    static addMaxLive(value, operation = 'add') {
      if (!value) return;
      if (operation === 'add') {
        const _val = Math.round(value * Utils_Actor.makeLiveRate);
        useAlert({ text: TranslateUtils.getText(`最大体力值上升了`) + _val });
        actorConfig.maxLive += _val;
      } else if (operation === 'minus') {
        actorConfig.maxLive -= value;
      } else if (operation === 'set') {
        actorConfig.maxLive = value;
      }
    }
    /**
     * 修改理智
     * @param {number} value 理智值
     * @param {'add'|'minus'|'set'} operation 操作类型
     */
    static addSan(value, operation = 'add') {
      if (!value) return;
      if (operation === 'add') {
        actorConfig.san += value;
      } else if (operation === 'minus') {
        actorConfig.san -= value;
      } else if (operation === 'set') {
        actorConfig.san = value;
      }
    }
    /** 初始化数据 */
    static initGameData() {
      actorConfig.initGameData();
    }
  }
  window.Utils_Actor = Utils_Actor;

  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'addLive', args => {
      const _value = Number(args.value);
      Utils_Actor.addLive(_value, args.operation);
    });
    PluginManager.registerCommand(PluginName, 'addMaxLive', args => {
      const _value = Number(args.value);
      Utils_Actor.addMaxLive(_value, args.operation);
    });
    PluginManager.registerCommand(PluginName, 'addSan', args => {
      const _value = Number(args.value);
      Utils_Actor.addSan(_value, args.operation);
    });
  }
  
})();