//=============================================================================
// ** RPG Maker MZ - Hakubox_Module_Scene.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 定制模块 - 游戏场景插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MZ
 * 
 * @help
 * 【与傲娇妹妹的治愈日常】游戏场景相关插件。
 * 
 * 
 * @command changeScene
 * @text 切换场景
 * 
 * @arg sceneName
 * @text 场景名称
 * @type select
 * @option 小镇
 * @value scene_map
 * @option 家中卧室
 * @value scene_bedroom
 * @option 便利店
 * @value scene_shop
 * @option 神秘商店
 * @value scene_r18_shop
 * @option 街道
 * @value scene_street
 * @option 街道_便利店
 * @value scene_street_shop
 * @option 睡眠奸
 * @value scene_bedroom_yobai
 * @option 森林
 * @value scene_forest
 * @default scene_map
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isFadeout
 * @text 是否渐隐
 * @desc 是否渐隐
 * @type boolean
 * @on 是
 * @off 否
 * @default true
 * 
 * @arg isFadein
 * @text 是否渐显
 * @desc 是否渐显
 * @type boolean
 * @on 是
 * @off 否
 * @default true
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command changeBgAndFg
 * @text 切换CG背景
 * 
 * @arg backImg
 * @text 背景图
 * @desc 背景图，基础路径为"img/pictures"
 * @type file
 * @dir img/pictures
 * 
 * @arg foreImg
 * @text 前景图
 * @desc 前景图，基础路径为"img/pictures"
 * @type file
 * @dir img/pictures
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command simpleChangeBackground
 * @text 简易切换背景
 * 
 * @arg img
 * @text 图片路径
 * @desc 图片路径，基础路径为"img/pictures/"
 * @type file
 * @dir img/pictures/
 * 
 * 
 * @command changeBackground
 * @text 切换背景
 * 
 * @arg imgs
 * @text 图片路径
 * @desc 图片路径，基础路径为"img/pictures/"
 * @type file[]
 * @dir img/pictures/
 * 
 * @arg duration
 * @text 持续时间
 * @desc 背景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command loopChangeBackground
 * @text 重复切换背景
 * 
 * @arg imgs
 * @text 图片路径
 * @desc 图片路径，基础路径为"img/pictures/"
 * @type file[]
 * @dir img/pictures/
 * @default []
 * 
 * @arg delay
 * @text 间隔时长
 * @desc 背景图片切换持续时间，单位为帧数
 * @type number
 * @min 10
 * @default 30
 * 
 * @arg times
 * @text 循环次数
 * @desc 循环次数
 * @type number
 * @min 2
 * @default 3
 * 
 * @arg isReset
 * @text 结束后恢复原始背景
 * @desc 结束后恢复原始背景
 * @type boolean
 * @on 结束后恢复
 * @off 不恢复
 * @default true
 * 
 * 
 * 
 * @command resetSceneObjects
 * @text 重置场景交互对象
 * @desc 重置场景交互对象
 * 
 * 
 * @command setBlackMask
 * @text 屏幕变黑
 * 
 * 
 * @command setMaskAlpha
 * @text 设置黑屏透明度
 * 
 * @arg alpha
 * @text 透明度
 * @desc 透明度：0~255，0为完全不透明
 * @type number
 * @max 255
 * @min 0
 * @default 255
 * 
 * @arg duration
 * @text 时长
 * @desc 时长（帧数）
 * @type number
 * @max 1000
 * @min 0
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command resetBackground
 * @text 还原背景
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command changeForeground
 * @text 切换前景
 * 
 * @arg imgs
 * @text 图片路径
 * @desc 图片路径，基础路径为"img/pictures/"
 * @type file[]
 * @dir img/pictures/
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command resetForeground
 * @text 还原前景
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command resetFilter
 * @text 重置时段滤镜
 * @desc 重置时段滤镜
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * 
 * @command clearFilter
 * @text 暂时清理时段滤镜
 * @desc 暂时清理时段滤镜
 * 
 * @arg duration
 * @text 持续时间
 * @desc 前景图片切换持续时间，单位为帧数
 * @type number
 * @default 30
 * 
 * 
 * @command actorMove
 * @text 角色移动
 * 
 * @arg actor
 * @text 角色名称
 * @desc 角色名称
 * @type select
 * @option 主角（玩家）
 * @value player
 * @option 加奈（カナ）
 * @value kana
 * @option 千夏（チナツ）
 * @value chinatsu
 * @option 由纪（ゆき）
 * @value yuki
 * @option 艾露（エーレ）
 * @value ere
 * @default player
 * 
 * @arg scene
 * @text 移动场景
 * @desc 移动场景
 * @type select
 * @option 小镇
 * @value scene_map
 * @option 家中卧室
 * @value scene_bedroom
 * @option 便利店
 * @value scene_shop
 * @option 神秘商店
 * @value scene_r18_shop
 * @option 服装店
 * @value scene_clothing_shop
 * @option 街道
 * @value scene_street
 * @option 公园
 * @value scene_park
 * @option 森林
 * @value scene_forest
 * @option 学校（并不存在）
 * @value school
 * @default scene_map
 * 
 * 
 * @command withMove
 * @text 和某个角色一起移动
 * 
 * @arg actor
 * @text 角色名称
 * @desc 角色名称
 * @type select
 * @option 加奈（カナ）
 * @value kana
 * @option 千夏（チナツ）
 * @value chinatsu
 * @option 由纪（ゆき）
 * @value yuki
 * @option 艾露（エーレ）
 * @value ere
 * @default kana
 * 
 * @arg scene
 * @text 移动场景
 * @desc 移动场景
 * @type select
 * @option 小镇
 * @value scene_map
 * @option 家中卧室
 * @value scene_bedroom
 * @option 便利店
 * @value scene_shop
 * @option 神秘商店
 * @value scene_r18_shop
 * @option 服装店
 * @value scene_clothing_shop
 * @option 街道
 * @value scene_street
 * @option 公园
 * @value scene_park
 * @option 森林
 * @value scene_forest
 * @default scene_bedroom
 * 
 * @arg duration
 * @text 移动时长
 * @desc 移动时长
 * @type number
 * @default 30
 * 
 * @arg isFadeout
 * @text 是否渐隐
 * @desc 是否渐隐
 * @type boolean
 * @on 是
 * @off 否
 * @default true
 * 
 * @arg isFadein
 * @text 是否渐显
 * @desc 是否渐显
 * @type boolean
 * @on 是
 * @off 否
 * @default true
 * 
 * 
 * 
 * @param current
 * @text 当前角色/玩家
 * 
 * @param currentLocVariable
 * @parent current
 * @text 玩家所在的场景变量ID
 * @desc 玩家所在的场景变量ID
 * @type variable
 * 
 * @param currentActionVariable
 * @parent current
 * @text 玩家当前的行为变量ID
 * @desc 玩家当前的行为变量ID
 * @type variable
 * 
 * 
 * @param kana
 * @text 加奈（カナ）
 * 
 * @param kanaLocVariable
 * @parent kana
 * @text 加奈所在的场景变量ID
 * @desc 加奈所在的场景变量ID
 * @type variable
 * 
 * @param kanaActionVariable
 * @parent kana
 * @text 加奈当前的行为变量ID
 * @desc 加奈当前的行为变量ID
 * @type variable
 * 
 * 
 * @param chinatsu
 * @text 千夏（チナツ）
 * 
 * @param chinatsuLocVariable
 * @parent chinatsu
 * @text 千夏所在的场景变量ID
 * @desc 千夏所在的场景变量ID
 * @type variable
 * 
 * @param chinatsuActionVariable
 * @parent chinatsu
 * @text 千夏当前的行为变量ID
 * @desc 千夏当前的行为变量ID
 * @type variable
 * 
 * 
 * @param yuki
 * @text 由纪（ゆき）
 * 
 * @param yukiLocVariable
 * @parent yuki
 * @text 由纪所在的场景变量ID
 * @desc 由纪所在的场景变量ID
 * @type variable
 * 
 * @param yukiActionVariable
 * @parent yuki
 * @text 由纪当前的行为变量ID
 * @desc 由纪当前的行为变量ID
 * @type variable
 * 
 * 
 * @param ere
 * @text 艾露（エーレ）
 * 
 * @param ereLocVariable
 * @parent ere
 * @text 艾露所在的场景变量ID
 * @desc 艾露所在的场景变量ID
 * @type variable
 * 
 * @param ereActionVariable
 * @parent ere
 * @text 艾露当前的行为变量ID
 * @desc 艾露当前的行为变量ID
 * @type variable
 * 
 */
// #endregion
(() => {

  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_Scene";
  
  const typeDefine = {
  };
  // #endregion

  // #region 插件参数

  const params = PluginParamsParser.parse(PluginManager.parameters(PluginName), typeDefine);

  /**
   * @typedef SceneItem
   * @property {boolean} isMain 是否为主场景
   * @property {string} name 场景名称
   * @property {string} title 场景标题
   * @property {string|object[]} background 获取场景背景
   * @property {string|object[]} foreground 前景图片
   * @property {string} description 场景描述
   * @property {string[]} commands 场景命令列表
   * 
   * @property {object[]} sceneObjects 场景内可交互对象表
   * @property {boolean} hasFilter 是否包含染色层
   * @property {function} customFilter 自定义染色层
   * 
   */

  /**
   * 场景列表 @type {SceneItem[]}
   */
  const sceneList = [
    {
      isMain: true,
      name: "scene_map",
      title: "小镇",
      get scene() { return Scene_Haku_Map },
      get background() {
        return "map/map_bg";
      },
      hasFilter: true,
      get bgm() {
        if (Utils_Time.timeSpan === 'daytime') {
          return "BGM_map";
        } else {
          return "BGM_night_1";
        }
      },
    }, {
      isMain: true,
      name: "scene_bedroom",
      title: "家中卧室",
      get scene() { return Scene_Home },
      get background() {
        const _imgs = ['bg/home/bedroom'];
        const _scene = SceneManager._scene;

        const _base = "bg/home";
        switch (Utils_Time.timeSpan) {
          case 'daytime':
            break;
          case 'dusk':
            _imgs.push({ path: 'bg/background_dusk_mask', blendMode: 2 });
            break;
          case 'night':
            if (!_scene.isOnLight) {
              _imgs.push({ path: _base + '/background_night_moon_mask_1', blendMode: 2 });
              _imgs.push({ path: _base + '/background_night_moon_mask_2', blendMode: 3 });
            } else {
              _imgs.push({ path: _base + '/bedroom_night_light_back', blendMode: 3 });
            }
            break;
          default:
            break;
        }
        return _imgs;
      },
      get foreground() { return []; },
      sceneObjects: [
        {
          name: 'pillow_1', x: 393, y: 346.5,
          description: '加奈睡觉的枕头',
          normalImg: 'bg/home/items/pillow_1',
          hoverImg: 'bg/home/items/pillow_1_hover',
          condition() { return !$gameSwitches.value(7); },
          onTap() {
            startTalk([{ text: '加奈一直用的枕头，时间长了略微的有点发黄了呢。' }])
          }
        }, {
          name: 'pillow_2', x: 393, y: 346.5,
          description: '给加奈买的新枕头',
          normalImg: 'bg/home/items/pillow_2',
          hoverImg: 'bg/home/items/pillow_2_hover',
          condition() { return $gameSwitches.value(7); },
          onTap() {
            startTalk([{ text: '给加奈买的新枕头，感觉比原来那个会舒服上不少的样子。' }])
          }
        }, {
          name: 'monkey', x: 268, y: 195,
          description: '由纪送给加奈的猴子',
          normalImg: 'bg/home/items/monkey',
          hoverImg: 'bg/home/items/monkey_hover',
          condition() { return false; },
          onTap() {
            startTalk([{ text: '这好像是当时由纪送给加奈的猴子…会有什么寓意吗？' }])
          }
        }, {
          name: 'mirror', x: 108.5, y: 147,
          description: '艾露送的穿衣镜',
          normalImg: 'bg/home/items/mirror',
          hoverImg: 'bg/home/items/mirror_hover',
          condition() { return false; },
          onTap() {
            startTalk([{ text: '果然还是要可爱的女孩子使用这面镜子比较好' }])
          }
        }, {
          name: 'computer', x: 716, y: 445,
          description: '加奈的笔记本电脑',
          normalImg: 'bg/home/items/computer',
          hoverImg: 'bg/home/items/computer_hover',
          condition() { return Utils_Scene.sameKana(); },
          onTap() {
            startTalk([{ text: '加奈的笔记本，平时不要随便偷看会比较好' }])
          }
        }, {
          name: 'kuma', x: 345, y: 326,
          description: '哥哥送的小熊',
          normalImg: 'bg/home/items/kuma',
          hoverImg: 'bg/home/items/kuma_hover',
          condition() { return $gameSwitches.value(26); },
          onTap() {
            startTalk([{ text: '这是之前送给妹妹的小熊，某种程度上还真是和妹妹很相配…' }])
          }
        }, {
          name: 'sleep', x: 220, y: 319,
          description: '睡觉的妹妹',
          normalImg: 'bg/home/items/sleep',
          hoverImg: 'bg/home/items/sleep_hover',
          condition() { return this.isSleep; },
          onTap() {
            startTalk([{ text: '妹妹似乎还在睡觉的样子' }])
          }
        }
      ],
      hasFilter: true,
      customFilter() {
        const _scene = SceneManager._scene;
        const _sprites = [];

        if (Utils_Time.timeSpan === 'dusk') {
          const _filter1 = new Sprite(ImageManager.loadPicture('bg/background_dusk_mask'));
          _filter1.blendMode = 2;
          _sprites.push(_filter1);
        } else if (Utils_Time.timeSpan === 'night') {
          if (_scene.isOnLight) {
            const _filter1 = new Sprite(ImageManager.loadPicture('bg/home/bedroom_night_light_front'));
            _filter1.blendMode = 2;
            _sprites.push(_filter1);
          } else {
            const _filter1 = new Sprite(ImageManager.loadPicture('bg/background_night_mask'));
            _filter1.blendMode = 2;
            _sprites.push(_filter1);
          }
        }

        return _sprites;
      },
      get bgm() {
        if (Utils_Time.timeSpan === 'daytime') {
          return "BGM1";
        } else {
          return "BGM_home_night";
        }
      },
      description: "主卧室。",
    }, {
      isMain: true,
      name: "scene_shower",
      title: "浴室",
      get scene() { return Scene_MainScene },
      get background() {
        return "bg/home-else/shower_daytime";
      },
      bgm: "BGM1",
      description: "浴室场景。",
    }, {
      isMain: true,
      name: "scene_bedroom_yobai",
      get scene() { return Hakubox_Scene_Yobai },
      // get background() {
      //   return "bg/background_night_mask";
      // },
      bgm: "BGM5",
      description: "妹妹的夜袭场景。",
      hasFilter: true,
      customFilter() {
        const _scene = SceneManager._scene;
        const _sprites = [];

        if (Utils_Time.timeSpan === 'dusk') {
          const _filter1 = new Sprite(ImageManager.loadPicture('bg/background_dusk_mask'));
          _filter1.blendMode = 2;
          _sprites.push(_filter1);
        } else if (Utils_Time.timeSpan === 'night') {
          if (_scene.isOnLight) {
            const _filter1 = new Sprite(ImageManager.loadPicture('bg/home/bedroom_night_light_front'));
            _filter1.blendMode = 2;
            _sprites.push(_filter1);
          } else {
            const _filter1 = new Sprite(ImageManager.loadPicture('bg/background_night_mask2'));
            _filter1.blendMode = 2;
            _sprites.push(_filter1);
          }
        }

        return _sprites;
      },
    }, {
      isMain: true,
      name: "scene_bedroom_interact",
      get scene() { return Hakubox_Scene_Interact },
      // get background() {
      //   return "bg/home-bed-interact/bg-bed";
      // },
      bgm: "BGM1",
      description: "和妹妹的交互场景。",
    }, {
      isMain: true,
      name: "scene_shop",
      title: "便利店",
      get scene() { return Scene_Haku_Shop },
      get background() {
        return "bg/shop/counter_bg";
      },
      get foreground() {
        return "bg/shop/counter_fg";
      },
      bgm: "BGM_shop",
      description: "街道的便利店，千夏是唯一的员工。",
    }, {
      isMain: true,
      name: "scene_r18_shop",
      title: "神秘商店",
      get scene() { return Scene_Haku_R18_Shop },
      get background() {
        return "bg/r18-shop/counter_bg2";
      },
      get foreground() {
        return "bg/r18-shop/counter_fg2";
      },
      bgm: "BGM6",
      description: "街角的神秘商店，由纪是代理店主，一般会卖一些神奇的道具。",
    }, {
      isMain: true,
      name: "scene_street_shop",
      title: "街道_便利店",
      get scene() { return Scene_MainScene },
      get background() {
        return "bg/street/street_shop_daytime";
      },
      bgm: "BGM_shop",
      description: "便利店外的街道。",
    }, {
      isMain: true,
      name: "scene_park",
      title: "公园",
      get scene() { return Scene_Haku_Park },
      get background() {
        switch (Utils_Time.timeSpan) {
          case 'daytime': return "bg/park/park_daytime";
          case 'dusk': return "bg/park/park_dusk";
          case 'night': return "bg/park/park_night";
        }
      },
      bgm: "BGM_park",
      hasFilter: true,
      description: "艾露经营的服装店。",
    }, {
      isMain: true,
      name: "scene_street",
      title: "街道",
      get scene() { return Scene_Haku_Street },
      get background() {
        return "bg/street/street_daytime";
      },
      get bgm() {
        if (Utils_Time.timeSpan === 'daytime') {
          return "BGM_night_2";
        } else {
          return "BGM_street";
        }
      },
      hasFilter: true,
      description: "小镇中的繁华街道。",
    }, {
      isMain: true,
      name: "scene_clothing_shop",
      title: "服装店",
      get scene() { return Scene_Haku_Clothing_Shop },
      get background() {
        return "bg/clothing-shop/clothing_shop";
      },
      bgm: "BGM_clothing_shop",
      description: "艾露经营的服装店。",
    }, {
      isMain: true,
      name: "scene_forest",
      title: "森林",
      get scene() { return Scene_Haku_Forest },
      get background() {
        return "bg/forest/forest_2";
      },
      bgm: "BGM_forest",
      hasFilter: true,
      description: "森林。",
    }
  ];

  /** 场景模块 */
  const sceneModule = {
    /** 玩家所在场景名称 @type {string} */
    get currentScene() {
      return $gameVariables.value(params.currentLocVariable);
    },
    set currentScene(value) {
      $gameVariables.setValue(params.currentLocVariable, value);
    },
    /** 玩家当前的行为名称 @type {string} */
    get currentAction() {
      return $gameVariables.value(params.currentActionVariable);
    },
    set currentAction(value) {
      $gameVariables.setValue(params.currentActionVariable, value);
    },
    /** 加奈所在的场景名称 @type {string} */
    get kanaScene() {
      return $gameVariables.value(params.kanaLocVariable);
    },
    set kanaScene(value) {
      $gameVariables.setValue(params.kanaLocVariable, value);
    },
    /** 加奈当前的行为名称 @type {string} */
    get kanaAction() {
      return $gameVariables.value(params.kanaActionVariable);
    },
    /** 千夏所在的场景名称 @type {string} */
    get chinatsuScene() {
      return $gameVariables.value(params.chinatsuLocVariable);
    },
    set chinatsuScene(value) {
      $gameVariables.setValue(params.chinatsuLocVariable, value);
    },
    /** 千夏当前的行为名称 @type {string} */
    get chinatsuAction() {
      return $gameVariables.value(params.chinatsuActionVariable);
    },
    /** 由纪所在的场景名称 @type {string} */
    get yukiScene() {
      return $gameVariables.value(params.yukiLocVariable);
    },
    set yukiScene(value) {
      $gameVariables.setValue(params.yukiLocVariable, value);
    },
    /** 由纪当前的行为名称 @type {string} */
    get yukiAction() {
      return $gameVariables.value(params.yukiActionVariable);
    },
    /** 艾露所在的场景名称 @type {string} */
    get ereScene() {
      return $gameVariables.value(params.ereLocVariable);
    },
    set ereScene(value) {
      $gameVariables.setValue(params.ereLocVariable, value);
    },
    /** 艾露当前的行为名称 @type {string} */
    get ereAction() {
      return $gameVariables.value(params.ereActionVariable);
    },
    /** 判断其他特殊地点 */
    checkElseLoc(loc1, loc2) {
      // 特殊地点1：卧室和床上
      const _arr1 = ['scene_bedroom', 'scene_bedroom_interact', 'scene_bedroom_yobai'];
      if (_arr1.includes(loc1) && _arr1.includes(loc2)) {
        return true;
      } else {
        return false;
      }
    },
    /**
     * 是否和任意女角色在一起
     * @returns {boolean}
     */
    sameOne(sceneName) {
      return this.sameKana(sceneName) || this.sameChinatsu(sceneName) || this.sameYuki(sceneName) || this.sameEre(sceneName);
    },
    /**
     * 是否和加奈在一起
     * @returns {boolean}
     */
    sameKana(sceneName) {
      const _scene = sceneName || this.currentScene;
      return _scene === this.kanaScene || this.checkElseLoc(_scene, this.kanaScene);
    },
    /**
     * 是否和千夏在一起
     * @returns {boolean}
     */
    sameChinatsu(sceneName) {
      const _scene = sceneName || this.currentScene;
      return _scene === this.chinatsuScene || this.checkElseLoc(_scene, this.chinatsuScene);
    },
    /**
     * 是否和由纪在一起
     * @returns {boolean}
     */
    sameYuki(sceneName) {
      const _scene = sceneName || this.currentScene;
      return _scene === this.yukiScene || this.checkElseLoc(_scene, this.yukiScene);
    },
    /**
     * 是否和艾露在一起
     * @returns {boolean}
     */
    sameEre(sceneName) {
      const _scene = sceneName || this.currentScene;
      return _scene === this.ereScene || this.checkElseLoc(_scene, this.ereScene);
    },
    /** 显示场景 */
    showScene() {

    },
    /** 获取场景信息，未传值则获取当前场景信息 */
    getSceneInfo(sceneName) {
      const _sceneName = sceneName || this.currentScene;
      return sceneList.find(i => i.name === _sceneName);
    },
    /** 加载场景数据 */
    loadSceneData(scene) {
    },
    /** 初始化游戏场景数据 */
    initGameData() {
      this.currentScene = "scene_bedroom";
      this.prevSceneName = "scene_bedroom";
      this.kanaScene = "scene_bedroom";
      this.chinatsuScene = "scene_shop";
      this.yukiScene = "scene_r18_shop";
      this.ereScene = "scene_clothing_shop";
    },
    /**
     * 角色移动
     * @param {string} actor 角色
     * @param {string} scene 场景
     */
    actorMove(actor, scene) {
      if (actor === "player") {
        this.currentScene = scene;
        SceneManager._scene.changeScene(scene);
      } else if (actor === "kana") {
        this.kanaScene = scene;
      } else if (actor === "chinatsu") {
        this.chinatsuScene = scene;
      } else if (actor === "yuki") {
        this.yukiScene = scene;
      } else if (actor === "ere") {
        this.ereScene = scene;
      }
    },
    /**
     * 一起移动
     * @param {string} actor 角色
     * @param {string} scene 场景
     * @param {number} duration 切换时长（帧）
     * @param {boolean} [isFadeout=true] 是否渐隐
     * @param {boolean} [isFadein=true] 是否渐显
     */
    withMove(actor, scene, duration = 30, isFadeout = true, isFadein = true) {
      this.currentScene = scene;
      if (actor === "kana") {
        this.kanaScene = scene;
      } else if (actor === "chinatsu") {
        this.chinatsuScene = scene;
      } else if (actor === "yuki") {
        this.yukiScene = scene;
      } else if (actor === "ere") {
        this.ereScene = scene;
      }
      SceneManager._scene.changeScene({
        sceneNam: scene,
        speed: duration,
        notGotoScene: true,
        isFadeout,
        isFadein,
        callback: () => {
        }
      });
    }
  };

  

  /** 场景工具类 */
  class Utils_Scene {
    /** 场景列表 */
    static get sceneList() {
      return sceneList;
    }
    /** 上一个场景名称 */
    static get prevSceneName() {
      return sceneModule.prevSceneName;
    }
    /** 玩家所在场景名称 @type {string} */
    static get currentScene() {
      return sceneModule.currentScene;
    }
    /** 玩家所在场景名称 @type {string} */
    static set currentScene(value) {
      sceneModule.currentScene = value;
    }
    /** 玩家所在场景 @type {object} */
    static get currentSceneInfo() {
      if (!sceneModule.currentScene) {
        throw new Error("当前场景未设置");
      }
      return sceneList.find(i => i.name === sceneModule.currentScene);
    }
    /** 加奈所在的场景名称 @type {string} */
    static get kanaScene() {
      return sceneModule.kanaScene;
    }
    static set kanaScene(value) {
      sceneModule.kanaScene = value;
    }
    static get chinatsuScene() {
      return sceneModule.chinatsuScene;
    }
    static set chinatsuScene(value) {
      sceneModule.chinatsuScene = value;
    }
    static get yukiScene() {
      return sceneModule.yukiScene;
    }
    static set yukiScene(value) {
      sceneModule.yukiScene = value;
    }
    static get ereScene() {
      return sceneModule.ereScene;
    }
    static set ereScene(value) {
      sceneModule.ereScene = value;
    }
    static get currentAction() {
      return sceneModule.currentAction;
    }
    static set currentAction(value) {
      sceneModule.currentAction = value;
    }
    static get kanaAction() {
      return sceneModule.kanaAction;
    }
    static get chinatsuAction() {
      return sceneModule.chinatsuAction;
    }
    static get yukiAction() {
      return sceneModule.yukiAction;
    }
    static get ereAction() {
      return sceneModule.ereAction;
    }
    /**
     * 是否和任意女角色在一起
     * @returns {boolean}
     */
    static sameOne(sceneName) {
      return sceneModule.sameOne(sceneName);
    }
    /**
     * 是否和加奈在一起
     * @returns {boolean}
     */
    static sameKana(sceneName) {
      return sceneModule.sameKana(sceneName);
    }
    /**
     * 是否和千夏在一起
     * @returns {boolean}
     */
    static sameChinatsu(sceneName) {
      return sceneModule.sameChinatsu(sceneName);
    }
    /**
     * 是否和由纪在一起
     * @returns {boolean}
     */
    static sameYuki(sceneName) {
      return sceneModule.sameYuki(sceneName);
    }
    /**
     * 是否和艾露在一起
     * @returns {boolean}
     */
    static sameEre(sceneName) {
      return sceneModule.sameEre(sceneName);
    }
    /** 切换场景 */
    static changeScene({
      sceneName, duration = 30, isFadeout = true, isFadein = true, isWait = true, props, callback
    }) {
      if (SceneManager._scene instanceof Scene_MainScene) {
        sceneModule.prevSceneName = sceneModule.currentScene;
        sceneModule.currentScene = sceneName;
        if (arguments.length === 1 && typeof arguments[0] === "string") {
          SceneManager._scene.changeScene({
            sceneName: arguments[0],
            speed: 30,
            isFadeout: true,
            isFadein: true,
            props,
            callback
          });
        } else {
          SceneManager._scene.changeScene({
            sceneName,
            speed: duration,
            isFadeout,
            isFadein,
            isWait,
            props,
            callback
          }, this);
        }
      } else {
        throw new Error("场景切换失败，当前场景不是主场景");
      }
    }
    /** 加载场景数据 */
    static loadSceneData(scene) {
      sceneModule.loadSceneData(scene);
    }
    /** 初始化场景数据 */
    static initGameData() {
      sceneModule.initGameData();
    }
    /** 获取场景信息，未传值则获取当前场景信息 */
    static getSceneInfo(sceneName) {
      return sceneModule.getSceneInfo(sceneName);
    }
    /**
     * 角色移动
     * @param {string} actor 角色
     * @param {string} scene 场景
     */
    static actorMove(actor, scene) {
      sceneModule.actorMove(actor, scene);
    }
    /**
     * 和某个角色一起移动
     * @param {string} actor 角色
     * @param {string} scene 场景
     * @param {number} [duration=30] 切换时长（帧）
     */
    static withMove(actor, scene, duration = 30, isFadeout = true, isFadein = true) {
      sceneModule.withMove(actor, scene, duration, isFadeout, isFadein);
    }
    /**
     * 判断两个场景是否相同
     * @param {string} loc1 地点1
     * @param {string} loc2 地点2
     * @returns {boolean}
     */
    static checkSameLoc(loc1, loc2) {
      if (loc1 === loc2) return true;
      else return sceneModule.checkElseLoc(loc1, loc2);
    }
    /**
     * 设置遮罩透明度
     */
    static setMaskAlpha(alpha, duration = 0, isWait = true, callback) {
      const _duration = Number(duration) || 0;
      const _alpha = Number(alpha) || 0;
      const _oldAlpha = $gameScreen._brightness;

      if (!_duration) {
        $gameScreen._brightness = _alpha;
        callback && callback();
      } else {
        useTimeout(() => {
          $gameScreen._brightness = _alpha;
          callback && callback();
        }, _duration, (progress, frameIndex) => {
          const _val = _oldAlpha + (_alpha - _oldAlpha) * progress;
          $gameScreen._brightness = Math.round(_val);
        });
      }

      if (_duration && isWait) {
        this.wait(_duration);
      }
    }
  }
  window.Utils_Scene = Utils_Scene;

  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'changeScene', function(args) {
      Utils_Scene.changeScene.call(this, {
        duration: args.duration ? Number(args.duration) : undefined,
        sceneName: args.sceneName,
        isFadeout: args.isFadeout !== 'false',
        isFadein: args.isFadein !== 'false',
        isWait: args.isWait === 'true',
      });
    });
    PluginManager.registerCommand(PluginName, 'changeBgAndFg', function(args) {
      const _isWait = args.isWait === "true";

      SceneManager._scene.changeBackground(args.backImg ? [args.backImg] : undefined, args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
      SceneManager._scene.changeForeground(args.foreImg ? [args.foreImg] : undefined, args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'changeBackground', function(args) {
      const _imgs = args.imgs ? JSON.parse(args.imgs) : undefined;
      const _isWait = args.isWait === "true";
      SceneManager._scene.changeBackground(_imgs, args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'simpleChangeBackground', function(args) {
      SceneManager._scene.changeBackground([args.img], 0, null);
    });
    PluginManager.registerCommand(PluginName, 'loopChangeBackground', function(args) {
      const _times = args.times ? Number(args.times) : 1;
      const _delay = args.duration ? Number(args.delay) : 30;
      const _imgs = args.imgs ? JSON.parse(args.imgs) : [];
      const _maxTime = _delay * _imgs.length * _times;
      const _isReset = args.isReset === "true";

      const _firstBg = cloneLoop($gameData.tempBackground || SceneManager._scene.sceneInfo.background);

      if (_imgs.length < 2) throw new Error("循环图片数量至少为2");

      this.wait(_maxTime);

      let _imgIndex = 0;
      
      SceneManager._scene.changeBackground([_imgs[0]], 0, this);

      useTimeout(() => {
        if (_isReset) {
          SceneManager._scene.changeBackground(_firstBg, 0, this);
        }
      }, _maxTime + _delay, (progress, frameIndex) => {
        if ((frameIndex + 1) % _delay === 0) {
          _imgIndex++;
          if (_imgIndex >= _imgs.length) {
            _imgIndex = 0;
          }

          SceneManager._scene.changeBackground([_imgs[_imgIndex]], 0, this);
        }
      });
    });
    
    PluginManager.registerCommand(PluginName, 'changeForeground', function(args) {
      const _imgs = args.imgs ? JSON.parse(args.imgs) : undefined;
      const _isWait = args.isWait === "true";
      SceneManager._scene.changeForeground(_imgs, args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'resetBackground', function(args) {
      const _isWait = args.isWait === "true";
      SceneManager._scene.changeBackground('', args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'resetForeground', function(args) {
      const _isWait = args.isWait === "true";
      SceneManager._scene.changeForeground('', args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'resetFilter', function(args) {
      const _isWait = args.isWait === "true";
      SceneManager._scene.changeFilter('', args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'clearFilter', function(args) {
      const _isWait = args.isWait === "true";
      SceneManager._scene.clearFilter(args.duration ? Number(args.duration) : undefined, _isWait ? this : null);
    });
    PluginManager.registerCommand(PluginName, 'setBlackMask', function(args) {
      $gameScreen._brightness = 0;
      // SceneManager._scene.transferBlackMask(args.isMask);
    });
    PluginManager.registerCommand(PluginName, 'setMaskAlpha', function(args) {
      Utils_Scene.setMaskAlpha.call(this, args.alpha, args.duration, args.isWait === "true");
    });
    PluginManager.registerCommand(PluginName, 'resetSceneObjects', function(args) {
      if (SceneManager._scene instanceof Scene_MainScene) {
        SceneManager._scene.drawSceneObjects();
      } else {
        console.warn("刷新场景对象失败，当前场景不是主场景");
      }
    });

    PluginManager.registerCommand(PluginName, 'actorMove', function(args) {
      Utils_Scene.actorMove(args.actor, args.scene);
    });
    PluginManager.registerCommand(PluginName, 'withMove', function(args) {
      Utils_Scene.withMove(args.actor, args.scene, Number(args.duration), args.isFadeout !== 'false', args.isFadein !== 'false');
    });
    
  }
})();

