//=============================================================================
// ** RPG Maker MZ - Hakubox_Module_Time.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 定制模块 - 时间日期插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MZ
 * 
 * @help
 * 【与傲娇妹妹的治愈日常】时间日期相关插件。
 * 
 * 
 * @command setTime
 * @text 设置时间
 * @desc 设置时间
 * 
 * @arg time
 * @text 分钟数
 * @desc 分钟数
 * @type number
 * 
 * @arg operation
 * @text 操作类型
 * @desc 操作类型
 * @type select
 * @option 增加 - add
 * @value add
 * @option 减少 - minus
 * @value minus
 * @option 设为固定值 - set
 * @value set
 * @default add
 * 
 * 
 * @command setDay
 * @text 设置天数
 * @desc 设置天数
 * 
 * @arg day
 * @text 天数
 * @desc 天数
 * @type number
 * 
 * @arg operation
 * @text 操作类型
 * @desc 操作类型
 * @type select
 * @option 增加 - add
 * @value add
 * @option 减少 - minus
 * @value minus
 * @option 设为固定值 - set
 * @value set
 * @default add
 * 
 * 
 * 
 * @command timeElapse
 * @text 时间流逝
 * @desc 时间流逝（分钟）
 * 
 * @arg time
 * @text 流逝分钟数
 * @type number
 * 
 * @arg mask
 * @text 是否显示遮罩
 * @desc 是否显示遮罩
 * @type boolean
 * @default true
 * 
 * @arg delay
 * @text 延迟时间
 * @desc 延迟时间（帧）
 * @type number
 * @default 10
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * @arg addLive
 * @text 是否恢复体力
 * @desc 是否恢复体力
 * @type boolean
 * @on 恢复体力
 * @off 不恢复
 * @default false
 * 
 * @arg force
 * @text 是否强制流逝
 * @desc 是否强制流逝
 * @type boolean
 * @on 强制流逝
 * @off 不强制
 * @default false
 * 
 * 
 * @command dayElapse
 * @text 天数流逝
 * 
 * @arg day
 * @text 流逝天数
 * @type number
 * 
 * @arg mask
 * @text 是否显示遮罩
 * @desc 是否显示遮罩
 * @type boolean
 * @default true
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * @command elapseEventTo
 * @text 时间流逝到指定时间
 * @desc 时间流逝到指定时间
 * 
 * @arg hour
 * @text 时刻
 * @desc 时刻（24小时制）
 * @type number
 * 
 * @arg minute
 * @text 分钟
 * @desc 分钟，默认为0
 * @type number
 * 
 * @arg mask
 * @text 是否显示遮罩
 * @desc 是否显示遮罩
 * @type boolean
 * @default true
 * 
 * @arg delay
 * @text 延迟时间
 * @desc 延迟时间（秒）
 * @type number
 * @default 10
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * @arg addLive
 * @text 是否恢复体力
 * @desc 是否恢复体力
 * @type boolean
 * @on 恢复体力
 * @off 不恢复
 * @default false
 * 
 * @arg force
 * @text 是否强制流逝
 * @desc 是否强制流逝
 * @type boolean
 * @on 强制流逝
 * @off 不强制
 * @default false
 * 
 * 
 * @command sleepTo
 * @text 晚上睡觉
 * @desc 晚上睡觉
 * 
 * @arg delay
 * @text 延迟时间
 * @desc 延迟时间（秒）
 * @type number
 * @default 5
 * 
 * @arg isWait
 * @text 是否等待
 * @desc 是否等待
 * @type boolean
 * @on 等待
 * @off 不等待
 * @default true
 * 
 * 
 * 
 * @param time
 * @text —————— 时间 ——————
 * 
 * @param dayVariableId
 * @parent time
 * @text 天数变量ID
 * @desc 天数变量ID（从周末开始第一天）
 * @type variable
 * 
 * @param timeVariableId
 * @parent time
 * @text 时间变量ID
 * @desc 时间变量ID（分钟数）
 * @type variable
 * 
 * @param timeElapseEventId
 * @parent time
 * @text 时间流逝事件ID
 * @desc 时间流逝事件ID
 * @type common_event
 * 
 */
// #endregion
(() => {
  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_Module_Time";
  
  const typeDefine = {
  };
  
  const params = PluginParamsParser.parse(PluginManager.parameters(PluginName), typeDefine);

  /** 天数变量ID */
  const _dayVariableId = params.dayVariableId;
  /** 时间（分钟数）变量ID */
  const _timeVariableId = params.timeVariableId;
  /** 时间流逝事件ID */
  const _timeElapseEventId = params.timeElapseEventId;

  /** 星期几列表 */
  const weekList = ["星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六"];
  
  const timeModule = {
    /** 当前天数 @type {number} */
    get day() {
      return $gameVariables.value(_dayVariableId);
    },
    set day(value) {
      if (value < 0) throw new Error("天数不能为负数");
      let _day = value;
      $gameVariables.setValue(_dayVariableId, _day);
    },
    /** 当天分钟数 @type {number} */
    get time() {
      return $gameVariables.value(_timeVariableId);
    },
    set time(value) {
      let _time = value;
      if (_time < 0) throw new Error("时间不能为负数");
      if (_time >= 1440) {
        let _day = Math.floor(_time / 1440);
        _time = _time % 1440;
        this.day += _day;
      }
      $gameVariables.setValue(_timeVariableId, _time);
    },
    /** 当前星期几 @type {number} */
    get week() {
      return this.day % 7;
    },
    /** 当前是否是工作日 @type {boolean} */
    get isWorkDay() {
      return this.week !== 6 && this.week !== 0;
    },
    /** 当前小时 @type {number} */
    get hour() {
      return Math.floor(this.time / 60);
    },
    set hour(value) {
      const time = this.time % 60;
      this.time = value * 60 + time;
    },
    /** 当前分钟数 @type {number} */
    get minutes() {
      return this.time % 60;
    },
    set minutes(value) {
      const hourCount = Math.floor(this.time / 60);
      const minuteCount = this.time % 60;
      this.time = hourCount * 60 - minuteCount + value;
    },
    /** 当前时间文本 */
    get timeStr() {
      return `${('' + this.hour).padStart(2, '0')}:${('' + this.minutes).padStart(2, '0')}`;
    },
    /** 当前日期文本 */
    get dateStr() {
      return TranslateUtils.getText(`第$1天`, [this.day]);
    },
    /** 当前时间段（白天:daytime/黄昏:dusk/夜晚:night） */
    get timeSpan() {
      if (this.hour >= 6 && this.hour < 16) {
        return "daytime";
      } else if (this.hour >= 16 && this.hour < 20) {
        return "dusk";
      } else {
        return "night";
      }
    },
    /** 当前时间段（白天:daytime/黄昏:dusk/夜晚:night） */
    get timeSpanStr() {
      switch (this.timeSpan) {
        case 'daytime': return "白天";
        case 'dusk': return "黄昏";
        case 'night': return "夜晚";
        default:
          break;
      }
    },
    /** 当前星期几 */
    get week() {
      return this.day % 7;
    },
    /** 当前星期几 */
    get weekStr() {
      return TranslateUtils.getText(weekList[this.day % 7]);
    },
    /**
     * 时间流逝
     * @param {number} durationTime 持续时间（分钟）
     */
    timeElapse(durationTime = 0, config) {
      this.elapseEvent(durationTime, config);
    },
    /**
     * 天数流逝
     * @param {number} durationDay 持续天数
     */
    dayElapse(durationDay = 0, config) {
      this.elapseEvent(durationDay * 24 * 60, config);
    },
    /**
     * 自动触发事件
     */
    autoTriggerEvent() {
      if (this.day < 1) return;

      // 当小时数为7，已保存状态设为false
      if ($gameData.todayData.isSaved && this.hour == 7) {
        $gameData.todayData.isSaved = false;
      }

      // 当小时数为8，自动保存并清空所有第一次交互信息
      if (!$gameData.todayData.isSaved && this.hour === 8 && !$gameData.isEventBusy) {
        $gameData.todayData.isSaved = true;

        console.warn('[Important]自动保存并清空所有第一次交互信息')

        // 调整为非睡觉状态 及 开灯状态
        if (SceneManager._scene instanceof Scene_Home) {
          SceneManager._scene.isSleep = false;
          SceneManager._scene.isOnLight = false;
          SceneManager._scene.drawSceneObjects();
        }
        
        $gameData.todayData.haveDinner = false;
        $gameData.todayData.havBath = false;

        // 1. 自动保存
        if (this.day % 7 == 0 && this.day > 1) {
          // 自动保存（按周）
          autoSave(2);
        }
        // 自动保存（按天）
        autoSave(1);
        
        // 商店进货（第一次）
        if (this.day === 7) {
          // 便利店补货
          Utils_Item.addStock('chinatsu', { 1: 5, 2: 5, 3: 3, 4: 3, 5: 3, 6: 3, 7: 5, 8: 1, 9: 1 });
          // 神秘商店补货
          Utils_Item.addStock('yuki', { 12: 5, 13: 5 });
        }
        // 商店进货（第一次）【暂未编写】

        // 2. 调整想法
        if (this.minutes === 0) {
          if ($gameData.thought.kana.currentThoughtId != 0) {
            const _execTime = $gameData.thought.kana.execTime;
            const _duration = $gameData.thought.kana.duration;

            if (_execTime + _duration <= this.time) {
              $gameData.thought.kana.currentThoughtId = 0;
              $gameData.thought.kana.execTime = 0;
              $gameData.thought.kana.duration = 0;
            }
          }
        }
        
        // 3. 检查是否有新的成就
        Utils_State.checkAchievement();

        // 4. 清空当天的数据
        Utils_GameData.clearTodayData({
          isSaved: true
        });

        // 5. 还原心情
        if (Utils_Favor.kanaMood != 'normal') {
          const _day = Utils_Time.day - $gameData.totalData.kanaMoodDay;
          const _random = Math.random();
          if (_day == 1 && _random <= 0.2) {
            Utils_Favor.kanaMood = 'normal';
          } else if (_day == 2 && _random <= 0.5) {
            Utils_Favor.kanaMood = 'normal';
          } else if (_day == 3 && _random <= 0.75) {
            Utils_Favor.kanaMood = 'normal';
          } else if (_day == 4) {
            Utils_Favor.kanaMood = 'normal';
          }
        }

        // 6. 修改各项数值
        // Utils_Favor.kanaFavor += 2;
        Utils_Favor.kanaSex -= 5;
        Utils_Actor.san += 2;
      }
    },
    /**
     * 计算当前时间和预定时间差值
     * @param {*} hour 时刻
     * @param {*} [minute=0] 分钟
     */
    getTimeDiff(hour, minute) {
      let _endTime = hour * 60 + minute;
      let _startTime = Utils_Time.hour * 60 + Utils_Time.minutes;

      let _duration = 0;

      if (_endTime > _startTime) {
        _duration += _endTime - _startTime;
      } else {
        _duration += 24 * 60 - _startTime + _endTime;
      }

      return _duration;
    },
    /**
     * 时间流逝到具体时分
     * @param {number} hour 时刻
     * @param {number} [minute=0] 分钟
     * @param {object} config 配置项
     */
    elapseEventTo(hour, minute, config) {
      if (minute === undefined) minute = 0;

      if (hour == Utils_Time.hour && minute == Utils_Time.minutes) {
        timeModule.elapseEvent(undefined, config);
        return;
      } else {
        const _duration = this.getTimeDiff(hour, minute);
        timeModule.elapseEvent(_duration, config);
      }
    },
    /**
     * 事件流逝触发事件
     * @param {number} duration 流逝时长（分钟）
     * @param {object} config 配置项
     * @param {boolean} [config.mask=true] 是否显示遮罩
     * @param {number} [config.step=10] 每次变化步长
     * @param {number} [config.delay=10] 时间变动帧数
     * @param {number} [config.force=false] 是否强制跳转
     * @param {string} [config.easingType='Linear'] 动画类型
     * @param {string} [config.inout='None'] 动画出入函数
     * @param {boolean} [config.isSleep=false] 是否睡觉
     * @param {string[]} [config.qImages=[]] 睡觉图片
     * @param {boolean} [addLive=false] 是否恢复体力
     * @param {function} [config.callback] 回调函数
     */
    elapseEvent(duration, {
      mask = false, step = 10, delay = 10, force = false, easingType = 'Linear', inout = 'None', isSleep = false, isWait = true, addLive = false, interpreter, qImages = [], callback = () => {}
    } = {
      mask: false, step: 10, delay: 10, force: false, easingType: 'Linear', inout: 'None', isSleep: false, isWait: true, addLive: false, qImages: [], callback: () => {}
    }) {
      if (duration === undefined) duration = 0;

      const _maskAlpha = isSleep ? 255 : mask ? 155 : 0;
      const _initTime = this.time;

      let sleepSpine;

      let sceneChange;
      let eventTrigger;

      if (SceneManager._scene instanceof Scene_MainScene) {
        sceneChange = SceneManager._scene.checkSisterState;
        eventTrigger = SceneManager._scene.checkEventTrigger;

        if (mask || isSleep) {
          SceneManager._scene.hideGameUI();
        }
      }

      // 是否等待
      if (duration && isWait && interpreter && interpreter instanceof Game_Interpreter) {
        interpreter.wait(duration * delay / step + 20);
      }

      // 是否恢复体力
      if (addLive && duration) Utils_Actor.live += Math.floor(duration / 6);

      const _updateTime = () => {
        if (SceneManager._scene.updateTime) {
          SceneManager._scene.updateTime();
        }
      };

      const _startElapse = () => {
        // 如果是睡觉
        if (isSleep) {
          // $gameSystem.saveBgm();
          AudioManager.fadeOutBgm(2);
          AudioManager.playSe({ name: "sleep", volume: 100, pitch: 100, pan: 0 });
          // 测试Q版睡觉
          const sleepSource = SpineManager.preloadSpine("sleep");
          sleepSource.addListener(sleepSource => {
            const { spine, anime } = SpineManager.createSpine(sleepSource);
            const spineSleep = spine;
            spineSleep.visible = true;
            spineSleep.position.set(620, 420);
            SceneManager._scene.addChild(spineSleep);

            sleepSpine = spineSleep;
            // anime.timeScale = 0;
          });
        } else if (mask && qImages && qImages.length > 0) {
          SceneManager._scene.showQImage(qImages, duration * delay / step);
        }

        // 是否有遮罩
        if (mask || isSleep) {
          useTimeout(() => {
            $gameScreen._brightness = 255 - _maskAlpha;
          }, 20, (progress) => {
            $gameScreen._brightness = 255 - _maskAlpha * progress;
          });
        }
      };

      const _endElapse = () => {
        let _re = _initTime + duration;
        if (_re >= 1440) _re = _re % 1440;
        this.time = _re;
        _updateTime();

        this.autoTriggerEvent();

        useTimeout(() => {
          if (mask || isSleep) {
            $gameScreen._brightness = 255;
          }
          SceneManager._scene._isBusy = false;
          callback && callback();

          if (sleepSpine) sleepSpine.destroy();
          if (SceneManager._scene.drawFilter) SceneManager._scene.drawFilter(0);
          Utils_Event.triggerEventByTime(undefined, false, true);
        }, 20, (progress) => {
          if (mask || isSleep) {
            $gameScreen._brightness = 255 - _maskAlpha + _maskAlpha * progress;
          }
          // if (isSleep) {
          //   $gameSystem.replayBgm();
          // }
          if (sleepSpine) {
            sleepSpine.alpha = 1 - progress;
          }
        });
      };

      if ((SceneManager._scene instanceof Scene_Game) && delay && duration) {
        SceneManager._scene._isBusy = true;
        let _currentTime = 0;

        const _anime = useAnime({
          frameDelay: delay,
          frameCount: duration / step,
          autoUpdate: true,
        }, progress => {

          this.autoTriggerEvent();

          if (!force) {
            const _event = Utils_Event.triggerEventByTime(undefined, false, true);
            if (_event) {

              if (eventTrigger) eventTrigger.call(SceneManager._scene);

              $gameData.autoPlayTime = duration - _currentTime;
              $gameData.audoPlayTimeDelay = delay;
              _anime.stop();
              _endElapse();
              return;
            }
          }

          _currentTime += step;
          this.time += step;
          
          _updateTime();

          if (sceneChange) sceneChange.call(SceneManager._scene);

          if (progress >= 1) {
            _anime.stop();
            _endElapse();
          }
        });

        _startElapse();

        _anime.start();
      } else {
        const _endTime = this.time + (duration || 0);
        let _currentTime = this.time;
  
        if (!force) {
          const _event = Utils_Event.triggerEventByTime(undefined, false, true);
          if (_event) {
            $gameData.autoPlayTime = 0;
            $gameData.audoPlayTimeDelay = delay;
            _endElapse();
            return;
          }
        }

        while (_currentTime < _endTime) {
          _currentTime += step;
          this.time += step;
          _updateTime();
  
          this.autoTriggerEvent();
  
          if (!force) {
            const _event = Utils_Event.triggerEventByTime(undefined, false, true);
            if (_event) {
              $gameData.autoPlayTime = duration - _currentTime - this.time;
              $gameData.audoPlayTimeDelay = delay;
              _endElapse();
              break;
            }
          }
        }
          
        if (sceneChange) sceneChange.call(SceneManager._scene);
        
        Utils_Event.triggerEventByTime(undefined, false, true);
      }
    },
    /** 动画相关信息 */
    animeInfo: {
      /** 动画是否开始 */
      isStart: false,
      /** 动画帧计数器 */
      frameCounter: 0,
      /** 动画帧延迟 */
      frameDelay: 2,
      /** 动画当前帧索引 */
      frameIndex: 0,
      /** 动画总帧数 */
      frameCount: 20,
    },
    /** 开始动画 */
    startAnime() {

    },
    /** 结束动画 */
    endAnime() {

    },
    /** 时间更新 */
    update() {

    },
    /** 时间相关事件 */
    events: [

    ],
    /**
     * 判断时间段
     */
    on(condition, fn) {

    },
    initGameData() {
      this.day = 0;
      this.hour = 8;
      this.elapseEvent();
    },
  };

  /** 时间工具类 */
  class Utils_Time {
    /** 当前天数 @type {number} */
    static get day() {
      return timeModule.day;
    }
    static set day(value) {
      timeModule.day = value;
    }
    /** 当前分钟数 @type {number} */
    static get time() {
      return timeModule.time;
    }
    static set time(value) {
      timeModule.time = value;
    }
    static get week() {
      return timeModule.week;
    }
    static get weekStr() {
      return timeModule.weekStr;
    }
    static get hour() {
      return timeModule.hour;
    }
    static set hour(value) {
      timeModule.hour = value;
    }
    static get minutes() {
      return timeModule.minutes;
    }
    static set minutes(value) {
      timeModule.minutes = value;
    }
    static get timeStr() {
      return timeModule.timeStr;
    }
    static get dateStr() {
      return timeModule.dateStr;
    }
    /** 当前时间段（白天:daytime/黄昏:dusk/夜晚:night） */
    static get timeSpan() {
      return timeModule.timeSpan;
    }
    /** 当前时间段（白天/黄昏/夜晚） */
    static get timeSpanStr() {
      return timeModule.timeSpanStr;
    }
    /** 是否白天 */
    static isDaytime() {
      return timeModule.timeSpan === "daytime";
    }
    /** 是否黄昏 */
    static isDusk() {
      return timeModule.timeSpan === "dusk";
    }
    /** 是否夜晚 */
    static isNight() {
      return timeModule.timeSpan === "night";
    }
    /** 当前是否是工作日 @type {boolean} */
    static get isWorkDay() {
      return timeModule.isWorkDay;
    }
    /**
     * 时间流逝
     * @param {number} durationTime 持续时间（分钟）
     */
    static timeElapse(durationTime = 0, config) {
      timeModule.elapseEvent(durationTime, config);
    }
    /**
     * 时间流逝
     * @param {number} hour 小时
     * @param {number} minute 分钟
     */
    static timeElapseTo(hour, minute, config) {
      timeModule.elapseEventTo(hour, minute, config);
    }
    /**
     * 天数流逝
     * @param {number} durationDay 持续天数
     */
    static dayElapse(durationDay = 0) {
      timeModule.dayElapse(durationDay);
    }
    /**
     * 计算当前时间和预定时间差值
     * @param {*} hour 时刻
     * @param {*} [minute=0] 分钟
     */
    static getTimeDiff(hour, minute) {
      return timeModule.getTimeDiff(hour, minute);
    }
    /** 初始化游戏数据 */
    static initGameData() {
      timeModule.initGameData();
    }
  }
  window.Utils_Time = Utils_Time;


  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'setTime', (args) => {
      const _time = Number(args.time);
      const _operation = args.operation;
      switch (_operation) {
        case 'add':
          timeModule.time += _time;
          break;
        case 'sub':
          timeModule.time -= _time;
          break;
        default:
          timeModule.time = _time;
          break;
      }
      timeModule.elapseEvent(undefined, { mask: false, isWait: false });
    });
    PluginManager.registerCommand(PluginName, 'setDay', (args) => {
      const _day = Number(args.day);
      const _operation = args.operation;
      switch (_operation) {
        case 'add':
          timeModule.day += _day;
          break;
        case 'sub':
          timeModule.day -= _day;
          break;
        default:
          timeModule.day = _day;
          break;
      }
      timeModule.elapseEvent(undefined, { mask: false, isWait: false });
    });
    PluginManager.registerCommand(PluginName, 'timeElapse', function(args) {
      const _time = Number(args.time);
      const _delay = args.delay !== undefined ? Number(args.delay) : 10;
      timeModule.timeElapse(_time, {
        mask: args.mask === "true",
        delay: _delay,
        isWait: args.isWait === "true",
        addLive: args.addLive ? args.addLive === "true" : false,
        interpreter: this,
        force: args.force === "true",
      });
    });
    PluginManager.registerCommand(PluginName, 'elapseEventTo', function(args) {
      const _hour = Number(args.hour);
      const _minute = Number(args.minute || 0) || 0;
      const _delay = args.delay !== undefined ? Number(args.delay) : 10;
      timeModule.elapseEventTo(_hour, _minute, {
        mask: args.mask === "true",
        delay: _delay,
        isWait: args.isWait === "true",
        addLive: args.addLive ? args.addLive === "true" : false,
        interpreter: this,
        force: args.force === "true",
      });
    });
    PluginManager.registerCommand(PluginName, 'sleepTo', function(args) {
      const _delay = args.delay !== undefined ? Number(args.delay) : 10;
      timeModule.elapseEventTo(8, 0, {
        mask: true,
        delay: _delay,
        isWait: args.isWait === "true",
        addLive: true,
        isSleep: true,
        interpreter: this,
        force: true,
      });
    });
    PluginManager.registerCommand(PluginName, 'dayElapse', function(args) {
      const _day = Number(args.day);
      timeModule.dayElapse(_day, { mask: args.mask === "true", isWait: args.isWait === "true", interpreter: this });
    });
    
  }
})();