//=============================================================================
// ** RPG Maker MZ - Hakubox_Touch_Plugin.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc Hakubox Touch工具插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MV MZ
 *
 * @help
 * Hakubox的鼠标操作插件。
 * 
 * 注：当前插件不适宜直接使用，而是为其他插件调用基础功能。
 * 
 * 
 * ■ [ 可用脚本列表 ] .............................................
 * 以下是当前插件提供的脚本代码。
 * 
 * 〇 判断当前是否正在点击/触摸。
 * TouchPlugin.isTap(config)
 *   - config.button {0|1|2} 鼠标按键，0：左键，1：中键，2：右键
 *   - config.rect {x, y, width, height} 点击范围
 * 示例: TouchPlugin.isTap({ rect: { x: 100, y: 100, width: 200, height: 200 } })
 * 
 * 〇 判断当前是否鼠标正在划过。
 * HakuSkit.isHover(config)
 *   - config.rect {x, y, width, height} 划过的范围
 * 示例: TouchPlugin.isHover({ rect: { x: 100, y: 100, width: 200, height: 200 } })
 * 
 * 〇 判断当前是否鼠标正在连续点击/触摸。
 * HakuSkit.isRepeatTap(config)
 *   - config.delay {number} 延迟帧数
 *   - config.button {0|1|2} 鼠标按键，0：左键，1：中键，2：右键
 *   - config.rect {x, y, width, height} 点击范围
 * 示例: TouchPlugin.isRepeatTap({ delay: 300, rect: { x: 100, y: 100, width: 200, height: 200 }})
 * 
 * 〇 判断当前是否正在长按。
 * HakuSkit.isHoldTap(config)
 *   - config.duration {number} 最短按键帧数
 *   - config.button {0|1|2} 鼠标按键，0：左键，1：中键，2：右键
 *   - config.rect {x, y, width, height} 点击范围
 * 示例: TouchPlugin.isHoldTap({ duration: 300, rect: { x: 100, y: 100, width: 200, height: 200 }})
 * 
 * 〇 判断当前是否正在拖拽。
 * TouchPlugin.isDrag(config)
 *   - config.angle {number} 角度
 *   - config.distance {number} 拖拽距离
 *   - config.button {0|1|2} 鼠标按键，0：左键，1：中键，2：右键
 *   - config.rect {x, y, width, height} 点击范围
 *   - 返回值: {progress: number, speed: number, speedPixelsPerSecond: number} 或 false
 *     - progress: 拖拽进度百分比
 *     - speed: 拖拽速度 (像素/毫秒)
 *     - speedPixelsPerSecond: 拖拽速度 (像素/秒)
 * 示例: const dragInfo = TouchPlugin.isDrag({ angle: 90, distance: 100, rect: { x: 100, y: 100, width: 200, height: 200 }})
 *       if (dragInfo) { console.log('拖拽进度:', dragInfo.progress, '速度:', dragInfo.speedPixelsPerSecond, '像素/秒'); }
 * 
 * 〇 判断当前鼠标是否来回晃动/移动。（可用于摸头等动作）
 * HakuSkit.isShake(config)
 *   - config.angle {number} 角度
 *   - config.distance {number} 拖拽距离
 *   - config.button {0|1|2} 鼠标按键，0：左键，1：中键，2：右键
 *   - config.rect {x, y, width, height} 点击范围
 * 示例: TouchPlugin.isShake({ angle: 90, distance: 100, rect: { x: 100, y: 100, width: 200, height: 200 }})
 * 
 * 〇 判断当前是否正在画圈。
 * HakuSkit.isDrawRing(config)
 *   - config.count {number} 判断次数，建议设置为50左右
 *   - config.rect {x, y, width, height} 点击范围
 * 示例: TouchPlugin.isDrawRing({ count: 50 })
 * 
 * 
 * ■ [ 联系方式 ] ........................................................
 * 
 * 微信：hakubox
 * 邮箱：hakubox＠outlook.com  （需要手动将＠替换为半角字符）
 * 
 * 
 * ■ [ License ] ........................................................
 * 
 * MIT license
 * 
 * 授予任何人使用、复制、修改和分发本插件的权利。
 * 软件是按“现状”提供的，不提供任何明示或暗示的担保，包括但不限于适销性、特定
 * 用途适用性和非侵权的担保。
 * 在任何情况下，版权所有者或许可方均不对因使用本软件或其他交易而引起或与之
 * 相关的任何索赔、损害或其他责任承担责任，无论是因合同、侵权或其他原因引起。
 * 
 * 
 * ■ [ 修订历史 ] ........................................................
 *  v1.0.0  2025/01/25  初版。
 * 
 * 
 */
(() => {

  /** 鼠标操作类 */
  const handleModule = {
    /** 信息 */
    info: {
      /** X坐标 */
      x: 0,
      /** Y坐标 */
      y: 0,
      /** 按下X坐标 */
      downX: 0,
      /** 按下Y坐标 */
      downY: 0,
      /** 上次按下的时间 */
      prevTapTime: 0,
      /** 按下时间 */
      tapTime: 0,
      /** 是否刚刚才点击 */
      isTapActive: false,
      /** 鼠标键位 0：左键，1：中键，2：右键 */
      button: -1,
      /** 是否开始按下 */
      isTapDown: false,
      /** 是否按着Ctrl键 */
      ctrlKey: false,
      /** 是否按着Alt键 */
      altKey: false,
      /** 点击位置，触屏可点击多个位置 */
      tapPoints: [],
      /** 画布处于页面中的区域 */
      domRect: {
        x: 0,
        y: 0,
        width: 0,
        height: 0
      },
      /** 拖拽方向，1为正，2为负，0为未拖拽 @type {0|1|2} */
      direction: 0,
      /** 上一次的拖拽长度 */
      prevDistance: 0,
      /** 总的拖拽长度 */
      totalDistance: 0,
      /** 画圆坐标 */
      touchMoveX: [],
      touchMoveY: [],
      /** 画圆次数 */
      ringCount: 0,
      /** 上次拖拽时间 */
      lastDragTime: 0,
      /** 上次拖拽X坐标 */
      lastDragX: 0,
      /** 上次拖拽Y坐标 */
      lastDragY: 0,
      /** 当前拖拽速度 (像素/毫秒) */
      dragSpeed: 0,
      /** 拖拽速度历史记录 (用于平滑计算) */
      dragSpeedHistory: [],
    },
    // #region 鼠标相关

    /** 鼠标按下事件 */
    mousedown_event(event) {
      const _pageX = event.touches ? event.touches[0].clientX : event.clientX;
      const _pageY = event.touches ? event.touches[0].clientY : event.clientY;

      const _x = Graphics.pageToCanvasX(_pageX);
      const _y = Graphics.pageToCanvasY(_pageY);
      if (!Graphics.isInsideCanvas(_x, _y)) return;

      this.info.isTapActive = true;
      this.info.isTapDown = true;
      this.info.x = _x;
      this.info.y = _y;
      this.info.downX = _x;
      this.info.downY = _y;
      if (event.pointerType === 'mouse') this.info.button = event.button;
      else this.info.button = undefined;
      this.info.ctrlKey = event.ctrlKey;
      this.info.altKey = event.altKey;
      this.info.shiftKey = event.shiftKey;
      this.info.prevTapTime = this.info.tapTime;
      this.info.tapTime = Date.now();
    },
    /** 鼠标移动事件 */
    mousemove_event(event) {
      const _pageX = event.touches ? event.touches[0].clientX : event.clientX;
      const _pageY = event.touches ? event.touches[0].clientY : event.clientY;

      const _x = Graphics.pageToCanvasX(_pageX);
      const _y = Graphics.pageToCanvasY(_pageY);
      if (this.info.isTapDown) {
        this.info.x = _x;
        this.info.y = _y;
      } else if (Graphics.isInsideCanvas(_x, _y)) {
        this.info.x = _x;
        this.info.y = _y;
      }
    },
    /** 鼠标松开事件 */
    mouseup_event() {
      this.info.isTapDown = false;
      this.info.x = 0;
      this.info.y = 0;
      this.info.button = undefined;
    },
    // #endregion


    // #region 触屏相关

    // #endregion
    /** 校验点击位置 */
    checkRect(x, y, width, height) {
      return this.info.x >= x && this.info.x <= x + width && this.info.y >= y && this.info.y <= y + height;
    },
    /** 刷新 */
    update() {
      this.info.isTapActive = false;
    },
    /** 修改窗口大小 */
    resize() {
      this.info.domRect = Graphics._canvas.getBoundingClientRect();
    },

    /** 根据角度和距离转换点 */
    calculateNewPoint(x, y, distance, radians) {
      const newX = x + distance * Math.cos(radians);
      const newY = y + distance * Math.sin(radians);

      return { x: newX, y: newY };
    },
    /** 将新点根据中心坐标转换 */
    rotatePoint(x, y, centerX, centerY, radians) {
      const cos = Math.cos(radians);
      const sin = Math.sin(radians);

      // 将点平移到旋转中心
      const translatedX = x - centerX;
      const translatedY = y - centerY;

      // 旋转点
      const xRotated = translatedX * cos - translatedY * sin;
      const yRotated = translatedX * sin + translatedY * cos;

      // 将旋转后的点平移回原始位置
      const finalX = xRotated + centerX;
      const finalY = yRotated + centerY;

      return { x: finalX, y: finalY };
    },
  };

  // #region 系统修改

  /** 初始化函数 */
  const init = () => {
    document.addEventListener('pointerdown', handleModule.mousedown_event.bind(handleModule));
    document.addEventListener('pointermove', handleModule.mousemove_event.bind(handleModule));
    document.addEventListener('pointerup', handleModule.mouseup_event.bind(handleModule));


    window.addEventListener('resize', handleModule.resize.bind(handleModule));

    handleModule.resize();
  };

  const SceneManager_initGraphics = SceneManager.initGraphics;
  SceneManager.initGraphics = function () {
    SceneManager_initGraphics.call(this);
    init();
  }


  const Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function () {
    Scene_Base_update.call(this);
    // const rect = {
    //   x: 100, y: 100, width: 200, height: 200
    // };
    // 鼠标操作
    // if (TouchPlugin.isTap({ rect })) {
    //   console.log('点击', handleModule.info.x, handleModule.info.y);
    // }
    // if (TouchPlugin.isHover({ rect })) {
    //   console.log('划过', handleModule.info.x, handleModule.info.y);
    // }
    // if (TouchPlugin.isRepeatTap()) {
    //   console.log('连击', handleModule.info.x, handleModule.info.y);
    // }
    // if (TouchPlugin.isHoldTap()) {
    //   console.log('长按', handleModule.info.x, handleModule.info.y);
    // }
    // const dragInfo = TouchPlugin.isDrag({ distance: 100, angle: 90 });
    // if (dragInfo !== false) {
    //   console.log('拖拽进度:', dragInfo.progress, '速度:', dragInfo.speedPixelsPerSecond, '像素/秒');
    // }
    // const _progress = TouchPlugin.isShake({ distance: 500, angle: 90 });
    // if (_progress !== false) {
    //   console.log('反复拖拽滑动', _progress);
    // }
    // const _progress = TouchPlugin.isDrawRing({ count: 50 });
    // if (_progress !== false) {
    //   console.log('画圆', _progress);
    // }

    handleModule.update();
  }

  // #endregion

  /** Touch工具类 */
  class TouchPlugin {
    /** 是否一直按着鼠标 @type {boolean} */
    static isTapDown() {
      return handleModule.info.isTapDown;
    }
    /** 清空鼠标状态 @type {boolean} */
    static clearTap() {
      handleModule.info.isTapDown = false;
    }
    /**
     * 判断当前是否正在点击/触摸
     * @param {object} config 配置项
     * @param {0|1|2} [config.button] 鼠标按键，0：左键，1：中键，2：右键
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 点击范围
     */
    static isTap(config) {
      if (handleModule.info.isTapActive) {
        if (config) {
          if (config.button !== undefined && handleModule.info.button !== undefined && config.button !== handleModule.info.button) return false;
          if (config.rect && !handleModule.checkRect(config.rect.x, config.rect.y, config.rect.width, config.rect.height)) return false;
        }
        return true;
      }
      return false;
    }
    /**
     * 判断当前是否鼠标正在划过
     * @param {object} config 配置项
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 划过的范围
     */
    static isHover(config) {
      if (config) {
        if (config.rect) return handleModule.checkRect(config.rect.x, config.rect.y, config.rect.width, config.rect.height);
      }
      return false;
    }
    /**
     * 判断当前是否正在连续点击/触摸
     * @param {object} config 配置项
     * @param {number} delay 延迟帧数
     * @param {0|1|2} [config.button=0] 鼠标按键，0：左键，1：中键，2：右键
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 点击范围
     */
    static isRepeatTap(config) {
      if (handleModule.info.isTapActive) {
        let _delay = config ? config.delay || 400 : 400;
        if (handleModule.info.prevTapTime + _delay < Date.now()) return false;
        if (config) {
          if (config.button !== undefined && handleModule.info.button !== undefined && config.button !== handleModule.info.button) return false;
          if (config.rect && !handleModule.checkRect(config.rect.x, config.rect.y, config.rect.width, config.rect.height)) return false;
        }
        return true;
      }
      return false;
    }
    /**
     * 判断当前是否正在长按
     * @param {object} config 配置项
     * @param {number} duration 最短按键帧数
     * @param {0|1|2} [config.button=0] 鼠标按键，0：左键，1：中键，2：右键
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 点击范围
     */
    static isHoldTap(config) {
      if (handleModule.info.isTapDown) {
        let _duration = config ? config.duration || 300 : 300;
        if (handleModule.info.tapTime + _duration > Date.now()) return false;
        if (config) {
          if (config.button !== undefined && handleModule.info.button !== undefined && config.button !== handleModule.info.button) return false;
          if (config.rect && !handleModule.checkRect(config.rect.x, config.rect.y, config.rect.width, config.rect.height)) return false;
        }
        return true;
      }
      return false;
    }
    /**
     * 判断当前是否正在拖拽。
     * @param {object} config 配置项
     * @param {number} config.angle 角度
     * @param {number} config.minDistance 拖拽距离
     * @param {number} config.distance 拖拽距离
     * @param {0|1|2} [config.button=0] 鼠标按键，0：左键，1：中键，2：右键
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 点击范围
     * @return {object|false} 返回包含拖拽百分比和速度的对象，或false
     */
    static isDrag(config) {
      if (handleModule.info.isTapDown) {

        if (config.angle === undefined) throw new Error('缺少 angle 角度参数');
        if (config.distance === undefined) throw new Error('缺少 distance 距离参数');

        if (config) {
          if (config.button !== undefined && handleModule.info.button !== undefined && config.button !== handleModule.info.button) return false;
          if (config.rect && !handleModule.checkRect(config.rect.x, config.rect.y, config.rect.width, config.rect.height)) return false;

          const currentTime = Date.now();
          const startX = handleModule.info.downX;
          const startY = handleModule.info.downY;

          const radians = (-(config.angle - 90) * Math.PI) / 180;

          /** 坐标原点 */
          const _originPoint = handleModule.calculateNewPoint(startX, startY, 0, radians);

          /** 应该是当前拖拽到的点 */
          const _endPoint = handleModule.rotatePoint(
            handleModule.info.x, handleModule.info.y,
            _originPoint.x, _originPoint.y,
            radians
          );
          
          if (config.minDistance && (_endPoint.x - _originPoint.x) < config.minDistance) {
            return false;
          }

          // 计算拖拽进度百分比
          const _progress = ((_endPoint.x - _originPoint.x) / config.distance) * 100;

          // 计算拖拽速度
          if (handleModule.info.lastDragTime > 0) {
            const timeDiff = currentTime - handleModule.info.lastDragTime;
            if (timeDiff > 0) {
              // 计算距离差值
              const distanceDiff = Math.sqrt(
                Math.pow(handleModule.info.x - handleModule.info.lastDragX, 2) +
                Math.pow(handleModule.info.y - handleModule.info.lastDragY, 2)
              );
              
              // 计算当前速度 (像素/毫秒)
              const currentSpeed = distanceDiff / timeDiff;
              
              // 添加到历史记录用于平滑计算
              handleModule.info.dragSpeedHistory.push(currentSpeed);
              
              // 保持历史记录在合理范围内（最多保留10个记录）
              if (handleModule.info.dragSpeedHistory.length > 10) {
                handleModule.info.dragSpeedHistory.shift();
              }
              
              // 计算平均速度以平滑结果
              const avgSpeed = handleModule.info.dragSpeedHistory.reduce((sum, speed) => sum + speed, 0) / handleModule.info.dragSpeedHistory.length;
              handleModule.info.dragSpeed = avgSpeed;
            }
          }

          // 更新上次拖拽信息
          handleModule.info.lastDragTime = currentTime;
          handleModule.info.lastDragX = handleModule.info.x;
          handleModule.info.lastDragY = handleModule.info.y;

          // console.log(`两个点：${_endPoint.x}、${_originPoint.x}`);
          // console.log(`当前角度: ${radians.toFixed(2)}°`);
          // console.log(`当前距离: ${_endPoint.x - _originPoint.x}px`);
          // console.log(`拖拽进度百分比: ${_progress.toFixed(2)}%`);
          // console.log(`拖拽速度: ${handleModule.info.dragSpeed.toFixed(4)} 像素/毫秒`);

          return {
            progress: _progress,
            speed: handleModule.info.dragSpeed,
            speedPixelsPerSecond: handleModule.info.dragSpeed * 1000 // 转换为像素/秒
          };
        }
      } else {
        // 重置拖拽速度相关信息
        handleModule.info.lastDragTime = 0;
        handleModule.info.lastDragX = 0;
        handleModule.info.lastDragY = 0;
        handleModule.info.dragSpeed = 0;
        handleModule.info.dragSpeedHistory = [];
      }
      return false;
    }
    /**
     * 判断当前鼠标是否来回晃动/移动。（可用于摸头等动作）
     * @param {object} config 配置项
     * @param {number} config.angle 角度，默认为90度，即横向移动
     * @param {number} config.distance 拖拽距离
     * @param {0|1|2} [config.button=0] 鼠标按键，0：左键，1：中键，2：右键
     * @param {{x: number, y: number, width: number, height: number}} [config.moveRect] 移动范围
     * @return {number} 移动百分比
     */
    static isShake(config) {
      if (handleModule.info.isTapDown) {

        if (config.angle === undefined) throw new Error('缺少 angle 角度参数');
        if (config.distance === undefined) throw new Error('缺少 distance 距离参数');

        if (config) {
          if (config.button !== undefined && this.info.button !== undefined && config.button !== this.info.button) return false;
          if (
            config.moveRect && handleModule.info.totalDistance < 1 &&
            !handleModule.checkRect(config.moveRect.x, config.moveRect.y, config.moveRect.width, config.moveRect.height)
          ) return false;

          const startX = handleModule.info.downX;
          const startY = handleModule.info.downY;

          let radians = (-(config.angle - 90) * (handleModule.info.direction === 2 ? 1 : -1) * Math.PI) / 180;

          /** 坐标原点 */
          const _originPoint = handleModule.calculateNewPoint(
            startX, startY, 0, radians
          );

          /** 应该是当前拖拽到的点 */
          const _endPoint = handleModule.rotatePoint(
            handleModule.info.x, handleModule.info.y,
            _originPoint.x, _originPoint.y,
            radians
          );

          const _distance = _endPoint.x - _originPoint.x;

          if (handleModule.info.prevDistance > _distance) {
            handleModule.info.direction = 1;
          } else if (handleModule.info.prevDistance < _distance) {
            handleModule.info.direction = 2;
          }

          handleModule.info.totalDistance += Math.abs(_distance - handleModule.info.prevDistance);
          handleModule.info.prevDistance = _distance;

          return handleModule.info.totalDistance / config.distance * 100;
        }
      } else {
        handleModule.info.direction = 0;
        handleModule.info.totalDistance = 0;
        handleModule.info.prevDistance = 0;
      }
      return false;
    }
    /**
     * 判断当前是否正在画圈。
     * @param {object} config 配置项
     * @param {number} [config.count = 50] 判断次数，建议设置为50左右
     * @param {{x: number, y: number, width: number, height: number}} [config.rect] 点击位置
     */
    static isDrawRing(config) {
      if (handleModule.info.isTapDown) {
        if (!config.count) config.count = 50;

        if (!(
          handleModule.info.touchMoveX.includes(handleModule.info.x) && 
          handleModule.info.touchMoveY.includes(handleModule.info.y)
        )) {
          handleModule.info.touchMoveX.push(handleModule.info.x);
          handleModule.info.touchMoveY.push(handleModule.info.y);
        }
        
        
        if (handleModule.info.touchMoveX.length < 30) return false;
        let totalAmount = handleModule.info.touchMoveX.length;
        // sum up all coordinates and divide them by total length 
        // the average is a cheap approximation of the center.
        let averageX = handleModule.info.touchMoveX.reduce((a, b) => a + b) / totalAmount;
        let averageY = handleModule.info.touchMoveY.reduce((a, b) => a + b) / totalAmount;

        // compute distance to approximated center from each point
        let distances = handleModule.info.touchMoveX.map((x, index) => {
          let y = handleModule.info.touchMoveY[index];        
          return Math.sqrt(Math.pow(x - averageX, 2) + Math.pow(y - averageY, 2));
        });
        // average of those distance is 
        let averageDistance = distances.reduce((a, b) => a + b) / distances.length;

        let min = averageDistance * 0.6;
        let max = averageDistance * 1.4;
        // filter out the ones not inside the min and max boundaries 
        let inRange = distances.filter(d => d > min && d < max).length;

        let minPercentInRange = 80;
        let percentInRange = inRange / totalAmount * 100;
        // by the % of points within those boundaries we can guess if it's circle

        if (handleModule.info.touchMoveX.length > 60) {
          handleModule.info.touchMoveX = handleModule.info.touchMoveX.slice(-60);
          handleModule.info.touchMoveY = handleModule.info.touchMoveY.slice(-60);
        }

        if (percentInRange > minPercentInRange) { 
          handleModule.info.ringCount += 1;
          //it's probably a circle
          return handleModule.info.ringCount / config.count;
        }
      } else {
        handleModule.info.touchMoveX = [];
        handleModule.info.touchMoveY = [];
        handleModule.info.ringCount = 0;
      }
      return false;
    }
  }
  window.TouchPlugin = TouchPlugin;

})();
