
//=============================================================================
// ** RPG Maker MZ - Hakubox_SLG_Plugin.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc Hakubox SLG工具插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MV MZ
 *
 * @help
 * Hakubox的Galgame成套UI配置常规组件。
 * 
 * 
 */

(() => {

  /**
   * @typedef {object} HakuComponent_Style 组件样式
   * @property {string} [fontSize] 字体大小
   * @property {string} [fontColor] 字体颜色
   * @property {string} [fontWeight] 字体粗细
   * @property {string} [cursor] 光标样式
   * @property {string} [backgroundImage] 背景图片路径
   * @property {number} [transitionDuration] 切换时长
   * @property {string} [transitionTimingFunction] 切换时间函数
   */

  /**
   * @typedef {object} HakuComponent_StyleConfig 组件样式
   * @property {HakuComponent_Style} styles 样式表
   * @property {'normal'|'hover'} [type='normal'] 字体颜色
   */
  

  /**
   * @typedef {object} HakuComponent_Params 基础组件参数
   * @property {string} [id] 组件ID
   * @property {boolean} [isDisabled = false] 是否禁用
   * @property {string} [text] 显示文本
   * @property {HakuComponent_Style} [style] 样式
   * @property {string} [className] 样式类名
   * @property {string} [normalBgUrl] 正常的背景图片路径
   * @property {string} [hoverBgUrl] 鼠标划过的背景图片路径
   * @property {string} [selectedBgUrl] 选中的背景图片路径
   * @property {string} [pressedBgUrl] 按下的背景图片路径
   * @property {string} [disabledBgUrl] 禁用的背景图片路径
   */

  /**
   * @typedef {HakuComponent_Params} HakuButton_Params 按钮组件参数
   */

  /**
   * @typedef {HakuComponent_Params} HakuCheckbox_Params 复选框组件参数
   * @property {boolean} [checked = false] 是否选中
   */

  /**
   * 全局类列表
   * @type {Record<string, HakuComponent_Style>}
   */
  const Haku_ClassList = {
    'hover:btn': {}
  };

  /**
   * 
   */
  const Haku_Anime_Config = {
    /** 动画配置 */
    hover1: {
      exec(component, config) {
        config.initX = component.x;
        config.initY = component.y;
      },
      update(component, progress, config) {
        let _scale;
        if (progress <= 0.5) {
          _scale = 1 + progress * 0.5;
        } else {
          _scale = 1.5 - progress * 0.5;
        }

        let _left = config.initX - (_scale - 1) * component.width / 2;
        let _top = config.initY - (_scale - 1) * component.height / 2;

        component.scale.x = _scale;
        component.scale.y = _scale;
        component.x = _left;
        component.y = _top;
      }
    },
    /**  */
    hover2: {
      exec(component, config) {
        config.initX = component.x;
        config.initY = component.y;
      },
      update(component, progress, config) {
        const _step = 1 / 3;
        let _scaleX = 1;
        let _scaleY = 1;
        if (progress <= _step) {
          _scaleX = 1 + progress * 0.4;
          _scaleY = 1;
        } else if (progress <= _step * 2) {
          _scaleX = 1 + (_step - (progress - _step)) * 0.4;
          _scaleY = 1 + (progress - _step) * 0.4;
        } else {
          _scaleX = 1;
          _scaleY = 1 + (_step - (progress - _step * 2)) * 0.4;
        }

        let _left = config.initX - (_scaleX - 1) * component.width / 2;
        let _top = config.initY - (_scaleY - 1) * component.height / 2;

        component.scale.x = _scaleX;
        component.scale.y = _scaleY;
        component.x = _left;
        component.y = _top;
      }
    },

  }

  // #region UIHandler
  const UIHandler = {
    /**
     * 转换style
     * @param {Haku_Component} component 组件
     * @param {'tap'|'hover'|'change'} actionType 动作类型
     */
    transformClassName(component, actionType) {
      
      // 获取所有符合规则的样式类名
      const matchClasList = [];
      component._className.split(' ').map(fullClassName => {
        const [prefix, className] = fullClassName.split(':');

        if (!prefix) {
          matchClasList.push(className);
        } else if (prefix === 'tap' && component.isTap && (!actionType || actionType === 'tap')) {
          matchClasList.push(className);
        } else if (prefix === 'hover' && component.isHover && (!actionType || actionType === 'hover')) {
          matchClasList.push(className);
        } else if (prefix === 'checked' && component.isChecked && (!actionType || actionType === 'checked')) {
          matchClasList.push(className);
        }
        
      });

      // 获取所有样式
      const _globalStyles = {};
      for (let i = 0; i < matchClasList.length; i++) {
        if (!matchClasList[i]) continue;

        const _styles = Haku_ClassList[matchClasList[i]];

        Object.entries(_styles).forEach(([styleName, styleValue]) => {
          _globalStyles[styleName] = styleValue;
        });
      }

      Object.entries(component.style).forEach(([styleName, styleValue]) => {
        _globalStyles[styleName] = styleValue;
      });
      if (component.isHover && component['hover:style'] && (!actionType || actionType === 'hover')) {
        Object.entries(component['hover:style']).forEach(([styleName, styleValue]) => {
          _globalStyles[styleName] = styleValue;
        });
      }
      if (component.isTap && component['tap:style'] && (!actionType || actionType === 'tap')) {
        Object.entries(component['tap:style']).forEach(([styleName, styleValue]) => {
          _globalStyles[styleName] = styleValue;
        });
      }
      if (component.isChecked && component['checked:style'] && (!actionType || actionType === 'checked')) {
        Object.entries(component['checked:style']).forEach(([styleName, styleValue]) => {
          _globalStyles[styleName] = styleValue;
        });
      }
      if (component.isHover && component.isChecked && component['checked:hover:style'] && (!actionType || actionType === 'checked')) {
        Object.entries(component['checked:hover:style']).forEach(([styleName, styleValue]) => {
          _globalStyles[styleName] = styleValue;
        });
      }

      return _globalStyles;
    },
    /**
     * 更新动画
     * @param {Haku_Component} component 组件
     * @param {number} duration 切换时长（帧）
     */
    updateComponentStyle(component, duration) {

    },
  }
  // #endregion

  // #region 基础组件
  /**
   * Hakubox基础组件
   */
  class Haku_Component extends PIXI.Container {
    /**
     * @param {HakuComponent_Params} config 参数
     */
    constructor(config) {
      super();
      this._config = config;
      this._isDisabled = config.isDisabled || false;
      /** 组件ID */
      this._id = config.id || this.getRandomId();
      this._width = config.width || 0;
      this._height = config.height || 0;
      /** 事件 @type {Record<string, Function[]>} */
      this.events = {};
      /** 样式类名 */
      this._className = config.className || "";
      /** 样式 */
      this._style = config.style || {};
      /** hover样式 */
      this['hover:style'] = config['hover:style'] || {};
      /** tap样式 */
      this['tap:style'] = config['tap:style'] || {};
      /** 显示文本 */
      this._text = config.text || "";
      /** 是否为点击状态 */
      this._isTap = false;
      /** 是否为鼠标移入状态 */
      this._isHover = false;

      /** 背景图片 */
      this.backgroundImage = new Sprite(new Bitmap(this.width, this.height));

      if (config.width) {
        this.width = config.width;
      }
      if (config.height) {
        this.height = config.height;
      }

      /** 动画配置 */
      this.animeConfig = {
        isStart: false,
        frameCounter: 0,
        frameDelay: 0,
        frameIndex: 0,
        frameCount: 0,
        config: {},
        type: ''
      };

      this.addChild(this.backgroundImage);
    }

    /** 获取随机ID */
    getRandomId() {
      return Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);
    }

    // initialize(config) {
    // }

    /**
     * 指定某一类动画开始
     * @param {'hover'|'tap'|'change'} actionType 动作类型
     */
    startAnime(actionType) {
      if (!this.animeConfig.isStart) {
        const styles = UIHandler.transformClassName(this, actionType);
        if (styles.animeName) {
          this.animeConfig.type = styles.animeName;
          this.animeConfig.isStart = true;
          this.animeConfig.frameCounter = 0;
          this.animeConfig.frameDelay = 2;
          this.animeConfig.frameIndex = 0;
          this.animeConfig.frameCount = 10;
          this.animeConfig.config = {};
        }
      }
    }

    stopAnime() {
      this.animeConfig.isStart = false;
      this.animeConfig.frameCounter = 0;
      this.animeConfig.frameIndex = 0;
      this.animeConfig.config = {};
    }

    /** 是否按下 */
    get isTap() {
      return this._isTap;
    }
    set isTap(value) {
      if (value === this._isTap) return;
      this._isTap = value;
      if (value) {
        this.startAnime('tap');
      }
    }
    /** 是否光标在上方 */
    get isHover() {
      return this._isHover;
    }
    set isHover(value) {
      if (value === this._isHover) return;
      this._isHover = value;
      if (value) {
        SoundManager.playCursor();
        this.startAnime('hover');
      }
    }

    get style() {
      return this._style;
    }
    set style(value) {
      this._style = value;
    }



    /** 样式类名 */
    get className() {
      return this._className;
    }
    set className(value) {
      value.split(' ').forEach(fullClassName => {
        const [prefix, className] = fullClassName.split(':');
        if (!Haku_ClassList[className]) {
          console.warn(`不存在的类名：${className}`);
          return;
        }
      });

      this._className = value;
    }

    /** 更新动画 */
    updateAnime() {
      if (!this.animeConfig.type || !Haku_Anime_Config[this.animeConfig.type]) return;
      if (!this.animeConfig.isStart) return;

      if (this.animeConfig.frameCounter === 0 && this.animeConfig.frameIndex === 0) {
        if (Haku_Anime_Config[this.animeConfig.type].exec) {
          Haku_Anime_Config[this.animeConfig.type].exec(this, this.animeConfig.config);
        }
      }
      
      this.animeConfig.frameCounter++;
      if (this.animeConfig.frameCounter >= this.animeConfig.frameDelay) {
        this.animeConfig.frameCounter = 0;
        
        if (Haku_Anime_Config[this.animeConfig.type].update) {
          Haku_Anime_Config[this.animeConfig.type].update(this, this.animeConfig.frameIndex / this.animeConfig.frameCount, this.animeConfig.config);
        }

        this.animeConfig.frameIndex++;
        if (this.animeConfig.frameIndex > this.animeConfig.frameCount) {
          if (this.animeConfig.loop) {
            this.animeConfig.frameIndex = 0;
          } else {
            this.animeConfig.isStart = false;
            this.animeConfig.type = '';
          }
        }
      }
    }

    /** 宽度 @type {number} */
    get width() {
      return this._width;
    }
    set width(value) {
      this._width = value;
      this.refreshSprite();
    }
    /** 高度 @type {number} */
    get height() {
      return this._height;
    }
    set height(value) {
      this._height = value;
      this.refreshSprite();
    }

    /** 宽高度调整时调整精灵图 */
    refreshSprite() {
      this.backgroundImage.bitmap = new Bitmap(this.width, this.height);
    }

    /**
     * 切换背景图
     * @param {string} url 图片地址
     */
    changeBackgroundImage(url) {
      if (!url || url == this.backgroundImage.bitmap._imageUrl) return;

      const _fileNameIndex = url.lastIndexOf('/');
      const _folder = url.substring(0, _fileNameIndex + 1);
      const _fileName = url.substring(_fileNameIndex + 1);

      this.backgroundImage.bitmap._imageUrl = url;
      this.backgroundImage.bitmap.smooth = false;
      const bitmap = ImageManager.loadBitmap(_folder, _fileName);
      if (bitmap.isReady()) {
        this.backgroundImage.bitmap.clear();
        this.backgroundImage.bitmap.blt(bitmap, 
          0, 0, bitmap.width, bitmap.height, 
          0, 0, this.backgroundImage.width, this.backgroundImage.height
        );
      } else {
        bitmap.addLoadListener(() => {
          this.backgroundImage.bitmap.clear();
          this.backgroundImage.bitmap.blt(bitmap, 
            0, 0, bitmap.width, bitmap.height, 
            0, 0, this.backgroundImage.width, this.backgroundImage.height
          );
        });
      }
    }

    /** 是否焦点 */
    get isActive() {
      return this._isActive;
    }

    /** 是否禁用 */
    get isDisabled() {
      return this._isDisabled;
    }

    destroy() {
      super.destroy();
    }

    /**
     * 设置事件
     * @param {'tap'|'keydown'|'hover'} eventType 事件类型
     * @param {Function} callback 回调函数 
     */
    on(eventType, callback) {
      if (!this.events[eventType]) {
        this.events[eventType] = [];
      }
      this.events[eventType].push(callback);
    }

    /**
     * 触发事件
     * @param {'tap'|'keydown'|'hover'|'mouseenter'|'mouseout'|'change'} eventType 事件类型
     * @param {object} params 参数
     */
    emit(eventType, params) {
      if (this.events[eventType]) {
        for (let i = 0; i < this.events[eventType].length; i++) {
          const _callback = this.events[eventType][i];
          _callback.call(this, params);
        }
      }
    }

    /** 判断重叠 */
    checkColl() {
      const _mouseX = TouchInput.x;
      const _mouseY = TouchInput.y;

      const _width = this.width;
      const _height = this.height;
      const _x = this.x - _width * this.backgroundImage.anchor.x;
      const _y = this.y - _height * this.backgroundImage.anchor.y;

      if (_mouseX >= _x && _mouseX <= _x + _width && _mouseY >= _y && _mouseY <= _y + _height) {
        return {
          x: _mouseX - _x,
          y: _mouseY - _y
        };
      }

      return false;
    }

    draw() {
      const styles = UIHandler.transformClassName(this);

      for (let key in styles) {
        if (key === 'backgroundImage') {
          this.changeBackgroundImage(styles[key]);
        }
      }
    }

    update() {
      if (!this.isDisabled) {
        const _isInArea = this.checkColl();

        // Hover相关
        if (_isInArea) {
          if (!this.isHover) {
            this.emit('hover');
            this.isHover = true;
          }
        } else if (this._isHover) {
          this.isHover = false;
        }

        // Tap相关
        if (_isInArea) {
          if (TouchInput.isTriggered() && !this.isTap) {
            this.emit('tap');
            this.isTap = true;
          } else if (!TouchInput.isPressed() && this.isTap) {
            this.isTap = false;
          }
        } else if (!TouchInput.isPressed() && this.isTap) {
          this.isTap = false;
        }
      }
    
      this.draw();

      this.updateAnime();
    }
  }
  // #endregion

  // #region 按钮
  /** Hakubox标准按钮 */
  class Haku_Button extends Haku_Component {

    /**
     * @param {HakuButton_Params} config 参数
     */
    constructor(config) {
      super(config);

      this._textSprite = new Sprite(new Bitmap(this.width, this.height + 10));
      this.addChild(this._textSprite);
    }

    refreshSprite() {
      this.backgroundImage.bitmap = new Bitmap(this.width, this.height);
      this._textSprite.bitmap = new Bitmap(this.width, this.height + 10);
    }

    draw() {
      super.draw();
      if (!this._textSprite || !this._textSprite.bitmap) return;

      this._textSprite.bitmap.clear();
      
      const styles = UIHandler.transformClassName(this);

      if (styles.textColor) {
        this._textSprite.bitmap.textColor = styles.textColor || this.style.textColor || '#FFFFFF';
      }
      if (styles.fontSize) {
        this._textSprite.bitmap.fontSize = styles.fontSize || this.style.fontSize || 24;
      }

      const _textX = styles.textX || this.style.textX || 0;
      const _textY = styles.textY || this.style.textY || 0;

      this._textSprite.bitmap.drawText(this._text, _textX, _textY,
        this.width - _textX, this.height - _textY, styles.textAlign || 'center'
      );

      if (!styles.fontSize) this._textSprite.bitmap.fontSize = this._style.fontSize || 24;
      if (!styles.textColor) this._textSprite.bitmap.textColor = this._style.textColor || '#FFFFFF';

    }
  }
  // #endregion

  // #region 复选框
  /** Hakubox标准复选框 */
  class Haku_Checkbox extends Haku_Button {

    /**
     * @param {HakuCheckbox_Params} config 参数
     */
    constructor(config) {
      super(config);
      /** tap样式 */
      this['checked:style'] = config['checked:style'] || {};
      this['checked:hover:style'] = config['checked:hover:style'] || {};
      this._isChecked = config.isChecked || false;
      this.on('tap', () => {
        this.isChecked = !this.isChecked;
      });
    }

    /** 是否选中 */
    get isChecked() {
      return this._isChecked;
    }
    set isChecked(value) {
      if (value === this._isChecked) return;
      this._isChecked = value;
      this.emit('change', this._isChecked);
    }

    draw() {
      super.draw();
      
      const styles = UIHandler.transformClassName(this);
    }
  }
  // #endregion

  // #region 滑块
  /** Hakubox标准滑块 */
  class Haku_Silder extends Haku_Component {

    /**
     * @param {HakuSilder_Params} config 参数
     */
    constructor(config) {
      super(config);
      /** tap样式 */
      this._max = config.max || 100;
      this._min = config.min || 0;
      this._value = config.value || this.min;

      this.textWidth = config.textWidth || 150;
      /** 使用用户指定的文本 */
      this.useUserText = !!config.text;
      this.text = config.text || "";
      
      /** 文本是否初始化了 */
      this.isInitText = false;

      /** 拖拽条底图精灵 */
      this._thumbSprite = new Sprite(new Bitmap(this.width - this.textWidth, this.height));
      this.addChild(this._thumbSprite);

      /** 拖拽条滑块精灵 */
      this._trackSprite = new Sprite(new Bitmap(this.width - this.textWidth, this.height));
      this.addChild(this._trackSprite);

      /** 拖拽条滑块精灵 */
      this._barSprite = new Sprite(new Bitmap(32, 36));
      this.addChild(this._barSprite);

      this.hideText = config.hideText || false;

      if (!this.hideText) {
        const fontSize = config.fontSize || 24;
        /** 文本图片 */
        this._textSprite = new Sprite(new Bitmap(this.textWidth, fontSize));
        this._textSprite.x = this.width - this.textWidth;
        this._textSprite.y = (this.height - fontSize) / 2 - 2;
        this.addChild(this._textSprite);
      }

      this.loadSprite(this._thumbSprite, config.thumbImage, 'thumbOriginBitmap');
      this.loadSprite(this._trackSprite, config.trackImage, 'trackOriginBitmap');
      this.loadSprite(this._barSprite, config.barImage, 'barOriginBitmap');
    }

    refreshSprite() {
      if (this.backgroundImage) this.backgroundImage.bitmap = new Bitmap(this.width, this.height);
      if (this._thumbSprite) this._thumbSprite.bitmap = new Bitmap(this.width - this.textWidth, this.height);
      if (this._barSprite) {
        this._barSprite.bitmap = new Bitmap(32, 36);
        this._barSprite.y = 0 - ((36 - this.height) / 2);
      }

      if (this._trackSprite) this._trackSprite.bitmap = new Bitmap(this.width - this.textWidth, this.height);
      if (this._textSprite) this._textSprite.bitmap = new Bitmap(this.width - this.textWidth, this._config.fontSize || 24);
      if (this._textSprite) {
        const fontSize = this._config.fontSize || 24;
        this._textSprite.x = this.width - this.textWidth;
        this._textSprite.y = (this.height - fontSize) / 2 - 2;
      }
    }

    /** 加载图片（不参与重绘） */
    loadSprite(sprite, url, originBitmap) {
      if (!url) {
        throw new Error('图片地址不能为空');
      }

      if (url.startsWith('/')) url = url.substring(1);

      const _fileNameIndex = url.lastIndexOf('/');
      const _folder = url.substring(0, _fileNameIndex + 1);
      const _fileName = url.substring(_fileNameIndex + 1);

      sprite.bitmap._imageUrl = url;
      sprite.bitmap.smooth = false;
      this[originBitmap] = ImageManager.loadBitmap(_folder, _fileName);
      if (this[originBitmap].isReady()) {
        sprite.bitmap.clear();
        sprite.bitmap.blt(this[originBitmap], 
          0, 0, this[originBitmap].width, this[originBitmap].height, 
          0, 0, sprite.width, sprite.height
        );
      } else {
        this[originBitmap].addLoadListener(() => {
          // sprite.bitmap = new Bitmap(this.width, this.height);
          sprite.bitmap.clear();
          sprite.bitmap.blt(this[originBitmap], 
            0, 0, this[originBitmap].width, this[originBitmap].height, 
            0, 0, sprite.width, sprite.height
          );
        });
      }
    }

    /** 最大值 @type {number} */
    get max() {
      return this._max;
    }
    set max(value) {
      if (value === this._max) return;
      const _max = Math.max(value, this.min);
      this._max = _max;
      if (this.value > _max) {
        this.value = _max;
      }
    }

    /** 最小值 @type {number} */
    get min() {
      return this._min;
    }
    set min(value) {
      if (value === this._min) return;
      const _min = Math.min(value, this.max);
      this._min = _min;
      if (this.value < _min) {
        this.value = _min;
      }
    }

    /** 当前值 @type {number} */
    get value() {
      return this._value;
    }
    set value(value) {
      if (value === this._value) return;
      const _value = Math.min(Math.max(value, this.min), this.max);
      this._value = _value;
      this.emit('change', this._value);
    }

    draw() {
      super.draw();
    }

    /** 判断重叠 */
    checkColl() {
      const _mouseX = TouchInput.x;
      const _mouseY = TouchInput.y;

      const _width = this.width;
      const _height = this.height + 8;
      const _x = this.x - _width * this.backgroundImage.anchor.x;
      const _y = this.y - _height * this.backgroundImage.anchor.y - 4;

      if (_mouseX >= _x && _mouseX <= _x + _width && _mouseY >= _y && _mouseY <= _y + _height) {
        return {
          x: _mouseX - _x,
          y: _mouseY - _y
        };
      }

      return false;
    }
    
    update() {
      super.update();

      if (!this.barOriginBitmap || !this.barOriginBitmap.isReady()) return;

      
      this.loadSprite(this._thumbSprite, this._config.thumbImage, 'thumbOriginBitmap');

      if (this._isHover || this._isTap) {
        this.loadSprite(this._barSprite, this._config.barImage_hover, 'barOriginBitmap');
      } else {
        this.loadSprite(this._barSprite, this._config.barImage, 'barOriginBitmap');
      }

      if (this._isTap) {
        const _x = TouchInput.x - (this.x - this.width * this.backgroundImage.anchor.x);

        const _percent = _x / (this.width - this.textWidth);
        
        const _val = Math.round(_percent * (this.max - this.min) + this.min);
      
        if (_val !== this.value) {
          this.value = _val;
          this.drawBar();
          this.drawText();
        }
      }

      if (!this.isInitText) {
        this.isInitText = true;
        this.drawBar();
        this.drawText();
      }

    }

    drawBar() {
      const _percent = (this.value - this.min) / (this.max - this.min);

      this._trackSprite.bitmap.clear();
      this._trackSprite.bitmap.blt(this.trackOriginBitmap, 
        0, 0, this.trackOriginBitmap.width * _percent, this.trackOriginBitmap.height, 
        0, 0, this._trackSprite.width * _percent, this._trackSprite.height
      );
      this._barSprite.x = this._trackSprite.width * _percent - 32 / 2;
    }

    drawText() {
      const styles = UIHandler.transformClassName(this);
          
      if (!this.hideText) {
        // /{0}/g 的正则表达式会提示报错
        const _text = (styles.text ? styles.text.replace(/\{0\}/g, this.value) : this.value) + '%';
        const fontSize = styles.fontSize || 24;

        this._textSprite.bitmap.fontSize = fontSize;
        this._textSprite.bitmap.fontBold = styles.fontBold || true;
        this._textSprite.bitmap.textColor = styles.fontColor || '#FFFFFF';
        this._textSprite.bitmap.outlineWidth = 2;
        this._textSprite.bitmap.outlineColor = '#FFFFFF';
        this._textSprite.bitmap.clear();
        this._textSprite.bitmap.drawText(_text, 0, 0,
          this.textWidth, fontSize, styles.textAlign || 'left'
        );
      }
    }
  }
  // #endregion

  // #region 单选按钮组
  /** Hakubox标准单选按钮组 */
  class Haku_RadioButton extends Haku_Component {

    /**
     * @param {HakuSilder_Params} config 参数
     */
    constructor(config) {
      super(config);
      /** tap样式 */
      this._max = config.max || 100;
      this._min = config.min || 0;
      this._value = config.value || this.min;

      this.textWidth = config.textWidth || 150;
      /** 使用用户指定的文本 */
      this.useUserText = !!config.text;
      this.text = config.text || "";
      this.items = config.items || [];

      this.hideText = config.hideText || false;

      if (!this.hideText) {
        const fontSize = config.fontSize || 24;
        /** 文本图片 */
        this._textSprite = new Sprite(new Bitmap(this.textWidth, fontSize));
        this._textSprite.x = this.width - this.textWidth;
        this._textSprite.y = (this.height - fontSize) / 2 - 2;
        this.addChild(this._textSprite);
      }

      this.drawItems();
    }

    /** 绘制所有项 */
    drawItems(isInit = true) {

      for (let i = 0; i < this.items.length; i++) {
        
        const _colIndex = i % this._config.maxCol;
        const _rowIndex = Math.floor(i / this._config.maxCol);

        const item = this.items[i];
        const _url = this.value === item.value ? item.checked : item.unchecked;
        const _fileNameIndex = _url.lastIndexOf('/');
        const _folder = _url.substring(0, _fileNameIndex + 1);
        const _fileName = _url.substring(_fileNameIndex + 1);
        const _realBitmap = ImageManager.loadBitmap(_folder, _fileName);
        let _sprite;
        if (isInit) {
          const _bitmap = new Bitmap(this._config.itemWidth, this._config.itemHeight);
          _sprite = new Sprite(_bitmap);
          _sprite.x = this.x + this._config.padding + (this._config.itemWidth + this._config.itemPadding[0]) * _colIndex;
          _sprite.y = this.y + this._config.padding + (this._config.itemHeight + this._config.itemPadding[1]) * _rowIndex;
        } else {
          _sprite = item.sprite;
        }
  
        _sprite.bitmap._imageUrl = _url;
        _sprite.bitmap.smooth = false;
        if (_realBitmap.isReady()) {
          _sprite.bitmap.clear();
          _sprite.bitmap.blt(_realBitmap, 
            0, 0, _realBitmap.width, _realBitmap.height, 
            0, 0, _sprite.width, _sprite.height
          );
        } else {
          _realBitmap.addLoadListener(() => {
            _sprite.bitmap.clear();
            _sprite.bitmap.blt(_realBitmap, 
              0, 0, _realBitmap.width, _realBitmap.height, 
              0, 0, _sprite.width, _sprite.height
            );
          });
        }
        if (isInit) {
          item.sprite = _sprite;
          this.addChild(_sprite);
        }
      }
    }

    refreshSprite() {
      if (this.backgroundImage) this.backgroundImage.bitmap = new Bitmap(this.width, this.height);
    }

    /** 当前值 @type {number} */
    get value() {
      return this._value;
    }
    set value(value) {
      if (value === this._value) return;
      this._value = value;
      this.drawItems(false);
      this.emit('change', this._value);
    }

    draw() {
      super.draw();
    }

    /**
     * 判断重叠
     * @param {object} item 项
     * @param {number} index 项索引
     */
    checkCollItem(item, index) {
      const _mouseX = TouchInput.x;
      const _mouseY = TouchInput.y;

      const _width = this._config.itemWidth;
      const _height = this._config.itemHeight;
      
      const _colIndex = index % this._config.maxCol;
      const _rowIndex = Math.floor(index / this._config.maxCol);
      
      const _x = this.x + this._config.padding + (this._config.itemWidth + this._config.itemPadding[0]) * _colIndex;
      const _y = this.y + this._config.padding + (this._config.itemHeight + this._config.itemPadding[1]) * _rowIndex;

      if (_mouseX >= _x && _mouseX <= _x + _width && _mouseY >= _y && _mouseY <= _y + _height) {
        return {
          x: _mouseX - _x,
          y: _mouseY - _y
        };
      }

      return false;
    }

    changeImg(item, imgSrc) {
      if (item && item.sprite) {
        const _url = imgSrc;
        const _fileNameIndex = _url.lastIndexOf('/');
        const _folder = _url.substring(0, _fileNameIndex + 1);
        const _fileName = _url.substring(_fileNameIndex + 1);
        const _realBitmap = ImageManager.loadBitmap(_folder, _fileName);
        item.sprite.bitmap._imageUrl = _url;
        item.sprite.bitmap.smooth = false;
        if (_realBitmap.isReady()) {
          item.sprite.bitmap.clear();
          item.sprite.bitmap.blt(_realBitmap, 
            0, 0, _realBitmap.width, _realBitmap.height, 
            0, 0, item.sprite.width, item.sprite.height
          );
        } else {
          _realBitmap.addLoadListener(() => {
            item.sprite.bitmap.clear();
            item.sprite.bitmap.blt(_realBitmap, 
              0, 0, _realBitmap.width, _realBitmap.height, 
              0, 0, item.sprite.width, item.sprite.height
            );
          });
        }
      }
    }
    
    update() {
      super.update();

      let i = 0;
      for (; i < this.items.length; i++) {
        const item = this.items[i];
        const _inArea = this.checkCollItem(item, i);
        
        if (_inArea) {
          if (this._hoverValue !== item.value) {
            SoundManager.playCursor();
          }
          this._hoverValue = item.value;
          if (this.value === item.value) {
            this.changeImg(item, item.checked_hover);
          } else {
            this.changeImg(item, item.unchecked_hover);
          }
        } else {
          if (this.value === item.value) {
            this.changeImg(item, item.checked);
          } else {
            this.changeImg(item, item.unchecked);
          }

          if (this._hoverValue === item.value) {
            this._hoverValue = '';
          }
        }

        if (_inArea && item.value !== this.value && TouchInput.isPressed()) {
          this.value = item.value;
          SoundManager.playOk();
        }
      }
    }

    drawBar() {
      const _percent = (this.value - this.min) / (this.max - this.min);

      this._trackSprite.bitmap.clear();
      this._trackSprite.bitmap.blt(this.trackOriginBitmap, 
        0, 0, this.trackOriginBitmap.width * _percent, this.trackOriginBitmap.height, 
        0, 0, this._trackSprite.width * _percent, this._trackSprite.height
      );
      this._barSprite.x = this._trackSprite.width * _percent - 32 / 2;
    }

    drawText() {
      const styles = UIHandler.transformClassName(this);
          
      if (!this.hideText) {
        const _text = (styles.text ? styles.text.replace(/\{0\}/g, this.value) : this.value) + '%';
        const fontSize = styles.fontSize || 24;

        this._textSprite.bitmap.fontSize = fontSize;
        this._textSprite.bitmap.fontBold = styles.fontBold || true;
        this._textSprite.bitmap.textColor = styles.fontColor || '#FFFFFF';
        this._textSprite.bitmap.outlineWidth = 2;
        this._textSprite.bitmap.outlineColor = '#FFFFFF';
        this._textSprite.bitmap.clear();
        this._textSprite.bitmap.drawText(_text, 0, 0,
          this.textWidth, fontSize, styles.textAlign || 'left'
        );
      }
    }
  }
  // #endregion

  // #region 对话框
  /** Hakubox标准对话框 */
  class Haku_Dialog extends Haku_Component {
    constructor(config) {
      super(config);

      this.alpha = 1;
      this._visible = false;

      this.textWidth = config.textWidth || (this.width - 100);
      
      /** 文本内容精灵 */
      this._textSprite = new Sprite(new Bitmap(this.width, this.height));
      this.addChild(this._textSprite);

      /** 遮罩精灵 */
      this._maskSprite = new Sprite(new Bitmap(Graphics.width, Graphics.height));
      this._maskSprite.x = -this.x;
      this._maskSprite.y = -this.y;
      this.addChild(this._maskSprite);

      /** 按钮间距 */
      const _btnPadding = (this.width - config.confirmConfig.width - config.cancelConfig.width) / 3;

      const confirmButton = new Haku_Button(config.confirmConfig);
      confirmButton.width = config.confirmConfig.width;
      confirmButton.height = config.confirmConfig.height;
      confirmButton.x = config.confirmConfig.x || _btnPadding;
      confirmButton.y = config.confirmConfig.y || 0;
      confirmButton.on('tap', () => {
        this.emit('confirm');
      });
      this.addChild(confirmButton);
      this.confirmButton = confirmButton;
      
      const cancelButton = new Haku_Button(config.cancelConfig);
      cancelButton.width = config.cancelConfig.width;
      cancelButton.height = config.cancelConfig.height;
      cancelButton.x = config.confirmConfig.x || (_btnPadding * 2 + config.confirmConfig.width);
      cancelButton.y = config.confirmConfig.y || 0;
      cancelButton.on('tap', () => {
        this.emit('cancel');
        this.hide();
      });
      this.addChild(cancelButton);
      this.cancelButton = cancelButton;

      this.loadSprite(this._maskSprite, config.maskImage);
      this.loadSprite(this.backgroundImage, config.style.backgroundImage);
    }
    

    /** 是否显示 @type {boolean} */
    get visible() {
      return this._visible;
    }
    set visible(value) {
      if (value === this._visible) return;
      this._visible = value;
      if (value) {
        this.show();
      }
    }
    
    refreshSprite() {
      this.x = (Graphics.width - this._config.width) / 2;
      this.y = (Graphics.height - this._config.height) / 2;
      this.backgroundImage.bitmap = new Bitmap(this.width, this.height);
      
      this._textSprite.bitmap = new Bitmap(this._config.textWidth, 100);
      this._textSprite.x = (this.width - this.textWidth) / 2;
      this._textSprite.y = 50;

      this._maskSprite.x = -this.x;
      this._maskSprite.y = -this.y;
    }
    
    /** 加载图片（不参与重绘） */
    loadSprite(sprite, url) {
      if (!url) {
        throw new Error('图片地址不能为空');
      }
      
      if (url.startsWith('/')) url = url.substring(1);

      const _fileNameIndex = url.lastIndexOf('/');
      const _folder = url.substring(0, _fileNameIndex + 1);
      const _fileName = url.substring(_fileNameIndex + 1);

      sprite.bitmap._imageUrl = url;
      sprite.bitmap.smooth = false;
      const bitmap = ImageManager.loadBitmap(_folder, _fileName);
      if (bitmap.isReady()) {
        sprite.bitmap.clear();
        sprite.bitmap.blt(bitmap, 
          0, 0, bitmap.width, bitmap.height, 
          0, 0, sprite.width, sprite.height
        );
      } else {
        bitmap.addLoadListener(() => {
          sprite.bitmap.clear();
          sprite.bitmap.blt(bitmap, 
            0, 0, bitmap.width, bitmap.height, 
            0, 0, sprite.width, sprite.height
          );
        });
      }
    }

    /** 显示对话框 */
    show() {
      this.alpha = 1;
    }
    hide() {
      this.alpha = 0;
    }

    update() {

    }

  }
  // #endregion

  

  // #region 测试

  const Scene_Base_createWindowLayer = Scene_Base.prototype.createWindowLayer;
  Scene_Base.prototype.createWindowLayer = function() {
    Scene_Base_createWindowLayer.call(this);

    // const dialog = new Haku_Dialog({
    //   text: "确认返回主菜单？\n这样可能会丢失所有进度",
    //   maskImage: '/img/ui/dialog/dialog_mask',
    //   textWidth: 400,
    //   style: {
    //     backgroundImage: '/img/ui/dialog/confirm_idle',
    //   },
    //   confirmConfig: {
    //     text: "确认",
    //     width: 148,
    //     height: 52,
    //     y: 200,
    //     style: {
    //       backgroundImage: '/img/ui/dialog/confirm_idle',
    //       fontColor: '#FFFF00',
    //       fontSize: 18,
    //       textAlign: 'center',
    //       textX: 20
    //     },
    //     'hover:style': {
    //       backgroundImage: '/img/ui/dialog/confirm_hover',
    //     }
    //   },
    //   cancelConfig: {
    //     text: "取消",
    //     width: 148,
    //     height: 52,
    //     y: 200,
    //     style: {
    //       backgroundImage: '/img/ui/dialog/confirm_idle',
    //       fontColor: '#FFFF00',
    //       fontSize: 18,
    //       textAlign: 'center',
    //       textX: 20
    //     },
    //     'hover:style': {
    //       backgroundImage: '/img/ui/dialog/confirm_hover',
    //     }
    //   },
    // });
    // dialog.width = 936;
    // dialog.height = 411;
    // dialog.on('confirm', () => {
    //   console.log('确认了');
    // });
    // dialog.on('cancel', () => {
    //   console.log('取消了');
    // });
    
    
    
    // const checkbox1 = new Haku_Checkbox({
    //   text: "测试复选框1",
    //   style: {
    //     backgroundImage: '/img/ui/checkbox/checkbox_background',
    //     fontColor: '#FFFF00',
    //     fontSize: 18,
    //     textAlign: 'center',
    //     textX: 20
    //   },
    //   'hover:style': {
    //     backgroundImage: '/img/ui/checkbox/checkbox_hover_background',
    //     animeName: 'hover2'
    //   },
    //   'checked:style': {
    //     backgroundImage: '/img/ui/checkbox/checkbox_selected_background'
    //   },
    //   'checked:hover:style': {
    //     backgroundImage: '/img/ui/checkbox/checkbox_selected_hover_background'
    //   },
    // });
    // checkbox1.width = 181;
    // checkbox1.height = 36;
    // checkbox1.x = 200;
    // checkbox1.y = 300;
    // checkbox1.on('change', (val) => {
    //   console.log('切换选中，目前状态是：', val ? '选中' : '未选中');
    // });
    // this.addChild(checkbox1);
    
    // const slider1 = new Haku_Silder({
    //   thumbImage: '/img/ui/scrollbar/scroll-thumb',
    //   barImage: '/img/ui/scrollbar/scroll-bar',
    //   text: '',
    //   textWidth: 50,
    //   style: {
    //     fontColor: '#FFFF00',
    //     fontSize: 18,
    //     textAlign: 'center',
    //     textX: 20
    //   },
    // });
    // slider1.width = 181;
    // slider1.height = 8;
    // slider1.x = 200;
    // slider1.y = 400;
    // slider1.on('change', (val) => {
    //   console.log('当前值为：', val);
    // });
    // this.addChild(slider1);
  }

  // #endregion



  window.HakuComponent = {
    /** 按钮组件 */
    Button: Haku_Button,
    /** 复选框组件 */
    Checkbox: Haku_Checkbox,
    /** 滑块组件 */
    Silder: Haku_Silder,
    /** 单选组按钮组件 */
    RadioButton: Haku_RadioButton,
  };

})();