//=============================================================================
// ** RPG Maker MZ - Hakubox_Utils.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 基础插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MV MZ
 * 
 * @help
 * 【与傲娇妹妹的治愈日常】定制插件。
 * 
 * 
 * 
 * @command startGameTransform
 * @text 开始界面变换
 * @desc 开始界面变换
 * 
 * @arg bgScale
 * @text 背景缩放（百分比）
 * @desc 背景缩放（百分比）
 * @type number
 * @default 100
 * 
 * @arg bgTransformX
 * @text X轴移动
 * @desc X轴移动
 * @type number
 * @max 9999
 * @min -9999
 * @default 1
 * 
 * @arg bgTransformY
 * @text Y轴移动
 * @desc Y轴移动
 * @type number
 * @max 9999
 * @min -9999
 * @default 1
 * 
 * @arg duration
 * @text 持续时间
 * @desc 持续时间
 * @type number
 * @default 30
 * @min 10
 * 
 * 
 * @command eventEnd
 * @text 事件结束
 * @desc 事件结束
 * 
 * 
 * @command useAlert
 * @text 显示文本提示框
 * @desc 显示文本提示框
 * 
 * @arg text
 * @text 提示内容
 * @desc 提示内容
 * @type text
 * 
 * 
 * @command saveEvent
 * @text 保存事件
 * @desc 保存事件
 * 
 * @arg hasExecStep
 * @text 是否保存执行步骤
 * @desc 是否保存执行步骤
 * @type boolean
 * @on 保存
 * @off 不保存
 * @default false
 * 
 * 
 * @command restoreEvent
 * @text 恢复事件
 * @desc 恢复事件
 * 
 * @arg hasExecStep
 * @text 是否恢复执行步骤
 * @desc 是否恢复执行步骤
 * @type boolean
 * @on 恢复
 * @off 不恢复
 * @default false
 * 
 * 
 * @command talkAbout
 * @text 显示主题对话
 * @desc 显示主题对话
 * 
 * @arg topicId
 * @text 对话主题ID
 * @desc 对话主题ID
 * @type combo
 * @option 加奈的随机对话
 * @option 回家太晚的担心
 * @option 欢迎回家
 * @option 晚上出门的担心
 * @option 加奈-路上小心
 * @option 千夏-路上小心
 * @option 由纪-路上小心
 * @option 有什么事情明天再说
 * @option 现在加奈什么都不想说
 * @option 我先出门咯
 * @option 加奈回家了
 * @option 加奈身体接触
 * @option 加奈做爱
 * 
 * @arg params
 * @text 自定义参数
 * @desc 自定义参数，内容为可执行代码，所以如果要字符串得加引号。
 * @type text[]
 * 
 * @arg isOver
 * @text 对话完成是否结束
 * @desc 对话完成是否结束
 * @on 结束
 * @off 继续
 * @type boolean
 * @default true
 * 
 */
// #endregion
(() => {

  // Game_Interpreter.prototype.executeCommand = function() {
  //   const command = this.currentCommand();
  //   if (command) {
  //     this._indent = command.indent;
  //     const methodName = "command" + command.code;
  //     if (typeof this[methodName] === "function") {
  //       if (!this[methodName].call(this, command.parameters)) {
  //         return false;
  //       }
  //     }
  //     this._index++;
  //   } else {
  //     this.terminate();
  //   }
  //   return true;
  // };

  PluginManager.callCommand = function(self, pluginName, commandName, args) {
    const key = pluginName + ":" + commandName;
    const func = this._commands[key];
    if (typeof func === "function") {
      func.call(self, args);
    }
  };

  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_Utils";


  /**
   * 动画曲线函数
   */
  window.MotionEasing = Object.freeze({
    Linear: Object.freeze({
      None: function (amount) {
        return amount;
      },
      In: function (amount) {
        return amount;
      },
      Out: function (amount) {
        return amount;
      },
      InOut: function (amount) {
        return amount;
      },
    }),
    Quadratic: Object.freeze({
      In: function (amount) {
        return amount * amount;
      },
      Out: function (amount) {
        return amount * (2 - amount);
      },
      InOut: function (amount) {
        if ((amount *= 2) < 1) {
          return 0.5 * amount * amount;
        }
        return -0.5 * (--amount * (amount - 2) - 1);
      },
    }),
    Cubic: Object.freeze({
      In: function (amount) {
        return amount * amount * amount;
      },
      Out: function (amount) {
        return --amount * amount * amount + 1;
      },
      InOut: function (amount) {
        if ((amount *= 2) < 1) {
          return 0.5 * amount * amount * amount;
        }
        return 0.5 * ((amount -= 2) * amount * amount + 2);
      },
    }),
    Quartic: Object.freeze({
      In: function (amount) {
        return amount * amount * amount * amount;
      },
      Out: function (amount) {
        return 1 - --amount * amount * amount * amount;
      },
      InOut: function (amount) {
        if ((amount *= 2) < 1) {
          return 0.5 * amount * amount * amount * amount;
        }
        return -0.5 * ((amount -= 2) * amount * amount * amount - 2);
      },
    }),
    Quintic: Object.freeze({
      In: function (amount) {
        return amount * amount * amount * amount * amount;
      },
      Out: function (amount) {
        return --amount * amount * amount * amount * amount + 1;
      },
      InOut: function (amount) {
        if ((amount *= 2) < 1) {
          return 0.5 * amount * amount * amount * amount * amount;
        }
        return 0.5 * ((amount -= 2) * amount * amount * amount * amount + 2);
      },
    }),
    Sinusoidal: Object.freeze({
      In: function (amount) {
        return 1 - Math.sin(((1.0 - amount) * Math.PI) / 2);
      },
      Out: function (amount) {
        return Math.sin((amount * Math.PI) / 2);
      },
      InOut: function (amount) {
        return 0.5 * (1 - Math.sin(Math.PI * (0.5 - amount)));
      },
    }),
    Exponential: Object.freeze({
      In: function (amount) {
        return amount === 0 ? 0 : Math.pow(1024, amount - 1);
      },
      Out: function (amount) {
        return amount === 1 ? 1 : 1 - Math.pow(2, -10 * amount);
      },
      InOut: function (amount) {
        if (amount === 0) {
          return 0;
        }
        if (amount === 1) {
          return 1;
        }
        if ((amount *= 2) < 1) {
          return 0.5 * Math.pow(1024, amount - 1);
        }
        return 0.5 * (-Math.pow(2, -10 * (amount - 1)) + 2);
      },
    }),
    Circular: Object.freeze({
      In: function (amount) {
        return 1 - Math.sqrt(1 - amount * amount);
      },
      Out: function (amount) {
        return Math.sqrt(1 - --amount * amount);
      },
      InOut: function (amount) {
        if ((amount *= 2) < 1) {
          return -0.5 * (Math.sqrt(1 - amount * amount) - 1);
        }
        return 0.5 * (Math.sqrt(1 - (amount -= 2) * amount) + 1);
      },
    }),
    Elastic: Object.freeze({
      In: function (amount) {
        if (amount === 0) {
          return 0;
        }
        if (amount === 1) {
          return 1;
        }
        return -Math.pow(2, 10 * (amount - 1)) * Math.sin((amount - 1.1) * 5 * Math.PI);
      },
      Out: function (amount) {
        if (amount === 0) {
          return 0;
        }
        if (amount === 1) {
          return 1;
        }
        return Math.pow(2, -10 * amount) * Math.sin((amount - 0.1) * 5 * Math.PI) + 1;
      },
      InOut: function (amount) {
        if (amount === 0) {
          return 0;
        }
        if (amount === 1) {
          return 1;
        }
        amount *= 2;
        if (amount < 1) {
          return -0.5 * Math.pow(2, 10 * (amount - 1)) * Math.sin((amount - 1.1) * 5 * Math.PI);
        }
        return 0.5 * Math.pow(2, -10 * (amount - 1)) * Math.sin((amount - 1.1) * 5 * Math.PI) + 1;
      },
    }),
    Back: Object.freeze({
      In: function (amount) {
        var s = 1.70158;
        return amount === 1 ? 1 : amount * amount * ((s + 1) * amount - s);
      },
      Out: function (amount) {
        var s = 1.70158;
        return amount === 0 ? 0 : --amount * amount * ((s + 1) * amount + s) + 1;
      },
      InOut: function (amount) {
        var s = 1.70158 * 1.525;
        if ((amount *= 2) < 1) {
          return 0.5 * (amount * amount * ((s + 1) * amount - s));
        }
        return 0.5 * ((amount -= 2) * amount * ((s + 1) * amount + s) + 2);
      },
    }),
    Bounce: Object.freeze({
      In: function (amount) {
        return 1 - Easing.Bounce.Out(1 - amount);
      },
      Out: function (amount) {
        if (amount < 1 / 2.75) {
          return 7.5625 * amount * amount;
        }
        else if (amount < 2 / 2.75) {
          return 7.5625 * (amount -= 1.5 / 2.75) * amount + 0.75;
        }
        else if (amount < 2.5 / 2.75) {
          return 7.5625 * (amount -= 2.25 / 2.75) * amount + 0.9375;
        }
        else {
          return 7.5625 * (amount -= 2.625 / 2.75) * amount + 0.984375;
        }
      },
      InOut: function (amount) {
        if (amount < 0.5) {
          return Easing.Bounce.In(amount * 2) * 0.5;
        }
        return Easing.Bounce.Out(amount * 2 - 1) * 0.5 + 0.5;
      },
    }),
    generatePow: function (power) {
      if (power === void 0) { power = 4; }
      power = power < Number.EPSILON ? Number.EPSILON : power;
      power = power > 10000 ? 10000 : power;
      return {
        In: function (amount) {
          return Math.pow(amount, power);
        },
        Out: function (amount) {
          return 1 - Math.pow((1 - amount), power);
        },
        InOut: function (amount) {
          if (amount < 0.5) {
            return Math.pow((amount * 2), power) / 2;
          }
          return (1 - Math.pow((2 - amount * 2), power)) / 2 + 0.5;
        },
      };
    },

    /**
     * 获取需要的动画曲线函数
     * @param {'Linear'|'Quadratic'|'Cubic'|'Quartic'|'Quintic'|'Sinusoidal'|'Exponential'} easingName 曲线类型名称
     * @param {'None'|'In'|'Out'|'InOut'} type 时间曲线出入方式
     */
    getEasing(easingName = 'Linear', type = 'None') {
      if (!easingName || easingName.length < 1) {
        throw new Error('easingName is error');
      }
      const easingType = easingName.substring(0, 1).toUpperCase() + easingName.substring(1).toLowerCase();
      return MotionEasing[easingType][type];
    }
  });

  Bitmap.prototype._startLoading = function () {
    this._image = new Image();
    this._image.onload = this._onLoad.bind(this);
    this._image.onerror = this._onError.bind(this);
    this._destroyCanvas();
    this._loadingState = "loading";
    if (Utils.hasEncryptedImages()) {
      this._startDecrypting();
    } else {
      this._image.src = this._url;
      // 修复浏览器刷新图片无法加载的问题
      // if (this._image.width > 0) {
      //     this._image.onload = null;
      //     this._onLoad();
      // }
    }
  }

  /**
   * 加载指定路径的位图
   * @param {string} path 图片路径
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   */
  ImageManager.loadImg = function(path, prefix = "img/pictures/") {
    return new Promise((resolve, reject) => {
      const _bitmap = ImageManager.loadBitmap(prefix, path);
      if (_bitmap.isReady()) {
        resolve(_bitmap);
      } else {
        _bitmap.addLoadListener(() => {
          resolve(_bitmap);
        });
      }
    });
  }

  /**
   * 加载指定路径的位图
   * @param {string} path 图片路径
   * @param {Function} callback 回调函数
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   */
  ImageManager.loadImgCb = function(path, callback, prefix = "img/pictures/") {
    const _bitmap = ImageManager.loadBitmap(prefix, path);
    if (_bitmap.isReady()) {
      callback(_bitmap);
    } else {
      _bitmap.addLoadListener(() => {
        callback(_bitmap);
      });
    }
  }

  window.inArea = function(x, y, left = 0, top = 0, width = 0, height = 0) {
    return (
      x >= left && x <= left + width && y >= top && y <= top + height
    )
  }
  
  /** 在范围内 */
  Window_Base.prototype.inArea = function(x, y, ox = 0, oy = 0, ow = 0, oh = 0) {
    return (
      x >= this.x + ox && x <= this.x + this.width + ow + ox && 
      y >= this.y + oy && y <= this.y + this.height + oh + oy
    );
  }

  /** 在范围内 */
  PIXI.Container.prototype.inArea = function(x, y, ox = 0, oy = 0, ow = 0, oh = 0) {
    if (!this.transform) return false;
    return (
      x >= this.x + ox && x <= this.x + this.width + ow + ox && 
      y >= this.y + oy && y <= this.y + this.height + oh + oy
    );
  };

  /** 在范围内 */
  Sprite.prototype.inArea = function(x, y, ox = 0, oy = 0, ow = 0, oh = 0) {
    return (
      x >= this.x + ox && x <= this.x + this.width + ow + ox && 
      y >= this.y + oy && y <= this.y + this.height + oh + oy
    );
  };

  /**
   * 切换图片
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   */
  Sprite.prototype.changePic = function(path, prefix = "img/pictures/") {
    if (this.bitmap._url != prefix + path + '.png') {
      this.bitmap = ImageManager.loadBitmap(prefix, path);
      this._refresh();
      // this.bitmap = ImageManager.loadPicture(path);
    }
  };

  /**
   * 绘制图片到当前位图
   * @param {string} path 图片路径
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   * @param {object} config 配置项
   * @param {Function} callback 回调函数
   */
  Bitmap.prototype.bltImg = function(path, prefix = "img/pictures/", config = { x: 0, y: 0, clear: false }, callback) {
    const _bitmap = ImageManager.loadBitmap(prefix, path);
    if (_bitmap.isReady()) {
      if (config.clear) {
        this.clearRect(config.x || 0, config.y || 0, config.width || this.width, config.height || this.height);
      }
      this.blt(_bitmap, config.x2 || 0, config.y2 || 0, config.width2 || _bitmap.width, config.height2 || _bitmap.height, 
        config.x || 0, config.y || 0, config.width || this.width, config.height || this.height);
      if (callback) callback();
    } else {
      _bitmap.addLoadListener(() => {
        if (config.clear) {
          this.clearRect(config.x || 0, config.y || 0, config.width || this.width, config.height || this.height);
        }
        this.blt(_bitmap, 0, 0, config.width2 || _bitmap.width, config.height2 || _bitmap.height, 
          config.x || 0, config.y || 0, config.width || this.width, config.height || this.height);
        if (callback) callback();
      });
    }
  }

  /**
   * 批量绘制图片到当前位图
   * @param {string[]} pathList 图片路径
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   * @param {object} config 配置项
   * @param {Function} callback 回调函数
   */
  Bitmap.prototype.bltImgList = function(pathList, prefix = "img/pictures/", config = { x: 0, y: 0, clear: false }, callback) {
    let _progress = 0;
    let _count = pathList.length;
    let _isComplete = false;

    const _resolve = () => {
      if (_isComplete) return;
      _isComplete = true;

      for (let i = 0; i < pathList.length; i++) {
        const _path = pathList[i];
        const _bitmap = ImageManager.loadBitmap(prefix, _path);
        this.blt(_bitmap, 0, 0, _bitmap.width, _bitmap.height, 
          config.x || 0, config.y || 0, config.width || this.width, config.height || this.height);
      }
      callback && callback();
    }

    for (let i = 0; i < pathList.length; i++) {
      const _path = pathList[i];
      const _bitmap = ImageManager.loadBitmap(prefix, _path);
      if (_bitmap.isReady()) {
        _progress++;
        if (_progress === _count) {
          _resolve();
        }
      } else {
        _bitmap.addLoadListener(() => {
          _progress++;
          if (_progress === _count) {
            _resolve();
          }
        });
      }
    }
  }

  /**
   * 绘制图片到当前位图
   * @param {string} path 图片路径
   * @param {string} prefix 前缀路径，默认为 "img/pictures/"
   * @param {object} config 配置项
   */
  Bitmap.prototype.bltImgAndClear = function(path, prefix = "img/pictures/", config = { x: 0, y: 0 }) {
    const _bitmap = ImageManager.loadBitmap(prefix, path);
    if (_bitmap.isReady()) {
      this.clear();
      this.blt(_bitmap, 0, 0, _bitmap.width, _bitmap.height, 0, 0, this.width, this.height);
    } else {
      _bitmap.addLoadListener(() => {
        this.clear();
        this.blt(_bitmap, 0, 0, _bitmap.width, _bitmap.height, 0, 0, this.width, this.height);
      });
    }
  }

  
  /** 
   * 节流函数
   * @param {() => void} func 回调函数
   * @param {number} limit 限制时间间隔
   */
  window.throttle = function(func, limit = 20) {
    let inThrottle;
    return function(...args) {
      if (!inThrottle) {
        func.apply(this, args);
        inThrottle = true;
        setTimeout(() => (inThrottle = false), limit);
      }
    };
  }

  /**
   * 等待函数
   * @param {(cb: Function) => void} callback 回调函数
   * @param {number} [delay=100] 等待毫秒数
   */
  window.useWait = function(callback, delay = 100) {
    let _isComplete = false;
    let _currentIndex = 0;
    let _maxCount = 100;
    const _timer = () => {
      _currentIndex++;
      callback(() => {
        _isComplete = true;
      });

      if (!_isComplete && _currentIndex < _maxCount) {
        setTimeout(() => {
          _timer();
        }, delay);
      }
    };

    _timer();
  }

  /**
   * 使用延时功能
   * @param {() => void} callback 回调函数
   * @param {number} delay 等待帧数
   * @param {(progress: number) => void} update 更新函数
   * @param {{ easingType: string, inout: string }} info 动画信息
   */
  window.useTimeout = function(callback, delay, update, info = { easingType: 'Linear', inout: 'None' }) {
    const _timer = {
      frameIndex: 0,
      frameCount: delay,
      stop() {
        this.frameIndex = 0;
        moduleTicker.off(_timerUpdate);
      }
    };

    const _timerUpdate = () => {
      _timer.frameIndex++;

      if (update) {
        const progress = MotionEasing.getEasing(info.easingType, info.inout)(_timer.frameIndex / _timer.frameCount);
        update(progress, _timer.frameIndex);
      }

      if (_timer.frameIndex >= _timer.frameCount) {
        _timer.frameIndex = 0;
        moduleTicker.off(_timerUpdate);
        callback();
      }
    };

    moduleTicker.on(_timerUpdate);
    return _timer;
  };

  /**
   * 使用图片切换功能
   * @param {Sprite} sprite 精灵对象
   * @param {Bitmap} bitmap 图片路径
   * @param {number} delay 等待帧数
   * @param {object} config 配置项
   * @param {string} [config.prefix] 前缀路径，默认为 "img/pictures/"
   */
  window.changeImg = function(sprite, bitmap, delay = 30, config = {
    prefix: "img/pictures/", scale: 1
  }) {
    if (!sprite) throw new Error("sprite is required");
    const _parent = sprite.parent;
    if (!_parent) throw new Error("parent is required");
    if (sprite.loading) return;
    // if (sprite.bitmap._url === bitmap._url) return;

    sprite.loading = true;
    const _scale = config.scale || 1;

    const _tempSprite = new Sprite(bitmap);
    _tempSprite.x = sprite.x;
    _tempSprite.y = sprite.y;
    _tempSprite.alpha = 0;
    _parent.addChildAt(_tempSprite, _parent.getChildIndex(sprite));

    useTimeout(() => {
      sprite.alpha = 1;
      sprite.bitmap = _tempSprite.bitmap;
      sprite.scale.set(1, 1);
      sprite.loading = false;
      if (_tempSprite && _tempSprite._texture) _tempSprite.destroy();
    }, delay, progress => {
      _tempSprite.alpha = progress;
      sprite.alpha = 1 - progress;
      if (_scale != 1) sprite.scale.set(_scale * progress, _scale * progress);
    }, {
      easingType: 'Quadratic',
      inout: 'InOut'
    });
  }

  /**
   * 使用图片切换功能
   * @param {PIXI.Container} container 容器对象
   * @param {number} delay 等待帧数
   */
  window.changeImgs = function(container, imgObj, delay = 30) {
    if (!container) throw new Error("container is required");

    const _newSprites = [];
    
    if (typeof imgObj === 'string') {
      _newSprites.push(new Sprite(ImageManager.loadPicture(imgObj)));
    } else if (imgObj instanceof Sprite) {
      _newSprites.push(imgObj);
    } else if (Array.isArray(imgObj)) {
      for (let i = 0; i < imgObj.length; i++) {
        const _img = imgObj[i];
        if (typeof _img === 'string') {
          _newSprites.push(new Sprite(ImageManager.loadPicture(_img)));
        } else if (_img instanceof Sprite) {
          _newSprites.push(_img);
        } else if (typeof _foreground === 'object') {
          const _sprite = new Sprite(_img.path);
          if (_img.blendMode) {
            _sprite.blendMode = _img.blendMode;
          }
          _newSprites.push(_sprite);
        }
      }
    }

    if (delay) {
      const _newContainer = new PIXI.Container();
      const _oldContainer = new PIXI.Container();
      _oldContainer.alpha = 0;

      for (let i = 0; i < _newSprites.length; i++) {
        const _item = _newSprites[i];
        _newContainer.addChild(_item);
      }

      for (let i = 0; i < container.children.length; i++) {
        const _item = container.children[i];
        _oldContainer.addChild(_item);
      }

      container.addChild(_oldContainer);
      container.addChild(_newContainer);

      useTimeout(() => {
        _newContainer.alpha = 1;
        for (let i = 0; i < _newContainer.children.length; i++) {
          const _item = _newContainer.children[i];
          container.addChild(_item);
        }
        _newContainer.destroy();
        _oldContainer.destroy();
      }, delay, progress => {
        _newContainer.alpha = progress;
        _oldContainer.alpha = 1 - progress;
      }, {
        easingType: 'Quadratic',
        inout: 'InOut'
      });
    } else {
      container.removeChildren();
      for (let i = 0; i < _newSprites.length; i++) {
        const _item = _newSprites[i];
        container.addChild(_item);
      }
    }
  }
  
  /**
   * 使用延时功能（返回Promise）
   * @param {number} delay 等待帧数
   */
  window.useDuration = function(delay) {
    return new Promise((resolve) => {
      let _frameIndex = 0;
      let _frameCount = delay;

      const _timerUpdate = () => {
        _frameIndex++;
        if (_frameIndex >= _frameCount) {
          _frameIndex = 0;
          resolve();
        }
      };

      Graphics._app.ticker.add(_timerUpdate);
    });
  };

  /**
   * 使用动画功能
   * @param {{ autoUpdate: boolean, frameDelay: number, frameCount: number, count: number, easingType: string, inout: string }} config 
   * @param {(progress: number) => void} callback 
   */
  window.useAnime = function(config, callback) {
    if (config.frameDelay === undefined || config.frameCount === undefined) {
      throw new Error("frameDelay or frameCount are required");
    }
    
    const start = () => {
      _info.isStart = true;
      _info.frameCounter = 0;
      _info.frameIndex = 0;
    }
    const stop = () => {
      _info.isStart = false;
      _info.frameCounter = 0;
      _info.frameIndex = 0;
      
      if (config.autoUpdate) {
        moduleTicker.off(_timerUpdate);
      }
    }
    const update = () => {
      if (_info.isStart) {
        _timerUpdate.call(_info);
      }
    }

    const _info = {
      cid: randomId(),
      isStart: config.isStart || false,
      frameCounter: 0,
      frameDelay: config.frameDelay,
      frameIndex: 0,
      frameCount: config.frameCount,
      count: config.count || 1,
      easingType: config.easingType || 'Linear',
      inout: config.inout || 'None',
      description: config.description || '',
      start, stop, update
    };
    const _timerUpdate = function() {
      if (_info.isStart) {
        _info.frameCounter++;
        if (_info.frameCounter > _info.frameDelay) {
          _info.frameCounter = 0;
          _info.frameIndex++;

          const progress = MotionEasing.getEasing(_info.easingType, _info.inout)(_info.frameIndex / _info.frameCount);
          callback(progress, _info.frameIndex);

          if (_info.frameIndex >= _info.frameCount && _info.isStart) {
            _info.stop();
          }
        }
      }
    };

    if (config.autoUpdate) {
      moduleTicker.on(_timerUpdate);
      if (!_info.isStart) _info.start();
    }

    return _info;
  }

  /** 警告框 */
  const __alertList = [];

  const __alertHandler = (progress = 0) => {
    if (__alertList.length <= 0) return;

    const _y = progress * (__alertList[0].height + 15);

    for (let i = 0; i < __alertList.length; i++) {
      const item = __alertList[i];

      if (item.container.transform) {
        if (i == 0) {
          item.container.y = 120 + i * (item.height + 15);
        } else {
          item.container.y = 120 + i * (item.height + 15) - _y;
        }
      }
    }
  }

  /**
   * 警告框组件
   * @param {object} config 组件配置
   */
  window.useAlert = function(config) {

    const _info = {
      height: 0,
      container: new PIXI.Container(),
      animeType: 'in',
      close() {
        this.animeType = 'out';
        _info.animeInOutInfo.start();
        moduleTicker.off(this.update);

        const _index = __alertList.indexOf(_info);
        __alertList.splice(_index, 1);
        __alertHandler();
        _info.container.destroy();
      },
      /** 更新 */
      update() {
        _info.animeInOutInfo.update();
      }
    };

    const onInit = function(cfg) {
      this._text = cfg.text;
      this._bgImage = cfg.bgImage || 'ui/alert_bg';
      this._duration = cfg.duration || 180;
      this._offset = cfg.offset || 50;

      ImageManager.loadImg(this._bgImage).then(bitmap => {
        
        if (SceneManager._scene instanceof Scene_Haku_Adventure) {
          _info.x = (853 - bitmap.width) / 2;
          _info.container.x = (853 - bitmap.width) / 2 + _info._offset;
        } else {
        _info.x = (Graphics.width - bitmap.width) / 2;
        _info.container.x = (Graphics.width - bitmap.width) / 2 + _info._offset;
        }

        _info.height = bitmap.height;

        _info.bgSprite = new Sprite(bitmap);
        _info.bgSprite.x = 0;
        _info.container.addChild(_info.bgSprite);

        _info.textSprite = new Sprite(new Bitmap(bitmap.width, bitmap.height));
        _info.textSprite.bitmap.fontFace = 'Source Han Sans VF';
        _info.textSprite.bitmap.textColor = '#FC8919';
        _info.textSprite.bitmap.fontSize = 26;
        _info.textSprite.bitmap.outlineColor = '#FFFFFF';
        _info.textSprite.bitmap.outlineWidth = 4;
        _info.textSprite.bitmap.drawText(this._text, 0, 0, bitmap.width, bitmap.height, 'center');
        _info.container.addChild(_info.textSprite);

        SceneManager._scene.addChild(_info.container);
        
        __alertList.push(_info);
        __alertHandler();

        _info.animeInOutInfo.start();
      });

      // 动画信息
      _info.animeInOutInfo = useAnime({
        frameDelay: 0,
        frameCount: 15,
        easingType: 'Linear',
        inout: 'None',
      }, progress => {
        if (!_info.container || !_info.container.parent) {
          _info.close();
          return;
        }

        if (_info.animeType === 'in') {
          _info.container.alpha = progress;
          _info.container.x = parseInt(_info.x - _info._offset + progress * _info._offset);

          if (progress >= 1) {
            useTimeout(() => {
              _info.animeType = 'out';
              _info.animeInOutInfo.start();
            }, this._duration);
          }
        } else if (_info.animeType === 'out') {
          _info.container.alpha = 1 - progress;
          _info.container.x = parseInt(_info.x - progress * _info._offset);
          if (progress >= 1) {
            _info.close();
          } else {
            const _index = __alertList.indexOf(_info);
            if (_index === 0) {
              __alertHandler(progress);
            }
          }
        }
      });
    }

    onInit.call(_info, config);
      
    moduleTicker.on(_info.update);

    return _info;
  };

  /**
   * 自动保存
   * @param {*} savefileId 
   * @param {*} callback 
   */
  window.autoSave = function(savefileId, callback) {
    if (savefileId === undefined) {
      throw new Error("savefileId is required");
    } else if (savefileId != 1 && savefileId != 2) {
      throw new Error("savefileId must be 1 or 2");
    }

    if (SceneManager._scene instanceof Scene_Game) {
      SceneManager._scene.saveTempInfo();
      SceneManager.screenShot();
    }
    $gameSystem.setSavefileId(100 + savefileId);
    $gameSystem.onBeforeSave();
    DataManager.saveGame(100 + savefileId).then(() => {
      // $gameData.tempSceneInfo.canUse = false;
      // if (savefileId != 2) useAlert({ text: '自动保存成功' });
      callback && callback();
    });
  };

  // #region 确认框组件
  /**
   * 确认框组件
   * @param {object} config 组件配置
   */
  // window.useSelect = function(config) {

  //   const onInit = function(cfg) {

  //     // 判断是否已有弹出框，有弹出框则不弹出
  //     if (SceneManager._scene instanceof Scene_Game) {
  //       if (SceneManager._scene.visibleDialog) {
  //         return;
  //       } else {
  //         SceneManager._scene.visibleDialog = true;
  //       }
  //     }

  //     this._width = cfg.width || 643;
  //     this._height = cfg.height || 165;
  //     this._hasCancel = cfg.hasCancel !== undefined ? cfg.hasCancel : true;
  //     this._okText = cfg.okText || '确定';
  //     this._okHandler = cfg.okHandler || (() => {});
  //     this._useMask = cfg.useMask !== undefined ? cfg.useMask : false;
  //     this._cancelHandler = () => {
  //       if (cfg.cancelHandler) cfg.cancelHandler();
  //       _info.close();
  //     };
  //     this._cancelText = cfg.cancelText || '取消';
  //     this._okBtnBgImg = cfg.okBtnBgImg || 'ui/button/button_6';
  //     this._okBtnBgHoverImg = cfg.okBtnBgHoverImg || 'ui/button/button_6_active';
  //     this._okBtnBgDisabledImg = cfg.okBtnBgDisabledImg || 'ui/button/button_6_disabled';
  //     this._cancelBtnBgImg = cfg.cancelBtnBgImg || 'ui/button/button_7';
  //     this._cancelBtnBgHoverImg = cfg.cancelBtnBgHoverImg || 'ui/button/button_7_active';
  //     this._cancelBtnBgDisabledImg = cfg.cancelBtnBgDisabledImg || 'ui/button/button_7_disabled';

  //     // 动画信息
  //     _info.animeInOutInfo = useAnime({
  //       frameDelay: 0,
  //       frameCount: 15,
  //       easingType: 'Quadratic',
  //       inout: 'InOut',
  //     }, progress => {
  //       if (!_info.contents.transform) {
  //         SceneManager._scene.visibleDialog = false;
  //         _info.animeInOutInfo.stop();
  //         _info.destroy();
  //         return;
  //       }

  //       const _baseY = (Graphics.height - this._height) / 2;
  //       if (_info.animeType === 'in') {
  //         _info.maskSprite.alpha = progress;
  //         _info.contents.alpha = progress;
  //         _info.contents.y = parseInt(_baseY - 20 + progress * 20);
  //       } else if (_info.animeType === 'out') {
  //         _info.maskSprite.alpha = 1 - progress;
  //         _info.contents.alpha = 1 - progress;
  //         _info.contents.y = parseInt(_baseY - progress * 20);
  //         if (progress >= 1) {
  //           SceneManager._scene.visibleDialog = false;
  //           _info.destroy();
  //         }
  //       }
  //     });
      
  //     const rect = new Rectangle(
  //       (Graphics.width - this._width) / 2, (Graphics.height - this._height) / 2, 
  //       this._width, this._height
  //     );
  //     _info.rect = rect;
    
  //     _info.contents.alpha = 0;
  //     _info.contents.x = rect.x;
  //     _info.contents.y = (Graphics.height - this._height) / 2;

  //     // 创建屏幕遮罩
  //     this.maskSprite = new Sprite(new Bitmap(Graphics.width, Graphics.height));
  //     this.maskSprite.bitmap.fillRect(0, 0, Graphics.width, Graphics.height, 'rgba(44,48,59,0.6)');
  //     this.maskSprite.alpha = 0;
  //     this.maskSprite.x = 0;
  //     this.maskSprite.y = 0;
  //     SceneManager._scene.addChild(this.maskSprite);

  //     // 创建背景（用于写入文本
  //     this.backround = new Sprite(new Bitmap(rect.width, rect.height));
  //     this.backround.bitmap.smooth = false;
  //     this.backround.x = 0;
  //     this.backround.y = 0;
  //     this.backround.bitmap.fontFace = 'ChillRoundF';
  //     this.backround.bitmap.fontSize = 22;
  //     this.backround.bitmap.outlineWidth = 4;
  //     this.backround.bitmap.textColor = '#5c474e';
  //     this.backround.bitmap.outlineColor = 'rgba(0,0,0,0)';
  //     this._content.split('\n').forEach((item, index) => {
  //       this.backround.bitmap.drawText(item, 0, this._contentY + index * 30, this.width, 30, 'center');
  //     });
  //     this.contents.addChild(this.backround);
      
  //     moduleTicker.on(_info.update);

  //     SceneManager._scene.addChild(this.contents);
  //     this.animeInOutInfo.start();
  //   };
    
  //   const _info = {
  //     contents: new PIXI.Container(),
  //     rect: undefined,
  //     animeType: 'in',
  //     destroy() {
  //       moduleTicker.off(this.update);
  //       setTimeout(() => {
  //         this.contents.destroy();
  //       }, 10);
  //     },
  //     get x() {
  //       return _info.contents.x;
  //     },
  //     set x(value) {
  //       _info.contents.x = value;
  //     },
  //     get y() {
  //       return _info.contents.y;
  //     },
  //     set y(value) {
  //       _info.contents.y = value;
  //     },
  //     get height() {
  //       return _info.rect.height || _info.contents.height;
  //     },
  //     get width() {
  //       return _info.rect.width || _info.contents.width;
  //     },
  //     /** 更新 */
  //     update() {
  //     },
  //   };

  //   onInit.call(_info, config);

  //   return _info;
  // };

  // #region 确认框组件
  /**
   * 确认框组件
   * @param {object} config 组件配置
   */
  window.useConfirm = function(config) {

    if (!config.content) {
      throw new Error("content is required");
    } else if (!config.okHandler) {
      throw new Error("okHandler is required");
    }

    const onInit = function(cfg) {

      // 判断是否已有弹出框，有弹出框则不弹出
      if (SceneManager._scene instanceof Scene_Game) {
        if (SceneManager._scene.visibleDialog) {
          return;
        } else {
          SceneManager._scene.visibleDialog = true;
        }
      }

      this._width = cfg.width || 643;
      this._height = cfg.height || 165;
      this._bgImage = cfg.bgImage || 'ui/dialog_panel_bg2';
      this._content = cfg.content || '内容';
      this._contentY = cfg.contentY || 38;
      this._hasCancel = cfg.hasCancel !== undefined ? cfg.hasCancel : true;
      this._okText = cfg.okText || '确定';
      this._okHandler = cfg.okHandler || (() => {});
      this._useMask = cfg.useMask !== undefined ? cfg.useMask : false;
      this._cancelHandler = () => {
        if (cfg.cancelHandler) cfg.cancelHandler();
        _info.close();
      };
      this._cancelText = cfg.cancelText || '取消';
      this._okBtnBgImg = cfg.okBtnBgImg || 'ui/button/button_6';
      this._okBtnBgHoverImg = cfg.okBtnBgHoverImg || 'ui/button/button_6_active';
      this._okBtnBgDisabledImg = cfg.okBtnBgDisabledImg || 'ui/button/button_6_disabled';
      this._cancelBtnBgImg = cfg.cancelBtnBgImg || 'ui/button/button_7';
      this._cancelBtnBgHoverImg = cfg.cancelBtnBgHoverImg || 'ui/button/button_7_active';
      this._cancelBtnBgDisabledImg = cfg.cancelBtnBgDisabledImg || 'ui/button/button_7_disabled';

      // 动画信息
      _info.animeInOutInfo = useAnime({
        frameDelay: 0,
        frameCount: 15,
        easingType: 'Quadratic',
        inout: 'InOut',
      }, progress => {
        if (!_info.contents.transform) {
          SceneManager._scene.visibleDialog = false;
          _info.animeInOutInfo.stop();
          _info.destroy();
          return;
        }

        const _baseY = (Graphics.height - this._height) / 2;
        if (_info.animeType === 'in') {
          _info.maskSprite.alpha = progress;
          _info.contents.alpha = progress;
          _info.contents.y = parseInt(_baseY - 20 + progress * 20);
        } else if (_info.animeType === 'out') {
          _info.maskSprite.alpha = 1 - progress;
          _info.contents.alpha = 1 - progress;
          _info.contents.y = parseInt(_baseY - progress * 20);
          if (progress >= 1) {
            SceneManager._scene.visibleDialog = false;
            _info.destroy();
          }
        }
      });
      
      const rect = new Rectangle(
        SceneManager._scene instanceof Scene_Haku_Adventure ? (853 - this._width) / 2 : (Graphics.width - this._width) / 2, 
        (Graphics.height - this._height) / 2, 
        this._width,
        this._height
      );
      _info.rect = rect;
    
      _info.contents.alpha = 0;
      _info.contents.x = rect.x;
      _info.contents.y = (Graphics.height - this._height) / 2;

      // 创建屏幕遮罩
      this.maskSprite = new Sprite(new Bitmap(SceneManager._scene instanceof Scene_Haku_Adventure ? 853 : Graphics.width, Graphics.height));
      this.maskSprite.bitmap.fillRect(0, 0, Graphics.width, Graphics.height, 'rgba(44,48,59,0.6)');
      this.maskSprite.alpha = 0;
      this.maskSprite.x = 0;
      this.maskSprite.y = 0;
      SceneManager._scene.addChild(this.maskSprite);

      // 创建背景
      this.bgSprite = new Sprite(ImageManager.loadPicture(this._bgImage));
      this.bgSprite.x = 0;
      this.bgSprite.y = 0;
      this.contents.addChild(this.bgSprite);

      // 创建背景（用于写入文本
      this.backround = new Sprite(new Bitmap(rect.width, rect.height));
      this.backround.bitmap.smooth = false;
      this.backround.x = 0;
      this.backround.y = 0;
      this.backround.bitmap.fontFace = 'ChillRoundF';
      this.backround.bitmap.fontSize = 22;
      this.backround.bitmap.outlineWidth = 4;
      this.backround.bitmap.textColor = '#5c474e';
      this.backround.bitmap.outlineColor = 'rgba(0,0,0,0)';
      this._content.split('\n').forEach((item, index) => {
        this.backround.bitmap.drawText(item, 0, this._contentY + index * 30, this.width, 30, 'center');
      });
      this.contents.addChild(this.backround);

      // 创建确认按钮
      this.okButton = new Sprite(new Bitmap(150, 46));
      this.okButton.bitmap.smooth = false;
      if (this._hasCancel) {
        this.okButton.x = (rect.width / 2 - 150) / 2 + 30;
      } else {
        this.okButton.x = rect.width / 2 - 75;
      }
      this.okButton.y = rect.height - 75;
      this.okButton.bitmap.fontFace = 'ChillRoundF';
      this.okButton.bitmap.textColor = 'white';
      this.okButton.bitmap.outlineColor = '#5c474e';
      this.okButton.bitmap.outlineWidth = 4;
      this.okButton.bitmap.fontSize = 18;
      this.contents.addChild(this.okButton);
      this.drawOk();

      // 创建取消按钮
      if (this._hasCancel) {
        this.cancelButton = new Sprite(new Bitmap(150, 46));
        this.cancelButton.bitmap.smooth = false;
        this.cancelButton.x = rect.width / 2 + (rect.width / 2 - 150) / 2 - 30;
        this.cancelButton.y = rect.height - 75;
        this.cancelButton.bitmap.fontFace = 'ChillRoundF';
        this.cancelButton.bitmap.textColor = '#5c474e';
        this.cancelButton.bitmap.outlineColor = 'rgba(0,0,0,0)';
        this.cancelButton.bitmap.outlineWidth = 1;
        this.cancelButton.bitmap.fontSize = 18;
        this.contents.addChild(this.cancelButton);
        this.drawCancel();
      }
      
      moduleTicker.on(_info.update);

      SceneManager._scene.addChild(this.contents);
      this.animeInOutInfo.start();
    };
    
    const _info = {
      contents: new PIXI.Container(),
      rect: undefined,
      animeType: 'in',
      close() {
        this.animeType = 'out';
        _info.animeInOutInfo.start();
      },
      drawCancel() {
        let _imgSrc = '';
        if (this.cancelButton.disabled) {
          this.cancelButton.bitmap.textColor = '#999999';
          _imgSrc = this._cancelBtnBgDisabledImg;
        } else if (this.cancelButton.isHover) {
          this.cancelButton.bitmap.textColor = '#5c474e';
          _imgSrc = this._cancelBtnBgHoverImg;
        } else {
          _imgSrc = this._cancelBtnBgImg;
        }
        this.cancelButton.bitmap.bltImg(_imgSrc, undefined, { clear: true }, () => {
          this.cancelButton.bitmap.drawText(this._cancelText, 
            0, 0, this.cancelButton.width, this.cancelButton.height, 'center'
          );
          this.cancelButton.bitmap.textColor = '#5c474e';
        });
      },
      drawOk() {
        let _imgSrc = '';
        if (this.okButton.disabled) {
          this.okButton.bitmap.textColor = '#999999';
          this.okButton.bitmap.outlineColor = '#CCCCCC';
          _imgSrc = this._okBtnBgDisabledImg;
        } else if (this.okButton.isHover) {
          this.okButton.bitmap.textColor = '#5c474e';
          this.okButton.bitmap.outlineColor = '#ffffff';
          _imgSrc = this._okBtnBgHoverImg;
        } else {
          _imgSrc = this._okBtnBgImg;
        }
        this.okButton.bitmap.bltImg(_imgSrc, undefined, { clear: true }, () => {
          this.okButton.bitmap.smooth = false;
          this.okButton.bitmap.drawText(this._okText, 
            0, 0, this.okButton.width, this.okButton.height, 'center'
          );
          this.okButton.bitmap.textColor = '#ffffff';
          this.okButton.bitmap.outlineColor = '#5c474e';
        });
      },
      destroy() {
        moduleTicker.off(this.update);
        setTimeout(() => {
          this.contents.destroy();
        }, 10);
      },
      get x() {
        return _info.contents.x;
      },
      set x(value) {
        _info.contents.x = value;
      },
      get y() {
        return _info.contents.y;
      },
      set y(value) {
        _info.contents.y = value;
      },
      get height() {
        return _info.rect.height || _info.contents.height;
      },
      get width() {
        return _info.rect.width || _info.contents.width;
      },
      /** 更新 */
      update() {
        _info.animeInOutInfo.update();
        const _x = TouchInput.x;
        const _y = TouchInput.y;

        if (TouchInput.isCancelled()) {
          _info.close();
          return;
        }

        if (!_info.contents.transform) return;
        const _okInArea = _info.okButton.inArea(_x, _y, _info.x, _info.y);
        if (_okInArea) {
          if (TouchInput.isTriggered()) {
            SoundManager.playOk();
            _info._okHandler(_info);
          }
          if (!_info.okButton.isHover) {
            SoundManager.playCursor();
            _info.okButton.isHover = true;
            _info.drawOk();
          }
        } else if (_info.okButton.isHover) {
          _info.okButton.isHover = false;
          _info.drawOk();
        }

        if (_info._hasCancel) {
          const _cancelInArea = _info.cancelButton.inArea(_x, _y, _info.x, _info.y);
          if (_cancelInArea) {
            if (TouchInput.isTriggered()) {
              SoundManager.playCancel();
              _info._cancelHandler();
            }
            if (!_info.cancelButton.isHover) {
              SoundManager.playCursor();
              _info.cancelButton.isHover = true;
              _info.drawCancel();
            }
          } else if (_info.cancelButton.isHover) {
            _info.cancelButton.isHover = false;
            _info.drawCancel();
          }
        }
      },
    };

    onInit.call(_info, config);

    return _info;
  };

  // #region 滚动条组件
  window.useScrollBar = function(rect, config) {
    const _scrollBar = new PIXI.Container();
    _scrollBar.x = rect.x;
    _scrollBar.y = rect.y;

    if (!config.barImage) {
      throw new Error("barImage is required");
    } else if (!config.thumbImage) {
      throw new Error("thumbImage is required");
    } else if (!config.pageHeight) {
      throw new Error("pageHeight is required");
    }

    const onInit = function(cfg) {
      this._scrollTop = 0;
      this._type = cfg.type;
      this._radius = cfg.radius || 5;
      this._rect = rect;
      this._direction = cfg.direction || 'vertical'; // 横向：horizontal 纵向：vertical
      this._scrollBarHeight = cfg.scrollBarHeight || 0;
      this._scrollBarWidth = cfg.scrollBarWidth || this.rect.width;
      this._pageHeight = cfg.pageHeight;
      this._contentHeight = rect.height;
      this._background = cfg.backround || '#FFFFFF';
      this._outlineColor = cfg.scrollOutlineColor || '#9EC9ED';
      this._outlineWidth = cfg.scrollOutlineWidth || 0;
      this._barBackground = cfg.barBackground || '#9EC9ED';
      this._barOutlineColor = cfg.scrollBarOutlineColor || '#9EC9ED';
      this._barOutlineWidth = cfg.scrollBarOutlineWidth || 0;
      this._dragMove = cfg.dragMove || null;
      this._barWidth = cfg.barWidth || this._rect.width;

      this._scrollBarTop = 0;

      this._mouseUp = this.onMouseUp.bind(this);
      this._mouseDown = this.onMouseDown.bind(this);
      this._mouseMove = this.onMouseMove.bind(this);

      // 拖拽相关
      this.isStartDragScroll = false;
      this.startDragY = 0;
      this.prevDragY = 0;

      // 创建拖拽条
      this.thumbSprite = new Sprite(ImageManager.loadPicture(config.thumbImage));
      this.contents.addChild(this.thumbSprite);
      this.thumbSprite.x = 0;
      this.thumbSprite.y = 5;

      this.barSprite = new Sprite(ImageManager.loadPicture(config.barImage));
      this.contents.addChild(this.barSprite);
      this.barSprite.x = 0;
      this.barSprite.y = 0;

      ImageManager.loadImg(config.thumbImage).then(thumbImage => {
        ImageManager.loadImg(config.barImage).then(img => {
          this.thumbSprite.x = (this.width - thumbImage.width) / 2;
          if (!cfg.scrollBarHeight) this._scrollBarHeight = img.height;
          if (!cfg.scrollBarWidth) this._scrollBarWidth = img.width;
        });
      });
      
      // 绑定事件
      document.addEventListener('mousedown', this._mouseDown);
      document.addEventListener('mouseup', this._mouseUp);
      document.addEventListener('mousemove', this._mouseMove);

      _info.update();
    };
    
    const _info = {
      contents: _scrollBar,
      rect,
      refresh() {
        // 判断是否应该显示滚动条
        if (this._contentHeight < this._pageHeight && this.contents.alpha) {
          this.contents.alpha = 0;
        } else if (this.contents.alpha === 0) {
          this.contents.alpha = 1;
        }
      },
      destroy() {
        _scrollBar.destroy();
        document.removeEventListener('mousedown', this._mouseDown);
        document.removeEventListener('mouseup', this._mouseUp);
        document.removeEventListener('mousemove', this._mouseMove);
        this._mouseMove = null;
      },
      get x() {
        return this.contents.x;
      },
      set x(value) {
        this.contents.x = value;
      },
      get y() {
        return this.contents.y;
      },
      set y(value) {
        this.contents.y = value;
      },
      get height() {
        return rect.height || this.contents.height;
      },
      get width() {
        return rect.width || this.contents.width;
      },
      /** 设置内容高度 */
      get contentHeight() {
        return this._contentHeight;
      },
      set contentHeight(height) {
        this._contentHeight = height;
      },
      /** 整个滚动条高度 */
      get scrollHeight() {
        return this._rect.height;
      },
      /** 更新 */
      update(value) {
        if (value !== undefined) {
          this.setScroll(value);
          this.barSprite.y = this._scrollBarTop;
        }
      },
      onMouseDown(e) {
        if (e.button === 0 && e.x >= this.x && e.x <= this.x + this._rect.width) {
          if (e.y >= this.y + this._scrollBarTop && e.y <= this.y + this._scrollBarTop + this._scrollBarHeight) {
            this.isStartDragScroll = true;
            this.startDragY = e.y - this.y - this._scrollBarTop - 5;
            this.prevDragY = this._scrollBarTop;
          } else if (e.y >= this.y && e.y <= this.y + this._rect.height) {
            this.isStartDragScroll = false;
            if (e.y < this.y + this._scrollBarTop) {
              if (this._dragMove) this._dragMove(this._scrollTop - this._pageHeight, true);
            } else if (e.y > this.y + this._scrollBarTop + this._scrollBarHeight) {
              if (this._dragMove) this._dragMove(this._scrollTop + this._pageHeight, true);
            }
          }
        }
      },
      onMouseUp(e) {
        this.isStartDragScroll = false;
      },
      onMouseMove(e) {
        if (this.isStartDragScroll) {
          const moveY = e.y - this.y - this.prevDragY - this.startDragY;
          const _barTop = this.prevDragY + moveY;
          let _re = _barTop / (this.scrollHeight - this._scrollBarHeight) * (this._contentHeight - this._pageHeight);
          if (_re < 0) _re = 0;
          else if (_re >= this._contentHeight - this._pageHeight) _re = this._contentHeight - this._pageHeight;
          if (this._dragMove) this._dragMove(_re, false);
        }
      },
      /** 滚动 */
      onScroll(scrollTop) {
        this.setScroll(scrollTop);
      },
      /** 设置滚动位置 */
      setScroll(scrollTop) {
        this._scrollTop = scrollTop;
        if (scrollTop < 0 || this._contentHeight < this._pageHeight) {
          this._scrollBarTop = 0;
        } else if (scrollTop > this._contentHeight - this._pageHeight) {
          this._scrollBarTop = this.height - this._scrollBarHeight;
        } else {
          //this._scrollBarTop = Math.floor(this.scrollHeight * this._scrollTop / this._contentHeight);
          this._scrollBarTop = this._scrollTop * (this.scrollHeight - this._scrollBarHeight) / (this._contentHeight - this._pageHeight);
        }
      }
    };

    onInit.call(_info, config);

    _info.refresh();

    return _info;
  };
  // #endregion

  /**
   * 创建文字副本
   * @param {object} config
   */
  window.createFloatText = function({
    sprite, x = 0, y = 0, text = "???", callback, style, frameCount, frameDelay, easingType, inout
  } = { style, frameCount: 50, frameDelay: 0, easingType: 'Quadratic', inout: 'InOut' }) {

    if (!style && !sprite.bitmap) {
      style = {
        fontFace: 'ChillRoundF',
        fontSize: 18,
        textColor: '#ffffff',
        outlineColor: 'rgba(0,0,0,0.9)',
        outlineWidth: 3
      };
    }

    // 初始化
    let bitmap = sprite.bitmap || { width: sprite.width, height: sprite.height };
    let _sprite = new Sprite(new Bitmap(bitmap.width, bitmap.height));
    if (style) {
      _sprite.bitmap.fontFace = style.fontFace || bitmap.fontFace;
      _sprite.bitmap.fontSize = style.fontSize || bitmap.fontSize;
      _sprite.bitmap.textColor = style.textColor || bitmap.textColor;
      _sprite.bitmap.outlineColor = style.outlineColor || bitmap.outlineColor;
      _sprite.bitmap.outlineWidth = style.outlineWidth || bitmap.outlineWidth;
    } else {
      _sprite.bitmap.fontFace = bitmap.fontFace;
      _sprite.bitmap.fontSize = bitmap.fontSize;
      _sprite.bitmap.textColor = bitmap.textColor;
      _sprite.bitmap.outlineColor = bitmap.outlineColor;
      _sprite.bitmap.outlineWidth = bitmap.outlineWidth;
    }
    _sprite.bitmap.drawText(text, 0, 0, bitmap.width, bitmap.height);
    _sprite.x = x;
    _sprite.y = y;
    sprite.parent.addChild(_sprite);

    const _frameCount = frameCount || 50;

    useAnime({
      frameDelay: frameDelay || 0,
      frameCount: _frameCount,
      autoUpdate: true,
      easingType: easingType || 'Quadratic',
      inout: inout || 'InOut',
    }, progress => {
      if (_sprite && _sprite.transform) {
        if (progress >= 0.5) {
          _sprite.alpha = 1 - (progress - 0.5) * 2;
        }
        _sprite.y = _sprite.y - progress * 0.6;
      }
      if (progress >= 1) {
        if (_sprite && _sprite.transform && _sprite.parent) {
          _sprite.destroy();
          _sprite = undefined;
        }
        if (callback) callback();
      }
    });

    return _sprite;
  }

  /**
   * 创建文字副本2
   * @param {object} config
   */
  window.createFloatText2 = function({ scene, bitmap, text, callback }) {
    const _info = {
      scene,
      sprite: undefined,
      anime: {
        frameIndex: 0,
        frameCount: 50,
      },
      update() {
        if (frameIndex < frameCount) {
          if (frameIndex >= frameCount / 2) {
            _info.sprite.alpha = 1 - (frameIndex - 0.5) / frameCount * 2;
          }
          _info.sprite.y = _info.sprite.y - frameIndex * 2;
          _info.anime.frameIndex++;
        } else {
          _info.sprite.destroy();
          _info.sprite = undefined;
          _info = undefined;
          if (callback) callback();
        }
      }
    };

    // 初始化
    const _sprite = new Sprite(new Bitmap(bitmap.width, bitmap.height));
    _sprite.bitmap.fontFace = bitmap.fontFace;
    _sprite.bitmap.fontSize = bitmap.fontSize;
    _sprite.bitmap.textColor = bitmap.textColor;
    _sprite.bitmap.outlineColor = bitmap.outlineColor;
    _sprite.bitmap.outlineWidth = bitmap.outlineWidth;
    _sprite.bitmap.drawText(text, 0, 0, bitmap.width, bitmap.height);
    _sprite.x = x;
    _sprite.y = y;
    _info.sprite = _sprite;
    scene.addChild(_sprite);

    return _info;
  }

  /**
   * 创建视频
   * @param {string} src 视频文件名
   * @param {{ x: number, y: number, parent: PIXI.Container }} config 
   */
  window.createVideo = function(src, { x, y, width, height, parent, alpha, floor }, callback) {
    let _canPlay = false;
    let _x = x || 0;
    let _y = y || 0;

    // 创建视频元素
    let videoSprite;
    const video = document.createElement('video');

    const _destroy = () => {
      videoSprite.destroy();
      video.remove();
    };

    video.src = `movies/${src}.webm`; // 视频文件路径
    video.loop = true; // 循环播放
    video.muted = true; // 静音
    video.addEventListener('canplay', () => {
      if (!_canPlay) {
        _canPlay = true;
        video.play(); // 播放视频

        // 将视频转换为纹理
        const texture = PIXI.Texture.from(video);
        texture.baseTexture.alphaMode = PIXI.ALPHA_MODES.PMA;
        // 创建精灵并添加到舞台
        videoSprite = new PIXI.Sprite(texture);
        videoSprite.x = _x || 0;
        videoSprite.y = _y || 0;
        if (alpha) videoSprite.alpha = alpha;

        videoSprite.width = width || Graphics.width;
        videoSprite.height = height || Graphics.height;

        if (typeof floor === 'function') {
          floor(videoSprite, parent);
        } else {
          if (floor) {
            if (floor > 0) {
              parent.addChildAt(videoSprite, floor);
            } else {
              parent.addChildAt(videoSprite, parent.children.length + floor);
            }
          } else {
            parent.addChild(videoSprite);
          }
        }

        if (callback) callback({ sprite: videoSprite, destroy: _destroy });
      }
    });

    return { destroy: _destroy, sprite: videoSprite };
  }

  /**
   * 创建小型CG视频
   * @param {string} src 视频文件名
   * @param {{ x: number, y: number, maskPath: string, parent: PIXI.Container }} config 
   */
  window.createMiniVideo = function(src, { x, y, parent, maskPath }) {

    ImageManager.loadImg('ui/interact/interact_mini_cg_bg').then(bgBitmap => {
      let _canPlay = false;
      let _x = x;
      let _y = y;

      // 创建视频元素
      const video = document.createElement('video');
      video.src = `movies/${src}.webm`; // 视频文件路径
      video.loop = true; // 循环播放
      video.muted = true; // 静音
      video.addEventListener('canplay', () => {
        if (!_canPlay) {
          _canPlay = true;
          video.play(); // 播放视频

          const videoContainer = new PIXI.Container();
          videoContainer.x = _x;
          videoContainer.y = _y;

          const videoPanelSprite = new Sprite(bgBitmap);
          videoContainer.addChild(videoPanelSprite);

          // 将视频转换为纹理
          const texture = PIXI.Texture.from(video);
          // 创建精灵并添加到舞台
          const videoSprite = new PIXI.Sprite(texture);

          const maskTexture = PIXI.Texture.from('img/pictures/ui/interact/interact_mini_cg_mask.png');
          const maskSprite = new PIXI.Sprite(maskTexture);

          videoSprite.width = bgBitmap.width;
          videoSprite.height = bgBitmap.height;
          videoSprite.mask = maskSprite;

          videoContainer.addChild(videoSprite);
          videoContainer.addChild(maskSprite);
          
          parent.addChild(videoContainer);
        }
      });
    });
  }

  /**
   * 根据角度、初始坐标和距离，计算新的坐标
   * @param {number} angle - 角度（以度为单位，0 到 360）
   * @param {number} x - 初始 x 坐标
   * @param {number} y - 初始 y 坐标
   * @param {number} distance - 距离
   * @returns {Object} - 返回计算后的新坐标 { x, y }
   */
  window.calculateAnglePosition = function(angle, x, y, distance) {
    // 将角度转换为弧度
    const radians = (angle * Math.PI) / 180;

    // 计算新的 x 和 y 坐标
    const newX = x + distance * Math.cos(radians);
    const newY = y + distance * Math.sin(radians);

    // 返回新坐标
    return { x: newX, y: newY };
  }

  /**
   * 开始对话
   * @param {{ name: string, text: string }[]} taskList - 对话列表
   * @param {{ name: string, text: string }[]} extraList - 附加功能列表
   * @param {() => void} callback - 回调函数
   * @param {boolean} [isOver=true] - 对话完成是否结束
   */
  window.startTalk = function(taskList = [], extraList = [], callback, isOver = true) {
    if (!taskList.length) return;

    const _talkList = [];
    for (let i = 0; i < taskList.length; i++) {
      if (taskList[i].face) {
        if (typeof taskList[i].face === 'string') {
          _talkList.push({ code: 355, indent: 0, parameters: [
            `Utils_Tachie.changeFace('${taskList[i].face}', undefined, '${taskList[i].actor || 'kana'}');`
          ] });
        } else if (Array.isArray(taskList[i].face)) {
          _talkList.push({ code: 355, indent: 0, parameters: [
            `Utils_Tachie.changeFace('${taskList[i].face[0]}', ${taskList[i].face[1] ? ('\'' + taskList[i].face[1] + '\'') : 'undefined'}, '${taskList[i].actor || 'kana'}');`
          ] });
        }
      }
      _talkList.push({ code: 101, indent: 0, parameters: [ "", 0, 0, 2, taskList[i].name || "" ] });
      taskList[i].text.split('\n').forEach(line => {
        const item = { code: 401, indent: 0, parameters: [ line ] };
        _talkList.push(item);
      });
    }
    for (let i = 0; i < extraList.length; i++) {
      _talkList.push(extraList[i]);
    }
    if (isOver) {
      _talkList.push({ code: 0, indent: 0, parameters: [] });
    }
    
    if (isOver) {
      if (this instanceof Game_Interpreter) {
        // this.setup([ { code: 117, indent: 0, parameters: [] } ], undefined, callback);
        this._childInterpreter = new Game_Interpreter(this._depth + 1);
        this._childInterpreter.setup(_talkList, 0, callback);
      } else {
        $gameMap._interpreter.setup(_talkList, undefined, callback);
      }
    } else {
      if (this instanceof Game_Interpreter) {
        let _commandCount = 0;
        while (this._list[_commandCount] && [357, 657].includes(this._list[_commandCount].code)) {
          _commandCount++;
        }


        this._list.splice(this._index, _commandCount);
        let _index = this._index;
        for (let i = 0; i < _talkList.length; i++, _index++) {
          const _item = _talkList[i];
          this._list.splice(_index, 0, _item);
        }
        this._index -= 1;
      } else {
        $gameMap._interpreter.setup(_talkList, undefined, callback);
      }
    }
  }

  /** 两个背景图是否相同 */
  window.isSameImage = (imgA, imgB) => {
    if (imgA === imgB) return true;

    const _imgAList = Array.isArray(imgA) ? imgA : [imgA];
    const _imgBList = Array.isArray(imgB) ? imgB : [imgB];
    if (_imgAList.length === _imgBList.length) {
      return _imgAList[0] === _imgBList[0];
    }

    return false;
  }

  /**
   * 打开网页
   * @param {string} url URL地址
   */
  window.openUrl = function(url) {
    if (window.nw && nw.Shell) {
      nw.Shell.openExternal(url);
    } else if (window.plus && window.plus.runtime) {
	    plus.runtime.openURL(url);
    } else if (window.uni) {
      uni.openURL({
        url: url,
        success: () => {},
        fail: (err) => {
          uni.showToast({
            title: '打开失败，请稍后重试',
            icon: 'none'
          });
          console.error('openURL Failed:' + url, err);
        }
      });
    } else {
      window.open(url, '_system');
    }
  }

  /**
   * 输入文本
   * @param {string} title 标题
   * @param {string} defaultText 默认文本
   * @param {text => void} onConfirm 确认事件
   * @param {() => void} onCancel 取消事件
   * @param {object} config 配置项
   * @param {number} config.maxlength 配置项
   */
  window.inputText = function(title, defaultText, onConfirm, onCancel, {
    maxlength = 10
  } = {
    maxlength: 10
  }) {
    const _container = new PIXI.Container();
    let _text = TranslateUtils.getText(defaultText) || '';
    let _isHover = false;
    let _width = 583;
    let _height = 375;

    const _scene = SceneManager._scene;
    _container.x = (Graphics.width - 583) / 2;
    _container.y = (Graphics.height - 375) / 2;

    // 背景
    const _bgSprite = new Sprite(ImageManager.loadPicture('ui/input-text/input-text-bg'));
    _container.addChild(_bgSprite);

    // 标题
    const _titleSprite = new Sprite(new Bitmap(583, 80));
    _titleSprite.y = 48;
    _titleSprite.bitmap.smooth = false;
    _titleSprite.bitmap.fontFace = 'Source Han Sans VF';
    _titleSprite.bitmap.fontSize = 30;
    _titleSprite.bitmap.textColor = '#FC8919';
    _titleSprite.bitmap.outlineColor = 'white';
    _titleSprite.bitmap.outlineWidth = 6;
    _titleSprite.bitmap.drawText(TranslateUtils.getText(title), 0, 0, 583, 80, 'center');
    _container.addChild(_titleSprite);
    
    // 输入框背景
    const _inputBgSprite = new Sprite(ImageManager.loadPicture('ui/input-text/input-text-input'));
    _inputBgSprite.x = (_width - 300) / 2;
    _inputBgSprite.y = 170;
    _container.addChild(_inputBgSprite);

    // 输入框内容
    const _inputField = new Sprite(new Bitmap(300, 40));
    _inputField.x = (_width - _inputField.width) / 2;
    _inputField.y = 170;
    _inputField.alpha = 0;
    _inputField.bitmap.smooth = false;
    _inputField.bitmap.fontSize = 20;
    _inputField.bitmap.textColor = '#000000';
    _inputField.bitmap.fontFace = 'Source Han Sans VF';
    _inputField.bitmap.outlineColor = '#FC8919';
    _inputField.bitmap.outlineWidth = 4;
    _inputField.bitmap.drawText(_text, 0, 0, _inputField.width, _inputField.height, 'center');
    _container.addChild(_inputField);
    
    // 按钮背景
    let _confirmButtonBg = new Sprite(ImageManager.loadPicture('ui/button/button_2'));
    _confirmButtonBg.x = (_width - 180) / 2;
    _confirmButtonBg.y = 260;
    _container.addChild(_confirmButtonBg);
    // 按钮文本
    let _confirmButtonText = new Sprite(new Bitmap(180, 46));
    _confirmButtonText.x = (_width - 180) / 2;
    _confirmButtonText.y = 260;
    _confirmButtonText.bitmap.smooth = false;
    _confirmButtonText.bitmap.fontSize = 20;
    _confirmButtonText.bitmap.textColor = '#FC8919';
    _confirmButtonText.bitmap.outlineColor = 'white';
    _confirmButtonText.bitmap.outlineWidth = 4;
    _confirmButtonText.bitmap.drawText(TranslateUtils.getText('确认'), 0, 0, 180, 46, 'center');
    _container.addChild(_confirmButtonText);

    _scene.addChild(_container);

    const _inputDom = document.createElement('input');
    _inputDom.setAttribute('class','sys-text-input');
    _inputDom.setAttribute('maxlength', (maxlength || 10) + '');
    _inputDom.setAttribute('type','text');
    _inputDom.setAttribute('tabindex','-1');
    document.body.appendChild(_inputDom);

    setTimeout(() => {
      _inputDom.focus();
      // _inputDom.style.pointerEvents = 'none';
      _inputDom.style.userSelect = 'text';
      _inputDom.style.position = 'fixed';
      _inputDom.style.textAlign = 'center';
      _inputDom.style.zIndex = '99';
      // _inputDom.style.opacity = '0.3';
      const _scale = Graphics._realScale;
      _inputDom.style.left = Math.round(_scale * (_inputField.x + _container.x) + gameVideo.offsetLeft) + 'px';
      _inputDom.style.top = Math.round(_scale * (_inputField.y + _container.y) + gameVideo.offsetTop) + 'px';
      _inputDom.style.width = _scale * (_inputField.width) + 'px';
      _inputDom.style.height = _scale * (_inputField.height) + 'px';
      _inputDom.style.fontSize = Math.round(20 * _scale) + 'px';
      _inputDom.style.lineHeight = _scale * (_inputField.height - 2) + 'px';
      _inputDom.value = _text;

      _inputDom.addEventListener('input', () => {
        _text = _inputDom.value;
        _inputField.bitmap.clear();
        _inputField.bitmap.drawText(_text, 0, 0, _inputField.width, _inputField.height, 'center');
      });
      _inputDom.addEventListener('keydown', (event) => {
        _text = _inputDom.value;
        _inputField.bitmap.clear();
        _inputField.bitmap.drawText(_text, 0, 0, _inputField.width, _inputField.height, 'center');

        event.stopPropagation();
      });
      
      // 开启更新
      moduleTicker.on(_onUpdate);
    }, 100);

    const _onSizeChange = () => {
      const _scale = Graphics._realScale;
      if (_inputDom && _inputField && _inputField.transform) {
        _inputDom.style.left = Math.round(_scale * (_inputField.x + _container.x) + gameVideo.offsetLeft) + 'px';
        _inputDom.style.top = Math.round(_scale * (_inputField.y + _container.y) + gameVideo.offsetTop) + 'px';
        _inputDom.style.width = _scale * (_inputField.width) + 'px';
        _inputDom.style.height = _scale * (_inputField.height) + 'px';
        _inputDom.style.fontSize = Math.round(20 * _scale) + 'px';
        _inputDom.style.lineHeight = _scale * (_inputField.height - 2) + 'px';
      }
    };

    window.addEventListener('resize', _onSizeChange);

    // 更新事件
    const _onUpdate = () => {
      if (!_container.transform) {
        _inputDom.remove();
        return;
      }
      const _x = TouchInput.x;
      const _y = TouchInput.y;
      if (_x >= _container.x + _confirmButtonBg.x && _x <= _container.x + _confirmButtonBg.x + _confirmButtonBg.width && 
        _y >= _container.y + _confirmButtonBg.y && _y <= _container.y + _confirmButtonBg.y + _confirmButtonBg.height
      ) {
        if (!_isHover) {
          _confirmButtonBg.bitmap = ImageManager.loadPicture('ui/button/button_2_active');
          _confirmButtonText.bitmap.textColor = '#d46f0c';
          _confirmButtonText.bitmap.clear();
          _confirmButtonText.bitmap.drawText(TranslateUtils.getText('确认'), 0, 0, 180, 46, 'center');
          _isHover = true;
        }
        if (TouchInput.isPressed()) {
          if (_text.trim() === '') {
            _text = defaultText;
          }
          window.removeEventListener('resize', _onSizeChange);
          moduleTicker.off(_onUpdate);
          _container.destroy();
          _inputDom.remove();
          onConfirm(_text);
        }
      } else {
        if (_isHover) {
          _confirmButtonBg.bitmap = ImageManager.loadPicture('ui/button/button_2');
          _confirmButtonText.bitmap.textColor = '#FC8919';
          _confirmButtonText.bitmap.clear();
          _confirmButtonText.bitmap.drawText(TranslateUtils.getText('确认'), 0, 0, 180, 46, 'center');
          _isHover = false;
        }
      }
    };
  };

  /**
   * 合并对象，补充对象中缺失的属性
   * @param {object} a - 第一个对象
   * @param {object} b - 第二个对象
   * @returns {object} - 合并后的对象
   */
  window.mergeMissing = function(a, b) {
    // 如果 a 或 b 不是对象，直接返回 a
    if (typeof a !== 'object' || a === null || typeof b !== 'object' || b === null) {
      return a;
    }
    
    // 创建结果对象，保持原对象的原型链
    const result = Array.isArray(a) ? [...a] : { ...a };
    
    // 遍历 b 的所有属性
    for (const key in b) {
      if (b.hasOwnProperty(key)) {
        // 如果 a 中不存在该属性，直接赋值
        if (!(key in result)) {
          result[key] = b[key];
        } else if (typeof result[key] === 'object' && result[key] !== null && typeof b[key] === 'object' && b[key] !== null) {
          // 如果两边都是对象，递归处理
          result[key] = mergeMissing(result[key], b[key]);
        }
        // 如果 a 中已存在该属性且不是对象，跳过不处理
      }
    }
    
    return result;
  }

  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'startGameTransform', function(args) {
      $gameScreen.startGameTransform(
        args.bgScale ? (Number(args.bgScale) / 100) : undefined, 
        args.bgTransformX ? Number(args.bgTransformX) : undefined, 
        args.bgTransformY ? Number(args.bgTransformY) : undefined, 
        args.duration ? Number(args.duration) : undefined
      );
    });
    PluginManager.registerCommand(PluginName, 'eventEnd', function(args) {
      $gameData.isEventBusy = false;
    });
    PluginManager.registerCommand(PluginName, 'useAlert', function(args) {
      useAlert({ text: args.text, type: args.type,  });
    });
    
    PluginManager.registerCommand(PluginName, 'saveEvent', function(args) {
      this._historyList = cloneLoop(this._list);
      if (args.hasExecStep === 'true') {
        this._historyIndex = this._index;
      }
    });
    
    PluginManager.registerCommand(PluginName, 'restoreEvent', function(args) {
      if (!this._historyList) return;

      this._list = cloneLoop(this._historyList);
      if (args.hasExecStep === 'true' && this._historyIndex !== undefined) {
        this._index = this._historyIndex;
      }
      this._historyList = undefined;
      this._historyIndex = undefined;
    });
    
    PluginManager.registerCommand(PluginName, 'talkAbout', function(args) {
      const _params = args.params ? JSON.parse(args.params).map(i => i ? TranslateUtils.getText(new Function('return ' + i)()) : '') : undefined;
      let _chatInfo = Utils_ChatMessage.getChat(args.topicId, _params);

      if (!_chatInfo || _chatInfo.length < 1) {
        throw new Error(`找不到话题 ${args.topicId}`);
      }
      // else {
      //   _chatInfo = _chatInfo[0];
      // }
      
      if ($gameMessage.isBusy()) {
        return false;
      }

      startTalk.call(this, _chatInfo, [], undefined, args.isOver === 'false' ? false : true);

      // this.setWaitMode("message");

      return true;
    });

  }

})();