//=============================================================================
// ** RPG Maker MZ - Hakubox_Video_Play.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 视频背景组件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox@outlook.com
 * @target MZ
 * 
 * @help
 * 
 * 视频背景组件可以让你在游戏中播放视频，并将其作为背景，不过首先需要配置对应视频列表。
 * 
 * 调用代码：
 * videoModule.playVideo("视频路径或者别名");
 * videoModule.playVideo("视频路径或者别名", { x: 0, y: 0 });
 * videoModule.stopVideo();
 * 
 * 
 * @command playVideo
 * @text 播放视频
 * @desc 播放视频。
 * 
 * @arg path
 * @text 视频文件路径
 * @desc 输入视频文件的路径，用于在游戏中显示，不用包含后缀名".webm"和前缀"movies/"。
 * @type text
 * @dir movies
 * 
 * @arg floor
 * @text 插入层级
 * @desc 视频精灵的插入层级，默认top为最上层，normal在地图上则会插入到对话框下方。
 * @type select
 * @option top - 最上层
 * @value top
 * @option normal - 普通
 * @value normal
 * @option custom - 自定义
 * @value custom
 * @default top
 * 
 * @arg customFloor
 * @parent floor
 * @text 自定义层级
 * @desc 自定义层级，如果floor选择custom，则使用此层级，在层级中可编写代码。
 * @type text
 * 
 * 
 * @param replacePlayVideo
 * @text 是否替代播放视频功能
 * @type boolean
 * @desc 是否替代RM自带的“播放视频”功能。
 * @on 替代
 * @off 不替代
 * @default false
 * 
 * @param basePath
 * @text 基础路径
 * @type text
 * @desc 所有视频的基础路径，默认为"movies/"。
 * @default movies/
 * 
 * @param videoList
 * @text 视频列表
 * @type struct<VideoInfo>[]
 * @desc 视频列表，可以设置多个视频，每个视频可以设置多个属性。
 * @default []
 * 
 * 
 */
/*~struct~VideoInfo:
 * 
 * @param path
 * @text 视频文件路径
 * @type text
 * @desc 输入视频文件的路径，用于在游戏中显示，不用包含后缀名".webm"和前缀"movies/"。
 * 
 * @param alias
 * @text 别名
 * @desc 当前视频的别名，可以当路径使用。
 * @type note
 * 
 * @param desc
 * @text 备注
 * @desc 当前视频的备注。
 * @type note
 * 
 * @param config
 * @text 配置
 * 
 * @param loop
 * @parent config
 * @text 是否循环播放
 * @type boolean
 * @desc 是否循环播放视频，不循环则播放完成后自动销毁。
 * @on 循环
 * @off 不循环
 * @default true
 * 
 * @param speed
 * @parent config
 * @text 播放速度
 * @desc 播放速度
 * @type text
 * @default 1.0
 * 
 * @param duration
 * @parent config
 * @text 播放时长（秒）
 * @type number
 * @desc 是否限制播放时长，限制则播放时长结束后自动销毁。
 * 
 * @param x
 * @parent config
 * @text X坐标
 * @desc X坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * @param y
 * @parent config
 * @text Y坐标
 * @desc Y坐标
 * @type number
 * @default 0
 * @min -10000
 * @max 10000
 * 
 * @param width
 * @parent config
 * @text 宽度
 * @desc 宽度，默认为原始视频宽度
 * @type number
 * 
 * @param height
 * @parent config
 * @text 高度
 * @desc 高度，默认为原始视频高度
 * @type number
 * 
 * @param alpha
 * @parent config
 * @text 不透明度
 * @desc 不透明度，默认为1，最小为0则完全透明。
 * @type text
 * @default 1
 * 
 * @param blendMode
 * @parent config
 * @text 混合模式
 * @type select
 * @option NORMAL - 默认,0
 * @value NORMAL
 * @option ADD - 线性减淡 (添加),1
 * @value ADD
 * @option MULTIPLY - 正片叠底,2
 * @value MULTIPLY
 * @option SCREEN - 滤色,3
 * @value SCREEN
 * @option OVERLAY - 叠加,4
 * @value OVERLAY
 * @option DARKEN - 变暗,5
 * @value DARKEN
 * @option LIGHTEN - 变亮,6
 * @value LIGHTEN
 * @option COLOR_DODGE - 颜色减淡,7
 * @value COLOR_DODGE
 * @option COLOR_BURN - 颜色加深,8
 * @value COLOR_BURN
 * @option HARD_LIGHT - 强光,9
 * @value HARD_LIGHT
 * @option SOFT_LIGHT - 柔光,10
 * @value SOFT_LIGHT
 * @option DIFFERENCE - 差值,11
 * @value DIFFERENCE
 * @option EXCLUSION - 排除,12
 * @value EXCLUSION
 * @option HUE - 色相,13
 * @value HUE
 * @option SATURATION - 饱和度,14
 * @value SATURATION
 * @option COLOR - 颜色,15
 * @value COLOR
 * @option LUMINOSITY - 明度,16
 * @value LUMINOSITY
 * @default NORMAL
 * 
 * @param useClearColor
 * @text 是否使用移除背景色滤镜
 * @type boolean
 * @desc 视频播放时，是否清除某个背景颜色。
 * @on 使用
 * @off 不使用
 * @default false
 * 
 * @param floor
 * @text 插入层级
 * @desc 视频精灵的插入层级，默认top为最上层，normal在地图上则会插入到对话框下方。
 * @type select
 * @option top - 最上层
 * @value top
 * @option normal - 普通
 * @value normal
 * @option custom - 自定义
 * @value custom
 * @default top
 * 
 * @param customFloor
 * @parent floor
 * @text 自定义层级
 * @desc 自定义层级，如果floor选择custom，则使用此层级，在层级中可编写代码。
 * @type text
 * 
 * @param clearColor
 * @parent useClearColor
 * @text 要消除的背景颜色
 * @type text
 * @desc 视频播放时，消除的背景颜色，颜色值必须为"x,x,x"，例如"1.0,1.0,1.0"代表白色。
 * 
 * @param uSensitivity
 * @parent useClearColor
 * @text 中心阈值
 * @desc 中心阈值。抠图的中心范围
 * @type text
 * @default 0.2
 * 
 * @param uSmoothing
 * @parent useClearColor
 * @text 平滑范围
 * @desc 平滑范围。边缘平滑度，值越大边缘越柔和。
 * @type text
 * @default 0.1
 * 
 * @param useAdjustmentFilter
 * @text 是否使用调节滤波器
 * @type boolean
 * @desc 视频播放时，是否使用调节滤波器，使用可以更好清除黑色背景。【注：要使用当前功能必须引入pixi-filters.js文件】
 * @on 使用
 * @off 不使用
 * @default false
 * 
 * @param contrast
 * @parent useAdjustmentFilter
 * @text 调整对比度
 * @desc 调整对比度。增加对比度会使亮部更亮，暗部更暗。
 * @type text
 * @default 2
 * 
 * @param gamma
 * @parent useAdjustmentFilter
 * @text 调整伽马值
 * @desc 调整伽马值。增加伽马值会主要压暗中间色调和暗部。
 * @type text
 * @default 1.5
 * 
 * @param useFade
 * @text 是否使用淡入淡出
 * @desc 是否使用淡入淡出
 * @type boolean
 * @on 使用
 * @off 不使用
 * @default false
 * 
 * @param fadeInDuration
 * @parent useFade
 * @text 淡入时长（帧数）
 * @desc 视频开始播放显示的淡入时长（帧数）
 * @type number
 * @on 使用
 * @off 不使用
 * @default 0
 * 
 * @param fadeOutDuration
 * @parent useFade
 * @text 淡出时长（帧数）
 * @desc 视频结束播放显示的淡出时长（帧数）
 * @type number
 * @on 使用
 * @off 不使用
 * @default 0
 * 
 */
// #endregion
(() => {
  /** 插件名称 */
  const PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_Video_Play";
  
  // #region 插件参数解释器
  class PluginParamsParser {
    constructor(predictEnable = true) {
      this._predictEnable = predictEnable;
    }
    static parse(params, typeData, predictEnable = true) {
      return new PluginParamsParser(predictEnable).parse(params, typeData);
    }
    parse(params, typeData, loopCount = 0) {
      if (++loopCount > 255)
        throw new Error("endless loop error");
      const result = {};
      for (const name in typeData) {
        if (params[name] === "" || params[name] === undefined) {
          result[name] = null;
        }
        else {
          result[name] = this.convertParam(params[name], typeData[name], loopCount);
        }
      }
      if (!this._predictEnable)
        return result;
      if (typeof params === "object" && !(params instanceof Array)) {
        for (const name in params) {
          if (result[name])
            continue;
          const param = params[name];
          const type = this.predict(param);
          result[name] = this.convertParam(param, type, loopCount);
        }
      }
      return result;
    }
    convertParam(param, type, loopCount) {
      if (typeof type === "string") {
        let str = param;
        if (str[0] == '"' && str[str.length - 1] == '"') {
          str = str.substring(1, str.length - 1).replace(/\\n/g, '\n').replace(/\\"/g, '"')
        }
        return this.cast(str, type);
      }
      else if (typeof type === "object" && type instanceof Array) {
        const aryParam = JSON.parse(param);
        if (type[0] === "string") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else if (type[0] === "number") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else {
          if (!aryParam.length) return [];
          else return aryParam.map((strParam) => this.parse(JSON.parse(strParam), type[0]), loopCount);
        }
      }
      else if (typeof type === "object") {
        return this.parse(JSON.parse(param), type, loopCount);
      }
      else {
        throw new Error(`${type} is not string or object`);
      }
    }
    cast(param, type) {
      switch (type) {
        case "any":
          if (!this._predictEnable)
            throw new Error("Predict mode is disable");
          return this.cast(param, this.predict(param));
        case "string":
          return param;
        case "number":
          if (param.match(/^\-?\d+\.\d+$/))
            return parseFloat(param);
          return parseInt(param);
        case "boolean":
          return param === "true";
        default:
          throw new Error(`Unknow type: ${type}`);
      }
    }
    predict(param) {
      if (param.match(/^\-?\d+$/) || param.match(/^\-?\d+\.\d+$/)) {
        return "number";
      }
      else if (param === "true" || param === "false") {
        return "boolean";
      }
      else {
        return "string";
      }
    }
  }
  // #endregion

  const typeDefine = {
    videoList: []
  };

  const params = PluginParamsParser.parse(PluginManager.parameters(PluginName), typeDefine);

  const videoList = params.videoList;

  // GLSL代码字符串
  const finalChromaKeyShader = `
    // --- Advanced Chroma Key Shader (Standalone) ---
    varying vec2 vTextureCoord;
    uniform sampler2D uSampler; // 前景视频纹理

    uniform vec3 uKeyColor;   // 要移除的颜色
    uniform float uThreshold; // 灵敏度/阈值
    uniform float uSoftness;  // 边缘柔和度
    uniform float uSpill;     // 色彩溢出抑制强度 (0.0 到 1.0)

    // 函数：将RGB转换为YUV色彩空间，以获得更精确的颜色判断
    vec3 rgb2yuv(vec3 c) {
      return vec3(
        0.299 * c.r + 0.587 * c.g + 0.114 * c.b, // Y (亮度)
        -0.169 * c.r - 0.331 * c.g + 0.5 * c.b,  // U (色度)
        0.5 * c.r - 0.419 * c.g - 0.081 * c.b   // V (色度)
      );
    }

    void main() {
      // 1. 获取前景颜色
      vec4 fgColor = texture2D(uSampler, vTextureCoord);

      // 2.【高级算法】在YUV空间中计算遮罩（Matte）
      // 这种方法能更好地抵抗亮度变化，抠图更干净
      vec3 keyYUV = rgb2yuv(uKeyColor);
      vec3 pixelYUV = rgb2yuv(fgColor.rgb);
      
      // 我们只比较色度(UV分量)，忽略亮度(Y分量)，让抠图更稳定
      float chromaDist = distance(keyYUV.gb, pixelYUV.gb); 
      float matte = smoothstep(uThreshold, uThreshold + uSoftness, chromaDist);

      // 3.【高级算法】执行色彩溢出抑制
      // 计算蓝色分量超出红绿平均值的部分，这就是"溢出"的蓝光
      float spillAmount = max(0.0, fgColor.b - (fgColor.r + fgColor.g) * 0.5);
      vec3 deSpilledColor = vec3(
          fgColor.r,
          fgColor.g,
          fgColor.b - spillAmount * uSpill
      );

      // 4.【专业输出】输出预乘Alpha (Pre-multiplied Alpha) 格式的颜色
      // 这是图形学中进行Alpha混合的最佳实践，能避免边缘出现黑边或白边
      // 最终颜色 = 去色边后的颜色 * 遮罩透明度
      // 最终透明度 = 遮罩透明度
      gl_FragColor = vec4(deSpilledColor * matte, matte);
    }
  `;

  /** 视频容器 */
  class VideoSprite extends PIXI.Sprite {
    constructor(path, config = {}) {
      super();

      if (!path) throw new Error("视频路径参数不能为空");

      const videoInfo = videoList.find(i => i.path === path || i.alias === path || `${params.basePath}${i.path}`.replace('movies/', '') === path.replace('movies/', ''));

      if (!videoInfo) throw new Error(`找不到视频 ${path}`);

      this.info = JSON.parse(JSON.stringify(videoInfo));

      this._canPlay = false;
      this.video = document.createElement('video');

      const _path = path.startsWith(params.basePath) ? videoInfo.path : `${params.basePath}${videoInfo.path}`;

      this.video.src = `${_path}.webm`; // 视频文件路径
      this.video.loop = videoInfo.loop || true; // 循环播放
      this.video.muted = true; // 静音
      
      // 移动设备的特殊处理
      if (Utils.isMobileDevice()) {
        this.video.setAttribute('playsinline', '');
        this.video.setAttribute('webkit-playsinline', '');
      }

      // 不循环
      if (config.loop === false) {
        video.addEventListener("ended", (event) => {
          this.destroy();
        });
      }

      // 限制时长
      if (videoInfo.duration) {
        this.video.addEventListener("timeupdate", (event) => {
          if (this.video) {
            if (this.video.currentTime >= videoInfo.duration) {
              this.destroy();
            }
          }
        });
      }

      // 将视频转换为纹理
      const texture = PIXI.Texture.from(this.video);
      // 创建精灵并添加到舞台
      this.texture = texture;
      this.x = videoInfo.x || config.x || 0;
      this.y = videoInfo.y || config.y || 0;
      this.video = this.video;
      this.filters = [];
      if (videoInfo.useFade && videoInfo.fadeInDuration) {
        this.alpha = 0;
        this.finalAlpha = videoInfo.alpha || config.alpha || 1;

        // 淡入
        videoModule.useTimeout(() => {
          this.alpha = this.finalAlpha;
        }, videoInfo.fadeInDuration, (progress) => {
          this.alpha = this.finalAlpha * progress;
        });
      } else {
        this.alpha = videoInfo.alpha || config.alpha || 1;
      }

      // 合成模式
      if (videoInfo.blendMode) {
        this.blendMode = PIXI.BLEND_MODES[videoInfo.blendMode];
      }
      
      // 移除对应颜色
      if (videoInfo.useClearColor) {
        if (!videoInfo.clearColor) throw new Error("在移除背景色的情况下，clearColor必须设置");
        // 创建滤镜实例，注意uniforms里多了一项
        const ultimateFilter = new PIXI.Filter(null, finalChromaKeyShader, {
          uKeyColor: videoInfo.clearColor.split(','),
          uThreshold: videoInfo.uThreshold || 0.2,   // 初始阈值，根据视频微调
          uSoftness: videoInfo.uSoftness || 0.2,    // 初始柔和度，根据视频微调
          uSpill: videoInfo.uSpill || 0.6        // 去蓝边强度，一般设为1.0即可
        });
        this.filters.push(ultimateFilter);
      }

      // 使用调节滤波器
      if (videoInfo.useAdjustmentFilter) {
        // this.blendMode = 0;
        const adjustmentFilter = new PIXI.filters.AdjustmentFilter();
        this.filters.push(adjustmentFilter);
        // 大幅增加对比度。这会使亮部更亮，暗部更暗。
        adjustmentFilter.contrast = videoInfo.contrast || 2; // 从 2 开始尝试，逐步增加
        // 增加伽马值。这会主要压暗中间色调和暗部。
        adjustmentFilter.gamma = videoInfo.gamma || 1.5; // 从 1.5 开始尝试，逐步增加
      }

      if (Utils.RPGMAKER_NAME === "MZ" && this.texture.baseTexture) {
        this.texture.baseTexture.alphaMode = PIXI.ALPHA_MODES.PMA;
      }

      this.width = videoInfo.width || config.width || Graphics.width;
      this.height = videoInfo.height || config.height || Graphics.height;

      videoModule.videoSprite = this;

      this.loadCount = 0;
      this.maxCount = 100;

      this.loadVideo();
    }
    
    /** 设置播放速度 */
    setSpeed(speed) {
      if (this.video) {
        this.video.playbackRate = speed;
      }
    }

    loadVideo() {
      setTimeout(() => {
        if (this.loadCount > this.maxCount) {
          throw new Error("视频加载超时");
        }
        this.loadCount++;
        if (this.video && (this.video.readyState == 1 || this.video.readyState == 4)) {
          if (!this._canPlay) {
            this._canPlay = true;
            this.video.play(); // 播放视频
            // 在VideoSprite构造函数中
            this.video.playbackRate = this.info.speed || 1.0;
          }
        } else {
          this.loadVideo();
        }
      }, 50);
    }

    destroy() {
      if (this.info.useFade && this.info.fadeOutDuration) {
        // 淡出
        videoModule.useTimeout(() => {
          this.alpha = 0;
          super.destroy();
          this.video.remove();
          this.video = null;
          videoModule.videoSprite = undefined;
        }, this.info.fadeOutDuration, (progress) => {
          this.alpha = this.finalAlpha - this.finalAlpha * progress;
        });
      } else {
        super.destroy();
        this.video.remove();
        this.video = null;
        videoModule.videoSprite = undefined;
      }
    }

    update() {
      // 调用模块更新
      videoModule.update();
      // 如果视频和纹理都存在，确保纹理更新
      if (this._updateCounter % 5 === 0) { // 每两帧更新一次
        if (this && this.texture && this.texture.baseTexture && this.video) {
          this.texture.baseTexture.update();
        }
        this._updateCounter = 0;
      }
      this._updateCounter++;
    }
  }
  window.VideoSprite = VideoSprite;

  /** 视频模块 */
  const videoModule = {
    videoSprite: undefined,
    _tickers: [],
    on(callback) {
      for (let i = 0; i < this._tickers.length; i++) {
        if (this._tickers[i] === undefined) {
          this._tickers[i] = callback;
          return;
        }
      }
      this._tickers.push(callback);
    },
    off(callback) {
      const _index = this._tickers.indexOf(callback);
      if (_index >= 0) {
        this._tickers.splice(_index, 1);
      }
    },
    update() {
      for (let i = 0; i < this._tickers.length; i++) {
        if (this._tickers[i]) this._tickers[i]();
      }
    },
    /**
     * 使用延时功能
     * @param {() => void} callback 回调函数
     * @param {number} delay 等待帧数
     * @param {(progress: number) => void} update 更新函数
     * @param {{ easingType: string, inout: string }} info 动画信息
     */
    useTimeout(callback, delay, update, info = { easingType: 'Linear', inout: 'None' }) {
      const _timer = {
        frameIndex: 0,
        frameCount: delay,
        stop() {
          this.frameIndex = 0;
          videoModule.off(_timerUpdate);
        }
      };

      const _timerUpdate = () => {
        _timer.frameIndex++;

        if (update) {
          const progress = MotionEasing.getEasing(info.easingType, info.inout)(_timer.frameIndex / _timer.frameCount);
          update(progress, _timer.frameIndex);
        }

        if (_timer.frameIndex >= _timer.frameCount) {
          _timer.frameIndex = 0;
          videoModule.off(_timerUpdate);
          callback();
        }
      };

      videoModule.on(_timerUpdate);
      return _timer;
    },
    /**
     * 切换视频层级
     * @param {object} videoInfo 视频信息
     * @param {PIXI.Container} videoSprite 视频容器
     * @param {string | (() => void)} replaceFloor 重设层级
     */
    changeFloor(videoInfo, videoSprite, replaceFloor) {
      let _floor = replaceFloor || videoInfo.floor || 'top';

      if (typeof _floor === 'function') {
        _floor.call(SceneManager._scene, videoSprite);
      } else if (_floor === 'custom' && videoInfo.customFloor) {
        new Function('sprite', videoInfo.customFloor).call(SceneManager._scene, videoSprite);
      } else {
        switch (_floor) {
          case 'top':
            SceneManager._scene.addChild(videoSprite);
            break;
          case 'normal':
            if (this instanceof Scene_Map) {
              SceneManager._scene._spriteset.addChild(videoSprite);
            } else {
              SceneManager._scene.addChild(videoSprite);
            }
            break;
          default:
            SceneManager._scene.addChild(videoSprite);
            break;
        }
      }
    },
    /** 播放视频 */
    playVideo(path, { floor, x, y } = {}) {
      videoModule.stopVideo();

      const _videoSprite = new VideoSprite(path, {
        x: x || 0,
        y: y || 0
      });

      videoModule.changeFloor(_videoSprite.info, _videoSprite, floor);
    },
    /** 停止视频 */
    stopVideo() {
      if (videoModule.videoSprite) {
        videoModule.videoSprite.destroy();
      }
    }
  }
  window.videoModule = videoModule;

  Game_Interpreter.prototype.command261 = function(p) {
    if (params.replacePlayVideo) {
      videoModule.playVideo(p[0]);
    } else {
      if ($gameMessage.isBusy()) return false;
      const name = p[0];
      if (name.length > 0) {
        const ext = this.videoFileExt();
        Video.play("movies/" + name + ext);
        this.setWaitMode("video");
      }
    }
    return true;
  };

  // MZ 指令
  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(PluginName, 'playVideo', (args) => {
      videoModule.playVideo(args.path, {
        floor: args.floor
      });
    });
  }
  
  // MV 指令
  const Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'videoplay') {
      videoModule.playVideo(args[0], { floor: args[1] });
    } else if (command === 'videostop') {
      videoModule.stopVideo();
    }
  };

})();