//=============================================================================
// ScreenNoise.js
//=============================================================================
// [Update History]
// 2017.Feb.12 Ver0.0.0 Closed Version
// 2019.Jun.27 Ver1.0.0 First Release

/*:
 * @plugindesc 在地图画面和标题画面显示噪点效果
 * @author 神無月サスケ (感谢 Rokan)
 *
 * @param Use_At_Title
 * @desc 是否在标题画面显示
 * @type boolean
 * @default true
 *
 * @requiredAssets img/system/noise_base
 * @requiredAssets img/system/noise_line
 * @requiredAssets img/system/noise_dot
 * 
 * @help 
 * 使用前准备:
 * 本插件需要3个图像文件才能运行。
 * 请将noise_base、noise_line、noise_dot放入img/system文件夹。
 *
 * 插件功能:
 * 在地图画面显示类似老式胶片的噪点效果。
 * 
 * 插件命令:
 * 在地图画面中调用以下插件命令：
 * ScreenNoise start   # 开始噪点效果
 * ScreenNoise end     # 结束噪点效果
 *
 * 版权声明:
 * 本插件基于Rokan的RGSS3脚本素材开发
 * 参见"回想领域" http://kaisou-ryouiki.sakura.ne.jp/
 * 特别感谢Rokan。
 *
 * 许可证:
 * 本插件使用MIT许可证发布
 * 详细内容请参考：
 * http://opensource.org/licenses/mit-license.php
 */
/*:ja
 * @plugindesc マップ画面とタイトル画面にノイズをかけます
 * @author 神無月サスケ (原案:ろかん)
 * 
 * @param Use_At_Title
 * @desc タイトル画面での表示 0:しない 1:する
 * @type boolean
 * @default true
 *
 * @requiredAssets img/system/noise_base
 * @requiredAssets img/system/noise_line
 * @requiredAssets img/system/noise_dot
 * 
 * @help 
 * 最初にすべきこと:
 * このプラグインの実行には、添付の画像ファイルが必要です。
 * img/system フォルダに、noise_base, noise_line, noise_dotを置いて下さい。
 *
 * プラグイン概要:
 * 古い映像フィルムのようなノイズ効果を与えます。
 * タイトルでの表示はパラメータで設定してください。
 * マップでの表示切替はプラグインコマンドで行ってください。
 * 
 * プラグインコマンド:
 * ScreenNoise start   # ノイズ開始
 * ScreenNoise end     # ノイズ終了
 *
 * 著作権表記:
 * このプラグインは、ろかん氏のRGSS3素材をベースに作成しました。
 * Webサイト：回想領域 http://kaisou-ryouiki.sakura.ne.jp/
 * ろかん氏に謝意を示します。
 *
 * ライセンス表記:
 * このプラグインは MIT ライセンスで配布されます。
 * ご自由にお使いください。
 * http://opensource.org/licenses/mit-license.php
 */

(function() {
  //
  // 处理参数
  //
  var parameters = PluginManager.parameters('ScreenNoise');
  var useAtTitle = parameters['Use_At_Title'] == 'true';

  //
  // 处理插件命令
  //
  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'ScreenNoise') {
      switch (args[0]) {
      case 'start':
        $gameScreen.startScreenNoise();
        break;
      case 'end':
        $gameScreen.endScreenNoise();
        break;
      }
    }
  };

  //
  // 游戏画面类
  //
  var _Game_Screen_initialize = Game_Screen.prototype.initialize;
  Game_Screen.prototype.initialize = function() {
    _Game_Screen_initialize.call(this);
    this.displayScreenNoise = false;
  };

  Game_Screen.prototype.startScreenNoise = function() {
    this.displayScreenNoise = true;
  };

  Game_Screen.prototype.endScreenNoise = function() {
    this.displayScreenNoise = false;
  };

  // ------------------------------------------------------
  //
  // 噪点背景精灵
  //
  function Sprite_NoiseBase() {
    this.initialize.apply(this, arguments);
  }

  Sprite_NoiseBase.prototype = Object.create(Sprite.prototype);
  Sprite_NoiseBase.prototype.constructor = Sprite_NoiseBase;

  Sprite_NoiseBase.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.reset();
  };

  // 0: "源混合"（SRC_OVER）- 默认混合模式，新图层覆盖在旧图层上
  // 1: "叠加"（ADD）- 颜色值相加，产生更亮的效果
  // 2: "正片叠底"（MULTIPLY）- 颜色值相乘，产生更暗的效果
  // 3: "滤色"（SCREEN）- 反相相乘后再反相，产生更亮的效果
  // 4: "叠加"（OVERLAY）- 结合正片叠底和滤色，增强对比度
  // 5: "变暗"（DARKEN）- 选择源和目标中较暗的像素
  // 6: "变亮"（LIGHTEN）- 选择源和目标中较亮的像素
  // 7: "颜色减淡"（COLOR_DODGE）- 使底色更亮以反映源颜色
  // 8: "颜色加深"（COLOR_BURN）- 使底色更暗以反映源颜色
  // 9: "强光"（HARD_LIGHT）- 类似叠加，但更强烈
  // 10: "柔光"（SOFT_LIGHT）- 类似强光，但效果更柔和
  // 11: "差值"（DIFFERENCE）- 从较亮的颜色中减去较暗的颜色
  // 12: "排除"（EXCLUSION）- 类似差值，但对比度较低
  // 13: "色相"（HUE）- 应用源的色相，保留目标的饱和度和亮度
  // 14: "饱和度"（SATURATION）- 应用源的饱和度，保留目标的色相和亮度
  // 15: "颜色"（COLOR）- 应用源的色相和饱和度，保留目标的亮度
  // 16: "亮度"（LUMINOSITY）- 应用源的亮度，保留目标的色相和饱和度
  // 17: "正常非预乘"（NORMAL_NPM）- 类似源混合，但不预乘alpha
  // 18: "叠加非预乘"（ADD_NPM）- 类似叠加，但不预乘alpha
  // 19: "滤色非预乘"（SCREEN_NPM）- 类似滤色，但不预乘alpha
  // 20: "无"（NONE）- 不应用任何混合
  // 21: "源内"（SRC_IN）- 显示源在目标内部的部分
  // 22: "源外"（SRC_OUT）- 显示源在目标外部的部分
  // 23: "源顶"（SRC_ATOP）- 显示源在目标顶部的部分
  // 24: "目标上"（DST_OVER）- 显示目标在源上方的部分
  // 25: "目标内"（DST_IN）- 显示目标在源内部的部分
  // 26: "擦除"（ERASE）- 擦除重叠区域
  // 27: "目标顶"（DST_ATOP）- 显示目标在源顶部的部分
  // 28: "减去"（SUBTRACT）- 从目标中减去源
  // 29: "异或"（XOR）- 排除源和目标重叠的部分
  
  Sprite_NoiseBase.prototype.reset = function() {
    this.blendMode = 2;
    this.bitmap = null;
    this.blink = true;
    return this;
  };

  Sprite_NoiseBase.prototype.setup = function() {
    this.bitmap = ImageManager.loadSystem('noise_base');
    this.blink = true;
    this.updateMain();
    return this;
  };

  Sprite_NoiseBase.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (NoiseEffects.needUpdate()) {
      this.updateMain();
    }
  };

  Sprite_NoiseBase.prototype.updateMain = function() {
    this.x = -(Math.randomInt(this.width - Graphics.width));
    this.opacity = this.blink ? 230 : 255;
    this.blink = !this.blink;
  };

  // ------------------------------------------------------
  //
  // 垂直线条噪点精灵
  //
  function Sprite_NoiseLine() {
    this.initialize.apply(this, arguments);
  }

  Sprite_NoiseLine.prototype = Object.create(Sprite.prototype);
  Sprite_NoiseLine.prototype.constructor = Sprite_NoiseLine;

  Sprite_NoiseLine.VectorX = [-3, -2, -1, 0, 1, 2, 3];
  Sprite_NoiseLine.OpacitySpeed = [-75, -30, -15, 0, 15, 30, 45, 75];


  Sprite_NoiseLine.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.reset();
  };

  Sprite_NoiseLine.prototype.reset = function() {
    this.bitmap = null;
    return this;
  };

  Sprite_NoiseLine.prototype.setup = function(startX) {
    this.bitmap = ImageManager.loadSystem('noise_line');
    this.x = startX || 0;
    var vx = Sprite_NoiseLine.VectorX;
    this.vectorX = vx[Math.randomInt(vx.length)];
    var os = Sprite_NoiseLine.OpacitySpeed;
    this.opacitySpeed = os[Math.randomInt(os.length)];
    this.updateMain();
    return this;
  };

  Sprite_NoiseLine.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (NoiseEffects.needUpdate()) {
      this.updateMain();
    }
  };

  Sprite_NoiseLine.prototype.updateMain = function() {
    this.x = (this.x + this.vectorX).clamp(0, Graphics.width);
    this.y = -Math.randomInt(this.height - Graphics.height);
    this.opacity += this.opacitySpeed;
    if(Math.randomInt(6) === 0) {
      var vx = Sprite_NoiseLine.VectorX;
      this.vectorX = vx[Math.randomInt(vx.length)];
    }
    if(Math.randomInt(6) === 0) {
      var os = Sprite_NoiseLine.OpacitySpeed;
      this.opacitySpeed = os[Math.randomInt(os.length)];
    }
  };

  // ------------------------------------------------------
  //
  // 点状噪点精灵
  //
  function Sprite_NoiseDot() {
    this.initialize.apply(this, arguments);
  }

  Sprite_NoiseDot.prototype = Object.create(Sprite.prototype);
  Sprite_NoiseDot.prototype.constructor = Sprite_NoiseDot;

  Sprite_NoiseDot.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.reset();
  };

  Sprite_NoiseDot.prototype.reset = function() {
    this.bitmap = null;
    return this;
  };

  Sprite_NoiseDot.prototype.setup = function(startX) {
    this.bitmap = ImageManager.loadSystem('noise_dot');
    this.updateMain();
    return this;
  };

  Sprite_NoiseDot.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (NoiseEffects.needUpdate()) {
      this.updateMain();
    }
  };

  Sprite_NoiseDot.prototype.updateMain = function() {
    if (Math.randomInt(50) === 0) {
      this.scale.x = (Math.randomInt(100) + 1.0) / 100.0;
      this.scale.y = (Math.randomInt(100) + 1.0) / 100.0;
      this.rotation = Math.randomInt(360) * Math.PI / 180;
      this.x = Math.randomInt(Graphics.width);
      this.y = Math.randomInt(Graphics.height);
      this.opacity = 255;
    } else {
      this.opacity = 0;
    }
  };

  // ------------------------------------------------------
  //
  // 噪点效果精灵组
  //
  function NoiseEffects() {
    this.initialize.apply(this, arguments);
  }

  NoiseEffects.prototype = Object.create(Sprite.prototype);
  NoiseEffects.prototype.constructor = NoiseEffects;

  NoiseEffects.needUpdate = function() {
    return Graphics.frameCount % 3 === 0;
  };

  NoiseEffects.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.sprites = [];
    this.reset();
  };

  NoiseEffects.prototype.reset = function() {
    while (this.sprites.length > 0) {
      this.removeChild(this.sprites.shift().reset());
    }
    this.working = false;
  };

  NoiseEffects.prototype.setup = function() {
    this.reset();
    this.createNoise();
    this.working = true;
  };

  NoiseEffects.prototype.createNoise = function() {
    this.sprites.push((new Sprite_NoiseBase()).setup());
    this.sprites.push((new Sprite_NoiseLine()).setup(75));
    this.sprites.push((new Sprite_NoiseLine()).setup(525));
    this.sprites.push((new Sprite_NoiseLine()).setup(600));
    this.sprites.push((new Sprite_NoiseLine()).setup(750));
    this.sprites.push((new Sprite_NoiseDot()).setup(0));
    for(var i = 0; i < this.sprites.length; i++) {
      this.addChild(this.sprites[i]);
    }
  };

  NoiseEffects.prototype.update = function() {
    this.updateDisplayState();
    Sprite.prototype.update.call(this);
  };

  NoiseEffects.prototype.updateDisplayState = function() {
    if (this.working !== !!$gameScreen.displayScreenNoise) {
      if (!!$gameScreen.displayScreenNoise) {
        this.setup();
      } else {
        this.reset();
      }
    }
  };

  // ------------------------------------------------------
  //
  // 标题画面噪点效果
  //
  function NoiseEffectsTitle() {
    this.initialize.apply(this, arguments);
  }

  NoiseEffectsTitle.prototype = Object.create(NoiseEffects.prototype);
  NoiseEffectsTitle.prototype.constructor = NoiseEffectsTitle;

  NoiseEffectsTitle.prototype.initialize = function() {
    NoiseEffects.prototype.initialize.call(this);
    this.setup();
  };

  NoiseEffectsTitle.prototype.updateDisplayState = function() {
  };

  var _Scene_Title_createScreenNoise = Scene_Title.prototype.createScreenNoise;
  Scene_Title.prototype.createScreenNoise = function() {
    _Scene_Title_createScreenNoise.call(this);
    this.noiseEffects = new NoiseEffects();
    this.addChild(this.noiseEffects);

    if (useAtTitle) {
      this.addChild(new NoiseEffectsTitle());
    }
  };

  window.createScreenNoise = function() {
    const _scene = SceneManager._scene;
    
    _scene.noiseEffects = new NoiseEffects();
    _scene.addChild(_scene.noiseEffects);

    _scene.addChild(new NoiseEffectsTitle());
  };

})();
