//=============================================================================
// X_Error.js
//=============================================================================
/*:
 * @plugindesc 美化报错界面
 * @author X
 *
 * @param PageTitle
 * @text 页面标题
 * @desc 错误页面主标题（支持HTML）
 * @default <h2>⚠️ 游戏运行异常！</h2>
 *
 * @param ErrorDetailTitle
 * @text 技术细节标题
 * @desc 错误堆栈部分的标题（支持HTML）
 * @default <h3>■ 技术细节 ■</h3>
 *
 * @param CloseButtonText
 * @text 关闭按钮文字
 * @desc 关闭按钮显示的文字
 * @default 安全退出游戏
 *
 * @param ReloadButtonText
 * @text 重开按钮文字
 * @desc 重开按钮显示的文字
 * @default 重新载入游戏
 * 
 *
 * @param LogNotice
 * @text 日志提示
 * @desc 日志保存提示（支持HTML）
 * @default <p>エラーログはゲームのルートディレクトリのError.logファイルに保存されました。</p><p>The error log has been saved to the Error.log file in the game's root directory.</p><p>错误日志已保存至游戏根目录的Error.log文件</p>
 *
 * @param ContactInfo
 * @text 联系方式
 * @desc 显示的联系信息（支持HTML）
 * @default <p><div>エラーログはゲームのルートディレクトリのError.logファイルに保存されました。</div><div>The error log has been saved to the Error.log file in the game's root directory.</div><div>错误日志已保存至游戏根目录的Error.log文件</div></p>
 *
 * @param TextColor
 * @text 文字颜色
 * @desc 主要文字颜色（十六进制）
 * @default #030303
 *
 *
 * @param AccentColor
 * @text 强调色
 * @desc 强调元素颜色（十六进制）
 * @default #FF3F45
 *
 * @param BackgroundImage
 * @text 背景图片
 * @desc 背景图片路径（相对游戏根目录）
 * @default img/error.png
 *
 * @param BackgroundSize
 * @text 背景尺寸
 * @desc 背景图片显示方式（cover/contain）
 * @default cover
 *
 * @help
 * 美化报错时候的界面
 * 除了插件设置之外，如果你会使用css，可以直接修改代码里的css来美化
 */
(function() {
    if (!window.require) {
        return;
    }

    let fs;
    let path;

    if (window.require) {
        fs = require('fs');
        path = require('path');
    } else {
        fs = {
            appendFileSync() {},
        };
        path = [];
    }
    const parameters = PluginManager.parameters('X_Error');

    // 配置参数
    const config = {
        // 文本内容
        pageTitle: parameters['PageTitle'],
        errorDetailTitle: parameters['ErrorDetailTitle'],
        reloadButtonText: parameters['ReloadButtonText'],
        closeButtonText: parameters['CloseButtonText'],
        logNotice: parameters['LogNotice'],
        contactInfo: parameters['ContactInfo'],
        
        // 颜色配置
        colors: {
            text: parameters['TextColor'] || '#ffffff',
            accent: parameters['AccentColor'] || '#ff5555'
        },
        
        // 背景配置
        background: {
            image: parameters['BackgroundImage'] || '',
            size: parameters['BackgroundSize'] || 'cover'
        }
    };

    // 生成背景样式
    function createBackgroundStyle() {
        if (!config.background.image) return '';
        
        let imagePath;
        
        if (window.process) {
            imagePath = path.join(process.cwd(), config.background.image)
                .replace(/\\/g, '/')
                .replace(/ /g, '%20');
        } else {
            imagePath = config.background.image;
        }

        return `
            background-image: url("file:///${imagePath}");
            background-size: ${config.background.size};
            background-position: center;
            background-repeat: no-repeat;
        `;
    }

    // 创建样式表
    const errorStyle = document.createElement('style');
    errorStyle.textContent = `
        .x-error-container {
            position: fixed;
            user-select: text;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: black;
            color: ${config.colors.text};
            z-index: 99999;
            overflow: auto;
            font-family: 'Segoe UI', sans-serif;
            line-height: 1.6;
            ${createBackgroundStyle()}
        }
        .x-error-content {
            max-width: 1000px;
            margin: 20px auto;
            padding: 25px;
            padding-top: 5px;
            border-radius: 10px;
            font-size: 18px;
        }
        .x-error-content pre {
            font-size: 13px;
        }
        .x-error-btn-group {
            display: block;
            text-align: center;
        }
        .x-error-btn {
            display: inline-block;
            padding: 12px 30px;
            background: ${config.colors.accent};
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 16px;
            margin: 25px auto;
            cursor: pointer;
            transition: all 0.2s;
        }
        .x-error-btn:hover {
            filter: brightness(1.1);
        }
        .x-error-btn + .x-error-btn {
            margin-left: 30px;
        }
        h2, h3 {
            color: ${config.colors.accent};
            margin: 20px 0;
        }
        pre {
            background: rgba(255,255,255,0.3);
            padding: 15px;
            border-left: 3px solid ${config.colors.accent};
            overflow-x: auto;
        }
        .x-error-container a {
            color: #FF5D5D;
            font-weight: bold;
            font-size: 18px;
        }
    `;
    document.head.appendChild(errorStyle);

    // 错误容器
    const errorContainer = document.createElement('div');
    errorContainer.className = 'x-error-container';
    errorContainer.style.display = 'none';
    document.body.appendChild(errorContainer);

    // 错误处理核心逻辑
    function handleError(name, message, stack) {
        if (errorContainer.style.display === 'block') return;
        Graphics.isError = true;

        errorContainer.innerHTML = `
            <div class="x-error-content">
                ${config.pageTitle}
                ${config.errorDetailTitle}
                <pre>${stack}</pre>
                ${config.logNotice}
                ${config.contactInfo}
                <div class="x-error-btn-group">
                    <button class="x-error-btn" onclick="SceneManager.reloadGame()">
                        ${config.reloadButtonText}
                    </button>
                    <button class="x-error-btn" onclick="SceneManager.quitGame()">
                        ${config.closeButtonText}
                    </button>
                </div>
            </div>
        `;
        errorContainer.style.display = 'block';
        writeErrorLog(name, message, stack);
        stopGame();
    }

    // 错误日志记录
    function writeErrorLog(name, message, stack) {
        const logContent = `[${new Date().toLocaleString()}]
==============================
${name}
${message}
${stack}
==============================\n\n`;
        fs.appendFileSync(path.join(process.cwd(), 'Error.log'), logContent);
    }

    SceneManager.quitGame = function() {
        if (window.nw && nw.App) {
            nw.App.quit();
        } else {
            window.close();
        }
    }

    // 停止游戏运行
    function stopGame() {
        SceneManager.stop();
        // Graphics.stop();
        [ 'keydown', 'mousedown', 'touchstart' ].forEach(event => {
            window.addEventListener(event, e => e.stopImmediatePropagation(), true);
        });
    }

    // 重写错误处理接口
    window.onerror = function(msg, url, line, col, error) {
        handleError(
            error?.name || 'Error',
            error?.message || msg,
            error?.stack || `at ${url}:${line}:${col}`
        );
    };

    Graphics.printLoadingError = function(url) {
        handleError(
            'ResourceLoadError',
            `无法加载资源: ${url}`,
            new Error().stack
        );
    };

    SceneManager.onError = function(event) {
        // console.error(event.filename, event.lineno);
        try {
            this.stop();
            Graphics.printError("Error", event.message, event);
            AudioManager.stopAll();
        } catch (e) {
            //
        }
    };

    SceneManager.catchException = function(e) {
        handleError(e.name, e.message, e.stack);
        this.requestUpdate();
    };

    Graphics._makeErrorHtml = function(name, message) {
        if (!name && !message) return ''; 
        handleError(name, message, new Error().stack);
        return '';
    };
})();