/*:
 * @target MZ
 * @author Zaun
 * @plugindesc 性能优化插件 ver 1.35
 * @base Zaun_Core
 * @orderAfter Zaun_Core
 * @orderBefore Zaun_Spine
 * @help
 * 2024.5.5 PM 17:00
 * 初版完成 1.00
 * 该插件可以和spine一起用，但
 * 必须放在Zaun_Spine之前
 * 除此之外
 * 该插件需要和Zaun_Core一起使用，
 * 且置于后者下方
 * 2024.5.6 PM 20:00
 * 版本变化 1.00 -> 1.01
 * 优化了游戏的刷新逻辑
 * 取消了PIXI.Application的使用
 * 移除项 Graphics.app
 * 增加项，现在可以直接访问Graphics._stage
 * 
 * 2024.6.15 PM 16:00
 * 1.31->1.32
 * Graphics.app虽然被移除
 * 但为了兼容性，Graphics.app.renderer 依然可以访问
 * 等价于Graphics.renderer
 * 建议直接访问后者
 * 移除了加载相关的简化
 * 某些插件的兼容性考虑
 * 窗口的updateFilterArea优化了一下
 * 在上次更新中
 * 已经 存在一个Sprite的属性
 * worldPosition 用来获取当前精灵的世界坐标点
 * 这个属性只存在于rm的sprite上
 * 
 * @param maxFps
 * @text 锁定设备的最大fps
 * @type select
 * @option 30fps
 * @value 30
 * @option 60fps
 * @value 60
 * @option 90fps
 * @value 90
 * @option 120fps
 * @value 120
 * @default 60
 * 
 * @param disableAnimationUpgrade
 * @text 是否禁用粒子动画优化
 * @type boolean
 * @on 禁用
 * @off 启用
 * @default false
 */
"use strict";
Zaun.Performance = ((t) => {
    const { ParseSystem } = Zaun.Core;
    const pluginName = "Zaun_Performance";
    const origParameters = PluginManager.parameters(pluginName);
    const parameters = ParseSystem.toParse(origParameters);
    const { maxFps, disableAnimationUpgrade } = parameters;
    Graphics.initialize = function () {
        this._width = 0;
        this._height = 0;
        this._defaultScale = 1;
        this._realScale = 1;
        this._errorPrinter = null;
        this._tickHandler = null;
        this._canvas = null;
        this._fpsCounter = null;
        this._loadingSpinner = null;
        this._stretchEnabled = this._defaultStretchMode();
        this._effekseer = null;
        this._wasLoading = false;
        this.frameCount = 0;
        this._deltaTime = 0;
        this._deltaMs = 0;
        this.boxWidth = this._width;
        this.boxHeight = this._height;
        this.requestId = 0;
        this._stage = null;
        this.updateHandler = this.update.bind(this);
        this.lastTime = 0;
        this._app = Object.create(null);
        this._updateRealScale();
        this._createAllElements();
        this._disableContextMenu();
        this._setupEventHandlers();
        this._createPixiApp();
        this._createEffekseerContext();
        return true;
    };
    // Reflect.defineProperty(Graphics, "app", {
    //     get() {
    //         console.warn("Deprecated! app has been removed");
    //         return this._app;
    //     },
    // });

    Reflect.defineProperty(Graphics, "stage", {
        get() {
            return this._stage;
        },
    });
    Graphics.resize = function (width, height) {
        this._width = width;
        this._height = height;
        this.renderer.resize(width, height);
        this._updateAllElements();
    };
    Graphics.startGameLoop = function () {
        this.update();
    };
    Graphics.FPSCounter.prototype.endTick = function (deltaTime, change) {
        this.fps = (1000 / deltaTime) >> 0;
        this.duration = change >> 0;
        if (this._tickCount++ % 30 === 0) {
            this._update();
        }
    };
    Graphics.FPSCounter.prototype._update = function () {
        const count = this._showFps ? this.fps : this.duration;
        this._labelDiv.textContent = this._showFps ? "FPS" : "ms";
        this._numberDiv.textContent = count >> 0;
    };
    function getInternal(targetFps) {
        switch (targetFps) {
            case 30: {
                return 32;
            }
            case 60: {
                return 15;
            }
            case 90: {
                return 10;
            }
            case 120: {
                return 7;
            }
            default:
                return 15;
        }
    }
    const INTERNAL = getInternal(maxFps);
    Graphics.update = function (time) {
        const deltaTime = time - this.lastTime;
        if (deltaTime >= INTERNAL) {
            this.lastTime = time;
            this._deltaTime = deltaTime;
            this._deltaMs = deltaTime / 1000;
            this._tickHandler(deltaTime);
            this._stage !== null && this.renderer.render(this._stage);
            const now = performance.now();
            const change = now - time;
            this._fpsCounter.endTick(deltaTime, change);
        }
        this.requestId = requestAnimationFrame(this.updateHandler);
    };
    Graphics.stopGameLoop = function () {
        cancelAnimationFrame(this.requestId);
    };
    Graphics._createPixiApp = function () {
        try {
            const renderer = PIXI.autoDetectRenderer({
                view: this._canvas,
                powerPreference: "high-performance",
                backGround: "#000000",
                hello: true,
            });
            this.renderer = renderer;
            this._app.renderer = renderer;
        } catch (e) {
            console.error(e);
        }
    };
    Graphics._createEffekseerContext = function () {
        if (this.renderer && globalThis.effekseer) {
            try {
                this._effekseer = effekseer.createContext();
                if (this._effekseer) {
                    this._effekseer.init(this.renderer.gl);
                    this._effekseer.setRestorationOfStatesFlag(false);
                }
            } catch (e) {
                console.error(e);
            }
        }
    };
    Graphics.setStage = function (stage) {
        this._stage = stage;
    };

    DataManager.metaRegexp = /<([^<>:]+)(:?)([^>]*)>/g;
    DataManager.getDataType = function (object) {
        switch (object) {
            case globalThis.$dataArmors: {
                return "isArmor";
            }
            case globalThis.$dataWeapons: {
                return "isWeapon";
            }
            case globalThis.$dataSkills: {
                return "isSkill";
            }
            case globalThis.$dataItems: {
                return "isItem";
            }
            case globalThis.$dataStates: {
                return "isState";
            }
            case globalThis.$dataActors: {
                return "isActor";
            }
            default:
                return "isEvent";
        }
    };
    DataManager.extractArrayMetadata = function (array) {
        if (Array.isArray(array)) {
            const type = this.getDataType(array);
            for (let i = 1; i < array.length; i++) {
                const data = array[i];
                if (data !== null) {
                    data.isEmpty = data.name === "";
                    data[type] = true;
                    if (Reflect.has(data, "note")) {
                        this.extractMetadata(data);
                    }
                }
            }
        }
    };
    DataManager.isSkill = function (item) {
        return item && item.isSkill;
    };
    DataManager.isItem = function (item) {
        return item && item.isItem;
    };
    DataManager.isWeapon = function (item) {
        return item && item.isWeapon;
    };
    DataManager.isArmor = function (item) {
        return item && item.isArmor;
    };
    DataManager.extractMetadata = function (data) {
        const meta = {};
        const regexp = this.metaRegexp;
        for (; ;) {
            const match = regexp.exec(data.note);
            if (match === null) break;
            const [, name, pattern, value] = match;
            meta[name] = pattern === ":" ? value : true;
        }
        data.meta = meta;
    };
    ColorManager.initialize = function () {
        this.colorCache = [
            "#ffffff",
            "#20a0d6",
            "#ff784c",
            "#66cc40",
            "#99ccff",
            "#ccc0ff",
            "#ffffa0",
            "#808080",
            "#c0c0c0",
            "#2080cc",
            "#ff3810",
            "#00a010",
            "#3e9ade",
            "#a098ff",
            "#ffcc20",
            "#000000",
            "#84aaff",
            "#ffff40",
            "#ff2020",
            "#202040",
            "#e08040",
            "#f0c040",
            "#4080c0",
            "#40c0f0",
            "#80ff80",
            "#c08080",
            "#8080ff",
            "#ff80ff",
            "#00a040",
            "#00e060",
            "#a060e0",
            "#c080ff",
        ];
        this._paddingColor = "#34cfff";
        this._outlineColor = "rgba(0, 0, 0, 0.6)";
        this._dimColor1 = "rgba(0, 0, 0, 0.6)";
        this._dimColor2 = "rgba(0, 0, 0, 0)";
        this._itemBackColor1 = "rgba(32, 32, 32, 0.5)";
        this._itemBackColor2 = "rgba(0, 0, 0, 0.5)";
    };
    ColorManager.textColor = function (n) {
        return this.colorCache[n];
    };
    ColorManager.pendingColor = function () {
        return this._paddingColor;
    };
    ColorManager.hpColor = function (actor) {
        if (!actor) {
            return this.normalColor();
        } else if (actor.isDead()) {
            return this.deathColor();
        } else if (actor.isDying()) {
            return this.crisisColor();
        } else {
            return this.normalColor();
        }
    };
    ColorManager.mpColor = function (/*actor*/) {
        return this.normalColor();
    };
    ColorManager.tpColor = function (/*actor*/) {
        return this.normalColor();
    };
    ColorManager.paramchangeTextColor = function (change) {
        if (change > 0) {
            return this.powerUpColor();
        } else if (change < 0) {
            return this.powerDownColor();
        } else {
            return this.normalColor();
        }
    };
    ColorManager.outlineColor = function () {
        return this._outlineColor;
    };
    ColorManager.dimColor1 = function () {
        return this._dimColor1;
    };
    ColorManager.dimColor2 = function () {
        return this._dimColor2;
    };
    ColorManager.itemBackColor1 = function () {
        return this._itemBackColor1;
    };
    ColorManager.itemBackColor2 = function () {
        return this._itemBackColor2;
    };
    ColorManager.initialize();
    const PIXI_Container_updateTransform = PIXI.Container.prototype.updateTransform;
    PIXI.Container.prototype.updateTransform = function () {
        PIXI_Container_updateTransform.call(this);
        this.updateWorldPosition();
    }
    PIXI.Container.prototype.updateWorldPosition = function () {
    }
    const Sprite_initialize = Sprite.prototype.initialize;
    Sprite.prototype.initialize = function (...args) {
        this.worldPosition = new Point();
        this.worldTestPoint = new Point();
        Sprite_initialize.call(this, ...args);
    }
    Sprite.prototype.updateWorldPosition = function () {
        this.worldTestPoint.set(0, 0);
        this.worldTransform.apply(this.worldTestPoint, this.worldPosition);
    }
    if (!disableAnimationUpgrade) {
        Sprite_Animation.prototype.setup = function (targets, animation, mirror, delay) {
            this._targets = targets;
            this._animation = animation;
            this._mirror = mirror;
            this._delay = delay;
            this._effect = EffectManager.load(animation.effectName);
            this._playing = true;
            const timings = animation.soundTimings.concat(animation.flashTimings);
            for (const timing of timings) {
                if (timing.frame > this._maxTimingFrames) {
                    this._maxTimingFrames = timing.frame;
                }
            }
            const x = mirror ? -1 : 1;
            const y = -1;
            const p = -(this._viewportSize / Graphics._height);
            this.projectMatrix = [x, 0, 0, 0, 0, y, 0, 0, 0, 0, 1, p, 0, 0, 0, 1,];
            this.cameraMatrix = [1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, -10, 1];
            this.enableRender = false;
        };
        Sprite_Animation.prototype.update = function () {
            if (this._delay > 0) {
                this._delay--;
            } else if (this._playing) {
                if (!this._started) {
                    if (this._effect.isLoaded) {
                        this._handle = Graphics.effekseer.play(this._effect);
                        this._started = true;
                        this.enableRender = true;
                    }
                } else {
                    this.updateEffectGeometry();
                    this.updateMain();
                    this.updateFlash();
                }
            } else {
                this.enableRender = false;
            }
        }
        Sprite_Animation.prototype.setViewport = function (renderer) {
            const animation = this._animation;
            const vw = 4096;
            const vh = 4096;
            const vx = animation.offsetX - 2048;
            const vy = animation.offsetY - 2048;
            let x = 0, y = 0;
            if (animation.displayType === 2) {
                x = Graphics._width / 2;
                y = Graphics._height / 2;
            } else {
                const targets = this._targets;
                const length = targets.length;
                for (let i = 0; i < length; i++) {
                    const sprite = targets[i];
                    const pos = this.targetSpritePosition(sprite);
                    x += pos.x;
                    y += pos.y;
                }
                x /= length;
                y /= length;
            }
            renderer.gl.viewport(vx + x, vy + y, vw, vh);
        };
        Sprite_Animation.prototype.targetSpritePosition = function (sprite) {
            const point = sprite.worldPosition;
            point.y -= sprite._texture._frame.height / 2;
            if (this._animation.alignBottom) {
                point.y = 0;
            }
            return point;
        }
        Sprite_Animation.prototype.updateEffectGeometry = function () {
            const animation = this._animation;
            const rotation = animation.rotation;
            const handle = this._handle;
            const scale = animation.scale / 100;
            const r = Math.PI / 180;
            const rx = rotation.x * r;
            const ry = rotation.y * r;
            const rz = rotation.z * r;
            handle.setLocation(0, 0, 0);
            handle.setRotation(rx, ry, rz);
            handle.setScale(scale, scale, scale);
            handle.setSpeed(animation.speed / 100);
        }
        Sprite_Animation.prototype._render = function (renderer) {
            if (this.enableRender) {
                this.onBeforeRender(renderer);
                const effekseer = Graphics._effekseer;
                effekseer.setProjectionMatrix(this.projectMatrix);
                effekseer.setCameraMatrix(this.cameraMatrix);
                this.setViewport(renderer);
                effekseer.beginDraw();
                effekseer.drawHandle(this._handle);
                effekseer.endDraw();
                this.resetViewport(renderer);
                this.onAfterRender(renderer);
            }
        };
    }
    Scene_Base.prototype.createColorFilter = function () {
        (this._colorFilter = new ColorFilter()),
            (this.filters = [this._colorFilter]),
            (this.filterArea = new Rectangle(
                0,
                0,
                2000,
                2000
            ));
    };
    Spriteset_Base.prototype.createOverallFilters = function () {
        (this.filters = []),
            (this._overallColorFilter = new ColorFilter()),
            (this.filterArea = new Rectangle(
                0,
                0,
                2000,
                2000
            )),
            this.filters.push(this._overallColorFilter);
    };
    Spriteset_Base.prototype.createBaseFilters = function () {
        (this._baseSprite.filters = []),
            (this._baseColorFilter = new ColorFilter()),
            (this._baseSprite.filterArea = new Rectangle(
                0,
                0,
                2000,
                2000
            )),
            this._baseSprite.filters.push(this._baseColorFilter);
    };
    SceneManager.snapForBackground = function () {
        if (this._backgroundBitmap === null) {
            this._backgroundBitmap = new Bitmap(Graphics._width, Graphics._height);
        }
        Bitmap.snap2(this._scene, this._backgroundBitmap);
    };
    Bitmap.snap2 = function (scene, bitmap) {
        const w = Graphics._width;
        const h = Graphics._height;
        bitmap.resize(w, h);
        if (bitmap.renderTexture === void 0) {
            bitmap.renderTexture = PIXI.RenderTexture.create(w, h);
        }
        bitmap.renderTexture.resize(w, h);
        const renderTexture = bitmap.renderTexture;
        const renderer = Graphics._app.renderer;
        const gl = renderer.gl;
        renderer.render(scene, renderTexture);
        const pixels = new Uint8Array(4 * w * h);
        gl.readPixels(0, 0, w, h, gl.RGBA, gl.UNSIGNED_BYTE, pixels);
        const imageData = new ImageData(
            new Uint8ClampedArray(pixels.buffer),
            w,
            h
        );
        const context = bitmap._context;
        context.putImageData(imageData, 0, 0);
        bitmap._baseTexture.update();
    }
    Bitmap.prototype.resize = function (t, e) {
        const canvas = this.canvas;
        if (t !== canvas.width || e !== canvas.height) {
            t = canvas.width = Math.max(t || 0, 1);
            e = canvas.height = Math.max(e || 0, 1);
            this._baseTexture.setSize(t, e, 1);
        }
    };
    JsonEx._encode = function (t, e) {
        if (e >= this.maxDepth) throw new Error("Object too deep");
        var i = Object.prototype.toString.call(t);
        if ("[object Object]" === i || "[object Array]" === i) {
            null !== Reflect.getPrototypeOf(t) &&
                "Object" !== (i = t.constructor.name) &&
                "Array" !== i &&
                (t["@"] = i);
            for (const r of Reflect.ownKeys(t)) t[r] = this._encode(t[r], e + 1);
        }
        return t;
    };
    Game_BattlerBase.prototype.initMembers = function () {
        (this._hp = 1),
            (this._mp = 0),
            (this._tp = 0),
            (this._hidden = !1),
            (this._states = []),
            this.clearParamPlus(),
            this.clearStates(),
            this.clearBuffs();
    }
    Game_BattlerBase.prototype.clearStates = function () {
        (this._states.length = 0), (this._stateTurns = {});
    };
    Game_ActionResult.prototype.initialize = function () {
        (this.addedStates = []),
            (this.removedStates = []),
            (this.addedBuffs = []),
            (this.addedDebuffs = []),
            (this.removedBuffs = []),
            this.clear();
    };
    Game_ActionResult.prototype.clear = function () {
        (this.used = !1),
            (this.missed = !1),
            (this.evaded = !1),
            (this.physical = !1),
            (this.drain = !1),
            (this.critical = !1),
            (this.success = !1),
            (this.hpAffected = !1),
            (this.hpDamage = 0),
            (this.mpDamage = 0),
            (this.tpDamage = 0),
            (this.addedStates.length = 0),
            (this.removedStates.length = 0),
            (this.addedBuffs.length = 0),
            (this.addedDebuffs.length = 0),
            (this.removedBuffs.length = 0);
    };
    SceneManager.initGraphics = function () {
        if (!Graphics.initialize())
            throw new Error("Failed to initialize graphics.");
        Graphics.setTickHandler(this.updateMain.bind(this));
    };
    t.VERSION = "1.35";
    return t
})(Object.create(null));