/*:
 * @target MZ
 * @plugindesc Play footstep sounds when the player moves.
 * @author Malomi
 * @url https://gendaigoyaku.com/
 *
 * @param Footstep Sound
 * @text Footstep sound effect file name.
 * @desc This is the file name of the sound effect to be played as footsteps. Please place it in the /audio/se folder.
 * @default Move6
 * @type file
 * @dir audio/se/
 *
 * @param Volume
 * @text Sound effect volume.
 * @desc Sound effect volume (0-100). The higher the value, the louder the sound.
 * @default 90
 * @type number
 * @max 100
 * @min 0
 *
 * @param Pitch
 * @text Sound effect pitch.
 * @desc Sound effect pitch (50-150). Lower values produce lower sounds, higher values produce higher sounds. 100 is the default.
 * @default 100
 * @type number
 * @max 150
 * @min 50
 *
 * @param Interval
 * @text Footstep interval.
 * @desc Footstep interval (in frames). Lower values result in slower intervals, while higher values result in faster intervals.
 * @default 30
 * @type number
 * @min 1
 *
 * @command EnableFootstep
 * @text Enable footsteps.
 * @desc Enable footstep sounds when the player moves.
 *
 * @command DisableFootstep
 * @text Disable footsteps.
 * @desc Disable footstep sounds when the player moves.
 *
 * @help
 * ----------------------------------------------------------------------------
 * PlayerFootstep.js
 * ----------------------------------------------------------------------------
 * Play the specified footstep sound effect when the player moves.
 * 
 * Please place the sound effect files in the /audio/se folder.
 * 
 * You can toggle footstep sound playback ON/OFF using the plugin command.
 * 
 * Plugin Command:
 * - "EnableFootstep" -> Enable footstep sound playback.
 * - "DisableFootstep" -> Disable footstep sound playback.
 * ----------------------------------------------------------------------------
 * License:
 * This is the MIT License. 
 * It can be used for both commercial and non-commercial purposes.
 * ----------------------------------------------------------------------------
 */

/*:ja
 * @target MZ
 * @plugindesc プレイヤーが移動する際に足音を再生するプラグイン
 * @author まろみ
 * @url https://gendaigoyaku.com/
 *
 * @param Footstep Sound
 * @text 足音の効果音ファイル名
 * @desc 足音として再生する効果音のファイル名を設定してね。効果音のファイルを置く場所は、/audio/se フォルダに配置してね。
 * @default Move6
 * @type file
 * @dir audio/se/
 *
 * @param Volume
 * @text 効果音のボリューム
 * @desc 効果音のボリューム（0～100）数値が大きいほど音も大きくなるよ
 * @default 90
 * @type number
 * @max 100
 * @min 0
 *
 * @param Pitch
 * @text 効果音のピッチ
 * @desc 効果音のピッチ（50～150）数値が低いと低い音、数値が高いと高い音だよ。100が真ん中、ふつうだよ。
 * @default 100
 * @type number
 * @max 150
 * @min 50
 *
 * @param Interval
 * @text 足音の間隔
 * @desc 足音を再生する間隔（フレーム数）数値が低いとゆっくり、数値が高いと速くなるよ。
 * @default 30
 * @type number
 * @min 1
 *
 * @command EnableFootstep
 * @text 足音を有効化
 * @desc プレイヤー移動時の足音を有効にします。
 *
 * @command DisableFootstep
 * @text 足音を無効化
 * @desc プレイヤー移動時の足音を無効にします。
 *
 * @help
 * ----------------------------------------------------------------------------
 * PlayerFootstep.js
 * ----------------------------------------------------------------------------
 * プレイヤーが移動する際に指定した足音の効果音を再生します。
 * 
 * 効果音のファイルは /audio/se フォルダに配置してください。
 * 
 * プラグインコマンドを使用して足音の再生をON/OFFに切り替えることができます。
 * 
 * プラグインコマンド:
 * - "足音を有効化" -> 足音の再生を有効にします。
 * - "足音を無効化" -> 足音の再生を無効にします。
 * ----------------------------------------------------------------------------
 * 利用規約:
 * 商用・非商用問わず利用可能です。
 * ----------------------------------------------------------------------------
 */

(() => {
    'use strict';

    const parameters = PluginManager.parameters('PlayerFootstep');
    const footstepSound = String(parameters['Footstep Sound'] || 'Footstep');
    const volume = Number(parameters['Volume'] || 90);
    const pitch = Number(parameters['Pitch'] || 100);
    const interval = Number(parameters['Interval'] || 30);

    let footstepEnabled = true;
    let footstepTimer = 0;

    PluginManager.registerCommand('PlayerFootstep', 'EnableFootstep', () => {
        footstepEnabled = true;
    });

    PluginManager.registerCommand('PlayerFootstep', 'DisableFootstep', () => {
        footstepEnabled = false;
    });

    let wasMoving = false;

    const _Game_Player_update = Game_Player.prototype.update;
    Game_Player.prototype.update = function (sceneActive) {
        _Game_Player_update.call(this, sceneActive);
        this.checkFootstep();
    };

    Game_Player.prototype.checkFootstep = function () {
        const isMoving = this.isMoving();
        if (footstepEnabled && isMoving) {
            footstepTimer--;
            if (footstepTimer <= 0) {
                AudioManager.playSe({
                    name: footstepSound,
                    volume: volume,
                    pitch: pitch
                });
                footstepTimer = interval;
            }
        } else if (footstepEnabled && !isMoving && wasMoving) {
            AudioManager.stopSe();
        }
        wasMoving = isMoving;
    };
})();
