//=============================================================================
// Plugin for RPG Maker MV
// BattleVoice.js
//=============================================================================
// [Update History]
// - BattleVoice.js
// 2015.Nov    Ver1.0.0 First Release
// 2016.Aug    Ver1.1.0 Strict Option Input
// 2019.Feb.27 Ver1.2.0 Random Play
// - BattleVoiceMZ
// 2020.Jan    Ver1.0.0 First release: Add plugin commands
// 2020.Oct.06 Ver1.1.0 Add situations: on counter attack and on reflect magic
// 2020.Nov.09 Ver1.2.0 Add situations: on evade attack and on battle starts
// 2021.Feb.21 Ver1.3.0 Add situations: on receive recover magic from ally
// - BattleVoice.js again
// 2021.Feb.21 Ver1.4.0 forcedly adapt MV version(closed version)
// 2021.Feb.24 Ver2.0.0 Ver.up based on MZ Ver1.3.0
// 2021.Mar.25 Ver2.1.0 enables to set delay when recovery received
// 2024.Jan.30 Ver2.1.1 Add situation: on evade magical attack.
// 2024.Jan.30 Ver2.2.0 It also be able to set voices to enemies' note.
// 2024.Oct.01 Ver2.3.0 Enables to select to do when plural actor bark at once
// 2024.Oct.05 Ver2.3.1 Fix bug: played at once when someone is dead
// 2024.Oct.13 Ver2.3.2 Fix bug: At battle starts, non-battler might bark.
// 2024.Dec.12 Ver2.4.0 Enables to set battle voice volume independent from SE
//                      Enables to assign voice when an actor uses an item

/*:
 * @target MV
 * @plugindesc [Ver2.4.0]Play voice SE at battle when actor does spcified action
 * @author Sasuke KANNAZUKI
 * 
 * @param ON switch ID
 * @desc play se only when the switch is ON.
 * This setting interlocks with option Battle Voice.
 * @type switch
 * @default 1
 * 
 * @param volume
 * @desc volume of SEs. this setting is common among all voice SEs.
 * (Default:90)
 * @type number
 * @min 0
 * @max 100000
 * @default 90
 * 
 * @param pitch
 * @desc pitch of SEs. this setting is common among all voice SEs.
 * (Default:100)
 * @type number
 * @min 10
 * @max 100000
 * @default 100
 *
 * @param pan
 * @desc pan of SEs. this setting is common among all actor's voice SEs.
 * 0:center, <0:left, >0:right (Default:100)
 * @type number
 * @min -100
 * @max 100
 * @default 100
 * 
 * @param enemyPan
 * @text pan of enemies
 * @desc pan of SEs. this setting is common among all enemy's voice SEs.
 * 0:center, <0:left, >0:right (Default:-100)
 * @type number
 * @min -100
 * @max 100
 * @default -100
 *
 * @param Battle Voice Name at Option
 * @desc display name at option
 * @type text
 * @default Battle Voice
 *
 * @param Does Change Volume
 * @desc Whether To Display Volume Change On Option Window
 * @type boolean
 * @default false
 *
 * @param Volume Variable ID
 * @parent Does Change Volume
 * @desc Variable ID That Interlocks With Volume Percentage. Set 100 When Starting Game.
 * @type variable
 * @default 1
 *
 * @param Volume Display Name
 * @parent Does Change Volume
 * @desc Display Name On Option Window
 * @type string
 * @default BattleVoice Volume
 * 
 * @param advanced
 * @text Advanced Setting
 * @desc No need to change unless it has some problems.
 * @type string
 * @default 
 *
 * @param waitForReceive
 * @parent advanced
 * @text Delay Frames Of Receiver
 * @desc Set the frames from recover invoke to receive
 * @type number
 * @default 30
 *
 * @param voiceAtAllDamage
 * @parent advanced
 * @text Multiple Damage Style
 * @desc Style when display multiple damages at once(by another plugin)
 * @type select
 * @option Not Change
 * @value normal
 * @option One of Them
 * @value alone
 * @option Make Wait
 * @value wait
 * @default normal
 *
 * @param waitForPlural
 * @parent voiceAtAllDamage
 * @text Plural Wait Count
 * @desc Frame count when you select "Make Wait" (1sec = 60frame)
 * @type number
 * @default 30
 *
 * @noteParam attackVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam recoverVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam friendMagicVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam magicVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam skillVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam damageVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam evadeVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam magicEvadeVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam defeatedVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam firstVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 * 
 * @noteParam victoryVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 * 
 * @noteParam counterVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam reflectVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam fromAllyVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @help
 * This plugin runs under RPG Maker MV.
 *
 * This plugin enables to play SE (assumed battle voice) at
 *  various situations.
 *
 * [Summary]
 * Player can change voice ON/OFF by Option Scene (except Title).
 * This setting interlocks switch ID set at plugin parameter.
 *
 * [note specification]
 * write down each actor's note at following format to set SE filename.
 * [[Voices when an actor perform something]]
 * <attackVoice:filename>  plays when actor does normal attack.
 * <recoverVoice:filename>   plays when actor uses HP recovering magic.
 * <friendMagicVoice:filename> plays when actor spells magic for friend
 *  except HP recovering. if this is not set but <magicVoice:filename> is set,
 *  it plays <magicVoice:filename> setting file.
 * <magicVoice:filename>   plays when actor spells magic(except for friend).
 * <skillVoice:filename>   plays when actor uses special skill except magic.
 *  *NOTE* Here 'magic' skill means its 'Skill Type' is included in 
 *   '[SV]Magic Skills' on 'System '.
 * [[Voices when an actor affected from any battler]]
 * <damageVoice:filename>    plays when actor takes damage.
 * <evadeVoice:filename>    plays when actor evades enemy physical attack.
 * <magicEvadeVoice:filename> plays when actor evades enemy magical attack.
 *    'magical attack' means the Hit Type of the skill is "Magical Attack".
 * <defeatedVoice:filename>   plays when actor is died.
 * <counterVoice:filename>   plays when counter attack invokes.
 * <reflectVoice:filename>   plays when actor reflects magic.
 * <fromAllyVoice:filename> plays when actor received HP recover magic.
 *   It doesn't play when magic user is the same as receiver.
 *   It assumes the phrase 'Thank you' and so on.
 * [[Voices when battle exceeds]]
 * if plural actors attend the battle, randomly selected actor's SE is adopted.  * <firstVoice:filename>   plays when battle starts except surprised.
 * <victoryVoice:filename>   plays when battle finishes.
 *
 * [Advanced option 1 : Set plural voices]
 * If you want to play one of several voices randomly,
 * write filenames with colon as follows:
 * <attackVoice:atk1,atk2,atk3>
 * in this case, at attack, plays atk1 atk2, or atk3 randomly.
 *
 * If set no SE one of filenames, 
 * <attackVoice:atk1,atk2,$>
 * in this case, at attack, plays atk1 atk2, or doesn't play SE.
 *
 * You can set the same filename twice or more than.
 * <attackVoice:atk1,atk2,atk2,$>
 * in this case, 25% atk1, 50% atk2, 25% don't play.
 *
 * *NOTE* When set SEs at this notation, these files might be excluded at
 *  deployment with option 'Exclude unused files'.
 *  To prevent this, I recommend to make dummy event and set each SE to
 *  'Play SE' on the Contents.
 *
 * [Advanced Option 2: Set enemy's voices] (Since Ver2.2.0)
 * The voice settings are also valid to enemies' note.
 * But 'firstVoice' and 'victoryVoice' are invalid (they are simply ignored).
 *
 * [Advanced Option 3: Set behavior when plural actors bark at once]
 * (Since Ver2.3.0)
 * If you introduce any plugin that enables to display plural damage popup
 * at once, you can select how to do at option.
 * - normal - default(play all battlers' voice at once)
 * - alone - play voice of one of the target
 * - wait - forcedly make wait specified frame to play each voice.
 *
 * [Plugin Commands]
 * 
 * **Set voice on each situation**
 * BattleVoice set arg1 arg2 arg3
 *  - arg1 must be actor id.
 *  - arg2 must be a situation.
 *   attack , recover, friendMagic, magic, skill, damage, evaded, magicEvaded,
 *   dead, counter, reflect, fromAlly, first or victory
 *  -arg3 must be voice file name. the same as note, by split comma,
 *   plural setting enables.
 * ex.
 * BattleVoice 1 attack attackVoice
 *  set actor whose id is 1 voice for attack.
 *
 * **Reset voice on each situation**
 * BattleVoice reset arg1 arg2
 *  - arg1 must be actor id.
 *  - arg2 must be a situation.
 * ex.
 * BattleVoice reset 2 attack
 *  reset actor whose id is 2 voice for attack.
 *
 * **Reset all situations' voice to default**
 * BattleVoice allReset arg1
 *  - arg1 must be actor id to reset all situation
 *   (note: voice for actor's each skill is not reset)
 *
 * **Assign actor voice to each skill**
 * BattleVoice skillSet arg1 arg2 arg3
 *  - arg1 must be actor id
 *  - arg2 must be skill id to set original voice
 *  - arg3 must be voice file name. the same as note, by split comma,
 *   plural setting enables.
 * ex.
 * BattleVoice skillSet 3 8 fire1
 *  sets fire1 when an actor whose id is 3 uses skill whose id is 8
 *
 * **Reset actor voice to each skill**
 * BattleVoice skillReset arg1 arg2
 *  - arg1 must be actor id
 *  - arg2 must be skill id to reset original voice
 * ex.
 * BattleVoice skillReset 2 15
 *  reset voice for an actor whose id is 2's skill whose id is 15.
 *
 * **Reset all voices assigned to skills**
 * BattleVoice skillAllReset arg1
 *  - arg1 must be actor id to reset all skill's voice.
 *  (note: voice for situations are not reset)
 *

 * **Assign actor voice to each item** (since Ver2.4.0)
 * BattleVoice itemSet arg1 arg2 arg3
 *  - arg1 must be actor id
 *  - arg2 must be item id to set special voice
 *  - arg3 must be voice file name. the same as note, by split comma,
 *   plural setting enables.
 * ex.
 * BattleVoice itemSet 3 8 fire1
 *  sets fire1 when an actor whose id is 3 uses item whose id is 8
 *
 * **Reset actor voice to each item** (since Ver2.4.0)
 * BattleVoice skillReset arg1 arg2
 *  - arg1 must be actor id
 *  - arg2 must be item id to reset special voice
 * ex.
 * BattleVoice skillReset 2 15
 *  reset voice for an actor whose id is 2's item whose id is 15.
 *
 * **Reset all voices assigned to items** (since Ver2.4.0)
 * BattleVoice skillAllReset arg1
 *  - arg1 must be actor id to reset all item's voice.
 *
 * [License]
 * this plugin is released under MIT license.
 * http://opensource.org/licenses/mit-license.php
 */
/*:ja
 * @target MV
 * @plugindesc [Ver2.4.0]アクターの戦闘時の行動にボイスSEを設定します。
 * @author 神無月サスケ
 * 
 * @param ON switch ID
 * @text ボイス演奏スイッチID
 * @desc このスイッチが ON の時のみ、ボイスSEを演奏します。
 * オプション「バトルボイス」と連動します。
 * @type switch
 * @default 1
 *
 * @param volume
 * @text 共通ボリューム
 * @desc ボイスSEのボリュームです。この設定が全てのボイスSEの
 * 共通となります。(既定値:90)
 * @type number
 * @min 0
 * @max 100000
 * @default 90
 *
 * @param pitch
 * @text 共通ピッチ
 * @desc ボイスSEのピッチです。この設定が全てのボイスSEの
 * 共通となります。(既定値:100)
 * @type number
 * @min 10
 * @max 100000
 * @default 100
 *
 * @param pan
 * @text 共通位相
 * @desc 全アクター共通のボイスSEの位相
 * 0:中央, 負数:左寄り, 正数:右寄り (既定値:100)
 * @type number
 * @min -100
 * @max 100
 * @default 100
 *
 * @param enemyPan
 * @text 敵キャラ共通位相
 * @desc 全敵キャラ共通のボイスSEの位相
 * 0:中央, 負数:左寄り, 正数:右寄り (既定値:-100)
 * @type number
 * @min -100
 * @max 100
 * @default -100
 *
 * @param Battle Voice Name at Option
 * @text バトルボイス表示名
 * @desc オプション画面での表示名です。
 * @type string
 * @default バトルボイス
 *
 * @param Does Change Volume
 * @text ボリューム調整機能使用？
 * @desc バトルボイスの音量をオプションで変更可能にするか
 * @type boolean
 * @default false
 *
 * @param Volume Variable ID
 * @parent Does Change Volume
 * @text ボリューム変数ID
 * @desc ボリューム％と連動する変数です。ゲーム開始時に100に初期化されます
 * @type variable
 * @default 1
 *
 * @param Volume Display Name
 * @parent Does Change Volume
 * @text ボリューム表示名
 * @desc オプション画面での表示名
 * @type string
 * @default バトルボイス 音量
 *
 * @param advanced
 * @text 高度な設定
 * @desc 通常は変更の必要はありません。
 * @type string
 * @default 
 *
 * @param waitForReceive
 * @parent advanced
 * @text 被回復時ウェイト
 * @desc 回復魔法が使われてから受け取ったアクターが発声するまでのフレーム数
 * @type number
 * @default 30
 *
 * @param voiceAtAllDamage
 * @parent advanced
 * @text 複数人ダメージ表示スタイル
 * @desc 別プラグインの導入で、複数人のダメージを一斉に再生する際の処理
 * @type select
 * @option 通常再生
 * @value normal
 * @option 一人のボイスのみ
 * @value alone
 * @option ウェイト発生
 * @value wait
 * @default normal
 *
 * @param waitForPlural
 * @parent voiceAtAllDamage
 * @text ウェイト数
 * @desc ウェイト発生の際のフレーム数(60フレーム＝1秒)
 * @type number
 * @default 30
 *
 * @noteParam attackVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam recoverVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam friendMagicVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam magicVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam skillVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam damageVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam evadeVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam magicEvadeVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam defeatedVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam firstVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 * 
 * @noteParam victoryVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 * 
 * @noteParam counterVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam reflectVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 * @noteParam fromAllyVoice
 * @noteRequire 1
 * @noteDir audio/se/
 * @noteType file
 * @noteData actors
 *
 @help
 * このプラグインは、RPGツクールMVに対応しています。
 * 
 * 戦闘中のシチュエーションに応じてにバトルボイスを演奏可能にします。
 *
 * ■概要
 * ゲーム中のオプション画面(タイトル画面以外)でON/OFFが可能です。
 * この設定は、このプラグインのパラメータで指定したスイッチと連動しています。
 * デフォルトではOFFになっています。
 *
 * ■メモ設定方法
 * それぞれのアクターのメモに以下の書式で書いてください。
 * filename はボイスSEのファイル名にしてください。
 * ◆アクター行動時
 * <attackVoice:filename>  通常攻撃の時に再生されるボイスです。
 * <recoverVoice:filename>   HP回復魔法を使用した時に再生されるボイスです。
 * <friendMagicVoice:filename>   HP回復以外の味方向け魔法を使用した時に
 *  再生されるボイスです。省略された場合で<magicVoice:filename>が
 *  設定されている場合は、そちらが再生されます。
 * <magicVoice:filename> 味方向け以外の魔法を使用した時に再生されるボイスです。
 * <skillVoice:filename>   必殺技を使用した時に再生されるボイスです。
 * 　注意：ここでいう「魔法」の定義は、そのスキルのスキルタイプが、
 * 「システム2」タブの「[SV]魔法スキル」に含まれているものです。
 * ◆アクターが対象になった時
 * <damageVoice:filename>    ダメージを受けた時に再生されるボイスです。
 * <evadeVoice:filename>    物理攻撃を回避した時に再生されるボイスです。
 * <magicEvadeVoice:filename> 魔法攻撃を回避した時に再生されるボイスです。
 *   「魔法攻撃」の定義は、スキルの「命中タイプ」が「魔法攻撃」のものです。
 * <defeatedVoice:filename>   戦闘不能になった時に再生されるボイスです。
 * <counterVoice:filename>   カウンター攻撃発動時に再生されるボイスです。
 * <reflectVoice:filename>   魔法を反射する時に再生されるボイスです。
 * <fromAllyVoice:filename> HP回復魔法を受けた時に再生されるボイスです。
 *  自分自身に使った場合は再生されません
 *  「ありがとう」など感謝の言葉を想定しています。
 * ◆バトルの進捗に応じて
 * アクターが複数いる場合、生きているアクターの中からランダムで再生されます。
 * <firstVoice:filename>    戦闘開始時に再生されるボイスです。
 *  ただし、不意打ちの際は再生されません。
 * <victoryVoice:filename>   戦闘勝利時に再生されるボイスです。
 *
 * ■拡張機能１：ボイスの複数指定
 * 上記のメモのfilename を、コロンで複数指定すると、その中からランダムで
 * 再生されます。例えば、以下のように指定した場合、
 * <attackVoice:atk1,atk2,atk3>
 * atk1 atk2 atk3 のいずれかのボイスがランダムで再生されます。
 *
 * 無音を指定したい場合は、$ を入れてください。
 * <attackVoice:atk1,atk2,$>
 * この場合、atk1, atk2, 無音の中から選ばれます。
 * 
 * 同じファイル名を複数回指定可能です。
 * <attackVoice:atk1,atk2,atk2,$>
 * この場合、25%でatk1、50%でatk2、25%で演奏なしになります。
 *
 * 注意：この形式で設定を行った場合、デプロイメントの「不要ファイルの削除」で
 *  削除される可能性があります。例えばダミーイベントを作り、これらのSEを
 *  演奏するなどして、適宜対処してください。
 *
 * ■拡張機能２ 敵キャラでも設定可(Ver2.2.0～)
 * 上記のタグを敵キャラのメモに設定することも可能です。
 * ただし、「バトルの進捗に応じて」(戦闘開始時、勝利時)の設定は無効です。
 *
 * ■拡張機能３ 複数人が同時に発声する際の処理を選択可能 (Ver2.3.0～)
 * 他のプラグインで、複数の仲間が同じタイミングでダメージを受けた場合、
 * 通常では全員のボイスが同時に再生されていました。
 * この際の挙動を、プラグインパラメータにて設定可能です。
 * - 通常 - これまで通り
 * - 一人のボイスのみ再生 - 代表の一人のボイスを再生します
 * - 再生ウェイト発生 - 指定されたフレームの間、各アクター毎にウェイトを
 *   発生させてボイスを再生します。
 *
 * ■プラグインコマンド
 * ◆各シチュエーションでのボイスの変更
 * BattleVoice set arg1 arg2 arg3
 *  - arg1 は対象のアクターID
 *  - arg2 はシチュエーション文字列。以下から選んでください
 *   attack (通常攻撃時), recover (回復魔法使用時),
 *   friendMagic (味方対象魔法使用時) magic (通常魔法使用時)
 *   skill (非魔法スキル使用時), damage (被ダメージ時),
 *   evaded (物理攻撃回避時), magicEvaded(魔法攻撃回避時)
 *   dead (戦闘不能時), counter (カウンター発動時), reflect (魔法反射発動時),
 *   fromAlly (回復魔法を受けた時),
 *   first (戦闘開始時) or victory (戦闘勝利時).
 *  -arg3 はボイスSE名。セミコロンで複数指定可能
 * 例.
 * BattleVoice 1 attack attackVoice
 *  アクターID1の攻撃時(attack)にattackVoiceというSEを再生します。
 *
 * ◆各シチュエーションでのボイスのリセット
 * BattleVoice reset arg1 arg2
 *  - arg1 は対象のアクターID
 *  - arg2 はシチュエーション文字列。上記を参考に。
 * 例.
 * BattleVoice reset 2 attack
 *  アクターID2の攻撃時ボイスをリセットしメモ設定に戻します。
 *
 * ◆全シチュエーションのボイスの一括リセット
  * BattleVoice allReset arg1
 *   - arg1 は対象のアクターID
 *   (注: スキル毎に設定されたボイスはリセットされません)
 *
 * ◆スキル番号にボイスを割り当てる
 * BattleVoice skillSet arg1 arg2 arg3
 *  - arg1 は対象のアクターID
 *  - arg2 はスキルID
 *  - arg3 は演奏するSE名
 * 例.
 * BattleVoice skillSet 3 8 fire1
 *   アクターID3がスキルID8を使用した時にfire1を演奏します
 *
 * ◆スキル番号に割り当てたボイスを解除
 * BattleVoice skillReset arg1 arg2
 *  - arg1 は対象のアクターID
 *  - arg2 はスキルID
 * 例.
 * BattleVoice skillReset 2 15
 *  アクターID2がスキルID15を使用した時のボイスをなくします。
 *
 * ◆スキル番号に割り当てられた全ボイスを一括リセット
 * BattleVoice skillAllReset arg1
 *  - arg1 は対象のアクターID
 *  (注意: シチュエーションに割り当てられたボイスはリセットされません。)
 *
 * ◆アイテム番号にボイスを割り当てる(Ver2.4.0～)
 * BattleVoice itemSet arg1 arg2 arg3
 *  - arg1 は対象のアクターID
 *  - arg2 はアイテムID
 *  - arg3 は演奏するSE名
 * 例.
 * BattleVoice itemSet 3 8 fire1
 *   アクターID3がID8のアイテムを使用した時にfire1を演奏します
 *
 * ◆アイテム番号に割り当てたボイスを解除(Ver2.4.0～)
 * BattleVoice itemReset arg1 arg2
 *  - arg1 は対象のアクターID
 *  - arg2 はアイテムID
 * 例.
 * BattleVoice itemReset 2 15
 *  アクターID2がID15のアイテムを使用した時のボイスをなくします。
 *
 * ◆アイテム番号に割り当てられた全ボイスを一括リセット(Ver2.4.0～)
 * BattleVoice itemAllReset arg1
 *  - arg1 は対象のアクターID
 *
 * ■ライセンス表記
 * このプラグインは MIT ライセンスで配布されます。
 * ご自由にお使いください。
 * http://opensource.org/licenses/mit-license.php
 */
(function() {
  var pluginName = 'BattleVoice';

  //
  // process parameters
  //
  var parameters = PluginManager.parameters(pluginName);
  var pitch = Number(parameters['pitch'] || 100);
  var volume = Number(parameters['volume'] || 90);
  var pan = Number(parameters['pan'] || 0);
  var enemyPan = Number(parameters['enemyPan'] || 0);
  var playSwitchId = Number(parameters['ON switch ID'] || 1);
  var strBattleVoice = parameters['Battle Voice Name at Option'] ||
    'Battle Voice';
  var waitForReceive = Number(parameters['waitForReceive'] || 30);
  var voiceAtAllDamage = parameters['voiceAtAllDamage'] || "normal";
  var waitForPlural = Number(parameters['waitForPlural'] || 30);

  var doesChangeVolume = eval(parameters['Does Change Volume'] || "false");
  var volumeVariableId = Number(parameters['Volume Variable ID'] || 0);
  var volumeDisplayName = parameters['Volume Display Name'] || "Voice Volume";

  //
  // process plugin commands
  //
  var _Game_Interpreter_pluginCommand =
   Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === pluginName) {
      var actor = $gameActors.actor(+args[1]);
      if (!actor) {
        return;
      }
      actor.battleVoices = actor.battleVoices || {};
      actor.skillVoices = actor.skillVoices || {};
      actor.itemVoices = actor.itemVoices || {};
      switch (args[0]) {
      case 'set':
        actor.battleVoices[args[2]] = args[3];
        break;
      case 'reset':
        actor.battleVoices[args[2]] = null;
        break;
      case 'allReset':
        actor.battleVoices = {};
        break;
      case 'skillSet':
        actor.skillVoices[+args[2]] = args[3];
        break;
      case 'skillReset':
        actor.skillVoices[+args[2]] = null;
        break;
      case 'skillAllReset':
        actor.skillVoices = {};
        break;
      case 'itemSet':
        actor.itemVoices[+args[2]] = args[3];
        break;
      case 'itemReset':
        actor.itemVoices[+args[2]] = null;
        break;
      case 'itemAllReset':
        actor.skillVoices = {};
        break;
      }
    }
  };

  //
  // set play options (interlock with switch)
  //
  var doesDisplaySpecialOptions = function() {
    return !SceneManager.isPreviousScene(Scene_Title);
  };

  var _DataManager_setupNewGame = DataManager.setupNewGame;
  DataManager.setupNewGame = function() {
    _DataManager_setupNewGame.call(this);
    $gameSwitches.setValue(playSwitchId, true);
    if (doesChangeVolume) {
      $gameVariables.setValue(volumeVariableId, 100);
    }
  };

  var _Window_Options_makeCommandList =
   Window_Options.prototype.makeCommandList;
  Window_Options.prototype.makeCommandList = function() {
    if (doesDisplaySpecialOptions()) {
      this.addCommand(strBattleVoice, 'battleVoice');
    }
    _Window_Options_makeCommandList.call(this);
    if (doesChangeVolume && doesDisplaySpecialOptions()) {
      this.addCommand(volumeDisplayName, 'battleVoiceVolume');
    }
  };

  var _Window_Options_getConfigValue =
   Window_Options.prototype.getConfigValue;
  Window_Options.prototype.getConfigValue = function(symbol) { 
    switch (symbol) {
    case 'battleVoice':
      return $gameSwitches.value(playSwitchId);
    case 'battleVoiceVolume':
      return $gameVariables.value(volumeVariableId);
    default:
      return _Window_Options_getConfigValue.call(this, symbol);
    }
  };

  var _Window_Options_setConfigValue =
   Window_Options.prototype.setConfigValue;
  Window_Options.prototype.setConfigValue = function(symbol, volume) {
    switch (symbol) {
    case 'battleVoice':
      $gameSwitches.setValue(playSwitchId, volume);
    case 'battleVoiceVolume':
      $gameVariables.setValue(volumeVariableId, volume);
    default:
      _Window_Options_setConfigValue.call(this, symbol, volume);
    }
  };

  var _Scene_Options_maxCommands = Scene_Options.prototype.maxCommands;
  Scene_Options.prototype.maxCommands = function() {
    var rowNum = _Scene_Options_maxCommands.call(this);
    if (doesDisplaySpecialOptions()) {
      rowNum++;
      if (doesChangeVolume) {
        rowNum++;
      }
    }
    return rowNum;
  };

  //
  // play battle voice at specified volume
  //
  AudioManager.playVoice = function(voice) {
    if (!doesChangeVolume) {
      this.playSe(voice);
      return;
    }
    if (voice.name) {
      // since battle voices are SE, set it at SE's buffer.
      this._seBuffers = this._seBuffers.filter(function(audio) {
        return audio.isPlaying();
      });
      var buffer = this.createBuffer('se', voice.name);
      this.updateVoiceParameters(buffer, voice);
      buffer.play(false);
      this._seBuffers.push(buffer);
    }
  };

  AudioManager.updateVoiceParameters = function(buffer, voice) {
    var volume = $gameVariables.value(volumeVariableId);
    this.updateBufferParameters(buffer, volume, voice);
  };

  //
  // play battler voice
  //
  var canPlayBattlerVoice = function() {
    return $gameSwitches.value(playSwitchId);
  };

  var split = function(name) {
    if (!name) {
      return name;
    }
    var names = name.split(',');
    return names[Math.randomInt(names.length)];
  };

  var createAudioByFileName = function(name, isActor) {
    var audio = {};
    audio.name = name;
    audio.pitch = pitch;
    audio.volume = volume;
    audio.pan = isActor ? pan : enemyPan;
    return audio;
  };

  var playBattlerVoice = function(battler, type) {
    if (!canPlayBattlerVoice() || $gameTemp.voicePlayed) {
      return;
    }
    var name = '';
    var a = battler.battleVoices || {};
    var m = battler.isActor() ? battler.actor().meta : battler.enemy().meta;
    switch(type){
      case 'attack':
        name = split(a.attack || m.attackVoice);
        break;
      case 'recover':
        name = split(a.recover || m.recoverVoice);
        break;
      case 'friendmagic':
        name = split(a.friendMagic || m.friendMagicVoice || m.magicVoice);
        break;
      case 'magic':
        name = split(a.magic || m.magicVoice);
        break;
      case 'skill':
        name = split(a.skill || m.skillVoice);
        break;
      case 'damage':
        name = split(a.damage || m.damageVoice);
        waitForVoice(name);
        break;
      case 'evade':
        name = split(a.evaded || m.evadeVoice);
        waitForVoice(name);
        break;
      case 'magicEvade':
        name = split(a.magicEvaded || m.magicEvadeVoice);
        waitForVoice(name);
        break;
      case 'dead':
        name = split(a.dead || m.defeatedVoice);
        waitForVoice(name);
        break;
      case 'counter':
        name = split(a.counter || m.counterVoice);
        waitForVoice(name);
        break;
      case 'reflect':
        name = split(a.reflect || m.reflectVoice);
        waitForVoice(name);
        break;
      case 'fromAlly':
        name = split(a.fromAlly || m.fromAllyVoice);
        break;
      case 'first':
        name = split(a.first || m.firstVoice);
        break;
      case 'victory':
        name = split(a.victory || m.victoryVoice);
        break;
    }
    if (name && name !=="$") {
      var audio = createAudioByFileName(name, battler.isActor());
      AudioManager.playVoice(audio);
    }
  };

  //
  // play voice when an actor uses skill
  //
  var isSkillVoice = function(battler, action) {
    if (!battler.skillVoices || !action.isSkill()) {
      return false;
    }
    return !!battler.skillVoices[action._item.itemId()];
  };

  var playSkillVoice = function(actor, action) {
    if (!canPlayBattlerVoice()) {
      return;
    }
    var name = split(actor.skillVoices[action._item.itemId()]);
    if (name && name !=="$") {
      var audio = createAudioByFileName(name, true);
      AudioManager.playVoice(audio);
    }
  };

  //
  // play voice when an actor uses item
  //
  var isItemVoice = function(battler, action) {
    if (!battler.itemVoices || !action.isItem()) {
      return false;
    }
    return !!battler.itemVoices[action._item.itemId()];
  };

  var playItemVoice = function(actor, action) {
    if (!canPlayBattlerVoice()) {
      return;
    }
    var name = split(actor.itemVoices[action._item.itemId()]);
    if (name && name !=="$") {
      var audio = createAudioByFileName(name, true);
      AudioManager.playVoice(audio);
    }
  };

  //
  // process wait (if it displays all popup damages together)
  //
  var waitForVoice = function(name) {
    if (name && name !=="$") {
      switch (voiceAtAllDamage) {
      case 'normal':
        break;
      case 'alone':
        $gameTemp.voicePlayed = true;
        break;
      case 'wait':
        BattleManager._logWindow._waitCount = waitForPlural;
      }
    }
  };

  var shuffleTargetsIfNeeds = function() {
    if (voiceAtAllDamage === 'alone') {
      var targets = BattleManager._targets;
      // Change the top of BattleManager._targets
      var index = Math.randomInt(targets.length);
      var topElementArray = targets.splice(index, 1);
      targets.unshift(topElementArray[0]);
    }
  };

  var _BattleManager_startAction = BattleManager.startAction;
  BattleManager.startAction = function() {
    $gameTemp.voicePlayed = false;
    _BattleManager_startAction.call(this);
    shuffleTargetsIfNeeds();
  };

  var _BattleManager_endAction = BattleManager.endAction;
  BattleManager.endAction = function() {
    $gameTemp.voicePlayed = false;
    _BattleManager_endAction.call(this);
  };

  //
  // functions for call actor voice.
  //
  var _Game_Battler_performAction = Game_Battler.prototype.performAction;
  Game_Battler.prototype.performAction = function(action) {
    _Game_Battler_performAction.call(this, action);
    if (isSkillVoice(this, action)) {
      playSkillVoice(this, action);
    } else if (isItemVoice(this, action)) {
       playItemVoice(this, action);
    } else if (action.isAttack()) {
      playBattlerVoice(this, 'attack');
    } else if (action.isMagicSkill() && action.isHpRecover()) {
      playBattlerVoice(this, 'recover');
    } else if (action.isMagicSkill() && action.isForFriend()) {
      playBattlerVoice(this, 'friendmagic');
    } else if (action.isMagicSkill()) {
      playBattlerVoice(this, 'magic');
    } else if (action.isSkill() && !action.isGuard()) {
      playBattlerVoice(this, 'skill');
    }
  };

  var _Game_Battler_performDamage = Game_Battler.prototype.performDamage;
  Game_Battler.prototype.performDamage = function() {
    _Game_Battler_performDamage.call(this);
    // When selected actor is dead, play dead voice instead of damage voice.
    if (voiceAtAllDamage !== "alone" || this.isAlive()) {
      playBattlerVoice(this, 'damage');
    }
  };

  var _Game_Battler_performEvasion = Game_Battler.prototype.performEvasion;
  Game_Battler.prototype.performEvasion = function() {
    _Game_Battler_performEvasion.call(this);
    playBattlerVoice(this, 'evade');
  };

  var _Game_Battler_performMagicEvasion =
   Game_Battler.prototype.performMagicEvasion;
  Game_Battler.prototype.performMagicEvasion = function() {
    _Game_Battler_performMagicEvasion.call(this);
    playBattlerVoice(this, 'magicEvade');
  };

  var _Game_Battler_performCollapse = Game_Battler.prototype.performCollapse;
  Game_Battler.prototype.performCollapse = function() {
    _Game_Battler_performCollapse.call(this);
    if ($gameParty.inBattle()) {
      playBattlerVoice(this, 'dead');
    }
  };

  var _BattleManager_invokeCounterAttack = BattleManager.invokeCounterAttack;
  BattleManager.invokeCounterAttack = function(subject, target) {1
    playBattlerVoice(target, 'counter');
    _BattleManager_invokeCounterAttack.call(this, subject, target);
  };

  var _BattleManager_invokeMagicReflection =
    BattleManager.invokeMagicReflection;
  BattleManager.invokeMagicReflection = function(subject, target) {
    playBattlerVoice(target, 'reflect');
    _BattleManager_invokeMagicReflection.call(this, subject, target);
  };

  Game_Party.prototype.aliveBattleMembers = function() {
    return this.battleMembers().filter(member => member.isAlive());
  };

  var _Game_System_onBattleStart = Game_System.prototype.onBattleStart;
  Game_System.prototype.onBattleStart = function() {
    _Game_System_onBattleStart.call(this);
    var candidates = $gameParty.aliveBattleMembers().filter(function(actor) {
       return actor.actor().meta.firstVoice || 
         (actor.battleVoices && actor.battleVoices.first);
    });
    if (candidates.length > 0) {
      var index = Math.randomInt(candidates.length);
      var actor = candidates[index];
      if (!BattleManager._surprise) {
        playBattlerVoice(actor, 'first');
      }
    }
  };

  var _BattleManager_processVictory = BattleManager.processVictory;
  BattleManager.processVictory = function() {
    var candidates = $gameParty.aliveMembers().filter(function(actor) {
      return actor.actor().meta.victoryVoice ||
        (actor.battleVoices && actor.battleVoices.victory);
    });
    if (candidates.length > 0) {
      var index = Math.randomInt(candidates.length);
      var actor = candidates[index];
      playBattlerVoice(actor, 'victory');
    }
    _BattleManager_processVictory.call(this);
  };

  //
  // When recover received, wait some frames until play.
  //
  Game_Battler.prototype.doesPlayFromAlly = function() {
    // not play voice if target is the same as magic user
    return this !== BattleManager._subject;
  };

  Window_BattleLog.prototype.waitAlly = function() {
    this._waitCount = waitForReceive;
  };

  Window_BattleLog.prototype.playReceiveVoice = function(target) {
    playBattlerVoice(target, 'fromAlly');
  };

  var _Window_BattleLog_performRecovery =
    Window_BattleLog.prototype.performRecovery;
  Window_BattleLog.prototype.performRecovery = function(target) {
    if (target.doesPlayFromAlly()) {
      this.push('waitAlly');
      this.push('playReceiveVoice', target);
    }
    _Window_BattleLog_performRecovery.call(this, target);
  };

})();
