 //=============================================================================
// WindowSpace.js
//=============================================================================
 /*:
 * @plugindesc メッセージウィンドウ内の文字開始位置をずらします
 * ※MPP_MessageEXがないと正しく機能しません。
 * @author TRS
 * @help このスクリプトとMPP_MessageEx、MPP_Message_op1等のウィンドウ関連スクリプトを兼ねる場合は、
 * これらのウィンドウ関連スクリプトの上に配置してください
 *
 * 
 * @param -------設定-----
 * @desc セパレータ
 * 
 * @param 有効無効SW
 * @desc メッセージウィンドウ内の文字移動を有効・無効するスイッチ
 * @type switch
 * @default 1
 *
 * @param ずらすドット数
 * @desc 文字開始位置をずらすドット数
 * @type number
 * @max 9999
 * @default 0
 *
 * @param 縁変更有効無効SW
 * @desc アウトライン変更有効・無効スイッチ
 * @type switch
 * @default 0
 *
 * @param 変更後の縁の大きさ
 * @desc 文字のアウトラインの大きさ
 * 初期値:4
 * @type number
 * @default 4
 *
 *
 *@help 
 * ◆プラグインコマンド 
 *  ・ウィンドウ設定 [X座標][Y座標][幅][列数]
 *       [X座標]……メッセージウィンドウのX座標(pix)
 *       [Y座標]……メッセージウィンドウのY座標(pix)
 *       [幅]   ……メッセージウィンドウの幅(pix)
 *       [列数]   ……メッセージウィンドウの列
 
 *   →ウィンドウの表示の指定です。ウィンドウ設定解除で解除するまでその指定位置に表示されます。
 *    
 *  ・ウィンドウ設定解除
 *   →ウィンドウの表示の指定を解除し、デフォルトの位置にウィンドウが出ます
 *
 *  ・フォント変更 [フォント名]
 *       [フォント名]……fontsフォルダにあるフォントファイル名（拡張子(.ttf 他)まで必要）
 
 *   →ウィンドウに表示されるフォントを指定のフォントに変えます。
 *   ※デフォに戻すには、元のフォントファイル名を指定してください
 *   ※フォントの切り替えにはツクールの都合上、2フレーム以上かかるため、
 *     このコマンド実行時、10フレームのウェイトが入ります。
 *   
 *   ・フォント色変更 [フォント色]
 *     [フォント色]……RGB 16進数で設定してください R 0～FF G 0～FF B 0～FF
 *                   （#が内部で付与されます）
 *
 *     ※文章中では\c[n]の機能も使えます。\c[0]を設定された場合は従来通りに、文字色が白になります。
 *
 *     設定例：フォント色変更 FFFF00
 *            ↑このプラグインコマンドを設定しますと、以降のウィンドウメッセージの文字色が黄色になります
 *
 *   ・フォント色初期化
 *     →フォント色を初期値に戻します
 *
 *   ・指定範囲の文字列のみ色を変える(工事中)
 *     \cex[FF0000]文字列\cex[FFFFFF]	//文字列のRGB指定
 */
(function() 
{
	//--捕まえる関連処理
 	var parameters = PluginManager.parameters('WindowSpace');
 	var params = [];
 	
 	params.switch = Number ( parameters['有効無効SW'] || 1 );
	params.shiftPix = Number ( parameters['ずらすドット数'] || 0 );
	
	params.outLineSwitch = Number( parameters['縁変更有効無効SW'] || 0 )
	params.outLineSize = Number( parameters['変更後の縁の大きさ'] || 4 );
	
	var windowInfo = {};				//ウィンドウインフォ
	
	windowInfo.xPos = 0;
	windowInfo.yPos = 0;
	windowInfo.rows = 4;
	windowInfo.width = 0;
	windowInfo.orgPos = false;
	windowInfo.color = '#FFFFFF'
	windowInfo.defaultColor = '#FFFFFF'
	
	//-----------------------------------------------------------------------------
	// Game_System
	//
	// The game object class for the system 
	
	var _game_system_prototype_initialize = Game_System.prototype.initialize;
	Game_System.prototype.initialize = function() {
	    _game_system_prototype_initialize.call(this);
	    
	    this._windowSpaceInfo = null;
	    
	};
	
	//-----------------------------------------------------------------------------
	// Window_Message
	//
	// The window for displaying text messages.
	
	//var _window_Message_prototype_newPage = Window_Message.prototype.newPage;
	
	Window_Message.prototype.initialize = function() 
	{
	    var width = this.windowWidth();
	    var height = this.windowHeight();
	    var x = (Graphics.boxWidth - width) / 2;
	    Window_Base.prototype.initialize.call(this, x, 0, width, height);
	    this.openness = 0;
	    this.initMembers();
	    this.createSubWindows();
	    this.updatePlacement();
	    windowInfo.defaultColor = this.normalColor();
	    
	};

	Window_Message.prototype.newPage = function(textState)
	{
		//console.log($gameSystem._windowSpaceInfo)
	    if($gameSystem._windowSpaceInfo !== null ){
	    	windowInfo = $gameSystem._windowSpaceInfo;
	    }
		
		var shiftPix = 0;
		if( $gameSwitches.value( params.switch ) == true)
		{
			shiftPix = params.shiftPix;
		}
		
	    this.contents.clear();
	    this.resetFontSettings();
	    this.clearFlags();
	    this.loadMessageFace();
	    textState.x = this.newLineX() + shiftPix; 
	    textState.y = 0;
	    textState.left = this.newLineX() + shiftPix;
	    textState.height = this.calcTextHeight(textState, false);
	};
	
	Window_Message.prototype.windowHeight = function() 
	{
		if( windowInfo.orgPos === false )
    		return this.fittingHeight( this.numVisibleRows() );
    	else
    		return this.fittingHeight( windowInfo.rows );
	};
	
	var _window_message_prototype_updatePlacement = Window_Message.prototype.updatePlacement;
	
	Window_Message.prototype.updatePlacement = function() 
	{
		this.height = this.windowHeight();
		if( windowInfo.orgPos === false )
		{
			_window_message_prototype_updatePlacement.call(this);
			this.width = this.windowWidth();
			this.x = ( Graphics.boxWidth -  this.windowWidth() ) / 2;
			
		}
		
		//このモードは数値を直接指定
		else
		{
			this.x = windowInfo.xPos;
			this.y = windowInfo.yPos;
			this.width = windowInfo.width;
			
			this._goldWindow.y = this.y > 0 ? 0 : Graphics.boxHeight - this._goldWindow.height;
		}
		
	};
	
	/**
	 * アウトラインの変更
	 */
	 var _window_Message_prototype_changeTextColor = Window_Message.prototype.changeTextColor;
	Window_Message.prototype.changeTextColor = function(color) 
	{
		color = windowInfo.color;
		_window_Message_prototype_changeTextColor.call(this, color);
    	if( $gameSwitches.value(params.outLineSwitch) )
    		this.contents.outlineWidth = params.outLineSize;
    		
    	else
    		this.contents.outlineWidth = 4;
	};
	
	/**
	 * インタプリタ
	 */
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args)
	{
		_Game_Interpreter_pluginCommand.call(this, command, args);
		if(command === "ウィンドウ設定")
		{
			windowInfo.xPos = Number(args[0]);
			windowInfo.yPos = Number(args[1]);
			windowInfo.width = Number(args[2]);
			windowInfo.rows = Number(args[3]);
			windowInfo.orgPos = true;
			
			//args[0]……X座標
			//args[1]……Y座標
			//args[2]……幅
			//args[3]……列数
			$gameSystem._windowSpaceInfo = windowInfo;
		}
		
		else if(command === "ウィンドウ設定解除")
		{
			windowInfo.xPos = 0;
			windowInfo.yPos = 0;
			windowInfo.width = Graphics.boxWidth;
			windowInfo.rows = 4;
			windowInfo.orgPos = false;
			$gameSystem._windowSpaceInfo = windowInfo;
		}
		
		else if( command === "フォント変更" )
		{
			Graphics.loadFont("GameFont", "fonts/" + args[0]);
			this.wait(10);
			$gameSystem._windowSpaceInfo = windowInfo;
		}
		
		else if( command === "フォント色変更" )
		{
			windowInfo.color = '#' + String(args[0]);
			$gameSystem._windowSpaceInfo = windowInfo;
		}
		
		else if( command === "フォント色初期化" )
		{
			windowInfo.color = windowInfo.defaultColor;
			$gameSystem._windowSpaceInfo = windowInfo;
		}
		
	}
	
})();
