//=============================================================================
// ** RPG Maker MZ/MV - Hakubox_HistoryMessage.js
//=============================================================================

// #region 脚本注释
/*:
 * @plugindesc 对话历史记录插件 (v1.0.0)
 * @version 1.0.0
 * @author hakubox
 * @email hakubox226@gmail.com
 * @target MV MZ
 *
 * @help
 * 一个对话历史记录插件。
 * 
 *
 * ■ [ 从脚本中使用插件命令 ] .............................................
 * 插件命令的功能也可以从脚本中使用。
 * MV版本不支持插件命令本身，因此如果您想在插件命令中使用该功能，需要
 * 使用此脚本，以下是可调用的脚本列表，同时脚本命令与MZ插件指令完全相同。
 * 
 * HistoryMessageUtils.show();          // 打开历史记录面板
 * HistoryMessageUtils.hide();          // 关闭历史记录面板
 * HistoryMessageUtils.addItem({        // 新增一条消息
 *   message: string,
 *   face: string,
 *   faceIndex: number,
 *   title: string,
 *   type: string
 * });
 * 
 * 
 * ■ [ 联系方式 ] ........................................................
 * 
 * 微信：hakubox
 * 邮箱：hakubox＠outlook.com  （需要手动将＠替换为半角字符）
 * 
 * 
 * ■ [ License ] ........................................................
 * 
 * MIT license
 * 
 * 授予任何人使用、复制、修改和分发本插件的权利。
 * 软件是按“现状”提供的，不提供任何明示或暗示的担保，包括但不限于适销性、特定
 * 用途适用性和非侵权的担保。
 * 在任何情况下，版权所有者或许可方均不对因使用本软件或其他交易而引起或与之
 * 相关的任何索赔、损害或其他责任承担责任，无论是因合同、侵权或其他原因引起。
 * 
 * 
 * ■ [ 修订历史 ] ........................................................
 *  v1.0.0  2024/11/06  初版。
 * 
 * 
 * @command show
 * @text 显示历史记录界面
 * @desc 显示主界面立绘，如果有绑定开关会直接修改开关
 * 
 * 
 * @command hide
 * @text 隐藏历史记录界面
 * @desc 隐藏主界面立绘，如果有绑定开关会直接修改开关
 * 
 * 
 * @command addItem
 * @text 添加一条历史记录
 * @desc 添加一条历史记录
 * 
 * @arg type
 * @text 历史记录类型
 * @desc 历史记录类型，默认为 message ，即普通消息
 * @type text
 * @default message
 * 
 * @arg message
 * @text 消息内容
 * @desc 消息内容
 * @type note
 * 
 * @arg face
 * @text 头像文件
 * @type file
 * @dir img/characters/
 * @desc 选择头像文件
 *
 * @arg faceIndex
 * @text 头像索引
 * @type number
 * @min 0
 * @desc 选择头像索引
 * @default 0
 * 
 * @arg title
 * @text 标题/姓名
 * @desc 标题/姓名
 * @type text
 * 
 * 
 * @param basic
 * @text ——————— 基础配置 ———————
 * @default ————————————————————————
 * 
 * @param showHotKey
 * @parent basic
 * @text 显示绑定快捷键
 * @desc 按快捷键会打开/关闭历史记录面板。
 * @type select
 * @option Backspace - 8
 * @value 8
 * @option Tab - 9
 * @value 9
 * @option Enter - 13
 * @value 13
 * @option Shift - 16
 * @value 16
 * @option Ctrl - 17
 * @value 17
 * @option Alt - 18
 * @value 18
 * @option Pause - 19
 * @value 19
 * @option CapsLock - 20
 * @value 20
 * @option Esc - 27
 * @value 27
 * @option Space - 32
 * @value 32
 * @option PageUp - 33
 * @value 33
 * @option PageDown - 34
 * @value 34
 * @option End - 35
 * @value 35
 * @option Home - 36
 * @value 36
 * @option Left - 37
 * @value 37
 * @option Up - 38
 * @value 38
 * @option Right - 39
 * @value 39
 * @option Down - 40
 * @value 40
 * @option Insert - 45
 * @value 45
 * @option Delete - 46
 * @value 46
 * @option 0 - 48
 * @value 48
 * @option 1 - 49
 * @value 49
 * @option 2 - 50
 * @value 50
 * @option 3 - 51
 * @value 51
 * @option 4 - 52
 * @value 52
 * @option 5 - 53
 * @value 53
 * @option 6 - 54
 * @value 54
 * @option 7 - 55
 * @value 55
 * @option 8 - 56
 * @value 56
 * @option 9 - 57
 * @value 57
 * @option A - 65
 * @value 65
 * @option B - 66
 * @value 66
 * @option C - 67
 * @value 67
 * @option D - 68
 * @value 68
 * @option E - 69
 * @value 69
 * @option F - 70
 * @value 70
 * @option G - 71
 * @value 71
 * @option H - 72
 * @value 72
 * @option I - 73
 * @value 73
 * @option J - 74
 * @value 74
 * @option K - 75
 * @value 75
 * @option L - 76
 * @value 76
 * @option M - 77
 * @value 77
 * @option N - 78
 * @value 78
 * @option O - 79
 * @value 79
 * @option P - 80
 * @value 80
 * @option Q - 81
 * @value 81
 * @option R - 82
 * @value 82
 * @option S - 83
 * @value 83
 * @option T - 84
 * @value 84
 * @option U - 85
 * @value 85
 * @option V - 86
 * @value 86
 * @option W - 87
 * @value 87
 * @option X - 88
 * @value 88
 * @option Y - 89
 * @value 89
 * @option Z - 90
 * @value 90
 * @option Left Win - 91
 * @value 91
 * @option Right Win - 92
 * @value 92
 * @option Select - 93
 * @value 93
 * @option Num 0 - 96
 * @value 96
 * @option Num 1 - 97
 * @value 97
 * @option Num 2 - 98
 * @value 98
 * @option Num 3 - 99
 * @value 99
 * @option Num 4 - 100
 * @value 100
 * @option Num 5 - 101
 * @value 101
 * @option Num 6 - 102
 * @value 102
 * @option Num 7 - 103
 * @value 103
 * @option Num 8 - 104
 * @value 104
 * @option Num 9 - 105
 * @value 105
 * @option Multiply - 106
 * @value 106
 * @option Add - 107
 * @value 107
 * @option Subtract - 109
 * @value 109
 * @option Decimal - 110
 * @value 110
 * @option Divide - 111
 * @value 111
 * @option F1 - 112
 * @value 112
 * @option F2 - 113
 * @value 113
 * @option F3 - 114
 * @value 114
 * @option F4 - 115
 * @value 115
 * @option F5 - 116
 * @value 116
 * @option F6 - 117
 * @value 117
 * @option F7 - 118
 * @value 118
 * @option F8 - 119
 * @value 119
 * @option F9 - 120
 * @value 120
 * @option F10 - 121
 * @value 121
 * @option F11 - 122
 * @value 122
 * @option F12 - 123
 * @value 123
 * @option Num Lock - 144
 * @value 144
 * @option Scroll Lock - 145
 * @value 145
 * @option Semicolon - 186
 * @value 186
 * @option Equal - 187
 * @value 187
 * @option Comma - 188
 * @value 188
 * @option Hyphen - 189
 * @value 189
 * @option Period - 190
 * @value 190
 * @option Forward Slash - 191
 * @value 191
 * @option Open Bracket - 219
 * @value 219
 * @option Back Slash - 220
 * @value 220
 * @option Close Bracket - 221
 * @value 221
 * @option Single Quote - 222
 * @value 222
 * 
 * @param bindShowSwitch
 * @parent basic
 * @text 显示绑定开关
 * @desc 可以根据开关控制历史记录面板的打开关闭。
 * @type switch
 * 
 * @param maxMessageCount
 * @parent basic
 * @text 最大消息条数
 * @desc 最大消息条数
 * @type number
 * @default 200
 * 
 * @param minMessageHeight
 * @parent basic
 * @text 最小消息高度
 * @desc 最小消息高度
 * @type number
 * 
 * @param messagePadding
 * @parent basic
 * @text 内边距
 * @desc 内边距
 * @type number
 * @default 10
 * 
 * @param sceneBGM
 * @parent basic
 * @text 场景音乐
 * @desc 场景音乐
 * @type file
 * @dir audio/bgm/
 * 
 * @param choiceFormat
 * @parent basic
 * @text 选择选项格式化文本
 * @desc 选择选项格式化文本
 * @type text
 * @default 选择了【{0}】
 * 
 * @param inputNumberFormat
 * @parent basic
 * @text 输入数字格式化文本
 * @desc 输入数字格式化文本
 * @type text
 * @default 输入了【{0}】
 * 
 * @param ignoreRegex
 * @parent basic
 * @text 忽略消息正则
 * @desc 忽略消息的正则表达式
 * @type text
 * 
 * 
 * @param detail
 * @text ——————— 详细配置 ———————
 * @default ————————————————————————
 * 
 * @param groupConfig
 * @parent detail
 * @text 消息分组配置
 * @type struct<GroupConfig>
 * @default { "actorNameRegex":"\\n[l|c|r]?<(.*?)>", "groupGetFunction": "return '';" }
 * 
 * @param normalMessageConfig
 * @parent detail
 * @text 普通消息配置
 * @type struct<NormalMessageConfig>
 * @default {"itemBgImage":"","titleFontConfig":"{\"fontSize\":\"32\",\"fontWeight\":\"false\",\"fontColor\":\"#765D13\",\"outlineWidth\":\"1\",\"outlineColor\":\"1\"}","faceImageConfig":"{\"visible\":\"true\",\"faceBackground\":\"\",\"width\":\"80\",\"height\":\"80\",\"outlineWidth\":\"4\",\"outlineColor\":\"#FFFFFF\",\"fillColor\":\"#FFFFFF\"}"}
 * 
 * @param specialMessageConfig
 * @parent detail
 * @text 特殊消息类型配置
 * @type struct<SpecialMessageItemConfig>[]
 * @default []
 * 
 * @param commonStyleConfig
 * @parent detail
 * @text 样式风格配置
 * @type struct<CommonStyleConfig>
 * @default {"windowPadding":"10","sceneImage":"","windowImage":"","sceneBlur":"4","splitImage":"","splitLeftImage":"","splitRightImage":"","splitLoc":"center","splitHeight":"50","facePadding":"6"}
 * 
 * @param windowConfig
 * @parent detail
 * @text 窗口配置
 * @desc 窗口配置
 * @type struct<WindowConfig>
 * 
 * @param scrollConfig
 * @parent detail
 * @text 滚动条配置
 * @desc 滚动条配置
 * @type struct<ScrollConfig>
 * @default {"visible":"true","paddingX":"5","paddingY":"10","width":"10","radius":"5","scrollColor":"#51618B","scrollOutlineColor":"#51618B","scrollOutlineWidth":"0","scrollBarColor":"#9EC9ED","scrollBarOutlineColor":"#9EC9ED","scrollBarOutlineWidth":"0"}
 *
 */
/*~struct~GroupConfig:
 * 
 * @param actorNameRegex
 * @text 角色名称获取正则表达式
 * @desc 角色名称获取正则表达式
 * @type text
 * @default \\n[l|c|r]?<(.*?)>
 * 
 * @param groupGetFunction
 * @text 分组标题获取代码
 * @desc 分组标题获取代码
 * @type note
 * @default return '';
 * 
 */
/*~struct~NormalMessageConfig:
 * 
 * @param itemBgImage
 * @text 消息背景图
 * @desc 消息背景图
 * @type file
 * @dir img/pictures/
 * 
 * @param titleFontConfig
 * @text 标题字体配置
 * @desc 标题字体配置
 * @type struct<FontConfig>
 * @default {"fontSize":"32","fontWeight":"false","fontColor":"#765D13","outlineWidth":"1","outlineColor":"#000000"}
 * 
 * @param normalFontConfig
 * @text 基础内容字体配置
 * @desc 基础内容字体配置
 * @type struct<FontConfig>
 * @default {"fontSize":"24","fontWeight":"false","fontColor":"#FFFFFF","outlineWidth":"1","outlineColor":"#000000"}
 * 
 * @param faceImageConfig
 * @text 头像配置
 * @desc 头像配置
 * @type struct<FaceImageConfig>
 * @default { "visible": "true", "width": "80", "height": "80", "outlineWidth": "4", "outlineColor": "#FFFFFF", "fillColor": "#FFFFFF" }
 * 
 */
/*~struct~SpecialMessageItemConfig:
 * 
 * @param messageName
 * @text 消息名称
 * @desc 消息名称
 * @type text
 * 
 * @param remark
 * @text 备注
 * @desc 没有作用，只是给开发者查看而已
 * @type note
 * 
 * @param itemBgImage
 * @text 消息背景图
 * @desc 消息背景图
 * @type file
 * @dir img/pictures/
 * 
 * @param clickBindVariableId
 * @text 点击绑定变量
 * @desc 点击绑定变量
 * @type variable
 * 
 * @param execCommonEventId
 * @text 启动公共事件
 * @desc 启动公共事件
 * @type common_event
 * 
 * @param titleFontConfig
 * @text 标题字体配置
 * @desc 标题字体配置
 * @type struct<FontConfig>
 * @default {}
 * 
 * @param contentFontConfig
 * @text 内容字体配置
 * @desc 内容字体配置
 * @type struct<FontConfig>
 * @default {}
 * 
 * @param faceImageConfig
 * @text 头像配置
 * @desc 头像配置
 * @type struct<FaceImageConfig>
 * @default { "visible": "false" }
 * 
 */
/*~struct~CommonStyleConfig:
 *
 * @param windowPadding
 * @text 窗口内边距
 * @desc 窗口内边距
 * @type number
 * @default 10
 *
 * @param sceneImage
 * @text 场景背景图
 * @desc 场景背景图，不配置默认显示为透明
 * @type file
 * @dir img/pictures/
 *
 * @param windowImage
 * @text 窗口背景图
 * @desc 窗口背景图，配置后会隐藏原有窗口
 * @type file
 * @dir img/pictures/
 * 
 * @param sceneBlur
 * @text 场景背景模糊度
 * @desc 场景背景模糊度，不配置默认为 4 ，为 0 则为不模糊
 * @type number
 * @default 4
 * 
 * @param groupFontConfig
 * @text 分组标题字体配置
 * @desc 分组标题字体配置
 * @type struct<FontConfig>
 * @default {"fontSize":"24","fontWeight":"false","fontColor":"#FFFFFF","outlineWidth":"0","outlineColor":"#FFFFFF"}
 * 
 * @param splitImage
 * @text 分组标题背景图
 * @desc 分组标题背景图
 * @type file
 * @dir img/pictures/
 * 
 * @param splitLeftImage
 * @text 左侧分组标题背景图
 * @desc 左侧分组标题背景图
 * @type file
 * @dir img/pictures/
 * 
 * @param splitRightImage
 * @text 右侧分组标题背景图
 * @desc 右侧分组标题背景图
 * @type file
 * @dir img/pictures/
 * 
 * @param splitLoc
 * @text 分组标题位置
 * @desc 分组标题位置，可选左、中、右，默认中间
 * @type select
 * @option 左 - left
 * @value left
 * @option 中 - center
 * @value center
 * @option 右 - right
 * @value right
 * @default center
 * 
 * @param splitHeight
 * @text 分组标题高度
 * @desc 分组标题高度
 * @type number
 * @default 50
 * 
 * @param facePadding
 * @text 头像与文字间距
 * @desc 头像与文字间距
 * @type number
 * @default 6
 * 
 */
/*~struct~WindowConfig:
 *
 * @param windowWidth
 * @text 窗口宽度
 * @desc 窗口宽度
 * @type number
 *
 * @param windowHeight
 * @text 窗口高度
 * @desc 窗口高度
 * @type number
 * 
 */
/*~struct~FaceImageConfig:
 *
 * @param visible
 * @text 是否显示头像
 * @desc 是否显示头像
 * @type boolean
 * @on 显示
 * @off 不显示
 * @default true
 *
 * @param faceBackground
 * @text 头像背景图
 * @desc 头像背景图，会和其他选项冲突
 * @type file
 * @dir img/pictures/
 *
 * @param width
 * @text 头像宽度
 * @desc 头像宽度
 * @type number
 * @default 80
 *
 * @param height
 * @text 头像高度
 * @desc 头像高度
 * @type number
 * @default 80
 *
 * @param outlineWidth
 * @text 边框粗细
 * @desc 边框粗细，使用头像背景图时指内边距
 * @type number
 * @default 4
 *
 * @param outlineColor
 * @text 边框颜色
 * @desc 边框颜色
 * @type text
 * @default #FFFFFF
 *
 * @param fillColor
 * @text 背景颜色
 * @desc 背景颜色
 * @type text
 * @default #FFFFFF
 * 
 */
/*~struct~FontConfig:
 *
 * @param fontSize
 * @text 字体大小
 * @desc 字体大小
 * @type number
 * @default 24
 *
 * @param fontWeight
 * @text 是否粗体
 * @desc 是否粗体（MV不支持）
 * @type boolean
 * @on 加粗
 * @off 正常
 * @default false
 *
 * @param fontColor
 * @text 字体颜色
 * @desc 字体颜色
 * @type text
 * @default #FFFFFF
 *
 * @param outlineWidth
 * @text 边框粗细
 * @desc 边框粗细
 * @type number
 * @default 1
 *
 * @param outlineColor
 * @text 边框颜色
 * @desc 边框颜色
 * @type text
 * @default #FFFFFF
 *
 */
/*~struct~ScrollConfig:
 *
 * @param visible
 * @text 是否显示滚动条
 * @desc 是否显示滚动条
 * @type boolean
 * @on 显示
 * @off 不显示
 * @default true
 *
 * @param paddingX
 * @text 左右间距
 * @desc 左右间距
 * @type number
 * @default 5
 *
 * @param paddingY
 * @text 上下间距
 * @desc 上下间距
 * @type number
 * @default 10
 * 
 * @param width
 * @text 滚动条宽度
 * @desc 滚动条宽度
 * @type number
 * @default 10
 * 
 * @param radius
 * @text 滚动条圆角像素
 * @desc 滚动条圆角像素，建议为宽度的一半
 * @type number
 * @default 5
 * 
 * @param barWidth
 * @text 滚动条滑块宽度
 * @desc 滚动条滑块宽度
 * @type number
 * @default 10
 * 
 * @param scrollColor
 * @text 滚动条颜色
 * @desc 滚动条颜色
 * @type text
 * @default #51618B
 * 
 * @param scrollOutlineColor
 * @text 滚动条描边颜色
 * @desc 滚动条描边颜色
 * @type text
 * @default #51618B
 * 
 * @param scrollOutlineWidth
 * @text 滚动条描边宽度
 * @desc 滚动条描边宽度
 * @type number
 * @default 0
 * 
 * @param scrollBarColor
 * @text 滚动条滑块颜色
 * @desc 滚动条滑块颜色
 * @type text
 * @default #9EC9ED
 * 
 * @param scrollBarOutlineColor
 * @text 滚动条滑块描边颜色
 * @desc 滚动条滑块描边颜色
 * @type text
 * @default #9EC9ED
 * 
 * @param scrollBarOutlineWidth
 * @text 滚动条滑块描边宽度
 * @desc 滚动条滑块描边宽度
 * @type number
 * @default 0
 * 
 */

// * @param pageWidth
// * @text 画面宽度
// * @desc 画面宽度
// * @type number
// *
// * @param pageHeight
// * @text 画面高度
// * @desc 画面高度
// * @type number

// #endregion
/** 全局数据 */
let $gameHistoryMessage = {
  /**
   * 消息列表
   * @type {<{message: string, face: string, faceIndex: number, title: string, date: Date, type: string}>[]}
   */
  messages: [
  ],
  prevVisible: false
};
/** 插件名称 */

const Hakubox_PluginName = document.currentScript ? decodeURIComponent(document.currentScript.src.match(/^.*\/(.+)\.js$/)[1]) : "Hakubox_HistoryMessage";
(() => {

  // #region 插件参数解释器
  class PluginParamsParser {
    constructor(predictEnable = true) {
      this._predictEnable = predictEnable;
    }
    static parse(params, typeData, predictEnable = true) {
      return new PluginParamsParser(predictEnable).parse(params, typeData);
    }
    parse(params, typeData, loopCount = 0) {
      if (++loopCount > 255)
        throw new Error("endless loop error");
      const result = {};
      for (const name in typeData) {
        if (params[name] === "" || params[name] === undefined) {
          result[name] = null;
        }
        else {
          result[name] = this.convertParam(params[name], typeData[name], loopCount);
        }
      }
      if (!this._predictEnable)
        return result;
      if (typeof params === "object" && !(params instanceof Array)) {
        for (const name in params) {
          if (result[name])
            continue;
          const param = params[name];
          const type = this.predict(param);
          result[name] = this.convertParam(param, type, loopCount);
        }
      }
      return result;
    }
    convertParam(param, type, loopCount) {
      if (typeof type === "string") {
        let str = param;
        if (str[0] == '"' && str[str.length - 1] == '"') {
          str = str.substring(1, str.length - 1).replace(/\\n/g, '\n').replace(/\\"/g, '"')
        }
        return this.cast(str, type);
      }
      else if (typeof type === "object" && type instanceof Array) {
        const aryParam = JSON.parse(param);
        if (type[0] === "string") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else if (type[0] === "number") {
          return aryParam.map((strParam) => this.cast(strParam, type[0]));
        } else {
          if (!aryParam.length) return [];
          else return aryParam.map((strParam) => this.parse(JSON.parse(strParam), type[0]), loopCount);
        }
      }
      else if (typeof type === "object") {
        return this.parse(JSON.parse(param), type, loopCount);
      }
      else {
        throw new Error(`${type} is not string or object`);
      }
    }
    cast(param, type) {
      switch (type) {
        case "any":
          if (!this._predictEnable)
            throw new Error("Predict mode is disable");
          return this.cast(param, this.predict(param));
        case "string":
          return param;
        case "number":
          if (param.match(/^\-?\d+\.\d+$/))
            return parseFloat(param);
          return parseInt(param);
        case "boolean":
          return param === "true";
        default:
          throw new Error(`Unknow type: ${type}`);
      }
    }
    predict(param) {
      if (param.match(/^\-?\d+$/) || param.match(/^\-?\d+\.\d+$/)) {
        return "number";
      }
      else if (param === "true" || param === "false") {
        return "boolean";
      }
      else {
        return "string";
      }
    }
  }

  const typeDefine = {
    groupConfig: {},
    normalMessageConfig: {
      faceImageConfig: {},
      titleFontConfig: {}
    },
    specialMessageConfig: [
      {
        faceImageConfig: {},
        titleFontConfig: {},
        contentFontConfig: {}
      }
    ],
    commonStyleConfig: {},
    windowConfig: {},
    scrollConfig: {}
  };

  const params = PluginParamsParser.parse(PluginManager.parameters(Hakubox_PluginName), typeDefine);

  /** 窗口配置 */
  const windowConfig = params.windowConfig;
  /** 分组配置 */
  const groupConfig = params.groupConfig;
  /** 常规样式配置 */
  const commonStyleConfig = params.commonStyleConfig;
  /** 普通消息类型配置 */
  const normalMessageConfig = params.normalMessageConfig;
  /** 特殊消息类型配置 */
  const specialMessageConfig = params.specialMessageConfig;
  /** 滚动条配置 */
  const scrollConfig = params.scrollConfig;

  /** 窗口大小 */
  const windowSize = {
    /** 画面宽度 */
    get pageWidth() {
      return windowConfig.pageWidth || Graphics.width;
    },
    /** 画面高度 */
    get pageHeight() {
      return windowConfig.pageHeight || Graphics.height;
    },
    /** 页面区域宽度 */
    get windowWidth() {
      return windowConfig.windowWidth || windowConfig.pageWidth || Graphics.boxWidth + 8;
    },
    /** 页面区域高度 */
    get windowHeight() {
      return windowConfig.windowHeight || windowConfig.pageHeight || Graphics.boxHeight + 8;
    }
  }

  // #region 历史记录场景

  let SuperScene_Message;
  if (Utils.RPGMAKER_NAME === "MZ") {
    // superScene_Message = Scene_Base;
    SuperScene_Message = Scene_Message;

  } else {
    function Scene_Message_MV() {
      this.initialize(...arguments);
    }
    Scene_Message_MV.prototype = Object.create(Scene_Base.prototype);
    Scene_Message_MV.prototype.constructor = Scene_Message_MV;
    Scene_Message_MV.prototype.initialize = function () {
      Scene_Base.prototype.initialize.call(this);
    };
    Scene_Message_MV.prototype.isMessageWindowClosing = function () {
      return this._messageWindow.isClosing();
    };
    Scene_Message_MV.prototype.createAllWindows = function () {
      this.createMessageWindow();
    };
    Scene_Message_MV.prototype.createMessageWindow = function () {
      const rect = this.messageWindowRect();
      this._messageWindow = new Window_Message(rect);
      this.addWindow(this._messageWindow);
    };
    Scene_Message_MV.prototype.messageWindowRect = function () {
      const ww = Graphics.boxWidth;
      const wh = this.calcWindowHeight(4, false) + 8;
      const wx = (Graphics.boxWidth - ww) / 2;
      const wy = 0;
      return new Rectangle(wx, wy, ww, wh);
    };
    Scene_Base.prototype.calcWindowHeight = function (numLines, selectable) {
      if (selectable) {
        return Window_Selectable.prototype.fittingHeight(numLines);
      }
      else {
        return Window_Base.prototype.fittingHeight(numLines);
      }
    };
    SuperScene_Message = Scene_Message_MV;
  }

  let _cacheContentHeight = 0;
  let _scrollY = 0;

  class Scene_HistoryMessage extends SuperScene_Message {
    prepare() {
      super.prepare();
      Input.clear();
    }

    hasCommonEvent() {
      return !!this._commonEvent;
    }
    updateCommonEvent() {
      if (!this.hasCommonEvent()) {
        return;
      }
      this._commonEvent.update();
      $gameScreen.update();
      if (param.activateTimer) {
        $gameTimer.update(true);
      }
      this.checkGameover();
      this.updateTouchPicturesIfNeed();
    }
    setupEvents() {
      this._events = [];
      this._commonEvents = [];
      for (const event of $dataMap.events.filter(event => !!event)) {
        this._events[event.id] = new Game_Event(this._mapId, event.id);
      }
      for (const commonEvent of this.parallelCommonEvents()) {
        this._commonEvents.push(new Game_CommonEvent(commonEvent.id));
      }
      this.refreshTileEvents();
    }
    updateInterpreter() {
      for (; ;) {
        this._interpreter.update();
        if (this._interpreter.isRunning()) {
          return;
        }
        if (this._interpreter.eventId() > 0) {
          this.unlockEvent(this._interpreter.eventId());
          this._interpreter.clear();
        }
        if (!this.setupStartingEvent()) {
          return;
        }
      }
    }
    setupStartingEvent() {
      if (this._interpreter.setupReservedCommonEvent()) {
        return true;
      }
      if (this.setupAutorunCommonEvent()) {
        return true;
      }
      return false;
    }
    setupAutorunCommonEvent() {
      for (const commonEvent of this.autorunCommonEvents()) {
        if ($gameSwitches.value(commonEvent.switchId)) {
          this._interpreter.setup(commonEvent.list);
          return true;
        }
      }
      return false;
    }
    autorunCommonEvents() {
      return $dataCommonEvents.filter(
        commonEvent => commonEvent && commonEvent.trigger === 1
      );
    }
    executeCommonEvent(eventId) {
      let event = $dataCommonEvents[eventId];
      this._interpreter.isRunning(true);
      if (event && !this._interpreter.isRunning()) {
        this._interpreter.setup(event.list);
        this._interpreter.executeCommand();
      }
    }

    update() {
      this.updateCommonEvent();
      super.update();

      this.updateInterpreter();
      for (const commonEvent of this._commonEvents) {
        commonEvent.update();
      }

      if (this.scrollBar) {
        this.scrollBar.contentHeight = _cacheContentHeight;
        this.scrollBar.onScroll(_scrollY);
      }
    }

    createWindowLayer() {
      if (Utils.RPGMAKER_NAME === "MZ") {
        this._windowLayer = new WindowLayer();
        this._windowLayer.x = 0;
        this._windowLayer.y = 0;
        this.addChild(this._windowLayer);
      } else {
        var width = windowSize.pageWidth;
        var height = windowSize.pageHeight;
        var x = 0;
        var y = 0;
        this._windowLayer = new WindowLayer();
        this._windowLayer.move(x, y, width, height);
        this.addChild(this._windowLayer);
      }
    }
    createButtons() {
      if (ConfigManager.touchUI) {
        this._cancelButton = new Sprite_Button("cancel");

        this._cancelButton.x = windowSize.windowWidth + ((Graphics.width - windowSize.windowWidth) / 2) - this._cancelButton.width - 4;
        this._cancelButton.y = this.buttonY() + 6;
        this.addWindow(this._cancelButton);
      }
    }
    createAllWindow() {
      this.createWindow();
      super.createAllWindows();
    }

    createWindow() {
      const _paddingX = scrollConfig.visible ? scrollConfig.paddingX : 0;
      const _paddingY = scrollConfig.visible ? scrollConfig.paddingY : 0;
      const _scrollWidth = scrollConfig.visible ? scrollConfig.width : 0;

      this._window = new Window_HistoryMessage(new Rectangle(
        (windowSize.pageWidth - windowSize.windowWidth) / 2,
        (windowSize.pageHeight - windowSize.windowHeight) / 2,
        windowSize.windowWidth - _scrollWidth - _paddingX * 2 - 5,
        windowSize.windowHeight
      ), this, this._cancelButton);
      this.addWindow(this._window);

      if (scrollConfig.visible) {
        let _xSkew = Utils.RPGMAKER_NAME === "MZ" ? 2 : -5;
        this.scrollBar = new HakuScrollBar(new Rectangle(
          (windowSize.pageWidth + windowSize.windowWidth) / 2 - _paddingX - _scrollWidth + _xSkew,
          (windowSize.pageHeight - windowSize.windowHeight) / 2 + _paddingY,
          _scrollWidth,
          windowSize.windowHeight - _paddingY * 2
        ), {
          barWidth: scrollConfig.barWidth || scrollConfig.width,
          type: 'vertical',
          backround: scrollConfig.scrollColor || '#51618B',
          barBackground: scrollConfig.scrollBarColor || '#9EC9ED',
          radius: scrollConfig.radius || 5,
          dragMove: this._window.dragMove.bind(this._window),
        });
        this.addWindow(this.scrollBar);
        this._window.scrollBar = this.scrollBar;
      }
    }
    setBackgroundOpacity(opacity) {
      this._backgroundSprite.opacity = opacity;
    }
    createBackground() {
      this._backgroundSprite = new Sprite();
      if (params.commonStyleConfig.sceneImage) {
        this._backgroundSprite.bitmap = new Bitmap(windowSize.pageWidth, windowSize.pageHeight);
        const bgBitmap = ImageManager.loadBitmap("img/pictures/", params.commonStyleConfig.sceneImage);
        bgBitmap.addLoadListener(() => {
          this._backgroundSprite.bitmap.blt(bgBitmap,
            0, 0, 
            bgBitmap.width, bgBitmap.height, 
            0, 0,
            windowSize.pageWidth, windowSize.pageHeight
          );
          this.addChild(this._backgroundSprite);
        });
      } else {
        this._backgroundFilter = new PIXI.filters.BlurFilter();
        this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
        this._backgroundSprite.filters = [this._backgroundFilter];
        this.addChild(this._backgroundSprite);
        this.setBackgroundOpacity(192);
      }

      if (params.commonStyleConfig.windowImage) {
        const _bgImg = ImageManager.loadBitmap('img/pictures/', params.commonStyleConfig.windowImage);
        const _windowBgSprite = new Sprite(_bgImg);
        _windowBgSprite.bitmap = new Bitmap(windowSize.pageWidth, windowSize.pageHeight);
        _windowBgSprite.x = (windowSize.pageWidth - windowSize.windowWidth) / 2;
        _windowBgSprite.y = (windowSize.pageHeight - windowSize.windowHeight) / 2;
        _windowBgSprite.bitmap.blt(_bgImg,
          0, 0,
          _bgImg.width, _bgImg.height,
          0, 0,
          windowSize.windowWidth, windowSize.windowHeight
        );
        this.addChild(_windowBgSprite);
      }
    }
    create() {
      super.create();
      this.createBackground();
      this.createWindowLayer();
      this.createButtons();
      this.createAllWindow();

      this._commonEvents = [];

      if (!this._interpreter) {
        this._interpreter = new Game_Interpreter();
      }
    }
  }

  // #endregion

  // #region 滚动条组件

  class HakuScrollBar extends Window_Base {
    /**
     * @param {object} config 配置项
     * @param {'horizontal'|'vertical'} [config.direction='vertical'] 方向 
     * @param {number} [config.width=10] 宽度
     * @param {Rectangle} config.rect 滚动条区域
     * @param {number} [config.radius=10] 圆角半径
     */
    constructor(rect, config) {
      super(rect);

      this.padding = 0;
      this._scrollTop = 0;
      this._type = config.type;
      this._radius = config.radius || 5;
      this._rect = rect;
      this._scrollBarHeight = 0;
      this._pageHeight = rect.height;
      this._contentHeight = rect.height;
      this._background = config.backround || '#FFFFFF';
      this._outlineColor = config.scrollOutlineColor || '#9EC9ED';
      this._outlineWidth = config.scrollOutlineWidth || 0;
      this._barBackground = config.barBackground || '#9EC9ED';
      this._barOutlineColor = config.scrollBarOutlineColor || '#9EC9ED';
      this._barOutlineWidth = config.scrollBarOutlineWidth || 0;
      this._dragMove = config.dragMove || null;
      this._barWidth = config.barWidth || this._rect.width;

      this._scrollBarTop = 0;


      this.initialize.call(this, this._rect);

      this._mouseUp = this.onMouseUp.bind(this);
      this._mouseDown = this.onMouseDown.bind(this);
      this._mouseMove = this.onMouseMove.bind(this);

      // 拖拽相关
      this.isStartDragScroll = false;
      this.startDragY = 0;
      this.prevDragY = 0;
      
      document.addEventListener('mousedown', this._mouseDown);
      document.addEventListener('mouseup', this._mouseUp);
      document.addEventListener('mousemove', this._mouseMove);

      this.update();
    }

    standardPadding() {
      return 0;
    }
    textPadding() {
      return 0;
    }

    destory() {
      this._mouseMove = null;
      document.removeEventListener('mousedown', this._mouseDown);
      document.removeEventListener('mouseup', this._mouseUp);
      document.removeEventListener('mousemove', this._mouseMove);
    }

    _refreshAllParts() {
    }

    itemPadding() {
      return 0;
    }
    updatePadding() {
      this.padding = 0;
    }

    get context() {
      return this.contents.context;
    }
    /** 设置内容高度 */
    get contentHeight() {
      return this._contentHeight;
    }
    set contentHeight(height) {
      this._contentHeight = height;
    }
    /** 整个滚动条高度 */
    get scrollHeight() {
      return this._rect.height;
    }

    initialize(rect) {
      if (Utils.RPGMAKER_NAME === "MZ") {
        super.initialize.call(this, rect);
      } else {
        // @ts-ignore // MV Compatible
        this._windowRect = rect;
        // @ts-ignore // MV Compatible
        super.initialize(rect.x, rect.y, rect.width, rect.height);
      }
      // this.deactivate();
    }

    /** 更新 */
    update() {
      super.update();
      
      this.contents.drawRounded(
        0,
        0,
        this._rect.width,
        this._rect.height,
        {
          radius: this._radius,
          fillColor: this._background,
          outlineColor: this._outlineColor,
          outlineWidth: this._outlineWidth,
        }
      );

      if (!this.isStartDragScroll) {
        this.setScroll(_scrollY);
        this.updateScrollBarHeight();
      }

      this.contents.drawRounded(
        (this._rect.width - this._barWidth) / 2,
        this._scrollBarTop,
        this._barWidth,
        this._scrollBarHeight,
        {
          radius: this._radius,
          fillColor: this._barBackground,
          outlineColor: this._barOutlineColor,
          outlineWidth: this._barOutlineWidth
        }
      );
    }

    onMouseDown(e) {
      if (e.x >= this.x && e.x <= this.x + this._rect.width) {
        if (e.y >= this.y + this._scrollBarTop && e.y <= this.y + this._scrollBarTop + this._scrollBarHeight) {
          this.isStartDragScroll = true;
          this.startDragY = e.y - this.y - this._scrollBarTop;
          this.prevDragY = this._scrollBarTop;
        } else if (e.y >= this.y && e.y <= this.y + this._rect.height) {
          this.isStartDragScroll = false;
          if (e.y < this.y + this._scrollBarTop) {
            if (this._dragMove) this._dragMove(this._scrollTop - this._pageHeight, true);
          } else if (e.y > this.y + this._scrollBarTop + this._scrollBarHeight) {
            if (this._dragMove) this._dragMove(this._scrollTop + this._pageHeight, true);
          }
        }
      }
    }
    onMouseUp(e) {
      this.isStartDragScroll = false;
    }
    onMouseMove(e) {
      if (this.isStartDragScroll) {
        const moveY = e.y - this.y - this.prevDragY - this.startDragY;
        const _barTop = this.prevDragY + moveY;
        let _re = _barTop / (this.scrollHeight - this._scrollBarHeight) * (this._contentHeight - this._pageHeight);
        if (_re < 0) _re = 0;
        else if (_re >= this._contentHeight - this._pageHeight) _re = this._contentHeight - this._pageHeight;
        this._dragMove(_re, false);
      }
    }

    /** 更新滚动条滑块高度 */
    updateScrollBarHeight() {
      let _height = this._pageHeight * this._pageHeight / this._contentHeight;
      if (_height > this._pageHeight) {
        _height = this._pageHeight;
      } else if (_height < 30) {
        _height = 30;
      }
      this._scrollBarHeight = Math.floor(_height);
    }

    /** 滚动 */
    onScroll(scrollTop) {
      this.setScroll(scrollTop);
    }

    /** 设置滚动位置 */
    setScroll(scrollTop) {
      this._scrollTop = scrollTop;
      if (scrollTop < 0 || this._contentHeight < this._pageHeight) {
        this._scrollBarTop = 0;
      } else if (scrollTop > this._contentHeight - this._pageHeight) {
        this._scrollBarTop = this._contentHeight - this._pageHeight;
      } else {
        this._scrollBarTop = Math.floor(this.scrollHeight * this._scrollTop / this._contentHeight);
      }
    }
  }

  // #endregion

  // #region 历史记录窗口

  class HakuSelectableWindow extends Window_Selectable {
    initialize(rect) {

      if (Utils.RPGMAKER_NAME === "MZ") {
        super.initialize.call(this, rect);
      } else {
        // @ts-ignore // MV Compatible
        this._windowRect = rect;
        // @ts-ignore // MV Compatible
        super.initialize(rect.x, rect.y, rect.width, rect.height);
      }

      // this.deactivate();
    }

    _refreshAllParts() {
      if (!params.commonStyleConfig.windowImage) {
        super._refreshAllParts();
      }
    }
  }

  /** 展示页页面 */
  class Window_HistoryMessage extends HakuSelectableWindow {
    initialize(rect, scene, cancelButton) {
      /**
       * 分组信息
       * @type {{index: number, groupName: string}[]}
       */
      this.groupInfo = [];
      /**
       * 列表（包含分组项）
       * @type {object[]}
       */
      this.items = [];
      // 清空高度值
      /** 记录高度信息 */
      this.itemRects = [];

      if (commonStyleConfig.splitLeftImage) this._leftBitmap = ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitLeftImage);
      if (commonStyleConfig.splitRightImage) this._rightBitmap = ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitRightImage);
      if (commonStyleConfig.splitImage) this._titleBitmap = ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitImage);
      if (normalMessageConfig.itemBgImage) this._itemBgBitmap = ImageManager.loadBitmap('img/pictures/', normalMessageConfig.itemBgImage);
      const _faceCfg = normalMessageConfig.faceImageConfig;
      if (_faceCfg.visible && _faceCfg.faceBackground) this._bitmapFaceBg = ImageManager.loadBitmap('img/pictures/', _faceCfg.faceBackground);


      super.initialize.call(this, rect);
      this.padding = commonStyleConfig.windowPadding || 10;
      this._scene = scene;
      this._cancelButton = cancelButton;
      this.refresh(true);
      this.select(0);
      this.activate();

      if (params.sceneBGM) {
        AudioManager.playBgm({
          name: params.sceneBGM,
          pan: 0,
          pitch: 100,
          volume: 90,
        });
        AudioManager.fadeInBgm(1);
      }
      _cacheContentHeight = this.overallHeight();
    }

    dragMove(scrollTop, smooth = true) {
      if (Utils.RPGMAKER_NAME === "MZ") {
        if (smooth) {
          this.smoothScrollTo(0, scrollTop);
        } else {
          this.scrollTo(0, scrollTop);
        }
      } else {
        this._scrollY = scrollTop;
        this.refresh();
        this.updateCursor();
      }
    }

    paint() {
      this.contents.clear();
      super.paint();
    }

    update() {
      this.processHandling();
      super.update();
      

      _scrollY = this.scrollY();
    }

    processHandling() {
      if (this.isOpenAndActive()) {
        if (TouchInput.isCancelled() || Input.isRepeated("cancel")) {
          this.processCancel();
        }
        if (Input.isRepeated("ok")) {
          let _index = typeof this.index == 'function' ? this.index() : this.index;
          const _type = this.items[_index].type;
          for (let i = 0; i < specialMessageConfig.length; i++) {
            if (specialMessageConfig[i].messageName === _type) {
              $gameVariables.setValue(specialMessageConfig[i].clickBindVariableId, _index);
              $gameTemp.reserveCommonEvent(specialMessageConfig[i].execCommonEventId);
            }
          }
        }
      }
    }

    processCancel() {
      if (params.sceneBGM) {
        AudioManager.fadeOutBgm(1);
      }
      if (this.scrollBar) {
        this.scrollBar.destory();
      }
      SoundManager.playCancel();
      Input.update();
      TouchInput.update();
      this.deactivate();
      this.close();
      this._scene.stop();
      HistoryMessageUtils.hide();
    };

    // #region 刷新函数 
    refresh(isReView = false) {
      if (this.contents) {
        this.contents.clear();
      }

      if (isReView) {
        // 清空高度值
        this.itemRects = [];
        this.items = [];

        // 计算分组
        this.groupInfo = [];
        let _groupName = '';

        for (let i = 0; i < $gameHistoryMessage.messages.length; i++) {
          const item = $gameHistoryMessage.messages[i];
          if (item.date && item.date !== _groupName) {
            _groupName = item.date;
            this.groupInfo.push({ index: i, type: 'group', groupName: _groupName });
            this.items.push({
              index: i,
              type: 'group',
              group: _groupName
            });
            this.itemRects.push({
              height: commonStyleConfig.splitHeight || 50,
            });
          }
          this.items.push({
            index: i,
            type: i.type,
            group: _groupName,
            face: item.face,
            faceIndex: item.faceIndex,
            title: item.title,
            message: item.message,
            type: item.type,
          });
          let _height = 30;
          if (item.title) {
            _height += this.textSizeHk(item.title, () => {
              this.contents.fontSize = commonStyleConfig.titleSize || 30;
              this.contents.textColor = commonStyleConfig.titleColor || '#F7F79B';
            }).height || 34;
          }
          for (let o = 0; o < item.message.length; o++) {
            let _rect = this.textSizeHk(item.message[o]);
            if (_rect && _rect.height) {
              _height += _rect.height || 32;
            }
          }

          const _faceCfg = normalMessageConfig.faceImageConfig;

          if (item.face && _faceCfg.visible && _height < _faceCfg.height + 2 * _faceCfg.outlineWidth + 4 + params.messagePadding * 2) {
            _height = _faceCfg.height + 2 * _faceCfg.outlineWidth + 4 + params.messagePadding * 2;
          }

          if (!isNaN(params.minMessageHeight) && _height < params.minMessageHeight) {
            _height = params.minMessageHeight;
          }

          this.itemRects.push({
            height: _height,
          });
        }
      }

      super.refresh();
    }
    // #endregionregion


    itemHeight() {
      return 100;
    }

    maxItems() {
      return this.items.length;
    }

    /** 获取高度 */
    overallHeight(maxCount) {
      if (!this.itemRects || this.itemRects.length === 0) return 0;
      else if (this.itemRects.length === 1) return this.itemRects[0].height;
      else {
        let _height = 0;
        let _maxCount = maxCount === undefined ? this.itemRects.length : maxCount;
        for (let i = 0; i < _maxCount; i++) {
          _height += this.itemRects[i].height;
        }
        return _height;
      }
    }
    updateCursor() {
      const rect = this.itemRect(this._index);
      if (this._index >= 0 && this._index < this.items.length && this.items[this._index].type !== 'group') {
        this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
      }
    }
    scrollX() {
      return this._scrollX;
    }
    scrollY() {
      return this._scrollY;
    }
    topRow() {
      let _height = 0;
      const _scrollY = this.scrollY();
      for (let i = 0; i < this.itemRects.length; i++) {
        _height += this.itemRects[i].height;
        if (_scrollY < _height) {
          if (Utils.RPGMAKER_NAME === "MZ") {
            return i - 2 < 0 ? 0 : i - 2;
          } else {
            return i;
          }
        }
      }
    }
    bottomRow() {
      let _height = 0;
      let minIndex = undefined;
      const pageHeight = this.height - this.padding * 2;
      let _scrollY = this.scrollY();
      let _pageRows = undefined;
      for (let i = 0; i < this.itemRects.length; i++) {
        if (minIndex === undefined) {
          if (_height >= _scrollY) {
            _scrollY += pageHeight;
            minIndex = i;
          }
          _height += this.itemRects[i].height;
        } else {
          _height += this.itemRects[i].height;
          if (_height >= _scrollY) {
            _pageRows = i - minIndex + 1;
            break;
          }
        }
      }
      if (_pageRows === undefined) _pageRows = this.itemRects.length;
      return Math.max(0, this.topRow() + _pageRows - 1);
    }
    setTopRow(row) {
      if (!isNaN(row)) {
        var scrollY = this.overallHeight(row);
        if (this._scrollY !== scrollY) {
          this._scrollY = scrollY;
          this.refresh();
          this.updateCursor();
        }
      }
    }
    maxVisibleItems(topIndex) {
      let _height = 0;
      let _topIndex = topIndex < 0 ? 0 : topIndex;
      const _contentHeight = this.contentsHeight();
      for (let i = _topIndex; i < this.itemRects.length; i++) {
        _height += this.itemRects[i].height;
        if (_contentHeight <= _height) {
          return i - _topIndex + 4 > this.itemRects.length - 1 ? this.itemRects.length - 1 : i - _topIndex + 4;
        }
      }
      return this.itemRects.length - _topIndex;
    }
    drawAllItems() {
      const topIndex = this.topIndex();
      const maxItemCount = this.maxVisibleItems(topIndex);
      for (let i = 0; i < maxItemCount; i++) {
        const index = topIndex + i;
        if (index < this.maxItems()) {
          if (this.items[index]) {
            if (this.items[index].type !== 'group' && this.drawItemBackground) {
              this.drawItemBackground(index);
            }
          }
          this.drawItem(index);
        }
      }
    }
    hitTest(x, y) {
      let isArea;
      if (Utils.RPGMAKER_NAME === "MZ") {
        isArea = this.innerRect.contains(x, y);
      } else {
        isArea = this.isContentsArea(x, y);
      }
      if (isArea) {
        const cx = this.origin.x + x - this.padding;
        const cy = this.origin.y + y - this.padding;
        const topIndex = this.topIndex();
        const maxItemCount = this.maxVisibleItems(topIndex);
        for (let i = 0; i < maxItemCount; i++) {
          const index = topIndex + i;
          if (index < this.maxItems()) {
            const rect = this.itemRect(index);
            if (rect.contains(cx, cy)) {
              return index;
            }
          }
        }
      }
      return -1;
    }

    ensureCursorVisible(smooth) {
      if (!this.itemRects.length) return;
      if (Utils.RPGMAKER_NAME === "MZ") {
        if (this._cursorAll) {
          this.scrollTo(0, 0);
        } else if (this.innerHeight > 0 && this.row() >= 0) {
          const scrollY = this.scrollY();
          let itemTop = 0;
          let i = 0;
          let _row = this.row();

          for (; i < this.itemRects.length; i++) {
            if (i < _row) {
              itemTop += this.itemRects[i].height;
            }
          }

          const lastHeight = this.itemRects[i >= this.itemRects.length ? (i - 1) : i].height;

          const itemBottom = itemTop + lastHeight;
          const scrollMin = itemBottom - this.innerHeight;
          if (scrollY > itemTop) {
            if (smooth) {
              this.smoothScrollTo(0, itemTop);
            } else {
              this.scrollTo(0, itemTop);
            }
          } else if (scrollY < scrollMin) {
            if (smooth) {
              this.smoothScrollTo(0, scrollMin);
            } else {
              this.scrollTo(0, scrollMin);
            }
          }
        }
      } else {
        var row = this.row();
        if (row < this.topRow()) {
          this.setTopRow(row);
        } else if (row > this.bottomRow() - 1) {
          this.setBottomRow(row);
        }
      }
    }

    setBottomRow(row) {
      this.setTopRow(row - (this.maxPageRowsByIndex(row - this._index) - 1));
    }
    /** 获取当前页的行数 */
    maxPageRowsByIndex(row) {
      const pageHeight = this.height - this.padding * 2;
      let _maxPageRows = 0;
      let _height = 0;
      for (let i = row; i < this.itemRects.length; i++) {
        const rect = this.itemRects[i];
        if (_height + rect.height < pageHeight) {
          _height += rect.height;
          _maxPageRows++;
        } else {
          break;
        }
      }

      return _maxPageRows;
    }

    itemRectWithPadding(index) {
      const rect = this.itemRect(index);
      const padding = this.itemPadding();
      rect.x += padding;
      rect.width -= padding * 2;
      return rect;
    }

    itemPadding() {
      return 8;
    }

    itemLineRect(index) {
      const rect = this.itemRectWithPadding(index);
      // const padding = (rect.height - this.lineHeight()) / 2;
      // rect.height -= padding * 2;
      return rect;
    }

    redrawItem(index) {
      if (index >= 0) {
        this.clearItem(index);
        if (this.items[index]) {
          if (this.items[index].type !== 'group') {
            this.drawItemBackground(index);
          }
        }
        this.drawItem(index);
      }
    }

    refreshCursor() {
      if (this._cursorAll) {
        super.refreshCursorForAll();
      } else if (super.index() >= 0) {
        const _index = super.index();
        const rect = this.itemRect(_index);
        if (this.items[_index]) {
          if (this.items[_index].type !== 'group') {
            this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
          }
        }
      } else {
        this.setCursorRect(0, 0, 0, 0);
      }
    }

    // #region 绘制函数
    drawItem(index) {
      const rect = this.itemLineRect(index);
      const _message = this.items[index];

      let _attaY = 0;
      if (Utils.RPGMAKER_NAME === "MZ") {
        _attaY = 5;
      }

      if (_message.type === 'group') {
        // 画分组
        if (commonStyleConfig.groupFontConfig) {
          this.contents.fontSize = commonStyleConfig.groupFontConfig.fontSize || 24;
          this.contents.textColor = commonStyleConfig.groupFontConfig.fontColor || '#E8E197';
          this.contents.fontBold = !!commonStyleConfig.groupFontConfig.fontWeight;
          this.contents.outlineWidth = commonStyleConfig.groupFontConfig.outlineWidth == '' ? commonStyleConfig.groupFontConfig.outlineWidth : 1;
          this.contents.outlineColor = commonStyleConfig.groupFontConfig.outlineColor || '#000000';
        }
        this.drawText(_message.group, rect.x, rect.y + (rect.height - this.lineHeight()) / 2 + 4, rect.width, commonStyleConfig.splitLoc || "center");
        this.resetFontSettings();
        if (commonStyleConfig.splitImage) {
          this.contents.blt(this._titleBitmap, 0, 0, this._titleBitmap.width, this._titleBitmap.height, rect.x, rect.y, rect.width, rect.height);
        }

        if (commonStyleConfig.splitLeftImage || commonStyleConfig.splitRightImage) {
          // 获取文本宽度
          let _textWidth = this.textWidth(_message.group);
          let _leftImgLoc = (rect.width - _textWidth) / 2 - this._leftBitmap.width + this.padding;
          let _rightImgLoc = (rect.width + _textWidth) / 2 + this.padding;

          this.contents.blt(this._leftBitmap, 0, 0, this._leftBitmap.width, this._leftBitmap.height, _leftImgLoc, rect.y);
          this.contents.blt(this._rightBitmap, 0, 0, this._rightBitmap.width, this._rightBitmap.height, _rightImgLoc, rect.y);
        }
      } else if (_message.type === 'message') {
        // 画消息

        // 画背景
        if (normalMessageConfig._itemBgBitmap) {
          this.contents.blt(this._bitmapBg, 0, 0, this._bitmapBg.width, this._bitmapBg.height, rect.x, rect.y, rect.width, rect.height);
        }

        // 画头像
        const _faceCfg = normalMessageConfig.faceImageConfig;

        if (_message.face && _faceCfg.visible) {
          if (_faceCfg.faceBackground) {
            this.contents.blt(this._bitmapFaceBg,
              0,
              0,
              this._bitmapFaceBg.width,
              this._bitmapFaceBg.height,
              rect.x + params.messagePadding,
              rect.y + (rect.height - params.messagePadding - (_faceCfg.height + _faceCfg.outlineWidth)) / 2 + _attaY,
              _faceCfg.width + _faceCfg.outlineWidth,
              _faceCfg.height + _faceCfg.outlineWidth,
            );
          } else {
            this.contents.drawRounded(
              rect.x + params.messagePadding,
              rect.y + (rect.height - params.messagePadding - (_faceCfg.height + 2 * _faceCfg.outlineWidth)) / 2 + _attaY,
              _faceCfg.width + 2 * _faceCfg.outlineWidth,
              _faceCfg.height + 2 * _faceCfg.outlineWidth,
              {
                radius: 10,
                fillColor: _faceCfg.fillColor || '#89671C',
                strokeColor: _faceCfg.outlineColor || '#E9C94E',
                lineWidth: 4
              }
            );
          }
          this.drawFace(
            _message.face,
            _message.faceIndex,
            rect.x + _faceCfg.outlineWidth + params.messagePadding,
            rect.y + (rect.height - params.messagePadding - (_faceCfg.height)) / 2 + _attaY,
            0, 0,
            _faceCfg.width,
            _faceCfg.height
          );
          this.resetTextColor();
        }

        // 画标题
        let _y = rect.y + this.itemPadding();
        let _faceWidth = _faceCfg && _faceCfg.visible ? (_faceCfg.width + _faceCfg.outlineWidth * 2) : 0;
        let _x = _faceWidth + this.itemPadding() * 2 + (commonStyleConfig.facePadding || 6);

        if (_message.title) {
          this.drawTextHk(_message.title, rect.x + _x, rect.y + this.itemPadding(), rect.width, () => {
            this.contents.fontSize = normalMessageConfig.titleFontConfig.fontSize || 30;
            this.contents.textColor = normalMessageConfig.titleFontConfig.fontColor || '#E8E197';
            this.contents.fontBold = !!normalMessageConfig.titleFontConfig.fontWeight;
            this.contents.outlineWidth = normalMessageConfig.titleFontConfig.outlineWidth == '' ? normalMessageConfig.titleFontConfig.outlineWidth : 1;
            this.contents.outlineColor = normalMessageConfig.titleFontConfig.outlineColor || '#000000';
          });
          this.resetTextColor();
          this.resetFontSettings();

          _y += this.lineHeight();
        }

        // 画内容
        this.resetFontSettings();
        for (let i = 0; i < _message.message.length; i++) {
          this.drawTextEx(_message.message[i], rect.x + _x, _y, rect.width, () => {
            this.contents.fontSize = normalMessageConfig.normalFontConfig.fontSize || 24;
            this.contents.textColor = normalMessageConfig.normalFontConfig.fontColor || '#FFFFFF';
            this.contents.fontBold = !!normalMessageConfig.normalFontConfig.fontWeight;
            this.contents.outlineWidth = normalMessageConfig.normalFontConfig.outlineWidth == '' ? normalMessageConfig.normalFontConfig.outlineWidth : 1;
            this.contents.outlineColor = normalMessageConfig.normalFontConfig.outlineColor || '#000000';
          });
          _y += this.textSizeHk(_message.message[i]).height;
          // _y += this.contents.fontSize + 2;
        }

      } else {
        const _msgTypeConfig = specialMessageConfig.find(i => i.messageName === _message.type);
        if (!_msgTypeConfig) {
          this.drawText('[NOT FOUND TYPE]', rect.x, rect.y + 20, rect.width, "center");
          return;
        }
        // 画消息

        // 画背景
        if (_msgTypeConfig.itemBgImage) {
          let _bitmapBg = ImageManager.loadBitmap('img/pictures/', _msgTypeConfig.itemBgImage);
          this.contents.blt(_bitmapBg, 0, 0, _bitmapBg.width, _bitmapBg.height, rect.x, rect.y, rect.width, rect.height);
        }

        // 画头像
        const _faceCfg = _msgTypeConfig.faceImageConfig;

        if (_message.face && _faceCfg.visible) {
          if (_faceCfg.faceBackground) {
            let _bitmapFaceBg = ImageManager.loadBitmap('img/pictures/', _faceCfg.faceBackground);
            _bitmapFaceBg.addLoadListener(() => {
              this.contents.blt(_bitmapFaceBg,
                0,
                0,
                _bitmapFaceBg.width,
                _bitmapFaceBg.height,
                rect.x + params.messagePadding,
                rect.y + (rect.height - params.messagePadding - (_faceCfg.height + _faceCfg.outlineWidth)) / 2 + _attaY,
                _faceCfg.width + _faceCfg.outlineWidth,
                _faceCfg.height + _faceCfg.outlineWidth,
              );
            });
          } else {
            this.contents.textColor = '#765D13';
            this.contents.drawRounded(
              rect.x + params.messagePadding,
              rect.y + (rect.height - params.messagePadding - (_faceCfg.height + 2 * _faceCfg.outlineWidth)) / 2 + _attaY,
              _faceCfg.width + 2 * _faceCfg.outlineWidth,
              _faceCfg.height + 2 * _faceCfg.outlineWidth,
              {
                radius: 10,
                fillColor: _faceCfg.fillColor || '#89671C',
                strokeColor: _faceCfg.outlineColor || '#E9C94E',
                lineWidth: 4
              }
            );
          }
          this.drawFace(
            _message.face,
            _message.faceIndex,
            rect.x + _faceCfg.outlineWidth + params.messagePadding,
            rect.y + (rect.height - params.messagePadding - (_faceCfg.height)) / 2 + _attaY,
            0, 0,
            _faceCfg.width,
            _faceCfg.height,
          );
          this.resetTextColor();
        }

        // 画标题
        let _y = rect.y + this.itemPadding();
        let _faceWidth = _faceCfg && _faceCfg.visible ? (_faceCfg.width + _faceCfg.outlineWidth * 2) : 0;
        let _x = _faceWidth + this.itemPadding() * 2 + (commonStyleConfig.facePadding || 6);

        if (_message.title) {
          this.drawTextHk(_message.title, rect.x + _x, rect.y + this.itemPadding(), rect.width, () => {
            this.contents.fontSize = _msgTypeConfig.titleFontConfig.fontSize || 30;
            this.contents.textColor = _msgTypeConfig.titleFontConfig.fontColor || '#E8E197';
            this.contents.fontBold = !!_msgTypeConfig.titleFontConfig.fontWeight;
            this.contents.outlineWidth = _msgTypeConfig.titleFontConfig.outlineWidth == '' ? _msgTypeConfig.titleFontConfig.outlineWidth : 1;
            this.contents.outlineColor = _msgTypeConfig.titleFontConfig.outlineColor || '#000000';
          });
          this.resetTextColor();
          this.resetFontSettings();

          _y += this.lineHeight();
        }

        // 画内容
        for (let i = 0; i < _message.message.length; i++) {
          this.drawTextHk(_message.message[i], rect.x + _x, _y, rect.width, () => {
            this.contents.fontSize = _msgTypeConfig.contentFontConfig.fontSize || 30;
            this.contents.textColor = _msgTypeConfig.contentFontConfig.fontColor || '#FFFFFF';
            this.contents.fontBold = !!_msgTypeConfig.contentFontConfig.fontWeight;
            this.contents.outlineWidth = _msgTypeConfig.contentFontConfig.outlineWidth == '' ? _msgTypeConfig.contentFontConfig.outlineWidth : 1;
            this.contents.outlineColor = _msgTypeConfig.contentFontConfig.outlineColor || '#000000';
          });
          _y += this.lineHeight();
        }
      }
    }
    // #endregion

    updatePadding() {
      this.padding = 0;
    }

    drawFace(faceName, faceIndex, x, y, width, height, drawWidth, drawHeight) {
      width = width || ImageManager.faceWidth || Window_Base._faceWidth;
      height = height || ImageManager.faceHeight || Window_Base._faceHeight;
      var bitmap = ImageManager.loadFace(faceName);
      var pw = ImageManager.faceWidth || Window_Base._faceWidth;
      var ph = ImageManager.faceHeight || Window_Base._faceHeight;
      var sw = Math.min(width, pw);
      var sh = Math.min(height, ph);
      var dx = Math.floor(x + Math.max(width - pw, 0) / 2);
      var dy = Math.floor(y + Math.max(height - ph, 0) / 2);
      var sx = faceIndex % 4 * pw + (pw - sw) / 2;
      var sy = Math.floor(faceIndex / 4) * ph + (ph - sh) / 2;
      this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy, drawWidth || sw, drawHeight || sh);
    }

    drawTextHk(text, x, y, width, setState) {
      let textState;
      if (Utils.RPGMAKER_NAME === "MZ") {
        this.resetFontSettings();
        if (setState) setState();
        textState = this.createTextState(text, x, y, width);
        this.processAllText(textState);
        return textState.outputWidth;
      } else {
        if (text) {
          textState = { index: 0, x: x, y: y, left: x };
          textState.text = this.convertEscapeCharacters(text);
          textState.height = this.calcTextHeight(textState, false);
          this.resetFontSettings();
          if (setState) setState();
          while (textState.index < textState.text.length) {
            this.processCharacter(textState);
          }
          return textState.x - x;
        } else {
          return 0;
        }
      }
    }
    textSizeHk(text, setState) {
      let textState;

      if (Utils.RPGMAKER_NAME === "MZ") {
        this.resetFontSettings();
        if (setState) setState();
        textState = this.createTextState(text, 0, 0, 0);
        textState.drawing = false;
        this.processAllText(textState);
        return { width: textState.outputWidth, height: textState.outputHeight };
      } else {
        textState = { index: 0, x: -10000, y: 0, left: 0, width: 0.01 };
        textState.text = this.convertEscapeCharacters(text);
        textState.height = this.calcTextHeight(textState, false);
        this.resetFontSettings();
        if (setState) setState();
        while (textState.index < textState.text.length) {
          this.processCharacter(textState);
        }
        return { width: textState.x, height: textState.height };
      }
    }
    createTextState(text, x, y, width) {
      if (Utils.RPGMAKER_NAME === "MZ") {
        const rtl = Utils.containsArabic(text);
        const textState = {};
        textState.text = this.convertEscapeCharacters(text);
        textState.index = 0;
        textState.x = rtl ? x + width : x;
        textState.y = y;
        textState.width = width;
        textState.height = this.calcTextHeight(textState);
        textState.startX = textState.x;
        textState.startY = textState.y;
        textState.rtl = rtl;
        textState.buffer = this.createTextBuffer(rtl);
        textState.drawing = true;
        textState.outputWidth = 0;
        textState.outputHeight = 0;
        return textState;
      } else {
        var textState = { index: 0, x: x, y: y, left: x };
        textState.text = this.convertEscapeCharacters(text);
        textState.height = this.calcTextHeight(textState, false);
        return textState;
      }
    }

    /** 光标下移 */
    cursorDown(wrap) {
      let index = this.index();
      const maxItems = this.maxItems();
      const maxCols = this.maxCols();
      if (index < maxItems - maxCols || (wrap && maxCols === 1)) {
        index = (index + maxCols) % maxItems;
      }
      if (this.items[index].type === 'group') {
        if (index < maxItems - maxCols || (wrap && maxCols === 1)) {
          index = (index + maxCols) % maxItems;
        }
      }
      if (Utils.RPGMAKER_NAME === "MZ") {
        this.smoothSelect(index);
      } else {
        this.select(index);
      }
    }

    /** 光标上移 */
    cursorUp(wrap) {
      let index = this.index();
      const maxItems = this.maxItems();
      const maxCols = this.maxCols();
      if (index >= maxCols || (wrap && maxCols === 1)) {
        index = (index - maxCols + maxItems) % maxItems;
      }
      if (this.items[index].type === 'group') {
        if (index >= maxCols || (wrap && maxCols === 1)) {
          index = (index - maxCols + maxItems) % maxItems;
        }
      }
      if (Utils.RPGMAKER_NAME === "MZ") {
        this.smoothSelect(index);
      } else {
        this.select(index);
      }
    }

    /** 获取某段列表项的高度 */
    getTopByIndex(startIndex = 0, endIndex) {
      let top = 0;
      for (let i = startIndex; i < endIndex; i++) {
        top += this.itemRects[i].height;
      }
      return top;
    }

    itemRect(index) {
      if (index < 0 || index >= this.itemRects.length) return new Rectangle(0, 0, 0, 0);
      const maxCols = this.maxCols();
      const itemWidth = this.itemWidth();
      let itemHeight = this.itemRects[index].height;
      const colSpacing = this.colSpacing();
      const rowSpacing = this.rowSpacing();
      const col = index % maxCols;
      const x = col * itemWidth + colSpacing / 2 - (this.scrollBaseX ? this.scrollBaseX() : this._scrollX);
      const y = this.getTopByIndex(0, index) + rowSpacing / 2 - (this.scrollBaseY ? this.scrollBaseY() : this._scrollY);
      const width = itemWidth - colSpacing;
      const height = itemHeight - rowSpacing;
      return new Rectangle(x, y, width, height);
    }

    windowWidth() {
      // @ts-ignore // MV Compatible
      return this._windowRect.width;
    }
    windowHeight() {
      // @ts-ignore // MV Compatible
      return this._windowRect.height;
    }
    colSpacing() {
      return 8;
    }
    rowSpacing() {
      return 4;
    }

  }

  // #endregion

  // #region 图像缓存

  // 缓存部分
  const SceneManager_onSceneCreate = SceneManager.onSceneCreate;
  SceneManager.onSceneCreate = function () {
    $gameHistoryMessage.canCache = [Scene_Title, Scene_Load].indexOf(this._previousClass) >= 0;
    if (this._scene && this._scene.constructor) {
      $gameHistoryMessage.currentScene = this._scene.constructor.name;
    }
    SceneManager_onSceneCreate.call(this, arguments);
  }

  const Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
  Scene_Map.prototype.onMapLoaded = function () {
    Scene_Map_onMapLoaded.apply(this, arguments);

    if ($gameHistoryMessage.canCache === undefined || $gameHistoryMessage.canCache) {
      if (params.commonStyleConfig.sceneImage) {
        ImageManager.loadBitmap('img/pictures/', params.commonStyleConfig.sceneImage);
      }
      if (params.commonStyleConfig.windowImage) {
        ImageManager.loadBitmap('img/pictures/', params.commonStyleConfig.windowImage);
      }
      if (commonStyleConfig.splitImage) {
        ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitImage);
      }
      if (commonStyleConfig.splitLeftImage) {
        ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitLeftImage);
      }
      if (commonStyleConfig.splitRightImage) {
        ImageManager.loadBitmap('img/pictures/', commonStyleConfig.splitRightImage);
      }

      if (normalMessageConfig.itemBgImage) {
        ImageManager.loadBitmap('img/pictures/', normalMessageConfig.itemBgImage);
      }
      for (let i = 0; i < specialMessageConfig.length; i++) {
        const _cfg = specialMessageConfig[i];
        if (_cfg.itemBgImage) {
          ImageManager.loadBitmap('img/pictures/', _cfg.itemBgImage);
        }
      }
      $gameHistoryMessage.canCache = false;
    }
  };

  // #endregion

  // #region 存储数据相关

  /** 默认创建参数 */
  const defaultCreatedParams = {
    messages: [],
    prevVisible: false,
  };
  const _DataManager_createGameObjects = DataManager.createGameObjects;
  DataManager.createGameObjects = function () {
    _DataManager_createGameObjects.call(this);
    $gameHistoryMessage = defaultCreatedParams;
  };
  const _DataManager_makeSaveContents = DataManager.makeSaveContents;
  DataManager.makeSaveContents = function () {
    const contents = _DataManager_makeSaveContents.call(this);
    contents.$gameHistoryMessage = $gameHistoryMessage;
    return contents;
  };
  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function (contents) {
    _DataManager_extractSaveContents.call(this, contents);
    if (!contents.$gameHistoryMessage) return;
    $gameHistoryMessage = contents.$gameHistoryMessage;
  };

  // #endregion

  // #region 系统修改

  Window_Base.prototype.processNormalCharacter = function (textState) {
    var c = textState.text[textState.index++];
    var w = textState.width != undefined ? textState.width : this.textWidth(c);
    this.contents.drawText(c, textState.x, textState.y, w * 2, textState.height);
    textState.x += w;
  }

  let _prevChoices = [];

  const Game_Interpreter_command102 = Game_Interpreter.prototype.command102;
  Game_Interpreter.prototype.command102 = function () {
    if ($gameMessage.isBusy()) {
      return false;
    }
    if (this.currentCommand().parameters[0]) {
      _prevChoices = this.currentCommand().parameters[0];
    }

    return Game_Interpreter_command102.call(this, params);
  }

  const Game_Message_onChoice = Game_Message.prototype.onChoice;
  Game_Message.prototype.onChoice = function (n) {
    Game_Message_onChoice.call(this, n);

    if (_prevChoices && _prevChoices.length > 0 && _prevChoices[n]) {
      HistoryMessageUtils.addItem({
        message: [params.choiceFormat.replace('{0}', _prevChoices[n])],
      });
    }
  }

  const Window_NumberInput_processOk = Window_NumberInput.prototype.processOk;
  Window_NumberInput.prototype.processOk = function () {
    Window_NumberInput_processOk.call(this);

    HistoryMessageUtils.addItem({
      message: [params.inputNumberFormat.replace('{0}', this._number + '')],
    });
  }

  const Window_EventItem_onOk = Window_EventItem.prototype.onOk;
  Window_EventItem.prototype.onOk = function() {
    var item = this.item();
    var itemId = item ? item.id : 0;

    HistoryMessageUtils.addItem({
      message: [params.choiceFormat.replace('{0}', $dataItems[itemId].name)],
    });

    Window_EventItem_onOk.call(this);
  }

  const Game_Interpreter_command101 = Game_Interpreter.prototype.command101;
  Game_Interpreter.prototype.command101 = function () {
    if ($gameMessage.isBusy()) {
      return false;
    }
    let _parmas = JSON.parse(JSON.stringify(this.currentCommand().parameters));
    let _eventDetailIndex = this._index;
    let _strlines = [];
    _eventDetailIndex++;
    let _actionIndex = 0;
    let _name = _parmas[4] || '';

    while (this._list[_eventDetailIndex].code === 401) {
      let _message = this._list[_eventDetailIndex].parameters[0];

      if (Utils.RPGMAKER_NAME === "MV" && _actionIndex == 0) {
        _message = _message.replace(new RegExp(params.groupConfig.actorNameRegex, 'g'), (a, b, c) => {
          _name = b;
          return '';
        });
      }

      _strlines.push(_message);
      _eventDetailIndex++;
      _actionIndex++;
    }

    switch (this._list[_eventDetailIndex].code) {
      case 102:  // Show Choices
        if (this._list[_eventDetailIndex].parameters) {
          _prevChoices = this._list[_eventDetailIndex].parameters[0];
        }
        break;
    }

    Game_Interpreter_command101.call(this, _parmas);

    HistoryMessageUtils.addItem({
      message: _strlines,
      face: _parmas[0],
      faceIndex: _parmas[1],
      title: _name,
      date: eval('(function(){' + (HistoryMessageUtils.transformNote(groupConfig.groupGetFunction) || 'return \'\';') + '})()'),
    });
  }

  Bitmap.prototype.drawRoundedRect = function (x, y, width, height, radius) {
    const context = this.context;

    context.beginPath();
    context.moveTo(x + radius, y);
    context.lineTo(x + width - radius, y);
    context.quadraticCurveTo(x + width, y, x + width, y + radius);
    context.lineTo(x + width, y + height - radius);
    context.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    context.lineTo(x + radius, y + height);
    context.quadraticCurveTo(x, y + height, x, y + height - radius);
    context.lineTo(x, y + radius);
    context.quadraticCurveTo(x, y, x + radius, y);
    context.closePath();
  }

  Bitmap.prototype.drawRounded = function (x, y, width, height, {
    radius,
    fillColor,
    strokeColor,
    lineWidth
  }) {
    const context = this.context;
    context.save();

    context.fillStyle = fillColor || '#FFFFFF';
    this.drawRoundedRect(x, y, width, height, radius);
    context.fill();

    if (strokeColor && lineWidth) {
      context.strokeStyle = strokeColor;
      context.lineWidth = lineWidth;
      this.drawRoundedRect(x, y, width, height, radius);
      context.stroke();
      context.restore();
    }

    this._baseTexture.update();
  }



  // #region 设定快捷键

  setTimeout(() => {
    if (params.showHotKey) {
      Input.keyMapper[params.showHotKey] = 'hkb-history';
    }
  }, 1000);

  const Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    Scene_Map_update.call(this);

    if (!$gameHistoryMessage.prevVisible && Input.isTriggered('hkb-history')) {
      HistoryMessageUtils.show();
    }

    if (params.bindShowSwitch) {
      if (!$gameHistoryMessage.prevVisible && $gameSwitches.value(params.bindShowSwitch)) {
        HistoryMessageUtils.show();
      } else if ($gameHistoryMessage.prevVisible && !$gameSwitches.value(params.bindShowSwitch)) {
        HistoryMessageUtils.hide();
      }
    }
  };
  // #endregion

  // #region 工具类

  /** Hakubox工具类 */
  class HistoryMessageUtils {
    /**
     * 添加一条记录，注意消息是数组
     * @param {Object} item 记录项
     * @param {string} item.message 信息
     * @param {string} [item.type='message'] 消息类型
     * @param {string} [item.face] 头像文件
     * @param {number} [item.faceIndex] 头像序号
     * @param {string} [item.title] 标题/姓名
     */
    static addItem(item) {
      let obj = {
        date: eval('(function(){' + (HistoryMessageUtils.transformNote(groupConfig.groupGetFunction) || 'return \'\';') + '})()'),
      };
      if (!item.message) throw new Error('not found message');
      obj.message = Array.isArray(item.message) ? item.message : item.message.split('\n');
      if (!item.type) item.type = 'message';
      obj.type = item.type;
      if (item.face && item.faceIndex) {
        obj.face = item.face;
        obj.faceIndex = item.faceIndex;
      }
      if (item.title) obj.title = item.title;

      if (params.ignoreRegex) {
        if (RegExp(params.ignoreRegex, 'g').test(obj.message.join('\n'))) {
          return;
        }
      }

      $gameHistoryMessage.messages.push(obj);

      // 超出部分删除
      if ($gameHistoryMessage.messages > params.maxMessageCount) {
        $gameHistoryMessage.messages.splice(0, 1);
      }
    }
    /** 显示历史记录窗口 */
    static show() {
      $gameHistoryMessage.prevVisible = true;
      if (params.bindShowSwitch) $gameSwitches._data[params.bindShowSwitch] = true;
      SceneManager.push(Scene_HistoryMessage);
    }
    /** 隐藏历史记录窗口 */
    static hide() {
      $gameHistoryMessage.prevVisible = false;
      if (params.bindShowSwitch) $gameSwitches._data[params.bindShowSwitch] = false;
      SceneManager.pop();
    }
    /** 节流函数 */
    static throttle(func, wait) {
      let lastCall = 0;
      let timeout;

      return function() {
        const now = Date.now();
        const context = this;
        const _arguments = arguments;

        if (now - lastCall >= wait) {
          if (timeout) {
            clearTimeout(timeout);
            timeout = null;
          }
          lastCall = now; 
          func.apply(context, _arguments);
        } else {
          if (!timeout) {
            timeout = setTimeout(function () {
              lastCall = Date.now();
              func.apply(context, _arguments);
            }, wait - (now - lastCall));
          }
        }
      };
    }
    /** 浅层合并对象 */
    static mergeObj(obj, props) {
      const _props = Object.entries(props);
      for (let i = 0; i < _props.length; i++) {
        obj[_props[i][0]] = _props[i][1];
      }
      return obj;
    }
    /** 转换note参数 */
    static transformNote(str) {
      if (!str) return '';
      let reStr = str.replace(/\\n/g, "\n").replace(/\\"/g, '"');
      if (reStr[0] === '"' || reStr[reStr.length - 1] === '"') {
        reStr = reStr.substring(1, reStr.length - 1);
      }
      return reStr;
    }
  }
  window.HistoryMessageUtils = HistoryMessageUtils;
  // #endregion

  if (Utils.RPGMAKER_NAME === "MZ") {
    PluginManager.registerCommand(Hakubox_PluginName, 'show', (args) => {
      return HistoryMessageUtils.show();
    });
    PluginManager.registerCommand(Hakubox_PluginName, 'hide', (args) => {
      return HistoryMessageUtils.hide();
    });
    PluginManager.registerCommand(Hakubox_PluginName, 'addItem', (args) => {
      const item = {
        type: args.type || 'message',
        message: args.message.split('\n'),
      };
      if (args.title) item.title = args.title;
      if (args.face && args.faceIndex) {
        item.face = args.face;
        item.faceIndex = +args.faceIndex;
      }
      return HistoryMessageUtils.addItem(item);
    });

  }
})();