//=============================================================================
//  Keke_NearStopEvent - ニアストップ・イベント
//  バージョン: 1.1.6
//=============================================================================
// Copyright (c) 2022 ケケー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MV MZ
 * @plugindesc プレイヤーと近づいたイベントを立ち止まらせる
 * @author ケケー
 * @url https://kekeelabo.com
 * 
 * @help
 * 【ver.1.1.6】
 * プレイヤーと近づいたイベントを自動で立ち止まらせる。もしくは近寄らせる
 * 離れると元に戻る
 * 近づくと発動系ギミックにも活用可能
 * 
 * 
 * 【機能1】イベント別にニアストップを設定
 * => イベントコマンドの『注釈』を使う
 * ◎注釈を置く位置
 *  「現在のページの一番上」か「1ページ目の一番上」
 * 　まず現在のページのがあればそれを使い、なければ1ページ目のを使う
 * 　注釈は複数つなげることも可能
 * ◎置いた注釈の中に
 * <nearStop> {
 * scope:   break:   time:   noStop: 
 * through:   push:   dir:   come:  speed: 
 * freq:   common1:   common2:  
 * self:   noRegion:   doRegion:   
 * }
 * ★パラメータの説明
 * scope:    ストップ発動距離。1～。0 ならストップ無効(なお範囲は円形)
 * break:    ストップ解除距離。1～。-1 なら距離解除しない
 * time:     ストップ解除時間。1～。0 なら時間解除しない
 * noStop:   ストップを封印するか。1-有効 0-無効
 * through:  ストップ中はすり抜け。1-有効  0-無効
 * push:     押し込み時間。1～。0 ならすぐすり抜け
 * dir:      ストップ中はこっち向く。1-有効  0-無効
 * come:     ストップ中は近づく。1-有効  0-無効
 * speed:    移動速度変更。0 なら変更しない
 * freq:     移動頻度変更。0 なら変更しない
 * fast:     即座に近づく。1-有効  0-無効
 * common1:  発動時コモンイベント。IDで指定。0 ならコモン実行しない
 * common2:  解除時コモンイベント。IDで指定。0 ならコモン実行しない
 * self:     セルフスイッチ。a～d。a ならセルフスイッチ A をオン
 * noRegion: 無効リージョン。IDで指定。0 なら無効リージョンなし
 * doRegion: 有効リージョン。IDで指定。0 なら有効リージョンなし
 * ★例
 * <nearStop> { scope:5 }
 * 　プレイヤーが5マス以内に入ると停止
 * <nearStop> { scope:0 }
 * 　ニアストップ無効
 * <nearStop> { scope:5  come:1  break:10 }
 * 　プレイヤーが5マス以内に入ると近づいてくる。10マス以上離れると解除
 * <nearStop> {
 * scope:10  come:1  speed:6  freq:5  fast:1
 * common1:20  doRegion:11  
 * }
 * 　(店主用)
 * 　プレイヤーがイベントの10マス以内に入ると、コモンイペント20を実行
 * 　移動速度を6、移動頻度を5にして即座に近づいてくる
 * 　プレイヤーがリージョン11の上にいる時のみ有効
 * <nearStop> {
 * scope:10  come:1  speed:6  freq:5  fast:1
 * common1:20  common2:21  self:A  noRegion:10  
 * }
 * 　(追いかけてくる敵用)
 * 　プレイヤーがイベントの10マス以内に入ると、セルフスイッチAをオン
 * 　コモンイペント20を実行し、移動速度を6、移動頻度を5にして即座に近づいてくる
 * 　プレイヤーがリージョン10の上にいる時は無効
 * 　射程外に離れるかリージョン10に入ると解除され、セルフスイッチAをオフ
 * 　コモンイベント21を実行する
 * 
 * 【機能2】マップ上の全てのイベントにニアストップを設定
 * => マップのメモ欄に
 * <nearStop> {  }
 * 書式は上のイベント個別の場合と同じ
 * 
 * 
 * 【機能3】ニアストップを一時的に無効(MZ限定)
 * => プラグインコマンド → ニアストップ無効
 * ◎有効に戻す時は　プラグインコマンド → ニアストップ有効
 *
 * 
 * 【応用】「ストップ」「近づく」以外の動きを設定
 * ◎セルフスイッチを使う
 * [1]注釈での設定で
 * 　<nearStop> { self:A  noStop:1 }
 * 　とセルフスイッチを起動させ、ストップを封印する
 * [2]セルフスイッチを条件としたページを作り、
 * 　移動タイプ『カスタム』で好きな動きを設定する
 * 
 * 
 * 【付属機能】真ランダム移動
 * デフォルトのイベントのランダム移動はそれほどランダムでないので、
 * それを真のランダムにできる
 * ★設定のしかた
 * ・イベントを移動タイプ「ランダム」にした上で、
 * ・注釈に
 * <trueRondomMove>
 * 注釈を置く位置は上のニアストップの場合と同じ
 *　
 * 
 * 【注意】プレイヤーの進路妨害について
 * ◎「ストップ中はすり抜け」を無効にすると、
 * 　プレイヤーの進路を塞いでしまうおそれがある
 * ◎その場合は「無効リージョンID」「有効リージョンID」を使い、
 *　イベントが危険な位置で立ち止まらないようにしよう
 *
 *
 * ● 利用規約 ●
 * MITライセンスのもと、自由に使ってくれて大丈夫です
 * 
 * 
 * 
 * Stops the independent walk of the event the player approaches
 * (this is called a near stop)
 * When the player moves away, it starts moving again
 * 
 * 
 * [Function 1] Set a near stop for each event.
 * => Use "Annotation" of Event Command
 * ◎ Position to place annotations
 *   "Top of current page" or "Top of page 1
 *   If there is one on the current page, use it first;
 *   if not, use the one on the first page.
 *   Annotations can be connected to each other.
 * ◎ In the annotations you have placed
 * <nearStop> {
 * scope: break: time: noStop: 
 * through: push: dir: come: speed: 
 * freq: common1: common2:  
 * self: noRegion: doRegion:   
 * }
 * ★ Description of parameter
 * scope: distance at which the stop is triggered, from 1 to 0,
 *   otherwise the stop is disabled (and the range is circular).
 * break: Distance to release the stop,
 *   from 1 to 1. -If 1, the distance is not released.
 * time: time to release the stop, from 1 to 0, the time is not released.
 * noStop: Stop is sealed or not. 1-enable, 0-disable.
 * through: The time to slip through the stop. 1-Enable 0-Disable
 * push: Time to push the stop. 1-0, it will slip through immediately.
 * dir: to turn around during stop. 1-Enabled 0-Not enabled
 * come: Move closer during stop. 1-Enable 0-Disable
 * speed: change movement speed, 0 means no change
 * freq: change movement frequency. 0 means no change
 * fast: React immediately. 1-enabled 0-disabled
 * common1: common event when triggered,
 *   specified by ID. 0 means no common event
 * common2: common event on release, specified by ID,
 *   if 0, do not execute common.
 * self: self-switch, a-d, if a, self-switch A is on
 * noRegion: disabled region, by ID, 0 means no disabled region
 * doRegion: Valid region, by ID, 0 means no valid region
 * ★Example)
 * <nearStop> { scope:5 }
 *   stop when a player enters within 5 squares
 * <nearStop> { scope:0 }
 *   nearStop disabled
 * <nearStop> { scope:5  come:1  break:10 }
 *   comes closer when player comes within 5 squares;
 *   disables when player moves 10 squares or more away
 * <nearStop> {
 * scope:10  come:1  speed:6  freq:5  fast:1
 * common1:20  doRegion:11  
 * }
 *   (for shopkeeper)
 *   Execute common-imperative 20 when the player gets
 *   within 10 squares of the event.
 *   Approaches with a movement speed of 6 and a movement frequency of 5
 *   and immediately 
 *   Effective only when player is on top of region 11.
 * <nearStop> {
 * scope:10  come:1  speed:6  freq:5  fast:1
 * common1:20  common2:21  self:A  noRegion:10  
 * }
 *   (for enemies chasing after you)
 *   When the player gets within 10 squares of the event, self switch A on
 *   Executes commonIpent 20 and approaches with a movement speed
 *   of 6 and a movement frequency of 5 and immediately 
 *   Not valid when player is above Region 10
 *   Disengaged when player moves out of range or enters region 10,
 *   self-switch A is turned off
 *   Execute Common Event 21.
 * 
 * [Function 2] Set near stops for all events on the map.
 * => In the notes field of the map.
 * <nearStop> { }
 *   Format is the same as for individual events above
 * 
 * 
 * [Function 3] Temporarily disable near stop (MZ only)
 * => plugin command → disable near stop
 * ◎ To return to enable: Plug-in command → Near stop enable
 *
 * 
 * [Application] Set up movements other than "stop" and "near stop".
 * ◎ Use self-switch
 * [1] In the setting in the annotation
 *   <nearStop> { self:A noStop:1 }
 *   and activate self-switch to seal the stop
 * [2]Create a page with self-switch as a condition, and
 *   Set the movement type "custom" to any movement you like.
 * 
 * 
 * [Included function] True random movement
 * Since the default event random movement is not so random.
 * Can make it true random.
 * ★ How to set up
 * ・Set the event to move type "Random" and then.
 * ・With the annotation.
 * <trueRondomMove>
 *   The position to place the annotation is
 *   the same as for the near stop above
 *　
 * 
 * [Caution] About obstructing the path of a player.
 * ◎ If you disable "slip through during stop",
 *   you may block the player's path.
 * ◎ In such a case, use "Invalid Region ID" and "Valid Region ID" and
 *   Try not to stop at a position where the event is dangerous!
 *
 * 
 * ● Terms of Use ●
 * You are free to use it under the MIT license!
 *
 *
 *
 * @param ストップ条件
 * 
 * @param 自動ニアストップ
 * @parent ストップ条件
 * @desc ゲーム中の全ての自立歩行するイベントに、自動でニアストップを適用する
 * @type boolean
 * @default true
 * 
 * @param ストップ距離
 * @parent ストップ条件
 * @desc ストップを発動する距離。2.5 なら プレイヤーとの距離が 2.5マス 以内(円形範囲) で発動。空欄ならストップ無効
 * @default 2
 * 
 * @param ストップ解除距離
 * @parent ストップ条件
 * @desc ストップを解除する距離。5 なら プレイヤーとの距離が 5マス 以上で解除。-1 なら距離解除しない。空欄なら発動距離と同じ
 * @default 
 * 
 * @param ストップ時間
 * @parent ストップ条件
 * @desc ストップを解除するまでの時間。50 なら 50フレーム 経過後に解除。空欄なら時間解除しない
 * @default
 * 
 * @param ストップ中の挙動
 * 
 * @param ストップ中はすり抜け
 * @parent ストップ中の挙動
 * @desc ニアストップ中はプレイヤーからすり抜け可能にする。プレイヤー以外はすり抜けない
 * @type boolean
 * @default true
 * 
 * @param …押し込み時間
 * @parent ストップ中の挙動
 * @desc 一定時間押し込むことですり抜けるようにする。60 なら 60フレーム 押し込んだらすり抜け。空欄ならすぐすり抜け
 * @type number
 * @default 60
 * 
 * @param ストップ中はこっち向く
 * @parent ストップ中の挙動
 * @desc ニアストップ中は常にプレイヤーの方を向くようにする
 * @type boolean
 * @default true
 * 
 * @param ストップ中は近づく
 * @parent ストップ中の挙動
 * @desc ニアストップ時、停止ではなくプレイヤーに近づいてくるようになる
 * @type boolean
 * @default false
 * 
 * @param …移動速度変更
 * @parent ストップ中の挙動
 * @desc 近づく時、移動速度を変更する。5 なら移動速度 5 に、+1 なら元の移動速度に +1 する。空欄なら変更しない
 * @default 
 * 
 * @param …移動頻度変更
 * @parent ストップ中の挙動
 * @desc 近づく時、移動頻度を変更する。5 なら移動頻度 5 に、+1 なら元の移動頻度に +1 する。空欄なら変更しない
 * @default 
 * 
 * @param …即座に近づく
 * @parent ストップ中の挙動
 * @desc 近づく時、即座に近づくモードに切り替える。若干不自然な動きになることもあるが、素早い反応を優先したい場合に
 * @type boolean
 * @default 
 * 
 * @param コモン/スイッチ
 * 
 * @param 発動時コモンイベント
 * @parent コモン/スイッチ
 * @desc ニアストップ発動時にイベントに実行させるコモンイベント
 * @type common_event
 * @default 
 * 
 * @param 解除時コモンイベント
 * @parent コモン/スイッチ
 * @desc ニアストップ解除時にイベントに実行させるコモンイベント
 * @type common_event
 * @default  
 * 
 * @param セルフスイッチ
 * @parent コモン/スイッチ
 * @desc ニアストップ中にオンにするセルフスイッチ
 * @type select
 * @option A
 * @option B
 * @option C
 * @option D
 * @default  
 * 
 * @param 無効設定
 * 
 * @param 無効の移動タイプ
 * @parent 無効設定
 * @desc 自動ニアストップを適用しない移動タイプを設定できる
 * @type struct<noMoveType>
 * @default 
 * 
 * @param 無効リージョンID
 * @parent 無効設定
 * @desc ニアストップを無効にするリージョンのID。プレイヤーがこのリージョン上にいる時はストップしない
 * @type number
 * @default
 * 
 * @param 有効リージョンID
 * @parent 無効設定
 * @desc ニアストップを有効にするリージョンのID。プレイヤーがこのリージョン上にいる時のみストップする
 * @type number
 * @default
 * 
 *
 *
 * @command ニアストップ有効
 * @desc ニアストップを有効にする
 * 
 * @command ニアストップ無効
 * @desc ニアストップを無効にする
 */



//================================================== 
/*~struct~noMoveType:
//==================================================
 *
 * @param ランダム-無効
 * @desc 移動タイプが『ランダム』のイベントには、自動ニアストップ無効にする
 * @type boolean
 * @default 
 * 
 * @param 近づく-無効
 * @desc 移動タイプが『近づく』のイベントには、自動ニアストップ無効にする
 * @type boolean
 * @default 
 * 
 * @param カスタム-無効
 * @desc 移動タイプが『カスタム』のイベントには、自動ニアストップ無効にする
 * @type boolean
 * @default 
 */
 
 
 
(() => {
    //- プラグイン名
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
    
    
    //- ツクールMZか
    function isMz() {
        return typeof(ColorManager) != "undefined";
    };


    //==================================================
    //--  パラメータ受け取り
    //==================================================
    
    //- 真偽化
    function toBoolean(str) {
        if (!str) { return false; }
        const str2 = str.toString().toLowerCase();
        if (str2 == "true" || str2 == "on") { return true; }
        if (str2 == "false" || str2 == "off") { return false; }
        return Number(str);
    };

    const parameters = PluginManager.parameters(pluginName);
    
    //- ストップ条件
    const keke_autoNearStop = toBoolean(parameters["自動ニアストップ"]);
    const keke_stopScope = Number(parameters["ストップ距離"]);
    const keke_stopBreak = Number(parameters["ストップ解除距離"]);
    const keke_stopTime = Number(parameters["ストップ時間"]);

    //- ストップ時の挙動
    const keke_throughInStop = toBoolean(parameters["ストップ中はすり抜け"]);
    const keke_pushTime = Number(parameters["…押し込み時間"]);
    const keke_dirInStop = toBoolean(parameters["ストップ中はこっち向く"]);
    const keke_comeInStop = toBoolean(parameters["ストップ中は近づく"]);
    const keke_changeMoveSpeed = parameters["…移動速度変更"];
    const keke_changeMoveFreq = parameters["…移動頻度変更"];
    const keke_comeFast = parameters["…即座に近づく"];

    //- コモン/スイッチ
    const keke_runCommonEvent = parameters["発動時コモンイベント"];
    const keke_brkCommonEvent = parameters["解除時コモンイベント"];
    const keke_selfSwitch = parameters["セルフスイッチ"].toUpperCase();

    //- 無効設定
    const keke_noMoveType = parameters["無効の移動タイプ"] ? JSON.parse(parameters["無効の移動タイプ"]) : null;
    if (keke_noMoveType) { Object.keys(keke_noMoveType).forEach(k => { keke_noMoveType[k] = toBoolean(keke_noMoveType[k]) }); }
    const keke_noRegionId = Number(parameters["無効リージョンID"]);
    const keke_doRegionId = Number(parameters["有効リージョンID"]);
    
    
    
    //==================================================
    //--  プラグインコマンド
    //==================================================
    
    if (isMz()) {

    //- ニアストップ有効
    PluginManager.registerCommand(pluginName, "ニアストップ有効", args => {
        $gameMap._noNearStopKe = null;
    });

    
    //- ニアストップ無効
    PluginManager.registerCommand(pluginName, "ニアストップ無効", args => {
        $gameMap._noNearStopKe = true;
    });

    }

    
    
    //==================================================
    //--  ニアストップの取得
    //==================================================

    //- ゲームイベント/更新(コア追加)
    const _Game_Event_update = Game_Event.prototype.update;
    Game_Event.prototype.update = function() {
        // ニアストップの処理
        processNearStop(this);
        _Game_Event_update.apply(this);
    };


    //- ニアストップの処理
    function processNearStop(event) {
        //if ($gameMap.isEventRunning()) { return false; }
        if ($gameMap._noNearStopKe) { return false; }
        // ページ切り替わったら初期化
        initAtPageChange(event);
        // ニアストップパラムスの取得
        getNearStopParams(event);
        // ニアストップの実行
        executeNearStop(event);
    };


    //- ページが切り替わったら初期化
    function initAtPageChange(event) {
        // ページが切り替わったら
        if (event._currentPageKeNast != event._pageIndex) {
            // パラムスを初期化
            event._nearStopParamsKe = null;
            event._inNearStopKe = null;
        }
        // 現在のページを保存
        event._currentPageKeNast = event._pageIndex;
    };


    //- ニアストップパラムスの取得
    function getNearStopParams(event) {
        if (event._nearStopParamsKe) { return; }
        let params = null;
        // 注釈からのニアストップ判定
        params = chechNearStopByComment(event);
        if (params == "none") { return; }
        if (params != null && needsNearStop(params, event)) {
            event._nearStopParamsKe = params;
            return
        }
        // マップ注釈からのニアストップ判定
        params = chechNearStopByMapMemo(event);
        if (params == "none") { return; }
        if (params != null && needsNearStop(params, event)) {
            event._nearStopParamsKe = params;
            return;
        }
        // 自動ニアストップ判定
        params = chechNearStopAuto(event);
        if (params != null && needsNearStop(params, event)) {
            event._nearStopParamsKe = params || {};
            return;
        }
    };


    //- ニアストップが必要か
    function needsNearStop(params, event) {
        // 移動タイプが「固定」以外なら true
        if (event._moveType) { return true; }
        // コモンイベントがあれば true
        if (params.commonId1 || params.commonId2) { return true; }
        return false;
    };


    //- 注釈からのニアストップ判定
    function chechNearStopByComment(event) {
        // 最初の注釈の取得
        const comment = getFirstComment(event);
        if (!comment) { return null; }
        return getParameter(comment);
    };


    //- マップメモからのニアストップ判定
    function chechNearStopByMapMemo(event) {
        // メタ取得
        const note = $dataMap.note;
        if (!note) { return null; }
        return getParameter(note);
    };


    //- 自動ニアストップ判定
    function chechNearStopAuto(event) {
        // 無効の移動タイプならリターン
        if (isNoMoveType(event._moveType)) { return false;}
        if (!keke_autoNearStop) { return null; }
        // 距離がなければ無効
        const scope = keke_stopScope;
        if (scope <= 0) { return null; }
        return getParameter();
    };

    
    //- パラメータの取得
    function getParameter(note = "") {
        // ノートから設定部分を検索
        let desc = "";
        if (note) {
            let match = note.match((/<nearStop>\s*\n*\{([^\}]+)\}/i));
            if (!match || !match[1]) { return null; }
            desc = match[1];
        }
        // デフォルトのパラメータを取得
        const p = {};
        p.scope = keke_stopScope;
        p.break = keke_stopBreak;
        p.time = keke_stopTime;
        p.noStop = false;
        p.through = keke_throughInStop;
        p.dir = keke_dirInStop;
        p.come = keke_comeInStop;
        p.moveSpeed = keke_changeMoveSpeed;
        p.moveFreq = keke_changeMoveFreq;
        p.comeFast = keke_comeFast;
        p.commonId1 = keke_runCommonEvent;
        p.commonId2 = keke_brkCommonEvent;
        p.selfSwitch = keke_selfSwitch;
        p.noRegionId = keke_noRegionId;
        p.doRegionId = keke_doRegionId;
        // パラメータの検索
        if (desc) {
            searchParameter(desc, p);
        }
        return p;
    };


    //- パラメータの検索
    function searchParameter(desc, p) {
        // 距離
        match = desc.match(/scope:\s*(\-*\d+\.*\d*)/i);
        if (match) {
            p.scope = Number(match[1]);
            // 距離が 0 ならニアストップ無効
            if (p.scope == 0) { return "none"};
        }
        // 解除距離
        match = desc.match(/break:\s*(\-*\d+\.*\d*)/i);
        if (match) {
            p.break = Number(match[1]);
        }
        // 時間
        match = desc.match(/time:\s*(\d+)/i);
        if (match) {
            p.time = Number(match[1]);
        }
        // ストップ封印
        match = desc.match(/noStop:\s*(\d+)/i);
        if (match) {
            p.noStop = Number(match[1]);
        }
        // すり抜け
        match = desc.match(/through:\s*(\d+)/i);
        if (match) {
            p.through = Number(match[1]);
        }
        // こっち向く
        match = desc.match(/dir:\s*(\d+)/i);
        if (match) {
            p.dir = Number(match[1]);
        }
        // 近づく
        match = desc.match(/come:\s*(\d+)/i);
        if (match) {
            p.come = Number(match[1]);
        }
        // 移動速度変更
        match = desc.match(/speed:\s*(\+*\-*\d+\.*\d*)/i);
        if (match) {
            p.moveSpeed = match[1];
        }
        // 移動頻度変更
        match = desc.match(/freq:\s*(\+*\-*\d+\.*\d*)/i);
        if (match) {
            p.moveFreq = match[1];
        }
        // 即座に近づく
        match = desc.match(/fast:\s*(\d+)/i);
        if (match) {
            p.comeFast = Number(match[1]);
        }
        // 発動時コモンイベント
        match = desc.match(/common1:\s*(\d+)/i);
        if (match) {
            p.commonId1 = Number(match[1]);
        }
        // 解除時コモンイベント
        match = desc.match(/common2:\s*(\d+)/i);
        if (match) {
            p.commonId2 = Number(match[1]);
        }
        // セルフスイッチ
        match = desc.match(/self:\s*(\w+)/i);
        if (match) {
            p.selfSwitch = match[1].toUpperCase();
        }
        // 無効リージョン
        match = desc.match(/noRegion:\s*(\d+)/i);
        if (match) {
            p.noRegionId = Number(match[1]);
        }
        // 有効リージョン
        match = desc.match(/doRegion:\s*(\d+)/i);
        if (match) {
            p.doRegionId = Number(match[1]);
        }
    };


    //- 無効の移動タイプか
    function isNoMoveType(moveType) {
        const noMoveType = keke_noMoveType;
        if (!noMoveType) { return; }
        if (noMoveType["ランダム-無効"] && moveType == 1) {
            return true;
        }
        if (noMoveType["近づく-無効"] && moveType == 2) {
            return true;
        }
        if (noMoveType["カスタム-無効"] && moveType == 3) {
            return true;
        }
        return false;
    };



    //==================================================
    //--  ニアストップの実行
    //==================================================

    // イベント押し込み済みフラグ
    let pushedNearStop = null;
    // タッチ移動中フラグ
    let inTouchMove = null;

    //- ゲームイベント/自立移動の更新(コア追加)
    const _Game_Event_updateSelfMovement = Game_Event.prototype.updateSelfMovement;
    Game_Event.prototype.updateSelfMovement = function() {
        // ストップ状態ならリターン
        if (isStop(this)) { return; }
        _Game_Event_updateSelfMovement.apply(this);
    };


    //- ニアストップの実行
    function executeNearStop(event) {
        if (!event._nearStopParamsKe) { return; }
        if (!Object.keys(event._nearStopParamsKe).length) { return; }
        const p = event._nearStopParamsKe;
        // 即座に近づく判定
        if (event.isMoving() && !(p.come && p.comeFast)) { return; }
        // パラメータを取り出す
        const scope = p.scope;
        const break_ = p.break;
        const time = p.time;
        const noStop = p.noStop;
        const through = p.through;
        const dir = p.dir;
        const come = p.come;
        const moveSpeed = p.moveSpeed;
        const moveFreq = p.moveFreq;
        const commonId1 = p.commonId1;
        const commonId2 = p.commonId2;
        const selfSwitch = p.selfSwitch;
        const noRegionId = p.noRegionId;
        const doRegionId = p.doRegionId;
        // リージョンを取得
        const regionId = $gameMap.regionId(Math.round($gamePlayer._realX), Math.round($gamePlayer._realY));
        // 無効リージョンの上か
        const isNoRegion = noRegionId && noRegionId == regionId ? true : false;
        // 有効リージョンの上でないか
        const noDoRegion = doRegionId && doRegionId != regionId ? true : false;
        // イベントとプレイヤーの距離を算出
        const distance = charaDistance(event, $gamePlayer);
        // 発動チェック
        const running = checkRunning(event, distance, scope, isNoRegion, noDoRegion);
        // 解除チェック
        const breaker = checkBreaker(event, running, distance, break_, time, isNoRegion, noDoRegion);
        // ストップするか
        let stop = running && !breaker;
        // ストップ中の処理
        if (stop) {
            // ストップ発動時の処理
            if (!inNearStop(event)) {
                // ストップ中データに本来のパラメータを保存
                event._inNearStopKe = { moveTypeOri:event._moveType, moveSpeedOri:event._moveSpeed, moveFreqOri:event._moveFrequency, run:true, stop:true };
                const inStop = event._inNearStopKe;
                // プレイヤーにストップ中イベントを保存
                if (!$gamePlayer._nearStopEventsKe) { $gamePlayer._nearStopEventsKe = []; }
                $gamePlayer._nearStopEventsKe.push(event._eventId);
                // ストップ中はすり抜け
                if (through) {
                    inStop.through = true;
                    inStop.pushTime = keke_pushTime;
                    inStop.pushDura = inStop.pushTime;
                }
                // ストップ封印
                if (noStop) {
                    inStop.stop = false;
                }
                // ストップ中は近づく
                if (come) {
                    event._moveType = 2;
                    inStop.come = true;
                    inStop.stop = false;
                }
                // 移動速度の変更
                if (moveSpeed) {
                    changeMoveSpeed(event, moveSpeed);
                }
                // 移動頻度の変更
                if (moveFreq) {
                    changeMoveFreq(event, moveFreq);
                }
                // 発動コモンイベント
                if (commonId1) {
                    doCommonEvent(commonId1, event);
                }
                // セルフスイッチ
                if (selfSwitch) {
                    // セルフスイッチをオン
                    setSelfSwitch(event, selfSwitch, true);
                }
                // ストップカウントを 0 に
                event._realX = event._x;
                event._realY = event._y;
                event._stopCount = 0;
            }
            // ストップ中はプレイヤーの方を向く
            if (dir) {
                dirPlayer(event);
            }
        // ストップ解除時の処理
        } else if (breaker) {
            const inStop = event._inNearStopKe;
            // 移動タイプを戻す
            if (come) {
                event._moveType = inStop.moveTypeOri;
            }
            // 移動速度を戻す
            if (moveSpeed) {
                event._moveSpeed = inStop.moveSpeedOri;
            }
            // 移動頻度を戻す
            if (moveSpeed) {
                event._moveFrequency = inStop.moveFreqOri;
            }
            // イベントのストップ中データを消去
            event._inNearStopKe = {};
            // 時間解除時は解除ウェイト
            if (inStop.count != null && inStop.count == 0) {
                event._inNearStopKe.breakWait = true;
            }
            // プレイヤーのストップ中イベントをリムーブ
            $gamePlayer._nearStopEventsKe = $gamePlayer._nearStopEventsKe.filter(eventId => eventId != event._eventId);
            // 解除コモンイベント
            if (commonId2) {
                doCommonEvent(commonId2, event);
            }
            // セルフスイッチ
            if (selfSwitch) {
                // セルフスイッチをオフ
                setSelfSwitch(event, selfSwitch, false)
            }
        }
        // ストップ封印か、近づくの場合はストップしない
        if (noStop || come) { stop = false; }
        return stop;
    };


    //- 発動チェック
    function checkRunning(event, distance, scope, isNoRegion, noDoRegion) {
        if ($gameMap.isEventRunning()) { return false; }
        let run = true;
        // 無効リージョンなら false
        if (isNoRegion) { run = false; }
        // 有効リージョンでないなら false
        if (run && noDoRegion) { run = false; }
        // 発動距離内か
        if (run) { run = distance <= scope; }
        // 解除ウェイト中は範囲外に出るまで起動しない
        if (isBreakWait(event)) {
            if (run) {
                run = false;
            } else {
                event._inNearStopKe.breakWait = null;
            }
        }
        return run;

    };


    //- 解除チェック
    function checkBreaker(event, running, distance, break_, time, isNoRegion, noDoRegion) {
        if ($gameMap.isEventRunning()) { return false; }
        // ストップ中でなければ false
        if (!inNearStop(event)) { return false; }
        // 無効リージョンなら true
        if (isNoRegion) { return true; }
        // 有効リージョンでないなら true
        if (noDoRegion) { return true; }
        const inStop = event._inNearStopKe;
        // 解除時間がある場合は、解除時間を更新
        if (time) {
            if (inStop.count == null) { inStop.count = time + 1; }
            inStop.count--;
            // カウント 0 になったら true;
            if (inStop.count <= 0) {
                return true;
            }
        }
        // 解除距離がある場合は、解除距離より離れたらtrue
        if (break_) {
            if (break_ < 0) { return false; }
            if (distance > break_) { return true;}
        // ない場合は発動距離で判定
        } else {
            return !running;
        }
    };


    //- 移動速度の変更
    function changeMoveSpeed(event, moveSpeed) {
        let match = moveSpeed.match(/\d+\.*\d*/);
        const val = match ? Number(match[0]) : 0;
        match = moveSpeed.match(/[\+\-\*\/]/);
        const ope = match ? match[0] : "";
        if (ope == "+") {
            event._moveSpeed = event._moveSpeed + val;
        } else if (ope == "-") {
            event._moveSpeed = event._moveSpeed - val;
        } else if (ope == "*") {
            event._moveSpeed = event._moveSpeed * val;
        } else if (ope == "/") {
            event._moveSpeed = event._moveSpeed / val;
        } else {
            event._moveSpeed = val;
        }
    };


    //- 移動頻度の変更
    function changeMoveFreq(event, moveFreq) {
        let match = moveFreq.match(/\d+\.*\d*/);
        const val = match ? Number(match[0]) : 0;
        match = moveFreq.match(/[\+\-\*\/]/);
        const ope = match ? match[0] : "";
        if (ope == "+") {
            event._moveFrequency = event._moveFrequency + val;
        } else if (ope == "-") {
            event._moveFrequency = event._moveFrequency - val;
        } else if (ope == "*") {
            event._moveFrequency = event._moveFrequency * val;
        } else if (ope == "/") {
            event._moveFrequency = event.moveFrequency / val;
        } else {
            event._moveFrequency = val;
        }
    };


    //- プレイヤーの方を向く
    function dirPlayer(event) {
        if (event._directionFix) { return; }
        const sx = event.deltaXFrom($gamePlayer.x);
        const sy = event.deltaYFrom($gamePlayer.y);
        if (Math.abs(sx) > Math.abs(sy)) {
            event._direction = sx > 0 ? 4 : 6;
        } else if (sy !== 0) {
            event._direction = sy > 0 ? 8 : 2;
        }
    };


    //- ニアストップ中は中プライオリティでも足元のイベントを起動する(コア追加)
    const _Game_Player_checkEventTriggerHere = Game_Player.prototype.checkEventTriggerHere;
    Game_Player.prototype.checkEventTriggerHere = function(triggers) {
        _Game_Player_checkEventTriggerHere.apply(this, arguments);
        if (this.canStartLocalEvents() && beStopEvents()) {
            this.startMapEvent(this.x, this.y, triggers, true);
        }
    };


    //- すり抜けの処理(コア追加)
    const _Game_Player_isCollidedWithEvents = Game_Player.prototype.isCollidedWithEvents;
    Game_Player.prototype.isCollidedWithEvents = function(x, y) {
        let result = _Game_Player_isCollidedWithEvents.apply(this, arguments);
        if (beStopEvents()) {
            // ヒット時
            if (result) {
                // すり抜け化判定
                result = $gameMap.eventsXyNt(x, y).some(event => event.isNormalPriority() && !isThrough(event));
            // 非ヒット時
            } else {
                // 押し込み時間を戻す
                restorePushTime();
            }
        }
        return result
    };


    //- すり抜け状態か
    function isThrough(event) {
        const inStop = event._inNearStopKe;
        if (!inNearStop(event) || !inStop.through) { return false; }
        // タッチ移動中以外は押し込み判定
        if (inStop.pushDura && !inTouchMove) {
            inStop.pushDura--;
            pushedNearStop = true;
            return false;
        }
        return true;
    };


    //- 押し込み時間を戻す
    function restorePushTime() {
        if (!pushedNearStop) { return; }
        pushedNearStop = null;
        $gamePlayer._nearStopEventsKe.forEach(eventId => {
            const event = $gameMap.event(eventId);
            if (!event) { return; }
            const inStop = event._inNearStopKe;
            if (!inNearStop(event) || !inStop.through || !inStop.pushTime) { return; }
            inStop.pushDura = inStop.pushTime;
        });
    };


    //- タッチ移動中ははすぐすり抜けするためのフラグ管理(コア追加)
    const _Game_Player_findDirectionTo = Game_Player.prototype.findDirectionTo;
    Game_Player.prototype.findDirectionTo = function(goalX, goalY) {
        // タッチ移動中フラグをオン
        inTouchMove = true;
        return _Game_Player_findDirectionTo.apply(this, arguments);
    };

    const _Game_Player_moveByInput = Game_Player.prototype.moveByInput;
    Game_Player.prototype.moveByInput = function() {
        _Game_Player_moveByInput.apply(this);
        // タッチ移動中フラグをオン
        inTouchMove = false;
    };


    //- 近づくの時は遠ざかるのを禁止(コア追加)
    const _Game_CharacterBase_canPass = Game_CharacterBase.prototype.canPass;
    Game_CharacterBase.prototype.canPass = function(x, y, d) {
        if (isCome(this) && !this._moveStyleKe) {
            // 現在地とプレイヤーの距離
            const distance = charaDistance($gamePlayer, this);
            // 移動先とプレイヤーの距離
            const x2 = $gameMap.roundXWithDirection(x, d);
            const y2 = $gameMap.roundYWithDirection(y, d);
            const distanceGo = charaPointDistance($gamePlayer, x2, y2);
            // プレイヤーから遠ざかるなら移動しない
            if (distanceGo > distance) { return false; }
        }
        return _Game_CharacterBase_canPass.apply(this, arguments);
    };

    
    //- ニアストップ中か
    function inNearStop(event) {
        const inStop = event._inNearStopKe;
        return inStop && inStop.run;
    };


    //- ストップ中のイベントがいるか
    function beStopEvents() {
        return $gamePlayer._nearStopEventsKe && $gamePlayer._nearStopEventsKe.length;
    };


    //- ストップ状態か
    function isStop(event) {
        return event._eventId && inNearStop(event) && event._inNearStopKe.stop;
    };


    //- 近づく状態か
    function isCome(event) {
        return event._eventId && inNearStop(event) && event._inNearStopKe.come;
    };


    //- 解除ウェイト中か
    function isBreakWait(event) {
        return event._inNearStopKe && event._inNearStopKe.breakWait;
    };



    //==================================================
    //--  真のランダム移動
    //==================================================


    //- ゲームイベント/ランダム移動(コア追加)
    const _Game_Event_moveTypeRandom = Game_Event.prototype.moveTypeRandom;
    Game_Event.prototype.moveTypeRandom = function() {
        // 真ランダム移動の処理
        if (processTrueRandomMove(this)) { return; }
        _Game_Event_moveTypeRandom.apply(this);
    };


    //- 真ランダム移動の処理
    function processTrueRandomMove(event) {
        if (event._moveType != 1) { return; }
        // 最初の注釈の取得
        const comment = getFirstComment(event);
        if (!comment) { return null; }
        // 真ランダム移動か判定
        if (!comment.match(/<trueRandomMove>/i)) { return false; }
        // ランダム移動
        event.moveRandom();
        return true;
    };



    //==================================================
    //--  コモンイベント /ベーシック
    //==================================================


    //- シーンマップ・更新
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.apply(this);
        // コモンイベントの更新
        updateCommonEvent(this);
    };


    //- コモンイベントの更新
    function updateCommonEvent(body) {
        const preters = body._commonPretersKeNast;
        if (!preters || !preters.length) { return; }
        // プリターを更新
        let del = false;
        preters.forEach((preter, i) => {
            preter.update();
            if (!preter.isRunning()) {
                preters[i] = null;
                del = true;
            }
        });
        // null を消去
        if (del) {
            body._commonPretersKeNast = preters.filter(p => p);
        }
    };


    //- コモンイベントの実行
    function doCommonEvent(commonId, event) {
        const body = SceneManager._scene;
        // コモンイベントを取得
        const commonEvent = $dataCommonEvents[commonId];
        if (!commonEvent) { return; }
        // インタープリターを作成
        const preter = new Game_Interpreter(0);
        if (!body._commonPretersKeNast) { body._commonPretersKeNast = []; }
        body._commonPretersKeNast.push(preter);
        // セットアップ
        const eventId = event ? event._eventId : 0;
        preter.setup(commonEvent.list, eventId);
    };



    //==================================================
    //-  注釈 /ベーシック
    //==================================================

    //- 最初の注釈の取得
    function getFirstComment(event) {
        const comments = [];
        // 現在のページの最初の注釈を取得
        const currentPage = event.page();
        comments.push(getFirstComment_page(currentPage));
        // 1ページの最初の注釈を取得
        if (event._pageIndex != 0) {
            const onePage = event.event().pages[0];
            comments.push(getFirstComment_page(onePage));
        }
        // 注釈を全て統合
        let comment = "";
        comments.forEach(cmt => {
            if (!cmt) { return; }
            comment += cmt + "\n";
        });
        return comment;
    };

    //- 最初の注釈の取得-ページ
    function getFirstComment_page(page) {
        if (!page) { return null; }
        let list = page.list;
        if (!list[0] || list[0].code != 108) { return null; }
        // 最初の注釈を全て読み込み
        let comment = "";
        let i = 0;
        while (true) {
            const command = list[i];
            if (!command || !(command.code == 108 || command.code == 408)) { break; }
            comment += command.parameters[0] + "\n";
            i++;
        }
        return comment;
    };

    //- 最初の注釈の取得
    /*function getFirstComment(event) {
        let page = event.page();
        let list = page ? event.page().list : [];
        let commandFirst = list[0] || {};
        if (commandFirst.code != 108) {
            page = event.event().pages[0];
            list = page ? page.list : [];
            commandFirst = list[0] || {};
            if (commandFirst.code != 108) { return; }
        }
        // 注釈を全て統合
        let comment = "";
        let i = 0;
        while (true) {
            const command = list[i];
            if (!command || !(command.code == 108 || command.code == 408)) { break; }
            comment += command.parameters[0] + "\n";
            i++;
        }
        return comment;
    };*/



    //==================================================
    //-  セルフスイッチ /ベーシック
    //==================================================

    //- セルフスイッチのセット
    function setSelfSwitch(event, ch, boolean) {
        if (!event._eventId) { return; }
        const key = [event._mapId, event._eventId, ch];
        $gameSelfSwitches.setValue(key, boolean);
    };


    //- セルフスイッチの取得
    function getSelfSwitch(event, ch) {
        if (!event._eventId) { return null; }
        const key = [event._mapId, event._eventId, ch];
        return $gameSelfSwitches.value(key);
    };


    
    //==================================================
    //-  距離計算 /ベーシック
    //==================================================

    //- キャラ間の距離
    function charaDistance(a, b) {
        return Math.sqrt(Math.pow($gameMap.deltaX(a._realX, b._realX), 2) + Math.pow($gameMap.deltaY(a._realY, b._realY), 2));
    };


    //- キャラとポイントの距離
    function charaPointDistance(a, px, py) {
        return Math.sqrt(Math.pow($gameMap.deltaX(a._realX, px), 2) + Math.pow($gameMap.deltaY(a._realY, py), 2));
    };


    //- 最短X距離
    /*function shortestDistanceX(ax, bx) {
        const distance1 = Math.abs(ax - bx);
        if (!$gameMap.isLoopHorizontal()) {
            return distance1;
        }
        const mw = $dataMap.width;
        const ax2 = ax + mw;
        const distance2 = Math.abs(ax2 - bx);
        const ax3 = ax - mw;
        const distance3 = Math.abs(ax3 - bx);
        return Math.min(distance1, distance2, distance3);
    };


    //- 最短Y距離
    function shortestDistanceY(ay, by) {
        const distance1 = Math.abs(ay - by);
        if (!$gameMap.isLoopVertical()) {
            return distance1;
        }
        const mh = $dataMap.height;
        const ay2 = ay + mh;
        const distance2 = Math.abs(ay2 - by);
        const ay3 = ay - mh;
        const distance3 = Math.abs(ay3 - by);
        return Math.min(distance1, distance2, distance3);
    };*/
    
})();