//=============================================================================
// NekoGakuen_MulitLanguage.js
// Version: 2.0.5
//=============================================================================
/*:
 * @target MV MZ
 * @plugindesc Multi-language content (Ver 2.0.5)
 * @author NekoGakuen
 * @url https://twitter.com/NekoGakuen
 * @help
 * ================================
 * Author: NekoGakuen
 * Version: 2.0.5
 * Twitter: https://twitter.com/NekoGakuen
 * ================================
 * 
 * -- Plugin Information --
 * Used in RPG Maker MV/MZ to support switching of game content in various languages.
 * 
 * 
 * -- Update Information --
 * V2.0.5 Fixed game level stuttering issues caused by CSV file preloading.
 * V2.0.4 Fixed the bug that the game title could not be updated, and changed the multi-language setting of the game title.
 * V2.0.3 Fixed a problem with preload language files.
 *        (Thanks to "Sang Hendrix" for his assistance.)
 * V2.0.2 Fixed the path problem of reading language settings.
 * V2.0.1 Fixed bug in RPG Maker MZ.
 * V2.0.0 Remove the language commands from the database and integrate the parameters of the language commands,
 *        fix the problem of displaying an error when there are spaces in the language commands.
 * V1.9.1 Fix some bugs.
 * V1.9.0 Fix some bugs.
 * V1.8.9 Fix some bugs.
 * V1.8.8 Removed the "Set Language Picture Path" feture and improved the module call of the code, etc.
 * V1.8.7 Added the option to set whether the game "Option" shows the language setting,
 *        and added the language selection to set the language from Steam.
 * V1.8.6 Added custom language command parameters, fix the problem of plugin commands.
 * V1.8.5 Added support for English language version (DeepL Translator).
 * V1.8.4 Fixed some bug issues.
 * V1.8.3 Updated plugin terms of use and added information about "Supported Platforms". 
 * V1.8.2 Fixed some problems in the code. 
 * V1.8.1 Fixed the bug that MZ can't read the plugins stored in subfolders when the version 1.3.0 is updated. 
 * V1.8.0 Added automatic alignment of the initial language selection window,
 *        and fixed the display problem when the initial language uses the background image. 
 * V1.7.8 Fixed the problem of line break when using semicolon in dialog. 
 * V1.7.7 Fixed the problem that the initial language selection cannot be read in order. 
 * V1.7.6 Fixed the problem of not updating text options and blank content selection on the options screen. 
 * V1.7.5 Fixed some functions and optimized code,
 *        and updated plugin terms of use and plugin command to remove specified CSV files. 
 * V1.7.0 Added function to change font by language,
 *        specify picture path, and add plugin command/text command to specify CSV file.
 * V1.6.4 Update plugin terms of use.
 * V1.6.3 Fixed the problem of displaying text messages in a few languages in MV/MZ version.
 * V1.6.2 Fixed the problem of displaying text message in the language in MV/MZ version.
 * V1.6.1 Fixed the problem of displaying wrong map name in MV version.
 * V1.6.0 Combined the plugins for MV version and MZ version.
 * V1.5.2 Fixed the problem of selecting language at the beginning of the game.
 * V1.5.1 Optimize the code of language selection at the beginning of the game.
 * V1.5.0 Added the function of selecting language at the beginning of the game.
 * V1.4.1 Fixed the problem that "\Say[<parameter>]" could not convert control characters when used.
 * V1.4.0 Fixed the bug when using "!Say <parameter>",
 *        and also added a link to the Google file for plugin instructions.
 * V1.3.0 Fixed the bug that "Show text..." The name box cannot read "External Text" when "Show Text...",
 *        and also optimize the code.
 * V1.2.0 Fixed the problem that "External Text" parameter cannot be read successfully when reading files.
 * V1.1.0 Fixed some bugs.
 * V1.0.0 Release plugin.
 * 
 * 
 * -- Use Description --
 * 1. There are a few pre-requisite steps to complete,
 *    please refer to the manual link in the Manual folder.
 * 2. Load the plugin in the "Plugin Manager" of RPG Maker MV/MZ.
 * 3. Select the Advanced section of Event Commands "Plugin Command..." or "Script...", 
 *   and set the parameters of the "Plugin Command" or "Script" to be executed.
 * 
 * 
 * -- Plugin Command / Script / Contol Characters --
 * 
 * [ Change Game Language ]
 * -- Description: 
 *    Set the message language parameter, and it must match the field name in the CSV file.
 * -- Plugin Command: 
 *    NekoCommands MulitLang_SET <parameters>
 * -- Script: 
 *    $gameSystem.mulitLangSET("<parameters>");
 * 
 * [ Get Game Language ]
 * -- Description: 
 *    Get Current Language, and it must match the field name in the CSV file.
 * >> Parameter: 
 *    Variable ID.
 * -- Plugin Command: 
 *    NekoCommands MulitLang_GET <parameters>
 * -- Conditional Branch: 
 *    $gameSystem.mulitLangGET() == "<Conditional Args>"
 * >> Conditional Args: 
 *    language Args.
 * 
 * [ Call Multilingual Messages]
 * -- Description: 
 * Calling a specified text item, "Say" can be changed in the "Language Command".
 * >> Parameter01: the name of the field you set in the CSV file.
 * >> Parameter02: the "Parameter Name" you set in "Text Language List..." of Plugin Manager.
 * -- Contol Characters:
 *    \Say[<parameters>]
 * -- Contol Characters:
 *    \Say[<parameter01>,<parameter02>]
 * -- Note: It is better not to have any half-space in the above mentioned parameter naming style.
 * If there are any blanks, you can change them to "_" instead of half blanks.
 * 
 * [ Call Multilingual Messages (＊Retaining the original text for annotation)]
 * -- Description: 
 * Add the text item specified by the call before or after the original text, "Say" can be changed in the "Language Command".
 * >> Parameter01: the name of the field you set in the CSV file.
 * >> Parameter02: the "Parameter Name" you set in "Text Language List..." of Plugin Manager.
 * -- Contol Characters:
 *    \Say[<parameters>,true]
 * -- Contol Characters:
 *    \Say[<parameter01>,<parameter02>,true]
 * -- Note: It is better not to have any half-space in the above mentioned parameter naming style.
 * If there are any blanks, you can change them to "_" instead of half blanks.
 * 
 * 
 * -- Supported Platforms --
 * - NWjs:
 *   [√ Yes(Windows、macOS)]
 * - Electron:
 *   [√ Yes(Windows、macOS)]
 * - Google Chrome:
 *   [√ Yes(Windows、macOS、Android)]
 * - Mozilla Firefox:
 *   [√ Yes(Windows、macOS、Android)]
 * - Microsoft Edge:
 *   [√ Yes(Windows、macOS、Android)]
 * - Apple Safari:
 *   [√ Yes(macOS)]
 * - Android:
 *   [√ Yes]
 * - iOS:
 *   [× No]
 * 
 * 
 * 
 * -- Terms of Use --
 * No prior notice is required to modify or translate this plugin, and if the plugin has bugs you can report them.
 * The copyright of this plugin is owned by NekoGakuen.
 * We also reserve the right to modify and change the rules of use of the plugin.
 * 
 * --------------------
 * - Credit: 
 *   [△ Not required, but appreciated if you have one. (#1)]
 * - Commercial: 
 *   [√ OK]
 * - Adults:
 *   [√ OK]
 * 
 * #1：If you want to attach it, you can mark it with "NekoGakuen".
 * --------------------
 * 
 * @command NekoCommands MulitLang_SET
 * @text Change Game Language
 * @desc Set the message language parameter.
 *
 * @arg langTextArgs
 * @type string
 * @default en_US
 * @text Language Args
 * @desc Set the Messages language parameter, and it must be the same as the field name in the CSV file.
 * 
 * @command NekoCommands MulitLang_GET
 * @text Get Game Language
 * @desc Set the message language parameter.
 *
 * @arg varld
 * @type variable
 * @default 0
 * @text Variable ID
 * @desc In the case of event command "Conditional Branch", this parameter can be omitted.
 * 
 * 
 * @param Translate Patch
 * @text ＃→Translation Plugin Patch
 * @desc Translates the plugin file name of the patch without the extension.
 * @type string
 * @default 
 * 
 * @param Custom Fontlist
 * @text Custom Fontlist...
 * @desc Put the font file in the fonts folder of the game project directory, and enter the file name of the font file (excluding the file extension) in this parameter, and leave it blank if it is not used.
 * If you choose "System Default Fonts", you can directly enter the "font name" (e.g., Segoe UI, etc.).
 * @parent Font Group
 * @type struct<fonts>[]
 * @default []
 * 
 * @param Lancsv List
 * @text Text Language List...
 * @desc Import various external text CSV files.
 * @type struct<Lancsv>[]
 * @default ["{\"Lancsv Name\":\"Text01\",\"Lancsv Path\":\"data/Text.csv\"}"]
 * 
 * @param Custom Langlist
 * @text Custom Game Language...
 * @desc Set the language that you can choose to switch between for your own games.
 * @type struct<Langlist>[]
 * @default ["{\"Lang Key\":\"JP\",\"Lang Name\":\"日本語\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"言語を選択してください？\"}","{\"Lang Key\":\"EN\",\"Lang Name\":\"English\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"Please select language?\"}"]
 * 
 * @param Message CommandName
 * @text Language Command
 * @desc The language command for setting up the Message Box and Database is recommended to use English naming.
 * @type string
 * @default Say
 * 
 * @param Select Lang Type
 * @text Language Select Type
 * @desc Set the type of language selection. If you select the language from Steam, you need to install the "NekoGakuen_SteamworksAPI" plugin.
 * @type select
 * @default Player Select
 * @option Player's choice
 * @value Player Select
 * @option From the Steam client
 * @value Steam Client
 * @option From the Steam Game
 * @value Steam Game
 * 
 * @param Config Lang Boolean
 * @text Show Option Language
 * @desc Whether to enable the language setting in the game settings "Option".
 * @type boolean
 * @on Enable
 * @off Disable
 * @default true
 * 
 * @param Config Lang
 * @text Option Name
 * @desc Set the name of the "Options" setting on the title screen.
 * If you want to support multiple languages, you can use the "Language Command".
 * @type string
 * @default Language Setting
 * 
 * @param TitleText Lang
 * @text Game Title
 * @desc Set the title of the game.
 * If you want to support multiple languages, you can use the "Language Command".
 * @type string
 * @default 
 * 
 * @param InitLang Set
 * @text Init language select...
 * @desc Set the language select parameters at the beginning of the game.
 * @type struct<InitLan>
 * @default {"InitLan Switch":"true","InitLan Images":"","InitLan AlignTypeX":"Auto","InitLan Auto SelectX":"Center","InitLan Custom SelectX":"283","InitLan AlignTypeY":"Auto","InitLan Auto SelectY":"Center","InitLan Custom SelectY":"250","InitLan Select Width":"250","InitLan Select Opacity":"255","InitLan Select Align":"left","InitLan HelpX":"0","InitLan HelpY":"0","InitLan Help Opacity":"255"}
 * 
 */
/*~struct~Lancsv:
 * 
 * @param Lancsv Name
 * @text CSV Name
 * @desc Specifies the parameter name of the external text CSV file.
 * @type string
 * 
 * @param Lancsv Path
 * @text CSV File
 * @desc Specifies the CSV file path of the external text.
 * @type string
 * 
 */
/*~struct~Langlist:
 * 
 * @param Lang Key
 * @text Language Key
 * @desc Specify the parameter name of the game language, which must be the same as the field name of the CSV file.
 * @type string
 * @default en_US
 * 
 * @param Lang Name
 * @text Language Name
 * @desc Specify the display name of the game language.
 * @type string
 * @default English
 * 
 * @param Lang FontFace
 * @text Language FontName
 * @desc Specify the font name to be displayed in the game language, remember to set it in the "Custom Font List" parameter first.
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Lang FontSize
 * @text Language FontSize
 * @desc Specifies the font size to be displayed for that game language.
 * @type number
 * @default 28
 * 
 * @param Init Help
 * @text Init Language Description
 * @desc Specify the text description that will be displayed when the initial game language is selected.
 * This parameter is not necessary if "Init language select" is turned off.
 * @type string
 * @default Please select a language?
 * 
 */
/*~struct~InitLan:
 * 
 * @param InitLan Switch
 * @text Init Language
 * @desc Whether you need to open the language select at the beginning of the game.
 * @type boolean
 * @on Enable
 * @off Disable
 * @default true
 * 
 * @param InitLan Images
 * @text Picture Background
 * @desc Specify the background image for the initial language selection screen.
 * The picture file is placed in the img/pictures folder.
 * @type file
 * @dir img/pictures/
 * @require 1
 * 
 * @param InitLan AlignTypeX
 * @text Select window X Align
 * @desc Specifies the X-axis alignment type of the language selection window.
 * @type select
 * @default Auto
 * @option Auto Align
 * @value Auto
 * @option Set Align
 * @value Custom
 * 
 * @param InitLan Auto SelectX
 * @text Auto Align X
 * @desc The X-axis display coordinates of the specified language selection window.
 * @type select
 * @parent InitLan AlignTypeX
 * @default Center
 * @option Left
 * @value Left
 * @option Center
 * @value Center
 * @option Right
 * @value Right
 * 
 * @param InitLan Custom SelectX
 * @text Set X-axis coordinates
 * @desc The X-axis display coordinates of the specified language selection window.
 * @type number
 * @parent InitLan AlignTypeX
 * @min -9999
 * @default 283
 * 
 * @param InitLan AlignTypeY
 * @text Select window Y Align
 * @desc Specifies the Y-axis alignment type of the language selection window.
 * @type select
 * @default Auto
 * @option Auto Align
 * @value Auto
 * @option Set Align
 * @value Custom
 * 
 * @param InitLan Auto SelectY
 * @text Auto Align Y
 * @desc The Y-axis display coordinates of the specified language selection window.
 * @type select
 * @parent InitLan AlignTypeY
 * @default Center
 * @option Top
 * @value Top
 * @option Center
 * @value Center
 * @option Bottom
 * @value Bottom
 * 
 * @param InitLan Custom SelectY
 * @text Set Y-axis coordinates
 * @desc The Y-axis display coordinates of the specified language selection window.
 * @type number
 * @parent InitLan AlignTypeY
 * @min -9999
 * @default 250
 * 
 * @param InitLan Select Width
 * @text Select Window Width
 * @desc Specifies the display width of the language selection window.
 * @type number
 * @min 100
 * @default 250
 * 
 * @param InitLan Select Opacity
 * @text Select Window Opacity
 * @desc Specifies the opacity of the language selection window.
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 * @param InitLan Select Align
 * @text Text Alignment
 * @desc Specifies the alignment of the text display in the language selection window.
 * @type select
 * @default left
 * @option Left
 * @value left
 * @option Center
 * @value center
 * @option Right
 * @value right
 * 
 * @param InitLan Help Boolean
 * @text Show Language Help
 * @desc Whether to show the language help at the beginning of the game.
 * @type boolean
 * @on Show
 * @off Hide
 * @default true
 * 
 * @param InitLan AlignTypeHelpY
 * @text Help window Y Align
 * @desc Specifies the Y-axis alignment type of the language help window.
 * @type select
 * @default Auto
 * @option Auto Align
 * @value Auto
 * @option Set Align
 * @value Custom
 * 
 * @param InitLan Auto SelectHelpY
 * @text Auto Align Y
 * @desc The Y-axis display coordinates of the specified language help window.
 * @type select
 * @parent InitLan AlignTypeHelpY
 * @default Top
 * @option Top
 * @value Top
 * @option Bottom
 * @value Bottom
 * 
 * @param InitLan Custom HelpY
 * @text Set Y-axis coordinates
 * @desc The Y-axis display coordinates of the specified language help window.
 * @type number
 * @parent InitLan AlignTypeHelpY
 * @min -9999
 * @default 0
 * 
 * @param InitLan Help Opacity
 * @text Description Opacity
 * @desc Specifies the opacity of the language description window.
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 */
/*~struct~fonts:
 * 
 * @param Fonts File
 * @text Fonts File Name
 * @desc Set the font file name(excluding the file extension), if you select "System Default Fonts", the font file already installed on the PC will be read.
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Fonts Format
 * @text Fonts File Format
 * @desc Set the font file extension format, if you select "System Default Fonts", the font file already installed on the PC will be read.
 * @type select
 * @default ttf
 * @option System Default Fonts
 * @value local
 * @option OTF (OpenType Font)
 * @value otf
 * @option TTF (TrueType Font)
 * @value ttf
 * @option WOFF (Web Open Font Format)
 * @value woff
 * 
 */
/*:zh
 * @target MV MZ
 * @plugindesc 多國語言文本 (Ver 2.0.5)
 * @author 貓咪學園 NekoGakuen
 * @url https://twitter.com/NekoGakuen
 * @help
 * ================================
 * 作者：貓咪學園 NekoGakuen
 * 版本：2.0.5
 * 聯絡推特：https://twitter.com/NekoGakuen
 * ================================
 * 
 * 
 * ─ 插件簡介 ─
 * 在RPG Maker MV/MZ中用於支援各國語言文本內容的切換功能。
 * 
 * 
 * ─ 更新履歷 ─
 * V2.0.5 修正 CSV 檔案預先載入所導致的遊戲關卡卡頓問題。
 * V2.0.4 修正遊戲標題無法更新的錯誤，變更遊戲標題的多語言設定。
 * V2.0.3 修正預先讀取語言檔案的問題。
 *        (感謝 Sang Hendrix 的協助。)
 * V2.0.2 修正讀取語言設定的路徑問題。
 * V2.0.1 修正在RPG Maker MZ的錯誤。
 * V2.0.0 移除資料庫的語言命令，並整合語言命令的參數，修正當語言命令有空格的情況會顯示錯誤的問題。
 * V1.9.1 修正一些錯誤。
 * V1.9.0 修正一些錯誤。
 * V1.8.9 修正一些錯誤。
 * V1.8.8 移除「指定語言圖片路徑」功能，並改善程式碼的模組呼叫等。
 * V1.8.7 新增指定遊戲「選項」是否顯示語言設定，以及新增語言選擇指定來自Steam語言。
 * V1.8.6 新增自訂語言命令參數，修正插件命令的問題。
 * V1.8.5 新增英文翻譯版本的支援 (DeepL翻譯)。
 * V1.8.4 修正部分的錯誤問題。
 * V1.8.3 更新插件使用條款，以及新增關於「支援平台」資訊。 
 * V1.8.2 修正程式碼上的一些問題。 
 * V1.8.1 修正MZ在1.3.0版本更新時，無法讀取存放在子資料夾內的插件。 
 * V1.8.0 新增初期語言選擇視窗的自動對齊，並修正初期語言使用背景圖片時的顯示問題。 
 * V1.7.8 修正對話內容使用半形逗號之斷行問題。 
 * V1.7.7 修正初始化語言選擇之後無法按照順序讀取的問題。 
 * V1.7.6 修正選項設定畫面無法即時更新文字選項，以及空白內容選擇的問題。 
 * V1.7.5 修正一些功能和最佳化程式碼，並更新插件的使用條款及移除指定CSV檔案的插件命令。 
 * V1.7.0 新增依語言更改字型功能、指定圖片路徑，以及新增指定CSV檔案的插件命令/文字指令。
 * V1.6.4 更新插件使用條款。
 * V1.6.3 修正在MV/MZ版本時少部分的用語方面顯示文字訊息等問題。
 * V1.6.2 修正在MV/MZ版本時用語方面顯示文字訊息等問題。
 * V1.6.1 修正在MV版本時顯示地圖名稱錯誤的問題。
 * V1.6.0 將MV版本用及MZ版本用的插件合併。
 * V1.5.2 修正遊戲初期選擇語言功能的判斷問題。
 * V1.5.1 將遊戲初期選擇語言程式碼做最佳化處理。
 * V1.5.0 新增遊戲初期選擇語言的功能。
 * V1.4.1 修正「\Say[<參數>]」使用時無法轉換控制字符的問題。
 * V1.4.0 修正「!Say <參數>」使用時的錯誤問題，另外也新增插件使用說明的Google文件連結。
 * V1.3.0 修正「顯示文字..」時名字框無法讀取「外部文本」問題，同時也將程式碼做最佳化處理。
 * V1.2.0 修正讀檔時無法成功讀取「外部文本」參數問題。
 * V1.1.0 修正一些錯誤。
 * V1.0.0 初次版本的插件發佈。
 * 
 * 
 * ─ 使用說明 ─
 * 1.需先完成一些前置步驟，請參閱放在Manual資料夾內的使用手冊連結。
 * 2.在RPG Maker MV/MZ的「插件管理器」中匯入本插件，
 *   並在本插件的「參數」區塊設定即可。
 * 3.在事件頁中高級區塊選擇「插件命令...」，
 *   並設定選擇要執行的插件命令及參數即可。
 * 
 * 
 * ─ 插件命令/腳本/文字指令 ─
 * 
 * 【變更遊戲語言】
 * --說明：指定語言參數，必須跟CSV檔案的欄位名稱一致。
 * >>參數：你在CSV檔案的欄位名稱。
 * --插件命令 NekoCommands MulitLang_SET <參數>
 * --腳本 $gameSystem.mulitLangSET("<參數>");
 * 
 * 【取得目前語言】
 * --說明：取得目前選擇的語言，必須跟CSV檔案的欄位名稱一致。
 * >>參數：你的「變數 ID」。
 * --插件命令 NekoCommands MulitLang_GET <參數>
 * --條件分歧 $gameSystem.mulitLangGET() == "<條件參數>";
 * >>條件參數：你的語言參數。
 * 
 * 【呼叫多國語言文本】
 * --說明：呼叫指定的文本項目，「Say」可在「語言命令」變更。
 * >>參數01：你在CSV檔案的欄位名稱。
 * >>參數02：你在插件管理器「外部文本語言...」設定的「參數名稱」。
 * --文字指令 \Say[<參數01>]
 * --文字指令 \Say[<參數01>,<參數02>]
 * --注意：參數名稱盡量不使用半形空白，如有空白可以改成「_」取代半形空白。
 * 
 * 【呼叫多國語言文本(※保留原文做為註解的情況)】
 * --說明：在原文的前面或後面加入呼叫指定的文本項目，「Say」可在「語言命令」變更。
 * >>參數01：你在CSV檔案的欄位名稱。
 * >>參數02：你在插件管理器「外部文本語言...」設定的「參數名稱」。
 * --文字指令 (原文) \Say[<參數01>,true]
 * --文字指令 (原文) \Say[<參數01>,<參數02>,true]
 * --注意：參數名稱盡量不使用半形空白，如有空白可以改成「_」取代半形空白。
 * 
 * 
 * ─ 支援平台 ─
 * - NWjs：
 *  【√ 支援(Windows、macOS)】
 * - Electron：
 *  【√ 支援(Windows、macOS)】
 * - Google Chrome：
 *  【√ 支援(Windows、macOS、Android)】
 * - Mozilla Firefox：
 *  【√ 支援(Windows、macOS、Android)】
 * - Microsoft Edge：
 *  【√ 支援(Windows、macOS、Android)】
 * - Apple Safari：
 *  【√ 支援(macOS)】
 * - Android：
 *  【√ 支援】
 * - iOS：
 *  【? 未知】
 * 
 * 
 * 
 * ─ 著作聲明 ─
 * 修改或翻譯本插件無需事前告知，如果插件有BUG可以回報。
 * 本插件著作權為貓咪學園(NekoGakuen)所有。
 * 並且保留對插件使用規則的修改與更動之權利。
 * 
 * --------------------
 * -來源標示：【△ 不需要，但有的話會很感謝。 (註1)】
 * -商業營利：【√ 允許】
 * -成人用途：【√ 允許】
 * 
 * ※註1：但如有註明的話，可以註明「NekoGakuen」即可。
 * --------------------
 * 
 * @command NekoCommands MulitLang_SET
 * @text 變更遊戲語言
 * @desc 變更遊戲文本的語言。
 *
 * @arg langTextArgs
 * @type string
 * @default zh_TW
 * @text 語言參數
 * @desc 指定語言參數，必須跟CSV檔案的欄位名稱一致。
 * 
 * @command NekoCommands MulitLang_GET
 * @text 取得目前語言
 * @desc 取得目前選擇的語言，必須跟CSV檔案的欄位名稱一致。
 *
 * @arg varld
 * @text 變數 ID
 * @desc 如果是在事件命令「條件分歧」的情況，可以不用輸入此參數。
 * @type variable
 * @default 0
 * 
 * 
 * @param Translate Patch
 * @text ＃→翻譯插件補丁
 * @desc 翻譯補丁的插件檔案名稱，不包含副檔名。
 * @type string
 * @default 
 * 
 * @param Custom Fontlist
 * @text 自訂字型清單...
 * @desc 設定遊戲字型，如已安裝「NekoGakuen_FontManager」插件，則可以忽略此參數。
 * @type struct<fonts>[]
 * @default []
 * 
 * @param Lancsv List
 * @text 外部文本語言...
 * @desc 匯入外部文本CSV。
 * @type struct<Lancsv>[]
 * @default ["{\"Lancsv Name\":\"Text01\",\"Lancsv Path\":\"data/Text.csv\"}"]
 * 
 * @param Custom Langlist
 * @text 自訂遊戲語言...
 * @desc 設定支援的語言。
 * @type struct<Langlist>[]
 * @default ["{\"Lang Key\":\"zh_TW\",\"Lang Name\":\"中文\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"請選擇語言？\"}","{\"Lang Key\":\"JP\",\"Lang Name\":\"日本語\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"言語を選択してください？\"}","{\"Lang Key\":\"EN\",\"Lang Name\":\"English\",\"Lang FontFace\":\"mplus-1m-regular\",\"Lang FontSize\":\"28\",\"Init Help\":\"Please select language?\"}"]
 * 
 * @param Message CommandName
 * @text 語言命令
 * @desc 指定對話框和資料庫的語言命令，建議使用英文命名。
 * @type string
 * @default Say
 * 
 * @param Select Lang Type
 * @text 語言選擇類型
 * @desc 指定語言選擇的指定類型，如選擇來自Steam語言時，需安裝「NekoGakuen_SteamworksAPI」插件。
 * @type select
 * @default Player Select
 * @option 由玩家自行選擇
 * @value Player Select
 * @option 從Steam客戶端指定
 * @value Steam Client
 * @option 從Steam遊戲指定
 * @value Steam Game
 * 
 * @param Config Lang Boolean
 * @text 顯示選項語言
 * @desc 是否開啟在遊戲設定「選項」的語言設定。
 * @type boolean
 * @on 開啟
 * @off 關閉
 * @default true
 * 
 * @param Config Lang
 * @text 選項名稱
 * @desc 指定遊戲設定「選項」的語言設定，如需支援多國語言可使用語言命令。
 * @type string
 * @default 語言設定
 * 
 * @param TitleText Lang
 * @text 遊戲名稱
 * @desc 指定遊戲的標題名稱，如需支援多國語言可使用語言命令。
 * @type string
 * @default 
 * 
 * @param InitLang Set
 * @text 初期語言選擇...
 * @desc 設定初始語言選擇。
 * @type struct<InitLan>
 * @default {"InitLan Switch":"true","InitLan Images":"","InitLan AlignTypeX":"Auto","InitLan Auto SelectX":"Center","InitLan Custom SelectX":"283","InitLan AlignTypeY":"Auto","InitLan Auto SelectY":"Center","InitLan Custom SelectY":"250","InitLan Select Width":"250","InitLan Select Opacity":"255","InitLan Select Align":"left","InitLan HelpX":"0","InitLan HelpY":"0","InitLan Help Opacity":"255"}
 * 
 */
/*~struct~Lancsv:zh
 * 
 * @param Lancsv Name
 * @text 文本參數
 * @desc 指定CSV檔案的參數名稱。
 * @type string
 * 
 * @param Lancsv Path
 * @text 文本路徑
 * @desc 指定CSV檔案的路徑。
 * @type string
 * 
 */
/*~struct~Langlist:zh
 * 
 * @param Lang Key
 * @text 語言參數
 * @desc 指定遊戲語言的參數名稱，必須跟CSV檔案的欄位名稱一致。
 * @type string
 * @default zh_TW
 * 
 * @param Lang Name
 * @text 語言名稱
 * @desc 指定遊戲語言的顯示名稱。
 * @type string
 * @default 中文
 * 
 * @param Lang FontFace
 * @text 字型名稱
 * @desc 指定遊戲語言的字型名稱，可在「自訂字型清單...」參數設定。
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Lang FontSize
 * @text 字型大小
 * @desc 指定遊戲語言的字型大小。
 * @type number
 * @default 28
 * 
 * @param Init Help
 * @text 初期語言說明
 * @desc 指定初期選擇項遊戲語言的文字說明，如「初期語言功能」為關閉時則可以忽略此參數。
 * @type string
 * @default 請選擇語言？
 * 
 */
/*~struct~InitLan:zh
 * 
 * @param InitLan Switch
 * @text 初期語言功能
 * @desc 是否開啟遊戲初期時的語言選擇項。
 * @type boolean
 * @on 開啟
 * @off 關閉
 * @default true
 * 
 * @param InitLan Images
 * @text 初期語言背景
 * @desc 指定語言選擇項的背景，圖片放在img/pictures資料夾。
 * @type file
 * @dir img/pictures/
 * @require 1
 * 
 * @param InitLan AlignTypeX
 * @text 選擇項X軸對齊
 * @desc 指定選擇項的X軸類型。
 * @type select
 * @default Auto
 * @option 自動對齊
 * @value Auto
 * @option 自訂位置
 * @value Custom
 * 
 * @param InitLan Auto SelectX
 * @text 自動對齊X軸
 * @desc 指定選擇項的X軸對齊。
 * @type select
 * @parent InitLan AlignTypeX
 * @default Center
 * @option 靠左
 * @value Left
 * @option 置中
 * @value Center
 * @option 靠右
 * @value Right
 * 
 * @param InitLan Custom SelectX
 * @text 自訂位置X軸
 * @desc 指定選擇項的X軸座標。
 * @type number
 * @parent InitLan AlignTypeX
 * @min -9999
 * @default 283
 * 
 * @param InitLan AlignTypeY
 * @text 選擇項Y軸對齊
 * @desc 指定選擇項的Y軸類型。
 * @type select
 * @default Auto
 * @option 自動對齊
 * @value Auto
 * @option 自訂位置
 * @value Custom
 * 
 * @param InitLan Auto SelectY
 * @text 自動對齊Y軸
 * @desc 指定選擇項的Y軸對齊。
 * @type select
 * @parent InitLan AlignTypeY
 * @default Center
 * @option 靠上
 * @value Top
 * @option 置中
 * @value Center
 * @option 靠下
 * @value Bottom
 * 
 * @param InitLan Custom SelectY
 * @text 自訂位置Y軸
 * @desc 指定選擇項的Y軸座標。
 * @type number
 * @parent InitLan AlignTypeY
 * @min -9999
 * @default 250
 * 
 * @param InitLan Select Width
 * @text 選擇項寬度
 * @desc 指定選擇項的寬度。
 * @type number
 * @min 100
 * @default 250
 * 
 * @param InitLan Select Opacity
 * @text 選擇項不透明度
 * @desc 指定選擇項的不透明度。
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 * @param InitLan Select Align
 * @text 選擇項文字對齊
 * @desc 指定選擇項的文字對齊。
 * @type select
 * @default left
 * @option 靠左
 * @value left
 * @option 置中
 * @value center
 * @option 靠右
 * @value right
 * 
 * @param InitLan Help Boolean
 * @text 顯示語言說明
 * @desc 是否在遊戲初期時的顯示語言說明。
 * @type boolean
 * @on 顯示
 * @off 隱藏
 * @default true
 * 
 * @param InitLan AlignTypeHelpY
 * @text 說明Y軸對齊
 * @desc 指定說明的Y軸類型。
 * @type select
 * @default Auto
 * @option 自動對齊
 * @value Auto
 * @option 自訂位置
 * @value Custom
 * 
 * @param InitLan Auto SelectHelpY
 * @text 自動對齊Y軸
 * @desc 指定說明的Y軸對齊。
 * @type select
 * @parent InitLan AlignTypeHelpY
 * @default Top
 * @option 靠上
 * @value Top
 * @option 靠下
 * @value Bottom
 * 
 * @param InitLan Custom HelpY
 * @text 自訂位置Y軸
 * @desc 指定說明的Y軸座標。
 * @type number
 * @parent InitLan AlignTypeHelpY
 * @min -9999
 * @default 0
 * 
 * @param InitLan Help Opacity
 * @text 語言說明不透明度
 * @desc 指定說明的不透明度。
 * @type number
 * @min 0
 * @max 255
 * @default 255
 * 
 */
/*~struct~fonts:zh
 * 
 * @param Fonts File
 * @text 字型名稱
 * @desc 指定字型的檔案名稱，不包含副檔名。
 * @type string
 * @default mplus-1m-regular
 * 
 * @param Fonts Format
 * @text 指定字型格式
 * @desc 指定字型的副檔名格式。
 * @type select
 * @default ttf
 * @option 系統內建字型
 * @value local
 * @option OTF (OpenType Font)
 * @value otf
 * @option TTF (TrueType Font)
 * @value ttf
 * @option WOFF (Web Open Font Format)
 * @value woff
 * 
 */
//=============================================================================

let NekoGakuen_MulitLanguage = {};
let NekoGakuen_MulitLanguage_PluginName = PluginManager._scripts.includes(PluginManager.parameters("NekoGakuen_MulitLanguage")["Translate Patch"]) ? String(PluginManager.parameters("NekoGakuen_MulitLanguage")["Translate Patch"]) : "NekoGakuen_MulitLanguage";
NekoGakuen_MulitLanguage.Parameters = PluginManager.parameters(NekoGakuen_MulitLanguage_PluginName);
NekoGakuen_MulitLanguage = {
    Config_Lang_Boolean: String(NekoGakuen_MulitLanguage.Parameters['Config Lang Boolean'] || "true"),
    Config_Lang: String(NekoGakuen_MulitLanguage.Parameters['Config Lang'] || (navigator.language == 'zh-TW' ? "語言設定" : "Language Setting")),
    TitleText_Lang: String(NekoGakuen_MulitLanguage.Parameters['TitleText Lang'] || ""),
    Select_Lang_Type: String(NekoGakuen_MulitLanguage.Parameters['Select Lang Type'] || "Player Select"),
    Message_CommandName: String(NekoGakuen_MulitLanguage.Parameters['Message CommandName'] || "Say"),
    cfl: JSON.parse(NekoGakuen_MulitLanguage.Parameters['Custom Fontlist']),
    fonts_file: Array(),
    fonts_format: Array(),
    InitLang_Set: JSON.parse(NekoGakuen_MulitLanguage.Parameters['InitLang Set']),
    Custom_Langlist: JSON.parse(NekoGakuen_MulitLanguage.Parameters['Custom Langlist']),
    Lancsv_List: JSON.parse(NekoGakuen_MulitLanguage.Parameters['Lancsv List']),
    ConsoleError01: (navigator.language == 'zh-TW' ? '無法讀取CSV檔案。' : 'Cannot read CSV files.'),
    ConsoleError02: (navigator.language == 'zh-TW' ? '無法讀取圖片。' : 'Cannot read Pictures.'),
    ConsoleError03: (navigator.language == 'zh-TW' ? 'CSV 解析失敗：' : 'CSV parsing failed: '),
    ConsoleError04: (navigator.language == 'zh-TW' ? '無法載入CSV檔案：' : 'Failed to load CSV file: '),
    ConsoleError05: (navigator.language == 'zh-TW' ? '讀取 CSV 失敗：' : 'CSV read error: ')
};

let args_InitLanSet = NekoGakuen_MulitLanguage.InitLang_Set;
let args_LanNameList = NekoGakuen_MulitLanguage.Custom_Langlist;
let args_LancsvFileList = NekoGakuen_MulitLanguage.Lancsv_List;
let args_Lancsv1a = Array();
let args_Lancsv1b = Array();
let args_Lan2a = Array();
let args_Lan2b = Array();
let args_Lan2d = Array();
let args_Lan2e = Array();
let args_Lan2c = Array();
let args_LancsvPath, args_LanName;

if (Utils.RPGMAKER_NAME === "MZ") {
    (() => {

        //-----------------------------------------------------------------------------
        // ◆PluginManager (Add Features)
        // 
        //  為插件管理的靜態類別新增額外功能。
        //  Add additional features to the static class that manages the plugins.
        //-----------------------------------------------------------------------------

        PluginManager.isPlugins = function (pluginsName) {
            return this._scripts.includes(pluginsName);
        };

        PluginManager.isPlatformFlag = function () {
            return Utils.isNwjs() ? (typeof Utils.isElectronjs === "function" ? (Utils.isElectronjs() ? "Electronjs" : "Nwjs") : "Nwjs") : null;
        };

        PluginManager.isMobileMode = function () {
            return screen.width <= 1000;
        };

        PluginManager.fsModule = function () {
            return this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs" ? require('fs') : null;
        };

        PluginManager.pathModule = function () {
            return this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs" ? require('path') : null;
        };

        PluginManager.isFilePath = function () {
            return this.isPlatformFlag() == "Electronjs" ? this.pathModule().dirname(__filename) : this.pathModule().dirname(process.mainModule.filename);
        };

        PluginManager.checkFile = function (url) {
            if (this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs") {
                const filePath = this.pathModule().join(this.isFilePath(), url);
                if (this.fsModule().existsSync(filePath)) {
                    return true;
                } else {
                    return false;
                }
            } else {
                const xmlhttp = new XMLHttpRequest();
                xmlhttp.open("GET", url, false);
                xmlhttp.send(null);
                if (xmlhttp.readyState == 4) {
                    if (xmlhttp.status == 200) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        };

        PluginManager.convText = function (text) {
            text = text.replace(/\\/g, '\x1b');
            const testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            const regexs = new RegExp(`\\x1b${testVar}`, 'gi');
            if (regexs.test(text)) {
                const regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let text01 = '';
                text = text.replace(regex, (_, p1) => {
                    let args = (p1.replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        text01 = MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                    } else if (args[1]) {
                        if (args[1] == 'true') {
                            text01 = MulitLanguageArgs.getLangDataTextO(args[0]);
                        } else {
                            return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                        }
                    } else {
                        return MulitLanguageArgs.getLangDataTextO(args[0]);
                    }
                });
                return text01 ? text01 != undefined ? text01 : "" : text;
            } else {
                return text;
            }
        };

        PluginManager.registerCommand(NekoGakuen_MulitLanguage_PluginName, "NekoCommands MulitLang_SET", args => {

            $gameSystem.mulitLangSET(args.langTextArgs);
        });

        PluginManager.registerCommand(NekoGakuen_MulitLanguage_PluginName, "NekoCommands MulitLang_GET", args => {
            $gameSystem.mulitLangGET(args.varld);
        });


        for (let i = 0; i < args_LancsvFileList.length; i++) {
            args_LancsvPath = JSON.parse(args_LancsvFileList[i]);
            args_Lancsv1a.push(String(args_LancsvPath["Lancsv Name"]));
            args_Lancsv1b.push(String(args_LancsvPath["Lancsv Path"]));
        };

        for (let i = 0; i < args_LanNameList.length; i++) {
            args_LanName = JSON.parse(args_LanNameList[i]);
            args_Lan2a.push(String(args_LanName["Lang Key"]));
            args_Lan2b.push(String(args_LanName["Lang Name"]));
            args_Lan2d.push(String(args_LanName["Lang FontFace"]));
            args_Lan2e.push(Number(args_LanName["Lang FontSize"]));
            args_Lan2c.push(String(args_LanName["Init Help"]));
        };

        let args_Csvindex = args_Lancsv1b[0];
        let args_Lanindex = args_Lan2a[0];

        if (!PluginManager._csvCache) {
            PluginManager._csvCache = {};
        }

        if (!PluginManager._csvCache[args_Csvindex]) {
            let request = new XMLHttpRequest();
            request.open("GET", args_Csvindex, false);
            request.send(null);

            if (request.status === 200) {
                let jsonObject = request.responseText.split(/\r\n|\r/);
                let csvData = [];
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
                PluginManager._csvCache[args_Csvindex] = csvData;
            } else {
                console.error(NekoGakuen_MulitLanguage.ConsoleError01);
                return;
            }

            let csvData = PluginManager._csvCache[args_Csvindex];
            csvData = new Array();
            let jsonObject = request.responseText.split(/\r\n|\r/);
            if (PluginManager.checkFile(args_Csvindex)) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            } else {
                console.log(NekoGakuen_MulitLanguage.ConsoleError01);
            }
        }

        Graphics.localFont = function (name) {
            const style = document.createElement('style');
            const head = document.getElementsByTagName('head');
            const rule = '@font-face { font-family: "' + name + '"; src: local("' + name + '"); }';
            style.type = 'text/css';
            head.item(0).appendChild(style);
            style.sheet.insertRule(rule, 0);
            this._createFontLoader(name);
        };


        if (PluginManager.isPlugins("NekoGakuen_FontManager")) {
            NekoGakuen_MulitLanguage._Game_System_mainFontFace = Game_System.prototype.mainFontFace;
            Game_System.prototype.mainFontFace = function () {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2d[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Game_System_mainFontFace.call(this);
                }
            };

            NekoGakuen_MulitLanguage._Game_System_mainFontSize = Game_System.prototype.mainFontSize;
            Game_System.prototype.mainFontSize = function () {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2e[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Game_System_mainFontSize.call(this);
                }
            };
        } else {
            for (let i = 0; i < NekoGakuen_MulitLanguage.cfl.length; i++) {
                let Read_FontManager = JSON.parse(NekoGakuen_MulitLanguage.cfl[i]);
                NekoGakuen_MulitLanguage.fonts_file.push(Read_FontManager["Fonts File"]);
                NekoGakuen_MulitLanguage.fonts_format.push(Read_FontManager["Fonts Format"]);
            }

            Scene_Boot.prototype.loadGameFonts = function () {
                const advanced = $dataSystem.advanced;
                FontManager.load("rmmz-mainfont", advanced.mainFontFilename);
                FontManager.load("rmmz-numberfont", advanced.numberFontFilename);
                for (let i = 0; i < NekoGakuen_MulitLanguage.cfl.length; ++i) {
                    let filename = NekoGakuen_MulitLanguage.fonts_file[i].trim();
                    if (NekoGakuen_MulitLanguage.fonts_format[i] != 'local') {
                        const url = './fonts/' + filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i];
                        if (PluginManager.checkFile(url)) {
                            FontManager.load(filename, filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i]);
                        }
                    } else {
                        Graphics.localFont(filename);
                    }
                }
            };

            NekoGakuen_MulitLanguage._Game_System_mainFontFace = Game_System.prototype.mainFontFace;
            Game_System.prototype.mainFontFace = function () {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2d[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Game_System_mainFontFace.call(this);
                }
            };

            NekoGakuen_MulitLanguage._Game_System_mainFontSize = Game_System.prototype.mainFontSize;
            Game_System.prototype.mainFontSize = function () {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2e[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Game_System_mainFontSize.call(this);
                }
            };
        }

        NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
        Window_Base.prototype.convertEscapeCharacters = function (text) {
            let content = NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters.call(this, text);
            const testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            const regexs = new RegExp(`\\\\${testVar}`, 'gi');
            if (regexs.test(text)) {
                const regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let content01 = '';
                content = content.replace(regex, (_, p1) => {
                    let args = (p1.replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        content01 = this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                    } else if (args[1]) {
                        if (args[1] == 'true') {
                            content01 = this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                        } else {
                            return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                        }
                    } else {
                        return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                    }
                });
                return content01 ? content01 : content;
            } else {
                return content;
            }
        };


        Game_Interpreter.prototype.checkTextByData = function (text) {
            text = text.replace(/\\/g, '\x1b');
            const testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            const regexs = new RegExp(`\\x1b${testVar}`, 'gi');
            if (regexs.test(text)) {
                const regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let text01 = '';
                text = text.replace(regex, (_, p1) => {
                    let args = (p1.replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        text01 = MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                    } else if (args[1]) {
                        if (args[1] == 'true') {
                            text01 = MulitLanguageArgs.getLangDataTextO(args[0]);
                        } else {
                            return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                        }
                    } else {
                        return MulitLanguageArgs.getLangDataTextO(args[0]);
                    }
                });
                return text01 ? text01 : text;
            } else {
                return text;
            }
        };

        Game_Interpreter.prototype.setupChoices = function (params) {
            const choices = params[0].clone();
            for (let i = 0; i < choices.length; i++) {
                choices[i] = this.checkTextByData(choices[i]);
            }
            const cancelType = params[1] < choices.length ? params[1] : -2;
            const defaultType = params.length > 2 ? params[2] : 0;
            const positionType = params.length > 3 ? params[3] : 2;
            const background = params.length > 4 ? params[4] : 0;
            $gameMessage.setChoices(choices, defaultType, cancelType);
            $gameMessage.setChoiceBackground(background);
            $gameMessage.setChoicePositionType(positionType);
            $gameMessage.setChoiceCallback(n => {
                this._branch[this._indent] = n;
            });
        };


        TextManager.basic = function (basicId) {
            return PluginManager.convText($dataSystem.terms.basic[basicId]) || '';
        };

        TextManager.param = function (paramId) {
            return PluginManager.convText($dataSystem.terms.params[paramId]) || '';
        };

        TextManager.command = function (commandId) {
            return PluginManager.convText($dataSystem.terms.commands[commandId]) || '';
        };

        TextManager.message = function (messageId) {
            return PluginManager.convText($dataSystem.terms.messages[messageId]) || '';
        };


        Window_BattleLog.prototype.displayAddedStates = function (target) {
            const result = target.result();
            const states = result.addedStateObjects();
            for (const state of states) {
                const stateText = target.isActor() ? PluginManager.convText(state.message1) : PluginManager.convText(state.message2);
                if (state.id === target.deathStateId()) {
                    this.push("performCollapse", target);
                }
                if (stateText) {
                    this.push("popBaseLine");
                    this.push("pushBaseLine");
                    this.push("addText", stateText.format(target.name()));
                    this.push("waitForEffect");
                }
            }
        };


        Game_Actor.prototype.displayLevelUp = function (newSkills) {
            const text = TextManager.levelUp.format(
                PluginManager.convText(this._name),
                TextManager.level,
                this._level
            );
            $gameMessage.newPage();
            $gameMessage.add(text);
            for (const skill of newSkills) {
                $gameMessage.add(TextManager.obtainSkill.format(skill.name));
            }
        };

        BattleManager.displayDropItems = function () {
            const items = this._rewards.items;
            if (items.length > 0) {
                $gameMessage.newPage();
                for (const item of items) {
                    $gameMessage.add(TextManager.obtainItem.format(PluginManager.convText(item.name)));
                }
            }
        };

        Object.defineProperty(TextManager, "currencyUnit", {
            get: function () {
                return PluginManager.convText($dataSystem.currencyUnit);
            },
            configurable: true
        });


        Game_Actor.prototype.name = function () {
            return PluginManager.convText(this._name);
        };

        Game_Actor.prototype.nickname = function () {
            return PluginManager.convText(this._nickname);
        };

        Game_Actor.prototype.profile = function () {
            return PluginManager.convText(this._profile);
        };


        Window_Base.prototype.drawItemName = function (item, x, y, width) {
            if (item) {
                const iconY = y + (this.lineHeight() - ImageManager.iconHeight) / 2;
                const textMargin = ImageManager.iconWidth + 4;
                const itemWidth = Math.max(0, width - textMargin);
                this.resetTextColor();
                this.drawIcon(item.iconIndex, x, iconY);
                this.drawText(PluginManager.convText(item.name), x + textMargin, y, itemWidth);
            }
        };


        Window_Help.prototype.refresh = function () {
            const rect = this.baseTextRect();
            this.contents.clear();
            this.drawTextEx(PluginManager.convText(this._text), rect.x, rect.y, rect.width);
        };


        Window_SkillType.prototype.makeCommandList = function () {
            if (this._actor) {
                const skillTypes = this._actor.skillTypes();
                for (const stypeId of skillTypes) {
                    const name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                    this.addCommand(name, "skill", true, stypeId);
                }
            }
        };


        Window_ActorCommand.prototype.addSkillCommands = function () {
            const skillTypes = this._actor.skillTypes();
            for (const stypeId of skillTypes) {
                const name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                this.addCommand(name, "skill", true, stypeId);
            }
        };


        Window_BattleLog.prototype.displayItemMessage = function (fmt, subject, item) {
            if (fmt) {
                this.push("addText", PluginManager.convText(fmt).format(subject.name(), PluginManager.convText(item.name)));
            }
        };


        Window_StatusBase.prototype.drawActorClass = function (actor, x, y, width) {
            width = width || 168;
            this.resetTextColor();
            this.drawText(PluginManager.convText(actor.currentClass().name), x, y, width);
        };

        Window_StatusBase.prototype.actorSlotName = function (actor, index) {
            const slots = actor.equipSlots();
            return PluginManager.convText($dataSystem.equipTypes[slots[index]]);
        };


        Game_Enemy.prototype.battlerName = function () {
            return PluginManager.convText(this.enemy().battlerName);
        };

        Game_Enemy.prototype.originalName = function () {
            return PluginManager.convText(this.enemy().name);
        };


        Game_Map.prototype.displayName = function () {
            return PluginManager.convText($dataMap.displayName);
        };


        NekoGakuen_MulitLanguage._Scene_Boot_start = Scene_Boot.prototype.start;
        Scene_Boot.prototype.start = async function () {
            await MulitLanguageArgs.preloadCsvData();
            NekoGakuen_MulitLanguage._Scene_Boot_start.call(this);
            if (PluginManager.isPlatformFlag() == "Nwjs") {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !PluginManager.fsModule().existsSync(PluginManager.pathModule().join(PluginManager.isFilePath(), 'save/config.rmmzsave'))) {
                    ConfigManager.language = "init";
                }
            } else if (PluginManager.isPlatformFlag() == "Electronjs") {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !PluginManager.fsModule().existsSync(Utils.isOptionValid("test") ? PluginManager.pathModule().join(PluginManager.isFilePath(), 'save/config.rmmzsave') : PluginManager.pathModule().join(PluginManager.isFilePath(), '../../save/config.rmmzsave'))) {
                    ConfigManager.language = "init";
                }
            } else {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !StorageManager.forageExists('config')) {
                    ConfigManager.language = "init";
                }
            }
            if (ConfigManager.language == "init") {
                if (!DataManager.isBattleTest() && !DataManager.isEventTest()) {
                    SceneManager.goto(Scene_InitialLanguage);
                }
            }
        };

        Scene_Boot.prototype.updateDocumentTitle = function () {
            document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
        };

        NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages = Scene_Boot.prototype.loadSystemImages;
        Scene_Boot.prototype.loadSystemImages = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages.call(this);
            ImageManager.loadPicture(args_InitLanSet["InitLan Images"]);
        };


        NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle = Scene_Title.prototype.drawGameTitle;
        Scene_Title.prototype.drawGameTitle = function () {
            $dataSystem.gameTitle = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
            if (PluginManager.isPlugins("NekoGakuen_FontManager")) {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
                } else {
                    this._gameTitleSprite.bitmap.fontFace = $gameSystem.mainFontFace();
                }
            } else {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
                } else {
                    this._gameTitleSprite.bitmap.fontFace = $gameSystem.mainFontFace();
                }
            }
            NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle.call(this);
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_LanguageSelect
        // 
        //  在遊戲初始化前選擇遊戲語言的視窗。
        //  The window to select the game language before the game initialization.
        //-----------------------------------------------------------------------------

        function Window_LanguageSelect() {
            this.initialize(...arguments);
        };

        Window_LanguageSelect.prototype = Object.create(Window_Selectable.prototype);
        Window_LanguageSelect.prototype.constructor = Window_LanguageSelect;

        Window_LanguageSelect.prototype.initialize = function (rect) {
            Window_Selectable.prototype.initialize.call(this, rect);
            this.opacity = Number(args_InitLanSet["InitLan Select Opacity"] || 255);
            this.width = this.windowWidth();
            this.height = this.windowHeight();
            this.setTopRow(0);
            this.select(0);
            this.activate();
            this.refresh();
        };

        Window_LanguageSelect.prototype.maxCols = function () {
            return 1;
        };

        Window_LanguageSelect.prototype.colSpacing = function () {
            return 16;
        };

        Window_LanguageSelect.prototype.maxItems = function () {
            return args_LanNameList ? args_LanNameList.length : 1;
        };

        Window_LanguageSelect.prototype.itemHeight = function () {
            return PluginManager.isMobileMode() ? 60 : Window_Scrollable.prototype.itemHeight.call(this) + 8;
        };

        Window_LanguageSelect.prototype.lineHeight = function () {
            return PluginManager.isMobileMode() ? 60 : Window_Base.prototype.lineHeight.call(this);
        };

        Window_LanguageSelect.prototype.windowWidth = function () {
            return Number(args_InitLanSet["InitLan Select Width"] || 250);
        };

        Window_LanguageSelect.prototype.windowHeight = function () {
            return this.fittingHeight(this.numVisibleRows());
        };

        Window_LanguageSelect.prototype.drawItem = function (index) {
            const name = this.itemName(index);
            const rect = this.itemLineRect(index);
            this.resetTextColor();
            this.drawText(name, rect.x, rect.y, rect.width, String(args_InitLanSet["InitLan Select Align"] || "left"));
        };

        Window_LanguageSelect.prototype.numVisibleRows = function () {
            return args_LanNameList.length;
        };

        Window_LanguageSelect.prototype.itemName = function (value) {
            return args_Lan2b[value];
        };

        Window_LanguageSelect.prototype.item = function () {
            return this.itemAt(this.index());
        };

        Window_LanguageSelect.prototype.itemAt = function (index) {
            return args_LanNameList ? this.itemName(index) : null;
        };

        Window_LanguageSelect.prototype.refresh = function () {
            Window_Selectable.prototype.refresh.call(this);
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_LanguageHelp
        // 
        //  在遊戲初始化前顯示說明文字的視窗。
        //  The window that Show the description text before the game initializes.
        //-----------------------------------------------------------------------------  

        function Window_LanguageHelp() {
            this.initialize(...arguments);
        };

        Window_LanguageHelp.prototype = Object.create(Window_Base.prototype);
        Window_LanguageHelp.prototype.constructor = Window_LanguageHelp;

        Window_LanguageHelp.prototype.initialize = function (rect) {
            Window_Base.prototype.initialize.call(this, rect);
            this._text = "";
            this.opacity = Number(args_InitLanSet["InitLan Help Opacity"] || 255);
        };

        Window_LanguageHelp.prototype.setText = function (text) {
            if (this._text !== text) {
                this._text = text;
                this.refresh();
            }
        };

        Window_LanguageHelp.prototype.clear = function () {
            this.setText("");
        };

        Window_LanguageHelp.prototype.setItem = function (value) {
            this.setText(args_Lan2c[value]);
        };

        Window_LanguageHelp.prototype.refresh = function () {
            const rect = this.baseTextRect();
            this.contents.clear();
            this.drawTextEx(this._text, rect.x, rect.y, rect.width);
        };


        //-----------------------------------------------------------------------------
        // ◆ Scene_InitialLanguage
        // 
        //  語言選擇畫面的場景類別。
        //  The scene class of the language select screen.
        //-----------------------------------------------------------------------------

        function Scene_InitialLanguage() {
            this.initialize(...arguments);
        };

        Scene_InitialLanguage.prototype = Object.create(Scene_Base.prototype);
        Scene_InitialLanguage.prototype.constructor = Scene_InitialLanguage;

        Scene_InitialLanguage.prototype.initialize = function () {
            Scene_Base.prototype.initialize.call(this);
        };

        Scene_InitialLanguage.prototype.create = function () {
            Scene_Base.prototype.create.call(this);
            this.createBackground();
            this.createHelpWindow();
            this.createLanguageSelectWindow();
        };

        Scene_InitialLanguage.prototype.createBackground = function () {
            this._background = new Sprite(ImageManager.loadPicture(args_InitLanSet["InitLan Images"]));
            this.addChild(this._background);
            this.scaleSprite(this._background);
            this.centerSprite(this._background);
        };

        Scene_InitialLanguage.prototype.langSelectWindowRect = function () {
            let ww = Number(args_InitLanSet["InitLan Select Width"] || 250);
            let wh = this.calcWindowHeight(args_LanNameList.length, false);
            let wx = 0;
            let wy = 0;
            if (args_InitLanSet["InitLan AlignTypeX"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Left') {
                    wx = 4;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Center') {

                    wx = (Graphics.boxWidth - ww) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Right') {
                    wx = Graphics.boxWidth - ww + 4;
                }
            } else {
                wx = Number(args_InitLanSet["InitLan Custom SelectX"] || 283);
            }
            if (args_InitLanSet["InitLan AlignTypeY"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Top') {
                    if (String(args_InitLanSet["InitLan Help Boolean"]) == "true") {
                        wy = (this._langhelpWindow.y >= this._langhelpWindow.height ? 4 : this._langhelpWindow.height + 8);
                    } else {
                        wy = 4;
                    }
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Center') {
                    wy = (Graphics.boxHeight - wh) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Bottom') {
                    if (String(args_InitLanSet["InitLan Help Boolean"]) == "true") {
                        wy = (this._langhelpWindow.y >= Graphics.boxHeight - this._langhelpWindow.height ? Graphics.boxHeight - wh - this._langhelpWindow.height - 4 : Graphics.boxHeight - wh - 20);
                    } else {
                        wy = Graphics.boxHeight - wh - 20;
                    }
                }
            } else {
                wy = Number(args_InitLanSet["InitLan Custom SelectY"] || 250);
            }
            return new Rectangle(wx, wy, ww, wh);
        };

        Scene_InitialLanguage.prototype.needsCancelButton = function () {
            return false;
        };

        Scene_InitialLanguage.prototype.createLanguageSelectWindow = function () {
            const rect = this.langSelectWindowRect();
            this._langSelectWindow = new Window_LanguageSelect(rect);
            this._langSelectWindow.setHandler("ok", this.onLangSetOk.bind(this));
            this.addChild(this._langSelectWindow);
        };

        Scene_InitialLanguage.prototype.createHelpWindow = function () {
            const rect = this.helpWindowRect();
            this._langhelpWindow = new Window_LanguageHelp(rect);
            this.addChild(this._langhelpWindow);
            if (String(args_InitLanSet["InitLan Help Boolean"]) != "true") {
                this._langhelpWindow.hide();
            }
        };

        Scene_InitialLanguage.prototype.helpWindowRect = function () {
            const ww = Graphics.boxWidth;
            const wh = this.calcWindowHeight(1, false);
            const wx = 4;
            const wy = (args_InitLanSet["InitLan AlignTypeHelpY"] == "Auto" ? (args_InitLanSet["InitLan Auto SelectHelpY"] == 'Top' ? 0 : Graphics.boxHeight - wh) : Number(args_InitLanSet["InitLan Custom HelpY"])) + 4;
            return new Rectangle(wx, wy, ww, wh);
        };

        Scene_InitialLanguage.prototype.update = function () {
            Scene_Base.prototype.update.call(this);
            this._lastSelect = this._langSelectWindow.index();
            this._langhelpWindow.setItem(this._lastSelect);
        };

        Scene_InitialLanguage.prototype.onLangSetOk = function () {
            const langset = args_Lan2a[this._langSelectWindow.index()]
            ConfigManager.language = this._langSelectWindow.index();
            ConfigManager.save();
            MulitLanguageArgs.setLangData(String(langset));
            this._langSelectWindow.close();
            this.fadeOutAll();
            SceneManager.goto(Scene_Boot);
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_Options (Add Features & Modify Features)
        // 
        //  為選項畫面新增額外功能和修改功能。
        //  Add additional features and modifications on the options screen.
        //-----------------------------------------------------------------------------

        NekoGakuen_MulitLanguage._Window_Options_statusText = Window_Options.prototype.statusText;
        Window_Options.prototype.statusText = function (index) {
            const symbol = this.commandSymbol(index);
            const value = this.getConfigValue(symbol);
            if (this.isLanguageSymbol(symbol)) {
                return this.langSymbol(value);
            } else {
                return NekoGakuen_MulitLanguage._Window_Options_statusText.call(this, index);
            }
        };

        Window_Options.prototype.isLanguageSymbol = function (symbol) {
            return symbol.contains('language');
        };

        Window_Options.prototype.langSymbol = function (value) {
            return args_Lan2b[value];
        };

        if (NekoGakuen_MulitLanguage.Config_Lang_Boolean == "true") {
            NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
            Window_Options.prototype.addGeneralOptions = function () {
                this.addCommand(PluginManager.convText(NekoGakuen_MulitLanguage.Config_Lang), 'language');
                NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions.call(this);
            };
        }

        Window_Options.prototype.processOk = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, true, true);
            } else {
                this.changeValue(symbol, !this.getConfigValue(symbol));
            }
            if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            }
        };

        Window_Options.prototype.langOffset = function () {
            return 1;
        };

        Window_Options.prototype.cursorLeft = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, false, false);
            } else {
                this.changeValue(symbol, false);
            }
            if (this.isLanguageSymbol(symbol)) {
                value -= this.langOffset();
                let config = args_Lan2a;
                if (value < 0) {
                    value = config.length - 1;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
                return false;
            }
        };

        Window_Options.prototype.cursorRight = function () {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            let value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                this.changeVolume(symbol, true, false);
            } else {
                this.changeValue(symbol, true);
            }
            if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
                return false;
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ Scene_Options (Add Features & Modify Features)
        // 
        //  為選項畫面新增額外功能和修改功能。
        //  Add additional features and modifications on the options screen.
        //-----------------------------------------------------------------------------

        NekoGakuen_MulitLanguage._Scene_Options_maxCommands = Scene_Options.prototype.maxCommands;
        Scene_Options.prototype.maxCommands = function () {
            return NekoGakuen_MulitLanguage._Scene_Options_maxCommands.call(this) + (NekoGakuen_MulitLanguage.Config_Lang_Boolean == "true" ? 1 : 0);
        };

        NekoGakuen_MulitLanguage._Scene_Options_maxVisibleCommands = Scene_Options.prototype.maxVisibleCommands;
        Scene_Options.prototype.maxVisibleCommands = function () {
            return NekoGakuen_MulitLanguage._Scene_Options_maxVisibleCommands.call(this) + (NekoGakuen_MulitLanguage.Config_Lang_Boolean == "true" ? 1 : 0);
        };


        //-----------------------------------------------------------------------------
        // ◆ ConfigManager (Add Features & Modify Features)
        // 
        //  為靜態類別新增額外功能和修改功能。
        //  Add additional features and modifications to static class.
        //-----------------------------------------------------------------------------

        if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Player Select") {
            ConfigManager.language = 0;
        } else {
            if (PluginManager.isPlugins("NekoGakuen_SteamworksAPI")) {
                if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Steam Client") {
                    ConfigManager.language = args_Lan2a.indexOf(SteamworksAPIManager.getCurrentUILanguage());
                }
                if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Steam Game") {
                    ConfigManager.language = args_Lan2a.indexOf(SteamworksAPIManager.getCurrentGameLanguage());
                }
            } else {
                ConfigManager.language = 0;
            }
        }

        NekoGakuen_MulitLanguage._configManager_makeData = ConfigManager.makeData;
        ConfigManager.makeData = function () {
            const config = NekoGakuen_MulitLanguage._configManager_makeData.call(this);
            config.language = this.language;
            return config;
        };

        NekoGakuen_MulitLanguage._configManager_applyData = ConfigManager.applyData;
        ConfigManager.applyData = function (config) {
            NekoGakuen_MulitLanguage._configManager_applyData.call(this, config);
            this.language = this.readLanguage(config, 'language');
            args_Lanindex = args_Lan2a[this.language];
        };

        ConfigManager.readLanguage = function (config, name) {
            if (name in config) {
                return Number(config[name]).clamp(0, args_LanNameList.length - 1);
            } else {
                return 0;
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ Game_System (Add Features)
        // 
        //  為遊戲物件類別新增額外功能。
        //  Add additional features to the game object class.
        //-----------------------------------------------------------------------------

        Game_System.prototype.mulitLangSET = function (lanargs) {
            ConfigManager.language = args_Lan2a.indexOf(lanargs);
            ConfigManager.save();
            MulitLanguageArgs.setLangData(String(lanargs));
        };

        Game_System.prototype.mulitLangGET = function (variablesId) {
            if (variablesId) {
                $gameVariables.setValue(Number(variablesId), String(args_Lanindex));
            } else {
                return args_Lanindex;
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ MulitLanguageArgs
        // 
        //  管理多語言功能的靜態類別。
        //  The static class that manages multi-language features.
        //-----------------------------------------------------------------------------

        function MulitLanguageArgs() {
            throw new Error('This is a static class');
        }
        
        let preloadedCsvData = {};
        MulitLanguageArgs.setLangData = function (lanindex) {
            const index = args_Lan2a.indexOf(lanindex);
            if (index !== -1) {
                args_Lanindex = args_Lan2a[index];
            }
        };
        
        MulitLanguageArgs.loadCsvAsync = function (filePath) {
            return new Promise((resolve, reject) => {
                const request = new XMLHttpRequest();
                request.open("GET", filePath, true);
                request.onload = function () {
                    if (request.status >= 200 && request.status < 300) {
                        try {
                            const parsedData = MulitLanguageArgs.parseCsv(request.responseText);
                            resolve(parsedData);
                        } catch (error) {
                            console.error(`${NekoGakuen_MulitLanguage.ConsoleError03}${filePath}`, error);
                            resolve([]);
                        }
                    } else {
                        console.error(`${NekoGakuen_MulitLanguage.ConsoleError04}${filePath}`);
                        resolve([]);
                    }
                };
                request.onerror = () => reject(`${NekoGakuen_MulitLanguage.ConsoleError05}${filePath}`);
                request.send();
            });
        };
        
        MulitLanguageArgs.parseCsv = function (content) {
            const rows = [];
            let currentRow = [];
            let insideQuotes = false;
            let cell = "";
            for (let char of content) {
                if (char === '"') {
                    insideQuotes = !insideQuotes;
                } else if (char === ',' && !insideQuotes) {
                    currentRow.push(cell.trim());
                    cell = "";
                } else if (char === '\n' && !insideQuotes) {
                    currentRow.push(cell.trim());
                    rows.push(currentRow);
                    currentRow = [];
                    cell = "";
                } else {
                    cell += char;
                }
            }
            if (cell.length > 0) {
                currentRow.push(cell.trim());
            }
            if (currentRow.length > 0) {
                rows.push(currentRow);
            }
        
            return rows;
        };
        
        MulitLanguageArgs.preloadCsvData = async function () {
            const promises = args_LancsvFileList.map(async fileInfo => {
                const fileData = JSON.parse(fileInfo);
                const csvFilePath = fileData["Lancsv Path"];
                preloadedCsvData[csvFilePath] = await MulitLanguageArgs.loadCsvAsync(csvFilePath);
            });
            await Promise.all(promises);
        };
        
        MulitLanguageArgs.getLangDataTextS = function (textArgs, csvName) {
            const csvPath = MulitLanguageArgs.getCsvPath(csvName);
            const csvArray = preloadedCsvData[csvPath] || [];
            const rowIndex = csvArray.findIndex(row => row[0] === textArgs);
        
            if (rowIndex > 0) {
                const columnIndex = csvArray[0].indexOf(args_Lanindex);
                let text = csvArray[rowIndex][columnIndex] || '';
                return text.replace(/^\"|\"$/g, '');
            }
            return '';
        };
        
        // 取得預設CSV中的文字 (getLangDataTextO)
        MulitLanguageArgs.getLangDataTextO = function (textArgs) {
            const csvPath = args_Lancsv1b[0]; 
            const csvArray = preloadedCsvData[csvPath] || [];
            const rowIndex = csvArray.findIndex(row => row[0] === textArgs);
        
            if (rowIndex > 0) {
                const columnIndex = csvArray[0].indexOf(args_Lanindex);
                let text = csvArray[rowIndex][columnIndex] || '';
                return text.replace(/^\"|\"$/g, '').replace(/\\/g, '\x1b'); 
            }
            return '';
        };
        
        // 取得CSV檔案路徑
        MulitLanguageArgs.getCsvPath = function (csvName) {
            const fileInfo = args_LancsvFileList.find(info => JSON.parse(info)["Lancsv Name"] === csvName);
            return fileInfo ? JSON.parse(fileInfo)["Lancsv Path"] : '';
        };
        
    })();
}

if (Utils.RPGMAKER_NAME === "MV") {
    (function () {

        //-----------------------------------------------------------------------------
        // ◆PluginManager (Add Features)
        // 
        //  為插件管理的靜態類別新增額外功能。
        //  Add additional features to the static class that manages the plugins.
        //-----------------------------------------------------------------------------

        PluginManager.isPlugins = function (pluginsName) {
            return this._scripts.includes(pluginsName);
        };

        PluginManager.isPlatformFlag = function () {
            return Utils.isNwjs() ? (typeof Utils.isElectronjs === "function" ? (Utils.isElectronjs() ? "Electronjs" : "Nwjs") : "Nwjs") : null;
        };

        PluginManager.isMobileMode = function () {
            return Utils.isMobileDevice() && PluginManager.isPlugins("TouchUI");
        };

        PluginManager.fsModule = function () {
            return this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs" ? require('fs') : null;
        };

        PluginManager.pathModule = function () {
            return this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs" ? require('path') : null;
        };

        PluginManager.isFilePath = function () {
            return this.isPlatformFlag() == "Electronjs" ? this.pathModule().dirname(__filename) : this.pathModule().dirname(process.mainModule.filename);
        };

        PluginManager.checkFile = function (url) {
            if (this.isPlatformFlag() == "Electronjs" || this.isPlatformFlag() == "Nwjs") {
                var filePath = this.pathModule().join(this.isFilePath(), url);
                if (this.fsModule().existsSync(filePath)) {
                    return true;
                } else {
                    return false;
                }
            } else {
                var xmlhttp = new XMLHttpRequest();
                xmlhttp.open("GET", url, false);
                xmlhttp.send(null);
                if (xmlhttp.readyState == 4) {
                    if (xmlhttp.status == 200) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        };

        PluginManager.convText = function (text) {
            text = text.replace(/\\/g, '\x1b');
            var testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            var regexs = new RegExp(`\\x1b${testVar}`, 'gi');
            if (regexs.test(text)) {
                var regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let text01 = '';
                text = text.replace(regex, function () {
                    var args = (arguments[1].replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        text01 = MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                    } else if (args[1]) {
                        if (args[1] == 'true') {
                            text01 = MulitLanguageArgs.getLangDataTextO(args[0]);
                        } else {
                            return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                        }
                    } else {
                        return MulitLanguageArgs.getLangDataTextO(args[0]);
                    }
                }.bind(this));
                return text01 ? text01 != undefined ? text01 : "" : text;
            } else {
                return text;
            }
        };


        for (let i = 0; i < args_LancsvFileList.length; i++) {
            args_LancsvPath = JSON.parse(args_LancsvFileList[i]);
            args_Lancsv1a.push(String(args_LancsvPath["Lancsv Name"]));
            args_Lancsv1b.push(String(args_LancsvPath["Lancsv Path"]));
        };

        for (let i = 0; i < args_LanNameList.length; i++) {
            args_LanName = JSON.parse(args_LanNameList[i]);
            args_Lan2a.push(String(args_LanName["Lang Key"]));
            args_Lan2b.push(String(args_LanName["Lang Name"]));
            args_Lan2d.push(String(args_LanName["Lang FontFace"]));
            args_Lan2e.push(Number(args_LanName["Lang FontSize"]));
            args_Lan2c.push(String(args_LanName["Init Help"]));
        };

        let args_Csvindex = args_Lancsv1b[0];
        let args_Lanindex = args_Lan2a[0];
        if (!PluginManager._csvCache) {
            PluginManager._csvCache = {};
        }

        if (!PluginManager._csvCache[args_Csvindex]) {
            let request = new XMLHttpRequest();
            request.open("GET", args_Csvindex, false);
            request.send(null);

            if (request.status === 200) {
                let jsonObject = request.responseText.split(/\r\n|\r/);
                let csvData = [];
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
                PluginManager._csvCache[args_Csvindex] = csvData;
            } else {
                console.error(NekoGakuen_MulitLanguage.ConsoleError01);
                return;
            }

            let csvData = PluginManager._csvCache[args_Csvindex];
            csvData = new Array();
            let jsonObject = request.responseText.split(/\r\n|\r/);
            if (PluginManager.checkFile(args_Csvindex)) {
                for (let i = 0; i < jsonObject.length; i++) {
                    jsonObject[i] = jsonObject[i].replace(/,\s*(?=([^"]*"[^"]*")*[^"]*$)/g, ';;;');
                    csvData.push(jsonObject[i].split(';;;'));
                }
            } else {
                console.log(NekoGakuen_MulitLanguage.ConsoleError01);
            }
        }

        Graphics.localFont = function (name) {
            var style = document.createElement('style');
            var head = document.getElementsByTagName('head');
            var rule = '@font-face { font-family: "' + name + '"; src: local("' + name + '"); }';
            style.type = 'text/css';
            head.item(0).appendChild(style);
            style.sheet.insertRule(rule, 0);
            this._createFontLoader(name);
        };


        if (PluginManager.isPlugins("NekoGakuen_FontManager")) {
            NekoGakuen_MulitLanguage._Window_Base_standardFontFace = Window_Base.prototype.standardFontFace;
            Window_Base.prototype.standardFontFace = function () {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2d[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Window_Base_standardFontFace.call(this);
                }
            };

            NekoGakuen_MulitLanguage._Window_Base_standardFontSize = Window_Base.prototype.standardFontSize;
            Window_Base.prototype.standardFontSize = function () {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2e[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Window_Base_standardFontSize.call(this);
                }
            };
        } else {
            for (var i = 0; i < NekoGakuen_MulitLanguage.cfl.length; i++) {
                var Read_FontManager = JSON.parse(NekoGakuen_MulitLanguage.cfl[i]);
                NekoGakuen_MulitLanguage.fonts_file.push(Read_FontManager["Fonts File"]);
                NekoGakuen_MulitLanguage.fonts_format.push(Read_FontManager["Fonts Format"]);
            };

            for (var i = 0; i < NekoGakuen_MulitLanguage.cfl.length; ++i) {
                var filename = NekoGakuen_MulitLanguage.fonts_file[i].trim();
                if (NekoGakuen_MulitLanguage.cfl.length > 0) {
                    if (NekoGakuen_MulitLanguage.fonts_format != 'local') {
                        Graphics.loadFont(filename, './fonts/' + filename + '.' + NekoGakuen_MulitLanguage.fonts_format[i]);
                    } else {
                        Graphics.localFont(filename);
                    }
                }
            };

            NekoGakuen_MulitLanguage._Window_Base_standardFontFace = Window_Base.prototype.standardFontFace;
            Window_Base.prototype.standardFontFace = function () {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2d[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Window_Base_standardFontFace.call(this);
                }
            };

            NekoGakuen_MulitLanguage._Window_Base_standardFontSize = Window_Base.prototype.standardFontSize;
            Window_Base.prototype.standardFontSize = function () {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    return args_Lan2e[ConfigManager.language];
                } else {
                    return NekoGakuen_MulitLanguage._Window_Base_standardFontSize.call(this);
                }
            };
        };


        NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
        Window_Base.prototype.convertEscapeCharacters = function (text) {
            let content = NekoGakuen_MulitLanguage._Window_Base_convertEscapeCharacters.call(this, text);
            var testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            var regexs = new RegExp(`\\\\${testVar}`, 'gi');
            if (regexs.test(text)) {
                var regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let content01 = '';
                content = content.replace(regex, function () {
                    var args = (arguments[1].replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        content01 = this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                    } else if (args[1]) {
                        if (args[1] == 'true') {

                            content01 = this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                        } else {
                            return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextS(args[0], args[1]));
                        }
                    } else {
                        return this.convertEscapeCharacters(MulitLanguageArgs.getLangDataTextO(args[0]));
                    }
                }.bind(this));
                return content01 ? content01 : content;
            } else {
                return content;
            }
        };


        NekoGakuen_MulitLanguage._Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
        Game_Interpreter.prototype.pluginCommand = function (command, args) {
            NekoGakuen_MulitLanguage._Game_Interpreter_pluginCommand.call(this, command, args);
            if (command === 'MulitLang') {
                switch (args[0]) {
                    case 'SET':
                        $gameSystem.mulitLangSET(args[1]);
                        break;
                    case 'GET':
                        $gameSystem.mulitLangGET(args[1]);
                        break;
                }
            }
            if (command === 'NekoCommands') {
                switch (args[0]) {
                    case 'MulitLang_SET':
                        $gameSystem.mulitLangSET(args[1]);
                        break;
                    case 'MulitLang_GET':
                        $gameSystem.mulitLangGET(args[1]);
                        break;
                }
            }
        };

        Game_Interpreter.prototype.checkTextByData = function (text) {
            text = text.replace(/\\/g, '\x1b');
            var testVar = NekoGakuen_MulitLanguage.Message_CommandName;
            var regexs = new RegExp(`\\x1b${testVar}`, 'gi');
            if (regexs.test(text)) {
                var regex = new RegExp(`\\x1b${testVar}\\[(.*?)\\]`, 'gi');
                let text01 = '';
                text = text.replace(regex, function () {
                    var args = (arguments[1].replace(/\s/g, '')).split(",");
                    if (args[2] == 'true') {
                        text01 = MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                    } else if (args[1]) {
                        if (args[1] == 'true') {
                            text01 = MulitLanguageArgs.getLangDataTextO(args[0]);
                        } else {
                            return MulitLanguageArgs.getLangDataTextS(args[0], args[1]);
                        }
                    } else {
                        return MulitLanguageArgs.getLangDataTextO(args[0]);
                    }
                }.bind(this));
                return text01 ? text01 : text;
            } else {
                return text;
            }
        };

        Game_Interpreter.prototype.setupChoices = function (params) {
            var choices = params[0].clone();
            for (let i = 0; i < choices.length; i++) {
                choices[i] = this.checkTextByData(choices[i]);
            }
            var cancelType = params[1];
            var defaultType = params.length > 2 ? params[2] : 0;
            var positionType = params.length > 3 ? params[3] : 2;
            var background = params.length > 4 ? params[4] : 0;
            if (cancelType >= choices.length) {
                cancelType = -2;
            }
            $gameMessage.setChoices(choices, defaultType, cancelType);
            $gameMessage.setChoiceBackground(background);
            $gameMessage.setChoicePositionType(positionType);
            $gameMessage.setChoiceCallback(function (n) {
                this._branch[this._indent] = n;
            }.bind(this));
        };


        TextManager.basic = function (basicId) {
            return PluginManager.convText($dataSystem.terms.basic[basicId]) || '';
        };

        TextManager.param = function (paramId) {
            return PluginManager.convText($dataSystem.terms.params[paramId]) || '';
        };

        TextManager.command = function (commandId) {
            return PluginManager.convText($dataSystem.terms.commands[commandId]) || '';
        };

        TextManager.message = function (messageId) {
            return PluginManager.convText($dataSystem.terms.messages[messageId]) || '';
        };


        Object.defineProperty(TextManager, 'currencyUnit', {
            get: function () {
                return PluginManager.convText($dataSystem.currencyUnit);
            },
            configurable: true
        });


        Game_Actor.prototype.name = function () {
            return PluginManager.convText(this._name);
        };

        Game_Actor.prototype.nickname = function () {
            return PluginManager.convText(this._nickname);
        };

        Game_Actor.prototype.profile = function () {
            return PluginManager.convText(this._profile);
        };


        Window_Base.prototype.drawItemName = function (item, x, y, width) {
            width = width || 312;
            if (item) {
                var iconBoxWidth = Window_Base._iconWidth + 4;
                this.resetTextColor();
                this.drawIcon(item.iconIndex, x + 2, y + 2);
                this.drawText(PluginManager.convText(item.name), x + iconBoxWidth, y, width - iconBoxWidth);
            }
        };

        Window_Base.prototype.drawActorClass = function (actor, x, y, width) {
            width = width || 168;
            this.resetTextColor();
            this.drawText(PluginManager.convText(actor.currentClass().name), x, y, width);
        };


        Window_Help.prototype.refresh = function () {
            this.contents.clear();
            this.drawTextEx(PluginManager.convText(this._text), this.textPadding(), 0);
        };


        Window_SkillType.prototype.makeCommandList = function () {
            if (this._actor) {
                var skillTypes = this._actor.addedSkillTypes();
                skillTypes.sort(function (a, b) {
                    return a - b;
                });
                skillTypes.forEach(function (stypeId) {
                    var name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                    this.addCommand(name, 'skill', true, stypeId);
                }, this);
            }
        };


        Window_ActorCommand.prototype.addSkillCommands = function () {
            var skillTypes = this._actor.addedSkillTypes();
            skillTypes.sort(function (a, b) {
                return a - b;
            });
            skillTypes.forEach(function (stypeId) {
                var name = PluginManager.convText($dataSystem.skillTypes[stypeId]);
                this.addCommand(name, 'skill', true, stypeId);
            }, this);
        };


        Window_BattleLog.prototype.displayAction = function (subject, item) {
            var numMethods = this._methods.length;
            if (DataManager.isSkill(item)) {
                if (item.message1) {
                    this.push('addText', subject.name() + PluginManager.convText(item.message1).format(PluginManager.convText(item.name)));
                }
                if (item.message2) {
                    this.push('addText', PluginManager.convText(item.message2).format(PluginManager.convText(item.name)));
                }
            } else {
                this.push('addText', PluginManager.convText(TextManager.useItem).format(subject.name(), PluginManager.convText(item.name)));
            }
            if (this._methods.length === numMethods) {
                this.push('wait');
            }
        };


        Window_EquipSlot.prototype.slotName = function (index) {
            var slots = this._actor.equipSlots();
            return this._actor ? PluginManager.convText($dataSystem.equipTypes[slots[index]]) : '';
        };


        Game_Enemy.prototype.battlerName = function () {
            return PluginManager.convText(this.enemy().battlerName);
        };

        Game_Enemy.prototype.originalName = function () {
            return PluginManager.convText(this.enemy().name);
        };


        Game_Map.prototype.displayName = function () {
            return PluginManager.convText($dataMap.displayName);
        };



        NekoGakuen_MulitLanguage._Scene_Boot_start = Scene_Boot.prototype.start;
        Scene_Boot.prototype.start = async function () {
            await MulitLanguageArgs.preloadCsvData();
            NekoGakuen_MulitLanguage._Scene_Boot_start.call(this);
            if (PluginManager.isPlatformFlag() == "Nwjs") {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !PluginManager.fsModule().existsSync(PluginManager.pathModule().join(PluginManager.isFilePath(), 'save/config.rpgsave'))) {
                    ConfigManager.language = "init";
                }
            } else if (PluginManager.isPlatformFlag() == "Electronjs") {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !PluginManager.fsModule().existsSync(Utils.isOptionValid("test") ? PluginManager.pathModule().join(PluginManager.isFilePath(), 'save/config.rpgsave') : PluginManager.pathModule().join(PluginManager.isFilePath(), '../../save/config.rpgsave'))) {
                    ConfigManager.language = "init";
                }
            } else {
                if (args_InitLanSet["InitLan Switch"] == 'true' && !localStorage.getItem('RPG Config')) {
                    ConfigManager.language = "init";
                }
            }
            if (ConfigManager.language == "init") {
                if (!DataManager.isBattleTest() && !DataManager.isEventTest()) {
                    SceneManager.goto(Scene_InitialLanguage);
                }
            }
        };

        Scene_Boot.prototype.updateDocumentTitle = function () {
            document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
        };

        NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages = Scene_Boot.loadSystemImages;
        Scene_Boot.loadSystemImages = function () {
            NekoGakuen_MulitLanguage._Scene_Boot_loadSystemImages.call(this);
            ImageManager.loadPicture(args_InitLanSet["InitLan Images"]);
        };


        NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle = Scene_Title.prototype.drawGameTitle;
        Scene_Title.prototype.drawGameTitle = function () {
            $dataSystem.gameTitle = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
            if (PluginManager.isPlugins("NekoGakuen_FontManager")) {
                if (NekoGakuen_FontManager.cfl.length > 0 && ConfigManager.language != "init") {
                    this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
                } else {
                    this._gameTitleSprite.bitmap.fontFace = 'GameFont';
                }
            } else {
                if (NekoGakuen_MulitLanguage.cfl.length > 0 && ConfigManager.language != "init") {
                    this._gameTitleSprite.bitmap.fontFace = args_Lan2d[ConfigManager.language];
                } else {
                    this._gameTitleSprite.bitmap.fontFace = 'GameFont';
                }
            }
            NekoGakuen_MulitLanguage._Scene_Title_drawGameTitle.call(this);
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_LanguageSelect
        // 
        //  在遊戲初始化前選擇遊戲語言的視窗。
        //  The window to select the game language before the game initialization.
        //-----------------------------------------------------------------------------

        function Window_LanguageSelect() {
            this.initialize.apply(this, arguments);
        };

        Window_LanguageSelect.prototype = Object.create(Window_Selectable.prototype);
        Window_LanguageSelect.prototype.constructor = Window_LanguageSelect;

        Window_LanguageSelect.prototype.initialize = function (x, y, width, height) {
            var width = this.windowWidth();
            var height = this.windowHeight();
            Window_Selectable.prototype.initialize.call(this, x, y, width, height);
            this.opacity = Number(args_InitLanSet["InitLan Select Opacity"] || 255);
            this.setTopRow(0);
            this.select(0);
            this.activate();
            this.refresh();
        };

        Window_LanguageSelect.prototype.maxCols = function () {
            return 1;
        };

        Window_LanguageSelect.prototype.spacing = function () {
            return 48;
        };

        Window_LanguageSelect.prototype.maxItems = function () {
            return args_LanNameList ? args_LanNameList.length : 1;
        };

        Window_LanguageSelect.prototype.itemHeight = function () {
            return PluginManager.isMobileMode() ? 60 : this.lineHeight();
        };

        Window_LanguageSelect.prototype.lineHeight = function () {
            return PluginManager.isMobileMode() ? 60 : Window_Base.prototype.lineHeight.call(this);
        };

        Window_LanguageSelect.prototype.windowWidth = function () {
            return Number(args_InitLanSet["InitLan Select Width"] || 250);
        };

        Window_LanguageSelect.prototype.windowHeight = function () {
            return this.fittingHeight(this.numVisibleRows());
        };

        Window_LanguageSelect.prototype.drawItem = function (index) {
            var name = args_Lan2b[index];
            var rect = this.itemRectForText(index);
            this.drawText(name, rect.x, rect.y, rect.width, String(args_InitLanSet["InitLan Select Align"] || "left"));
        };

        Window_LanguageSelect.prototype.numVisibleRows = function () {
            return args_LanNameList.length;
        };

        Window_LanguageSelect.prototype.item = function () {
            return args_LanNameList ? args_Lan2b[this.index()] : null;
        };

        Window_LanguageSelect.prototype.makeItemList = function () {

        };

        Window_LanguageSelect.prototype.refresh = function () {
            this.createContents();
            this.drawAllItems();
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_LanguageHelp
        // 
        //  在遊戲初始化前顯示說明文字的視窗。
        //  The window that Show the description text before the game initializes.
        //-----------------------------------------------------------------------------

        function Window_LanguageHelp() {
            this.initialize.apply(this, arguments);
        };

        Window_LanguageHelp.prototype = Object.create(Window_Base.prototype);
        Window_LanguageHelp.prototype.constructor = Window_LanguageHelp;

        Window_LanguageHelp.prototype.initialize = function () {
            var width = Graphics.boxWidth;
            var height = this.fittingHeight(1);
            Window_Base.prototype.initialize.call(this, 0, 0, width, height);
            this._text = '';
            this.opacity = Number(args_InitLanSet["InitLan Help Opacity"] || 255);
        };

        Window_LanguageHelp.prototype.setText = function (text) {
            if (this._text !== text) {
                this._text = text;
                this.refresh();
            }
        };

        Window_LanguageHelp.prototype.clear = function () {
            this.setText("");
        };

        Window_LanguageHelp.prototype.setItem = function (value) {
            this.setText(args_Lan2c[value]);
        };

        Window_LanguageHelp.prototype.refresh = function () {
            this.contents.clear();
            this.drawTextEx(this._text, this.textPadding(), 0);
        };


        //-----------------------------------------------------------------------------
        // ◆ Scene_InitialLanguage
        // 
        //  語言選擇畫面的場景類別。
        //  The scene class of the language select screen.
        //-----------------------------------------------------------------------------

        function Scene_InitialLanguage() {
            this.initialize.apply(this, arguments);
        };

        Scene_InitialLanguage.prototype = Object.create(Scene_Base.prototype);
        Scene_InitialLanguage.prototype.constructor = Scene_InitialLanguage;

        Scene_InitialLanguage.prototype.initialize = function () {
            Scene_Base.prototype.initialize.call(this);
        };

        Scene_InitialLanguage.prototype.create = function () {
            Scene_Base.prototype.create.call(this);
            this.createBackground();
            this.createHelpWindow();
            this.createLanguageSelectWindow();
        };

        Scene_InitialLanguage.prototype.createBackground = function () {
            this._background = new Sprite(ImageManager.loadPicture(args_InitLanSet["InitLan Images"]));
            this.addChild(this._background);
            this.scaleSprite(this._background);
            this.centerSprite(this._background);
        };

        Scene_InitialLanguage.prototype.scaleSprite = function (sprite) {
            var scaleX = Graphics.width / sprite.bitmap.baseTexture.width;
            var scaleY = Graphics.height / sprite.bitmap.baseTexture.height;
            var scale = Math.max(scaleX, scaleY, 1.0);
            sprite.scale.x = scale;
            sprite.scale.y = scale;
        };

        Scene_InitialLanguage.prototype.centerSprite = function (sprite) {
            sprite.x = Graphics.width / 2;
            sprite.y = Graphics.height / 2;
            sprite.anchor.x = 0.5;
            sprite.anchor.y = 0.5;
        };

        Scene_InitialLanguage.prototype.createLanguageSelectWindow = function () {
            this._langSelectWindow = new Window_LanguageSelect();
            if (args_InitLanSet["InitLan AlignTypeX"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Left') {
                    this._langSelectWindow.x = 0;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Center') {

                    this._langSelectWindow.x = (Graphics.boxWidth - this._langSelectWindow.width) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectX"] == 'Right') {
                    this._langSelectWindow.x = Graphics.boxWidth - this._langSelectWindow.width;
                }
            } else {
                this._langSelectWindow.x = Number(args_InitLanSet["InitLan Custom SelectX"] || 283);
            }
            if (args_InitLanSet["InitLan AlignTypeY"] == "Auto") {
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Top') {
                    if (String(args_InitLanSet["InitLan Help Boolean"]) == "true") {
                        this._langSelectWindow.y = (this._langhelpWindow.y >= this._langhelpWindow.height ? 0 : this._langhelpWindow.height);
                    } else {
                        this._langSelectWindow.y = 0;
                    }
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Center') {
                    this._langSelectWindow.y = (Graphics.boxHeight - this._langSelectWindow.height) / 2;
                }
                if (args_InitLanSet["InitLan Auto SelectY"] == 'Bottom') {
                    if (String(args_InitLanSet["InitLan Help Boolean"]) == "true") {
                        this._langSelectWindow.y = (this._langhelpWindow.y >= Graphics.boxHeight - this._langhelpWindow.height ? Graphics.boxHeight - this._langSelectWindow.height - this._langhelpWindow.height : Graphics.boxHeight - this._langSelectWindow.height);
                    } else {
                        this._langSelectWindow.y = Graphics.boxHeight - this._langSelectWindow.height;
                    }
                }
            } else {
                this._langSelectWindow.y = Number(args_InitLanSet["InitLan Custom SelectY"] || 250);
            }
            this._langSelectWindow.setHandler("ok", this.onLangSetOk.bind(this));
            this.addChild(this._langSelectWindow);
        };

        Scene_InitialLanguage.prototype.createHelpWindow = function () {
            this._langhelpWindow = new Window_LanguageHelp();
            this._langhelpWindow.x = 0;
            if (args_InitLanSet["InitLan AlignTypeHelpY"] == "Auto") {
                this._langhelpWindow.y = (args_InitLanSet["InitLan Auto SelectHelpY"] == 'Top' ? 0 : Graphics.boxHeight - this._langhelpWindow.height);
            } else {
                this._langhelpWindow.y = Number(args_InitLanSet["InitLan Custom HelpY"]);
            }
            this.addChild(this._langhelpWindow);
            if (String(args_InitLanSet["InitLan Help Boolean"]) != "true") {
                this._langhelpWindow.hide();
            }
        };

        Scene_InitialLanguage.prototype.update = function () {
            Scene_Base.prototype.update.call(this);
            this._lastSelect = this._langSelectWindow.index();
            this._langhelpWindow.setItem(this._lastSelect);
        };

        Scene_InitialLanguage.prototype.onLangSetOk = function () {
            let langset = args_Lan2a[this._langSelectWindow.index()]
            ConfigManager.language = this._langSelectWindow.index();
            ConfigManager.save();
            MulitLanguageArgs.setLangData(langset);
            this._langSelectWindow.close();
            this.fadeOutAll();
            SceneManager.goto(Scene_Boot);
        };


        //-----------------------------------------------------------------------------
        // ◆ Window_Options (Add Features & Modify Features)
        // 
        //  為選項畫面新增額外功能和修改功能。
        //  Add additional features and modifications on the options screen.
        //-----------------------------------------------------------------------------

        NekoGakuen_MulitLanguage._Window_Options_statusText = Window_Options.prototype.statusText;
        Window_Options.prototype.statusText = function (index) {
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isLanguageSymbol(symbol)) {
                return this.langSymbol(value);
            } else {
                return NekoGakuen_MulitLanguage._Window_Options_statusText.call(this, index);
            }
        };

        Window_Options.prototype.isLanguageSymbol = function (symbol) {
            return symbol.contains('language');
        };

        Window_Options.prototype.langSymbol = function (value) {
            return args_Lan2b[value];
        };

        if (NekoGakuen_MulitLanguage.Config_Lang_Boolean == "true") {
            NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
            Window_Options.prototype.addGeneralOptions = function () {
                this.addCommand(PluginManager.convText(NekoGakuen_MulitLanguage.Config_Lang), 'language');
                NekoGakuen_MulitLanguage._Window_Options_addGeneralOptions.call(this);
            };
        }

        Window_Options.prototype.processOk = function () {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isVolumeSymbol(symbol)) {
                value += this.volumeOffset();
                if (value > 100) {
                    value = 0;
                }
                value = value.clamp(0, 100);
                this.changeValue(symbol, value);
            } else {
                this.changeValue(symbol, !value);
            }
            if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
            }
            if (PluginManager.isPlugins("TouchUI")) {
                if (symbol === "cancel" && this.isHandled('cancel')) {
                    this._handlers['cancel']();
                }
            }
        };

        Window_Options.prototype.langOffset = function () {
            return 1;
        };

        Window_Options.prototype.cursorLeft = function (wrap) {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isLanguageSymbol(symbol)) {
                value -= this.langOffset();
                let config = args_Lan2a;
                if (value < 0) {
                    value = config.length - 1;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
                return false;
            } else {
                this.changeValue(symbol, true);
            }
        };

        Window_Options.prototype.cursorRight = function (wrap) {
            var index = this.index();
            var symbol = this.commandSymbol(index);
            var value = this.getConfigValue(symbol);
            if (this.isLanguageSymbol(symbol)) {
                value += this.langOffset();
                let config = args_Lan2a;
                if (value > config.length - 1) {
                    value = 0;
                }
                MulitLanguageArgs.setLangData(args_Lan2a[value]);
                document.title = NekoGakuen_MulitLanguage.TitleText_Lang ? PluginManager.convText(NekoGakuen_MulitLanguage.TitleText_Lang) : $dataSystem.gameTitle;
                this.changeValue(symbol, value);
                this.resetFontSettings();
                this.refresh();
                return false;
            } else {
                this.changeValue(symbol, true);
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ ConfigManager (Add Features & Modify Features)
        // 
        //  為靜態類別新增額外功能和修改功能。
        //  Add additional features and modifications to static class.
        //-----------------------------------------------------------------------------

        if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Player Select") {
            ConfigManager.language = 0;
        } else {
            if (PluginManager.isPlugins("NekoGakuen_SteamworksAPI")) {
                if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Steam Client") {
                    ConfigManager.language = args_Lan2a.indexOf(SteamworksAPIManager.getCurrentUILanguage());
                }
                if (NekoGakuen_MulitLanguage.Select_Lang_Type == "Steam Game") {
                    ConfigManager.language = args_Lan2a.indexOf(SteamworksAPIManager.getCurrentGameLanguage());
                }
            } else {
                ConfigManager.language = 0;
            }
        }

        NekoGakuen_MulitLanguage._configManager_makeData = ConfigManager.makeData;
        ConfigManager.makeData = function () {
            var config = NekoGakuen_MulitLanguage._configManager_makeData.call(this);
            config.language = this.language;
            return config;
        };

        NekoGakuen_MulitLanguage._configManager_applyData = ConfigManager.applyData;
        ConfigManager.applyData = function (config) {
            NekoGakuen_MulitLanguage._configManager_applyData.call(this, config);
            this.language = this.readLanguage(config, 'language');
            args_Lanindex = args_Lan2a[this.language];
        };

        ConfigManager.readLanguage = function (config, name) {
            if (name in config) {
                return Number(config[name]).clamp(0, args_LanNameList.length - 1);
            } else {
                return 0;
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ Game_System (Add Features)
        // 
        //  為遊戲物件類別新增額外功能。
        //  Add additional features to the game object class.
        //-----------------------------------------------------------------------------

        Game_System.prototype.mulitLangSET = function (lanargs) {
            ConfigManager.language = args_Lan2a.indexOf(lanargs);
            ConfigManager.save();
            MulitLanguageArgs.setLangData(lanargs);
        };

        Game_System.prototype.mulitLangGET = function (variablesId) {
            if (variablesId) {
                $gameVariables.setValue(Number(variablesId), String(args_Lanindex));
            } else {
                return args_Lanindex;
            }
        };


        //-----------------------------------------------------------------------------
        // ◆ MulitLanguageArgs
        // 
        //  管理多語言功能的靜態類別。
        //  The static class that manages multi-language features.
        //-----------------------------------------------------------------------------

        function MulitLanguageArgs() {
            throw new Error('This is a static class');
        }
        
        let preloadedCsvData = {};
        MulitLanguageArgs.setLangData = function (lanindex) {
            var index = args_Lan2a.indexOf(lanindex);
            if (index !== -1) {
                args_Lanindex = args_Lan2a[index];
            }
        };
        
        MulitLanguageArgs.loadCsvAsync = function (filePath) {
            return new Promise((resolve, reject) => {
                var request = new XMLHttpRequest();
                request.open("GET", filePath, true);
                request.onload = function () {
                    if (request.status >= 200 && request.status < 300) {
                        try {
                            var parsedData = MulitLanguageArgs.parseCsv(request.responseText);
                            resolve(parsedData);
                        } catch (error) {
                            console.error(`${NekoGakuen_MulitLanguage.ConsoleError03}${filePath}`, error);
                            resolve([]);
                        }
                    } else {
                        console.error(`${NekoGakuen_MulitLanguage.ConsoleError04}${filePath}`);
                        resolve([]);
                    }
                };
                request.onerror = () => reject(`${NekoGakuen_MulitLanguage.ConsoleError05}${filePath}`);
                request.send();
            });
        };
        
        MulitLanguageArgs.parseCsv = function (content) {
            var rows = [];
            let currentRow = [];
            let insideQuotes = false;
            let cell = "";
            for (let char of content) {
                if (char === '"') {
                    insideQuotes = !insideQuotes;
                } else if (char === ',' && !insideQuotes) {
                    currentRow.push(cell.trim());
                    cell = "";
                } else if (char === '\n' && !insideQuotes) {
                    currentRow.push(cell.trim());
                    rows.push(currentRow);
                    currentRow = [];
                    cell = "";
                } else {
                    cell += char;
                }
            }
            if (cell.length > 0) {
                currentRow.push(cell.trim());
            }
            if (currentRow.length > 0) {
                rows.push(currentRow);
            }
        
            return rows;
        };
        
        MulitLanguageArgs.preloadCsvData = async function () {
            var promises = args_LancsvFileList.map(async fileInfo => {
                var fileData = JSON.parse(fileInfo);
                var csvFilePath = fileData["Lancsv Path"];
                preloadedCsvData[csvFilePath] = await MulitLanguageArgs.loadCsvAsync(csvFilePath);
            });
            await Promise.all(promises);
        };
        
        MulitLanguageArgs.getLangDataTextS = function (textArgs, csvName) {
            var csvPath = MulitLanguageArgs.getCsvPath(csvName);
            var csvArray = preloadedCsvData[csvPath] || [];
            var rowIndex = csvArray.findIndex(row => row[0] === textArgs);
        
            if (rowIndex > 0) {
                var columnIndex = csvArray[0].indexOf(args_Lanindex);
                let text = csvArray[rowIndex][columnIndex] || '';
                return text.replace(/^\"|\"$/g, '');
            }
            return '';
        };
        
        MulitLanguageArgs.getLangDataTextO = function (textArgs) {
            var csvPath = args_Lancsv1b[0];
            var csvArray = preloadedCsvData[csvPath] || [];
            var rowIndex = csvArray.findIndex(row => row[0] === textArgs);
        
            if (rowIndex > 0) {
                var columnIndex = csvArray[0].indexOf(args_Lanindex);
                let text = csvArray[rowIndex][columnIndex] || '';
                return text.replace(/^\"|\"$/g, '').replace(/\\/g, '\x1b');
            }
            return '';
        };
        
        MulitLanguageArgs.getCsvPath = function (csvName) {
            var fileInfo = args_LancsvFileList.find(info => JSON.parse(info)["Lancsv Name"] === csvName);
            return fileInfo ? JSON.parse(fileInfo)["Lancsv Path"] : '';
        };
        
    })();
}