//=============================================================================
// TMPlugin - つちけむり
// バージョン: 2.1.3
// 最終更新日: 2023/06/20
// 配布元    : http://hikimoki.sakura.ne.jp/
//-----------------------------------------------------------------------------
// Copyright (c) 2016 tomoaky
// Released under the MIT license.
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @plugindesc 在跳跃和冲刺中添加灰尘效果（备注控制版）。
 * 也可以随时显示烟雾。
 *
 * @author tomoaky (http://hikimoki.sakura.ne.jp/)
 * @originalModifier 孤身守梦
 * @modifier Anonymous (优化功能)
 * @param dustImage
 * @text 烟尘图像
 * @desc 用作灰尘烟雾的图像文件名。
 * 初始值: Dust1
 * @default Dust1
 * @require 1
 * @dir img/system/
 * @type file
 *
 * @param maxDusts
 * @text 最大烟尘数量
 * @type number
 * @desc 可以同时显示的烟尘数量。
 * 初始值: 64
 * @default 64
 *
 * @param jumpDusts
 * @text 跳跃烟尘数量
 * @type number
 * @desc 跳跃落地时显示的烟尘数量（需在事件备注中添加<dustJump>标签）。
 * 初始值: 5
 * @default 5
 *
 * @param dashDusts
 * @text 冲刺烟尘数量
 * @type number
 * @desc 冲刺时显示的烟尘数量。
 * 初始值: 3
 * @default 3
 *
 * @param followerDustEnabled
 * @text 跟随者粉尘默认
 * @desc 跟随者粉尘效果默认是否启用 (true:启用, false:禁用)
 * @type boolean
 * @default true
 *
 * @help
 * TMPlugin（修订） - つちけむり ver2.1.3
 *
 * 修订版本: 孤身守梦
 *
 * 修订版本: v1.2.3
 * 修订内容: 
 *   - 新增跟随者粉尘效果开关控制
 *   - 跳跃烟尘现在需要通过事件备注<dustJump>标签触发
 *   - 修复了跳跃烟尘缺失的问题
 *   - 增加了跟随者的烟尘效果
 *   - 增加了烟尘的角度随机性
 *   - 增加了8方向适配性
 *   - 修订了烟尘扩散方向
 *   - 修复烟尘大小刷新错误
 * 
 * 使用方法:
 *
 *   跳跃烟尘触发:
 *     在事件的备注栏中添加 <dustJump> 标签
 *     当该事件跳跃落地时会显示烟尘效果
 *
 *   跟随者粉尘控制:
 *     使用插件指令: setFollowerDust on - 启用跟随者粉尘效果
 *     使用插件指令: setFollowerDust off - 禁用跟随者粉尘效果
 *
 *   将随插件分发的烟尘图像保存在img/system文件夹中。 
 *   文件名为Dust1.png。
 *   如果需要更改文件名，请同时更改插件参数【烟尘图像】
 *
 *   启用插件后，玩家冲刺移动时会显示烟尘。
 *
 *   也可以通过插件命令随时显示指定坐标处的烟尘。
 *
 *   此插件已使用RPG Maker MV版本1.5.0进行测试。
 *
 *   该插件在 MIT 许可下分发，可自由用于商业用途、修改、再分发等。
 * 
 *
 * 插件命令:
 *
 *   setDustXy 5 8
 *     在指定坐标处显示烟尘。
 *     数值与事件指令『场所移动』中使用的坐标相同，不是画面的点数。
 *     也可以通过输入小数来指定坐标(5，8)和坐标(6，8)之间的中间位置，
 *     例如setDustXy 5.5 8。
 *
 *   setDustXy 5 8 3
 *     在指定坐标处显示三个烟尘。
 *     如果省略设置，则只显示一个。
 *
 *   setDustXy 5 8 1 0.04
 *     设置显示的烟尘的移动速度。如果省略设置，将应用 0.02。
 *
 *   setDustXy 5 8 1 0.02 3.14
 *     限制要显示的烟尘的移动方向。
 *     数值以右边为0，顺时针6.28，为1周。
 *     如上所述，设置3.14时，烟尘会稍微向左移动。
 *
 *   setDustEvent 3
 *     在事件3的脚下显示烟尘。
 *     如果设置为0，则以执行命令的事件本身为目标，
 *     如果设置-1，则目标为玩家。
 *     与setDustXy一样，也可以设置烟尘的数量、移动方向和移动速度。
 *     在事件编号之后，按照烟尘数量、移动速度、移动方向的顺序添加数字。
 * 
 *   setJumpDusts 5
 *     将跳跃着地时显示的烟尘数量更改为设置值。
 * 
 *   setDashDusts 3
 *     将冲刺时显示的烟尘数量更改到设置值。
 * 
 *   stopDust
 *     新的烟尘将无法显示。
 *     不影响已经显示的烟尘。
 * 
 *   startDust
 *     移除stopDust的效果，可以显示新的烟尘。
 *
 *   命令后面的参数只能省略中间的参数。
 *   指定移动方向时，还需要设置个数和移动速度。
 *
 *
 * 插件参数补充:
 *
 *   dustImage
 *     设置烟尘的图像文件名(不带扩展名)。 
 *     请将文件保存在img/system文件夹中。
 *
 *   maxDusts
 *     如果尝试同时显示超过此参数指定数量的烟尘，
 *     则不会显示任何内容，插件命令将被忽略。
 *     增加数值可以显示很多烟尘，
 *     但处理会使运行内存过大，在低配置环境中会导致FPS降低。
 *
 *   jumpDusts
 *     通过事件命令『设置移动路线』等使角色跳跃后，
 *     落地时显示的烟尘数量（需在事件备注中添加<dustJump>标签）。
 *     烟尘重叠了数值的量，变成了更浓的烟尘。
 *     如果设置0，落地时的烟尘将不再显示。
 * 
 *   dashDusts
 *     玩家以冲刺移动时显示的烟尘数。
 *     烟尘重叠了数值的量，变成了更浓的烟尘。
 *     如果设置0，冲刺时将不再显示烟尘。
 *
 */

var Imported = Imported || {};
Imported.TMCloudDust = true;

var TMPlugin = TMPlugin || {};
TMPlugin.CloudDust = {};
TMPlugin.CloudDust.Parameters = PluginManager.parameters('TMCloudDust');
TMPlugin.CloudDust.DustImage = TMPlugin.CloudDust.Parameters['dustImage'] || 'Dust1';
TMPlugin.CloudDust.MaxDusts = +(TMPlugin.CloudDust.Parameters['maxDusts'] || 64);
TMPlugin.CloudDust.JumpDusts = +(TMPlugin.CloudDust.Parameters['jumpDusts'] || 5);
TMPlugin.CloudDust.DashDusts = +(TMPlugin.CloudDust.Parameters['dashDusts'] || 3);

function Game_CloudDust() {
  this.initialize.apply(this, arguments);
}

if (!TMPlugin.InterpreterBase) {
  TMPlugin.InterpreterBase = true;
  (function() {

    Game_Interpreter.prototype.convertEscapeCharactersTM = function(text) {
      text = text.replace(/\\/g, '\x1b');
      text = text.replace(/\x1b\x1b/g, '\\');
      text = text.replace(/\x1bV$$(\d+)$$/gi, function() {
        return $gameVariables.value(parseInt(arguments[1]));
      }.bind(this));
      text = text.replace(/\x1bV$$(\d+)$$/gi, function() {
        return $gameVariables.value(parseInt(arguments[1]));
      }.bind(this));
      text = text.replace(/\x1bN$$(\d+)$$/gi, function() {
        return this.actorNameTM(parseInt(arguments[1]));
      }.bind(this));
      text = text.replace(/\x1bP$$(\d+)$$/gi, function() {
        return this.partyMemberNameTM(parseInt(arguments[1]));
      }.bind(this));
      text = text.replace(/\x1bG/gi, TextManager.currencyUnit);
      return text;
    };
  
    Game_Interpreter.prototype.actorNameTM = function(n) {
      var actor = n >= 1 ? $gameActors.actor(n) : null;
      return actor ? actor.name() : '';
    };

    Game_Interpreter.prototype.partyMemberNameTM = function(n) {
      var actor = n >= 1 ? $gameParty.members()[n - 1] : null;
      return actor ? actor.name() : '';
    };

  })();
} // TMPlugin.InterpreterBase

(function() {

  //-----------------------------------------------------------------------------
  // Game_System
  //
Game_System.prototype.isFollowerDustEnabled = function() {
  if (this._followerDustEnabled === undefined) {
    this._followerDustEnabled = TMPlugin.CloudDust.FollowerDustEnabled;
  }
  return this._followerDustEnabled;
};

Game_System.prototype.enableFollowerDust = function() {
  this._followerDustEnabled = true;
};

Game_System.prototype.disableFollowerDust = function() {
  this._followerDustEnabled = false;
};
  Game_System.prototype.jumpDusts = function() {
    if (this._jumpDusts != null) return this._jumpDusts;
    return TMPlugin.CloudDust.JumpDusts;
  };

  Game_System.prototype.dashDusts = function() {
    if (this._dashDusts != null) return this._dashDusts;
    return TMPlugin.CloudDust.DashDusts;
  };

  Game_System.prototype.setJumpDusts = function(n) {
    this._jumpDusts = n;
  };

  Game_System.prototype.setDashDusts = function(n) {
    this._dashDusts = n;
  };

  Game_System.prototype.isDustEnabled = function() {
    if (this._dustEnabled == null) this._dustEnabled = true;
    return this._dustEnabled;
  };

  Game_System.prototype.enableDust = function() {
    this._dustEnabled = true;
  };

  Game_System.prototype.disableDust = function() {
    this._dustEnabled = false;
  };

  //-----------------------------------------------------------------------------
  // Game_Map
  //

  	
var _Game_Map_setup = Game_Map.prototype.setup;
  Game_Map.prototype.setup = function(mapId) {
    _Game_Map_setup.call(this, mapId);
    this.setupCloudDusts();
  };

  Game_Map.prototype.setupCloudDusts = function() {
    this._cloudDusts = [];
    for (var i = 0; i < TMPlugin.CloudDust.MaxDusts; i++) {
      this._cloudDusts.push(new Game_CloudDust());
    }
  };

  Game_Map.prototype.cloudDusts = function() {
    return this._cloudDusts;
  };
  
  Game_Map.prototype.addCloudDust = function(x, y, speed, radian) {
    if (!$gameSystem.isDustEnabled()) return;
    for (var i = 0; i < TMPlugin.CloudDust.MaxDusts; i++) {
      if (!this._cloudDusts[i].exists()) {
        this._cloudDusts[i].setup(x, y, speed, radian);
        break;
      }
    }
  };

  var _Game_Map_update = Game_Map.prototype.update;
  Game_Map.prototype.update = function(sceneActive) {
    _Game_Map_update.call(this, sceneActive);
    this.updateCloudDusts();
  };

  Game_Map.prototype.updateCloudDusts = function() {
    this.cloudDusts().forEach(function(cloudDust) {
      cloudDust.update();
    });
  };
  
  //-----------------------------------------------------------------------------
  // Game_CloudDust
  //

  Game_CloudDust.prototype.initialize = function() {
    this._x = 0;
    this._y = 0;
    this._count = 0;
    this._scale = new Point(1.0, 1.0);
  };

  Game_CloudDust.prototype.setup = function(x, y, speed, radian) {
    this._x = +x;
    this._y = +y;
    this._opacity = 180;
    this._count = 30;
    if (radian != null) {
        radian = +radian + Math.random() * 1.5 - 0.75;
    } else {
        radian = Math.random() * Math.PI * 2;
    }
    speed = +(speed || 0.02);
    this._vx = Math.cos(radian) * speed;
    this._vy = Math.sin(radian) * speed;
    this._rotation = radian;
    // 强制重置缩放比例
    this._scale = new Point(1.0, 1.0); // 新增此行
};

  Game_CloudDust.prototype.screenX = function() {
    var tw = $gameMap.tileWidth();
    return Math.round($gameMap.adjustX(this._x) * tw);
  };

  Game_CloudDust.prototype.screenY = function() {
    var th = $gameMap.tileHeight();
    return Math.round($gameMap.adjustY(this._y) * th);
  };

  Game_CloudDust.prototype.opacity = function() {
    return this._opacity;
  };

  Game_CloudDust.prototype.rotation = function() {
    return this._rotation;
  };

  Game_CloudDust.prototype.scale = function() {
    return this._scale;
  };

  Game_CloudDust.prototype.exists = function() {
    return this._count > 0;
  };

  Game_CloudDust.prototype.update = function() {
    if (this._count > 0) {
      this._count--;
      this._x += this._vx;
      this._y += this._vy;
      this._vy -= 0.0008;
      this._opacity -= 6;
      this._scale.x += 0.02;
      this._scale.y += 0.02;
    }
  };

  //-----------------------------------------------------------------------------
  // Game_CharacterBase
  //

  Game_CharacterBase.prototype.addCloudDust = function(speed, radian) {
    $gameMap.addCloudDust(this._realX + 0.5, this._realY + 1.0, speed, radian);
  };

  var _Game_CharacterBase_moveDiagonally = Game_CharacterBase.prototype.moveDiagonally;
Game_CharacterBase.prototype.moveDiagonally = function(horz, vert) {
    var d = Galv.DM.getDir(horz, vert); // 获取斜向方向（1、3、7、9）
    var n = $gameSystem.dashDusts();
    if (n > 0 && this.isDashing() && this.canPass(this.x, this.y, d)) {
        var radian;
        switch(d) {
            case 1:  // 左下
                radian = Math.PI * 0.75;
                break;
            case 3:  // 右下
                radian = Math.PI * 0.25;
                break;
            case 7:  // 左上
                radian = Math.PI * 1.25;
                break;
            case 9:  // 右上
                radian = Math.PI * 1.75;
                break;
            default:
                radian = 0;
        }
        for (var i = 0; i < n; i++) {
            this.addCloudDust(0.03, radian + Math.random() * 0.2 - 0.1);
        }
    }
    _Game_CharacterBase_moveDiagonally.call(this, horz, vert);
};

  //-----------------------------------------------------------------------------
  // Game_Player
  //

  var _Game_Player_moveStraight = Game_Player.prototype.moveStraight;
Game_Player.prototype.moveStraight = function(d) {
    var n = $gameSystem.dashDusts();
    if (n > 0) {
        if (this.isDashing() && this.canPass(this.x, this.y, d)) {
            var radian;
            switch(d) {
                case 1:  // 左下移动 → 烟尘向右上扩散（315度）
                    radian = Math.PI * 1.75;
                    break;
                case 3:  // 右下移动 → 烟尘向左上扩散（225度）
                    radian = Math.PI * 1.25;
                    break;
                case 7:  // 左上移动 → 烟尘向右下扩散（135度）
                    radian = Math.PI * 0.75;
                    break;
                case 9:  // 右上移动 → 烟尘向左下扩散（45度）
                    radian = Math.PI * 0.25;
                    break;
                case 2:  // 下移动 → 烟尘向下扩散（保持原样）
                     radian = Math.PI * 1.5;  // 270度（向下）
                    break;
                case 4:  // 左移动 → 烟尘向右扩散（0度）
                    radian = 0;
                    break;
                case 6:  // 右移动 → 烟尘向左扩散（180度）
                    radian = Math.PI;
                    break;
                case 8:  // 上移动 → 烟尘向上扩散（保持原样）
                    radian = Math.PI / 2;    // 90度（向上）
                    break;
                default:
                    radian = 0;
            }
            for (var i = 0; i < n; i++) {
                this.addCloudDust(0.03, radian + Math.random() * 0.2 - 0.1); // 添加随机角度偏移
            }
        }
    }
    _Game_Player_moveStraight.call(this, d);
};

 //-----------------------------------------------------------------------------
  // Game_Follower
  //

var _Game_Follower_moveStraight = Game_Follower.prototype.moveStraight;
Game_Follower.prototype.moveStraight = function(d) {
    if (!$gameSystem.isFollowerDustEnabled()) return _Game_Follower_moveStraight.call(this, d);
    var n = $gameSystem.dashDusts();
    if (n > 0) {
        if (this.isVisible() && $gamePlayer.isDashing() && this.canPass(this.x, this.y, d)) {
            var radian;
            switch(d) {
                case 1:  // 左下移动 → 烟尘向右上扩散（315度）
                    radian = Math.PI * 1.75;
                    break;
                case 3:  // 右下移动 → 烟尘向左上扩散（225度）
                    radian = Math.PI * 1.25;
                    break;
                case 7:  // 左上移动 → 烟尘向右下扩散（135度）
                    radian = Math.PI * 0.75;
                    break;
                case 9:  // 右上移动 → 烟尘向左下扩散（45度）
                    radian = Math.PI * 0.25;
                    break;
                case 2:  // 下移动 → 烟尘向上扩散（保持原样）
                    radian = Math.PI * 1.5;  // 270度（向下）
                    break;
                case 4:  // 左移动 → 烟尘向右扩散（0度）
                    radian = 0;
                    break;
                case 6:  // 右移动 → 烟尘向左扩散（180度）
                    radian = Math.PI;
                    break;
                case 8:  // 上移动 → 烟尘向下扩散（保持原样）
                    radian = Math.PI / 2;    // 90度（向上）
                    break;
                default:
                    radian = 0;
            }
            for (var i = 0; i < n; i++) {
                this.addCloudDust(0.03, radian + Math.random() * 0.2 - 0.1); // 添加随机角度偏移
            }
        }
    }
    _Game_Follower_moveStraight.call(this, d);
};

  //-----------------------------------------------------------------------------
  // Game_Event
  //
  
  // 添加方法检查事件是否有灰尘跳跃备注
  Game_Event.prototype.hasDustJumpNote = function() {
    return this.event().meta.dustJump !== undefined;
  };

  //-----------------------------------------------------------------------------
  // Game_CharacterBase
  //
  
  // 重写updateJump方法，添加备注检查
  var _Game_CharacterBase_updateJump = Game_CharacterBase.prototype.updateJump;
  Game_CharacterBase.prototype.updateJump = function() {
    _Game_CharacterBase_updateJump.call(this);
    
    // 跳跃结束时触发
    if (this._jumpCount === 0) {
      // 对于事件，检查是否有dustJump备注
      if (this instanceof Game_Event) {
        if (this.hasDustJumpNote()) {
          this.triggerDustJump();
        }
      } 
   // 玩家始终触发
      else if (this instanceof Game_Player) {
        this.triggerDustJump();
      }
      // 跟随者根据开关触发
      else if (this instanceof Game_Follower) {
        if ($gameSystem.isFollowerDustEnabled()) {
          this.triggerDustJump();
        }
      }
    }
  };
  
  // 触发跳跃灰尘的方法
  Game_CharacterBase.prototype.triggerDustJump = function() {
    var n = $gameSystem.jumpDusts();
    if (n > 0) {
      for (var i = 0; i < n; i++) {
        // 添加随机角度 (-0.75 ~ +0.75 弧度)
        var radian = i % 2 * Math.PI + Math.random() * 1.5 - 0.75;
        this.addCloudDust(0.02, radian);
      }
    }
  };

  //-----------------------------------------------------------------------------
  // Game_Interpreter
  //

  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function(command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);
    if (command === 'setDustXy') {
      var arr = args.map(this.convertEscapeCharactersTM, this);
      var n = parseInt(arr[2] || 1);
      for (var i = 0; i < n; i++) {
        $gameMap.addCloudDust(arr[0], arr[1], arr[3], arr[4]);
      }
    } else if (command === 'setDustEvent') {
      var arr = args.map(this.convertEscapeCharactersTM, this);
      var character = this.character(+arr[0]);
      if (character) {
        var n = parseInt(arr[1] || 1);
        for (var i = 0; i < n; i++) {
          character.addCloudDust(arr[2], arr[3]);
        }
      }
    } else if (command === 'setJumpDusts') {
      var arr = args.map(this.convertEscapeCharactersTM, this);
      $gameSystem.setJumpDusts(+arr[0]);
    } else if (command === 'setDashDusts') {
      var arr = args.map(this.convertEscapeCharactersTM, this);
      $gameSystem.setDashDusts(+arr[0]);
    } else if (command === 'stopDust') {
      $gameSystem.disableDust();
    } else if (command === 'startDust') {
      $gameSystem.enableDust();
    } else if (command === 'setFollowerDust') {
      var arg = args[0] ? args[0].toLowerCase() : '';
      if (arg === 'on') {
        $gameSystem.enableFollowerDust();
      } else if (arg === 'off') {
        $gameSystem.disableFollowerDust();
      }
    }
  };
  
  //-----------------------------------------------------------------------------
  // Sprite_CloudDust
  //

  function Sprite_CloudDust() {
    this.initialize.apply(this, arguments);
  }

  Sprite_CloudDust.prototype = Object.create(Sprite.prototype);
  Sprite_CloudDust.prototype.constructor = Sprite_CloudDust;

  Sprite_CloudDust.prototype.initialize = function(cloudDust) {
    Sprite.prototype.initialize.call(this);
    this._cloudDust = cloudDust;
    this.scale = this._cloudDust.scale();
    this.visible = false;
    this.createBitmap();
  };

Sprite_CloudDust.prototype.update = function() {
    Sprite.prototype.update.call(this);
    if (this._cloudDust.exists()) {
        this.visible = true;
        this.x = this._cloudDust.screenX();
        this.y = this._cloudDust.screenY();
        this.opacity = this._cloudDust.opacity();
        this.rotation = this._cloudDust.rotation();
        // 强制应用缩放值
        this.scale.x = this._cloudDust.scale().x;
        this.scale.y = this._cloudDust.scale().y;
    } else {
        this.visible = false;
    }
};

  Sprite_CloudDust.prototype.createBitmap = function() {
    this.bitmap = ImageManager.loadSystem(TMPlugin.CloudDust.DustImage);
    this.anchor.x = 0.5;
    this.anchor.y = 0.5;
    this.z = 3;
  };

  //-----------------------------------------------------------------------------
  // Spriteset_Map
  //

  var _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function() {
    _Spriteset_Map_createLowerLayer.call(this);
    this.createCloudDust();
  };

  Spriteset_Map.prototype.createCloudDust = function() {
    this._cloudDustSprites = [];
    $gameMap.cloudDusts().forEach(function(cloudDust) {
      this._cloudDustSprites.push(new Sprite_CloudDust(cloudDust));
    }, this);
    for (var i = 0; i < this._cloudDustSprites.length; i++) {
      this._tilemap.addChild(this._cloudDustSprites[i]);
    }
  };
  
})();