/*=============================================================================
 _Key_Basic_Cutscenes.js
----------------------------------------------------------------------------
 :::::::::  :::    ::: :::::::::  :::   :::    :::    ::: :::::::::: :::   ::: 
 :+:    :+: :+:    :+: :+:    :+: :+:   :+:    :+:   :+:  :+:        :+:   :+: 
 +:+    +:+ +:+    +:+ +:+    +:+  +:+ +:+     +:+  +:+   +:+         +:+ +:+  
 +#++:++#:  +#+    +:+ +#++:++#+    +#++:      +#++:++    +#++:++#     +#++:   
 +#+    +#+ +#+    +#+ +#+    +#+    +#+       +#+  +#+   +#+           +#+    
 #+#    #+# #+#    #+# #+#    #+#    #+#       #+#   #+#  #+#           #+#    
 ###    ###  ########  #########     ###       ###    ### ##########    ###    

 Key Basic Cutscene
 (C)2023 MiguelRPG - RubyKey
 Version: 0.0.3
 Date: 2024-05
 This software is released under the MIT License.
 http://opensource.org/licenses/mit-license.php
----------------------------------------------------------------------------
 Version
 0.0.7 2024/05/31 beta version
 0.0.6 2024/05/31 fix switch activation
 0.0.5 2024/05/31 english comments
 0.0.4 2024/05/30 add plugin commands
 0.0.3 2024/05/30 add event common start and end
 0.0.2 2024/05/29 add opacity and zoom in/out
 0.0.1 2024/05/28 create plugin
=============================================================================*/

var Imported = Imported || {};
Imported._key_Basic_Cutscenes = true;

var RubyKey = RubyKey || {};
RubyKey = RubyKey || {};
RubyKey.CUTSCENE = RubyKey.CUTSCENE || {};
RubyKey.CUTSCENE.version = 0.07;

/*:
 * @target MV
 * @plugindesc Plugin to create simple cutscenes without the need to create events on the map.
 * 
 * @author MiguelRPG - RubyKey
 * 
 * @help
 * -->------------------------------------------------------------------
 * [Introduction]
 * -->------------------------------------------------------------------
 * This plugin allows you to create simple cutscenes by adding horizontal black bars at the top and bottom to simulate film, without the need to create events on the map.
 * 
 * Through a plugin command, you can activate and deactivate the cutscene.
 * 
 * It is possible to trigger a common event when starting the cutscene and another when finishing the cutscene.
 * 
 * -->------------------------------------------------------------------
 * [Usage Configuration]
 * -->------------------------------------------------------------------
 * 
 * Switch ---------------------------------
 * 
 * ---> Switch to activate cutscene
 *    - When the switch is activated, the cutscene will be activated - but the start common event is not triggered.
 * 
 * ---> Switch to deactivate cutscene
 *    - When the switch is deactivated, the cutscene will be deactivated - but the end common event is not triggered.
 * 
 * -->------------------------------------------------------------------
 * [Plugin Command]
 * -->------------------------------------------------------------------
 * 
 * ---> StartCutscene
 * - To activate the cutscene, use the plugin command StartCutscene.
 *    * This way the switch will be activated and the start common event will be triggered.
 *    * Example: StartCutscene
 *
 * ---> EndCutscene
 * - To deactivate the cutscene, use the plugin command EndCutscene.
 *   * This way the switch will be deactivated and the end common event will be triggered.
 *   * Example: EndCutscene
 *    
 * 
 * - In the plugin settings, you can define the height of the black bars and the speed at which they appear.
 *    
 * -->------------------------------------------------------------------
 * [Settings]
 * -->------------------------------------------------------------------
 * 
 * @param switch
 * @text Switch to activate cutscene
 * @desc Switch to activate the cutscene
 * @default 100
 * 
 * @param black_bar_height
 * @text Height of the black bar
 * @desc Height of the black bar
 * @default 100
 * 
 * @param black_bar_speed
 * @text Speed of the black bar in frames
 * @desc Speed of the black bar in frames
 * @default 120
 * 
 * @param black_bar_transparece
 * @text Transparency of the black bar
 * @description Transparency of the black bar
 * @default 120
 * 
 * @param zoom_in_duration
 * @text Duration of the zoom in
 * @description Duration of the zoom in
 * @default 60
 * 
 * @param zoom_out_duration
 * @text Duration of the zoom out
 * @description Duration of the zoom out
 * @default 60
 * 
 * @param common_event_start
 * @text Common event when starting cutscene
 * @description Common event when starting cutscene
 * @default 10
 * 
 * @param common_event_end
 * @text Common event when finishing cutscene
 * @description Common event when finishing cutscene
 * @default 11
 * 
 * -->------------------------------------------------------------------
 * [End of Settings]
 * -->------------------------------------------------------------------
 * 
 */

/*:pt
 * @target MV
 * @plugindesc plugin para criar cutscenes simples sem a necessidade de criar eventos no mapa.
 * 
 * @author MiguelRPG - RubyKey
 * 
 * @help
 * -->------------------------------------------------------------------
 * [Introdução]
 * -->------------------------------------------------------------------
 * Este plugin permite criar cutscenes simples, inserindo barras pretas horizontais na parte 
 * superior e inferior para simular filme, sem a necessidade de criar eventos no mapa.
 * 
 * Através de comando de plugin, é possível ativar e desativar a cutscene.
 * 
 * É possível disparar um evento comum ao iniciar a cutscene e outro ao finalizar a cutscene.
 * 
 * -->------------------------------------------------------------------
 * [Configuração de uso]
 * -->------------------------------------------------------------------
 * 
 * Swith ---------------------------------
 * 
 * ---> Switch para ativar cutscene
 *    - Ao ativar a switch, a cutscene será ativada - mas o evento comum de início não é disparado.
 * 
 * ---> Switch para desativar cutscene
 *    - Ao desativar a switch, a cutscene será desativada - mas o evento comum de fim não é disparado.
 * 
 * -->------------------------------------------------------------------
 * [Plugin Command]
 * -->------------------------------------------------------------------
 * 
 * ---> StartCutscene
 * - Para ativar a cutscene, utilize o comando de plugin StartCutscene.
 *    * Desta forma a switch será ativada e o evento comum de início será disparado.
 *    * Exemplo: StartCutscene
 *
 * ---> EndCutscene
 * - Para desativar a cutscene, utilize o comando de plugin EndCutscene.
 *   * Desta forma a switch será desativada e o evento comum de fim será disparado.
 *   * Exemplo: EndCutscene
 *    
 * 
 * - Nas configurações do plugin, é possível definir a altura das barras pretas e a velocidade que elas aparecem.
 *    
 * -->------------------------------------------------------------------
 * [Configurações]
 * -->------------------------------------------------------------------
 * 
 * @param switch
 * @text Switch para ativar cutscene
 * @desc Switch para ativar a cutscene
 * @default 100
 * 
 * @param black_bar_height
 * @text Altura da barra preta
 * @desc Altura da barra preta
 * @default 100
 * 
 * @param black_bar_speed
 * @text Velocidade da barra preta em frames
 * @desc Velocidade da barra preta em frames
 * @default 120
 * 
 * @param black_bar_transparece
 * @text Transparência da barra preta
 * @description Transparência da barra preta
 * @default 120
 * 
 * @param zoom_in_duration
 * @text Duração do zoom in
 * @description Duração do zoom in
 * @default 60
 * 
 * @param zoom_out_duration
 * @text Duração do zoom out
 * @description Duração do zoom out
 * @default 60
 * 
 * @param common_event_start
 * @text Evento comum ao iniciar cutscene
 * @description Evento comum ao iniciar cutscene
 * @default 10
 * 
 * @param common_event_end
 * @text Evento comum ao finalizar cutscene
 * @description Evento comum ao finalizar cutscene
 * @default 11
 * 
 * -->------------------------------------------------------------------
 * [Fim das Configurações]
 * -->------------------------------------------------------------------
 * 
 */

(function () {

  var parameters = PluginManager.parameters('_key_Basic_Cutscenes');

  const configs = {
    switchId: Number(parameters['switch'] || 0),
    blackBarHeight: Number(parameters['black_bar_height'] || 100),
    blackBarSpeed: Number(parameters['black_bar_speed'] || 120),
    blackBarTransparece: Number(parameters['black_bar_transparece'] || 120),
    zoomInDuration: Number(parameters['zoom_in_duration'] || 60),
    zoomOutDuration: Number(parameters['zoom_out_duration'] || 60),
    commonEventStart: Number(parameters['common_event_start'] || 10),
    commonEventEnd: Number(parameters['common_event_end'] || 11),
  };

  RubyKey.CUTSCENE.CONFIG = configs;


  //=============================================================================
  // ** Scene_Map
  //=============================================================================

  // ALIAS: Scene_Map.prototype.update
  //  * processa cutscene bars
  //-----------------------------------------------------------------------------
  var _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);

    this.cutsceneBarsAnimation();
    this.updateCutsceneBars();
  };

  // NEW FUNCTION: Scene_Map -> createCutsceneBars
  //  * cria as barras pretas
  //-----------------------------------------------------------------------------
  Scene_Map.prototype.createCutsceneBars = function () {
    this._cutsceneBars = new Sprite();
    this._cutsceneBars.bitmap = new Bitmap(Graphics.boxWidth, Graphics.boxHeight);
    this._cutsceneBars.bitmap.fillRect(0, 0, Graphics.boxWidth, RubyKey.CUTSCENE.CONFIG.blackBarHeight, 'black');
    this._cutsceneBars.bitmap.fillRect(0, Graphics.boxHeight - RubyKey.CUTSCENE.CONFIG.blackBarHeight, Graphics.boxWidth, RubyKey.CUTSCENE.CONFIG.blackBarHeight, 'black');
    window._cutsceneBars = this._cutsceneBars;
    window.map = this;

    if (typeof Utils.RPGMAKER_NAME !== 'undefined' && Utils.RPGMAKER_NAME === 'MV')
      this.addChildAt(this._cutsceneBars, this.getChildIndex(this._mapNameWindow));
    else
      this.addChild(this._cutsceneBars);

    this._cutsceneBars.opacity = 0;
    this._cutsceneBars.visible = true;
    this._cutsceneBars.scale.x = 1.5; // Inicializa com zoom 1.5
    this._cutsceneBars.scale.y = 1.5; // Inicializa com zoom 1.5
    this._cutsceneBars.x = (Graphics.boxWidth - this._cutsceneBars.width * 1.5) / 2;
    this._cutsceneBars.y = (Graphics.boxHeight - this._cutsceneBars.height * 1.5) / 2;
  };

  // NEW FUNCTION: Scene_Map -> cutsceneBarsAnimation 
  //  * processa a criação e remoção das barras pretas
  //-----------------------------------------------------------------------------
  Scene_Map.prototype.cutsceneBarsAnimation = function () {
    if (!RubyKey.CUTSCENE.CONFIG.switchId > 0) return;

    if ($gameSwitches.value(RubyKey.CUTSCENE.CONFIG.switchId)) {
      if (!this._cutsceneBars) {
        this.createCutsceneBars();
      }
    } else {
      if (this._cutsceneBars && this._cutsceneBars.opacity === 0)
        this.removeCutsceneBars();
    }
  };

  // NEW FUNCTION: Scene_Map -> updateCutsceneBars
  //  * processa a animação das barras pretas
  //-----------------------------------------------------------------------------
  Scene_Map.prototype.updateCutsceneBars = function () {
    if (!RubyKey.CUTSCENE.CONFIG.switchId > 0) return;
    if (!this._cutsceneBars) return;

    if ($gameSwitches.value(RubyKey.CUTSCENE.CONFIG.switchId)) {
      if (this._cutsceneBars.opacity < RubyKey.CUTSCENE.CONFIG.blackBarTransparece)
        this._cutsceneBars.opacity += RubyKey.CUTSCENE.CONFIG.blackBarTransparece / RubyKey.CUTSCENE.CONFIG.blackBarSpeed;

      // Zoom in
      if (this._cutsceneBars.scale.x > 1.0) {
        if (this._cutsceneBars.scale.x - 0.5 / RubyKey.CUTSCENE.CONFIG.zoomInDuration > 1.0)
          this._cutsceneBars.scale.x -= 0.5 / RubyKey.CUTSCENE.CONFIG.zoomInDuration;
        else
          this._cutsceneBars.scale.x = 1.0;

        if (this._cutsceneBars.x < 0) {
          let scale = (Graphics.boxWidth - this._cutsceneBars.width * this._cutsceneBars.scale.x) / 2;
          this._cutsceneBars.x = scale < 0 ? scale : 0;
        } else {
          this._cutsceneBars.x = 0;
        }
      } else {
        if (this._cutsceneBars.scale.x < 1.0) {
          this._cutsceneBars.scale.x = 1.0;
        }
      }

      if (this._cutsceneBars.scale.y > 1.0) {

        if (this._cutsceneBars.scale.y - 0.5 / RubyKey.CUTSCENE.CONFIG.zoomOutDuration > 1.0)
          this._cutsceneBars.scale.y -= 0.5 / RubyKey.CUTSCENE.CONFIG.zoomOutDuration;
        else
          this._cutsceneBars.scale.y = 1.0;

        if (this._cutsceneBars.y < 0) {
          let scale = parseInt((Graphics.boxHeight - this._cutsceneBars.height * this._cutsceneBars.scale.y) / 2);
          this._cutsceneBars.y = scale < 0 ? scale : 0;
        } else {
          this._cutsceneBars.y = 0;
        }
      } else {
        if (this._cutsceneBars.scale.y < 1.0) {
          this._cutsceneBars.scale.y = 1.0;
        }
        if (this._cutsceneBars.y !== 0) {
          this._cutsceneBars.y = 0;
        }
        if (this._cutsceneBars.x !== 0) {
          this._cutsceneBars.x = 0;
        }
      }
    } else {
      if (this._cutsceneBars.opacity > 0)
        this._cutsceneBars.opacity -= RubyKey.CUTSCENE.CONFIG.blackBarTransparece / RubyKey.CUTSCENE.CONFIG.blackBarSpeed;

      // Zoom out
      if (this._cutsceneBars.scale.x < 1.5) {
        this._cutsceneBars.scale.x += 0.5 / RubyKey.CUTSCENE.CONFIG.zoomOutDuration;
        this._cutsceneBars.x = (Graphics.boxWidth - this._cutsceneBars.width * this._cutsceneBars.scale.x) / 2;
      }

      if (this._cutsceneBars.scale.y < 1.5) {
        this._cutsceneBars.scale.y += 0.5 / RubyKey.CUTSCENE.CONFIG.zoomOutDuration;
        this._cutsceneBars.y = parseInt((Graphics.boxHeight - this._cutsceneBars.height * this._cutsceneBars.scale.y) / 2);
      }
    }
  };

  // ALIAS: Scene_Map.prototype.terminate
  //  * remove cutscene bars
  //-----------------------------------------------------------------------------
  var _Scene_Map_terminate = Scene_Map.prototype.terminate;
  Scene_Map.prototype.terminate = function () {
    _Scene_Map_terminate.call(this);
    this.removeCutsceneBars();
  };

  // NEW FUNCTION: Scene_Map -> removeCutsceneBars
  //  * remove cutscene bars
  //-----------------------------------------------------------------------------
  Scene_Map.prototype.removeCutsceneBars = function () {
    this.removeChild(this._cutsceneBars);
    this._cutsceneBars = null;
  };


  //=============================================================================
  // ** Game_Interpreter
  //=============================================================================

  // NEW FUNCTION: Game_Interpreter -> pluginCommand
  //  * processa os comandos de plugin
  //-----------------------------------------------------------------------------  
  var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    _Game_Interpreter_pluginCommand.call(this, command, args);

    if (!command || command === '') return;

    if (command.toLocaleLowerCase() === 'StartCutscene'.toLocaleLowerCase()) {
      if (!RubyKey.CUTSCENE.CONFIG.switchId > 0) return;

      $gameSwitches.setValue(RubyKey.CUTSCENE.CONFIG.switchId, true);
      if (RubyKey.CUTSCENE.CONFIG.commonEventStart > 0) {
        // $gameTemp.reserveCommonEvent(RubyKey.CUTSCENE.CONFIG.commonEventStart);

        var commonEvent = $dataCommonEvents[RubyKey.CUTSCENE.CONFIG.commonEventStart];
        if (commonEvent) {
          var eventId = this.isOnCurrentMap() ? this._eventId : 0;
          this.setupChild(commonEvent.list, eventId);
        }
      }
    }

    if (command.toLocaleLowerCase() === 'EndCutscene'.toLocaleLowerCase()) {
      if (!RubyKey.CUTSCENE.CONFIG.switchId > 0) return;

      $gameSwitches.setValue(RubyKey.CUTSCENE.CONFIG.switchId, false);
      if (RubyKey.CUTSCENE.CONFIG.commonEventEnd > 0) {
        $gameTemp.reserveCommonEvent(RubyKey.CUTSCENE.CONFIG.commonEventEnd);
      }
    }
  };

})();
