/*:
 * @plugindesc (v1.0) QTE 시스템 플러그인 - Quick Time Event를 구현합니다.
 * @author kar
 *
 * @help
 * ===================================================
 * kar_QTE 플러그인
 * ===================================================
 * Quick Time Event (QTE) 시스템을 구현하는 플러그인입니다.
 * 원형 테두리와 입력 버튼 이미지를 사용하여 특정 타이밍에
 * 키를 입력하여 성공/실패를 판정할 수 있습니다.
 *
 * ■ 사용법
 * 1. QTE 플러그인을 활성화합니다.
 * 2. 플러그인 커맨드를 이벤트에서 호출하여 QTE를 설정하세요.
 * 3. QTEgood.png, QTEbad.png, 그리고 각 방향 키 이미지
 *    (QTEup.png, QTEdown.png, QTEleft.png, QTEright.png)를 
 *    프로젝트의 /img/pictures/ 폴더에 준비하세요.
 *
 * ■ 플러그인 명령어
 * QTE start x y button image speed time var successVar successSE failSE
 * - x: QTE 버튼의 x좌표 (화면상 위치)
 * - y: QTE 버튼의 y좌표 (화면상 위치)
 * - button: 입력 키 (up, down, left, right)
 * - image: 버튼 이미지 파일명 (예: QTEup)
 * - speed: 테두리 크기 줄어드는 속도 (기본값 120)
 * - time: 입력 유예 시간 (기본값 30)
 * - var: 입력 성공 여부를 저장할 변수 번호
 * - successVar: 성공 시 더할 변수 값
 * - successSE: 성공 시 재생할 SE 파일명.
 * - failSE: 실패 시 재생할 SE 파일명.
 *
 * QTE combo hide  (QTE 콤보 카운터 숨기기)
 * QTE combo show  (QTE 콤보 카운터 표시)
 * 예제:
 * QTE combo hide
 * QTE start 500 300 right QTEright 120 30 1 10 QTESuccess QTEFail
 * => 화면 (500, 300)에 QTE 버튼이 표시되고, `오른쪽 키`를 입력. 콤보 표시하지 않음.
 *    성공 시 변수 1에 10을 더합니다. 성공시 QTESuccess 사운드가 실패시 QTEFail 사운드가 재생됩니다.
 *    ReadyToRumble.js 플러그인을 사용 할 경우 성공 입력시에 게임 패드 진동기능이 생깁니다. 기본설정 Rumble(0.3, 0.3, 200)
 *
 * ■ 준비할 이미지 파일
 * 1. QTEgood.png: 성공 시 표시할 이미지
 * 2. QTEbad.png: 실패 시 표시할 이미지
 * 3. QTEup.png, QTEdown.png, QTEleft.png, QTEright.png:
 *    각 방향 키의 버튼 이미지
 * 4. QTEg.png: 테두리 이미지 (중앙 버튼 크기와 동일한 원형 테두리)
 *   중요!  원형 버튼과 원형의 테두리 이미지는 크기가 같아야 합니다. 
 *
 * ===================================================
 * kar_QTE Plugin
 * Version: v1.0
 * ===================================================
 *
 * ■ 説明 (日本語)
 * このプラグインはQTEシステムを導入するためのものです。
 * QTEイベントはタイミングよくボタンを押すことで成功・失敗を判定します。
 *
 * ■ 使用方法
 * 1. プラグインを有効化してください。
 * 2. プラグインコマンドをイベントで呼び出してください。
 * 3. 必要な画像をプロジェクトの /img/pictures/ フォルダに用意してください。
 *
 * ■ プラグインコマンド
* QTE start x y button image speed time var successVar successSE failSE
* - x: QTEボタンのx座標 (画面上の位置)
* - y: QTEボタンのy座標(画面上位置)
* - button: 입력 키 (up, down, left, right)
* - image:ボタンイメージファイル名(例:QTEup)
* - speed: 枠 サイズ 減る 速度 (基本値 120)
* - time:入力猶予時間(基本値30)
* - var: 入力の成否を保存する変数番号
* - successVar:成功時に追加する変数値
* - success SE : 成功時に再生するSE ファイル名。
* - failSE:失敗時に再生するSEファイル名。
*
* QTE combo hide (QTE コンボ カウンター 隠す)
* QTE combo show(QTE コンボ カウンター 表示)
* 例題:
* QTE combo hide
* QTE start 500 300 right QTEright 120 30 1 10 QTESuccess QTEFail
* =>画面(500,300)にQTEボタンが表示され、「右キー」を入力。 コンボ表示しない。
* 成功時に変数 1 に 10 を加えます。 成功するとQTESuccessサウンドが失敗すると、QTEFailサウンドが再生されます。
* Ready TORumble.jsプラグインを使用する場合、成功入力時にゲームパッド振動機能が発生します。 基本設定 Rumble(0.3, 0.3, 200)
 *
 * ■ 必要な画像ファイル
 * 1. QTEgood.png: 成功時に表示される画像
 * 2. QTEbad.png: 失敗時に表示される画像
 * 3. QTEup.png, QTEdown.png, QTEleft.png, QTEright.png:
 *    各方向キーのボタン画像
 * 4. QTEg.png: 枠線画像 (中央ボタンと同じ円形枠)
 *
 * ===================================================
 *
 * ■ 업데이트 로그
 * v1.0 - 기본 QTE 시스템 구현
 * 
 * ===================================================
 */
function isGamepadConnected() {
    const gamepads = navigator.getGamepads ? navigator.getGamepads() : [];
    return gamepads && gamepads[0] && gamepads[0].connected; // 연결 여부 확인
}

function getGamepadButtonIndex(action) {
    const mapping = Input.gamepadMapper || {}; // 매퍼가 없을 경우 기본값으로 빈 객체 사용
    for (let index in mapping) {
        if (mapping[index] === action) return Number(index);
    }
    return -1; // 해당 액션이 없으면 -1 반환
}

function getGamepadButtonImage(buttonIndex) {
    switch (buttonIndex) {
        case 0: return "QTEA"; // A 버튼
        case 1: return "QTEB"; // B 버튼
        case 2: return "QTEX"; // X 버튼
        case 3: return "QTEY"; // Y 버튼
        case 12: return "QTEup"; // 위 방향키
        case 13: return "QTEdown"; // 아래 방향키
        case 14: return "QTEleft"; // 왼쪽 방향키
        case 15: return "QTEright"; // 오른쪽 방향키
        default: 
            console.warn(`알 수 없는 게임패드 버튼 인덱스: ${buttonIndex}`);
            return null; // 이미지 없음
    }
}

function getKeyboardButtonImage(button) {
    switch (button) {
        case "ok": return "QTEZ"; // Z/Enter/Space
        case "cancel": return "QTEEsc"; // X/Esc
        case "up": return "QTEup"; // 위 방향키
        case "down": return "QTEdown"; // 아래 방향키
        case "left": return "QTEleft"; // 왼쪽 방향키
        case "right": return "QTEright"; // 오른쪽 방향키
        default:
            console.warn(`알 수 없는 키보드 버튼: ${button}`);
            return null; // 이미지 없음
    }
}

(function () {
    let activeQTEs = []; // 실행 중인 QTE 리스트
    let comboCount = 0; // 현재 콤보 카운트
    let comboSprite = null; // 콤보 표시 스프라이트
    let comboVisible = true; // 콤보 표시 상태 (기본값: true)

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (command === "QTE") {
            if (args[0] === "start") {
                const [x, y, button, image, speed, tolerance, variable, successValue, successSE, failSE] = args.slice(1);
                startQTE(
                    Number(x), Number(y), button, image,
                    Number(speed), Number(tolerance),
                    Number(variable), Number(successValue), successSE, failSE
                );
            } else if (args[0] === "combo") {
                if (args[1] === "show") {
                    comboVisible = true;
                    if (!comboSprite) {
                        comboSprite = createComboSprite();
                    }
                    updateComboSprite();
                } else if (args[1] === "hide") {
                    comboVisible = false;
                    if (comboSprite) {
                        comboSprite.bitmap.clear(); // 콤보 숨김
                    }
                }
            }
        }
    };

    function startQTE(x, y, button, image, speed, tolerance, variable, successValue, successSE, failSE) {
        const qteData = {
            x, y, button, image, speed, tolerance, variable, successValue,
            successSE, failSE,
            frame: 0,
            success: false,
            attempted: false,
            validStartFrame: 0,
            validEndFrame: 0,
            buttonSprite: null,
            borderSprite: null,
        };

        activeQTEs.push(qteData);
        createQTEUI(qteData);

        // 성공 입력 유효 시간 계산
        qteData.validStartFrame = Math.floor((speed / 4) * 3); // 테두리 축소가 1배 크기로 되는 시점
        qteData.validEndFrame = qteData.validStartFrame + tolerance;
    }

    function createQTEUI(data) {
        const scene = SceneManager._scene;

        let buttonImage = getKeyboardButtonImage(data.button); // 버튼 이미지 결정 (기본 키보드)
        if (isGamepadConnected()) {
            const gamepadButtonIndex = getGamepadButtonIndex(data.button); // 게임패드 매핑 확인
            buttonImage = getGamepadButtonImage(gamepadButtonIndex); // 매핑된 게임패드 이미지
        }

        if (!buttonImage) {
            console.error(`QTE 이벤트의 버튼 이미지가 유효하지 않습니다: ${data.button}`);
            return;
        }

        const buttonSprite = new Sprite(ImageManager.loadPicture(buttonImage));
        const borderSprite = new Sprite(ImageManager.loadPicture("QTEg"));

        buttonSprite.anchor.set(0.5, 0.5);
        borderSprite.anchor.set(0.5, 0.5);

        buttonSprite.x = data.x;
        buttonSprite.y = data.y;
        borderSprite.x = data.x;
        borderSprite.y = data.y;

        borderSprite.scale.set(4, 4); // 초기 크기

        scene.addChild(borderSprite);
        scene.addChild(buttonSprite);

        data.buttonSprite = buttonSprite;
        data.borderSprite = borderSprite;

        // 콤보 표시 초기화
        if (!comboSprite) {
            comboSprite = createComboSprite();
        }
    }

    function createComboSprite() {
        const scene = SceneManager._scene;
        const sprite = new Sprite(new Bitmap(Graphics.width, Graphics.height));
        sprite.bitmap.fontSize = 32; // 콤보 표시 글자 크기
        sprite.bitmap.textColor = "#ffffff"; // 글자 색상
        sprite.x = Graphics.width / 2;
        sprite.y = 50; // 화면 상단 중앙
        sprite.anchor.set(0.5, 0); // 중앙 정렬
        scene.addChild(sprite);
        return sprite;
    }

    function updateComboSprite() {
        if (!comboVisible || comboCount <= 0) {
            if (comboSprite) comboSprite.bitmap.clear();
            return;
        }

        comboSprite.bitmap.clear();
        comboSprite.bitmap.drawText(`COMBO: ${comboCount}`, 0, 0, Graphics.width, 48, "center");
    }

function updateQTE() {
    if (activeQTEs.length === 0) return;

    activeQTEs.forEach((data, index) => {
        if (data.attempted) return; // 이미 처리된 QTE는 스킵

        data.frame++;

        // 테두리 크기 줄어들기
        const progress = Math.max(1, 4 - (data.frame / data.speed) * 4);
        data.borderSprite.scale.set(progress, progress);

        // 성공 체크
        if (!data.success && progress <= 1 && data.frame >= data.validStartFrame && data.frame <= data.validEndFrame) {
            if (Input.isTriggered(data.button)) {
				    if (typeof Rumble === "function") {
        Rumble(0.3, 0.3, 200); // 진동 효과 추가
    }  
                data.attempted = true; // 입력 시도 플래그 설정
                handleSuccess(data, index);
            }
        }

        // 실패 처리 (유효 시간 종료 후)
        if (!data.success && data.frame > data.speed + data.tolerance) {
            data.attempted = true;
            handleFailure(data, index);
        }
    });
}



    function handleSuccess(data, index) {
        data.success = true;
        $gameVariables.setValue(data.variable, $gameVariables.value(data.variable) + data.successValue);

        if (data.successSE) {
            AudioManager.playSe({ name: data.successSE, volume: 90, pitch: 100 });
        }

        createResultImage("QTEgood", data.x, data.y);

        // 콤보 증가
        comboCount++;
        updateComboSprite();

        endQTE(index);
    }

    function handleFailure(data, index) {
        if (data.failSE) {
            AudioManager.playSe({ name: data.failSE, volume: 90, pitch: 100 });
        }

        createResultImage("QTEbad", data.x, data.y);

        // 콤보 초기화
        comboCount = 0;
        updateComboSprite();

        endQTE(index);
    }

    function createResultImage(imageName, x, y) {
        const scene = SceneManager._scene;
        const resultSprite = new Sprite(ImageManager.loadPicture(imageName));

        resultSprite.anchor.set(0.5, 0.5);
        resultSprite.x = x;
        resultSprite.y = y;

        scene.addChild(resultSprite);

        setTimeout(() => {
            scene.removeChild(resultSprite);
        }, 1000);
    }

    function endQTE(index) {
        const data = activeQTEs[index];
        const scene = SceneManager._scene;

        scene.removeChild(data.buttonSprite);
        scene.removeChild(data.borderSprite);

        activeQTEs.splice(index, 1);
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        updateQTE();
    };
})();


