//=============================================================================
// RPG Maker MZ - SunD_VRM
//=============================================================================

/*:
 * @target MZ
 * @plugindesc VRMモデル表示
 * @author MUMAMOMEMU
 * @url https://mumamomemu.booth.pm/
 * @base PluginCommonBase
 * @orderAfter PluginCommonBase
 *
 * @help SunD_VRM.js(ver1.0.1)
 *
 * SunD_VRM_MOD.jsと同じフォルダに配置してください。
 *
 * --------------------------
 * ■利用規約
 *
 * Copyright (c) 2024 MUMAMOMEMU
 * https://star-write-dream.com/
 *
 * 以下に定める条件に従い、本プラグインファイル（以下「ソフトウェア」）の
 * 購入者に対し、ソフトウェアを変更し、
 * 購入者の制作する作品（以下、制作物）に組み込むことを許可します。
 * 制作物の内容および公開形式に制約はありません。
 *
 * ソフトウェア単体で掲載、頒布、共有することはできません。
 * 上記の著作権表示および本許諾表示は変更できず、削除もできません。
 * 別途、制作物の重要な箇所にソフトウェアの著作権表示をする必要はありません。
 *
 * ソフトウェアは「現状のまま」で、明示であるか暗黙であるかを問わず、
 * 何らの保証もなく提供されます。ここでいう保証とは、商品性、
 * 特定の目的への適合性、および権利非侵害についての保証も含みますが、
 * それに限定されるものではありません。作者または著作権者は、契約行為、
 * 不法行為、またはそれ以外であろうと、ソフトウェアに起因または関連し、
 * あるいはソフトウェアの使用またはその他の扱いによって生じる一切の請求、
 * 損害、その他の義務について何らの責任も負わないものとします。
 * --------------------------
 *
 * @param models
 * @type struct<VrmModel>[]
 * @text VRMリスト
 * @desc VRMリストを登録します。
 * @default []
 *
 * @param motions
 * @type struct<Motion>[]
 * @text VRMAリスト
 * @desc VRMAリストを登録します。
 * @default []
 *
 * @param positions
 * @type struct<Position>[]
 * @text ポジションリスト
 * @desc ポジションリストを登録します。
 * @default ["{\"keyName\":\"center\",\"position\":\"0,-0.8,0\"}","{\"keyName\":\"right\",\"position\":\"0.9,-0.8,0\"}","{\"keyName\":\"left\",\"position\":\"-0.9,-0.8,0\"}","{\"keyName\":\"near_center\",\"position\":\"0,-1.3,1.8\"}","{\"keyName\":\"near_right\",\"position\":\"0.4,-1.3,1.8\"}","{\"keyName\":\"near_left\",\"position\":\"-0.4,-1.3,1.8\"}"]
 *
 * @param directions
 * @type struct<ModelRotation>[]
 * @text 向きリスト
 * @desc 向きのリストを登録します。
 * @default ["{\"keyName\":\"default\",\"y\":\"0\"}","{\"keyName\":\"right\",\"y\":\"45\"}","{\"keyName\":\"left\",\"y\":\"-45\"}"]
 *
 * @param cameras
 * @type struct<Camera>[]
 * @text カメラリスト
 * @desc カメラリストを登録します。
 * @default ["{\"keyName\":\"default\",\"position\":\"0,0,3\",\"rotation\":\"0,0,0\"}","{\"keyName\":\"lookDown\",\"position\":\"0.44,1.26,0.49\",\"rotation\":\"-58.75,26.03,35.87\"}"]
 *
 * @param devModeSW
 * @type switch
 * @text 開発モードスイッチ
 * @desc 指定のスイッチがONの間、開発モードとなります。
 * @default 0
 *
 * @param useSave
 * @type boolean
 * @text セーブする
 * @desc ON:表示可能モデルとカメラの情報をセーブします。
 * @default true
 *
 * @command showVRM
 * @text モデルの表示
 * @desc ロードされているモデルを表示します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc 表示したいモデルを指定します。複数指定可
 * @default
 *
 * @command hideVRM
 * @text モデルを隠す
 * @desc モデルを隠します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc 隠したいモデルを指定します。複数指定可
 * @default
 *
 * @command setPosition
 * @text 位置変更
 * @desc モデルの位置を変更します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc 位置を変更するモデルを指定します。複数指定可
 * @default
 *
 * @arg position
 * @type string
 * @text 位置
 * @desc パラメータで登録したキーを指定します。x,y,zで指定した場合、現行値に加算します。
 * @default
 *
 * @arg frame
 * @type string
 * @text フレーム
 * @desc 指定したフレーム数を使って移動します。
 * @default 1
 *
 * @command setDirection
 * @text 向き変更
 * @desc ロードしたモデルの向きを変更します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc 位置を変更するモデルを指定します。複数指定可
 * @default
 *
 * @arg modelRotation
 * @type string
 * @text 向き
 * @desc キーを指定します。-180から180の数値で指定した場合、現在値に加算します。「camera」でカメラを向きます。
 * @default
 *
 * @arg frame
 * @type string
 * @text フレーム
 * @desc 指定したフレーム数を使って移動します。
 * @default 1
 *
 * @command setCamera
 * @text カメラ変更
 * @desc カメラの設定を変更します。
 *
 * @arg cameraKey
 * @type string
 * @text カメラ名
 * @desc カメラのキーを指定します。
 * @default
 *
 * @command cameraLookAt
 * @text カメラをモデルに向ける
 * @desc カメラの方向を指定したモデルに向けます。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルのキーを指定します。
 * @default
 *
 * @arg adjustY
 * @type string
 * @text Y調整
 * @desc Yの調整値を指定します。
 * @default 0.8
 *
 * @command autoBlink
 * @text オートまばたき
 * @desc 自動でまばたきするかどうかを設定します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 * @arg enabled
 * @type boolean
 * @text 有効化
 * @desc ON:自動でまばたきします。別の表情を設定したい場合はOFFにしてください。
 * @default false
 *
 * @command lookCamera
 * @text カメラ目線
 * @desc カメラ目線にするかどうかを設定します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 * @arg enabled
 * @type boolean
 * @text 有効化
 * @desc ON:カメラ目線にします。
 * @default false
 *
 *
 * @command setExpression
 * @text 表情設定
 * @desc 表情を設定します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 * @arg expression
 * @type select
 * @text 表情
 * @desc 表情を選択します。
 * @default neutral
 *
 * @option ニュートラル
 * @value neutral
 * @option 怒り
 * @value angry
 * @option ハッピー
 * @value happy
 * @option リラックス
 * @value relaxed
 * @option 悲しみ
 * @value sad
 * @option おどろき
 * @value surprised
 * @option --------
 * @option 目線：上
 * @value lookUp
 * @option 目線：下
 * @value lookDown
 * @option 目線：左
 * @value lookLeft
 * @option 目線：右
 * @value lookRight
 * @option 目線：リセット
 * @value lookReset
 * @option --------
 * @option まばたき
 * @value blink
 * @option 左ウインク
 * @value blinkLeft
 * @option 右ウインク
 * @value blinkRight
 * @option --------
 * @option 口：ア
 * @value aa
 * @option 口：イ
 * @value ih
 * @option 口：ウ
 * @value ou
 * @option 口：エ
 * @value ee
 * @option 口：オ
 * @value oh
 * @option 口：リセット
 * @value mouthReset
 *
 * @arg value
 * @type string
 * @text 強さ
 * @desc 表情の強さを指定します。
 * @default 100
 *
 * @command disabledExpression
 * @text 表情機能の無効化
 * @desc プラグインによる表情設定を無効化します。(VRMAに表情が設定されている場合に使用)
 * 
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 * 
 * @arg disabled
 * @text 無効化する
 * @type boolean
 * @desc ON:プラグインによる表情設定を無効化 OFF:解除
 * @default off
 *
 * @command startMotion
 * @text モーションスタート
 * @desc モーションをスタートします。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 * @arg motionKey
 * @type string
 * @text モーション
 * @desc VRMAをキーで指定します。
 * @default
 *
 * @arg loop
 * @type select
 * @text ループ
 * @desc ループを設定します。
 * @default enabled
 *
 * @option ループする
 * @value enabled
 * @option ループしない
 * @value disabled
 * @option ピンポンループする
 * @value ping-pong
 *
 * @arg speed
 * @type string
 * @text 速度
 * @desc 再生速度を指定します。0で一時停止、100で等倍再生、-100で逆再生
 * @default 100
 *
 * @arg nextMotionKey
 * @type string
 * @text 次のモーションキー
 * @desc ループしない場合にのみ使用されます。モーション完了後、このモーションが速度1でループ再生されます。
 * @default
 *
 * @command motionSpeed
 * @text モーションの再生速度
 * @desc 実行中のモーションの再生速度を変更します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。
 * @default
 *
 * @arg speed
 * @type string
 * @text 速度
 * @desc 再生速度を指定します。0で一時停止、100で等倍再生、-100で逆再生
 * @default 100
 *
 * @command autoMouth
 * @text 口パク
 * @desc 文章の表示でモデルの口を動かします。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可。空欄でリセット。
 * @default
 *
 * @command dispose
 * @text モデルの破棄
 * @desc モデルを破棄します。
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 * @command loadSkipList
 * @text ロードスキップリストの編集
 * @desc ロードスキップリストを変更します。
 *
 * @arg mode
 * @type select
 * @text モード
 * @desc リストに追加するか、削除するかを指定します。
 * @default add
 *
 * @option 追加
 * @value add
 * @option 削除
 * @value remove
 *
 * @arg modelKey
 * @type string
 * @text モデル
 * @desc モデルをキーで指定します。複数指定可
 * @default
 *
 */

/*~struct~VrmModel:
 *
 * @param keyName
 * @type string
 * @text キー
 * @desc モデルの名前を指定。編集画面ではこの名前を使用します。「all」は使用不可。
 * @default
 *
 * @param fileName
 * @type string
 * @text VRM
 * @desc vrmフォルダ内のVRMを指定    vrm/goblin.vrm -> goblin    vrm/enemy/goblin.vrm -> enemy/goblin
 * @default
 *
 * @param preLoad
 * @type boolean
 * @text 必ずロード
 * @desc マップ開始時に必ず読み込みます。
 * @default true
 *
 */

/*~struct~Motion:
 *
 * @param keyName
 * @type string
 * @text キー
 * @desc アニメーションの名前を指定。編集画面ではこの名前を使用します。
 * @default
 *
 * @param fileName
 * @type string
 * @text VRMA
 * @desc vrm/vrma内のVRMAを指定    vrm/vrma/dance.vrma -> dance    vrm/vrma/enemy/dance.vrma -> enemy/dance
 * @default
 *
 */
/*~struct~Position:
 *
 * @param keyName
 * @type string
 * @text キー
 * @desc ポジションの名前を指定。編集画面ではこの名前を使用します。
 * @default
 *
 * @param position
 * @type string
 * @text ポジション
 * @desc ポジションをx,y,zで指定
 * @default 0,0,0
 *
 */
/*~struct~ModelRotation:
 *
 * @param keyName
 * @type string
 * @text キー
 * @desc 向きの名前を指定。編集画面ではこの名前を使用します。「camera」はシステム専用のため不可。
 * @default
 *
 * @param rotationY
 * @type number
 * @text 回転
 * @desc 回転Yを-180から180で指定
 * @default 0
 * @max 180
 * @min -180
 *
 */
/*~struct~Camera:
 *
 * @param keyName
 * @type string
 * @text キー
 * @desc カメラの名前を指定。編集画面ではこの名前を使用します。
 * @default
 *
 * @param position
 * @type string
 * @text ポジション
 * @desc ポジションをx,y,zで指定
 * @default 0,0,0
 *
 * @param rotation
 * @type string
 * @text 回転
 * @desc 回転をx,y,zで指定
 * @default 0,0,0
 *
 */

/*
■変更履歴
・v1.0.1
  コマンド「表情機能の無効化」を追加。
・v1.0.0
  公開
*/

if (!SUND_MODS) {
    // SunD_VRM_MODで設定
    var SUND_MODS = {
        THREE: undefined,
        GLTFLoader: undefined,
        OrbitControls: undefined,
        THREE_VRM: undefined,
        THREE_VRM_Animation: undefined,
    };
}

// メインデータ
var $sundVRMs = new Map();

(() => {
    "use strict";
    const script = document.currentScript;
    const params = PluginManagerEx.createParameter(script);

    // -----------------
    // SUND_VRM

    class SUND_VRM {
        static _params = {
            vrmMap: new Map(),
            vrmaMap: new Map(),
            positionMap: new Map(),
            directionMap: new Map(),
            cameraMap: new Map(),
            loadModels: new Set(),
            devModeSW: params.devModeSW,
        };

        static _renderer;
        static _scene;
        static _camera;
        static _light;
        static _animations = new Map();
        static _threeSprite;
        static _visibleAnyModel = false;
        static _AXIS_X = { x: 1, y: 0, z: 0 };
        static _AXIS_Y = { x: 0, y: 1, z: 0 };
        static _AXIS_Z = { x: 0, y: 0, z: 1 };
        static _dev = { touchX: null, touchY: null };
        static _autoMouth = { targets: [], duration: 0 };
        static _readyState = "";

        static #loader;
        static get loader() {
            return this.#loader;
        }

        static #moduleLoaded = false;
        static get moduleLoaded() {
            return this.#moduleLoaded;
        }

        static #clock;
        static get clock() {
            return this.#clock;
        }

        static #deltaTime = 0;
        static get deltaTime() {
            return this.#deltaTime;
        }

        // SunD_VRM_MODが実行する
        static setupMOD() {
            const loader = new SUND_MODS.GLTFLoader();
            this.#loader = loader;
            loader.register((parser) => {
                return new SUND_MODS.THREE_VRM.VRMLoaderPlugin(parser);
            });

            loader.register((parser) => {
                return new SUND_MODS.THREE_VRM_Animation.VRMAnimationLoaderPlugin(parser);
            });

            this._params.vrmaMap.forEach((value, key) => {
                this.loadVRMA(key, value);
            });

            // まだシステム設定の読み取り前
            const w = 816;
            const h = 624;

            const three = SUND_MODS.THREE;
            const scene = new three.Scene();
            this._scene = scene;

            const renderer = new three.WebGLRenderer({ alpha: true });
            renderer.setSize(w, h);
            renderer.domElement.style.zIndex = 0;
            renderer.domElement.style.position = "absolute";
            document.body.appendChild(renderer.domElement);
            this._renderer = renderer;

            const camera = new three.PerspectiveCamera(45, w / h, 0.1, 10000);
            camera.position.set(0, 0, 3);
            this._camera = camera;
            scene.add(camera);

            const controls = new SUND_MODS.OrbitControls(camera, renderer.domElement);
            controls.enabled = false;
            // この関数は独自に追加したもの。listenToKeyEvents()のマウス版
            controls.listenToMouseEvents(document.body);
            this._controls = controls;

            const light = new three.DirectionalLight(0xffffff, Math.PI);
            light.position.set(1.0, 1.0, 1.0);
            scene.add(light);
            this._light = light;

            const clock = new three.Clock();
            this.#deltaTime = clock.getDelta();
            this.#clock = clock;

            this.#moduleLoaded = true;
            Graphics._updateThreeElements();
        }

        static loadVRM(keyName, fileName) {
            this.loader.load(
                "./vrm/" + fileName + ".vrm",
                (gltf) => {
                    this.setupVRM(keyName, gltf);
                },
                (progress) => { },
                (error) => {
                    // ゲーム公開形式によってリトライ検討が必要
                    console.error(error);
                }
            );
        }

        static setupVRM(keyName, gltf) {
            const obj = $sundVRMs.get(keyName);
            const scene = this._scene;
            const vrm = gltf.userData.vrm;
            // 表示状態で配置。isReadyの過程で非表示にする
            scene.add(vrm.scene);

            const proxy = new SUND_MODS.THREE_VRM_Animation.VRMLookAtQuaternionProxy(vrm.lookAt);
            proxy.name = "VRMLookAtQuaternionProxy";
            vrm.scene.add(proxy);

            // デフォルトでやや下げる
            vrm.scene.position.y -= 0.8;

            obj._gltf = gltf;
            obj._vrm = vrm;
            obj._mixer = new SUND_MODS.THREE.AnimationMixer(vrm.scene);
            obj._needsTrialRender = true;
            obj._loaded = true;

            obj.startMotion({ motionKey: "idle", nextMotionKey: "", loop: "enabled", speed: 1 });
        }

        static loadVRMA(keyName, fileName) {
            this.loader.load(
                "./vrm/vrma/" + fileName + ".vrma",
                (gltf) => {
                    this.setupVRMA(keyName, gltf);
                },
                (progress) => { },
                (error) => {
                    // ゲーム公開形式によってリトライ検討が必要
                    console.error(error);
                }
            );
        }

        static setupVRMA(keyName, gltf) {
            const vrmAnimations = gltf.userData.vrmAnimations;
            if (!vrmAnimations || !vrmAnimations[0]) return;
            this._animations.set("" + keyName, vrmAnimations[0]);
        }

        static setActionLoop(strLoop, action) {
            switch (strLoop) {
                case "enabled":
                    action.setLoop(SUND_MODS.THREE.LoopRepeat);
                    break;
                case "ping-pong":
                    action.setLoop(SUND_MODS.THREE.LoopPingPong);
                    break;
                case "disabled":
                    action.setLoop(SUND_MODS.THREE.LoopOnce);
                    break;
                default:
                    action.setLoop(SUND_MODS.THREE.LoopRepeat);
            }
        }

        static dispose(keyName) {
            const obj = $sundVRMs.get(keyName);
            if (!obj) return;

            $sundVRMs.delete(keyName);

            // 念のため
            obj._animationActions.clear();
            obj._mixer._initMemoryManager();
            obj._mixer = null;

            // geometry,skeleton,material,texture
            const vrm = obj._vrm;
            const scene = vrm.scene;
            obj._vrm = null;
            SUND_MODS.THREE_VRM.VRMUtils.deepDispose(scene);

            // ImageBitmap,texture
            const cache = obj._gltf.parser.sourceCache;
            for (let key in cache) {
                cache[key].then(texture => {
                    if (texture.image && texture.image.close) {
                        texture.image.close();
                    }
                    texture.dispose();
                });
            }
            obj._gltf = null;

            // 絶縁
            scene.removeFromParent();
        }

        static isDevMode() {
            return $gameSwitches.value(this._params.devModeSW) && this._visibleAnyModel;
        }

        static dontMovePlayer() {
            return this.isDevMode();
        }

        static isLoadSkipModel(keyName) {
            return $gameParty._sundVRM.loadSkipModels.includes(keyName);
        }

        static allLoaded() {
            return !Array.from($sundVRMs.values()).some((el) => !el._loaded);
        }

        static isReady() {
            if (!this.moduleLoaded) return false;

            switch (this._readyState) {
                case "start":
                    this.mapLoadVRM();
                    this._readyState = "loading";
                    return false;
                case "loading":
                    if (this.allLoaded()) {
                        this._readyState = "trialRender";
                    }
                    return false;
                case "trialRender":
                    //初回レンダリングに時間がかかるのでやっておく
                    this._renderer.render(this._scene, this._camera);
                    $sundVRMs.forEach((el) => {
                        if (el._needsTrialRender) {
                            el._needsTrialRender = false;
                            el.hide();
                        }
                    });
                    this._readyState = "restore";
                    return false;
                case "restore":
                    $sundVRMs.forEach((el) => {
                        el.restoreContinueData();
                    });
                    this._readyState = "finished";
                    return false;
                case "finished":
                    return true;
                default:
                    return true;
            }
        }

        static mapLoadVRM() {
            const loadSet = new Set();
            const data = $gameParty._sundVRM;
            // 必ずロード
            this._params.loadModels.forEach((keyName) => {
                loadSet.add(keyName);
            });
            // マップ指定
            if (!DataManager.isEventTest() && $dataMap.meta["Load_VRM"]) {
                const keys = this.modelKeyToArray($dataMap.meta["Load_VRM"]);
                keys.forEach((key) => {
                    loadSet.add(key);
                });
            }
            // 保存データ
            if (this._afterContinue) {
                data.modelSaveData.forEach((el) => {
                    loadSet.add(el.keyName);
                });

                this.restoreCameraData();
            }

            // 読み込み
            loadSet.forEach((keyName) => {
                if (!this.isLoadSkipModel(keyName)) {
                    const obj = this.create(keyName);
                    if (this._afterContinue) {
                        const continueData = data.modelSaveData.find((el) => el.keyName === keyName);
                        obj._continueData = continueData;
                    }
                }
            });
            this._afterContinue = false;
        }

        static create(keyName) {
            if ($sundVRMs.has(keyName)) {
                return $sundVRMs.get(keyName);
            }
            const fileName = this._params.vrmMap.get(keyName);
            const obj = new this(keyName);
            $sundVRMs.set(keyName, obj);
            this.loadVRM(keyName, fileName);
            return obj;
        }

        static updateMap() {
            this.updateMapVRM();
            this.updateDevCamera();
            this.#deltaTime = this.clock.getDelta();
            this._renderer.render(this._scene, this._camera);
            this.drawThreeSprite();
        }

        static updateMapVRM() {
            let visible = false;
            $sundVRMs.forEach((el) => {
                if (!visible && el._vrm && el.visible()) {
                    visible = true;
                }
                el.update();
            });
            this._visibleAnyModel = visible;
        }

        static updateDevCamera() {
            if (!this._controls) return;
            if (!this.isDevMode() || Input.isPressed("shift")) {
                this._controls.enabled = false;
                return;
            }
            this._controls.enabled = true;
            this.updateDevCameraInformation();
        }

        static updateDevCameraInformation() {
            if (!Input.isPressed("control")) return;

            const cam = this._camera;
            if (Input.isTriggered("pageup")) {
                try {
                    this.copyClipboard(this.devNum(cam.position.x) + "," + this.devNum(cam.position.y) + "," + this.devNum(cam.position.z));
                } catch { }
                console.log(`==== [camera] ====`);
                console.log(`position: x:${this.devNum(cam.position.x)}, y:${this.devNum(cam.position.y)}, z:${this.devNum(cam.position.z)}`);
            } else if (Input.isTriggered("pagedown")) {
                const rX = SUND_MODS.THREE.MathUtils.radToDeg(cam.rotation.x);
                const rY = SUND_MODS.THREE.MathUtils.radToDeg(cam.rotation.y);
                const rZ = SUND_MODS.THREE.MathUtils.radToDeg(cam.rotation.z);
                try {
                    this.copyClipboard(this.devNum(rX) + "," + this.devNum(rY) + "," + this.devNum(rZ));
                } catch { }
                console.log(`==== [camera] ====`);
                console.log(`rotation: x:${this.devNum(rX)}, y:${this.devNum(rY)}, z:${this.devNum(rZ)}`);
            }
        }

        static copyClipboard(str) {
            if (!Utils.isOptionValid('test')) return;
            navigator.clipboard.writeText(str);
        }

        static drawThreeSprite() {
            // threeのcanvasは背面に隠されている。bltコピー
            if (!this._threeSprite) return;

            const bitmap = this._threeSprite.bitmap;
            bitmap.clear();
            const w = Graphics.width;
            const h = Graphics.height;
            bitmap.blt({ _canvas: this._renderer.domElement }, 0, 0, w, h, 0, 0);
        }

        static updateAutoMouth() {
            if (this._autoMouth.duration <= 0 || this._autoMouth.targets.length === 0) return;
            const m = this._autoMouth;

            const duration = --m.duration;
            const isRandom = duration % 4 === 0 && duration > 0;
            const isReset = duration <= 0;

            m.targets.forEach((el) => {
                const obj = $sundVRMs.get(el);
                if (obj) {
                    if (isRandom) {
                        obj.setRandomMouth();
                    } else if (isReset) {
                        obj.resetMouth();
                    }
                }
            });
        }

        static stopAutoMouth() {
            const m = this._autoMouth;
            m.duration = 0;
            const targets = [...m.targets];
            targets.forEach((el) => {
                const obj = $sundVRMs.get(el);
                if (obj) {
                    obj.resetMouth();
                }
            });
        }

        static setSaveData() {
            if (!params.useSave) return;

            const saveArr = [];
            $sundVRMs.forEach((el) => {
                const scene = el._vrm.scene;
                const m = el._motion;
                const obj = {
                    keyName: el._keyName,
                    position: this.makeObjectXYZ(),
                    rotation: this.makeObjectXYZ(),
                    visible: scene.visible,
                    motion: {
                        current: { name: m.current.name, loop: m.current.loop, speed: m.current.speed },
                        next: { name: m.next.name, loop: m.next.loop, speed: m.next.speed },
                        state: m.state,
                    },
                };
                this.assignmentXYZ(obj.position, scene.position);
                this.assignmentXYZ(obj.rotation, scene.rotation);
                saveArr.push(obj);
            });
            $gameParty._sundVRM.modelSaveData = saveArr;

            const cameraData = $gameParty._sundVRM.cameraSaveData;
            const camera = SUND_VRM._camera;
            this.assignmentXYZ(cameraData.position, camera.position);
            this.assignmentXYZ(cameraData.rotation, camera.rotation);
        }

        static restoreCameraData() {
            const data = $gameParty._sundVRM.cameraSaveData;
            const cam = SUND_VRM._camera;
            SUND_VRM.assignmentXYZ(cam.position, data.position);
            const rot = data.rotation;
            cam.rotateX(rot.x);
            cam.rotateY(rot.y);
            cam.rotateZ(rot.z);
        }

        static allHide() {
            $sundVRMs.forEach((el) => {
                el.hide();
            });
        }
        static allShow() {
            $sundVRMs.forEach((el) => {
                el.show();
            });
        }

        static makeObjectXYZ(x = 0, y = 0, z = 0) {
            return { x: x, y: y, z: z };
        }

        static assignmentXYZ(to, from) {
            to.x = from.x;
            to.y = from.y;
            to.z = from.z;
        }

        static stringTrim(key) {
            return ("" + key).trim();
        }

        static devNum(value) {
            return Math.round(value * 100) / 100;
        }

        static rotationWork180(value) {
            if (value > 180) {
                return -180 + (value - 180);
            } else if (value < -180) {
                return 180 + (value + 180);
            }
            return value;
        }

        static getRotationDegY(rotation) {
            const util = SUND_MODS.THREE.MathUtils;
            if (rotation.x <= -3.14) {
                if (rotation.y >= 0) {
                    return 180 - util.radToDeg(rotation.y);
                } else {
                    return -180 - util.radToDeg(rotation.y);
                }
            }
            return util.radToDeg(rotation.y);
        }

        static makeDegreesObject(radiansObj) {
            const obj = Object.assign({}, radiansObj);
            const util = SUND_MODS.THREE.MathUtils;
            obj.x = util.radToDeg(obj.x);
            obj.y = util.radToDeg(obj.y);
            obj.z = util.radToDeg(obj.z);
            return obj;
        }

        static makeRadiansObject(degreesObj) {
            const obj = Object.assign({}, degreesObj);
            const util = SUND_MODS.THREE.MathUtils;
            obj.x = util.degToRad(obj.x);
            obj.y = util.degToRad(obj.y);
            obj.z = util.degToRad(obj.z);
            return obj;
        }

        static modelKeyToArray(str) {
            const strKey = this.stringTrim(str);
            if (strKey.toLowerCase() === "all") {
                return Array.from(this._params.vrmMap.keys());
            } else {
                return strKey.split(",").map((el) => el.trim());
            }
        }

        static getRandomInt(min, max) {
            min = Math.ceil(min);
            max = Math.floor(max + 1);
            return Math.floor(Math.random() * (max - min) + min);
        }

        static getAngleToTarget(me, target, isRadians = true) {
            // 平面で2点間の角度を得るために以前作成した関数。謎の90は0を北とするためだったはず。
            const l = 1;
            const dist = Math.sqrt(Math.pow(me.x - target.x, 2) + Math.pow(me.z - target.z, 2));

            const goalX = (-l * me.x + (dist + l) * target.x) / dist;
            const goalY = (-l * me.z + (dist + l) * target.z) / dist;

            if (isRadians) {
                const value = 90 * (Math.PI / 180);
                return -Math.atan2(goalY - me.z, goalX - me.x) + value;
            } else {
                return -Math.atan2(goalY - me.z, goalX - me.x) * (180 / Math.PI) + 90;
            }
        }

        // ---------------
        // instance

        constructor(keyName) {
            this.initialize(keyName);
        }

        initialize(keyName) {
            this._keyName = keyName;
            this._loaded = false;
            this._needsDispose = false;
            this._disposed = false;
            this._needsTrialRender = true;
            this._vrm;
            this._mixer;
            this._lookCamera = true;
            this._animationActions = new Map();
            this._rotationDegY = 0;
            this._continueData;
            this._autoBlink = { enabled: true, nextBlink: 300, interval: 0, value: 0 };
            this._nextPosition = {
                changeValue: { x: 0, y: 0, z: 0 },
                duration: 0,
            };
            this._nextDirection = {
                changeValue: { x: 0, y: 0, z: 0 },
                duration: 0,
            };
            this._motion = {
                current: { name: "idle", loop: "enabled", speed: 1 },
                next: { name: "", loop: "enabled", speed: 1 },
                state: "",
            };
            this._dev = { touchX: null, touchY: null };
            this._disabledPluginExpression = false;
        }

        update() {
            if (this._disposed) return;
            if (!this._vrm) return;

            const vrm = this._vrm;
            this.updateMove(vrm);
            this.updateDirection(vrm);
            this.updateLookAt(vrm);
            this.updateMotion(vrm);
            this.updateBlink(vrm);
            this.updateDev(vrm);
            vrm.update(SUND_VRM.deltaTime);
            this.updateDispose(vrm);
        }

        updateMove(vrm) {
            if (this._nextPosition.duration <= 0) return;
            const n = this._nextPosition;
            n.duration--;
            const pos = vrm.scene.position;
            pos.x += n.changeValue.x;
            pos.y += n.changeValue.y;
            pos.z += n.changeValue.z;
        }

        updateDirection(vrm) {
            if (this._nextDirection.duration <= 0) return;
            const n = this._nextDirection;
            n.duration--;

            this._rotationDegY = SUND_VRM.rotationWork180(this._rotationDegY + n.changeValue.y);
            vrm.scene.setRotationFromAxisAngle(SUND_VRM._AXIS_Y, SUND_MODS.THREE.MathUtils.degToRad(this._rotationDegY));
        }

        updateLookAt(vrm) {
            if (!this._lookCamera) return;

            vrm.lookAt.lookAt(SUND_VRM._camera.position);
        }

        updateMotion(vrm) {
            if (!this.isMotionPlay() || !this._mixer) return;

            if (this._mixer._actions[0].isRunning()) {
                this._motion.state = "play";
                this._mixer.update(SUND_VRM.deltaTime);
            } else {
                this._motion.state = "pause";
                if (this._motion.next.name !== "") {
                    this.startMotion({ motionKey: this._motion.next.name, nextMotionKey: "", loop: "enabled", speed: 1 });
                }
            }
        }

        updateBlink(vrm) {
            if (this._disabledPluginExpression || !this._autoBlink.enabled) return;

            const b = this._autoBlink;
            b.interval++;
            if (b.interval >= b.nextBlink) {
                vrm.expressionManager.setValue("blink", 1);
                b.nextBlink = SUND_VRM.getRandomInt(180, 300);
                b.interval = 0;
            } else if (b.interval > 4) {
                this.resetBlink(vrm);
            }
        }

        resetBlink(vrm) {
            if (this._disabledPluginExpression) return;
            vrm.expressionManager.setValue("blink", 0);
        }

        updateDev(vrm) {
            if (!SUND_VRM.isDevMode()) return;
            if (!this.visible()) return;
            if (Input.isPressed("shift")) {
                this.updateDevVRM(vrm);
            }
        }

        updateDevVRM(vrm) {
            this.updateDevVRMPosition(vrm);
            this.updateDevVRMRotation(vrm);
            this.updateDevVRMInformation(vrm);
        }

        updateDevVRMPosition(vrm) {
            const position = vrm.scene.position;
            const moveAmount = 0.1;

            if (Input.isPressed("up")) {
                position.y += moveAmount;
            } else if (Input.isPressed("down")) {
                position.y -= moveAmount;
            } else if (Input.isPressed("left")) {
                position.x -= moveAmount;
            } else if (Input.isPressed("right")) {
                position.x += moveAmount;
            } else if (TouchInput.wheelY > 0) {
                position.z += moveAmount;
            } else if (TouchInput.wheelY < 0) {
                position.z -= moveAmount;
            }
        }

        updateDevVRMRotation(vrm) {
            const dev = SUND_VRM._dev;
            if (TouchInput.isMoved()) {
                dev.touchX = TouchInput._triggerX;
                const moveX = TouchInput._x - dev.touchX;
                const value = moveX > 0 ? 5 : -5;
                const curY = this._rotationDegY;
                const d = SUND_VRM.rotationWork180(curY + value);
                this._rotationDegY = d;
                vrm.scene.setRotationFromAxisAngle(SUND_VRM._AXIS_Y, SUND_MODS.THREE.MathUtils.degToRad(d));

                dev.touchX = TouchInput._x;
            } else {
                dev.touchX = 0;
            }
        }

        updateDevVRMInformation(vrm) {
            const scene = vrm.scene;
            if (Input.isTriggered("pageup")) {
                try {
                    SUND_VRM.copyClipboard(SUND_VRM.devNum(scene.position.x) + "," + SUND_VRM.devNum(scene.position.y) + "," + SUND_VRM.devNum(scene.position.z));
                } catch { }
                console.log(`==== [${this._keyName}] ====`);
                console.log(`position: x:${SUND_VRM.devNum(scene.position.x)}, y:${SUND_VRM.devNum(scene.position.y)}, z:${SUND_VRM.devNum(scene.position.z)}`);
            } else if (Input.isTriggered("pagedown")) {
                try {
                    SUND_VRM.copyClipboard(this._rotationDegY);
                } catch { }
                console.log(`==== [${this._keyName}] ====`);
                console.log(`rotation: y:${this._rotationDegY}`);
            }
        }

        restoreContinueData() {
            if (!this._continueData) return;
            const scene = this._vrm.scene;
            const data = this._continueData;
            SUND_VRM.assignmentXYZ(scene.position, data.position);
            const rot = data.rotation;
            scene.rotateX(rot.x);
            scene.rotateY(rot.y);
            scene.rotateZ(rot.z);
            const m = data.motion;
            const c = m.current;
            this.startMotion({ motionKey: c.name, nextMotionKey: m.next.name, loop: c.loop, speed: c.speed });
            if (data.visible) {
                this.show();
            }
            this._continueData = null;
        }

        updateDispose(vrm) {
            if (!this._needsDispose) return;
            this.hide();
            vrm.update(SUND_VRM.deltaTime);
            SUND_VRM.dispose(this._keyName);
            this._disposed = true;
        }

        setRandomMouth() {
            if (this._disabledPluginExpression) return;

            const vrm = this._vrm;
            const mouthArr = vrm.expressionManager.mouthExpressionNames;
            for (key in vrm.expressionManager._expressionMap) {
                if (mouthArr.includes(key)) {
                    vrm.expressionManager._expressionMap[key].weight = 0;
                }
            }
            let i = SUND_VRM.getRandomInt(0, 4);
            let w = SUND_VRM.getRandomInt(4, 7) / 10;
            vrm.expressionManager._expressionMap[mouthArr[i]].weight = w;
        }

        resetMouth() {
            if (this._disabledPluginExpression) return;

            const vrm = this._vrm;
            const mouthArr = vrm.expressionManager.mouthExpressionNames;
            for (key in vrm.expressionManager._expressionMap) {
                if (mouthArr.includes(key)) {
                    vrm.expressionManager._expressionMap[key].weight = 0;
                }
            }
        }

        setLookAt(name, value) {
            const lookAt = this._vrm.lookAt;
            const pos = this._vrm.scene.position.clone();
            pos.y += 1;
            pos.z += 1;
            const change = 5 * value;
            switch (name) {
                case "lookReset":
                    lookAt.reset();
                    return;
                case "lookUp":
                    pos.y += change;
                    break;
                case "lookDown":
                    pos.y -= change;
                    break;
                case "lookLeft":
                    pos.x += change;
                    break;
                case "lookRight":
                    pos.x -= change;
                    break;
            }
            lookAt.lookAt(pos);
            return;
        }

        setExpression(name, value) {
            if (this._disabledPluginExpression) return;

            const vrm = this._vrm;
            if (name === "lookReset" || vrm.expressionManager.lookAtExpressionNames.includes(name)) {
                this.setLookAt(name, value);
                return;
            }

            const mouthArr = this._vrm.expressionManager.mouthExpressionNames;
            const blinkArr = this._vrm.expressionManager.blinkExpressionNames;
            if (mouthArr.includes(name)) {
                for (key in this._vrm.expressionManager._expressionMap) {
                    if (mouthArr.includes(key)) {
                        this._vrm.expressionManager._expressionMap[key].weight = 0;
                    }
                }
            } else if (blinkArr.includes(name)) {
                for (key in this._vrm.expressionManager._expressionMap) {
                    if (blinkArr.includes(key)) {
                        this._vrm.expressionManager._expressionMap[key].weight = 0;
                    }
                }
            } else {
                for (key in this._vrm.expressionManager._expressionMap) {
                    this._vrm.expressionManager._expressionMap[key].weight = 0;
                }
            }
            this._vrm.expressionManager.setValue(name, value);
        }

        resetPluginExpression(){
            const vrm = this._vrm;
            vrm.lookAt.reset();

            const map = vrm.expressionManager._expressionMap;
            for (key in map) {
                map[key].weight = 0;
            }
        }

        visible() {
            return this._vrm.scene.visible;
        }

        isMotionPlay() {
            return this.visible() && this._motion.state === "play";
        }

        show() {
            if (!this._vrm) return;
            this._vrm.scene.visible = true;
        }

        hide() {
            if (!this._vrm) return;
            this._vrm.scene.visible = false;
        }

        motionSpeedChange(props) {
            const vrm = this._vrm;
            if (!vrm) return;
            const motionKey = this._motion.current.name;
            const action = this._animationActions.get(motionKey);
            if (!action) return;

            const motion = this._motion;
            const speed = props.speed;
            motion.current.speed = speed;
            action.timeScale = speed;
            if (props.speed === 0) {
                motion.state = "pause";
                this._mixer.update(SUND_VRM.deltaTime);
            } else {
                motion.state = "play";
            }
        }

        startMotion(props) {
            const vrm = this._vrm;
            const motionKey = "" + props.motionKey;
            const vrma = SUND_VRM._animations.get(motionKey);
            if (!vrm || !vrma) return;

            let action = this._animationActions.get(motionKey);
            if (!action) {
                const clip = SUND_MODS.THREE_VRM_Animation.createVRMAnimationClip(vrma, vrm);
                action = this._mixer.clipAction(clip);
                this._animationActions.set(motionKey, action);
            }
            const motion = this._motion;
            motion.current.name = motionKey;
            const loop = props.loop;
            motion.current.loop = loop;

            SUND_VRM.setActionLoop(loop, action);
            if (loop === "disabled") {
                motion.next.name = "" + props.nextMotionKey;
                action.clampWhenFinished = true;
            } else {
                motion.next.name = "";
                action.clampWhenFinished = false;
            }

            action.reset();

            const speed = props.speed;
            motion.current.speed = speed;
            action.timeScale = speed;
            action.play();

            this._animationActions.forEach((value, key) => {
                if (key !== motionKey) {
                    value.stop();
                }
            });

            if (speed === 0) {
                motion.state = "pause";
                this._mixer.update(SUND_VRM.deltaTime);
            } else {
                motion.state = "play";
            }
        }
    }
    window.SUND_VRM = SUND_VRM;

    // -----------------
    // パラメータ整形してSUND_VRMへ

    params.models.forEach((el) => {
        SUND_VRM._params.vrmMap.set("" + el.keyName, el.fileName);
        if (el.preLoad) {
            SUND_VRM._params.loadModels.add(el.keyName);
        }
    });

    params.motions.forEach((el) => SUND_VRM._params.vrmaMap.set("" + el.keyName, el.fileName));

    params.positions.forEach((el) => {
        const posArr = el.position.split(",").map(Number);
        const pos = SUND_VRM.makeObjectXYZ(posArr[0], posArr[1], posArr[2]);
        SUND_VRM._params.positionMap.set("" + el.keyName, pos);
    });

    params.directions.forEach((el) => {
        SUND_VRM._params.directionMap.set("" + el.keyName, Number(el.rotationY));
    });

    params.cameras.forEach((el) => {
        const posArr = el.position.split(",").map(Number);
        const pos = SUND_VRM.makeObjectXYZ(posArr[0], posArr[1], posArr[2]);
        const rotArr = el.rotation.split(",").map(Number);
        const rot = SUND_VRM.makeObjectXYZ(rotArr[0], rotArr[1], rotArr[2]);
        SUND_VRM._params.cameraMap.set("" + el.keyName, { position: pos, rotation: rot });
    });

    // -----------------
    // プラグインコマンド

    PluginManagerEx.registerCommand(script, "showVRM", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj.show();
            }
        });
    });

    PluginManagerEx.registerCommand(script, "hideVRM", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj.hide();
            }
        });
    });

    PluginManagerEx.registerCommand(script, "setPosition", function (args) {
        if (isNaN(args.frame)) return;
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const duration = args.frame <= 0 ? 1 : args.frame;

        let pos;
        let addMode = false;
        const argPos = SUND_VRM.stringTrim(args.position);
        if (argPos.split(",").length === 3) {
            const arr = argPos.split(",").map(Number);
            pos = { x: arr[0], y: arr[1], z: arr[2] };
            addMode = true;
        } else {
            pos = SUND_VRM._params.positionMap.get(argPos);
        }
        if (!pos) return;

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                const curPos = obj._vrm.scene.position;
                const addPos = SUND_VRM.makeObjectXYZ();
                if (addMode) {
                    addPos.x = pos.x + curPos.x;
                    addPos.y = pos.y + curPos.y;
                    addPos.z = pos.z + curPos.z;
                }

                const next = obj._nextPosition;
                next.changeValue.x = (pos.x + addPos.x - curPos.x) / duration;
                next.changeValue.y = (pos.y + addPos.y - curPos.y) / duration;
                next.changeValue.z = (pos.z + addPos.z - curPos.z) / duration;
                next.duration = duration;
            }
        });
    });

    PluginManagerEx.registerCommand(script, "setDirection", function (args) {
        if (isNaN(args.frame)) return;
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const paramRot = SUND_VRM.stringTrim(args.modelRotation);
        const duration = args.frame <= 0 ? 1 : args.frame;
        let rot;
        let addMode = false;
        let cameraMode = false;

        if (paramRot === "camera") {
            cameraMode = true;
        } else if (!isNaN(paramRot)) {
            rot = { x: 0, y: Number(paramRot), z: 0 };
            addMode = true;
        } else {
            const value = SUND_VRM._params.directionMap.get(paramRot);
            if (value == null) return;
            rot = { x: 0, y: value, z: 0 };
        }
        // rotはdegree

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                const next = obj._nextDirection;
                const curRot = obj._vrm.scene.rotation;
                if (addMode) {
                    next.changeValue.y = rot.y / duration;
                } else {
                    const curY = Math.round(SUND_VRM.getRotationDegY(curRot));

                    let goalY = 0;
                    if (cameraMode) {
                        goalY = SUND_VRM.rotationWork180(Math.round(SUND_VRM.getAngleToTarget(obj._vrm.scene.position, SUND_VRM._camera.position, false)));
                    } else {
                        goalY = Math.round(rot.y);
                    }

                    if (isNaN(curY) || isNaN(goalY)) return;
                    if (curY === goalY) return;

                    obj._rotationDegY = curY;

                    // より近い方向で回転
                    let right = 0;
                    let left = 0;
                    const cur = curY < 0 ? 360 + curY : curY;
                    const goal = goalY < 0 ? 360 + goalY : goalY;
                    if (cur <= goal) {
                        right = goal - cur;
                        left = cur + 360 - goal;
                    } else {
                        right = 360 - cur + goal;
                        left = cur - goal;
                    }
                    if (right <= left) {
                        next.changeValue.y = right / duration;
                    } else {
                        next.changeValue.y = -(left / duration);
                    }
                }
                next.duration = duration;
            }
        });
    });

    PluginManagerEx.registerCommand(script, "setCamera", function (args) {
        const camData = SUND_VRM._params.cameraMap.get(SUND_VRM.stringTrim(args.cameraKey));
        if (!camData) return;
        const pos = camData.position;
        const rot = SUND_VRM.makeRadiansObject(camData.rotation);

        SUND_VRM._camera.position.set(pos.x, pos.y, pos.z);
        SUND_VRM._camera.rotation.set(rot.x, rot.y, rot.z);
    });

    PluginManagerEx.registerCommand(script, "cameraLookAt", function (args) {
        if (isNaN(args.adjustY)) return;
        const obj = $sundVRMs.get(SUND_VRM.stringTrim(args.modelKey));
        if (obj) {
            const p = obj._vrm.scene.position.clone();
            p.y += args.adjustY;
            SUND_VRM._camera.lookAt(p);
        }
    });

    PluginManagerEx.registerCommand(script, "disabledExpression", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);

        if (args.disabled){
            keys.forEach((key) => {
                const obj = $sundVRMs.get(key);
                const index = SUND_VRM._autoMouth.targets.indexOf(key);
                SUND_VRM._autoMouth.targets.splice(index,1);
                if (obj){
                    obj.resetPluginExpression();
                    obj._disabledPluginExpression = true;
                }
            });
        } else {
            keys.forEach((key) => {
                const obj = $sundVRMs.get(key);
                if (obj){
                    obj._disabledPluginExpression = false;
                }
            });
        }

    });

    PluginManagerEx.registerCommand(script, "startMotion", function (args) {
        if (isNaN(args.speed)) return;
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        args.speed /= 100;

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj.startMotion(args);
            }
        });
    });

    PluginManagerEx.registerCommand(script, "motionSpeed", function (args) {
        if (isNaN(args.speed)) return;
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        args.speed /= 100;

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj.motionSpeedChange(args);
            }
        });
    });

    PluginManagerEx.registerCommand(script, "dispose", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj._needsDispose = true;
            }
        });
    });

    PluginManagerEx.registerCommand(script, "loadSkipList", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const mode = args.mode;
        const list = $gameParty._sundVRM.loadSkipModels;
        switch (mode) {
            case "add":
                keys.forEach((key) => {
                    if (!list.includes(key)) {
                        list.push(key);
                    }
                });
                break;
            case "remove":
                keys.forEach((key) => {
                    const index = list.indexOf(key);
                    if (index >= 0) {
                        list.splice(index, 1);
                    }
                });
                break;
        }
    });

    PluginManagerEx.registerCommand(script, "autoMouth", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const m = SUND_VRM._autoMouth;
        m.targets.splice(0);
        keys.forEach((key) => m.targets.push(key));
    });

    PluginManagerEx.registerCommand(script, "autoBlink", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const enabled = args.enabled;

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj._autoBlink.enabled = enabled;
                if (!enabled) {
                    obj.resetBlink(obj._vrm);
                }
            }
        });
    });

    PluginManagerEx.registerCommand(script, "lookCamera", function (args) {
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const enabled = args.enabled;

        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj._lookCamera = enabled;
            }
        });
    });

    PluginManagerEx.registerCommand(script, "setExpression", function (args) {
        if (isNaN(args.value)) return;
        const keys = SUND_VRM.modelKeyToArray(args.modelKey);
        const value = args.value / 100;
        keys.forEach((key) => {
            const obj = $sundVRMs.get(key);
            if (obj) {
                obj.setExpression(args.expression, value);
            }
        });
    });

    // ----------------
    // Bitmap

    // nw.js更新時の警告に対応するもの
    Bitmap.prototype._createCanvas = function (width, height) {
        this._canvas = document.createElement("canvas");
        this._context = this._canvas.getContext("2d", { willReadFrequently: true });
        this._canvas.width = width;
        this._canvas.height = height;
        this._createBaseTexture(this._canvas);
    };

    // ------------
    // Graphics

    const _Graphics__createAllElements = Graphics._createAllElements;
    Graphics._createAllElements = function () {
        this._createThreeArea();
        _Graphics__createAllElements.apply(this, arguments);
    };

    Graphics._createThreeArea = function () {
        const sc = document.createElement("script");
        sc.type = "module";
        const url = (() => {
            const start = script.src.indexOf("/js/");
            const end = script.src.indexOf("SunD_VRM.js");
            return script.src.substring(start + 1, end) + "SunD_VRM_MOD.js";
        })();
        sc.src = url;
        sc.async = false;
        sc.defer = true;
        sc._url = url;
        document.body.appendChild(sc);
        this._threeArea = sc;
    };

    const _Graphics__updateAllElements = Graphics._updateAllElements;
    Graphics._updateAllElements = function () {
        _Graphics__updateAllElements.apply(this, arguments);
        this._updateThreeElements();
    };

    Graphics._updateThreeElements = function () {
        if (!SUND_VRM.moduleLoaded) return;

        const width = this._width;
        const height = this._height;
        const renderer = SUND_VRM._renderer;
        renderer.setSize(width, height);
        SUND_VRM._camera.aspect = width / height;
        SUND_VRM._camera.updateProjectionMatrix();

        const canvas = renderer.domElement;
        canvas.width = width;
        canvas.height = height;
        this._centerElement(canvas);
    };

    // ------------
    // Scene_Map

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.apply(this, arguments);
        SUND_VRM.updateMap();
    };

    const _Scene_Map_isReady = Scene_Map.prototype.isReady;
    Scene_Map.prototype.isReady = function () {
        return _Scene_Map_isReady.apply(this, arguments) && SUND_VRM.isReady();
    };

    const _Scene_Map_create = Scene_Map.prototype.create;
    Scene_Map.prototype.create = function () {
        SUND_VRM._readyState = "start";
        _Scene_Map_create.apply(this, arguments);
    };

    // ------------
    // Scene_Load

    const _Scene_Load_executeLoad = Scene_Load.prototype.executeLoad;
    Scene_Load.prototype.executeLoad = function (savefileId) {
        // 直接ロード対策
        SUND_VRM.allHide();
        SUND_VRM._afterContinue = params.useSave;
        _Scene_Load_executeLoad.apply(this, arguments);
    };

    // ------------
    // Scene_Save

    const _Scene_Save_executeSave = Scene_Save.prototype.executeSave;
    Scene_Save.prototype.executeSave = function (savefileId) {
        SUND_VRM.setSaveData();
        _Scene_Save_executeSave.apply(this, arguments);
    };

    // ------------
    // Spriteset_Map

    const _Spriteset_Map_createPictures = Spriteset_Map.prototype.createPictures;
    Spriteset_Map.prototype.createPictures = function () {
        _Spriteset_Map_createPictures.apply(this, arguments);
        this.createThreeArea();
    };

    Spriteset_Map.prototype.createThreeArea = function () {
        const sp = new Sprite();
        const w = Graphics.width;
        const h = Graphics.height;
        sp.bitmap = new Bitmap(w, h);
        this.addChild(sp);
        SUND_VRM._threeSprite = sp;
    };

    // ------------
    // Window_Message

    const _Window_Message_startMessage = Window_Message.prototype.startMessage;
    Window_Message.prototype.startMessage = function () {
        _Window_Message_startMessage.apply(this, arguments);
        SUND_VRM._autoMouth.duration = $gameMessage.allText().length * 4;
    };

    const _Window_Message_update = Window_Message.prototype.update;
    Window_Message.prototype.update = function () {
        SUND_VRM.updateAutoMouth();
        _Window_Message_update.apply(this, arguments);
    };

    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function () {
        _Window_Message_terminateMessage.apply(this, arguments);
        SUND_VRM.stopAutoMouth();
    };

    // ------------
    // Game_Party

    const _Game_Party_initialize = Game_Party.prototype.initialize;
    Game_Party.prototype.initialize = function () {
        _Game_Party_initialize.apply(this, arguments);
        this._sundVRM = {
            loadSkipModels: [],
            modelSaveData: [],
            cameraSaveData: { position: SUND_VRM.makeObjectXYZ(0, 0, 3), rotation: SUND_VRM.makeObjectXYZ() },
        };
    };

    // ------------
    // Game_Player

    const _Game_Player_canMove = Game_Player.prototype.canMove;
    Game_Player.prototype.canMove = function () {
        if (SUND_VRM.dontMovePlayer()) {
            return false;
        }
        return _Game_Player_canMove.apply(this, arguments);
    };

    // ------------
    // Scene_Title

    const _Scene_Title_initialize = Scene_Title.prototype.initialize;
    Scene_Title.prototype.initialize = function () {
        SUND_VRM.allHide();
        _Scene_Title_initialize.apply(this, arguments);
    };
})();
