//=============================================================================
// SimpleVoiceChannelGroups.js
// ----------------------------------------------------------------------------
//  Add per-channel-group volume controls to SimpleVoice.js
// ----------------------------------------------------------------------------
//  Requires : PluginCommonBase.js, SimpleVoice.js (by トリアコンタン)
//  Author   : ChatGPT (OpenAI)
//  License  : MIT – free to use, modify, redistribute, commercial/non-commercial
//  Version  : 1.0.0 2025/05/14  First release
//=============================================================================

/*:
 * @base PluginCommonBase
 * @orderAfter SimpleVoice
 * @target MZ
 * @plugindesc <SV_ChannelGroups> チャンネルグループ単位で音量を変更できる機能を追加
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/SimpleVoice.js
 *
 * @param groups
 * @text チャンネルグループ設定
 * @type struct<Group>[]
 * @desc オプションに表示するグループを定義します。
 * 例）ボイス:0 SE:1,2,3
 * @default []
 *
 * @help
 * ■概要
 * SimpleVoice.js で再生される各チャンネルを任意の「グループ」にまとめ、
 * オプション画面からグループごとに音量を変更できるようにします。
 *
 * 例）
 *   ・グループ「ボイス」…チャンネル 0
 *   ・グループ「効果音」…チャンネル 1,2,3
 *
 * ■使い方
 * 1. 本プラグインを SimpleVoice.js の下に配置して有効化してください。
 * 2. プラグインパラメータ「チャンネルグループ設定」に任意の数だけ
 *    グループを追加します。
 * 3. ゲーム起動後、オプション画面に各グループの音量スライダーが
 *    追加されます。
 *
 * ■仕様
 * ・マスタ音量（SimpleVoice の「ボイス 音量」）×グループ音量 で
 *   実際の再生音量を決定します。
 * ・グループ未登録チャンネルはマスタ音量だけが適用されます。
 *
 * ■利用規約
 * MIT ライセンス。商用・非商用・年齢制限作品を問わずご自由にどうぞ。
 */

/*~struct~Group:
 *
 * @param name
 * @text グループ名
 * @type string
 * @default Group
 *
 * @param channels
 * @text 対象チャンネル番号
 * @type number[]
 * @default []
 *
 * @param defaultVolume
 * @text 初期音量
 * @type number
 * @min 0
 * @max 100
 * @default 100
 */

(() => {
    'use strict';

    const script = document.currentScript;
    const param  = PluginManagerEx.createParameter(script);

    /** @type {{name:string, channels:number[], defaultVolume:number, key:string}[]} */
    const Groups = (param.groups || []).map((g, i) => ({
        name:          String(g.name || `Group${i+1}`),
        channels:      (g.channels ?? []).map(Number).filter(c => !Number.isNaN(c)),
        defaultVolume: Number(g.defaultVolume || 100),
        key:           `voiceGroup${i}Volume`
    }));

    //-------------------------------------------------------------------------
    // AudioManager – グループ音量管理
    //-------------------------------------------------------------------------
    AudioManager._voiceGroupVolumes = Groups.map(g => g.defaultVolume);
    AudioManager._voiceGroupMap     = Groups.map(g => g.channels);

    /**
     * 指定チャンネルが属するグループ音量を返す（無ければ100）
     * @param {number} channel
     * @return {number}
     */
    AudioManager.groupVolumeForChannel = function (channel) {
        for (let i = 0; i < this._voiceGroupMap.length; i++) {
            if (this._voiceGroupMap[i].includes(channel)) {
                return this._voiceGroupVolumes[i];
            }
        }
        return 100;
    };

    // SimpleVoice が定義した updateVoiceParameters を上書き拡張
    AudioManager.updateVoiceParameters = function (buffer, voice) {
        const master   = this._voiceVolume ?? 100;
        const groupVol = this.groupVolumeForChannel(voice.channel ?? 0);
        const volume   = Math.round(master * groupVol / 100);
        this.updateBufferParameters(buffer, volume, voice);
    };

    //-------------------------------------------------------------------------
    // ConfigManager – グループ音量を保存 / 読込
    //-------------------------------------------------------------------------
    Groups.forEach((g, i) => {
        Object.defineProperty(ConfigManager, g.key, {
            get: ()  => AudioManager._voiceGroupVolumes[i],
            set: v   => { AudioManager._voiceGroupVolumes[i] = v; }
        });
    });

    // 既存メソッドをラップ
    const _makeData = ConfigManager.makeData;
    ConfigManager.makeData = function () {
        const config = _makeData.apply(this, arguments);
        Groups.forEach(g => { config[g.key] = this[g.key]; });
        return config;
    };

    const _applyData = ConfigManager.applyData;
    ConfigManager.applyData = function (config) {
        _applyData.apply(this, arguments);
        Groups.forEach(g => {
            if (config.hasOwnProperty(g.key)) {
                this[g.key] = this.readVolume(config, g.key);
            }
        });
    };

    const _load = ConfigManager.load;
    ConfigManager.load = function () {
        // 初期値セット
        Groups.forEach((g, i) => { AudioManager._voiceGroupVolumes[i] = g.defaultVolume; });
        _load.apply(this, arguments);
    };

    //-------------------------------------------------------------------------
    // Window_Options – スライダー追加
    //-------------------------------------------------------------------------
    const _addVolumeOptions = Window_Options.prototype.addVolumeOptions;
    Window_Options.prototype.addVolumeOptions = function () {
        _addVolumeOptions.apply(this, arguments); // 既存（BGM 等 + ボイス）を追加
        Groups.forEach(g => this.addCommand(g.name, g.key));
    };

    // コマンド数を加算
    const _maxCommands = Scene_Options.prototype.maxCommands;
    Scene_Options.prototype.maxCommands = function () {
        return _maxCommands.apply(this, arguments) + Groups.length;
    };
})();
