
/*:

@target MZ
@plugindesc 立ち絵などの差分のある絵をまとめて表示したりするプラグインです
@author 紫鳥＠メロウ紙芝居
@url https://meroukamisibai.com/

@param messageControlCharacter
@text 制御文字
@desc 文章の表示の名前の制御文字の文字を変更できます。デフォルトは「CP」です。
@type string
@default CP

@help MK_CharaPicture.js
ver1.2

立ち絵などの差分のある絵をまとめて表示したりするプラグインです。
立ち絵の設定はjsファイルにある
データベースで設定します。

立ち絵の画像はimgフォルダの中にある「mk_charaPicture」を読み込みます。
imgフォルダの中に「mk_charaPicture」フォルダがない場合は、
「mk_charaPicture」フォルダを作りましょう。

プラグインコマンドでキャラを用意した後は、
直ぐに「ピクチャの表示」をしてください。
画像は特に何も指定しません。

戦闘の立ち絵顔グラを更新したい時は、スクリプトで
「$gameTemp.requestBattleRefresh()」を
しましょう。


文章の表示の名前の制御文字:

\CP[ピクチャ番号:差分名] :
ピクチャ番号がキャラの場合、
指定した差分名に変更して、立ち絵顔グラも表示します。
差分名は「,」で複数指定できます。

例　\CP[1:幸せ,体_赤面]

また、差分名は指定しなくても問題ありません。

例　\CP[1]


スクリプト:

$gameScreen.MK_CharaPicture_DifferenceId(pictureId, partId) :
指定したピクチャIDの指定したパーツIDの差分IDを取得


自動操作条件参考スクリプト:

this.DifferenceId(partId) == 2 :
このキャラのパーツIDの差分IDが2の時


自動操作内容参考スクリプト:

this.Difference(partId, differenceId) :
このキャラの差分を変更します。

this.FrameSettings(x, y, width, height) :
このキャラの表示範囲設定を変更します。

this.FaceRange(x, y, width, height) :
このキャラの立ち絵で顔グラの表示設定をします。

this.AutoOperationActive(active) :
このキャラの自動操作を有効にするか変更します。

this.PartVisible(partId, visible) :
このキャラの部位の表示切替を変更します。

this.PartPosition(partId, x, y) :
このキャラの部位の位置を変更します。

this.PartOpacity(partId, opacity) :
このキャラの部位の不透明度を変更します。

this.PartTone(partId, tone) :
このキャラの部位の色調を変更します。

this.PartHue(partId, hue) :
このキャラの部位の色相を変更します。


@command charaSet
@text キャラの用意
@desc キャラを用意した後に、「ピクチャの表示」で画面上にキャラを表示します。

@arg charaId
@text キャラID
@desc 表示するキャラのIDです。
@type number
@min 1
@default 1

@command differenceChange
@text キャラの表示する差分を変更
@desc 指定したピクチャ番号のキャラの差分を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg difference
@text 差分
@desc 指定する差分の名前です。差分名は「,」で複数指定できます。例　幸せ,赤面
@type string

@command frameSettings
@text キャラの表示範囲設定を変更
@desc 指定したピクチャ番号のキャラの表示範囲の設定を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg x
@text x
@desc 表示する立ち絵の横の開始位置です。
@type number
@default 0

@arg y
@text y
@desc 表示する立ち絵の縦の開始位置です。
@type number
@default 0

@arg width
@text 幅
@desc 表示する立ち絵の幅です。
@type number
@default 0

@arg height
@text 高さ
@desc 表示する立ち絵の高さです。
@type number
@default 0

@command faceRange
@text キャラの立ち絵顔グラ設定を変更
@desc 指定したピクチャ番号のキャラの立ち絵顔グラの設定を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg x
@text x
@desc 表示する立ち絵の横の開始位置です。
@type number
@default 0

@arg y
@text y
@desc 表示する立ち絵の縦の開始位置です。
@type number
@default 0

@arg width
@text 幅
@desc 表示する立ち絵の幅です。
@type number
@default 144

@arg height
@text 高さ
@desc 表示する立ち絵の高さです。
@type number
@default 144

@command actorLinking
@text キャラとアクターの紐づけを変更
@desc 指定したピクチャ番号のキャラとアクターを紐づけます。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。０にすると、指定したアクターの紐づけが無くなります。
@type number
@max 100
@default 1

@arg actorId
@text アクター
@desc キャラと紐づけるアクターです。
@type actor
@default 1

@command autoOperationActive
@text キャラの自動操作の有効を変更
@desc 指定したピクチャ番号のキャラの自動操作の有効を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg active
@text 自動操作の有効
@desc 自動操作を有効にしますか？
@on 有効にする
@off 有効にしない
@type boolean
@default true

@command partVisible
@text キャラの部位の表示を変更
@desc 指定したピクチャ番号のキャラの部位の表示を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg visible
@text 部位の表示
@desc 指定した部位を表示しますか？
@on 表示する
@off 表示しない
@type boolean
@default true

@command partPosition
@text キャラの部位の位置を変更
@desc 指定したピクチャ番号のキャラの部位の位置を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg x
@text x
@desc 表示するx座標です。
@type number
@min -9999
@default 0

@arg y
@text y
@desc 表示するy座標です。
@type number
@min -9999
@default 0

@command partOpacity
@text キャラの部位の不透明度を変更
@desc 指定したピクチャ番号のキャラの部位の不透明度を変更します。

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg opacity
@text 不透明度
@desc 立ち絵の不透明度です。
@type number
@max 255
@min 0
@default 255

@command partTone
@text キャラの部位の色調を変更
@desc 指定したピクチャ番号のキャラの部位の色調を変更します。※ウィンドウ非対応

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg tintR
@text 色調　赤
@desc 色調のR成分です。
@type number
@max 255
@min -255
@default 0

@arg tintG
@text 色調　緑
@desc 色調のG成分です。
@type number
@max 255
@min -255
@default 0

@arg tintB
@text 色調　青
@desc 色調のB成分です。
@type number
@max 255
@min -255
@default 0

@arg tintGray
@text 色調　グレー
@desc グレースケールの強さです。
@type number
@max 255
@min 0
@default 0

@command partHue
@text キャラの部位の色相を変更
@desc 指定したピクチャ番号のキャラの部位の色相を変更します。※ウィンドウ非対応

@arg pictureId
@text ピクチャ番号
@desc 指定するピクチャ番号です。
@type number
@max 100
@min 1
@default 1

@arg part
@text 部位
@desc 指定する部位の名前です。
@type string

@arg hue
@text 色相
@desc 立ち絵の色相です。
@type number
@max 360
@min -360
@default 0

*/

/*~struct~database:

@param name
@text キャラ名
@desc キャラの名前です。
@type string

@param parts
@text 部位
@desc 部位の設定です。先頭から先に画像が表示されます。
@type struct<part>[]

@param autoOperations
@text 自動操作
@desc 自動操作の設定です。
@type struct<autoOperation>[]

*/

/*~struct~part:

@param name
@text 部位名
@desc 部位の名前です。
@type string

@param differences
@text 差分
@desc 差分の設定です。
@type struct<difference>[]

*/

/*~struct~difference:

@param name
@text 差分名
@desc 差分の名前です。
@type string

@param bitmaps
@text 表示画像
@desc 表示する画像の設定です。表示する画像は後ろから判定されます。
@type struct<bitmap>[]

*/

/*~struct~bitmap:

@param name
@text 画像
@desc 表示する画像です。
@type file
@dir img/mk_charaPicture/

@param condition
@text 条件
@desc 画像が表示される条件です。スクリプトで入力してboolean型を返してください。
@type multiline_string
@default true;

*/

/*~struct~autoOperation:

@param condition
@text 条件
@desc 自動操作が行われる条件です。スクリプトで入力してboolean型を返してください。
@type multiline_string
@default true;

@param action
@text 自動操作内容
@desc 自動操作の内容です。スクリプトで入力してください。
@type multiline_string

*/

var $data_mk_charaPicture = [null];

/////////////////立ち絵キャラデータ/////////////////

function MK_CharaPicture_Chara(charaId) {
    this.charaId = charaId;
    this.parts = [];
    this.autoOperationActive = false;
    this.frame = {
        display : false,
        x : 0,
        y : 0,
        width : 0,
        height : 0
    };
    this.face = {
        x : 0,
        y : 0,
        width : 0,
        height : 0
    };
    this.pictureName = "charaId_" + charaId;
    
    this.Initialize();
}

MK_CharaPicture_Chara.prototype.Initialize = function() {
    this.FaceRangeDefault();
    this.ActorLinkingDefault();
};

MK_CharaPicture_Chara.prototype.PartAdd = function(partId) {
    if(this.parts.findIndex(part => part.id == partId) == -1){
        this.parts.push({
            id : partId,
            differenceId : 1,
            visible : true,
            x : 0,
            y : 0,
            opacity : 255,
            tone : [0, 0, 0, 0],
            hue : 0
        });
    }
};

MK_CharaPicture_Chara.prototype.CheckPartId = function(partId) {
    const id = this.parts.findIndex(part => part.id == partId);
    if(id != -1){
        return id;
    }
    this.PartAdd(partId);
    return this.parts.length -1;
};

MK_CharaPicture_Chara.prototype.Difference = function(partId, differenceId) {
    this.parts[this.CheckPartId(partId)].differenceId = differenceId;
};

MK_CharaPicture_Chara.prototype.FrameSettings = function(x, y, width, height) {
    this.frame.display = true;
    this.frame.x = x;
    this.frame.y = y;
    this.frame.width = width;
    this.frame.height = height;
};

MK_CharaPicture_Chara.prototype.FaceRange = function(x, y, width, height) {
    this.face.x = x;
    this.face.y = y;
    this.face.width = width;
    this.face.height = height;
};

MK_CharaPicture_Chara.prototype.FaceRangeDefault = function() {
    const faceRange = $data_mk_charaPicture[this.charaId].faceRange;
    this.FaceRange(faceRange.x, faceRange.y, faceRange.width, faceRange.height);
};

MK_CharaPicture_Chara.prototype.ActorLinkingDefault = function() {
    const actorLinking = $data_mk_charaPicture[this.charaId].actorLinking;
    $gameSystem.MK_CharaPicture_ActorLinkingSet(actorLinking.pictureId, actorLinking.actorId);
};

MK_CharaPicture_Chara.prototype.AutoOperationActive = function(active) {
    this.autoOperationActive = active;
};

MK_CharaPicture_Chara.prototype.ProcessingAutoOperation = function() {
    if(this.autoOperationActive){
        const autoOperations = $data_mk_charaPicture[this.charaId].autoOperations;
        for(let r = 1;r < autoOperations.length;r++){
            if(eval(autoOperations[r].condition)){
                eval(autoOperations[r].action);
            }
        }
    }
};

MK_CharaPicture_Chara.prototype.PartVisible = function(partId, visible) {
    this.parts[this.CheckPartId(partId)].visible = visible;
};

MK_CharaPicture_Chara.prototype.PartPosition = function(partId, x, y) {
    const part = this.parts[this.CheckPartId(partId)];
    part.x = x;
    part.y = y;
};

MK_CharaPicture_Chara.prototype.PartOpacity = function(partId, opacity) {
    this.parts[this.CheckPartId(partId)].opacity = opacity;
};

MK_CharaPicture_Chara.prototype.PartTone = function(partId, tone) {
    this.parts[this.CheckPartId(partId)].tone = tone;
};

MK_CharaPicture_Chara.prototype.PartHue = function(partId, hue) {
    this.parts[this.CheckPartId(partId)].hue = hue;
};

//指定した部位の差分ID取得
MK_CharaPicture_Chara.prototype.DifferenceId = function(partId) {
    return this.parts[this.CheckPartId(partId)].differenceId;
};

MK_CharaPicture_Chara.prototype.PictureName = function() {
    return this.pictureName;
};

//指定した部位の差分の表示する画像返す
MK_CharaPicture_Chara.prototype.DifferenceBitmap = function(partId, differenceId) {//部位ID, 差分ID
    if(differenceId != -1){
        const bitmaps = $data_mk_charaPicture[this.charaId].parts[partId].differences[differenceId].bitmaps;
        for(let r = bitmaps.length - 1; r > 0; r--){
            if(eval(bitmaps[r].condition)){
                return bitmaps[r].bitmap;
            }
        }
    }
    return "";
};


/////////////////キャラ部位スプライト/////////////

function MK_CharaPicture_Sprite_Part(pictureId, partId) {
    this.mk_charaPicture_pictureId = pictureId;
    this.mk_charaPicture_partId = partId;
    this.mk_charaPicture_pictureName = "";
    this.mk_charaPicture_yuanFrame = {x : 0, y : 0, width : 0, height : 0};
    
    this.initialize(...arguments);
}

MK_CharaPicture_Sprite_Part.prototype = Object.create(Sprite_Clickable.prototype);
MK_CharaPicture_Sprite_Part.prototype.constructor = MK_CharaPicture_Sprite_Part;

const _Sprite_onBitmapLoad = Sprite.prototype._onBitmapLoad;
MK_CharaPicture_Sprite_Part.prototype._onBitmapLoad = function(bitmapLoaded) {
    _Sprite_onBitmapLoad.apply(this, arguments);
    this.mk_charaPicture_yuanFrame.width = this._frame.width;
    this.mk_charaPicture_yuanFrame.height = this._frame.height;
};

MK_CharaPicture_Sprite_Part.prototype.MK_CharaPicture_Update = function() {
    const chara = $gameScreen.picture(this.mk_charaPicture_pictureId).mk_charaPicture;
    const part = chara.parts[chara.CheckPartId(this.mk_charaPicture_partId)];
    const bitmapName = chara.DifferenceBitmap(part.id, part.differenceId);
    if(bitmapName != this.mk_charaPicture_pictureName){
        this.mk_charaPicture_pictureName = bitmapName;
        this.bitmap = ImageManager.MK_CharaPicture_LoadCharaPicture(bitmapName);
    }
    this.visible = part.visible && this.mk_charaPicture_pictureName;
    if(this.visible){
        this.move(part.x, part.y);
        this.opacity = part.opacity;
        this.setColorTone(part.tone);
        this.setHue(part.hue);
    }
};

MK_CharaPicture_Sprite_Part.prototype.MK_CharaPicture_UpdateOrigin = function(anchor_x, anchor_y) {
    this.anchor.x = anchor_x;
    this.anchor.y = anchor_y;
};

MK_CharaPicture_Sprite_Part.prototype.MK_CharaPicture_UpdateBlendMode = function(blendMode) {
    this.blendMode = blendMode;
};

MK_CharaPicture_Sprite_Part.prototype.MK_CharaPicture_UpdateFrame = function() {
    const frame = this._frame;
    const chara = $gameScreen.picture(this.mk_charaPicture_pictureId).mk_charaPicture;
    const charaFrame = chara.frame;
    if(charaFrame.display){
        if(charaFrame.x !== frame.x || charaFrame.y !== frame.y ||
        charaFrame.width !== frame.width || charaFrame.height !== frame.height){
            this.setFrame(charaFrame.x, charaFrame.y, charaFrame.width, charaFrame.height);
        }
    }else{
        const yuanFrame = this.mk_charaPicture_yuanFrame;
        if(yuanFrame.x !== frame.x || yuanFrame.y !== frame.y ||
        yuanFrame.width !== frame.width || yuanFrame.height !== frame.height){
            this.setFrame(yuanFrame.x, yuanFrame.y, yuanFrame.width, yuanFrame.height);
        }
    }
};


(() => {
    
    /////////////////プラグイン関係/////////////////
    
    const pluginName = "MK_CharaPicture";
    
    //----------プラグインパラメーター----------
    
    const parameters = PluginManager.parameters(pluginName);
    const p_messageControlCharacter = String(parameters["messageControlCharacter"]) || "CP";
    
    //----------キャラ表示関係----------
    
    //キャラの用意
    PluginManager.registerCommand(pluginName, "charaSet", args => {
        $gameTemp.MK_CharaPicture_CharaIdSet(Number(args.charaId));
        $gameTemp.MK_CharaPicture_CharaDisplay(true);
    });
    
    //キャラの差分を変更
    PluginManager.registerCommand(pluginName, "differenceChange", args => {
        $gameScreen.MK_CharaPicture_DifferenceWord(Number(args.pictureId), args.difference);
    });
    
    //キャラの表示範囲設定を変更
    PluginManager.registerCommand(pluginName, "frameSettings", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            picture.mk_charaPicture.FrameSettings(Number(args.x), Number(args.y), Number(args.width), Number(args.height));
        }
    });
    
    //キャラの立ち絵で顔グラの表示設定
    PluginManager.registerCommand(pluginName, "faceRange", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            picture.mk_charaPicture.FaceRange(Number(args.x), Number(args.y), Number(args.width), Number(args.height));
        }
    });
    
    //キャラの立ち絵とアクターを紐づける
    PluginManager.registerCommand(pluginName, "actorLinking", args => {
        $gameSystem.MK_CharaPicture_ActorLinkingSet(Number(args.pictureId), Number(args.actorId));
    });
    
    //キャラの自動操作を有効にするか
    PluginManager.registerCommand(pluginName, "autoOperationActive", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            picture.mk_charaPicture.AutoOperationActive((args.active == "true") ? true : false);
        }
    });
    
    //キャラの部位の表示切替
    PluginManager.registerCommand(pluginName, "partVisible", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            const partId = $gameScreen.MK_CharaPicture_SearchPartName(picture.mk_charaPicture.charaId, args.part);
            picture.mk_charaPicture.PartVisible(partId, (args.visible == "true") ? true : false);
        }
    });

    //キャラの部位の位置を変更
    PluginManager.registerCommand(pluginName, "partPosition", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            const partId = $gameScreen.MK_CharaPicture_SearchPartName(picture.mk_charaPicture.charaId, args.part);
            picture.mk_charaPicture.PartPosition(partId, Number(args.x), Number(args.y));
        }
    });
    
    //キャラの部位の不透明度を変更
    PluginManager.registerCommand(pluginName, "partOpacity", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            const partId = $gameScreen.MK_CharaPicture_SearchPartName(picture.mk_charaPicture.charaId, args.part);
            picture.mk_charaPicture.PartOpacity(partId, Number(args.opacity));
        }
    });
    
    //キャラの部位の色調を変更
    PluginManager.registerCommand(pluginName, "partTone", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            const partId = $gameScreen.MK_CharaPicture_SearchPartName(picture.mk_charaPicture.charaId, args.part);
            picture.mk_charaPicture.PartTone(partId, [Number(args.tintR), Number(args.tintG), Number(args.tintB), Number(args.tintGray)]);
        }
    });
    
    //キャラの部位の色相を変更
    PluginManager.registerCommand(pluginName, "partHue", args => {
        const picture = $gameScreen.picture(Number(args.pictureId));
        if(!!picture && !!picture.mk_charaPicture){
            const partId = $gameScreen.MK_CharaPicture_SearchPartName(picture.mk_charaPicture.charaId, args.part);
            picture.mk_charaPicture.PartHue(partId, Number(args.hue));
        }
    });
    
    
    /////////////////データベース/////////////////
    
    //----------データベースオブジェクト----------
    
    let data = new MK_CharaPicture_Database();

    function MK_CharaPicture_Database() {
        this.chara = null;
        this.id_chara = 0;
        this.id_autoOperation = 0;
        this.id_part = 0;
        this.id_difference = 0;
        this.id_bitmap = 0;
    }

    //キャラ
    MK_CharaPicture_Database.prototype.Chara = function(name) {
        this.id_chara++;
        this.id_autoOperation = 0;
        this.id_part = 0;
        this.id_difference = 0;
        this.id_bitmap = 0;
        $data_mk_charaPicture[this.id_chara] = {
            id : this.id_chara,
            name : name,
            faceRange : {x : 0, y : 0, width : 0, height : 0},
            actorLinking : {pictureId : 0, actorId : 0},
            parts : [null],
            autoOperations : [null]
        };
        this.chara = $data_mk_charaPicture[this.id_chara];
    };

    //顔グラ範囲
    MK_CharaPicture_Database.prototype.FaceRange = function(x, y, width, height) {
        const faceRange = this.chara.faceRange;
        faceRange.x = x;
        faceRange.y = y;
        faceRange.width = width;
        faceRange.height = height;
    };
    
    //キャラとアクターの紐づけ
    MK_CharaPicture_Database.prototype.ActorLinking = function(pictureId, actorId) {
        const actorLinking = this.chara.actorLinking;
        actorLinking.pictureId = pictureId;
        actorLinking.actorId = actorId;
    };

    //部位
    MK_CharaPicture_Database.prototype.Part = function(name) {
        this.id_part++;
        this.id_difference = 0;
        this.chara.parts[this.id_part] = {
            id : this.id_part,
            name : name,
            differences : [null]
        };
    };

    //差分
    MK_CharaPicture_Database.prototype.Difference = function(name) {
        this.id_difference++;
        this.id_bitmap = 0;
        this.chara.parts[this.id_part].differences[this.id_difference] = {
            id : this.id_difference,
            name : name,
            bitmaps : [null]
        };
    };

    //画像
    MK_CharaPicture_Database.prototype.Bitmap = function(name, condition) {//画像名, 条件js
        this.id_bitmap++;
        this.chara.parts[this.id_part].differences[this.id_difference].bitmaps[this.id_bitmap] =  {
            id : this.id_bitmap,
            bitmap : name,
            condition : condition　|| "true"
        };
    };
    
    //自動操作
    MK_CharaPicture_Database.prototype.AutoOperation = function(condition, action) {//条件js, js
        this.id_autoOperation++;
        this.chara.autoOperations[this.id_autoOperation] = {
            id : this.id_autoOperation,
            condition : condition,
            action : action
        };
    };

    
    //----------データベース入力場所----------
    //-----ここから-----


    //キャラID 1
    data.Chara("プルン");
        data.FaceRange(63, 43, 323, 323);
        data.ActorLinking(1, 1);

        //部位ID 1
        data.Part("体");
            //差分ID 1
            data.Difference("体_普通");
            data.Bitmap("purun_karada");

        //部位ID 2
        data.Part("表情");
            //差分ID 1
            data.Difference("普通");
            data.Bitmap("purun_hyouzyou_hutuu");
            //差分ID 2
            data.Difference("嬉しい");
            data.Bitmap("purun_hyouzyou_uresii");
            //差分ID 3
            data.Difference("にこり");
            data.Bitmap("purun_hyouzyou_nikori");
            //差分ID 4
            data.Difference("困る");
            data.Bitmap("purun_hyouzyou_komaru");
            //差分ID 5
            data.Difference("びっくり");
            data.Bitmap("purun_hyouzyou_biltukuri");
            //差分ID 6
            data.Difference("惚れ");
            data.Bitmap("purun_hyouzyou_hore");

    //-----ここまで-----
    
    
    /////////////////元々オブジェクト/////////////
    
    //----------データマネージャー----------
    
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.apply(this, arguments);
        if(typeof $gameSystem.mk_charaPicture_actorLinkings === "undefined"){
            $gameSystem.MK_CharaPicture_Initialize();
        }
    };
    
    //----------イメージマネージャー----------

    //imgにあるMK_charaPictureのファイルを読込ます
    ImageManager.MK_CharaPicture_LoadCharaPicture = function(filename) {
        return ImageManager.loadBitmap('img/mk_charaPicture/', filename);
    };
    
    //----------ゲームテンポラリ----------
    
    const _Game_Temp_initialize = Game_Temp.prototype.initialize;
    Game_Temp.prototype.initialize = function() {
        _Game_Temp_initialize.apply(this, arguments);
        this.MK_CharaPicture_Initialize();
    };
    
    Game_Temp.prototype.MK_CharaPicture_Initialize = function() {
        this.mk_charaPicture = {
            chara : false,
            charaId : 0
        };
    };
    
    Game_Temp.prototype.MK_CharaPicture_Chara = function() {
        return this.mk_charaPicture.chara;
    };
    
    Game_Temp.prototype.MK_CharaPicture_CharaDisplay = function(display) {
        this.mk_charaPicture.chara = display;
    };
    
    Game_Temp.prototype.MK_CharaPicture_CharaId = function() {
        return this.mk_charaPicture.charaId;
    };
    
    Game_Temp.prototype.MK_CharaPicture_CharaIdSet = function(charaId) {
        this.mk_charaPicture.charaId = charaId;
    };
    
    //----------ゲームシステム----------
    
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.apply(this, arguments);
        this.MK_CharaPicture_Initialize();
    };
    
    Game_System.prototype.MK_CharaPicture_Initialize = function() {
        this.mk_charaPicture_actorLinkings = {};
        this.mk_charaPicture_messagePictureId = -1;
    };
    
    Game_System.prototype.MK_CharaPicture_ActorLinking = function(actorId) {
        return this.mk_charaPicture_actorLinkings["actor_" + actorId];
    };
    
    Game_System.prototype.MK_CharaPicture_ActorLinkingSet = function(pictureId, actorId) {
        if(pictureId > 0){
            this.mk_charaPicture_actorLinkings["actor_" + actorId] = pictureId;
        }else{
            delete this.mk_charaPicture_actorLinkings["actor_" + actorId];
        }
    };
    
    Game_System.prototype.MK_CharaPicture_MessagePictureId = function() {
        return this.mk_charaPicture_messagePictureId;
    };
    
    Game_System.prototype.MK_CharaPicture_MessagePictureIdSet = function(pictureId) {
        if(pictureId > 0){
            this.mk_charaPicture_messagePictureId = pictureId;
        }else{
            this.mk_charaPicture_messagePictureId = -1;
        }
    };
    
    //----------ゲームメッセージ----------
    
    const charaPictureControlCharacter = new RegExp("\\\\" + p_messageControlCharacter + "\\[(.*?)\\]", "gi");
    
    const _Game_Message_setSpeakerName = Game_Message.prototype.setSpeakerName;
    Game_Message.prototype.setSpeakerName = function(speakerName) {
        if(!!speakerName){
            speakerName = speakerName.replace(charaPictureControlCharacter, (_, p1) => {
                let word = p1.split(':');
                word[0] = Number(word[0]);
                const picture = $gameScreen.picture(word[0]);
                if(!!picture && !!picture.mk_charaPicture){
                    if(word[1] != null){
                        $gameScreen.MK_CharaPicture_DifferenceWord(word[0],word[1]);
                    }
                    $gameSystem.MK_CharaPicture_MessagePictureIdSet(word[0])
                    return "";
                }
                return "";
                }
            );
        }
        _Game_Message_setSpeakerName.apply(this, arguments);
    };
    
    //----------ゲームスクリーン----------
    
    //名前検索にヒットしたキャラidを返す。なければ-1
    Game_Screen.prototype.MK_CharaPicture_SearchCharaName = function(name) {//検索するキャラ名
        return $data_mk_charaPicture.findIndex((chara) => !!chara && chara.name == name);
    };

    //名前検索にヒットしたキャラの部位idを返す。なければ-1
    Game_Screen.prototype.MK_CharaPicture_SearchPartName = function(charaId, name) {//キャラid,検索する部位名
        return $data_mk_charaPicture[charaId].parts.findIndex((part) => !!part && part.name == name);
    };

    //名前検索にヒットしたキャラの差分idを返す。なければ-1
    Game_Screen.prototype.MK_CharaPicture_SearchDifferenceName = function(charaId, partId, name) {//キャラid,部位id,検索する部位名
        return $data_mk_charaPicture[charaId].parts[partId].differences.findIndex((differences) => !!differences && differences.name == name);
    };
    
    //全キャラ自動操作させる
    Game_Screen.prototype.MK_CharaPicture_AllProcessingAutoOperation = function() {
        const charaPictures = this._pictures.filter(picture => !!picture.mk_charaPicture);
        charaPictures.forEach(chara => chara.mk_charaPicture.this.ProcessingAutoOperation());
    };
    
    //指定したピクチャIDのキャラの現在の差分idを返す。なければ-1
    Game_Screen.prototype.MK_CharaPicture_DifferenceId = function(pictureId, partId) {//ピクチャid,部位id
        const picture = this.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            return picture.mk_charaPicture.DifferenceId(partId);
        }
        return -1;
    };
    
    //プリロード
    Game_Screen.prototype.MK_CharaPicture_Preload = function(charaId) {
        const chara = $data_mk_charaPicture[charaId];
        let difference = null;
        let bitmap = "";
        if(chara){
            for(let r1 = 1; r1 < chara.parts.length; r1++){
                difference = chara.parts[r1].differences;
                for(let r2 = 1; r2 < difference.length; r2++){
                    for(let r3 = 1; r3 < difference[r2].bitmaps.length; r3++){
                        ImageManager.MK_CharaPicture_LoadCharaPicture(difference[r2].bitmaps[r3].bitmap);
                    }
                }
            }
        }
    };

    //文字列で差分を変更する
    Game_Screen.prototype.MK_CharaPicture_DifferenceWord = function(pictureId, difference) {//ピクチャid,差分名
        const picture = this.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            const differences = difference.split(',');
            const chara = $data_mk_charaPicture[picture.mk_charaPicture.charaId];
            chara.parts.forEach(p => {
                if(!!p){
                    differences.forEach(d => {
                        const differenceId = this.MK_CharaPicture_SearchDifferenceName(chara.id, p.id, d);
                        if(differenceId != -1){
                            picture.mk_charaPicture.Difference(p.id, differenceId);
                        }else if(d == ""){
                            picture.mk_charaPicture.Difference(p.id, -1);
                        }
                    });
                }
            });
        }
    };
    
    //----------ゲームピクチャ----------
    
    const _Game_Picture_initialize = Game_Picture.prototype.initialize;
    Game_Picture.prototype.initialize = function() {
        this.mk_charaPicture = null;
        if($gameTemp.MK_CharaPicture_Chara() && $data_mk_charaPicture[$gameTemp.MK_CharaPicture_CharaId()]){
            this.mk_charaPicture = new MK_CharaPicture_Chara($gameTemp.MK_CharaPicture_CharaId());
            $gameTemp.MK_CharaPicture_CharaDisplay(false);
        }
        _Game_Picture_initialize.apply(this, arguments);
    };
    
    
    //----------ゲームインタープリター----------
    
    const _Game_Interpreter_loadImages = Game_Interpreter.prototype.loadImages;
    Game_Interpreter.prototype.loadImages = function() {
        _Game_Interpreter_loadImages.apply(this, arguments);
        const list = this._list.slice(0, 200);
        for(const command of list){
            if(command.code == 357 && command.parameters[0] == pluginName && command.parameters[1] =="charaSet"){
                $gameScreen.MK_CharaPicture_Preload(Number(command.parameters[3].charaId));
            }else if(command.code == 357 && command.parameters[0] == pluginName && command.parameters[1] == "differenceChange"){
                this.MK_CharaPicture_loadImages(Number(command.parameters[3].pictureId));
            }else if(command.code == 101 && !!command.parameters[4] && command.parameters[4].includes("\\CP")){
                let word = command.parameters[4].replace(/\\CP/g, "");
                word = word.split(':');
                this.MK_CharaPicture_loadImages(Number(word[0]));
            }
        }
    };
    
    Game_Interpreter.prototype.MK_CharaPicture_loadImages = function(pictureId) {
        const picture = $gameScreen.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            $gameScreen.MK_CharaPicture_Preload(picture.mk_charaPicture.charaId);
        }
    };
    
    
    //----------スプライトピクチャ----------
    
    const _Sprite_Picture_initialize = Sprite_Picture.prototype.initialize;
    Sprite_Picture.prototype.initialize = function(pictureId) {
        this.mk_charaPicture_pictureName = "";
        this.mk_charaPicture_frameDisplay = false;
        _Sprite_Picture_initialize.apply(this, arguments);
    };
    
    const _Sprite_Picture_updateBitmap = Sprite_Picture.prototype.updateBitmap;
    Sprite_Picture.prototype.updateBitmap = function() {
        const picture = this.picture();
        if (!!picture && !!picture.mk_charaPicture) {
            const pictureName = picture.mk_charaPicture.PictureName();
            if(this.mk_charaPicture_pictureName != pictureName){
                this.mk_charaPicture_pictureName = pictureName;
                this.MK_CharaPicture_PartsDestroy();
                this.MK_CharaPicture_CharaRegister();
            }
            picture.mk_charaPicture.ProcessingAutoOperation();
            this.children.forEach(sprite => sprite.MK_CharaPicture_Update());
            this.visible = this.MK_CharaPicture_CheckPartsLoad() ? true : false;
        }else{
            if(this.mk_charaPicture_pictureName){
                this.MK_CharaPicture_PartsDestroy();
                this.mk_charaPicture_pictureName = "";
            }
        }
        _Sprite_Picture_updateBitmap.apply(this, arguments);
    };
    
    Sprite_Picture.prototype.MK_CharaPicture_PartsDestroy = function() {
        for(let r = this.children.length - 1; r >= 0; --r){
            if(this.children[r].constructor.name == "MK_CharaPicture_Sprite_Part"){
                this.children[r].destroy();
            }
        }
    };
    
    Sprite_Picture.prototype.MK_CharaPicture_CharaRegister = function() {
        const chara = this.picture().mk_charaPicture;
        const parts = $data_mk_charaPicture[chara.charaId].parts;
        let sprite = null;
        for(let r = 1; r < parts.length; r++){
            sprite = new MK_CharaPicture_Sprite_Part(this._pictureId, parts[r].id);
            this.addChild(sprite);
        }
    };
    
    Sprite_Picture.prototype.MK_CharaPicture_CheckPartsLoad = function() {
        return this.children.every((sprite) => {
            if(!sprite.bitmap || sprite.bitmap._loadingState == "loaded" || sprite.bitmap._url == ""){
                return true;
            }
            return false;
        });
    };
    
    const _Sprite_Picture_updateOther = Sprite_Picture.prototype.updateOther;
    Sprite_Picture.prototype.updateOther = function() {
        _Sprite_Picture_updateOther.apply(this, arguments);
        if(this.mk_charaPicture_pictureName){
            const picture = this.picture();
            const frame = picture.mk_charaPicture.frame;
            const sprites = this.children.filter(sprite => sprite.constructor.name == "MK_CharaPicture_Sprite_Part");
            sprites.forEach(sprite => {
                sprite.MK_CharaPicture_UpdateOrigin(this.anchor.x, this.anchor.y);
                sprite.MK_CharaPicture_UpdateBlendMode(this.blendMode);
                sprite.MK_CharaPicture_UpdateFrame();
            });
        }
    };
    
    
    //----------ベースウィンドウ----------
    
    const _Window_Base_initialize = Window_Base.prototype.initialize;
    Window_Base.prototype.initialize = function(rect) {
        _Window_Base_initialize.apply(this, arguments);
        this.mk_charaPicture_bitmapsDatas = [];
    };
    
    Window_Base.prototype.MK_CharaPicture_DrawCharaPicture = function(pictureId, x, y, scale_x, scale_y) {
        const picture = $gameScreen.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            scale_x = scale_x || 100;
            scale_y = scale_y || 100;
            const frame = picture.mk_charaPicture.frame;
            const sx = frame.display ? frame.x : 0;
            const sy = frame.display ? frame.y : 0;
            let sw = frame.display ? frame.width : 0;
            let sh = frame.display ? frame.height : 0;
            let bitmap = null;
            const charaData = this.MK_CharaPicture_DrawCharaPictureData(pictureId);
            const bitmapsDataId = this.MK_CharaPicture_NotUsedBitmapsDataId();
            charaData.forEach((data) => {
                bitmap = ImageManager.MK_CharaPicture_LoadCharaPicture(data.bitmap);
                if(!frame.display){
                    sw = bitmap.width;
                    sh = bitmap.height;
                }
                this.MK_CharaPicture_CreateBitmapData(bitmapsDataId, data.bitmap, data.opacity, sx, sy, sw, sh, data.x, data.y, sw * scale_x / 100, sh * scale_y / 100);
            });
        }
    };
    
    Window_Base.prototype.MK_CharaPicture_DrawCharaPictureFaceSize = function(pictureId, x, y, width, height) {
        const picture = $gameScreen.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            const face = picture.mk_charaPicture.face;
            const charaData = this.MK_CharaPicture_DrawCharaPictureData(pictureId);
            width = width || ImageManager.faceWidth;
            height = height || ImageManager.faceHeight;
            const pw = face.width;//元々の立ち絵顔グラサイズ
            const ph = face.height;
            const difference_w = (width - ImageManager.faceWidth) / ImageManager.faceWidth * 100;//指定したサイズは通常顔グラの何%の差があるか
            const difference_h = (height - ImageManager.faceHeight) / ImageManager.faceHeight * 100;
            const sw = Math.min(pw + pw * difference_w / 100, pw);////立ち絵顔グラの幅
            const sh = Math.min(ph + ph * difference_h / 100, ph);
            const dx = Math.floor(x + Math.max(width - ImageManager.faceWidth, 0) / 2);//ウィンドウの表示位置
            const dy = Math.floor(y + Math.max(height - ImageManager.faceHeight, 0) / 2);
            const sx = face.x + Math.max(pw - sw, 0) / 2;//立ち絵顔グラの表示開始位置
            const sy = face.y + Math.max(ph - sh, 0) / 2;
            let scaleWidth = sw;//大きさ
            let scaleHeight = sh;
            if(sw > width || sh > height){//必要なら縮小
                let w = 100;
                let h = 100;
                while(true){
                    if(sw * w / 100 <= width && sh * h / 100 <= height){
                        scaleWidth = sw * w / 100;
                        scaleHeight = sh * h / 100;
                        break;
                    }
                    w--;
                    h--;
                }
            }

            const bitmapsDataId = this.MK_CharaPicture_NotUsedBitmapsDataId();
            charaData.forEach((data) => {
                this.MK_CharaPicture_CreateBitmapData(bitmapsDataId, data.bitmap, data.opacity, sx - data.x, sy - data.y, sw, sh, dx, dy, scaleWidth, scaleHeight);
            });
        }
    };
    
    Window_Base.prototype.MK_CharaPicture_DrawCharaPictureData = function(pictureId) {
        const picture = $gameScreen.picture(pictureId);
        if(!!picture && !!picture.mk_charaPicture){
            const chara = picture.mk_charaPicture;
            const parts = chara.parts;
            const chara_data = $data_mk_charaPicture[chara.charaId];
            let charaDatas = [];
            let partId = -1;
            chara.ProcessingAutoOperation();
            for(let r = 1;r < chara_data.parts.length; r++){
                partId = picture.mk_charaPicture.CheckPartId(chara_data.parts[r].id);
                if(parts[partId].visible){
                    const data = new function(){
                        this.x = parts[partId].x;
                        this.y = parts[partId].y;
                        this.bitmap = chara.DifferenceBitmap(parts[partId].id, parts[partId].differenceId);
                        this.opacity = parts[partId].opacity * (picture._opacity / 255 * 100) / 100;
                    };
                    charaDatas.push(data);
                }
            }
            return charaDatas;
        }
    };
    
    Window_Base.prototype.MK_CharaPicture_NotUsedBitmapsDataId = function() {
        let bitmapsDataId = 0;
        while(true){
            if(!this.mk_charaPicture_bitmapsDatas[bitmapsDataId] || this.mk_charaPicture_bitmapsDatas[bitmapsDataId].length == 0){
                break;
            }
            bitmapsDataId++;
        }
        this.mk_charaPicture_bitmapsDatas[bitmapsDataId] = [];
        return bitmapsDataId;
    };
    
    Window_Base.prototype.MK_CharaPicture_CreateBitmapData = function(bitmapsDataId, bitmap, opacity, sx, sy, sw, sh, dx, dy, dw, dh) {//bitmapsDatasのid,画像,不透明度,切り取りx,y,幅x,y,位置x,y,拡大x,y
        const data = {
            bitmap :　ImageManager.MK_CharaPicture_LoadCharaPicture(bitmap),
            opacity : opacity,
            sx : sx,
            sy : sy,
            sw : sw,
            sh : sh,
            dx : dx,
            dy : dy,
            dw : dw,
            dh : dh
        };
        this.mk_charaPicture_bitmapsDatas[bitmapsDataId].push(data);
        data.bitmap.addLoadListener(() => {
            this.MK_CharaPicture_DrawBitmaps(bitmapsDataId);
        });
    };
    
    Window_Base.prototype.MK_CharaPicture_DrawBitmaps = function(bitmapsDataId) {
        if(this.MK_CharaPicture_CheckBitmapsDataLoad(bitmapsDataId)){
            let data = null;
            this.mk_charaPicture_bitmapsDatas[bitmapsDataId].forEach((data) => {
                const momentOpacity = this.contents.paintOpacity;
                this.contents.paintOpacity = data.opacity;
                this.contents.blt(data.bitmap, data.sx, data.sy, data.sw, data.sh, data.dx, data.dy, data.dw, data.dh);//画像,切り取りx,y,幅x,y,位置x,y,拡大x,y
                this.contents.paintOpacity = momentOpacity;
            });
            this.mk_charaPicture_bitmapsDatas[bitmapsDataId] = [];
        }
    };
    
    Window_Base.prototype.MK_CharaPicture_CheckBitmapsDataLoad = function(bitmapsDataId) {
        return this.mk_charaPicture_bitmapsDatas[bitmapsDataId].every((data) => {
            if(!data.bitmap || data.bitmap._loadingState == "loaded" || data.bitmap._url == ""){
                return true;
            }
            return false;
        });
    };
    
    
    //----------ステータスベースウィンドウ----------
    
    const _Window_StatusBase_drawActorFace = Window_StatusBase.prototype.drawActorFace;
    Window_StatusBase.prototype.drawActorFace = function(actor, x, y, width, height) {
        const actorLinking = $gameSystem.MK_CharaPicture_ActorLinking(actor._actorId);
        if(!!actorLinking){
            const picture = $gameScreen.picture(actorLinking);
            if(!!picture && !!picture.mk_charaPicture){
                this.MK_CharaPicture_DrawCharaPictureFaceSize(actorLinking, x, y, width, height);
            }
        }
        _Window_StatusBase_drawActorFace.apply(this, arguments);
    };
    
    
    //----------メッセージウィンドウ----------
    
    const _Window_Message_newLineX = Window_Message.prototype.newLineX;
    Window_Message.prototype.newLineX = function(textState) {
        const x = _Window_Message_newLineX.apply(this, arguments);
        if($gameSystem.MK_CharaPicture_MessagePictureId() != -1){
            const faceWidth = ImageManager.faceWidth;
            const spacing = 20;
            const margin = faceWidth + spacing;
            return textState.rtl ? this.innerWidth - margin : margin;
        }
        return x;
    };
    
    const _Window_Message_drawMessageFace = Window_Message.prototype.drawMessageFace;
    Window_Message.prototype.drawMessageFace = function() {
        const messagePictureId = $gameSystem.MK_CharaPicture_MessagePictureId();
        if(messagePictureId != -1){
            const picture = $gameScreen.picture(messagePictureId);
            if(!!picture && !!picture.mk_charaPicture){
                const rtl = $gameMessage.isRTL();
                const width = ImageManager.faceWidth;
                const height = this.innerHeight;
                const x = rtl ? this.innerWidth - width - 4 : 4;
                this.MK_CharaPicture_DrawCharaPictureFaceSize(messagePictureId, x, 0, width, height);
            }
            $gameSystem.MK_CharaPicture_MessagePictureIdSet(-1);
        }
        _Window_Message_drawMessageFace.apply(this, arguments);
    };
    
})();
