
/*:

@target MZ
@plugindesc ステージセレクト機能を追加します。
@author 紫鳥＠メロウ紙芝居
@url https://meroukamisibai.com/

@help MK_StageSelect.js
ver 1.3

スコアなどに変数を使いたい場合は、
「MK_PluginCommandConvert」プラグインを使おう。

選ばれたステージの確認方法はスクリプトで確認します。

例：
$gameSystem.MK_StageSelect_ChoiceSymbol() == "youzyo" :
選ばれたステージのシンボルが「youzyo」の時。

!$gameSystem.MK_StageSelect_ChoiceSymbol() :
選ばれたステージのシンボルが存在しない場合。
※ステージを選んでない場合も。


スクリプト:

$gameSystem.MK_StageSelect_ChoiceSymbol() :
選択されたステージのシンボルを返します。

$gameSystem.MK_StageSelect_ChoiceSymbolOption() :
選択されたステージのオプションのシンボルを返します。


@command title
@text タイトル名
@desc ステージセレクト画面のタイトルに名前を付けます。

@arg title
@text タイトル
@desc ステージセレクト画面のタイトル名です。
@type string

@command stageAdd
@text ステージ追加
@desc ステージを追加します。

@arg symbol
@text シンボル
@desc ステージのシンボルです。
@type string

@arg name
@text 名前
@desc ステージの名前です。
@type string

@arg icon
@text アイコン
@desc ステージの名前の先頭に表示させるアイコンの番号です。
@type number
@min -1
@default -1

@arg feasible
@text 選択可能か
@desc このステージは選択できるか
@on 選択できる
@off 選択できない
@type boolean
@default true

@arg score
@text スコア
@desc ステージのスコアの文字です。
@type string

@arg scoreValue
@text スコア値
@desc ステージのスコアの値です。
@type string

@arg option
@text オプション
@desc ステージのオプションです。
@type struct<option>[]
@default []

@arg picture
@text 画像
@desc ステージの画像です。
@type file
@dir img/
@default pictures

@command stageRemove
@text ステージ外す
@desc 追加したステージを外します。

@command stageStartCursor
@text ステージカーソル開始位置
@desc ステージセレクト画面を開いた時のステージのカーソルの開始位置を設定します。

@arg symbol
@text シンボル
@desc 開始位置となるステージのシンボルです。存在しない場合は、一番上になります。
@type string

@arg symbolOption
@text オプションシンボル
@desc オプションの開始位置となるステージのオプションのシンボルです。存在しない場合は、一番左になります。
@type string

@command stageSelect
@text ステージセレクト画面
@desc ステージセレクト画面を開きます。

*/

/*~struct~option:

@param symbol
@text シンボル
@desc オプションのシンボルです。
@type string

@param name
@text 名前
@desc オプションの名前です。
@type string

@param feasible
@text 選択可能か
@desc このオプションは選択できるか
@on 選択できる
@off 選択できない
@type boolean
@default true

@param score
@text スコア
@desc このオプション選択時のスコアの文字です。設定しない場合は、ステージの設定が適用されます。
@type string

@param scoreValue
@text スコア値
@desc このオプション選択時のスコアの値です。設定しない場合は、ステージの設定が適用されます。
@type string

@param picture
@text 画像
@desc このオプション選択時の画像です。設定しない場合は、ステージの設定が適用されます。
@type file
@dir img/
@default pictures

*/

/////////////////ステージセレクトシーン/////////////////

//シーン
function MK_StageSelect_Scene() {
    this.mk_stageSelect = $gameSystem.mk_stageSelect;
    this.mk_stages = this.mk_stageSelect.stages;
    this.mk_stage = null;//現在選択されてるステージ
    this.mk_option = null;//現在選択されてるオプション
    this.mk_cursorStageIndex = 0;//ステージカーソル位置
    this.mk_cursorOptionIndex = 0;//オプションカーソル位置
    this.mk_windowUpdate = false;//スコア・ピクチャウィンドウ更新するか
    this.initialize(...arguments);
}

MK_StageSelect_Scene.prototype = Object.create(Scene_MenuBase.prototype);
MK_StageSelect_Scene.prototype.constructor = MK_StageSelect_Scene;

MK_StageSelect_Scene.prototype.initialize = function() {
    Scene_MenuBase.prototype.initialize.call(this);
    this.mk_cursorStageIndex = this.MK_CursorStageIndex();
    this.MK_UpdateStage(this.mk_cursorStageIndex);
    this.mk_cursorOptionIndex = this.MK_CursorOptionIndex();
    this.MK_UpdateOption(this.mk_cursorOptionIndex);
};

//初期ステージカーソル
MK_StageSelect_Scene.prototype.MK_CursorStageIndex = function() {
    if(this.mk_stageSelect.startSymbol != "" && this.mk_stages.length > 0){
        const s = this.mk_stages.findIndex(stage => stage.symbol == this.mk_stageSelect.startSymbol);
        if(s != -1){
            return s;
        }else{
            return 0;
        }
    }else{
        return 0;
    }
};

//初期オプションカーソル
MK_StageSelect_Scene.prototype.MK_CursorOptionIndex = function() {
    if(this.mk_stageSelect.startSymbolOption != "" && this.mk_stage && this.mk_stage.options.length > 0){
        const s = this.mk_stage.options.findIndex(option => option.symbol == this.mk_stageSelect.startSymbolOption);
        if(s != -1){
            return s;
        }else{
            return 0;
        }
    }else{
        return 0;
    }
};

//ステージ更新
MK_StageSelect_Scene.prototype.MK_UpdateStage = function(index) {//ステージid
    if(this.mk_stages.length > 0){
        this.mk_stage = this.mk_stages[index];
    }else{
        this.mk_stage = null;
    }
};

//オプション更新
MK_StageSelect_Scene.prototype.MK_UpdateOption = function(index) {//オプションid
    if(this.mk_stage && this.mk_stage.options.length > 0){
        this.mk_option = this.mk_stage.options[index];
    }else{
        this.mk_option = null;
    }
};

MK_StageSelect_Scene.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
    this.MK_CreateTitleWindow();
    this.MK_CreateSelectWindow();
    this.MK_CreateOptionWindow();
    this.MK_CreateScoreWindow();
    this.MK_CreatePictureWindow();
    this.MK_WindowRefresh();
};

MK_StageSelect_Scene.prototype.MK_CreateTitleWindow = function() {
    const rect = this.MK_CreateTitleWindowRect();
    this.mk_stageSelect_window_title = new MK_StageSelect_Window_Title(rect);
    this.addWindow(this.mk_stageSelect_window_title);
};

MK_StageSelect_Scene.prototype.MK_CreateTitleWindowRect = function() {
    const ww = Graphics.boxWidth / 3.5;
    const wh = this.calcWindowHeight(1, false);
    const wx = 50 + (Graphics.boxWidth / 3 - ww) / 2;
    const wy = 55;
    return new Rectangle(wx, wy, ww, wh);
};

MK_StageSelect_Scene.prototype.MK_CreateSelectWindow = function() {
    const rect = this.MK_CreateSelectWindowRect();
    this.mk_stageSelect_window_select = new MK_StageSelect_Window_Select(rect);
    this.mk_stageSelect_window_select.setHandler('ok', this.onOk.bind(this));
    this.mk_stageSelect_window_select.setHandler('cancel', this.onCancel.bind(this));
    if(this.mk_stages.length > 0){
        this.mk_stageSelect_window_select.forceSelect(this.mk_cursorStageIndex);
    }else{
        this.mk_stageSelect_window_select.deselect();
    }
    this.mk_stageSelect_window_select.MK_OptionUpdate(this.mk_cursorOptionIndex);
    this.addWindow(this.mk_stageSelect_window_select);
};

MK_StageSelect_Scene.prototype.MK_CreateSelectWindowRect = function() {
    const ww = Graphics.boxWidth / 3;
    const wh = Graphics.boxHeight / 1.6;
    const wx = 50;
    const wy = 140;
    return new Rectangle(wx, wy, ww, wh);
};

MK_StageSelect_Scene.prototype.MK_CreateOptionWindow = function() {
    const rect = this.MK_CreateOptionWindowRect();
    this.mk_stageSelect_window_option = new MK_StageSelect_Window_Option(rect);
    this.mk_stageSelect_window_option.select(this.mk_cursorOptionIndex);
    this.mk_stageSelect_window_option.MK_UpdateOption(this.mk_stage);
    this.addWindow(this.mk_stageSelect_window_option);
};

MK_StageSelect_Scene.prototype.MK_CreateOptionWindowRect = function() {
    const ww = Graphics.boxWidth / 2.5;
    const wh = this.calcWindowHeight(1, true);
    const wx = 10 + Graphics.boxWidth / 2;
    const wy = 55;
    return new Rectangle(wx, wy, ww, wh);
};

MK_StageSelect_Scene.prototype.MK_CreateScoreWindow = function() {
    const rect = this.MK_CreateScoreWindowRect();
    this.mk_stageSelect_window_score = new MK_StageSelect_Window_Score(rect);
    this.addWindow(this.mk_stageSelect_window_score);
};

MK_StageSelect_Scene.prototype.MK_CreateScoreWindowRect = function() {
    const ww = Graphics.boxWidth / 2.5;
    const wh = this.calcWindowHeight(1, false);
    const wx = 50 + (Graphics.boxWidth / 3 - ww) / 2;
    const wy = Graphics.boxHeight - this.calcWindowHeight(1, false) - 15;
    return new Rectangle(wx, wy, ww, wh);
};

MK_StageSelect_Scene.prototype.MK_CreatePictureWindow = function() {
    const rect = this.MK_CreatePictureWindowRect();
    this.mk_stageSelect_window_picture = new MK_StageSelect_Window_Picture(rect);
    this.addWindow(this.mk_stageSelect_window_picture);
};

MK_StageSelect_Scene.prototype.MK_CreatePictureWindowRect = function() {
    const ww = Graphics.boxWidth / 2;
    const wh = Graphics.boxHeight / 1.6;
    const wx = 10 + Graphics.boxWidth / 2 + (Graphics.boxWidth / 2.5 - ww) / 2;
    const wy = 140;
    return new Rectangle(wx, wy, ww, wh);
};

MK_StageSelect_Scene.prototype.onOk = function() {
    const symbol = this.mk_stages.length > 0 ? this.mk_stage.symbol : "";
    let symbolOption = "";
    if(this.mk_option){
        symbolOption = this.mk_option.symbol;
    }
    $gameSystem.MK_StageSelect_ChoiceSymbolSet(symbol);
    $gameSystem.MK_StageSelect_ChoiceSymbolOptionSet(symbolOption);
    $gameSystem.MK_StageSelect_StageStartCursor(symbol, symbolOption);
    this.popScene();
};

MK_StageSelect_Scene.prototype.onCancel = function() {
    $gameSystem.MK_StageSelect_ChoiceSymbolSet("");
    $gameSystem.MK_StageSelect_ChoiceSymbolOptionSet("");
    this.popScene();
};

MK_StageSelect_Scene.prototype.update = function() {
    Scene_Base.prototype.update.call(this);
    if(this.mk_cursorStageIndex != this.mk_stageSelect_window_select._index){
        this.mk_cursorStageIndex = this.mk_stageSelect_window_select._index;
        this.MK_UpdateStage(this.mk_cursorStageIndex);
        this.MK_UpdateOption(this.mk_cursorOptionIndex);
        this.mk_stageSelect_window_option.MK_UpdateOption(this.mk_stage);
        this.mk_windowUpdate = true;
    }
    if(this.mk_cursorOptionIndex != this.mk_stageSelect_window_option._index){
        this.mk_cursorOptionIndex = this.mk_stageSelect_window_option._index;
        this.MK_UpdateOption(this.mk_cursorOptionIndex);
        this.mk_windowUpdate = true;
    }
    if(this.mk_windowUpdate){
        this.MK_WindowRefresh();
        this.mk_windowUpdate = false;
    }
};

MK_StageSelect_Scene.prototype.MK_WindowRefresh = function() {
    this.mk_stageSelect_window_select.MK_OptionUpdate(this.mk_cursorOptionIndex);
    this.mk_stageSelect_window_score.MK_StageUpdate(this.mk_stage, this.mk_cursorOptionIndex);
    this.mk_stageSelect_window_picture.MK_StageUpdate(this.mk_stage, this.mk_cursorOptionIndex);
};


/////////////////ステージセレクトタイトルウィンドウ/////////////////

function MK_StageSelect_Window_Title() {
    this.initialize(...arguments);
}

MK_StageSelect_Window_Title.prototype = Object.create(Window_Base.prototype);
MK_StageSelect_Window_Title.prototype.constructor = MK_StageSelect_Window_Title;

MK_StageSelect_Window_Title.prototype.initialize = function(rect) {
    Window_Base.prototype.initialize.call(this, rect);
    this.refresh();
};

MK_StageSelect_Window_Title.prototype.refresh = function() {
    const name = $gameSystem.mk_stageSelect.title;
    if(name){
        this.drawText(name, 0, 0, this.contentsWidth(), 'center');
    }else{
        this.hide();
    }
};


/////////////////ステージセレクトウィンドウ/////////////////

function MK_StageSelect_Window_Select() {
    this.mk_cursorOptionIndex = 0;//オプションカーソル位置
    
    this.initialize(...arguments);
}

MK_StageSelect_Window_Select.prototype = Object.create(Window_Selectable.prototype);
MK_StageSelect_Window_Select.prototype.constructor = MK_StageSelect_Window_Select;

MK_StageSelect_Window_Select.prototype.initialize = function(rect) {
    Window_Selectable.prototype.initialize.call(this, rect);
    this.select(0);
    this.activate();
    this.refresh();
};

MK_StageSelect_Window_Select.prototype.maxCols = function() {
    return 1;
};

MK_StageSelect_Window_Select.prototype.maxItems = function() {
    return $gameSystem.mk_stageSelect.stages.length;
};

MK_StageSelect_Window_Select.prototype.drawItem = function(index) {
    const rect = this.itemLineRect(index);
    this.changePaintOpacity($gameSystem.mk_stageSelect.stages[index].feasible);
    const textWidth = this.drawTextEx(this.MK_Icon(index) , rect.x, rect.y);
    this.drawText(this.MK_Name(index), rect.x  + textWidth, rect.y, rect.width - textWidth);
    this.changePaintOpacity(true);
};

//選択できるか
MK_StageSelect_Window_Select.prototype.MK_Enable = function(index) {
    if($gameSystem.mk_stageSelect.stages.length > 0 && $gameSystem.mk_stageSelect.stages[index].feasible){
        if($gameSystem.mk_stageSelect.stages[index].options.length > 0){
            return $gameSystem.mk_stageSelect.stages[index].options[this.mk_cursorOptionIndex].feasible;
        }
        return $gameSystem.mk_stageSelect.stages[index].feasible;
    }
    return false;
};

//ステージアイコン返す
MK_StageSelect_Window_Select.prototype.MK_Icon = function(index) {
    const id = $gameSystem.mk_stageSelect.stages[index].icon;
    if(id != -1){
        return "\\I[" + id + "]";
    }
    return "";
};

//ステージ名返す
MK_StageSelect_Window_Select.prototype.MK_Name = function(index) {
    return $gameSystem.mk_stageSelect.stages[index].name;
};

//決定キー押したとき
MK_StageSelect_Window_Select.prototype.isCurrentItemEnabled = function() {
    return this.MK_Enable(this._index);
};

//選べない時の効果音
MK_StageSelect_Window_Select.prototype.playBuzzerSound = function() {
    if($gameSystem.mk_stageSelect.stages.length > 0){
        Window_Base.prototype.playBuzzerSound.call(this);
    }
};

//オプションカーソル位置更新
MK_StageSelect_Window_Select.prototype.MK_OptionUpdate = function(cursorOptionIndex) {//オプションカーソル位置
    this.mk_cursorOptionIndex = cursorOptionIndex;
};


/////////////////ステージセレクトオプションウィンドウ/////////////////

function MK_StageSelect_Window_Option() {
    this.mk_options = null;//選ばれているステージのオプションデータ
    this.mk_cursorIndex = 0;//カーソル位置
    
    this.initialize(...arguments);
}

MK_StageSelect_Window_Option.prototype = Object.create(Window_Selectable.prototype);
MK_StageSelect_Window_Option.prototype.constructor = MK_StageSelect_Window_Option;

MK_StageSelect_Window_Option.prototype.initialize = function(rect) {
    Window_Selectable.prototype.initialize.call(this, rect);
};

MK_StageSelect_Window_Option.prototype.maxCols = function() {
    if(this.MK_ExistenceOption()){
        return this.mk_options.length;
   }else{
        return 1;
   }
};

MK_StageSelect_Window_Option.prototype.maxItems = function() {
    if(this.MK_ExistenceOption()){
        return this.mk_options.length;
   }else{
        return 0;
   }
};

MK_StageSelect_Window_Option.prototype.update = function() {
    Window_Selectable.prototype.update.call(this);
    if(this.MK_ExistenceOption() && this._index != this.mk_cursorIndex){
        this.refresh();
        this.mk_cursorIndex = this._index;
    }
};

MK_StageSelect_Window_Option.prototype.processTouch = function() {
    if (this.isOpenAndActive()) {
        if (TouchInput.isTriggered()) {
            this.onTouchSelect(true);
        }
        if (TouchInput.isClicked()) {
            this.onTouchOk();
        } else if (TouchInput.isCancelled()) {
            this.onTouchCancel();
        }
    }
};

MK_StageSelect_Window_Option.prototype.drawItem = function(index) {
    if(this._index == index){
        const rect1 = this.itemRect(index);
        const color = ColorManager.pendingColor();
        this.changePaintOpacity(false);
        this.contents.fillRect(rect1.x, rect1.y, rect1.width, rect1.height, color);
    }
    const rect2 = this.itemLineRect(index);
    this.changePaintOpacity(this.mk_options[index].feasible);
    this.drawText(this.mk_options[index].name, rect2.x, rect2.y, rect2.width, 'center');
    this.changePaintOpacity(true);
};

MK_StageSelect_Window_Option.prototype.refreshCursor = function() {
    if(this.index() >= 0){
    }else{
        Window_Selectable.prototype.refreshCursor.call(this);
    }
};

//ステージ変えた時のオプション更新
MK_StageSelect_Window_Option.prototype.MK_UpdateOption = function(stage) {//選択中のステージ
    this.mk_options = (!!stage) ? stage.options : null;
    if(this.MK_ExistenceOption()){
        this.activate();
        this.show();
        this.select((this._index < this.mk_options.length) ? this._index : 0);
        this.refresh();
        this.mk_cursorIndex = this._index;
    }else{
        this.deactivate();
        this.hide();
    }
};

//オプションがあるか
MK_StageSelect_Window_Option.prototype.MK_ExistenceOption = function() {
    return (this.mk_options　&& this.mk_options.length > 0) ? true : false;
};


/////////////////ステージセレクトスコアウィンドウ/////////////////

function MK_StageSelect_Window_Score() {
    this.mk_stage = null;//選ばれているステージ
    this.mk_cursorOptionIndex = 0;//オプションカーソル位置
    
    this.initialize(...arguments);
}

MK_StageSelect_Window_Score.prototype = Object.create(Window_Base.prototype);
MK_StageSelect_Window_Score.prototype.constructor = MK_StageSelect_Window_Score;

MK_StageSelect_Window_Score.prototype.initialize = function(rect) {
    Window_Base.prototype.initialize.call(this, rect);
};

MK_StageSelect_Window_Score.prototype.refresh = function() {
    if(this.mk_stage){
        const score = this.MK_Score();
        const scoreValue = this.MK_ScoreValue();
        if(score != "" || scoreValue != ""){
            this.contents.clear();
            this.show();
            this.drawText(score, 0, 0, this.contentsWidth());
            this.drawText(scoreValue, 0, 0, this.contentsWidth(), 'right');
        }else{
            this.hide();
        }
    }else{
        this.hide();
    }
};

//スコア返す
MK_StageSelect_Window_Score.prototype.MK_Score = function() {
    if(this.mk_stage.options.length > 0){
        const score = this.mk_stage.options[this.mk_cursorOptionIndex].score;
        if(score){
            return score;
        }
    }
    return this.mk_stage.score;
};

//スコア値返す
MK_StageSelect_Window_Score.prototype.MK_ScoreValue = function() {
    if(this.mk_stage.options.length > 0){
        const scoreValue = this.mk_stage.options[this.mk_cursorOptionIndex].scoreValue;
        if(scoreValue){
            return scoreValue;
        }
    }
    return this.mk_stage.scoreValue;
};

//ステージとオプションカーソル位置更新
MK_StageSelect_Window_Score.prototype.MK_StageUpdate = function(stage, cursorOptionIndex) {//ステージ, オプションカーソル位置
    this.mk_stage = stage;
    this.mk_cursorOptionIndex = cursorOptionIndex;
    this.refresh();
};


/////////////////ステージセレクトピクチャウィンドウ/////////////////

function MK_StageSelect_Window_Picture() {
    this.mk_stage = null;//選ばれているステージ
    this.mk_cursorOptionIndex = 0;//オプションカーソル位置
    this.mk_picture = "";//表示する画像名
    
    this.initialize(...arguments);
}

MK_StageSelect_Window_Picture.prototype = Object.create(Window_Base.prototype);
MK_StageSelect_Window_Picture.prototype.constructor = MK_StageSelect_Window_Picture;

MK_StageSelect_Window_Picture.prototype.initialize = function(rect) {
    Window_Base.prototype.initialize.call(this, rect);
};

MK_StageSelect_Window_Picture.prototype.refresh = function() {
    this.contents.clear();
    if(this.mk_stage){
        const picture = this.MK_Picture();
        if(picture != ""){
            this.mk_picture = picture;
            const bitmap = ImageManager.MK_StageSelect_LoadBitmap(picture);
            bitmap.addLoadListener(function() {
                if(this.mk_picture == picture){
                    let scaleWidth = bitmap.width;
                    let scaleHeight = bitmap.height;
                    if(bitmap.width > this.contentsWidth() || bitmap.height > this.contentsHeight()){
                        let w = 100;
                        let h = 100;
                        while(true){
                            if(bitmap.width * w / 100 <= this.contentsWidth() && bitmap.height * h / 100 <= this.contentsHeight()){
                                scaleWidth = bitmap.width * w / 100;
                                scaleHeight = bitmap.height * h / 100;
                                break;
                            }
                            w--;
                            h--;
                        }
                    }
                    dx = this.contentsWidth() / 2 - scaleWidth / 2;
                    dy = this.contentsHeight() / 2 - scaleHeight / 2;
                    this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, dx, dy, scaleWidth, scaleHeight);
                }
            }.bind(this));
        };
    }
};

//画像返す
MK_StageSelect_Window_Picture.prototype.MK_Picture = function() {
    if(this.mk_stage){
        if(this.mk_stage.options.length > 0 && this.mk_stage.options[this.mk_cursorOptionIndex].picture){
            const filenames = this.mk_stage.options[this.mk_cursorOptionIndex].picture.split("/");
            if(filenames[1]){
                return this.mk_stage.options[this.mk_cursorOptionIndex].picture;
            }
        }
        return this.mk_stage.picture;
    }
    return "";
};

//ステージとオプションカーソル位置更新
MK_StageSelect_Window_Picture.prototype.MK_StageUpdate = function(stage, cursorOptionIndex) {//ステージ, オプションカーソル位置
    this.mk_stage = stage;
    this.mk_cursorOptionIndex = cursorOptionIndex;
    this.refresh();
};


(() => {
    
    /////////////////プラグイン関係/////////////////
    
    const pluginName = "MK_StageSelect";
    
    //----------プラグインコマンド----------

    //タイトル名
    PluginManager.registerCommand(pluginName, "title", args => {
        $gameSystem.MK_StageSelect_TitleSet(args.title);
    });
    
    //ステージ追加
    PluginManager.registerCommand(pluginName, "stageAdd", args => {
        const argsOptions = JSON.parse(args.option);
        const options = [];
        argsOptions.forEach(option => {
            option = JSON.parse(option);
            option.feasible = option.feasible == "true" ? true : false;
            options.push($gameSystem.MK_StageSelect_OptionCreate(option.symbol, option.name, option.feasible, option.score, option.scoreValue, option.picture));
        });
            
        const feasible = args.feasible == "true" ? true : false;
        $gameSystem.MK_StageSelect_StageAdd(args.symbol, args.name, Number(args.icon), feasible, args.score, args.scoreValue, options, args.picture);
    });
    
    //ステージ外す
    PluginManager.registerCommand(pluginName, "stageRemove", args => {
        $gameSystem.MK_StageSelect_StageRemove();
    });
    
    //ステージカーソル開始位置
    PluginManager.registerCommand(pluginName, "stageStartCursor", args => {
        $gameSystem.MK_StageSelect_StageStartCursor(args.symbol, args.symbolOption);
    });
    
    //ステージセレクト画面
    PluginManager.registerCommand(pluginName, "stageSelect", args => {
        $gameSystem.MK_StageSelect_StageSelect();
    });
    
    
    /////////////////元々オブジェクト/////////////////
    
    //----------データマネージャー----------
    
    //ロード
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.apply(this, arguments);
        if(typeof $gameSystem.mk_stageSelect === "undefined"){
            $gameSystem.MK_StageSelect_Initialize();
        }
    };
    
    //----------イメージマネージャー----------

    ImageManager.MK_StageSelect_LoadBitmap = function(filename) {
        const word = filename.split("/");
        const folder = "img/" + word.shift() + "/";
        const fileName2 = word.join("/");
        return ImageManager.loadBitmap(folder, fileName2);
    };
    
    //----------ゲームシステム----------
    
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.apply(this, arguments);
        this.MK_StageSelect_Initialize();
    };
    
    Game_System.prototype.MK_StageSelect_Initialize = function() {
        this.mk_stageSelect = {
            title : "",
            stages : [],
            startSymbol : "",
            startSymbolOption : "",
            choiceSymbol : "",
            choiceSymbolOption : ""
        };
    };
    
    Game_System.prototype.MK_StageSelect_TitleSet = function(title) {
        this.mk_stageSelect.title = title;
    };
    
    Game_System.prototype.MK_StageSelect_OptionCreate = function(symbol, name, feasible, score, scoreValue, picture) {
        return {
            symbol : symbol || "",
            name : name || "",
            feasible : feasible,
            score : score || "",
            scoreValue : scoreValue || "",
            picture : picture || ""
        }
    };
    
    Game_System.prototype.MK_StageSelect_StageAdd = function(symbol, name, icon, feasible, score, scoreValue, options, picture) {
        this.mk_stageSelect.stages.push({
            symbol : symbol || "",
            name : name || "",
            icon : icon || -1,
            feasible : feasible,
            score : score || "",
            scoreValue : scoreValue || "",
            options : options || [],
            picture : picture || ""
        });
    };
    
    Game_System.prototype.MK_StageSelect_StageRemove = function() {
        this.mk_stageSelect.stages = [];
    };
    
    Game_System.prototype.MK_StageSelect_StageStartCursor = function(symbol, symbolOption) {
        this.mk_stageSelect.startSymbol = symbol || "";
        this.mk_stageSelect.startSymbolOption = symbolOption || "";
    };
    
    Game_System.prototype.MK_StageSelect_StageSelect = function() {
        SceneManager.push(MK_StageSelect_Scene);
    };
    
    Game_System.prototype.MK_StageSelect_ChoiceSymbol = function() {
        return this.mk_stageSelect.choiceSymbol;
    };
    
    Game_System.prototype.MK_StageSelect_ChoiceSymbolSet = function(symbol) {
        this.mk_stageSelect.choiceSymbol = symbol;
    };
    
    Game_System.prototype.MK_StageSelect_ChoiceSymbolOption = function() {
        return this.mk_stageSelect.choiceSymbolOption;
    };
    
    Game_System.prototype.MK_StageSelect_ChoiceSymbolOptionSet = function(symbol) {
        this.mk_stageSelect.choiceSymbolOption = symbol;
    };
    
    //----------ゲームインタープリター----------
    
    const _Game_Interpreter_loadImages = Game_Interpreter.prototype.loadImages;
    Game_Interpreter.prototype.loadImages = function() {
        _Game_Interpreter_loadImages.apply(this, arguments);
        const list = this._list.slice(0, 200);
        for(const command of list){
            if(command.code == 357 && command.parameters[0] == pluginName && command.parameters[1] =="stageAdd"){
                ImageManager.MK_StageSelect_LoadBitmap(command.parameters[3].picture);
                const argsOptions = JSON.parse(command.parameters[3].option);
                argsOptions.forEach(option => {
                    option = JSON.parse(option);
                    ImageManager.MK_StageSelect_LoadBitmap(option.picture);
                });
            }else if(command.code == 357 && command.parameters[0] == pluginName && command.parameters[1] =="stageSelect"){
                $gameSystem.mk_stageSelect.stages.forEach(stage => {
                    ImageManager.MK_StageSelect_LoadBitmap(stage.picture);
                    stage.options.forEach(option => {
                        ImageManager.MK_StageSelect_LoadBitmap(option.picture);
                    });
                });
            }
        }
    };
    
    
})();
