
/*:

@target MZ
@plugindesc 視界機能を追加します。
@author 紫鳥＠メロウ紙芝居
@url https://meroukamisibai.com/

@help MK_View.js

視界の範囲は「前方」の数だけ前に届き、幅は「横方」の数で広がります。
通れない地形があると、そこまでが視界の範囲になります。
無視する地形タグを設定すると、通れない地形でも視界の範囲は
止まりません。また、見つからない地形タグを設定すると、
見つからない地形にいる状態で視界に入っても
入ったことになりません。

条件分岐:

$mk_view.ViewPlayer(this._eventId) :
このイベントの視界を判定します。

$mk_view.ViewPlayer([イベントID]) :
指定したイベントの視界を判定します。


イベントのメモ:

<mk_view:[種類]> :
イベントの種類を設定します。


@command explanation_view
@text ---視界---
@desc

@command viewRange
@text 視界範囲変更
@desc 視界の範囲を変更します。

@arg forward
@text 前方
@desc 前に届く範囲です。
@type number
@default 1

@arg sideways
@text 横方
@desc 視界の範囲の幅です。
@type number
@default 1

@command explanation_ignore
@text ---無視する地形---
@desc

@command ignoreTerrainAdd
@text 無視する地形タグ追加
@desc 無視する地形タグを追加します。

@arg terrain
@text 地形タグ
@desc 無視する地形タグを追加します。
@type number
@max 7
@min 0
@default 0

@command ignoreTerrainRemove
@text 無視する地形タグ外す
@desc 無視する地形タグを外します。

@arg terrain
@text 地形タグ
@desc 無視する地形タグを外します。
@type number
@max 7
@min 0
@default 0

@command ignoreTerrainAllRemove
@text 無視する地形タグ全て外す
@desc 無視する地形タグを全て外します。

@command explanation_impossible
@text ---見つからない地形---
@desc

@command impossibleTerrainAdd
@text 見つからない地形タグ追加
@desc 見つからない地形タグを追加します。

@arg terrain
@text 地形タグ
@desc 見つからない地形タグを追加します。
@type number
@max 7
@min 0
@default 0

@command impossibleTerrainRemove
@text 見つからない地形タグ外す
@desc 見つからない地形タグを外します。

@arg terrain
@text 地形タグ
@desc 見つからない地形タグを外します。
@type number
@max 7
@min 0
@default 0

@command impossibleTerrainAllRemove
@text 見つからない地形タグ全て外す
@desc 見つからない地形タグを全て外します。

@command explanation_obstructEvent
@text ---遮るイベント種類---
@desc

@command obstructEventTypeAdd
@text 遮るイベントの種類追加
@desc 遮るイベントの種類を追加します。

@arg type
@text 種類
@desc 遮るイベントの種類を追加します。
@type string

@command obstructEventTypeRemove
@text 遮るイベントの種類外す
@desc 遮るイベントの種類を外します。

@arg type
@text 種類
@desc 遮るイベントの種類を外します。
@type string

@command obstructEventTypeAllRemove
@text 遮るイベントの種類全て外す
@desc 遮るイベントの種類を全て外します。

@command explanation_portrayal
@text ---視界の描写---
@desc

@command portrayalColor
@text 視界の色を変更
@desc 視界の色を変更します。

@arg r
@text r
@desc 赤色(0~255)。
@type number
@max 255
@min 0
@default 0

@arg g
@text g
@desc 緑色(0~255)。
@type number
@max 255
@min 0
@default 0

@arg b
@text b
@desc 青色(0~255)。
@type number
@max 255
@min 0
@default 0

@command portrayalOpacity
@text 視界の不透明度変更
@desc 視界の不透明度を変更します。不透明度は全ての視界共通です。

@arg opacity
@text 不透明度
@desc 不透明度(0~255)。
@type number
@max 255
@min 0
@default 0

*/


/////////////////グローバルデータ/////////////////
var $mk_view = null;//視界データ

/////////////////視界データ/////////////

function MK_View() {
    this.forward = 0;//前方
    this.sideways = 0;//横方
    this.obstructEventType = [];//視界を遮るイベントタイプ
    this.ignoreTerrain = [];//無視する地形
    this.impossibleTerrain = [];//見つからない地形
    this.viewsPortrayal = [];//描写する視界データ
    this.rgb_r = 0;//描写視界の赤色
    this.rgb_g = 0;//描写視界の緑色
    this.rgb_b = 0;//描写視界の青色
    this.viewOpacity = 50;//描写視界の不透明度
}

//----------基本----------

//視界範囲変更
MK_View.prototype.ViewRange = function(forward,sideways) {//前方,　横方
    this.forward = forward;
    this.sideways = sideways;
};

//無視する地形タグ追加
MK_View.prototype.IgnoreTerrainAdd = function(tag) {//地形タグ
    if(this.ignoreTerrain.indexOf(tag) == -1){
        this.ignoreTerrain.push(tag);
    }
};

//無視する地形タグ外す
MK_View.prototype.IgnoreTerrainRemove = function(tag) {//地形タグ
    const index = this.ignoreTerrain.indexOf(tag);
    if(index != -1){
        this.ignoreTerrain.splice(index, 1);
    }
};

//無視する地形タグ全て外す
MK_View.prototype.IgnoreTerrainAllRemove = function() {
    this.ignoreTerrain = [];
};

//見つからない地形タグ追加
MK_View.prototype.ImpossibleTerrainAdd = function(tag) {//地形タグ
    if(this.impossibleTerrain.indexOf(tag) == -1){
        this.impossibleTerrain.push(tag);
    }
};

//見つからない地形タグ外す
MK_View.prototype.ImpossibleTerrainRemove = function(tag) {//地形タグ
    const index = this.impossibleTerrain.indexOf(tag);
    if(index != -1){
        this.impossibleTerrain.splice(index, 1);
    }
};

//見つからない地形タグ全て外す
MK_View.prototype.ImpossibleTerrainAllRemove = function() {
    this.impossibleTerrain = [];
};

//遮るイベントの種類追加
MK_View.prototype.ObstructEventTypeAdd = function(type) {//遮るイベントの種類名
    if(this.obstructEventType.indexOf(type) == -1){
        this.obstructEventType.push(type);
    }
};

//遮るイベントの種類外す
MK_View.prototype.ObstructEventTypeRemove = function(type) {//遮るイベントの種類名
    const index = this.obstructEventType.indexOf(type);
    if(index != -1){
        this.obstructEventType.splice(index, 1);
    }
};

//遮るイベントの種類全て外す
MK_View.prototype.ObstructEventTypeAllRemove = function() {
    this.obstructEventType = [];
};

//描写視界のrgb変更
MK_View.prototype.PortrayalColor = function(r, g, b) {//赤, 緑, 青
    this.rgb_r = r;
    this.rgb_g = g;
    this.rgb_b = b;
};

//描写視界の不透明度変更
MK_View.prototype.PortrayalOpacity = function(opacity) {//不透明度
    this.viewOpacity = opacity;
};


//----------視界判定----------

//プレイヤーいるか判定
MK_View.prototype.ViewPlayer = function(id) {//イベントid
    const views = this.ViewsData(id);
    this.ViewsPortrayal(views, id);
    return views.some(function(view) {
        return (!view.impossible && view.x == $gamePlayer._x && view.y == $gamePlayer._y) ? true : false;
    });
};

//描写する視界データ追加
MK_View.prototype.ViewsPortrayal = function(views, id) {//視界データ, イベントid
    let position = [];
    const event = $gameMap.event(id);
    const realX = event._realX - event._x;
    const realY = event._realY - event._y;
    for(let r = 0; r < views.length; r++){
        position.push({x : views[r].x + realX, y : views[r].y + realY});
    }
    this.viewsPortrayal.push({
        position : position,
        rgb_r : this.rgb_r,
        rgb_g : this.rgb_g,
        rgb_b : this.rgb_b
    });
};


//----------視界範囲データ----------

//判定する視界データを作る
MK_View.prototype.ViewsData = function(id) {//イベントid
    let views = [];
    const event = $gameMap.event(id);
    const direction = event._direction;
    const direction_x = (direction == 6) ? 1 : (direction == 4) ? -1 : 0;
    const direction_y = (direction == 2) ? 1 : (direction == 8) ? -1 : 0;
    const x = event._x + direction_x;
    const y = event._y + direction_y;
    
    views = this.ViewsLookData(x, y, direction);
    if(views.length > 0){
        const sideways_d1 = (direction == 2 || direction == 8) ? 4 : 2;
        const sideways_d2 = (direction == 2 || direction == 8) ? 6 : 8;
        views = views.concat(this.ViewsLookSidewaysData(x, y, direction, sideways_d1));
        views = views.concat(this.ViewsLookSidewaysData(x, y, direction, sideways_d2));
    }

    return views;
};

//視界の見える範囲のデータ
MK_View.prototype.ViewsLookData = function(start_x, start_y, direction) {//前方1マスのx, y, 向き
    let views = [];
    const direction_x = (direction == 6) ? 1 : (direction == 4) ? -1 : 0;
    const direction_y = (direction == 2) ? 1 : (direction == 8) ? -1 : 0;
    let x = 0;
    let y = 0;
    
    for(let r = 0; r < this.forward; r++){
        x = start_x + direction_x * r;
        y = start_y + direction_y * r;
        if(this.ViewsLookCheck(x, y, direction)){
            views.push(this.ViewsLookDataCreate(x, y));
        }else{
            break;
        }
    }
    
    return views;
};

//横方の視界の見える範囲のデータ
MK_View.prototype.ViewsLookSidewaysData = function(start_x, start_y, event_d, sideways_d) {//前方1マスのx, y, 向き, 幅の伸びる向き
    let views = [];
    const direction_x = (sideways_d == 6) ? 1 : (sideways_d == 4) ? -1 : 0;
    const direction_y = (sideways_d == 2) ? 1 : (sideways_d == 8) ? -1 : 0;
    const direction_r = (sideways_d == 2) ? 8 : (sideways_d == 4) ? 6 : (sideways_d == 6) ? 4 : 2;
    let x = 0;
    let y = 0;
    
    for(var r = 1; r < this.sideways; r++){
        x = start_x + direction_x * r;
        y = start_y + direction_y * r;
        if(this.ViewsLookCheck(x, y, direction_r) && this.ViewsLookCheck(x - direction_x, y - direction_y, sideways_d)){
            views = views.concat(this.ViewsLookData(x, y, event_d));
        }else{
            break;
        }
    }
    
    return views;
};

//視界範囲データ作成
MK_View.prototype.ViewsLookDataCreate = function(x, y) {//x, y
    return {x : x, y : y, impossible : this.ViewsImpossibleTerrainCheck(x, y)};
};

//視界見えるか判定
MK_View.prototype.ViewsLookCheck = function(x, y, direction) {//x, y, 向き
    if(!this.ViewsLookObstructEventCheck(x, y)){
        const tag = $gameMap.terrainTag(x, y);
        if(this.ignoreTerrain.indexOf(tag) != -1 || $gameMap.isPassable(x,y,direction)){
            return true;
        }
    }
    return false;
};

//イベントで視界が遮られてるか判定
MK_View.prototype.ViewsLookObstructEventCheck = function(x, y) {//x, y
    let type = "";
    const events = $gameMap.eventsXy(x, y);
    return t = events.some(event => {
        type = $dataMap.events[event._eventId].meta["mk_view"];
        if($mk_view.obstructEventType.indexOf(type) != -1){
            return true;
        }else{
            return false;
        }
    });
};

//見つからない地形か判定
MK_View.prototype.ViewsImpossibleTerrainCheck = function(x, y) {//x,y
    const tag = $gameMap.terrainTag(x, y);
    if(this.impossibleTerrain.indexOf(tag) != -1){
        return true;
    }
    return false;
};


/////////////////視界スプライト/////////////////

//視界スプライト
function MK_View_Sprite() {
    this.mk_tileSize = 48;
    
    this.initialize(...arguments);
}

MK_View_Sprite.prototype = Object.create(Sprite.prototype);
MK_View_Sprite.prototype.constructor = MK_View_Sprite;

MK_View_Sprite.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.bitmap = new Bitmap(Graphics.boxWidth + this.mk_tileSize, Graphics.boxHeight + this.mk_tileSize);
};

MK_View_Sprite.prototype.update = function() {
    this.bitmap.clear();
    this.opacity = $mk_view.viewOpacity;
    let view = null;
    let x = 0;
    let y = 0;
    for(let r1 = 0; r1 < $mk_view.viewsPortrayal.length; r1++){
        view = $mk_view.viewsPortrayal[r1];
        for(let r2 = 0; r2 < view.position.length; r2++){
            x = view.position[r2].x * this.mk_tileSize - $gameMap._displayX * this.mk_tileSize;
            y = view.position[r2].y * this.mk_tileSize - $gameMap._displayY * this.mk_tileSize;
            this.bitmap.fillRect(x, y, this.mk_tileSize, this.mk_tileSize, "rgb("+ view.rgb_r + "," + view.rgb_g +  "," + view.rgb_b + ")");
        }
    }
    $mk_view.viewsPortrayal = [];
};


(() => {
    
    /////////////////プラグイン関係/////////////////
    
    const pluginName = "MK_View";

    //----------プラグインコマンド----------
    
    //指定した視界の範囲変更
    PluginManager.registerCommand(pluginName, "viewRange", args => {
        $mk_view.ViewRange(Number(args.forward), Number(args.sideways));
    });
    
    //無視する地形タグ追加
    PluginManager.registerCommand(pluginName, "ignoreTerrainAdd", args => {
        $mk_view.IgnoreTerrainAdd(Number(args.terrain));
    });
    
    //無視する地形タグ外す
    PluginManager.registerCommand(pluginName, "ignoreTerrainRemove", args => {
        $mk_view.IgnoreTerrainRemove(Number(args.terrain));
    });
    
    //無視する地形タグ全て外す
    PluginManager.registerCommand(pluginName, "ignoreTerrainAllRemove", args => {
        $mk_view.IgnoreTerrainAllRemove();
    });
    
    //見つからない地形タグ追加
    PluginManager.registerCommand(pluginName, "impossibleTerrainAdd", args => {
        $mk_view.ImpossibleTerrainAdd(Number(args.terrain));
    });
    
    //見つからない地形タグ外す
    PluginManager.registerCommand(pluginName, "impossibleTerrainRemove", args => {
        $mk_view.ImpossibleTerrainRemove(Number(args.terrain));
    });
    
    //見つからない地形タグ全て外す
    PluginManager.registerCommand(pluginName, "impossibleTerrainAllRemove", args => {
        $mk_view.ImpossibleTerrainAllRemove();
    });
    
    //遮るイベントの種類追加
    PluginManager.registerCommand(pluginName, "obstructEventTypeAdd", args => {
        $mk_view.ObstructEventTypeAdd(args.type);
    });
    
    //遮るイベントの種類外す
    PluginManager.registerCommand(pluginName, "obstructEventTypeRemove", args => {
        $mk_view.ObstructEventTypeRemove(args.type);
    });
    
    //遮るイベントの種類全て外す
    PluginManager.registerCommand(pluginName, "obstructEventTypeAllRemove", args => {
        $mk_view.ObstructEventTypeAllRemove();
    });
    
    //視界の色を変更
    PluginManager.registerCommand(pluginName, "portrayalColor", args => {
        $mk_view.PortrayalColor(Number(args.r), Number(args.g), Number(args.b));
    });
    
    //視界の不透明度変更
    PluginManager.registerCommand(pluginName, "portrayalOpacity", args => {
        $mk_view.PortrayalOpacity(Number(args.opacity));
    });
    
    /////////////////元々オブジェクト/////////////
    
    //----------スプライト関係----------
    
    //スプライト作る
    const _Spriteset_Map_createWeather = Spriteset_Map.prototype.createWeather;
    Spriteset_Map.prototype.createWeather = function() {
        this.mk_view_sprite = new MK_View_Sprite();
        this.addChild(this.mk_view_sprite);
        _Spriteset_Map_createWeather.apply(this, arguments);
    };
    
    //----------セーブ関係----------

    //ゲーム開始時作成データ
    const _DataManager_createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function() {
        _DataManager_createGameObjects.apply(this, arguments);
        $mk_view = new MK_View();
    };
    
    //セーブ
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function() {
        let contents = _DataManager_makeSaveContents.apply(this, arguments);
        contents.mk_view = $mk_view;
        return contents;
    };
    
    //ロード
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.apply(this, arguments);
        if(!!contents.mk_view){
            $mk_view = contents.mk_view;
        }
    };
    
})();
