//=============================================================================
// TYT_BlackWhite.js
//=============================================================================
// Copyright 2024 帝国妖異対策局
// Released under the MIT License.
// https://opensource.org/licenses/MIT
// Version: 
// 2024/09/12 初版
// 2024/09/14 名称変更
// Twitter: @tytinfo
// Blog: https://tyt.fanbox.cc/
//=============================================================================
/*:
 * @plugindesc ブラックホワイトゲームのメインプラグイン
 * @author 帝国妖異対策局
 * @target MZ
 * @base TYT_BlackWhite_Config
 * @orderAfter TYT_BlackWhite_Config
 * @orderAfter TYT_BlackWhite_Logic
 * @orderAfter TYT_BlackWhite_AI
 *
 * @param DefaultWhiteStoneImage
 * @text デフォルト白石の画像
 * @type file
 * @dir img/pictures
 * @desc デフォルトの白石として使用する画像ファイル
 *
 * @param DefaultBlackStoneImage
 * @text デフォルト黒石の画像
 * @type file
 * @dir img/pictures
 * @desc デフォルトの黒石として使用する画像ファイル
 * 
 * @param PlaceStoneSE
 * @text 石を置く効果音
 * @type file
 * @dir audio/se
 * @desc 石を置いたときに再生する効果音
 *
 * @param GameStartSE
 * @text ゲーム開始効果音
 * @type file
 * @dir audio/se
 * @desc ゲーム開始時に再生する効果音
 *
 * @param GameEndSE
 * @text ゲーム終了効果音
 * @type file
 * @dir audio/se
 * @desc ゲーム終了時に再生する効果音
 *
 * @command StartBlackWhiteGame
 * @text ブラックホワイトゲームを開始
 * @desc ブラックホワイトゲームを開始します。
 * 
 * @arg WhiteStoneImage
 * @text 白石の画像
 * @type file
 * @dir img/pictures
 * @desc このゲームで使用する白石の画像ファイル。指定しない場合はデフォルト画像を使用。
 *
 * @arg BlackStoneImage
 * @text 黒石の画像
 * @type file
 * @dir img/pictures
 * @desc このゲームで使用する黒石の画像ファイル。指定しない場合はデフォルト画像を使用。
 * 
 */

(() => {
    const pluginName = "TYT_BlackWhite";
    const parameters = PluginManager.parameters(pluginName);
    const DefaultWhiteStoneImage = parameters['DefaultWhiteStoneImage'] || '';
    const DefaultBlackStoneImage = parameters['DefaultBlackStoneImage'] || '';
    const WhiteStoneImage = parameters['WhiteStoneImage'] || '';
    const BlackStoneImage = parameters['BlackStoneImage'] || '';
    const PlaceStoneSE = parameters['PlaceStoneSE'] || '';
    const GameStartSE = parameters['GameStartSE'] || '';
    const GameEndSE = parameters['GameEndSE'] || '';
    let currentWhiteStoneImage = DefaultWhiteStoneImage;
    let currentBlackStoneImage = DefaultBlackStoneImage;

    let puzzleActive = false;

    // 右クリックイベントリスナーを更新
    document.addEventListener('contextmenu', (event) => {
        if (puzzleActive) {
            event.preventDefault();
            if (SceneManager._scene instanceof Scene_BlackWhite) {
                SceneManager._scene.onRightButtonDown();
            }
        }
    });

    class Sprite_BlackWhiteBoard extends Sprite {
        constructor() {
            super();
            this.initialize(...arguments);
        }

        initialize() {
            super.initialize();
            this.createBoard();
            this.createStones();
        }

        createBoard() {
            const bitmap = new Bitmap(410, 410);
            bitmap.fillAll('black');
            bitmap.fillRect(4, 4, 402, 402, 'green');
            for (let i = 0; i <= 8; i++) {
                const pos = i * 50 + 4;
                bitmap.fillRect(4, pos, 402, 2, 'black');
                bitmap.fillRect(pos, 4, 2, 402, 'black');
            }
            bitmap.fillRect(406, 4, 4, 406, 'black');
            bitmap.fillRect(4, 406, 406, 4, 'black');
            this.bitmap = bitmap;
        }

        createStones() {
            this._stones = [];
            for (let y = 0; y < 8; y++) {
                this._stones[y] = [];
                for (let x = 0; x < 8; x++) {
                    const stone = new Sprite_BlackWhiteStone();
                    stone.x = x * 50 + 29;
                    stone.y = y * 50 + 29;
                    this.addChild(stone);
                    this._stones[y][x] = stone;
                }
            }
        }

        updateBoard(gameState) {
            for (let y = 0; y < 8; y++) {
                for (let x = 0; x < 8; x++) {
                    this._stones[y][x].setType(gameState.board[y][x]);
                }
            }
        }

        onBoardTouch(x, y) {
            const boardX = Math.floor(x / 50);
            const boardY = Math.floor(y / 50);
            if (boardX >= 0 && boardX < 8 && boardY >= 0 && boardY < 8) {
                return { x: boardX, y: boardY };
            }
            return null;
        }

        setTheme(theme) {
            // テーマに応じてボードの見た目を変更する処理を実装
            // 例: this.bitmap.fillRect(4, 4, 402, 402, theme === 'classic' ? 'green' : 'blue');
        }
    }

    class Sprite_BlackWhiteStone extends Sprite {
        constructor() {
            super();
            this.initialize(...arguments);
        }

        initialize() {
            super.initialize();
            this._type = 'none';
            this.bitmap = new Bitmap(45, 45);
            this.anchor.x = 0.5;
            this.anchor.y = 0.5;
        }

        setType(type) {
            if (this._type !== type) {
                this._type = type;
                this.refresh();
            }
        }

        refresh() {
            this.bitmap.clear();
            if (this._type === 'white' || this._type === 'black') {
                const imageName = this._type === 'white' ? currentWhiteStoneImage : currentBlackStoneImage;
                const bitmap = ImageManager.loadPicture(imageName);
                bitmap.addLoadListener(() => {
                    this.bitmap = this.resizeImage(bitmap, 45, 45);
                });
            }
        }

        resizeImage(originalBitmap, width, height) {
            const resizedBitmap = new Bitmap(width, height);
            resizedBitmap.blt(originalBitmap, 0, 0, originalBitmap.width, originalBitmap.height, 0, 0, width, height);
            return resizedBitmap;
        }
    }

    class Sprite_TurnIndicator extends Sprite {
        constructor() {
            super();
            this._turn = '';
            this._gameState = null;
            this.createBitmap();
        }
    
        createBitmap() {
            this.bitmap = new Bitmap(300, 50);
            this.bitmap.fontSize = 24;
        }
    
        update(gameState) {
            if (!gameState) return;  // gameState が undefined の場合は処理をスキップ
    
            this._gameState = gameState;
            if (this._turn !== gameState.currentPlayer || gameState.gameOver) {
                this._turn = gameState.currentPlayer;
                this.redraw();
            }
        }
    
        redraw() {
            this.bitmap.clear();
            let text;
            if (this._gameState.gameOver) {
                const blackCount = this._gameState.blackCount;
                const whiteCount = this._gameState.whiteCount;
                if (blackCount > whiteCount) {
                    text = "黒の勝ち";
                } else if (whiteCount > blackCount) {
                    text = "白の勝ち";
                } else {
                    text = "引き分け";
                }
                text += ` (黒:${blackCount} 白:${whiteCount})`;
            } else {
                text = this._turn === 'black' ? '黒の番です' : '白の番です';
            }
            this.bitmap.drawText(text, 0, 0, 300, 50, 'center');
        }
    }

    class Sprite_ScoreDisplay extends Sprite {
        constructor() {
            super();
            this._blackScore = 0;
            this._whiteScore = 0;
            this.createBitmap();
        }

        createBitmap() {
            this.bitmap = new Bitmap(200, 50);
            this.bitmap.fontSize = 24;
        }

        update(gameState) {
            if (gameState && gameState.blackCount != null && gameState.whiteCount != null) {
                const blackScore = gameState.blackCount;
                const whiteScore = gameState.whiteCount;
                if (this._blackScore !== blackScore || this._whiteScore !== whiteScore) {
                    this._blackScore = blackScore;
                    this._whiteScore = whiteScore;
                    this.redraw();
                }
            }
        }

        redraw() {
            this.bitmap.clear();
            const text = `黒: ${this._blackScore}  白: ${this._whiteScore}`;
            this.bitmap.drawText(text, 0, 0, 200, 50, 'center');
        }
    }

    class Scene_BlackWhite extends Scene_Base {
        constructor() {
            super();
            this.initialize(...arguments);
        }

        initialize() {
            super.initialize();
            this._gameLogic = new Game_BlackWhiteLogic();
            this._aiPlayer = new Game_BlackWhiteAI();
            this._aiThinking = false;
            this._lastMoveApplied = false;
        }

        create() {
            super.create();
            this.createWindowLayer();  // ここで _windowLayer を作成
            this.createBackground();
            this.createBlackWhiteBoard();
            this.createUIElements();
            this.applyConfigSettings();
            this.playGameStartSound();
            puzzleActive = true;
            this._gameEnded = false; // ゲーム終了フラグを追加
        }

        createBackground() {
            this._backgroundSprite = new Sprite();
            this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
            this.addChild(this._backgroundSprite);
        }

        createBlackWhiteBoard() {
            this._BlackWhiteBoard = new Sprite_BlackWhiteBoard();
            this._BlackWhiteBoard.x = (Graphics.width - 410) / 2;
            this._BlackWhiteBoard.y = (Graphics.height - 410) / 2;
            this.addChild(this._BlackWhiteBoard);
        }

        createUIElements() {
            this._turnIndicator = new Sprite_TurnIndicator();
            this._turnIndicator.x = 20;
            this._turnIndicator.y = 20;
            this.addChild(this._turnIndicator);

            this._scoreDisplay = new Sprite_ScoreDisplay();
            this._scoreDisplay.x = Graphics.width - 220;
            this._scoreDisplay.y = 20;
            this.addChild(this._scoreDisplay);
        }

        update() {
            super.update();
            if (this._gameEnded && (Input.isTriggered('cancel') || TouchInput.isCancelled())) {
                this.onResultOk();
            } else {
                this.updateGame();
            }
        }

        updateGame() {
            const gameState = this._gameLogic.getGameState();
            console.log("Current game state:", gameState);  // デバッグログを追加
        
            if (!gameState) {
                console.error("Game state is undefined");
                return;
            }
        
            this._BlackWhiteBoard.updateBoard(gameState);
            this._turnIndicator.update(gameState);
            this._scoreDisplay.update(gameState);
    
            if (this._gameLogic.isGameOver()) {
                if (!this._gameEnded) {
                    this.endBlackWhiteGame();
                    this._gameEnded = true;
                }
                return;
            }
    
            if (!this._aiThinking) {
                if (this._gameLogic.currentPlayer === 'black') {
                    this.updatePlayerTurn();
                } else {
                    this.updateAITurn();
                }
            }
        }

        updatePlayerTurn() {
            if (this._gameLogic.getValidMoves('black').length === 0) {
                console.log("Player has no valid moves. Passing...");
                this._gameLogic.pass();
                this.updateGame(); // 即座に次のターンに移行
            } else if (TouchInput.isTriggered()) {
                const touchPos = this._BlackWhiteBoard.onBoardTouch(
                    TouchInput.x - this._BlackWhiteBoard.x,
                    TouchInput.y - this._BlackWhiteBoard.y
                );
                if (touchPos) {
                    const isValidMove = this._gameLogic.makeMove(touchPos.x, touchPos.y);
                    if (isValidMove) {
                        this.playPlaceStoneSound();
                        this.updateGame(); // 即座に次のターンに移行
                    }
                }
            }
        }

        updateAITurn() {
            this._aiThinking = true;
            const validMoves = this._gameLogic.getValidMoves('white');
            if (validMoves.length === 0) {
                console.log("AI has no valid moves. Passing...");
                this._gameLogic.pass();
                this._aiThinking = false;
                this.updateGame(); // 即座に次のターンに移行
            } else {
                this._aiPlayer.decideMoveWithDelay(this._gameLogic.getGameState())
                    .then(aiMove => {
                        if (aiMove) {
                            const isValidMove = this._gameLogic.makeMove(aiMove.x, aiMove.y);
                            if (isValidMove) {
                                this.playPlaceStoneSound();
                            }
                        }
                        this._aiThinking = false;
                        this.updateGame(); // 即座に次のターンに移行
                    });
            }
        }

        playPlaceStoneSound() {
            if (PlaceStoneSE) {
                AudioManager.playSe({ name: PlaceStoneSE, pan: 0, pitch: 100, volume: 90 });
            }
        }

        playGameStartSound() {
            if (GameStartSE) {
                AudioManager.playSe({ name: GameStartSE, pan: 0, pitch: 100, volume: 90 });
            }
        }

        playGameEndSound() {
            if (GameEndSE) {
                AudioManager.playSe({ name: GameEndSE, pan: 0, pitch: 100, volume: 90 });
            }
        }

        endBlackWhiteGame() {
            if (this._gameEnded) return; // 既にゲームが終了している場合は処理しない
            this._gameEnded = true;
            this.playGameEndSound();
            const gameState = this._gameLogic.getGameState();
            console.log("Game ended. Showing result...");
            this.showGameResult(gameState);
            puzzleActive = false;
        }

        showGameResult(gameState) {
            console.log("Showing game result with state:", gameState);
            if (!gameState || typeof gameState.blackCount === 'undefined' || typeof gameState.whiteCount === 'undefined') {
                console.error('Invalid gameState in showGameResult:', gameState);
                return;
            }
            if (!this._resultWindow) {
                const rect = new Rectangle(100, 100, 400, 200);
                this._resultWindow = new Window_BlackWhiteResult(rect);
                this._resultWindow.setGameState(gameState);
                this.addWindow(this._resultWindow);
                this._resultWindow.setHandler('ok', this.onResultOk.bind(this));
                this._resultWindow.activate();
            }
        }

        onResultOk() {
            if (this._resultWindow) {
                this._windowLayer.removeChild(this._resultWindow);
                this._resultWindow = null;
            }
            SceneManager.pop();
        }

        // 右クリック処理を Scene_BlackWhite クラス内に移動
        onRightButtonDown() {
            if (!this._gameEnded) {
                this.endBlackWhiteGame();
            } else {
                this.onResultOk();
            }
        }

        applyConfigSettings() {
            const difficulty = ConfigManager.BlackWhiteDifficulty;
            const aiThinkTime = ConfigManager.BlackWhiteAIThinkTime;
            const boardTheme = ConfigManager.BlackWhiteBoardTheme;

            if (this._aiPlayer) {
                this._aiPlayer.setDifficulty(difficulty);
                this._aiPlayer.setThinkTime(aiThinkTime);
            }
            if (this._BlackWhiteBoard) {
                this._BlackWhiteBoard.setTheme(boardTheme);
            }
        }
    }

    // Window_BlackWhiteResult クラスを以下のように修正します
    class Window_BlackWhiteResult extends Window_Command {
        constructor(rect) {
            super(rect);
            this._gameState = null;
        }

        setGameState(gameState) {
            this._gameState = gameState;
            this.refresh();
        }

        makeCommandList() {
            this.addCommand("OK", 'ok');
        }

        drawAllItems() {
            super.drawAllItems();
            if (this._gameState) {
                this.drawContents();
            }
        }

        drawContents() {
            if (!this._gameState) {
                console.error('_gameState is undefined in Window_BlackWhiteResult');
                return;
            }
            const rect = this.itemLineRect(0);
            this.resetTextColor();
            this.drawText("ゲーム終了", rect.x, rect.y, rect.width, 'center');
            const y = rect.y + this.lineHeight();
            this.drawText(`黒: ${this._gameState.blackCount}`, rect.x, y, rect.width / 2, 'center');
            this.drawText(`白: ${this._gameState.whiteCount}`, rect.x + rect.width / 2, y, rect.width / 2, 'center');
            const winner = this.getWinnerText();
            this.drawText(winner, rect.x, y + this.lineHeight(), rect.width, 'center');
        }

        getWinnerText() {
            if (this._gameState.blackCount > this._gameState.whiteCount) {
                return "黒の勝ち";
            } else if (this._gameState.blackCount < this._gameState.whiteCount) {
                return "白の勝ち";
            } else {
                return "引き分け";
            }
        }
    }

     // プラグインコマンドの登録
     PluginManager.registerCommand(pluginName, "StartBlackWhiteGame", args => {
        currentWhiteStoneImage = args.WhiteStoneImage || DefaultWhiteStoneImage;
        currentBlackStoneImage = args.BlackStoneImage || DefaultBlackStoneImage;
        SceneManager.push(Scene_BlackWhite);
    });

    // ハンバーガーメニューの表示/非表示を制御
    const _Scene_Base_createWindowLayer = Scene_Base.prototype.createWindowLayer;
    Scene_Base.prototype.createWindowLayer = function () {
        _Scene_Base_createWindowLayer.call(this);
        this.updateMenuButtonVisibility();
    };

    Scene_Base.prototype.updateMenuButtonVisibility = function () {
        if (this._menuButton) {
            this._menuButton.visible = !(SceneManager._scene instanceof Scene_BlackWhite);
        }
    };

    // シーン変更時にメニューボタンの表示/非表示を更新
    const _SceneManager_onSceneStart = SceneManager.onSceneStart;
    SceneManager.onSceneStart = function () {
        _SceneManager_onSceneStart.call(this);
        if (this._scene) {
            this._scene.updateMenuButtonVisibility();
        }
    };

    // TouchInput の右クリック処理を更新
    const _TouchInput_onRightButtonDown = TouchInput._onRightButtonDown;
    TouchInput._onRightButtonDown = function (event) {
        _TouchInput_onRightButtonDown.call(this, event);
        if (SceneManager._scene instanceof Scene_BlackWhite) {
            SceneManager._scene.onRightButtonDown();
        }
    };

    // グローバルスコープにクラスを公開
    window.Scene_BlackWhite = Scene_BlackWhite;
    window.Sprite_BlackWhiteBoard = Sprite_BlackWhiteBoard;
    window.Sprite_BlackWhiteStone = Sprite_BlackWhiteStone;

    // Spriteクラスにクリックリスナーを追加
    Sprite.prototype.setClickListener = function (listener) {
        this.interactive = true;
        this.buttonMode = true;
        // 既存のリスナーを削除
        this.removeAllListeners('pointerdown');
        if (listener) {
            this.on('pointerdown', () => {
                listener();
            });
        }
    };

    // Scene_Map.prototype.isRightButtonReserved をオーバーライド
    const _Scene_Map_isRightButtonReserved = Scene_Map.prototype.isRightButtonReserved;
    Scene_Map.prototype.isRightButtonReserved = function () {
        if (puzzleActive) {
            return true; // パズル中は右クリックを予約済みとする
        }
        return _Scene_Map_isRightButtonReserved.call(this);
    };

    // Scene_Map.prototype.createメソッドをオーバーライド
    const _Scene_Map_create = Scene_Map.prototype.create;
    Scene_Map.prototype.create = function () {
        _Scene_Map_create.call(this);
        if (puzzleActive) {
            this.hideMenuButton();
        }
    };

    // Scene_Map.prototype.update メソッドをオーバーライド
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        if (puzzleActive) {
            this.hideMenuButton();
        }
    };

    const _Scene_Map_processMapTouch = Scene_Map.prototype.processMapTouch;
    Scene_Map.prototype.processMapTouch = function () {
        if (puzzleActive) {
            return;
        }
        _Scene_Map_processMapTouch.call(this);
    };

    const _Game_Player_moveByInput = Game_Player.prototype.moveByInput;
    Game_Player.prototype.moveByInput = function () {
        if (puzzleActive) {
            return;
        }
        _Game_Player_moveByInput.call(this);
    };

    const _Scene_Map_isAnyButtonPressed = Scene_Map.prototype.isAnyButtonPressed;
    Scene_Map.prototype.isAnyButtonPressed = function () {
        if (puzzleActive) {
            return true;
        }
        return _Scene_Map_isAnyButtonPressed.call(this);
    };

    console.log(`Plugin ${pluginName} loaded`);
    console.log(`Debug mode: ${Utils.isOptionValid('test')}`);
    console.log('Plugin parameters:', parameters);
})();