/*:
 * @plugindesc ブラックホワイトゲームのAIプラグイン
 * @author 帝国妖異対策局
 * @target MZ
 */
class Game_BlackWhiteAI {
    constructor() {
        this.initialize();
    }

    initialize() {
        this._difficulty = 1; // デフォルトは中級
        this._thinkTime = 3; // デフォルトは3秒
    }

    setDifficulty(difficulty) {
        this._difficulty = difficulty;
    }

    setThinkTime(thinkTime) {
        this._thinkTime = thinkTime;
    }

    decideMoveWithDelay(gameState) {
        return new Promise(resolve => {
            setTimeout(() => {
                const move = this.decideMove(gameState);
                resolve(move);
            }, this._thinkTime * 1000);
        });
    }

    decideMove(gameState) {
        const depth = this.getDifficultyDepth();
        const gameLogic = new Game_BlackWhiteLogic();
        gameLogic.board = gameState.board;
        gameLogic.currentPlayer = gameState.currentPlayer;
        const validMoves = gameLogic.getValidMoves(gameState.currentPlayer);
    
        if (validMoves.length === 0) {
            return null;
        }
    
        // 最後の1手の場合は、深さを考慮せずに最善の手を選択
        if (gameLogic.isBoardAlmostFull()) {
            return this.getBestMoveForLastTurn(gameState, validMoves);
        }
    
        let bestMove = null;
        let bestScore = -Infinity;
    
        for (const move of validMoves) {
            const newState = this.makeMove(gameState, move);
            const score = this.minimax(newState, depth - 1, true, -Infinity, Infinity);
    
            if (score > bestScore) {
                bestScore = score;
                bestMove = move;
            }
        }
    
        return bestMove;
    }

    isLastMove(gameState) {
        const emptyCount = gameState.board.flat().filter(cell => cell === 'empty').length;
        return emptyCount === 1;
    }
    
    getBestMoveForLastTurn(gameState, validMoves) {
        let bestMove = null;
        let bestScore = -Infinity;
    
        for (const move of validMoves) {
            const newState = this.makeMove(gameState, move);
            const score = this.evaluateBoard(newState);
    
            if (score > bestScore) {
                bestScore = score;
                bestMove = move;
            }
        }
    
        return bestMove;
    }
    
   

    minimax(gameState, depth, isMaximizing, alpha, beta) {
        const gameLogic = new Game_BlackWhiteLogic();
        gameLogic.board = gameState.board;
        gameLogic.currentPlayer = gameState.currentPlayer;

        if (depth === 0 || gameLogic.isGameOver()) {
            return this.evaluateBoard(gameState);
        }

        const validMoves = gameLogic.getValidMoves();

        if (validMoves.length === 0) {
            // パスの場合
            const newState = this.pass(gameState);
            return this.minimax(newState, depth - 1, !isMaximizing, alpha, beta);
        }

        let bestScore = isMaximizing ? -Infinity : Infinity;

        for (const move of validMoves) {
            const newState = this.makeMove(gameState, move);
            const score = this.minimax(newState, depth - 1, !isMaximizing, alpha, beta);

            if (isMaximizing) {
                bestScore = Math.max(bestScore, score);
                alpha = Math.max(alpha, bestScore);
            } else {
                bestScore = Math.min(bestScore, score);
                beta = Math.min(beta, bestScore);
            }

            if (beta <= alpha) {
                break; // アルファベータ枝刈り
            }
        }

        return bestScore;
    }

    makeMove(gameState, move) {
        const gameLogic = new Game_BlackWhiteLogic();
        gameLogic.board = JSON.parse(JSON.stringify(gameState.board));
        gameLogic.currentPlayer = gameState.currentPlayer;
        gameLogic.makeMove(move.x, move.y);
        return gameLogic.getGameState();
    }

    pass(gameState) {
        const gameLogic = new Game_BlackWhiteLogic();
        gameLogic.board = JSON.parse(JSON.stringify(gameState.board));
        gameLogic.currentPlayer = gameState.currentPlayer === 'black' ? 'white' : 'black';
        return gameLogic.getGameState();
    }

    evaluateBoard(gameState) {
        const weights = [
            [100, -20, 10, 5, 5, 10, -20, 100],
            [-20, -50, -2, -2, -2, -2, -50, -20],
            [10, -2, -1, -1, -1, -1, -2, 10],
            [5, -2, -1, -1, -1, -1, -2, 5],
            [5, -2, -1, -1, -1, -1, -2, 5],
            [10, -2, -1, -1, -1, -1, -2, 10],
            [-20, -50, -2, -2, -2, -2, -50, -20],
            [100, -20, 10, 5, 5, 10, -20, 100]
        ];

        let score = 0;
        let ownStones = 0;
        let opponentStones = 0;

        for (let y = 0; y < 8; y++) {
            for (let x = 0; x < 8; x++) {
                if (gameState.board[y][x] === 'white') {
                    score += weights[y][x];
                    ownStones++;
                } else if (gameState.board[y][x] === 'black') {
                    score -= weights[y][x];
                    opponentStones++;
                }
            }
        }

        // 終盤では石の数を重視
        if (ownStones + opponentStones > 50) {
            score += (ownStones - opponentStones) * 10;
        }

        return score;
    }

    getDifficultyDepth() {
        switch (this._difficulty) {
            case 0: return 2; // 初級
            case 1: return 3; // 中級
            case 2: return 4; // 上級
            default: return 3;
        }
    }
}