"use strict";

/*:
 * @plugindesc ブラックホワイトゲームの設定管理プラグイン
 * @author 帝国妖異対策局
 * @target MZ
 *
 * @param DefaultDifficulty
 * @text デフォルト難易度
 * @type select
 * @option 初級
 * @value 0
 * @option 中級
 * @value 1
 * @option 上級
 * @value 2
 * @default 1
 *
 * @param DefaultAIThinkTime
 * @text デフォルトAI思考時間
 * @type number
 * @min 1
 * @max 10
 * @default 3
 *
 * @param DefaultBoardTheme
 * @text デフォルトボードテーマ
 * @type select
 * @option クラシック
 * @value classic
 * @option モダン
 * @value modern
 * @default classic
 *
 * @command OpenBlackWhiteConfig
 * @text ブラックホワイト設定画面を開く
 * @desc ブラックホワイトゲームの設定画面を開きます。
 */

(() => {
    const pluginName = "TYT_BlackWhite_Config";
    const parameters = PluginManager.parameters(pluginName);

    const Config = {
        defaultDifficulty: Number(parameters['DefaultDifficulty'] || 1),
        defaultAIThinkTime: Number(parameters['DefaultAIThinkTime'] || 3),
        defaultBoardTheme: parameters['DefaultBoardTheme'] || 'classic',
    };

    // ConfigManager拡張
    ConfigManager.BlackWhiteDifficulty = Config.defaultDifficulty;
    ConfigManager.BlackWhiteAIThinkTime = Config.defaultAIThinkTime;
    ConfigManager.BlackWhiteBoardTheme = Config.defaultBoardTheme;

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function() {
        const config = _ConfigManager_makeData.call(this);
        config.BlackWhiteDifficulty = this.BlackWhiteDifficulty;
        config.BlackWhiteAIThinkTime = this.BlackWhiteAIThinkTime;
        config.BlackWhiteBoardTheme = this.BlackWhiteBoardTheme;
        return config;
    };

    const _ConfigManager_applyData = ConfigManager.applyData;
    ConfigManager.applyData = function(config) {
        _ConfigManager_applyData.call(this, config);
        this.BlackWhiteDifficulty = this.readBlackWhiteConfig(config, 'BlackWhiteDifficulty', Config.defaultDifficulty);
        this.BlackWhiteAIThinkTime = this.readBlackWhiteConfig(config, 'BlackWhiteAIThinkTime', Config.defaultAIThinkTime);
        this.BlackWhiteBoardTheme = this.readBlackWhiteConfig(config, 'BlackWhiteBoardTheme', Config.defaultBoardTheme);
    };

    ConfigManager.readBlackWhiteConfig = function(config, name, defaultValue) {
        return config[name] != null ? config[name] : defaultValue;
    };

    // ブラックホワイト設定画面
    class Scene_BlackWhiteConfig extends Scene_MenuBase {
        create() {
            super.create();
            this.createBlackWhiteConfigWindow();
        }

        createBlackWhiteConfigWindow() {
            const rect = this.configWindowRect();
            this._configWindow = new Window_BlackWhiteConfig(rect);
            this._configWindow.setHandler('cancel', this.popScene.bind(this));
            this.addWindow(this._configWindow);
        }

        configWindowRect() {
            const ww = 400;
            const wh = this.calcWindowHeight(6, true);
            const wx = (Graphics.boxWidth - ww) / 2;
            const wy = (Graphics.boxHeight - wh) / 2;
            return new Rectangle(wx, wy, ww, wh);
        }
    }

    class Window_BlackWhiteConfig extends Window_Command {
        makeCommandList() {
            this.addCommand("難易度", 'difficulty');
            this.addCommand("AI思考時間", 'aiThinkTime');
            this.addCommand("ボードテーマ", 'boardTheme');
        }

        drawItem(index) {
            const rect = this.itemLineRect(index);
            const statusWidth = 120;
            const titleWidth = rect.width - statusWidth;
            this.changeTextColor(ColorManager.normalColor());
            this.drawText(this.commandName(index), rect.x, rect.y, titleWidth, 'left');
            this.drawText(this.statusText(index), rect.x + titleWidth, rect.y, statusWidth, 'right');
        }

        statusText(index) {
            const symbol = this.commandSymbol(index);
            switch (symbol) {
                case 'difficulty':
                    return ['初級', '中級', '上級'][ConfigManager.BlackWhiteDifficulty];
                case 'aiThinkTime':
                    return ConfigManager.BlackWhiteAIThinkTime + '秒';
                case 'boardTheme':
                    return ConfigManager.BlackWhiteBoardTheme === 'classic' ? 'クラシック' : 'モダン';
            }
            return '';
        }

        processOk() {
            const index = this.index();
            const symbol = this.commandSymbol(index);
            switch (symbol) {
                case 'difficulty':
                    this.changeDifficulty();
                    break;
                case 'aiThinkTime':
                    this.changeAIThinkTime();
                    break;
                case 'boardTheme':
                    this.changeBoardTheme();
                    break;
            }
        }

        changeDifficulty() {
            ConfigManager.BlackWhiteDifficulty = (ConfigManager.BlackWhiteDifficulty + 1) % 3;
            this.redrawItem(this.findSymbol('difficulty'));
            this.playCursorSound();
        }

        changeAIThinkTime() {
            ConfigManager.BlackWhiteAIThinkTime = (ConfigManager.BlackWhiteAIThinkTime % 10) + 1;
            this.redrawItem(this.findSymbol('aiThinkTime'));
            this.playCursorSound();
        }

        changeBoardTheme() {
            ConfigManager.BlackWhiteBoardTheme = ConfigManager.BlackWhiteBoardTheme === 'classic' ? 'modern' : 'classic';
            this.redrawItem(this.findSymbol('boardTheme'));
            this.playCursorSound();
        }
    }

    // プラグインコマンドの登録
    PluginManager.registerCommand(pluginName, "OpenBlackWhiteConfig", () => {
        SceneManager.push(Scene_BlackWhiteConfig);
    });

    // グローバルスコープに公開
    window.Scene_BlackWhiteConfig = Scene_BlackWhiteConfig;
})();