/*:
 * @plugindesc ブラックホワイトゲームのロジック管理プラグイン
 * @author 帝国妖異対策局
 * @target MZ
 */

(() => {
    const pluginName = "TYT_BlackWhite_Logic";

    class Game_BlackWhiteLogic {
        constructor() {
            this.initialize();
        }

        initialize() {
            this.resetGame();
        }

        resetGame() {
            this.board = this.createEmptyBoard();
            this.board[3][3] = this.board[4][4] = 'white';
            this.board[3][4] = this.board[4][3] = 'black';
            this.currentPlayer = 'black';
            this.gameOver = false;
            this.passCount = 0;
        }

        createEmptyBoard() {
            return Array(8).fill().map(() => Array(8).fill('empty'));
        }

        getGameState() {
            return {
                board: this.board,
                currentPlayer: this.currentPlayer,
                gameOver: this.gameOver,
                blackCount: this.countStones('black'),
                whiteCount: this.countStones('white')
            };
        }

        makeMove(x, y) {
            if (this.isValidMove(x, y)) {
                this.placeStone(x, y);
                this.flipStones(x, y);
                this.passCount = 0;
                this.switchPlayer();
                this.checkGameOver();
                return true;
            }
            return false;
        }

        isValidMove(x, y, player = this.currentPlayer) {
            if (this.board[y][x] !== 'empty') return false;
            return this.getFlipDirections(x, y, player).length > 0;
        }

        getValidMoves(player = this.currentPlayer) {
            const moves = [];
            for (let y = 0; y < 8; y++) {
                for (let x = 0; x < 8; x++) {
                    if (this.isValidMove(x, y, player)) {
                        moves.push({x, y});
                    }
                }
            }
            return moves;
        }

        getFlipDirections(x, y, player = this.currentPlayer) {
            const directions = [
                [-1, -1], [0, -1], [1, -1],
                [-1,  0],          [1,  0],
                [-1,  1], [0,  1], [1,  1]
            ];
            return directions.filter(([dx, dy]) => this.isValidDirection(x, y, dx, dy, player));
        }

        isValidDirection(x, y, dx, dy, player = this.currentPlayer) {
            let cx = x + dx;
            let cy = y + dy;
            if (!this.isOnBoard(cx, cy) || this.board[cy][cx] === player) return false;
            while (this.isOnBoard(cx, cy) && this.board[cy][cx] !== 'empty') {
                if (this.board[cy][cx] === player) return true;
                cx += dx;
                cy += dy;
            }
            return false;
        }

        isOnBoard(x, y) {
            return x >= 0 && x < 8 && y >= 0 && y < 8;
        }

        isBoardAlmostFull() {
            let emptyCount = 0;
            for (let y = 0; y < 8; y++) {
                for (let x = 0; x < 8; x++) {
                    if (this.board[y][x] === 'empty') {
                        emptyCount++;
                        if (emptyCount > 1) return false;
                    }
                }
            }
            return true;
        }

        placeStone(x, y) {
            this.board[y][x] = this.currentPlayer;
        }

        flipStones(x, y) {
            this.getFlipDirections(x, y).forEach(([dx, dy]) => {
                let cx = x + dx;
                let cy = y + dy;
                while (this.board[cy][cx] !== this.currentPlayer) {
                    this.board[cy][cx] = this.currentPlayer;
                    cx += dx;
                    cy += dy;
                }
            });
        }

        switchPlayer() {
            this.currentPlayer = this.currentPlayer === 'black' ? 'white' : 'black';
        }

        pass() {
            this.passCount++;
            this.switchPlayer();
            this.checkGameOver();
        }

        hasValidMoves(player = this.currentPlayer) {
            for (let y = 0; y < 8; y++) {
                for (let x = 0; x < 8; x++) {
                    if (this.isValidMove(x, y, player)) {
                        return true;
                    }
                }
            }
            return false;
        }

        countStones(color) {
            return this.board.flat().filter(stone => stone === color).length;
        }

        isGameOver() {
            return this.gameOver;
        }

        checkGameOver() {
            if (this.isBoardFull()) {
                this.gameOver = true;
                return true;
            }

            const currentPlayerMoves = this.hasValidMoves(this.currentPlayer);
            const otherPlayerMoves = this.hasValidMoves(this.currentPlayer === 'black' ? 'white' : 'black');

            if (!currentPlayerMoves && !otherPlayerMoves) {
                this.gameOver = true;
                return true;
            }

            if (!currentPlayerMoves && otherPlayerMoves) {
                this.pass();
                return false;
            }

            return false;
        }

        isBoardFull() {
            return this.board.every(row => row.every(cell => cell !== 'empty'));
        }

        getWinner() {
            if (!this.gameOver) return null;
            const blackCount = this.countStones('black');
            const whiteCount = this.countStones('white');
            if (blackCount > whiteCount) return 'black';
            if (whiteCount > blackCount) return 'white';
            return 'draw';
        }
    }

    window.Game_BlackWhiteLogic = Game_BlackWhiteLogic;
})();