//=============================================================================
// TYT_MonsterSweeper.js
//=============================================================================
// Copyright 2024 帝国妖異対策局
// Released under the MIT License.
// https://opensource.org/licenses/MIT
// Version: 
// 2024/09/13 初版
// Twitter: @tytinfo
// Blog: https://tyt.fanbox.cc/
//=============================================================================
/*:
 * @target MZ
 * @plugindesc モンスタースイーパのミニゲームを追加します。
 *
 * @command StartMonsterSweeper
 * @text モンスタースイーパを開始
 * @desc モンスタースイーパのミニゲームを開始します。
 *
 * @arg gridSize
 * @type number
 * @min 5
 * @max 20
 * @default 10
 * @text グリッドサイズ
 * @desc グリッドの縦横のマス数（5〜20）
 *
 * @arg mineCount
 * @type number
 * @min 1
 * @default 15
 * @text モンスター数
 * @desc ゲーム内のモンスターの数
 *
 * @arg tileSize
 * @type number
 * @min 16
 * @default 40
 * @text タイルサイズ
 * @desc 各タイルの大きさ（ピクセル）
 *
 * @arg bombImage
 * @type file
 * @dir img/pictures
 * @text 爆弾画像
 * @desc 爆弾に使用する画像ファイル
 *
 * @arg boardBackground
 * @type file
 * @dir img/parallaxes
 * @text ボード背景画像
 * @desc ボードの背景に使用する画像ファイル
 *
 * @arg gameOverSE
 * @type file
 * @dir audio/se
 * @text ゲームオーバーSE
 * @desc モンスターをクリックしたときのSE
 *
 * @arg safeSE
 * @type file
 * @dir audio/se
 * @text セーフSE
 * @desc 安全地帯をクリックしたときのSE
 *
 * @arg gameClearSE
 * @type file
 * @dir audio/se
 * @text ゲームクリアSE
 * @desc ゲームクリア時のSE
 *
 * @command SetGameOverSE
 * @text ゲームオーバーSEを設定
 * @desc ゲームオーバー時のSEを設定します。
 *
 * @arg gameOverSE
 * @type file
 * @dir audio/se
 * @text ゲームオーバーSE
 * @desc モンスターをクリックしたときのSE
 *
 * @command SetSafeSE
 * @text セーフSEを設定
 * @desc 安全地帯をクリックしたときのSEを設定します。
 *
 * @arg safeSE
 * @type file
 * @dir audio/se
 * @text セーフSE
 * @desc 安全地帯をクリックしたときのSE
 *
 * @command SetGameClearSE
 * @text ゲームクリアSEを設定
 * @desc ゲームクリア時のSEを設定します。
 *
 * @arg gameClearSE
 * @type file
 * @dir audio/se
 * @text ゲームクリアSE
 * @desc ゲームクリア時のSE
 *
 * @help
 * このプラグインは、モンスタースイーパのミニゲームを追加します。
 * プラグインコマンド「モンスタースイーパを開始」を使用してゲームを起動してください。
 *
 * プラグインコマンドのパラメータでグリッドサイズ、モンスター数、タイルサイズ、
 * 爆弾画像、ボード背景画像、SEを設定できます。
 *
 * SEは以下のプラグインコマンドで個別に設定することもできます。
 * - ゲームオーバーSEを設定
 * - セーフSEを設定
 * - ゲームクリアSEを設定
 */
 
(() => {
    const pluginName = 'TYT_MonsterSweeper';

    const MonsterSweeperSEs = {
        gameOverSE: '',
        safeSE: '',
        gameClearSE: ''
    };

    PluginManager.registerCommand(pluginName, 'StartMonsterSweeper', args => {
        const gridSize = Number(args.gridSize || 10);
        const monsterCount = Number(args.mineCount || 15);
        const tileSize = Number(args.tileSize || 40);
        const bombImage = args.bombImage || '';
        const boardBackground = args.boardBackground || '';
        const gameOverSE = args.gameOverSE || MonsterSweeperSEs.gameOverSE || '';
        const safeSE = args.safeSE || MonsterSweeperSEs.safeSE || '';
        const gameClearSE = args.gameClearSE || MonsterSweeperSEs.gameClearSE || '';
        SceneManager.push(Scene_MonsterSweeper);
        SceneManager.prepareNextScene(gridSize, monsterCount, tileSize, bombImage, boardBackground, gameOverSE, safeSE, gameClearSE);
    });

    PluginManager.registerCommand(pluginName, 'SetGameOverSE', args => {
        MonsterSweeperSEs.gameOverSE = args.gameOverSE || '';
    });

    PluginManager.registerCommand(pluginName, 'SetSafeSE', args => {
        MonsterSweeperSEs.safeSE = args.safeSE || '';
    });

    PluginManager.registerCommand(pluginName, 'SetGameClearSE', args => {
        MonsterSweeperSEs.gameClearSE = args.gameClearSE || '';
    });

    class Scene_MonsterSweeper extends Scene_Base {
        prepare(gridSize, monsterCount, tileSize, bombImage, boardBackground, gameOverSE, safeSE, gameClearSE) {
            this._gridSize = gridSize;
            this._monsterCount = monsterCount;
            this._tileSize = tileSize;
            this._bombImage = bombImage;
            this._boardBackground = boardBackground;
            this._gameOverSE = gameOverSE || '';
            this._safeSE = safeSE || '';
            this._gameClearSE = gameClearSE || '';
        }

        create() {
            super.create();
            this.createBackground();
            this.loadBombImage();
            this.createGrid();
            this.createReturnButton();
            this._isGameOver = false;
            this._isGameClear = false;

            // タイマー関連の初期化
            this._startTime = null;
            this._elapsedTime = 0;
            this.createTimerDisplay();
        }

        createBackground() {
            this._backgroundSprite = new Sprite();
            if (this._boardBackground) {
                this._backgroundSprite.bitmap = ImageManager.loadParallax(this._boardBackground);
            } else {
                this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
            }
            this.addChild(this._backgroundSprite);
        }

        loadBombImage() {
            if (this._bombImage) {
                this._bombBitmap = ImageManager.loadPicture(this._bombImage);
            } else {
                this._bombBitmap = null;
            }
        }

        createGrid() {
            this._tiles = [];
            this._flags = 0;
            this._revealedTiles = 0;

            const maxMonsters = this._gridSize * this._gridSize - 1;
            if (this._monsterCount > maxMonsters) {
                this._monsterCount = maxMonsters;
            }

            for (let y = 0; y < this._gridSize; y++) {
                this._tiles[y] = [];
                for (let x = 0; x < this._gridSize; x++) {
                    const tile = new Sprite_MonsterSweeperTile(x, y, this._tileSize);
                    tile.x = x * this._tileSize + (Graphics.width - this._gridSize * this._tileSize) / 2;
                    tile.y = y * this._tileSize + (Graphics.height - this._gridSize * this._tileSize) / 2;
                    this.addChild(tile);
                    this._tiles[y][x] = tile;
                }
            }

            let monstersPlaced = 0;
            while (monstersPlaced < this._monsterCount) {
                const x = Math.floor(Math.random() * this._gridSize);
                const y = Math.floor(Math.random() * this._gridSize);
                const tile = this._tiles[y][x];
                if (!tile.isMonster) {
                    tile.isMonster = true;
                    monstersPlaced++;
                }
            }

            for (let y = 0; y < this._gridSize; y++) {
                for (let x = 0; x < this._gridSize; x++) {
                    const tile = this._tiles[y][x];
                    if (!tile.isMonster) {
                        tile.monsterCount = this.countAdjacentMonsters(x, y);
                    }
                }
            }
        }

        createReturnButton() {
            const buttonWidth = 100;
            const buttonHeight = 40;
            this._returnButton = new Sprite_Clickable();
            this._returnButton.bitmap = new Bitmap(buttonWidth, buttonHeight);
            this._returnButton.bitmap.fillAll('#444444');
            this._returnButton.bitmap.strokeRect(0, 0, buttonWidth, buttonHeight, '#FFFFFF');
            this._returnButton.bitmap.drawText('中断', 0, 0, buttonWidth, buttonHeight, 'center');
            this._returnButton.x = Graphics.width - buttonWidth - 20;
            this._returnButton.y = 20;
            this._returnButton.onClick = this.onReturn.bind(this);
            this.addChild(this._returnButton);
        }

        onReturn() {
            SceneManager.pop();
        }

        createTimerDisplay() {
            const width = 200;
            const height = this.calcWindowHeight(1, false);
            const x = (Graphics.width - width) / 2;
            const y = Graphics.height - height - 20; // 画面下から20ピクセル上
            this._timerWindow = new Window_Base(new Rectangle(x, y, width, height));
            this._timerWindow.opacity = 0; // ウィンドウの背景を透明にする場合
            this.addChild(this._timerWindow);
        }

        updateTimerDisplay() {
            this._timerWindow.contents.clear();
            const timeText = `時間: ${this._elapsedTime.toFixed(2)} 秒`;
            this._timerWindow.drawText(timeText, 0, 0, this._timerWindow.contentsWidth(), 'center');
        }

        countAdjacentMonsters(x, y) {
            let count = 0;
            for (let dx = -1; dx <= 1; dx++) {
                for (let dy = -1; dy <= 1; dy++) {
                    if (dx === 0 && dy === 0) continue;
                    const nx = x + dx;
                    const ny = y + dy;
                    if (nx >= 0 && nx < this._gridSize && ny >= 0 && ny < this._gridSize) {
                        if (this._tiles[ny][nx].isMonster) {
                            count++;
                        }
                    }
                }
            }
            return count;
        }

        gameOver() {
            this._isGameOver = true;
            this.showMessage('ゲームオーバー');

            // タイマーを停止
            this._startTime = null;

            this.revealAllTiles();
            if (this._gameOverSE) {
                AudioManager.playSe({ name: this._gameOverSE, volume: 90, pitch: 100, pan: 0 });
            }
        }

        gameClear() {
            this._isGameClear = true;

            // クリア時間を計算
            const finalTime = (Graphics.frameCount - this._startTime) / 60;
            const timeText = `クリア時間: ${finalTime.toFixed(2)} 秒`;

            this.showMessage('ゲームクリア！', timeText);

            // セルフスイッチAをオンにする
            if ($gameMap && $gameMap._interpreter) {
                const eventId = $gameMap._interpreter.eventId();
                if (eventId > 0) {
                    $gameSelfSwitches.setValue([$gameMap.mapId(), eventId, 'A'], true);
                }
            }

            if (this._gameClearSE) {
                AudioManager.playSe({ name: this._gameClearSE, volume: 90, pitch: 100, pan: 0 });
            }
        }

        revealAllTiles() {
            for (let y = 0; y < this._gridSize; y++) {
                for (let x = 0; x < this._gridSize; x++) {
                    this._tiles[y][x].reveal();
                }
            }
        }

        showMessage(text, subText) {
            const windowHeight = subText ? this.calcWindowHeight(2, false) : this.calcWindowHeight(1, false);
            const window = new Window_Base(new Rectangle(0, 0, Graphics.width, windowHeight));
            window.y = (Graphics.height - windowHeight) / 2;
            window.drawText(text, 0, 0, Graphics.width, 'center');
            if (subText) {
                window.drawText(subText, 0, window.lineHeight, Graphics.width, 'center');
            }
            this.addChild(window);
            this._messageWindow = window;
        }
        
        update() {
            super.update();
            if (Input.isTriggered('cancel')) {
                SceneManager.pop();
            }

            // ゲーム中の場合、タイマーを更新
            if (this._startTime !== null && !this._isGameOver && !this._isGameClear) {
                this._elapsedTime = (Graphics.frameCount - this._startTime) / 60; // 60FPSの場合
                this.updateTimerDisplay();
            }

            if (this._isGameOver || this._isGameClear) {
                if (Input.isTriggered('ok') || TouchInput.isTriggered()) {
                    SceneManager.pop();
                }
            }
        }

        onTileRevealed() {
            this._revealedTiles++;
            if (this._revealedTiles + this._monsterCount === this._gridSize * this._gridSize) {
                this.gameClear();
            }
        }
    }

    class Sprite_MonsterSweeperTile extends Sprite_Clickable {
        constructor(x, y, size) {
            super();
            this._coordinateX = x;
            this._coordinateY = y;
            this._size = size;
            this.isMonster = false;
            this.isRevealed = false;
            this.isFlagged = false;
            this.monsterCount = 0;
            this.createBitmap();
        }

        createBitmap() {
            this.bitmap = new Bitmap(this._size, this._size);
            this.bitmap.fillAll('#888888');
            this.drawFrame();
        }

        drawFrame() {
            this.bitmap.strokeRect(0, 0, this._size, this._size, '#000000');
        }

        onClick() {
            if (this.isRevealed || SceneManager._scene._isGameOver) return;

            // 最初のタイルをクリックしたときにタイマーを開始
            if (SceneManager._scene._startTime === null) {
                SceneManager._scene._startTime = Graphics.frameCount;
            }

            this.reveal();
            if (this.isMonster) {
                SceneManager._scene.gameOver();
            } else {
                if (SceneManager._scene._safeSE) {
                    AudioManager.playSe({ name: SceneManager._scene._safeSE, volume: 90, pitch: 100, pan: 0 });
                }
                SceneManager._scene.onTileRevealed();
                if (this.monsterCount === 0) {
                    this.revealAdjacentTiles();
                }
            }
        }

        reveal() {
            if (this.isRevealed) return;
            this.isRevealed = true;
            this.bitmap.clear();
            this.drawFrame();
            if (this.isMonster) {
                if (SceneManager._scene._bombBitmap) {
                    this.bitmap.blt(
                        SceneManager._scene._bombBitmap,
                        0, 0,
                        SceneManager._scene._bombBitmap.width,
                        SceneManager._scene._bombBitmap.height,
                        0, 0,
                        this._size,
                        this._size
                    );
                } else {
                    this.bitmap.drawCircle(this._size / 2, this._size / 2, this._size / 4, '#FF0000');
                }
            } else if (this.monsterCount > 0) {
                this.bitmap.drawText(this.monsterCount, 0, 0, this._size, this._size, 'center');
            }
        }

        revealAdjacentTiles() {
            for (let dx = -1; dx <= 1; dx++) {
                for (let dy = -1; dy <= 1; dy++) {
                    const nx = this._coordinateX + dx;
                    const ny = this._coordinateY + dy;
                    if (nx >= 0 && nx < SceneManager._scene._gridSize && ny >= 0 && ny < SceneManager._scene._gridSize) {
                        const tile = SceneManager._scene._tiles[ny][nx];
                        if (!tile.isRevealed && !tile.isMonster) {
                            tile.reveal();
                            SceneManager._scene.onTileRevealed();
                            if (tile.monsterCount === 0) {
                                tile.revealAdjacentTiles();
                            }
                        }
                    }
                }
            }
        }
    }
})();
