//=============================================================================
// TYT_TowerOfHanoi.js
//=============================================================================
// Copyright 2024 帝国妖異対策局
// Released under the MIT License.
// https://opensource.org/licenses/MIT
// Version: 
// 2024/09/12 初版
// Twitter: @tytinfo
// Blog: https://tyt.fanbox.cc/
//=============================================================================
/*:
 * @target MZ
 * @plugindesc 帝国ブロックプラグイン（SE・画像をイベントごとに設定可能、エラー修正済み）
 * @author
 *
 * @param defaultBackgroundName
 * @text デフォルト背景画像
 * @desc 背景に使用するパララックス画像を指定します。
 * @type file
 * @dir img/parallaxes
 * @default
 *
 * @param defaultOverlayImageName
 * @text デフォルトオーバーレイ画像
 * @desc 背景とゲーム画面の間に表示する画像を指定します。
 * @type file
 * @dir img/pictures
 * @default
 *
 * @param defaultOverlayImageX
 * @text デフォルトオーバーレイ画像X座標
 * @desc オーバーレイ画像の表示X座標を指定します。
 * @type number
 * @min -10000
 * @max 10000
 * @default 0
 *
 * @param defaultOverlayImageY
 * @text デフォルトオーバーレイ画像Y座標
 * @desc オーバーレイ画像の表示Y座標を指定します。
 * @type number
 * @min -10000
 * @max 10000
 * @default 0
 *
 * @param defaultOverlayImageZoom
 * @text デフォルトオーバーレイ画像ズーム率
 * @desc オーバーレイ画像の拡大率を指定します。（1.0で等倍）
 * @type number
 * @decimals 2
 * @min 0.01
 * @max 10.0
 * @default 1.00
 *
 * @param defaultLineClearSE
 * @text デフォルト横ライン消去SE
 * @desc 横10個のラインが消去されたときに再生するSE
 * @type struct<SE>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 *
 * @param defaultColorClearSE
 * @text デフォルト同色ブロック消去SE
 * @desc 同じ色のブロックが10個以上消えたときに再生するSE
 * @type struct<SE>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 *
 * @param defaultGameStartSE
 * @text デフォルトゲーム開始SE
 * @desc ゲーム開始時に再生するSE
 * @type struct<SE>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 *
 * @param defaultGameOverSE
 * @text デフォルトゲーム終了SE
 * @desc ゲーム終了時に再生するSE
 * @type struct<SE>
 * @default {"name":"","volume":"90","pitch":"100","pan":"0"}
 *
 * @command Start
 * @text ゲーム開始
 * @desc テトリス風のゲームを開始します。各種SEや画像を指定できます。
 *
 * @arg backgroundName
 * @text 背景画像
 * @desc 背景に使用するパララックス画像を指定します。
 * @type file
 * @dir img/parallaxes
 * @default
 *
 * @arg overlayImageName
 * @text オーバーレイ画像
 * @desc 背景とゲーム画面の間に表示する画像を指定します。
 * @type file
 * @dir img/pictures
 * @default
 *
 * @arg overlayImageX
 * @text オーバーレイ画像X座標
 * @desc オーバーレイ画像の表示X座標を指定します。
 * @type number
 * @min -10000
 * @max 10000
 * @default 0
 *
 * @arg overlayImageY
 * @text オーバーレイ画像Y座標
 * @desc オーバーレイ画像の表示Y座標を指定します。
 * @type number
 * @min -10000
 * @max 10000
 * @default 0
 *
 * @arg overlayImageZoom
 * @text オーバーレイ画像ズーム率
 * @desc オーバーレイ画像の拡大率を指定します。（1.0で等倍）
 * @type number
 * @decimals 2
 * @min 0.01
 * @max 10.0
 * @default 1.00
 *
 * @arg lineClearSE
 * @text 横ライン消去SE
 * @desc 横10個のラインが消去されたときのSE設定
 * @type struct<SE>
 * @default
 *
 * @arg colorClearSE
 * @text 同色ブロック消去SE
 * @desc 同じ色のブロックが10個以上消えたときのSE設定
 * @type struct<SE>
 * @default
 *
 * @arg gameStartSE
 * @text ゲーム開始SE
 * @desc ゲーム開始時のSE設定
 * @type struct<SE>
 * @default
 *
 * @arg gameOverSE
 * @text ゲーム終了SE
 * @desc ゲーム終了時のSE設定
 * @type struct<SE>
 * @default
 * 
 * @help
 * このプラグインは、テトリス風のミニゲームを実装します。ブロックは3～5の■から構成され、
 * 7種類・7色のブロックが登場します。
 * 
 * 主な機能:
 * - 各イベントでSEや画像を設定可能
 * - ゲームオーバー後にセルフスイッチAをONに設定
 * - 画面右下に戻るボタンを追加（既存のボタン画像を使用）
 * - ESCキーやキャンセルボタンでゲームを終了してマップ画面に戻る
 * 
 * 使用方法:
 * 1. プラグイン管理でこのプラグインを導入します。
 * 2. 必要に応じてプラグインパラメータを設定します。
 * 3. イベントコマンドの「プラグインコマンド」から「ゲーム開始」を選択し、ゲームを開始します。
 * 
 * 注意事項:
 * - ゲームオーバー後、ゲームを開始したイベントのセルフスイッチAがONになります。
 * 　イベントページの条件にセルフスイッチAを使用して、イベントの進行を管理してください。
 * - 画像ファイルを使用する場合は、指定のフォルダに正しく配置してください。
 * 
 */

 /*~struct~SE:
 * @param name
 * @text ファイル名
 * @desc SEのファイル名を指定します。
 * @default
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param volume
 * @text 音量
 * @desc SEの音量を指定します。（0～100）
 * @default 90
 * @type number
 * @min 0
 * @max 100
 *
 * @param pitch
 * @text ピッチ
 * @desc SEのピッチを指定します。（50～150）
 * @default 100
 * @type number
 * @min 50
 * @max 150
 *
 * @param pan
 * @text 位相
 * @desc SEの位相を指定します。（-100～100）
 * @default 0
 * @type number
 * @min -100
 * @max 100
 * 
 */
 (() => {
    const pluginName = "TYT_TeikokuBlock";

    // プラグインパラメータの取得
    const parameters = PluginManager.parameters(pluginName);
    const defaultBackgroundName = parameters['defaultBackgroundName'] || '';
    const defaultOverlayImageName = parameters['defaultOverlayImageName'] || '';
    const defaultOverlayImageX = Number(parameters['defaultOverlayImageX'] || 0);
    const defaultOverlayImageY = Number(parameters['defaultOverlayImageY'] || 0);
    const defaultOverlayImageZoom = Number(parameters['defaultOverlayImageZoom'] || 1.0);

    // SEパラメータの解析関数
    function parseSEParameter(param) {
        if (!param) return null;
        try {
            const se = JSON.parse(param);
            return {
                name: se.name,
                volume: Number(se.volume),
                pitch: Number(se.pitch),
                pan: Number(se.pan)
            };
        } catch (e) {
            return null;
        }
    }

    const defaultLineClearSE = parseSEParameter(parameters['defaultLineClearSE']);
    const defaultColorClearSE = parseSEParameter(parameters['defaultColorClearSE']);
    const defaultGameStartSE = parseSEParameter(parameters['defaultGameStartSE']);
    const defaultGameOverSE = parseSEParameter(parameters['defaultGameOverSE']);

    // プラグインコマンドの登録
    PluginManager.registerCommand(pluginName, "Start", function (args) {
        const options = {
            backgroundName: args.backgroundName || defaultBackgroundName,
            overlayImageName: args.overlayImageName || defaultOverlayImageName,
            overlayImageX: Number(args.overlayImageX || defaultOverlayImageX),
            overlayImageY: Number(args.overlayImageY || defaultOverlayImageY),
            overlayImageZoom: Number(args.overlayImageZoom || defaultOverlayImageZoom),
            lineClearSE: parseSEParameter(args.lineClearSE) || defaultLineClearSE,
            colorClearSE: parseSEParameter(args.colorClearSE) || defaultColorClearSE,
            gameStartSE: parseSEParameter(args.gameStartSE) || defaultGameStartSE,
            gameOverSE: parseSEParameter(args.gameOverSE) || defaultGameOverSE,
            eventId: this.eventId(),        // イベントIDを追加
            mapId: $gameMap.mapId(),        // マップIDを追加

        };
        // オプションを一時的にSceneManagerに保存
        SceneManager._teikokuBlockOptions = options;
        SceneManager.push(Scene_TeikokuBlock);
    });

    // テトリス風ゲームのシーン
    class Scene_TeikokuBlock extends Scene_Base {
        constructor() {
            super();
            this.initialize();
        }

        initialize() {
            super.initialize();
            this._score = 0;
            this._level = 1;
            this._nextTeikokuCube = null;
            this._gameOver = false;
            this._dropSpeed = 60; // 初期の落下速度
            this._timer = 0;

            // オプションを取得
            const options = SceneManager._teikokuBlockOptions || {};
            this._backgroundName = options.backgroundName || '';
            this._overlayImageName = options.overlayImageName || '';
            this._overlayImageX = options.overlayImageX || 0;
            this._overlayImageY = options.overlayImageY || 0;
            this._overlayImageZoom = options.overlayImageZoom || 1.0;
            this._lineClearSE = options.lineClearSE;
            this._colorClearSE = options.colorClearSE;
            this._gameStartSE = options.gameStartSE;
            this._gameOverSE = options.gameOverSE;
            this._eventId = options.eventId || null; // イベントIDを保持
            this._mapId = options.mapId || null;     // マップIDを保持
        }

        create() {
            super.create();
            this.createBackground();
            this.createOverlayImage();
            this.createField();
            this.createTeikokuCube();
            this.createUI();
            this.createReturnButton(); // 戻るボタンを作成
            // ゲーム開始時のSEを再生
            if (this._gameStartSE && this._gameStartSE.name) {
                AudioManager.playSe(this._gameStartSE);
            }
            // オプションをクリア
            SceneManager._teikokuBlockOptions = null;
        }

        // 戻るボタンを作成するメソッド
        createReturnButton() {
            this._returnButton = new Sprite_Button("cancel"); // デフォルトのキャンセルボタンを使用
            this._returnButton.x = Graphics.boxWidth - this._returnButton.width - 10; // 右下に配置
            this._returnButton.y = Graphics.boxHeight - this._returnButton.height - 10;
            this._returnButton.setClickHandler(this.onReturnButtonClicked.bind(this));
            this.addChild(this._returnButton);
        }

        // 戻るボタンが押されたときの処理
        onReturnButtonClicked() {
            // セルフスイッチAをONにする（必要に応じて）
            if (this._eventId && this._mapId) {
                //  $gameSelfSwitches.setValue([this._mapId, this._eventId, 'A'], true);
            }
            // ゲームを終了してマップ画面に戻る
            SceneManager.pop();
        }

        createBackground() {
            if (this._backgroundName) {
                this._background = new TilingSprite();
                this._background.bitmap = ImageManager.loadParallax(this._backgroundName);
                this._background.move(0, 0, Graphics.width, Graphics.height);
                this.addChild(this._background);
            }
        }

        createOverlayImage() {
            if (this._overlayImageName) {
                this._overlaySprite = new Sprite();
                this._overlaySprite.bitmap = ImageManager.loadPicture(this._overlayImageName);
                this._overlaySprite.x = this._overlayImageX;
                this._overlaySprite.y = this._overlayImageY;
                this._overlaySprite.scale.x = this._overlayImageZoom;
                this._overlaySprite.scale.y = this._overlayImageZoom;
                this.addChild(this._overlaySprite);
            }
        }

        createField() {
            this._fieldWidth = 12; // フィールドの幅を12に変更
            this._fieldHeight = 20;
            this._blockSize = 24;
            this._field = [];
            for (let y = 0; y < this._fieldHeight; y++) {
                this._field[y] = [];
                for (let x = 0; x < this._fieldWidth; x++) {
                    this._field[y][x] = null;
                }
            }
            this._fieldSprite = new Sprite(new Bitmap(this._fieldWidth * this._blockSize, this._fieldHeight * this._blockSize));
            this._fieldSprite.x = (Graphics.width - this._fieldWidth * this._blockSize) / 2; // X座標を再計算
            this._fieldSprite.y = 20;
            // フィールド背景を黒で塗りつぶす
            this._fieldSprite.bitmap.fillAll('black');
            this.addChild(this._fieldSprite);
        }


        // ここでブロックの定義を変更します
        createTeikokuCube() {
            this._TeikokuCubees = [
                // I-3
                { shape: [[1, 1, 1]], color: 'cyan' },
                // L-4
                { shape: [[1, 1], [1, 0], [1, 1]], color: 'yellow' },
                // J-3
                { shape: [[1, 1], [0, 1], [1, 1]], color: 'purple' },
                // T-3
                { shape: [[1, 1, 1], [1, 1, 0]], color: 'green' },
                // S-3
                { shape: [[0, 1, 1], [1, 1, 0]], color: 'red' },
                // I-5
                { shape: [[1, 1, 1, 1, 1]], color: 'blue' },
                // 十字型（Cross-5）
                { shape: [[0, 1, 0], [1, 1, 1], [0, 1, 0]], color: 'orange' },
                // 金型（Gold-5）
                {
                    shape: [
                        [0, 1, 0],
                        [1, 1, 1],
                        [1, 0, 1]
                    ],
                    color: 'gold'
                },
            ];
            this.spawnTeikokuCube();
        }

        spawnTeikokuCube() {
            if (this._nextTeikokuCube === null) {
                this._currentTeikokuCube = this.randomTeikokuCube();
            } else {
                this._currentTeikokuCube = this._nextTeikokuCube;
            }
            this._nextTeikokuCube = this.randomTeikokuCube();
            this._TeikokuCubeX = Math.floor(this._fieldWidth / 2) - Math.floor(this._currentTeikokuCube.shape[0].length / 2);
            this._TeikokuCubeY = 0;
            if (this.isColliding(this._TeikokuCubeX, this._TeikokuCubeY, this._currentTeikokuCube.shape)) {
                this._gameOver = true;
                this.showGameOver();
            }
        }

        randomTeikokuCube() {
            const index = Math.floor(Math.random() * this._TeikokuCubees.length);
            return this._TeikokuCubees[index];
        }

        createUI() {
            // スコア表示用スプライト
            this._scoreText = new Sprite(new Bitmap(200, 50));
            this._scoreText.x = this._fieldSprite.x + this._fieldSprite.width + 10; // フィールドの右側に配置
            this._scoreText.y = this._fieldSprite.y;
            this.addChild(this._scoreText);
            this.updateScoreText();

            // 次のブロック表示用スプライト
            this._nextText = new Sprite(new Bitmap(200, 50));
            this._nextText.x = this._fieldSprite.x; // フィールドのX座標と同じ
            this._nextText.y = this._fieldSprite.y + this._fieldSprite.height + 10; // フィールドの下に配置
            this._nextText.bitmap.fontSize = 20;
            this._nextText.bitmap.drawText("", 0, 0, 200, 50, 'left');
            this.addChild(this._nextText);

            this._nextSprite = new Sprite();
            this._nextSprite.x = this._fieldSprite.x + 120; // フィールドの左端から少し内側に
            this._nextSprite.y = this._nextText.y + 10; // テキストの下に配置
            this.addChild(this._nextSprite);
        }

        updateScoreText() {
            this._scoreText.bitmap.clear();
            this._scoreText.bitmap.fontSize = 20;
            this._scoreText.bitmap.drawText("スコア: " + this._score, 50, 0, 200, 50, 'left');
        }

        update() {
            super.update();
            if (this._gameOver) {
                this.isTriggeredToTitle();
                return;
            }
            // キャンセルボタンの入力をチェック
            if (Input.isTriggered('cancel')) {
                this.onReturnButtonClicked(); // ゲームを終了してマップ画面に戻る処理を呼び出す
                return;
            }
            this._timer++;
            if (this._timer >= this._dropSpeed) {
                this._timer = 0;
                this.moveTeikokuCube(0, 1);
            }
            this.updateInput();
            this.renderField();
            this.renderNextTeikokuCube();
            this.updateLevel();
        }

        updateInput() {
            if (Input.isTriggered('left')) {
                this.moveTeikokuCube(-1, 0);
            }
            if (Input.isTriggered('right')) {
                this.moveTeikokuCube(1, 0);
            }
            if (Input.isPressed('down')) {
                this.moveTeikokuCube(0, 1);
            }
            if (Input.isTriggered('ok')) {
                this.rotateTeikokuCube();
            }
        }

        moveTeikokuCube(dx, dy) {
            if (!this.isColliding(this._TeikokuCubeX + dx, this._TeikokuCubeY + dy, this._currentTeikokuCube.shape)) {
                this._TeikokuCubeX += dx;
                this._TeikokuCubeY += dy;
            } else if (dy !== 0) {
                this.fixTeikokuCube();
                this.resolveMatches(); // 再帰的にマッチを解決
                this.spawnTeikokuCube();
            }
        }

        rotateTeikokuCube() {
            const rotatedShape = this.rotateMatrix(this._currentTeikokuCube.shape);
            if (!this.isColliding(this._TeikokuCubeX, this._TeikokuCubeY, rotatedShape)) {
                this._currentTeikokuCube.shape = rotatedShape;
            }
        }

        rotateMatrix(matrix) {
            const result = [];
            for (let y = 0; y < matrix[0].length; y++) {
                result[y] = [];
                for (let x = 0; x < matrix.length; x++) {
                    result[y][x] = matrix[matrix.length - x - 1][y];
                }
            }
            return result;
        }

        isColliding(x, y, shape) {
            for (let ty = 0; ty < shape.length; ty++) {
                for (let tx = 0; tx < shape[ty].length; tx++) {
                    if (shape[ty][tx]) {
                        const fx = x + tx;
                        const fy = y + ty;
                        if (fx < 0 || fx >= this._fieldWidth || fy >= this._fieldHeight) {
                            return true;
                        }
                        if (fy >= 0 && this._field[fy][fx]) {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        fixTeikokuCube() {
            const color = this._currentTeikokuCube.color;
            const shape = this._currentTeikokuCube.shape;
            for (let ty = 0; ty < shape.length; ty++) {
                for (let tx = 0; tx < shape[ty].length; tx++) {
                    if (shape[ty][tx]) {
                        const fx = this._TeikokuCubeX + tx;
                        const fy = this._TeikokuCubeY + ty;
                        if (fy >= 0 && fy < this._fieldHeight && fx >= 0 && fx < this._fieldWidth) {
                            this._field[fy][fx] = color; // 色を保持
                        }
                    }
                }
            }
        }

        // 再帰的にマッチを解決する関数
        resolveMatches() {
            let matchesFound;
            do {
                matchesFound = false;
                // 横ラインの消去
                const linesCleared = this.clearLines();
                if (linesCleared > 0) {
                    matchesFound = true;
                    this._score += linesCleared * 100;
                    this.updateScoreText();
                    // 横ライン消去SEを再生
                    if (this._lineClearSE && this._lineClearSE.name) {
                        AudioManager.playSe(this._lineClearSE);
                    }
                }
                // 同色ブロックの消去
                const blocksCleared = this.checkSameColorBlocks();
                if (blocksCleared > 0) {
                    matchesFound = true;
                    this._score += blocksCleared * 10;
                    this.updateScoreText();
                    // 同色ブロック消去SEを再生
                    if (this._colorClearSE && this._colorClearSE.name) {
                        AudioManager.playSe(this._colorClearSE);
                    }
                }
            } while (matchesFound);
        }

        clearLines() {
            let linesCleared = 0;
            for (let y = this._fieldHeight - 1; y >= 0; y--) {
                if (this._field[y].every(cell => cell)) {
                    this._field.splice(y, 1);
                    this._field.unshift(new Array(this._fieldWidth).fill(null));
                    linesCleared++;
                    y++;
                }
            }
            return linesCleared;
        }

        checkSameColorBlocks() {
            const visited = [];
            for (let y = 0; y < this._fieldHeight; y++) {
                visited[y] = [];
                for (let x = 0; x < this._fieldWidth; x++) {
                    visited[y][x] = false;
                }
            }
            let totalBlocksCleared = 0;
            for (let y = 0; y < this._fieldHeight; y++) {
                for (let x = 0; x < this._fieldWidth; x++) {
                    if (this._field[y][x] && !visited[y][x]) {
                        const color = this._field[y][x];
                        const connectedBlocks = this.getConnectedBlocks(x, y, color, visited);
                        if (connectedBlocks.length >= 10) {
                            this.removeBlocks(connectedBlocks);
                            totalBlocksCleared += connectedBlocks.length;
                        }
                    }
                }
            }
            return totalBlocksCleared;
        }

        // 4方向で同じ色のブロックを探索する関数
        getConnectedBlocks(x, y, color, visited) {
            const stack = [[x, y]];
            const connected = [];
            visited[y][x] = true;

            while (stack.length > 0) {
                const [cx, cy] = stack.pop();
                connected.push([cx, cy]);

                const directions = [
                    [0, -1],  // 上
                    [1, 0],   // 右
                    [0, 1],   // 下
                    [-1, 0],  // 左
                ];

                for (const [dx, dy] of directions) {
                    const nx = cx + dx;
                    const ny = cy + dy;
                    if (nx >= 0 && nx < this._fieldWidth && ny >= 0 && ny < this._fieldHeight) {
                        if (!visited[ny][nx] && this._field[ny][nx] === color) {
                            visited[ny][nx] = true;
                            stack.push([nx, ny]);
                        }
                    }
                }
            }

            return connected;
        }

        removeBlocks(blocks) {
            for (const [x, y] of blocks) {
                this._field[y][x] = null;
            }
            // ブロックを下に落とす処理
            let moved;
            do {
                moved = false;
                for (let y = this._fieldHeight - 1; y > 0; y--) {
                    for (let x = 0; x < this._fieldWidth; x++) {
                        if (this._field[y][x] === null && this._field[y - 1][x] !== null) {
                            this._field[y][x] = this._field[y - 1][x];
                            this._field[y - 1][x] = null;
                            moved = true;
                        }
                    }
                }
            } while (moved);
        }

        renderField() {
            this._fieldSprite.bitmap.clear();
            // フィールド背景を黒で塗りつぶす
            this._fieldSprite.bitmap.fillAll('black');
            // フィールドの描画
            for (let y = 0; y < this._fieldHeight; y++) {
                for (let x = 0; x < this._fieldWidth; x++) {
                    if (this._field[y][x]) {
                        this._fieldSprite.bitmap.fillRect(x * this._blockSize, y * this._blockSize, this._blockSize - 1, this._blockSize - 1, this._field[y][x]);
                    }
                }
            }
            // 現在のテトロミノの描画
            const shape = this._currentTeikokuCube.shape;
            const color = this._currentTeikokuCube.color;
            for (let ty = 0; ty < shape.length; ty++) {
                for (let tx = 0; tx < shape[ty].length; tx++) {
                    if (shape[ty][tx]) {
                        const fx = this._TeikokuCubeX + tx;
                        const fy = this._TeikokuCubeY + ty;
                        this._fieldSprite.bitmap.fillRect(fx * this._blockSize, fy * this._blockSize, this._blockSize - 1, this._blockSize - 1, color);
                    }
                }
            }
        }

        renderNextTeikokuCube() {
            if (!this._nextBitmap) {
                // ビットマップのサイズを調整（必要に応じて変更）
                const bitmapSize = this._blockSize * 4; // ブロックサイズに合わせて設定
                this._nextBitmap = new Bitmap(bitmapSize, bitmapSize);
                this._nextSprite.bitmap = this._nextBitmap;
            }
            this._nextBitmap.clear();
            const shape = this._nextTeikokuCube.shape;
            const color = this._nextTeikokuCube.color;
            const blockSize = this._blockSize;
            for (let ty = 0; ty < shape.length; ty++) {
                for (let tx = 0; tx < shape[ty].length; tx++) {
                    if (shape[ty][tx]) {
                        this._nextBitmap.fillRect(tx * blockSize, ty * blockSize, blockSize - 1, blockSize - 1, color);
                    }
                }
            }
        }


        updateLevel() {
            if (this._score >= this._level * 500) {
                this._level++;
                this._dropSpeed = Math.max(10, this._dropSpeed - 5);
            }
        }

        showGameOver() {
            // ゲームオーバー表示用スプライト
            this._gameOverText = new Sprite(new Bitmap(400, 100));
            this._gameOverText.x = Graphics.width / 2 - 200;
            this._gameOverText.y = Graphics.height / 2 - 50;
            this._gameOverText.bitmap.fontSize = 40;
            this._gameOverText.bitmap.drawText("ゲームオーバー", 0, 0, 400, 50, 'center');
            this.addChild(this._gameOverText);

            // リトライテキスト表示用スプライト
            this._retryText = new Sprite(new Bitmap(400, 100));
            this._retryText.x = Graphics.width / 2 - 200;
            this._retryText.y = Graphics.height / 2;
            this._retryText.bitmap.fontSize = 20;
            this._retryText.bitmap.drawText("戻るには決定キーを押してください", 0, 0, 400, 50, 'center');
            this.addChild(this._retryText);

            // ゲーム終了時のSEを再生
            if (this._gameOverSE && this._gameOverSE.name) {
                AudioManager.playSe(this._gameOverSE);
            }
        }

        isTriggeredToTitle() {
            if (this._gameOver && Input.isTriggered('ok')) {
                // セルフスイッチAをONにする
                if (this._eventId && this._mapId) {
                    $gameSelfSwitches.setValue([this._mapId, this._eventId, 'A'], true);
                }
                SceneManager.pop();
            }
        }

    }

    // タイトル画面に「テトリス風ゲーム」を追加
    const _Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function () {
        _Scene_Title_createCommandWindow.call(this);
        this._commandWindow.setHandler('teikokuBlock', this.commandTeikokuBlock.bind(this));
    };

    Scene_Title.prototype.commandTeikokuBlock = function () {
        // デフォルトのオプションを使用してシーンを開始
        SceneManager._teikokuBlockOptions = {
            backgroundName: defaultBackgroundName,
            overlayImageName: defaultOverlayImageName,
            overlayImageX: defaultOverlayImageX,
            overlayImageY: defaultOverlayImageY,
            overlayImageZoom: defaultOverlayImageZoom,
            lineClearSE: defaultLineClearSE,
            colorClearSE: defaultColorClearSE,
            gameStartSE: defaultGameStartSE,
            gameOverSE: defaultGameOverSE,
        };
        SceneManager.push(Scene_TeikokuBlock);
    };

    const _Window_TitleCommand_makeCommandList = Window_TitleCommand.prototype.makeCommandList;
    Window_TitleCommand.prototype.makeCommandList = function () {
        _Window_TitleCommand_makeCommandList.call(this);
        this.addCommand("帝国ブロックゲーム", 'teikokuBlock');
    };

})();
