// おもちゃ
//


//---
// mons_data
//---
// 素材作成用。スクリプトで素材を加工して作るのに使うデータ集
// ※※出力のみ※※・・・入出力のも一緒に入れた


//=============================================================================
// rpg_managers.js v1.6.2
//=============================================================================

//-----------------------------------------------------------------------------
// DataManager
//
// The static class that manages the database and game objects.

function DataManager() {
    throw new Error('This is a static class');
}

var $dataActors       = null;
var $dataClasses      = null;
var $dataSkills       = null;
var $dataItems        = null;
var $dataWeapons      = null;
var $dataArmors       = null;
var $dataEnemies      = null;
var $dataTroops       = null;
var $dataStates       = null;
var $dataAnimations   = null;
var $dataTilesets     = null;
var $dataCommonEvents = null;
var $dataSystem       = null;
var $dataMapInfos     = null;
var $dataMap          = null;
var $gameTemp         = null;
var $gameSystem       = null;
var $gameScreen       = null;
var $gameTimer        = null;
var $gameMessage      = null;
var $gameSwitches     = null;
var $gameVariables    = null;
var $gameSelfSwitches = null;
var $gameActors       = null;
var $gameParty        = null;
var $gameTroop        = null;
var $gameMap          = null;
var $gamePlayer       = null;
var $testEvent        = null;
var $monsVariables     = null; //セーブデータ関連が上手くいかないからこっちでやる
var $shopVariables     = null; //理由は不明
var $textDatas         = null; //

DataManager._globalId       = 'RPGMV';
DataManager._lastAccessedId = 1;
DataManager._errorUrl       = null;

DataManager._databaseFiles = [
    { name: '$dataActors',       src: 'Actors.json'       },
    { name: '$dataClasses',      src: 'Classes.json'      },
    { name: '$dataSkills',       src: 'Skills.json'       },
    { name: '$dataItems',        src: 'Items.json'        },
    { name: '$dataWeapons',      src: 'Weapons.json'      },
    { name: '$dataArmors',       src: 'Armors.json'       },
    { name: '$dataEnemies',      src: 'Enemies.json'      },
    { name: '$dataTroops',       src: 'Troops.json'       },
    { name: '$dataStates',       src: 'States.json'       },
    { name: '$dataAnimations',   src: 'Animations.json'   },
    { name: '$dataTilesets',     src: 'Tilesets.json'     },
    { name: '$dataCommonEvents', src: 'CommonEvents.json' },
    { name: '$dataSystem',       src: 'System.json'       },
    { name: '$dataMapInfos',     src: 'MapInfos.json'     }
];

DataManager.loadDatabase = function() {
    var test = this.isBattleTest() || this.isEventTest();
    var prefix = test ? 'Test_' : '';
    for (var i = 0; i < this._databaseFiles.length; i++) {
        var name = this._databaseFiles[i].name;
        var src = this._databaseFiles[i].src;
        this.loadDataFile(name, prefix + src);
    }
    if (this.isEventTest()) {
        this.loadDataFile('$testEvent', prefix + 'Event.json');
    }
};

DataManager.loadDataFile = function(name, src) {
    var xhr = new XMLHttpRequest();
    var url = 'data/' + src;
    xhr.open('GET', url);
    xhr.overrideMimeType('application/json');
    xhr.onload = function() {
        if (xhr.status < 400) {
            window[name] = JSON.parse(xhr.responseText);
            DataManager.onLoad(window[name]);
        }
    };
    xhr.onerror = this._mapLoader || function() {
        DataManager._errorUrl = DataManager._errorUrl || url;
    };
    window[name] = null;
    xhr.send();
};

DataManager.isDatabaseLoaded = function() {
    this.checkError();
    for (var i = 0; i < this._databaseFiles.length; i++) {
        if (!window[this._databaseFiles[i].name]) {
            return false;
        }
    }
    return true;
};

DataManager.loadMapData = function(mapId) {
    if (mapId > 0) {
        var filename = 'Map%1.json'.format(mapId.padZero(3));
        this._mapLoader = ResourceHandler.createLoader('data/' + filename, this.loadDataFile.bind(this, '$dataMap', filename));
        this.loadDataFile('$dataMap', filename);
    } else {
        this.makeEmptyMap();
    }
};

DataManager.makeEmptyMap = function() {
    $dataMap = {};
    $dataMap.data = [];
    $dataMap.events = [];
    $dataMap.width = 100;
    $dataMap.height = 100;
    $dataMap.scrollType = 3;
};

DataManager.isMapLoaded = function() {
    this.checkError();
    return !!$dataMap;
};

DataManager.onLoad = function(object) {
    var array;
    if (object === $dataMap) {
        this.extractMetadata(object);
        array = object.events;
    } else {
        array = object;
    }
    if (Array.isArray(array)) {
        for (var i = 0; i < array.length; i++) {
            var data = array[i];
            if (data && data.note !== undefined) {
                this.extractMetadata(data);
            }
        }
    }
    if (object === $dataSystem) {
        Decrypter.hasEncryptedImages = !!object.hasEncryptedImages;
        Decrypter.hasEncryptedAudio = !!object.hasEncryptedAudio;
        Scene_Boot.loadSystemImages();
    }
};

DataManager.extractMetadata = function(data) {
    var re = /<([^<>:]+)(:?)([^>]*)>/g;
    data.meta = {};
    for (;;) {
        var match = re.exec(data.note);
        if (match) {
            if (match[2] === ':') {
                data.meta[match[1]] = match[3];
            } else {
                data.meta[match[1]] = true;
            }
        } else {
            break;
        }
    }
};

DataManager.checkError = function() {
    if (DataManager._errorUrl) {
        throw new Error('Failed to load: ' + DataManager._errorUrl);
    }
};

DataManager.isBattleTest = function() {
    return Utils.isOptionValid('btest');
};

DataManager.isEventTest = function() {
    return Utils.isOptionValid('etest');
};

DataManager.isSkill = function(item) {
    return item && $dataSkills.contains(item);
};

DataManager.isItem = function(item) {
    return item && $dataItems.contains(item);
};

DataManager.isWeapon = function(item) {
    return item && $dataWeapons.contains(item);
};

DataManager.isArmor = function(item) {
    return item && $dataArmors.contains(item);
};

DataManager.createGameObjects = function() {
    $gameTemp          = new Game_Temp();
    $gameSystem        = new Game_System();
    $gameScreen        = new Game_Screen();
    $gameTimer         = new Game_Timer();
    $gameMessage       = new Game_Message();
    $gameSwitches      = new Game_Switches();
    $gameVariables     = new Game_Variables();
    $gameSelfSwitches  = new Game_SelfSwitches();
    $gameActors        = new Game_Actors();
    $gameParty         = new Game_Party();
    $gameTroop         = new Game_Troop();
    $gameMap           = new Game_Map();
    $gamePlayer        = new Game_Player();
	$monsVariables     = new mons_data(); // ***
	$shopVariables     = new Shop_Variables();
	$textDatas         = new Text_Datas()
	// ここも加工。フォントについて
	Graphics.loadFont("font1","fonts/x12y12pxMaruMinya.ttf")
	Graphics.loadFont("font2","fonts/x16y32pxGridGazer.ttf")
	//Graphics.loadFont("font3","fonts/x12y16pxLineLinker.ttf")
	//Graphics.loadFont("font4","fonts/x12y16pxSolidLinker.ttf")

};

DataManager.setupNewGame = function() {
    this.createGameObjects();
    this.selectSavefileForNewGame();
    $gameParty.setupStartingMembers();
    $gamePlayer.reserveTransfer($dataSystem.startMapId,
        $dataSystem.startX, $dataSystem.startY);
    Graphics.frameCount = 0;
};

DataManager.setupBattleTest = function() {
    this.createGameObjects();
    $gameParty.setupBattleTest();
    BattleManager.setup($dataSystem.testTroopId, true, false);
    BattleManager.setBattleTest(true);
    BattleManager.playBattleBgm();
};

DataManager.setupEventTest = function() {
    this.createGameObjects();
    this.selectSavefileForNewGame();
    $gameParty.setupStartingMembers();
    $gamePlayer.reserveTransfer(-1, 8, 6);
    $gamePlayer.setTransparent(false);
};

DataManager.loadGlobalInfo = function() {
    var json;
    try {
        json = StorageManager.load(0);
    } catch (e) {
        console.error(e);
        return [];
    }
    if (json) {
        var globalInfo = JSON.parse(json);
        for (var i = 1; i <= this.maxSavefiles(); i++) {
            if (!StorageManager.exists(i)) {
                delete globalInfo[i];
            }
        }
        return globalInfo;
    } else {
        return [];
    }
};

DataManager.saveGlobalInfo = function(info) {
    StorageManager.save(0, JSON.stringify(info));
};

DataManager.isThisGameFile = function(savefileId) {
    var globalInfo = this.loadGlobalInfo();
    if (globalInfo && globalInfo[savefileId]) {
        if (StorageManager.isLocalMode()) {
            return true;
        } else {
            var savefile = globalInfo[savefileId];
            return (savefile.globalId === this._globalId &&
                    savefile.title === $dataSystem.gameTitle);
        }
    } else {
        return false;
    }
};

DataManager.isAnySavefileExists = function() {
    var globalInfo = this.loadGlobalInfo();
    if (globalInfo) {
        for (var i = 1; i < globalInfo.length; i++) {
            if (this.isThisGameFile(i)) {
                return true;
            }
        }
    }
    return false;
};

DataManager.latestSavefileId = function() {
    var globalInfo = this.loadGlobalInfo();
    var savefileId = 1;
    var timestamp = 0;
    if (globalInfo) {
        for (var i = 1; i < globalInfo.length; i++) {
            if (this.isThisGameFile(i) && globalInfo[i].timestamp > timestamp) {
                timestamp = globalInfo[i].timestamp;
                savefileId = i;
            }
        }
    }
    return savefileId;
};

DataManager.loadAllSavefileImages = function() {
    var globalInfo = this.loadGlobalInfo();
    if (globalInfo) {
        for (var i = 1; i < globalInfo.length; i++) {
            if (this.isThisGameFile(i)) {
                var info = globalInfo[i];
                this.loadSavefileImages(info);
            }
        }
    }
};

DataManager.loadSavefileImages = function(info) {
    if (info.characters) {
        for (var i = 0; i < info.characters.length; i++) {
            ImageManager.reserveCharacter(info.characters[i][0]);
        }
    }
    if (info.faces) {
        for (var j = 0; j < info.faces.length; j++) {
            ImageManager.reserveFace(info.faces[j][0]);
        }
    }
};

DataManager.maxSavefiles = function() {
    return 20;
};

DataManager.saveGame = function(savefileId) {
    try {
        StorageManager.backup(savefileId);
        return this.saveGameWithoutRescue(savefileId);
    } catch (e) {
        console.error(e);
        try {
            StorageManager.remove(savefileId);
            StorageManager.restoreBackup(savefileId);
        } catch (e2) {
        }
        return false;
    }
};

DataManager.loadGame = function(savefileId) {
    try {
        return this.loadGameWithoutRescue(savefileId);
    } catch (e) {
        console.error(e);
        return false;
    }
};

DataManager.loadSavefileInfo = function(savefileId) {
    var globalInfo = this.loadGlobalInfo();
    return (globalInfo && globalInfo[savefileId]) ? globalInfo[savefileId] : null;
};

DataManager.lastAccessedSavefileId = function() {
    return this._lastAccessedId;
};

DataManager.saveGameWithoutRescue = function(savefileId) {
    var json = JsonEx.stringify(this.makeSaveContents());
    if (json.length >= 200000) {
        console.warn('Save data too big!');
    }
    StorageManager.save(savefileId, json);
    this._lastAccessedId = savefileId;
    var globalInfo = this.loadGlobalInfo() || [];
    globalInfo[savefileId] = this.makeSavefileInfo();
    this.saveGlobalInfo(globalInfo);
    return true;
};

DataManager.loadGameWithoutRescue = function(savefileId) {
    var globalInfo = this.loadGlobalInfo();
    if (this.isThisGameFile(savefileId)) {
        var json = StorageManager.load(savefileId);
        this.createGameObjects();
        this.extractSaveContents(JsonEx.parse(json));
        this._lastAccessedId = savefileId;
        return true;
    } else {
        return false;
    }
};

DataManager.selectSavefileForNewGame = function() {
    var globalInfo = this.loadGlobalInfo();
    this._lastAccessedId = 1;
    if (globalInfo) {
        var numSavefiles = Math.max(0, globalInfo.length - 1);
        if (numSavefiles < this.maxSavefiles()) {
            this._lastAccessedId = numSavefiles + 1;
        } else {
            var timestamp = Number.MAX_VALUE;
            for (var i = 1; i < globalInfo.length; i++) {
                if (!globalInfo[i]) {
                    this._lastAccessedId = i;
                    break;
                }
                if (globalInfo[i].timestamp < timestamp) {
                    timestamp = globalInfo[i].timestamp;
                    this._lastAccessedId = i;
                }
            }
        }
    }
};

DataManager.makeSavefileInfo = function() {
    var info = {};
    info.globalId   = this._globalId;
    info.title      = $dataSystem.gameTitle;
    info.characters = $gameParty.charactersForSavefile();
    info.faces      = $gameParty.facesForSavefile();
    info.playtime   = $gameSystem.playtimeText();
    info.timestamp  = Date.now();
	info.displayName= $gameMap.displayName() // セーブ画面でマップ名を付け加えたいので
	info.endingOK   = $gameSwitches.value(25) // エンディングが終わりゲームクリア済み判定
    return info;
};

DataManager.makeSaveContents = function() {
    // A save data does not contain $gameTemp, $gameMessage, and $gameTroop.
    var contents = {};
    contents.system       = $gameSystem;
    contents.screen       = $gameScreen;
    contents.timer        = $gameTimer;
    contents.switches     = $gameSwitches;
    contents.variables    = $gameVariables;
    contents.selfSwitches = $gameSelfSwitches;
    contents.actors       = $gameActors;
    contents.party        = $gameParty;
    contents.map          = $gameMap;
    contents.player       = $gamePlayer;
	contents.shop_variables    = $shopVariables;
    return contents;
};

DataManager.extractSaveContents = function(contents) {
    $gameSystem        = contents.system;
    $gameScreen        = contents.screen;
    $gameTimer         = contents.timer;
    $gameSwitches      = contents.switches;
    $gameVariables     = contents.variables;
    $gameSelfSwitches  = contents.selfSwitches;
    $gameActors        = contents.actors;
    $gameParty         = contents.party;
    $gameMap           = contents.map;
    $gamePlayer        = contents.player;
	$shopVariables    = contents.shop_variables;
};

//-----------------------------------------------------------------------------
// ConfigManager
//
// The static class that manages the configuration data.

function ConfigManager() {
    throw new Error('This is a static class');
}

ConfigManager.alwaysDash        = false;
ConfigManager.commandRemember   = false;

Object.defineProperty(ConfigManager, 'bgmVolume', {
    get: function() {
        return AudioManager._bgmVolume;
    },
    set: function(value) {
        AudioManager.bgmVolume = value;
    },
    configurable: true
});

Object.defineProperty(ConfigManager, 'bgsVolume', {
    get: function() {
        return AudioManager.bgsVolume;
    },
    set: function(value) {
        AudioManager.bgsVolume = value;
    },
    configurable: true
});

Object.defineProperty(ConfigManager, 'meVolume', {
    get: function() {
        return AudioManager.meVolume;
    },
    set: function(value) {
        AudioManager.meVolume = value;
    },
    configurable: true
});

Object.defineProperty(ConfigManager, 'seVolume', {
    get: function() {
        return AudioManager.seVolume;
    },
    set: function(value) {
        AudioManager.seVolume = value;
    },
    configurable: true
});

ConfigManager.load = function() {
    var json;
    var config = {};
    try {
        json = StorageManager.load(-1);
    } catch (e) {
        console.error(e);
    }
    if (json) {
        config = JSON.parse(json);
    }
    this.applyData(config);
};

ConfigManager.save = function() {
    StorageManager.save(-1, JSON.stringify(this.makeData()));
};

ConfigManager.makeData = function() {
    var config = {};
    config.alwaysDash = this.alwaysDash;
    config.commandRemember = this.commandRemember;
    config.bgmVolume = this.bgmVolume;
    config.bgsVolume = this.bgsVolume;
    config.meVolume = this.meVolume;
    config.seVolume = this.seVolume;
    return config;
};

ConfigManager.applyData = function(config) {
    this.alwaysDash = this.readFlag(config, 'alwaysDash');
    this.commandRemember = this.readFlag(config, 'commandRemember');
    this.bgmVolume = this.readVolume(config, 'bgmVolume');
    this.bgsVolume = this.readVolume(config, 'bgsVolume');
    this.meVolume = this.readVolume(config, 'meVolume');
    this.seVolume = this.readVolume(config, 'seVolume');
};

ConfigManager.readFlag = function(config, name) {
    return !!config[name];
};

ConfigManager.readVolume = function(config, name) {
    var value = config[name];
    if (value !== undefined) {
        return Number(value).clamp(0, 100);
    } else {
        return 100;
    }
};

//-----------------------------------------------------------------------------
// StorageManager
//
// The static class that manages storage for saving game data.

function StorageManager() {
    throw new Error('This is a static class');
}

StorageManager.save = function(savefileId, json) {
    if (this.isLocalMode()) {
        this.saveToLocalFile(savefileId, json);
    } else {
        this.saveToWebStorage(savefileId, json);
    }
};

StorageManager.load = function(savefileId) {
    if (this.isLocalMode()) {
        return this.loadFromLocalFile(savefileId);
    } else {
        return this.loadFromWebStorage(savefileId);
    }
};

StorageManager.exists = function(savefileId) {
    if (this.isLocalMode()) {
        return this.localFileExists(savefileId);
    } else {
        return this.webStorageExists(savefileId);
    }
};

StorageManager.remove = function(savefileId) {
    if (this.isLocalMode()) {
        this.removeLocalFile(savefileId);
    } else {
        this.removeWebStorage(savefileId);
    }
};

StorageManager.backup = function(savefileId) {
    if (this.exists(savefileId)) {
        if (this.isLocalMode()) {
            var data = this.loadFromLocalFile(savefileId);
            var compressed = LZString.compressToBase64(data);
            var fs = require('fs');
            var dirPath = this.localFileDirectoryPath();
            var filePath = this.localFilePath(savefileId) + ".bak";
            if (!fs.existsSync(dirPath)) {
                fs.mkdirSync(dirPath);
            }
            fs.writeFileSync(filePath, compressed);
        } else {
            var data = this.loadFromWebStorage(savefileId);
            var compressed = LZString.compressToBase64(data);
            var key = this.webStorageKey(savefileId) + "bak";
            localStorage.setItem(key, compressed);
        }
    }
};

StorageManager.backupExists = function(savefileId) {
    if (this.isLocalMode()) {
        return this.localFileBackupExists(savefileId);
    } else {
        return this.webStorageBackupExists(savefileId);
    }
};

StorageManager.cleanBackup = function(savefileId) {
	if (this.backupExists(savefileId)) {
		if (this.isLocalMode()) {
			var fs = require('fs');
            var dirPath = this.localFileDirectoryPath();
            var filePath = this.localFilePath(savefileId);
            fs.unlinkSync(filePath + ".bak");
		} else {
		    var key = this.webStorageKey(savefileId);
			localStorage.removeItem(key + "bak");
		}
	}
};

StorageManager.restoreBackup = function(savefileId) {
    if (this.backupExists(savefileId)) {
        if (this.isLocalMode()) {
            var data = this.loadFromLocalBackupFile(savefileId);
            var compressed = LZString.compressToBase64(data);
            var fs = require('fs');
            var dirPath = this.localFileDirectoryPath();
            var filePath = this.localFilePath(savefileId);
            if (!fs.existsSync(dirPath)) {
                fs.mkdirSync(dirPath);
            }
            fs.writeFileSync(filePath, compressed);
            fs.unlinkSync(filePath + ".bak");
        } else {
            var data = this.loadFromWebStorageBackup(savefileId);
            var compressed = LZString.compressToBase64(data);
            var key = this.webStorageKey(savefileId);
            localStorage.setItem(key, compressed);
            localStorage.removeItem(key + "bak");
        }
    }
};

StorageManager.isLocalMode = function() {
    return Utils.isNwjs();
};

StorageManager.saveToLocalFile = function(savefileId, json) {
    var data = LZString.compressToBase64(json);
    var fs = require('fs');
    var dirPath = this.localFileDirectoryPath();
    var filePath = this.localFilePath(savefileId);
    if (!fs.existsSync(dirPath)) {
        fs.mkdirSync(dirPath);
    }
    fs.writeFileSync(filePath, data);
};

StorageManager.loadFromLocalFile = function(savefileId) {
    var data = null;
    var fs = require('fs');
    var filePath = this.localFilePath(savefileId);
    if (fs.existsSync(filePath)) {
        data = fs.readFileSync(filePath, { encoding: 'utf8' });
    }
    return LZString.decompressFromBase64(data);
};

StorageManager.loadFromLocalBackupFile = function(savefileId) {
    var data = null;
    var fs = require('fs');
    var filePath = this.localFilePath(savefileId) + ".bak";
    if (fs.existsSync(filePath)) {
        data = fs.readFileSync(filePath, { encoding: 'utf8' });
    }
    return LZString.decompressFromBase64(data);
};

StorageManager.localFileBackupExists = function(savefileId) {
    var fs = require('fs');
    return fs.existsSync(this.localFilePath(savefileId) + ".bak");
};

StorageManager.localFileExists = function(savefileId) {
    var fs = require('fs');
    return fs.existsSync(this.localFilePath(savefileId));
};

StorageManager.removeLocalFile = function(savefileId) {
    var fs = require('fs');
    var filePath = this.localFilePath(savefileId);
    if (fs.existsSync(filePath)) {
        fs.unlinkSync(filePath);
    }
};

StorageManager.saveToWebStorage = function(savefileId, json) {
    var key = this.webStorageKey(savefileId);
    var data = LZString.compressToBase64(json);
    localStorage.setItem(key, data);
};

StorageManager.loadFromWebStorage = function(savefileId) {
    var key = this.webStorageKey(savefileId);
    var data = localStorage.getItem(key);
    return LZString.decompressFromBase64(data);
};

StorageManager.loadFromWebStorageBackup = function(savefileId) {
    var key = this.webStorageKey(savefileId) + "bak";
    var data = localStorage.getItem(key);
    return LZString.decompressFromBase64(data);
};

StorageManager.webStorageBackupExists = function(savefileId) {
    var key = this.webStorageKey(savefileId) + "bak";
    return !!localStorage.getItem(key);
};

StorageManager.webStorageExists = function(savefileId) {
    var key = this.webStorageKey(savefileId);
    return !!localStorage.getItem(key);
};

StorageManager.removeWebStorage = function(savefileId) {
    var key = this.webStorageKey(savefileId);
    localStorage.removeItem(key);
};

StorageManager.localFileDirectoryPath = function() {
    var path = require('path');

    var base = path.dirname(process.mainModule.filename);
    return path.join(base, 'save/');
};

StorageManager.localFilePath = function(savefileId) {
    var name;
    if (savefileId < 0) {
        name = 'config.rpgsave';
    } else if (savefileId === 0) {
        name = 'global.rpgsave';
    } else {
        name = 'file%1.rpgsave'.format(savefileId);
    }
    return this.localFileDirectoryPath() + name;
};

StorageManager.webStorageKey = function(savefileId) {
    if (savefileId < 0) {
        return 'RPG Config';
    } else if (savefileId === 0) {
        return 'RPG Global';
    } else {
        return 'RPG File%1'.format(savefileId);
    }
};

//-----------------------------------------------------------------------------
// ImageManager
//
// The static class that loads images, creates bitmap objects and retains them.

function ImageManager() {
    throw new Error('This is a static class');
}

ImageManager.cache = new CacheMap(ImageManager);

ImageManager._imageCache = new ImageCache();
ImageManager._requestQueue = new RequestQueue();
ImageManager._systemReservationId = Utils.generateRuntimeId();

ImageManager._generateCacheKey = function(path, hue){
    return  path + ':' + hue;
};

ImageManager.loadAnimation = function(filename, hue) {
    return this.loadBitmap('img/animations/', filename, hue, true);
};

ImageManager.loadBattleback1 = function(filename, hue) {
    return this.loadBitmap('img/battlebacks1/', filename, hue, true);
};

ImageManager.loadBattleback2 = function(filename, hue) {
    return this.loadBitmap('img/battlebacks2/', filename, hue, true);
};

ImageManager.loadEnemy = function(filename, hue) {
    return this.loadBitmap('img/enemies/', filename, hue, true);
};

ImageManager.loadCharacter = function(filename, hue) {
    return this.loadBitmap('img/characters/', filename, hue, false);
};

ImageManager.loadFace = function(filename, hue) {
    return this.loadBitmap('img/faces/', filename, hue, true);
};

ImageManager.loadParallax = function(filename, hue) {
    return this.loadBitmap('img/parallaxes/', filename, hue, true);
};

ImageManager.loadPicture = function(filename, hue) {
    return this.loadBitmap('img/pictures/', filename, hue, true);
};

ImageManager.loadSvActor = function(filename, hue) {
    return this.loadBitmap('img/sv_actors/', filename, hue, false);
};

ImageManager.loadSvEnemy = function(filename, hue) {
    return this.loadBitmap('img/sv_enemies/', filename, hue, true);
};

ImageManager.loadSystem = function(filename, hue) {
    return this.loadBitmap('img/system/', filename, hue, false);
};

ImageManager.loadTileset = function(filename, hue) {
    return this.loadBitmap('img/tilesets/', filename, hue, false);
};

ImageManager.loadTitle1 = function(filename, hue) {
    return this.loadBitmap('img/titles1/', filename, hue, true);
};

ImageManager.loadTitle2 = function(filename, hue) {
    return this.loadBitmap('img/titles2/', filename, hue, true);
};

ImageManager.loadBitmap = function(folder, filename, hue, smooth) {
    if (filename) {
        var path = folder + encodeURIComponent(filename) + '.png';
        var bitmap = this.loadNormalBitmap(path, hue || 0);
        bitmap.smooth = smooth;
        return bitmap;
    } else {
        return this.loadEmptyBitmap();
    }
};

ImageManager.loadEmptyBitmap = function() {
    var empty = this._imageCache.get('empty');
    if(!empty){
        empty = new Bitmap();
        this._imageCache.add('empty', empty);
        this._imageCache.reserve('empty', empty, this._systemReservationId);
    }

    return empty;
};

ImageManager.loadNormalBitmap = function(path, hue) {
    var key = this._generateCacheKey(path, hue);
    var bitmap = this._imageCache.get(key);
    if (!bitmap) {
        bitmap = Bitmap.load(decodeURIComponent(path));
        bitmap.addLoadListener(function() {
            bitmap.rotateHue(hue);
        });
        this._imageCache.add(key, bitmap);
    }else if(!bitmap.isReady()){
        bitmap.decode();
    }

    return bitmap;
};

ImageManager.clear = function() {
    this._imageCache = new ImageCache();
};

ImageManager.isReady = function() {
    return this._imageCache.isReady();
};

ImageManager.isObjectCharacter = function(filename) {
    var sign = filename.match(/^[\!\$]+/);
    return sign && sign[0].contains('!');
};

ImageManager.isBigCharacter = function(filename) {
    var sign = filename.match(/^[\!\$]+/);
    return sign && sign[0].contains('$');
};

ImageManager.isZeroParallax = function(filename) {
    return filename.charAt(0) === '!';
};


ImageManager.reserveAnimation = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/animations/', filename, hue, true, reservationId);
};

ImageManager.reserveBattleback1 = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/battlebacks1/', filename, hue, true, reservationId);
};

ImageManager.reserveBattleback2 = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/battlebacks2/', filename, hue, true, reservationId);
};

ImageManager.reserveEnemy = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/enemies/', filename, hue, true, reservationId);
};

ImageManager.reserveCharacter = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/characters/', filename, hue, false, reservationId);
};

ImageManager.reserveFace = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/faces/', filename, hue, true, reservationId);
};

ImageManager.reserveParallax = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/parallaxes/', filename, hue, true, reservationId);
};

ImageManager.reservePicture = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/pictures/', filename, hue, true, reservationId);
};

ImageManager.reserveSvActor = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/sv_actors/', filename, hue, false, reservationId);
};

ImageManager.reserveSvEnemy = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/sv_enemies/', filename, hue, true, reservationId);
};

ImageManager.reserveSystem = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/system/', filename, hue, false, reservationId || this._systemReservationId);
};

ImageManager.reserveTileset = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/tilesets/', filename, hue, false, reservationId);
};

ImageManager.reserveTitle1 = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/titles1/', filename, hue, true, reservationId);
};

ImageManager.reserveTitle2 = function(filename, hue, reservationId) {
    return this.reserveBitmap('img/titles2/', filename, hue, true, reservationId);
};

ImageManager.reserveBitmap = function(folder, filename, hue, smooth, reservationId) {
    if (filename) {
        var path = folder + encodeURIComponent(filename) + '.png';
        var bitmap = this.reserveNormalBitmap(path, hue || 0, reservationId || this._defaultReservationId);
        bitmap.smooth = smooth;
        return bitmap;
    } else {
        return this.loadEmptyBitmap();
    }
};

ImageManager.reserveNormalBitmap = function(path, hue, reservationId){
    var bitmap = this.loadNormalBitmap(path, hue);
    this._imageCache.reserve(this._generateCacheKey(path, hue), bitmap, reservationId);

    return bitmap;
};

ImageManager.releaseReservation = function(reservationId){
    this._imageCache.releaseReservation(reservationId);
};

ImageManager.setDefaultReservationId = function(reservationId){
    this._defaultReservationId = reservationId;
};


ImageManager.requestAnimation = function(filename, hue) {
    return this.requestBitmap('img/animations/', filename, hue, true);
};

ImageManager.requestBattleback1 = function(filename, hue) {
    return this.requestBitmap('img/battlebacks1/', filename, hue, true);
};

ImageManager.requestBattleback2 = function(filename, hue) {
    return this.requestBitmap('img/battlebacks2/', filename, hue, true);
};

ImageManager.requestEnemy = function(filename, hue) {
    return this.requestBitmap('img/enemies/', filename, hue, true);
};

ImageManager.requestCharacter = function(filename, hue) {
    return this.requestBitmap('img/characters/', filename, hue, false);
};

ImageManager.requestFace = function(filename, hue) {
    return this.requestBitmap('img/faces/', filename, hue, true);
};

ImageManager.requestParallax = function(filename, hue) {
    return this.requestBitmap('img/parallaxes/', filename, hue, true);
};

ImageManager.requestPicture = function(filename, hue) {
    return this.requestBitmap('img/pictures/', filename, hue, true);
};

ImageManager.requestSvActor = function(filename, hue) {
    return this.requestBitmap('img/sv_actors/', filename, hue, false);
};

ImageManager.requestSvEnemy = function(filename, hue) {
    return this.requestBitmap('img/sv_enemies/', filename, hue, true);
};

ImageManager.requestSystem = function(filename, hue) {
    return this.requestBitmap('img/system/', filename, hue, false);
};

ImageManager.requestTileset = function(filename, hue) {
    return this.requestBitmap('img/tilesets/', filename, hue, false);
};

ImageManager.requestTitle1 = function(filename, hue) {
    return this.requestBitmap('img/titles1/', filename, hue, true);
};

ImageManager.requestTitle2 = function(filename, hue) {
    return this.requestBitmap('img/titles2/', filename, hue, true);
};

ImageManager.requestBitmap = function(folder, filename, hue, smooth) {
    if (filename) {
        var path = folder + encodeURIComponent(filename) + '.png';
        var bitmap = this.requestNormalBitmap(path, hue || 0);
        bitmap.smooth = smooth;
        return bitmap;
    } else {
        return this.loadEmptyBitmap();
    }
};

ImageManager.requestNormalBitmap = function(path, hue){
    var key = this._generateCacheKey(path, hue);
    var bitmap = this._imageCache.get(key);
    if(!bitmap){
        bitmap = Bitmap.request(path);
        bitmap.addLoadListener(function(){
            bitmap.rotateHue(hue);
        });
        this._imageCache.add(key, bitmap);
        this._requestQueue.enqueue(key, bitmap);
    }else{
        this._requestQueue.raisePriority(key);
    }

    return bitmap;
};

ImageManager.update = function(){
    this._requestQueue.update();
};

ImageManager.clearRequest = function(){
    this._requestQueue.clear();
};

//-----------------------------------------------------------------------------
// AudioManager
//
// The static class that handles BGM, BGS, ME and SE.

function AudioManager() {
    throw new Error('This is a static class');
}

AudioManager._masterVolume   = 1;   // (min: 0, max: 1)
AudioManager._bgmVolume      = 100;
AudioManager._bgsVolume      = 100;
AudioManager._meVolume       = 100;
AudioManager._seVolume       = 100;
AudioManager._currentBgm     = null;
AudioManager._currentBgs     = null;
AudioManager._bgmBuffer      = null;
AudioManager._bgsBuffer      = null;
AudioManager._meBuffer       = null;
AudioManager._seBuffers      = [];
AudioManager._staticBuffers  = [];
AudioManager._replayFadeTime = 0.5;
AudioManager._path           = 'audio/';
AudioManager._blobUrl        = null;

Object.defineProperty(AudioManager, 'masterVolume', {
    get: function() {
        return this._masterVolume;
    },
    set: function(value) {
        this._masterVolume = value;
        WebAudio.setMasterVolume(this._masterVolume);
        Graphics.setVideoVolume(this._masterVolume);
    },
    configurable: true
});

Object.defineProperty(AudioManager, 'bgmVolume', {
    get: function() {
        return this._bgmVolume;
    },
    set: function(value) {
        this._bgmVolume = value;
        this.updateBgmParameters(this._currentBgm);
    },
    configurable: true
});

Object.defineProperty(AudioManager, 'bgsVolume', {
    get: function() {
        return this._bgsVolume;
    },
    set: function(value) {
        this._bgsVolume = value;
        this.updateBgsParameters(this._currentBgs);
    },
    configurable: true
});

Object.defineProperty(AudioManager, 'meVolume', {
    get: function() {
        return this._meVolume;
    },
    set: function(value) {
        this._meVolume = value;
        this.updateMeParameters(this._currentMe);
    },
    configurable: true
});

Object.defineProperty(AudioManager, 'seVolume', {
    get: function() {
        return this._seVolume;
    },
    set: function(value) {
        this._seVolume = value;
    },
    configurable: true
});

AudioManager.playBgm = function(bgm, pos) {
    if (this.isCurrentBgm(bgm)) {
        this.updateBgmParameters(bgm);
    } else {
        this.stopBgm();
        if (bgm.name) { 
            if(Decrypter.hasEncryptedAudio && this.shouldUseHtml5Audio()){
                this.playEncryptedBgm(bgm, pos);
            }
            else {
                this._bgmBuffer = this.createBuffer('bgm', bgm.name);
                this.updateBgmParameters(bgm);
                if (!this._meBuffer) {
                    this._bgmBuffer.play(true, pos || 0);
                }
            }
        }
    }
    this.updateCurrentBgm(bgm, pos);
};

AudioManager.playEncryptedBgm = function(bgm, pos) {
    var ext = this.audioFileExt();
    var url = this._path + 'bgm/' + encodeURIComponent(bgm.name) + ext;
    url = Decrypter.extToEncryptExt(url);
    Decrypter.decryptHTML5Audio(url, bgm, pos);
};

AudioManager.createDecryptBuffer = function(url, bgm, pos){
    this._blobUrl = url;
    this._bgmBuffer = this.createBuffer('bgm', bgm.name);
    this.updateBgmParameters(bgm);
    if (!this._meBuffer) {
        this._bgmBuffer.play(true, pos || 0);
    }
    this.updateCurrentBgm(bgm, pos);
};

AudioManager.replayBgm = function(bgm) {
    if (this.isCurrentBgm(bgm)) {
        this.updateBgmParameters(bgm);
    } else {
        this.playBgm(bgm, bgm.pos);
        if (this._bgmBuffer) {
            this._bgmBuffer.fadeIn(this._replayFadeTime);
        }
    }
};

AudioManager.isCurrentBgm = function(bgm) {
    return (this._currentBgm && this._bgmBuffer &&
            this._currentBgm.name === bgm.name);
};

AudioManager.updateBgmParameters = function(bgm) {
    this.updateBufferParameters(this._bgmBuffer, this._bgmVolume, bgm);
};

AudioManager.updateCurrentBgm = function(bgm, pos) {
    this._currentBgm = {
        name: bgm.name,
        volume: bgm.volume,
        pitch: bgm.pitch,
        pan: bgm.pan,
        pos: pos
    };
};

AudioManager.stopBgm = function() {
    if (this._bgmBuffer) {
        this._bgmBuffer.stop();
        this._bgmBuffer = null;
        this._currentBgm = null;
    }
};

AudioManager.fadeOutBgm = function(duration) {
    if (this._bgmBuffer && this._currentBgm) {
        this._bgmBuffer.fadeOut(duration);
        this._currentBgm = null;
    }
};

AudioManager.fadeInBgm = function(duration) {
    if (this._bgmBuffer && this._currentBgm) {
        this._bgmBuffer.fadeIn(duration);
    }
};

AudioManager.playBgs = function(bgs, pos) {
    if (this.isCurrentBgs(bgs)) {
        this.updateBgsParameters(bgs);
    } else {
        this.stopBgs();
        if (bgs.name) {
            this._bgsBuffer = this.createBuffer('bgs', bgs.name);
            this.updateBgsParameters(bgs);
            this._bgsBuffer.play(true, pos || 0);
        }
    }
    this.updateCurrentBgs(bgs, pos);
};

AudioManager.replayBgs = function(bgs) {
    if (this.isCurrentBgs(bgs)) {
        this.updateBgsParameters(bgs);
    } else {
        this.playBgs(bgs, bgs.pos);
        if (this._bgsBuffer) {
            this._bgsBuffer.fadeIn(this._replayFadeTime);
        }
    }
};

AudioManager.isCurrentBgs = function(bgs) {
    return (this._currentBgs && this._bgsBuffer &&
            this._currentBgs.name === bgs.name);
};

AudioManager.updateBgsParameters = function(bgs) {
    this.updateBufferParameters(this._bgsBuffer, this._bgsVolume, bgs);
};

AudioManager.updateCurrentBgs = function(bgs, pos) {
    this._currentBgs = {
        name: bgs.name,
        volume: bgs.volume,
        pitch: bgs.pitch,
        pan: bgs.pan,
        pos: pos
    };
};

AudioManager.stopBgs = function() {
    if (this._bgsBuffer) {
        this._bgsBuffer.stop();
        this._bgsBuffer = null;
        this._currentBgs = null;
    }
};

AudioManager.fadeOutBgs = function(duration) {
    if (this._bgsBuffer && this._currentBgs) {
        this._bgsBuffer.fadeOut(duration);
        this._currentBgs = null;
    }
};

AudioManager.fadeInBgs = function(duration) {
    if (this._bgsBuffer && this._currentBgs) {
        this._bgsBuffer.fadeIn(duration);
    }
};

AudioManager.playMe = function(me) {
    this.stopMe();
    if (me.name) {
        if (this._bgmBuffer && this._currentBgm) {
            this._currentBgm.pos = this._bgmBuffer.seek();
            this._bgmBuffer.stop();
        }
        this._meBuffer = this.createBuffer('me', me.name);
        this.updateMeParameters(me);
        this._meBuffer.play(false);
        this._meBuffer.addStopListener(this.stopMe.bind(this));
    }
};

AudioManager.updateMeParameters = function(me) {
    this.updateBufferParameters(this._meBuffer, this._meVolume, me);
};

AudioManager.fadeOutMe = function(duration) {
    if (this._meBuffer) {
        this._meBuffer.fadeOut(duration);
    }
};

AudioManager.stopMe = function() {
    if (this._meBuffer) {
        this._meBuffer.stop();
        this._meBuffer = null;
        if (this._bgmBuffer && this._currentBgm && !this._bgmBuffer.isPlaying()) {
            this._bgmBuffer.play(true, this._currentBgm.pos);
            this._bgmBuffer.fadeIn(this._replayFadeTime);
        }
    }
};

AudioManager.playSe = function(se) {
    if (se.name) {
        this._seBuffers = this._seBuffers.filter(function(audio) {
            return audio.isPlaying();
        });
        var buffer = this.createBuffer('se', se.name);
        this.updateSeParameters(buffer, se);
        buffer.play(false);
        this._seBuffers.push(buffer);
    }
};

AudioManager.updateSeParameters = function(buffer, se) {
    this.updateBufferParameters(buffer, this._seVolume, se);
};

AudioManager.stopSe = function() {
    this._seBuffers.forEach(function(buffer) {
        buffer.stop();
    });
    this._seBuffers = [];
};

AudioManager.playStaticSe = function(se) {
    if (se.name) {
        this.loadStaticSe(se);
        for (var i = 0; i < this._staticBuffers.length; i++) {
            var buffer = this._staticBuffers[i];
            if (buffer._reservedSeName === se.name) {
                buffer.stop();
                this.updateSeParameters(buffer, se);
                buffer.play(false);
                break;
            }
        }
    }
};

AudioManager.loadStaticSe = function(se) {
    if (se.name && !this.isStaticSe(se)) {
        var buffer = this.createBuffer('se', se.name);
        buffer._reservedSeName = se.name;
        this._staticBuffers.push(buffer);
        if (this.shouldUseHtml5Audio()) {
            Html5Audio.setStaticSe(buffer._url);
        }
    }
};

AudioManager.isStaticSe = function(se) {
    for (var i = 0; i < this._staticBuffers.length; i++) {
        var buffer = this._staticBuffers[i];
        if (buffer._reservedSeName === se.name) {
            return true;
        }
    }
    return false;
};

AudioManager.stopAll = function() {
    this.stopMe();
    this.stopBgm();
    this.stopBgs();
    this.stopSe();
};

AudioManager.saveBgm = function() {
    if (this._currentBgm) {
        var bgm = this._currentBgm;
        return {
            name: bgm.name,
            volume: bgm.volume,
            pitch: bgm.pitch,
            pan: bgm.pan,
            pos: this._bgmBuffer ? this._bgmBuffer.seek() : 0
        };
    } else {
        return this.makeEmptyAudioObject();
    }
};

AudioManager.saveBgs = function() {
    if (this._currentBgs) {
        var bgs = this._currentBgs;
        return {
            name: bgs.name,
            volume: bgs.volume,
            pitch: bgs.pitch,
            pan: bgs.pan,
            pos: this._bgsBuffer ? this._bgsBuffer.seek() : 0
        };
    } else {
        return this.makeEmptyAudioObject();
    }
};

AudioManager.makeEmptyAudioObject = function() {
    return { name: '', volume: 0, pitch: 0 };
};

AudioManager.createBuffer = function(folder, name) {
    var ext = this.audioFileExt();
    var url = this._path + folder + '/' + encodeURIComponent(name) + ext;
    if (this.shouldUseHtml5Audio() && folder === 'bgm') {
        if(this._blobUrl) Html5Audio.setup(this._blobUrl);
        else Html5Audio.setup(url);
        return Html5Audio;
    } else {
        return new WebAudio(url);
    }
};

AudioManager.updateBufferParameters = function(buffer, configVolume, audio) {
    if (buffer && audio) {
        buffer.volume = configVolume * (audio.volume || 0) / 10000;
        buffer.pitch = (audio.pitch || 0) / 100;
        buffer.pan = (audio.pan || 0) / 100;
    }
};

AudioManager.audioFileExt = function() {
    if (WebAudio.canPlayOgg() && !Utils.isMobileDevice()) {
        return '.ogg';
    } else {
        return '.m4a';
    }
};

AudioManager.shouldUseHtml5Audio = function() {
    // The only case where we wanted html5audio was android/ no encrypt
    // Atsuma-ru asked to force webaudio there too, so just return false for ALL    // return Utils.isAndroidChrome() && !Decrypter.hasEncryptedAudio;
 return false;
};

AudioManager.checkErrors = function() {
    this.checkWebAudioError(this._bgmBuffer);
    this.checkWebAudioError(this._bgsBuffer);
    this.checkWebAudioError(this._meBuffer);
    this._seBuffers.forEach(function(buffer) {
        this.checkWebAudioError(buffer);
    }.bind(this));
    this._staticBuffers.forEach(function(buffer) {
        this.checkWebAudioError(buffer);
    }.bind(this));
};

AudioManager.checkWebAudioError = function(webAudio) {
    if (webAudio && webAudio.isError()) {
        throw new Error('Failed to load: ' + webAudio.url);
    }
};

//-----------------------------------------------------------------------------
// SoundManager
//
// The static class that plays sound effects defined in the database.

function SoundManager() {
    throw new Error('This is a static class');
}

SoundManager.preloadImportantSounds = function() {
    this.loadSystemSound(0);
    this.loadSystemSound(1);
    this.loadSystemSound(2);
    this.loadSystemSound(3);
};

SoundManager.loadSystemSound = function(n) {
    if ($dataSystem) {
        AudioManager.loadStaticSe($dataSystem.sounds[n]);
    }
};

SoundManager.playSystemSound = function(n) {
    if ($dataSystem) {
        AudioManager.playStaticSe($dataSystem.sounds[n]);
    }
};

SoundManager.playCursor = function() {
    this.playSystemSound(0);
};

SoundManager.playOk = function() {
    this.playSystemSound(1);
};

SoundManager.playCancel = function() {
    this.playSystemSound(2);
};

SoundManager.playBuzzer = function() {
    this.playSystemSound(3);
};

SoundManager.playEquip = function() {
    this.playSystemSound(4);
};

SoundManager.playSave = function() {
    this.playSystemSound(5);
};

SoundManager.playLoad = function() {
    this.playSystemSound(6);
};

SoundManager.playBattleStart = function() {
    this.playSystemSound(7);
};

SoundManager.playEscape = function() {
    this.playSystemSound(8);
};

SoundManager.playEnemyAttack = function() {
    this.playSystemSound(9);
};

SoundManager.playEnemyDamage = function() {
    this.playSystemSound(10);
};

SoundManager.playEnemyCollapse = function() {
    this.playSystemSound(11);
};

SoundManager.playBossCollapse1 = function() {
    this.playSystemSound(12);
};

SoundManager.playBossCollapse2 = function() {
    this.playSystemSound(13);
};

SoundManager.playActorDamage = function() {
    this.playSystemSound(14);
};

SoundManager.playActorCollapse = function() {
    this.playSystemSound(15);
};

SoundManager.playRecovery = function() {
    this.playSystemSound(16);
};

SoundManager.playMiss = function() {
    this.playSystemSound(17);
};

SoundManager.playEvasion = function() {
    this.playSystemSound(18);
};

SoundManager.playMagicEvasion = function() {
    this.playSystemSound(19);
};

SoundManager.playReflection = function() {
    this.playSystemSound(20);
};

SoundManager.playShop = function() {
    this.playSystemSound(21);
};

SoundManager.playUseItem = function() {
    this.playSystemSound(22);
};

SoundManager.playUseSkill = function() {
    this.playSystemSound(23);
};

//-----------------------------------------------------------------------------
// TextManager
//
// The static class that handles terms and messages.

function TextManager() {
    throw new Error('This is a static class');
}

TextManager.basic = function(basicId) {
    return $dataSystem.terms.basic[basicId] || '';
};

TextManager.param = function(paramId) {
    return $dataSystem.terms.params[paramId] || '';
};

TextManager.command = function(commandId) {
    return $dataSystem.terms.commands[commandId] || '';
};

TextManager.message = function(messageId) {
    return $dataSystem.terms.messages[messageId] || '';
};

TextManager.getter = function(method, param) {
    return {
        get: function() {
            return this[method](param);
        },
        configurable: true
    };
};

Object.defineProperty(TextManager, 'currencyUnit', {
    get: function() { return $dataSystem.currencyUnit; },
    configurable: true
});

Object.defineProperties(TextManager, {
    level           : TextManager.getter('basic', 0),
    levelA          : TextManager.getter('basic', 1),
    hp              : TextManager.getter('basic', 2),
    hpA             : TextManager.getter('basic', 3),
    mp              : TextManager.getter('basic', 4),
    mpA             : TextManager.getter('basic', 5),
    tp              : TextManager.getter('basic', 6),
    tpA             : TextManager.getter('basic', 7),
    exp             : TextManager.getter('basic', 8),
    expA            : TextManager.getter('basic', 9),
    fight           : TextManager.getter('command', 0),
    escape          : TextManager.getter('command', 1),
    attack          : TextManager.getter('command', 2),
    guard           : TextManager.getter('command', 3),
    item            : TextManager.getter('command', 4),
    skill           : TextManager.getter('command', 5),
    equip           : TextManager.getter('command', 6),
    status          : TextManager.getter('command', 7),
    formation       : TextManager.getter('command', 8),
    save            : TextManager.getter('command', 9),
    gameEnd         : TextManager.getter('command', 10),
    options         : TextManager.getter('command', 11),
    weapon          : TextManager.getter('command', 12),
    armor           : TextManager.getter('command', 13),
    keyItem         : TextManager.getter('command', 14),
    equip2          : TextManager.getter('command', 15),
    optimize        : TextManager.getter('command', 16),
    clear           : TextManager.getter('command', 17),
    newGame         : TextManager.getter('command', 18),
    continue_       : TextManager.getter('command', 19),
    toTitle         : TextManager.getter('command', 21),
    cancel          : TextManager.getter('command', 22),
    buy             : TextManager.getter('command', 24),
    sell            : TextManager.getter('command', 25),
    alwaysDash      : TextManager.getter('message', 'alwaysDash'),
    commandRemember : TextManager.getter('message', 'commandRemember'),
    bgmVolume       : TextManager.getter('message', 'bgmVolume'),
    bgsVolume       : TextManager.getter('message', 'bgsVolume'),
    meVolume        : TextManager.getter('message', 'meVolume'),
    seVolume        : TextManager.getter('message', 'seVolume'),
    possession      : TextManager.getter('message', 'possession'),
    expTotal        : TextManager.getter('message', 'expTotal'),
    expNext         : TextManager.getter('message', 'expNext'),
    saveMessage     : TextManager.getter('message', 'saveMessage'),
    loadMessage     : TextManager.getter('message', 'loadMessage'),
    file            : TextManager.getter('message', 'file'),
    partyName       : TextManager.getter('message', 'partyName'),
    emerge          : TextManager.getter('message', 'emerge'),
    preemptive      : TextManager.getter('message', 'preemptive'),
    surprise        : TextManager.getter('message', 'surprise'),
    escapeStart     : TextManager.getter('message', 'escapeStart'),
    escapeFailure   : TextManager.getter('message', 'escapeFailure'),
    victory         : TextManager.getter('message', 'victory'),
    defeat          : TextManager.getter('message', 'defeat'),
    obtainExp       : TextManager.getter('message', 'obtainExp'),
    obtainGold      : TextManager.getter('message', 'obtainGold'),
    obtainItem      : TextManager.getter('message', 'obtainItem'),
    levelUp         : TextManager.getter('message', 'levelUp'),
    obtainSkill     : TextManager.getter('message', 'obtainSkill'),
    useItem         : TextManager.getter('message', 'useItem'),
    criticalToEnemy : TextManager.getter('message', 'criticalToEnemy'),
    criticalToActor : TextManager.getter('message', 'criticalToActor'),
    actorDamage     : TextManager.getter('message', 'actorDamage'),
    actorRecovery   : TextManager.getter('message', 'actorRecovery'),
    actorGain       : TextManager.getter('message', 'actorGain'),
    actorLoss       : TextManager.getter('message', 'actorLoss'),
    actorDrain      : TextManager.getter('message', 'actorDrain'),
    actorNoDamage   : TextManager.getter('message', 'actorNoDamage'),
    actorNoHit      : TextManager.getter('message', 'actorNoHit'),
    enemyDamage     : TextManager.getter('message', 'enemyDamage'),
    enemyRecovery   : TextManager.getter('message', 'enemyRecovery'),
    enemyGain       : TextManager.getter('message', 'enemyGain'),
    enemyLoss       : TextManager.getter('message', 'enemyLoss'),
    enemyDrain      : TextManager.getter('message', 'enemyDrain'),
    enemyNoDamage   : TextManager.getter('message', 'enemyNoDamage'),
    enemyNoHit      : TextManager.getter('message', 'enemyNoHit'),
    evasion         : TextManager.getter('message', 'evasion'),
    magicEvasion    : TextManager.getter('message', 'magicEvasion'),
    magicReflection : TextManager.getter('message', 'magicReflection'),
    counterAttack   : TextManager.getter('message', 'counterAttack'),
    substitute      : TextManager.getter('message', 'substitute'),
    buffAdd         : TextManager.getter('message', 'buffAdd'),
    debuffAdd       : TextManager.getter('message', 'debuffAdd'),
    buffRemove      : TextManager.getter('message', 'buffRemove'),
    actionFailure   : TextManager.getter('message', 'actionFailure'),
});

//-----------------------------------------------------------------------------
// SceneManager
//
// The static class that manages scene transitions.

function SceneManager() {
    throw new Error('This is a static class');
}

/*
 * Gets the current time in ms without on iOS Safari.
 * @private
 */
SceneManager._getTimeInMsWithoutMobileSafari = function() {
    return performance.now();
};

SceneManager._scene             = null;
SceneManager._nextScene         = null;
SceneManager._stack             = [];
SceneManager._stopped           = false;
SceneManager._sceneStarted      = false;
SceneManager._exiting           = false;
SceneManager._previousClass     = null;
SceneManager._backgroundBitmap  = null;
SceneManager._screenWidth       = 816;
SceneManager._screenHeight      = 624;
SceneManager._boxWidth          = 816;
SceneManager._boxHeight         = 624;
SceneManager._deltaTime = 1.0 / 60.0;
if (!Utils.isMobileSafari()) SceneManager._currentTime = SceneManager._getTimeInMsWithoutMobileSafari();
SceneManager._accumulator = 0.0;

SceneManager.run = function(sceneClass) {
    try {
        this.initialize();
        this.goto(sceneClass);
        this.requestUpdate();
    } catch (e) {
        this.catchException(e);
    }
};

SceneManager.initialize = function() {
    this.initGraphics();
    this.checkFileAccess();
    this.initAudio();
    this.initInput();
    this.initNwjs();
    this.checkPluginErrors();
    this.setupErrorHandlers();
};

SceneManager.initGraphics = function() {
    var type = this.preferableRendererType();
    Graphics.initialize(this._screenWidth, this._screenHeight, type);
    Graphics.boxWidth = this._boxWidth;
    Graphics.boxHeight = this._boxHeight;
    Graphics.setLoadingImage('img/system/Loading.png');
    if (Utils.isOptionValid('showfps')) {
        Graphics.showFps();
    }
    if (type === 'webgl') {
        this.checkWebGL();
    }
};

SceneManager.preferableRendererType = function() {
    if (Utils.isOptionValid('canvas')) {
        return 'canvas';
    } else if (Utils.isOptionValid('webgl')) {
        return 'webgl';
    } else {
        return 'auto';
    }
};

SceneManager.shouldUseCanvasRenderer = function() {
    return Utils.isMobileDevice();
};

SceneManager.checkWebGL = function() {
    if (!Graphics.hasWebGL()) {
        throw new Error('Your browser does not support WebGL.');
    }
};

SceneManager.checkFileAccess = function() {
    if (!Utils.canReadGameFiles()) {
        throw new Error('Your browser does not allow to read local files.');
    }
};

SceneManager.initAudio = function() {
    var noAudio = Utils.isOptionValid('noaudio');
    if (!WebAudio.initialize(noAudio) && !noAudio) {
        throw new Error('Your browser does not support Web Audio API.');
    }
};

SceneManager.initInput = function() {
    Input.initialize();
    TouchInput.initialize();
};

SceneManager.initNwjs = function() {
    if (Utils.isNwjs()) {
        var gui = require('nw.gui');
        var win = gui.Window.get();
        if (process.platform === 'darwin' && !win.menu) {
            var menubar = new gui.Menu({ type: 'menubar' });
            var option = { hideEdit: true, hideWindow: true };
            menubar.createMacBuiltin('Game', option);
            win.menu = menubar;
        }
    }
};

SceneManager.checkPluginErrors = function() {
    PluginManager.checkErrors();
};

SceneManager.setupErrorHandlers = function() {
    window.addEventListener('error', this.onError.bind(this));
    document.addEventListener('keydown', this.onKeyDown.bind(this));
};

SceneManager.requestUpdate = function() {
    if (!this._stopped) {
        requestAnimationFrame(this.update.bind(this));
    }
};

SceneManager.update = function() {
    try {
        this.tickStart();
        if (Utils.isMobileSafari()) {
            this.updateInputData();
        }
        this.updateManagers();
        this.updateMain();
        this.tickEnd();
    } catch (e) {
        this.catchException(e);
    }
};

SceneManager.terminate = function() {
    window.close();
};

SceneManager.onError = function(e) {
    console.error(e.message);
    console.error(e.filename, e.lineno);
    try {
        this.stop();
        Graphics.printError('Error', e.message);
        AudioManager.stopAll();
    } catch (e2) {
    }
};

SceneManager.onKeyDown = function(event) {
    if (!event.ctrlKey && !event.altKey) {
        switch (event.keyCode) {
        case 116:   // F5
            if (Utils.isNwjs()) {
                location.reload();
            }
            break;
        case 119:   // F8
            if (Utils.isNwjs() && Utils.isOptionValid('test')) {
                require('nw.gui').Window.get().showDevTools();
            }
            break;
        }
    }
};

SceneManager.catchException = function(e) {
    if (e instanceof Error) {
        Graphics.printError(e.name, e.message);
        console.error(e.stack);
    } else {
        Graphics.printError('UnknownError', e);
    }
    AudioManager.stopAll();
    this.stop();
};

SceneManager.tickStart = function() {
    Graphics.tickStart();
};

SceneManager.tickEnd = function() {
    Graphics.tickEnd();
};

SceneManager.updateInputData = function() {
    Input.update();
    TouchInput.update();
};

SceneManager.updateMain = function() {
    if (Utils.isMobileSafari()) {
        this.changeScene();
        this.updateScene();
    } else {
        var newTime = this._getTimeInMsWithoutMobileSafari();
        var fTime = (newTime - this._currentTime) / 1000;
        if (fTime > 0.25) fTime = 0.25;
        this._currentTime = newTime;
        this._accumulator += fTime;
        while (this._accumulator >= this._deltaTime) {
            this.updateInputData();
            this.changeScene();
            this.updateScene();
            this._accumulator -= this._deltaTime;
        }
    }
    this.renderScene();
    this.requestUpdate();
};

SceneManager.updateManagers = function() {
    ImageManager.update();
};

SceneManager.changeScene = function() {
    if (this.isSceneChanging() && !this.isCurrentSceneBusy()) {
        if (this._scene) {
            this._scene.terminate();
            this._scene.detachReservation();
            this._previousClass = this._scene.constructor;
        }
        this._scene = this._nextScene;
        if (this._scene) {
            this._scene.attachReservation();
            this._scene.create();
            this._nextScene = null;
            this._sceneStarted = false;
            this.onSceneCreate();
        }
        if (this._exiting) {
            this.terminate();
        }
    }
};

SceneManager.updateScene = function() {
    if (this._scene) {
        if (!this._sceneStarted && this._scene.isReady()) {
            this._scene.start();
            this._sceneStarted = true;
            this.onSceneStart();
        }
        if (this.isCurrentSceneStarted()) {
            this._scene.update();
        }
    }
};

SceneManager.renderScene = function() {
    if (this.isCurrentSceneStarted()) {
        Graphics.render(this._scene);
    } else if (this._scene) {
        this.onSceneLoading();
    }
};

SceneManager.onSceneCreate = function() {
    Graphics.startLoading();
};

SceneManager.onSceneStart = function() {
    Graphics.endLoading();
};

SceneManager.onSceneLoading = function() {
    Graphics.updateLoading();
};

SceneManager.isSceneChanging = function() {
    return this._exiting || !!this._nextScene;
};

SceneManager.isCurrentSceneBusy = function() {
    return this._scene && this._scene.isBusy();
};

SceneManager.isCurrentSceneStarted = function() {
    return this._scene && this._sceneStarted;
};

SceneManager.isNextScene = function(sceneClass) {
    return this._nextScene && this._nextScene.constructor === sceneClass;
};

SceneManager.isPreviousScene = function(sceneClass) {
    return this._previousClass === sceneClass;
};

SceneManager.goto = function(sceneClass) {
    if (sceneClass) {
        this._nextScene = new sceneClass();
    }
    if (this._scene) {
        this._scene.stop();
    }
};

SceneManager.push = function(sceneClass) {
    this._stack.push(this._scene.constructor);
    this.goto(sceneClass);
};

SceneManager.pop = function() {
    if (this._stack.length > 0) {
        this.goto(this._stack.pop());
    } else {
        this.exit();
    }
};

SceneManager.exit = function() {
    this.goto(null);
    this._exiting = true;
};

SceneManager.clearStack = function() {
    this._stack = [];
};

SceneManager.stop = function() {
    this._stopped = true;
};

SceneManager.prepareNextScene = function() {
    this._nextScene.prepare.apply(this._nextScene, arguments);
};

SceneManager.snap = function() {
    return Bitmap.snap(this._scene);
};

SceneManager.snapForBackground = function() {
    this._backgroundBitmap = this.snap();
    this._backgroundBitmap.blur();
};

SceneManager.backgroundBitmap = function() {
    return this._backgroundBitmap;
};

SceneManager.resume = function() {
    this._stopped = false;
    this.requestUpdate();
    if (!Utils.isMobileSafari()) {
        this._currentTime = this._getTimeInMsWithoutMobileSafari();
        this._accumulator = 0;
    }
};

//-----------------------------------------------------------------------------
// BattleManager
//
// The static class that manages battle progress.

function BattleManager() {
    throw new Error('This is a static class');
}

BattleManager.setup = function(troopId, canEscape, canLose) {
    this.initMembers();
    this._canEscape = canEscape;
    this._canLose = canLose;
    $gameTroop.setup(troopId);
    $gameScreen.onBattleStart();
    this.makeEscapeRatio();
};

BattleManager.initMembers = function() {
    this._phase = 'init';
    this._canEscape = false;
    this._canLose = false;
    this._battleTest = false;
    this._eventCallback = null;
    this._preemptive = false;
    this._surprise = false;
    this._actorIndex = -1;
    this._actionForcedBattler = null;
    this._mapBgm = null;
    this._mapBgs = null;
    this._actionBattlers = [];
    this._subject = null;
    this._action = null;
    this._targets = [];
    this._logWindow = null;
    this._statusWindow = null;
    this._spriteset = null;
    this._escapeRatio = 0;
    this._escaped = false;
    this._rewards = {};
    this._turnForced = false;
};

BattleManager.isBattleTest = function() {
    return this._battleTest;
};

BattleManager.setBattleTest = function(battleTest) {
    this._battleTest = battleTest;
};

BattleManager.setEventCallback = function(callback) {
    this._eventCallback = callback;
};

BattleManager.setLogWindow = function(logWindow) {
    this._logWindow = logWindow;
};

BattleManager.setStatusWindow = function(statusWindow) {
    this._statusWindow = statusWindow;
};

BattleManager.setSpriteset = function(spriteset) {
    this._spriteset = spriteset;
};

BattleManager.onEncounter = function() {
    this._preemptive = (Math.random() < this.ratePreemptive());
    this._surprise = (Math.random() < this.rateSurprise() && !this._preemptive);
};

BattleManager.ratePreemptive = function() {
    return $gameParty.ratePreemptive($gameTroop.agility());
};

BattleManager.rateSurprise = function() {
    return $gameParty.rateSurprise($gameTroop.agility());
};

BattleManager.saveBgmAndBgs = function() {
    this._mapBgm = AudioManager.saveBgm();
    this._mapBgs = AudioManager.saveBgs();
};

BattleManager.playBattleBgm = function() {
    AudioManager.playBgm($gameSystem.battleBgm());
    AudioManager.stopBgs();
};

BattleManager.playVictoryMe = function() {
    AudioManager.playMe($gameSystem.victoryMe());
};

BattleManager.playDefeatMe = function() {
    AudioManager.playMe($gameSystem.defeatMe());
};

BattleManager.replayBgmAndBgs = function() {
    if (this._mapBgm) {
        AudioManager.replayBgm(this._mapBgm);
    } else {
        AudioManager.stopBgm();
    }
    if (this._mapBgs) {
        AudioManager.replayBgs(this._mapBgs);
    }
};

BattleManager.makeEscapeRatio = function() {
    this._escapeRatio = 0.5 * $gameParty.agility() / $gameTroop.agility();
};

BattleManager.update = function() {
    if (!this.isBusy() && !this.updateEvent()) {
        switch (this._phase) {
        case 'start':
            this.startInput();
            break;
        case 'turn':
            this.updateTurn();
            break;
        case 'action':
            this.updateAction();
            break;
        case 'turnEnd':
            this.updateTurnEnd();
            break;
        case 'battleEnd':
            this.updateBattleEnd();
            break;
        }
    }
};

BattleManager.updateEvent = function() {
    switch (this._phase) {
        case 'start':
        case 'turn':
        case 'turnEnd':
            if (this.isActionForced()) {
                this.processForcedAction();
                return true;
            } else {
                return this.updateEventMain();
            }
    }
    return this.checkAbort();
};

BattleManager.updateEventMain = function() {
    $gameTroop.updateInterpreter();
    $gameParty.requestMotionRefresh();
    if ($gameTroop.isEventRunning() || this.checkBattleEnd()) {
        return true;
    }
    $gameTroop.setupBattleEvent();
    if ($gameTroop.isEventRunning() || SceneManager.isSceneChanging()) {
        return true;
    }
    return false;
};

BattleManager.isBusy = function() {
    return ($gameMessage.isBusy() || this._spriteset.isBusy() ||
            this._logWindow.isBusy());
};

BattleManager.isInputting = function() {
    return this._phase === 'input';
};

BattleManager.isInTurn = function() {
    return this._phase === 'turn';
};

BattleManager.isTurnEnd = function() {
    return this._phase === 'turnEnd';
};

BattleManager.isAborting = function() {
    return this._phase === 'aborting';
};

BattleManager.isBattleEnd = function() {
    return this._phase === 'battleEnd';
};

BattleManager.canEscape = function() {
    return this._canEscape;
};

BattleManager.canLose = function() {
    return this._canLose;
};

BattleManager.isEscaped = function() {
    return this._escaped;
};

BattleManager.actor = function() {
    return this._actorIndex >= 0 ? $gameParty.members()[this._actorIndex] : null;
};

BattleManager.clearActor = function() {
    this.changeActor(-1, '');
};

BattleManager.changeActor = function(newActorIndex, lastActorActionState) {
    var lastActor = this.actor();
    this._actorIndex = newActorIndex;
    var newActor = this.actor();
    if (lastActor) {
        lastActor.setActionState(lastActorActionState);
    }
    if (newActor) {
        newActor.setActionState('inputting');
    }
};

BattleManager.startBattle = function() {
    this._phase = 'start';
    $gameSystem.onBattleStart();
    $gameParty.onBattleStart();
    $gameTroop.onBattleStart();
    this.displayStartMessages();
};

BattleManager.displayStartMessages = function() {
    $gameTroop.enemyNames().forEach(function(name) {
        $gameMessage.add(TextManager.emerge.format(name));
    });
    if (this._preemptive) {
        $gameMessage.add(TextManager.preemptive.format($gameParty.name()));
    } else if (this._surprise) {
        $gameMessage.add(TextManager.surprise.format($gameParty.name()));
    }
};

BattleManager.startInput = function() {
    this._phase = 'input';
    $gameParty.makeActions();
    $gameTroop.makeActions();
    this.clearActor();
    if (this._surprise || !$gameParty.canInput()) {
        this.startTurn();
    }
};

BattleManager.inputtingAction = function() {
    return this.actor() ? this.actor().inputtingAction() : null;
};

BattleManager.selectNextCommand = function() {
    do {
        if (!this.actor() || !this.actor().selectNextCommand()) {
            this.changeActor(this._actorIndex + 1, 'waiting');
            if (this._actorIndex >= $gameParty.size()) {
                this.startTurn();
                break;
            }
        }
    } while (!this.actor().canInput());
};

BattleManager.selectPreviousCommand = function() {
    do {
        if (!this.actor() || !this.actor().selectPreviousCommand()) {
            this.changeActor(this._actorIndex - 1, 'undecided');
            if (this._actorIndex < 0) {
                return;
            }
        }
    } while (!this.actor().canInput());
};

BattleManager.refreshStatus = function() {
    this._statusWindow.refresh();
};

BattleManager.startTurn = function() {
    this._phase = 'turn';
    this.clearActor();
    $gameTroop.increaseTurn();
    this.makeActionOrders();
    $gameParty.requestMotionRefresh();
    this._logWindow.startTurn();
};

BattleManager.updateTurn = function() {
    $gameParty.requestMotionRefresh();
    if (!this._subject) {
        this._subject = this.getNextSubject();
    }
    if (this._subject) {
        this.processTurn();
    } else {
        this.endTurn();
    }
};

BattleManager.processTurn = function() {
    var subject = this._subject;
    var action = subject.currentAction();
    if (action) {
        action.prepare();
        if (action.isValid()) {
            this.startAction();
        }
        subject.removeCurrentAction();
    } else {
        subject.onAllActionsEnd();
        this.refreshStatus();
        this._logWindow.displayAutoAffectedStatus(subject);
        this._logWindow.displayCurrentState(subject);
        this._logWindow.displayRegeneration(subject);
        this._subject = this.getNextSubject();
    }
};

BattleManager.endTurn = function() {
    this._phase = 'turnEnd';
    this._preemptive = false;
    this._surprise = false;
    this.allBattleMembers().forEach(function(battler) {
        battler.onTurnEnd();
        this.refreshStatus();
        this._logWindow.displayAutoAffectedStatus(battler);
        this._logWindow.displayRegeneration(battler);
    }, this);
    if (this.isForcedTurn()) {
        this._turnForced = false;
    }
};

BattleManager.isForcedTurn = function () {
    return this._turnForced;
};

BattleManager.updateTurnEnd = function() {
    this.startInput();
};

BattleManager.getNextSubject = function() {
    for (;;) {
        var battler = this._actionBattlers.shift();
        if (!battler) {
            return null;
        }
        if (battler.isBattleMember() && battler.isAlive()) {
            return battler;
        }
    }
};

BattleManager.allBattleMembers = function() {
    return $gameParty.members().concat($gameTroop.members());
};

BattleManager.makeActionOrders = function() {
    var battlers = [];
    if (!this._surprise) {
        battlers = battlers.concat($gameParty.members());
    }
    if (!this._preemptive) {
        battlers = battlers.concat($gameTroop.members());
    }
    battlers.forEach(function(battler) {
        battler.makeSpeed();
    });
    battlers.sort(function(a, b) {
        return b.speed() - a.speed();
    });
    this._actionBattlers = battlers;
};

BattleManager.startAction = function() {
    var subject = this._subject;
    var action = subject.currentAction();
    var targets = action.makeTargets();
    this._phase = 'action';
    this._action = action;
    this._targets = targets;
    subject.useItem(action.item());
    this._action.applyGlobal();
    this.refreshStatus();
    this._logWindow.startAction(subject, action, targets);
};

BattleManager.updateAction = function() {
    var target = this._targets.shift();
    if (target) {
        this.invokeAction(this._subject, target);
    } else {
        this.endAction();
    }
};

BattleManager.endAction = function() {
    this._logWindow.endAction(this._subject);
    this._phase = 'turn';
};

BattleManager.invokeAction = function(subject, target) {
    this._logWindow.push('pushBaseLine');
    if (Math.random() < this._action.itemCnt(target)) {
        this.invokeCounterAttack(subject, target);
    } else if (Math.random() < this._action.itemMrf(target)) {
        this.invokeMagicReflection(subject, target);
    } else {
        this.invokeNormalAction(subject, target);
    }
    subject.setLastTarget(target);
    this._logWindow.push('popBaseLine');
    this.refreshStatus();
};

BattleManager.invokeNormalAction = function(subject, target) {
    var realTarget = this.applySubstitute(target);
    this._action.apply(realTarget);
    this._logWindow.displayActionResults(subject, realTarget);
};

BattleManager.invokeCounterAttack = function(subject, target) {
    var action = new Game_Action(target);
    action.setAttack();
    action.apply(subject);
    this._logWindow.displayCounter(target);
    this._logWindow.displayActionResults(target, subject);
};

BattleManager.invokeMagicReflection = function(subject, target) {
	this._action._reflectionTarget = target;
    this._logWindow.displayReflection(target);
    this._action.apply(subject);
    this._logWindow.displayActionResults(target, subject);
};

BattleManager.applySubstitute = function(target) {
    if (this.checkSubstitute(target)) {
        var substitute = target.friendsUnit().substituteBattler();
        if (substitute && target !== substitute) {
            this._logWindow.displaySubstitute(substitute, target);
            return substitute;
        }
    }
    return target;
};

BattleManager.checkSubstitute = function(target) {
    return target.isDying() && !this._action.isCertainHit();
};

BattleManager.isActionForced = function() {
    return !!this._actionForcedBattler;
};

BattleManager.forceAction = function(battler) {
    this._actionForcedBattler = battler;
    var index = this._actionBattlers.indexOf(battler);
    if (index >= 0) {
        this._actionBattlers.splice(index, 1);
    }
};

BattleManager.processForcedAction = function() {
    if (this._actionForcedBattler) {
        this._turnForced = true;
        this._subject = this._actionForcedBattler;
        this._actionForcedBattler = null;
        this.startAction();
        this._subject.removeCurrentAction();
    }
};

BattleManager.abort = function() {
    this._phase = 'aborting';
};

BattleManager.checkBattleEnd = function() {
    if (this._phase) {
        if (this.checkAbort()) {
            return true;
        } else if ($gameParty.isAllDead()) {
            this.processDefeat();
            return true;
        } else if ($gameTroop.isAllDead()) {
            this.processVictory();
            return true;
        }
    }
    return false;
};

BattleManager.checkAbort = function() {
    if ($gameParty.isEmpty() || this.isAborting()) {
        SoundManager.playEscape();
        this._escaped = true;
        this.processAbort();
    }
    return false;
};

BattleManager.processVictory = function() {
    $gameParty.removeBattleStates();
    $gameParty.performVictory();
    this.playVictoryMe();
    this.replayBgmAndBgs();
    this.makeRewards();
    this.displayVictoryMessage();
    this.displayRewards();
    this.gainRewards();
    this.endBattle(0);
};

BattleManager.processEscape = function() {
    $gameParty.performEscape();
    SoundManager.playEscape();
    var success = this._preemptive ? true : (Math.random() < this._escapeRatio);
    if (success) {
        this.displayEscapeSuccessMessage();
        this._escaped = true;
        this.processAbort();
    } else {
        this.displayEscapeFailureMessage();
        this._escapeRatio += 0.1;
        $gameParty.clearActions();
        this.startTurn();
    }
    return success;
};

BattleManager.processAbort = function() {
    $gameParty.removeBattleStates();
    this.replayBgmAndBgs();
    this.endBattle(1);
};

BattleManager.processDefeat = function() {
    this.displayDefeatMessage();
    this.playDefeatMe();
    if (this._canLose) {
        this.replayBgmAndBgs();
    } else {
        AudioManager.stopBgm();
    }
    this.endBattle(2);
};

BattleManager.endBattle = function(result) {
    this._phase = 'battleEnd';
    if (this._eventCallback) {
        this._eventCallback(result);
    }
    if (result === 0) {
        $gameSystem.onBattleWin();
    } else if (this._escaped) {
        $gameSystem.onBattleEscape();
    }
};

BattleManager.updateBattleEnd = function() {
    if (this.isBattleTest()) {
        AudioManager.stopBgm();
        SceneManager.exit();
    } else if (!this._escaped && $gameParty.isAllDead()) {
        if (this._canLose) {
            $gameParty.reviveBattleMembers();
            SceneManager.pop();
        } else {
            SceneManager.goto(Scene_Gameover);
        }
    } else {
        SceneManager.pop();
    }
    this._phase = null;
};

BattleManager.makeRewards = function() {
    this._rewards = {};
    this._rewards.gold = $gameTroop.goldTotal();
    this._rewards.exp = $gameTroop.expTotal();
    this._rewards.items = $gameTroop.makeDropItems();
};

BattleManager.displayVictoryMessage = function() {
    $gameMessage.add(TextManager.victory.format($gameParty.name()));
};

BattleManager.displayDefeatMessage = function() {
    $gameMessage.add(TextManager.defeat.format($gameParty.name()));
};

BattleManager.displayEscapeSuccessMessage = function() {
    $gameMessage.add(TextManager.escapeStart.format($gameParty.name()));
};

BattleManager.displayEscapeFailureMessage = function() {
    $gameMessage.add(TextManager.escapeStart.format($gameParty.name()));
    $gameMessage.add('\\.' + TextManager.escapeFailure);
};

BattleManager.displayRewards = function() {
    this.displayExp();
    this.displayGold();
    this.displayDropItems();
};

BattleManager.displayExp = function() {
    var exp = this._rewards.exp;
    if (exp > 0) {
        var text = TextManager.obtainExp.format(exp, TextManager.exp);
        $gameMessage.add('\\.' + text);
    }
};

BattleManager.displayGold = function() {
    var gold = this._rewards.gold;
    if (gold > 0) {
        $gameMessage.add('\\.' + TextManager.obtainGold.format(gold));
    }
};

BattleManager.displayDropItems = function() {
    var items = this._rewards.items;
    if (items.length > 0) {
        $gameMessage.newPage();
        items.forEach(function(item) {
            $gameMessage.add(TextManager.obtainItem.format(item.name));
        });
    }
};

BattleManager.gainRewards = function() {
    this.gainExp();
    this.gainGold();
    this.gainDropItems();
};

BattleManager.gainExp = function() {
    var exp = this._rewards.exp;
    $gameParty.allMembers().forEach(function(actor) {
        actor.gainExp(exp);
    });
};

BattleManager.gainGold = function() {
    $gameParty.gainGold(this._rewards.gold);
};

BattleManager.gainDropItems = function() {
    var items = this._rewards.items;
    items.forEach(function(item) {
        $gameParty.gainItem(item, 1);
    });
};

//-----------------------------------------------------------------------------
// PluginManager
//
// The static class that manages the plugins.

function PluginManager() {
    throw new Error('This is a static class');
}

PluginManager._path         = 'js/plugins/';
PluginManager._scripts      = [];
PluginManager._errorUrls    = [];
PluginManager._parameters   = {};

PluginManager.setup = function(plugins) {
    plugins.forEach(function(plugin) {
        if (plugin.status && !this._scripts.contains(plugin.name)) {
            this.setParameters(plugin.name, plugin.parameters);
            this.loadScript(plugin.name + '.js');
            this._scripts.push(plugin.name);
        }
    }, this);
};

PluginManager.checkErrors = function() {
    var url = this._errorUrls.shift();
    if (url) {
        throw new Error('Failed to load: ' + url);
    }
};

PluginManager.parameters = function(name) {
    return this._parameters[name.toLowerCase()] || {};
};

PluginManager.setParameters = function(name, parameters) {
    this._parameters[name.toLowerCase()] = parameters;
};

PluginManager.loadScript = function(name) {
    var url = this._path + name;
    var script = document.createElement('script');
    script.type = 'text/javascript';
    script.src = url;
    script.async = false;
    script.onerror = this.onError.bind(this);
    script._url = url;
    document.body.appendChild(script);
};

PluginManager.onError = function(e) {
    this._errorUrls.push(e.target._url);
};






















	
	
	function getRandomNum(min, max) { // min「以上」でmax「以下」の整数ランダム発生用
		if (min <= max){
			// floorを使っているので、MAXがMINより小さいと計算がおかしくなる。
			// 最大MAXを超えたら入れ替える処理
			return Math.floor(Math.random() * (max + 1 - min) + min)
		}else{
			return Math.floor(Math.random() * (min + 1 - max) + max)
		}
	}
	function renumber(number, min, max) { // numberの数字をminmaxの範囲に変換してしまう奴
		var range = max - min + 1;
		var adjustedNumber = ((number * 109 - min) % range + range) % range;
		return adjustedNumber + min;
	}
	// 
	function get_data2(term,v2){
		
		if (term == 0){ // 使わない。基本用
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			// 100代は普通の開閉
			// 前半は閉じているのがベース
		}else if (term == 100){ // ずっとひらいている
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
		}else if (term == 101){ //２回瞬き
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,1,2,2,2,2,1,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,1,2,2,2,2,1,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			
		}else if (term == 102){ // 目を開けて２回瞬き。口を閉じていて、２回開閉
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,1,2,2,2,2,1,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,1,2,2,2,2,1,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			
		}else if (term == 103){ // 目を開けて２回瞬き。口を閉じていて、２回開閉
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,1,2,2,2,2,1,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,1,2,2,2,2,1,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			
		}else if (term == 104){ // １かいだけまばたき
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,1,2,2,2,2,1,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
		}else if (term == 105){ // １かいだけまばたき
			switch (v2) {
				case 0:
				return [0,0,1,2,2,1,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			
			
		}else if (term == 106){ // はんはん
			switch (v2) {
				case 0:
				return [0,0,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,1,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,1,0]
				
			}
		}else if (term == 107){ // はんはん
			switch (v2) {
				case 0:
				return [0,1,2,2,2,2,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,1,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,0,0]
				
			}
			
			
		}else if (term == 108){ // 目を閉じていて２回瞬き。口を開けていて、２回開閉
			switch (v2) {
				case 0:
				return [0,1,2,2,2,2,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,1,0,0,0,0,1,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,1,0,0,0,0,1,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,1,0]
			}
		}else if (term == 109){ // 目を閉じていて3回瞬き。口を開けていて、3回開閉
			switch (v2) {
				case 0:
				return [0,1,2,2,2,2,2,2]
				case 1:
				return [2,1,0,0,0,0,1,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,1,0,0,0,0,1,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,1,0,0,0,0,1,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,1,0]
			}
			
			// 150から逆走
		}else if (term == 150){ //ずっと閉じている
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
		}else if (term == 151){ //目を開けて２回瞬き。口を閉じていて、２回開閉
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,1,0,0,0,0,1,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,1,0,0,0,0,1,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
		}else if (term == 152){ // 目を開けて２回瞬き。口を閉じていて、２回開閉
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,1,0,0,0,0,1,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,1,0,0,0,0,1,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
		}else if (term == 153){ // 目を開けて２回瞬き。口を閉じていて、２回開閉
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,1,0,0,0,0,1,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,1,0,0,0,0,1,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
		}else if (term == 154){ // １かいだけまばたき
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,1,0,0,0,0,1,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
		}else if (term == 155){ // １かいだけまばたき
			switch (v2) {
				case 0:
				return [2,2,1,0,0,1,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,2,2,2,2,2,2,2]
				case 4:
				return [2,2,2,2,2,2,2,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
			
		}else if (term == 156){ // はんはん
			switch (v2) {
				case 0:
				return [2,2,2,2,2,2,2,2]
				case 1:
				return [2,2,2,2,2,2,2,2]
				case 2:
				return [2,2,2,2,2,2,2,2]
				case 3:
				return [2,1,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,1,2]
				
			}
		}else if (term == 157){ // はんはん
			switch (v2) {
				case 0:
				return [2,1,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,0,0,0,0,0,1,2]
				case 5:
				return [2,2,2,2,2,2,2,2]
				case 6:
				return [2,2,2,2,2,2,2,2]
				case 7:
				return [2,2,2,2,2,2,2,2]
				
			}
			
			
		}else if (term == 158){ // 目を閉じていて２回瞬き。口を開けていて、２回開閉
			switch (v2) {
				case 0:
				return [2,1,0,0,0,0,0,0]
				case 1:
				return [0,0,0,0,0,0,0,0]
				case 2:
				return [0,1,2,2,2,2,1,0]
				case 3:
				return [0,0,0,0,0,0,0,0]
				case 4:
				return [0,1,2,2,2,2,1,0]
				case 5:
				return [0,0,0,0,0,0,0,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,1,2]
			}
		}else if (term == 159){ // 目を閉じていて3回瞬き。口を開けていて、3回開閉
			switch (v2) {
				case 0:
				return [2,1,0,0,0,0,0,0]
				case 1:
				return [0,1,2,2,2,2,1,0]
				case 2:
				return [0,0,0,0,0,0,0,0]
				case 3:
				return [0,1,2,2,2,2,1,0]
				case 4:
				return [0,0,0,0,0,0,0,0]
				case 5:
				return [0,1,2,2,2,2,1,0]
				case 6:
				return [0,0,0,0,0,0,0,0]
				case 7:
				return [0,0,0,0,0,0,1,2]
			}
			
			// 200代は射精関連
		}else if (term == 201){ // これは射精直前の痙攣。１回だけやる。
			switch (v2) {
				case 0:
				return [0,2,4,6,0,2,4,6]
				case 1:
				return [0,2,4,6,0,2,4,6]
				case 2:
				return [0,2,4,6,0,2,4,6]
				case 3:
				return [0,2,4,6,0,2,4,6]
				case 4:
				return [0,2,4,6,0,2,4,6]
				case 5:
				return [0,2,4,6,0,2,4,6]
				case 6:
				return [0,2,4,6,0,2,4,4]
				case 7:
				return [2,4,6,4,2,4,4,4]
				
			}
		}else if (term == 202){ // ここから射精
			switch (v2) {
				case 0:// 前回は押し込んだ状態[4,4]から、その続きとして。202と203はセット。
				return [0,0,1,1,1,1,1,1] //[0,1,1,1,2,2,3,3]
				case 1:
				return [2,2,3,3,3,3,3,3] //[2,2,4,5,5,5,5,5]
				case 2:
				return [4,4,5,5,5,5,5,5] //[5,5,5,5,5,5,5,5]
				case 3:
				return [5,5,5,5,5,5,5,5] //[5,5,5,5,5,5,5,5]
				case 4:
				return [5,5,5,5,5,5,5,5] //[6,7,7,7,7,7,7,7]
				case 5:
				return [5,5,5,5,5,5,5,5]
				case 6:
				return [5,5,5,5,5,5,5,5]
				case 7:
				return [5,5,5,5,5,5,5,5]
				
			}
		}else if (term == 203){
			switch (v2) {
				case 0:
				return [5,5,5,5,5,5,5,5]
				case 1:
				return [5,5,5,5,5,5,5,5]
				case 2:
				return [5,5,5,5,5,5,5,5]
				case 3:
				return [5,5,5,5,5,5,5,5]
				case 4:
				return [5,5,5,5,5,5,5,5]
				case 5:
				return [5,5,5,5,5,5,5,5]
				case 6:
				return [5,5,5,5,5,5,5,5]
				case 7:
				return [5,5,5,5,5,5,5,5]
				
			}
		}else if (term == 204){
			switch (v2) {
				case 0:
				return [6,6,6,6,6,6,6,6]
				case 1:
				return [6,6,6,6,6,6,6,6]
				case 2:
				return [6,6,6,6,6,6,6,6]
				case 3:
				return [6,6,6,6,6,6,6,6]
				case 4:
				return [6,6,6,6,6,6,6,6]
				case 5:
				return [6,6,6,6,6,6,6,6]
				case 6:
				return [6,6,6,6,6,6,6,6]
				case 7:
				return [6,6,6,6,6,6,6,6]
				
			}
		}else if (term == 205){
			switch (v2) {
				case 0:
				return [7,7,7,7,7,7,7,7]
				case 1:
				return [7,7,7,7,7,7,7,7]
				case 2:
				return [7,7,7,7,7,7,7,7]
				case 3:
				return [7,7,7,7,7,7,7,7]
				case 4:
				return [7,7,7,7,7,7,7,7]
				case 5:
				return [7,7,7,7,7,7,7,7]
				case 6:
				return [7,7,7,7,7,7,7,7]
				case 7:
				return [7,7,7,7,7,7,7,7]
				
			}
			
			// 細かい変化パターンjump
			//1 目を開けて２回瞬き。口を閉じていて、２回開閉
			//2 目を開けて２回瞬き。口を閉じていて、２回開閉
			//3 目を開けて２回瞬き。口を閉じていて、２回開閉
			//4 １かいだけまばたき
			//5 １かいだけまばたき
			//6 はんはん
			//7 はんはん
			//8 目を閉じていて２回瞬き。口を開けていて、２回開閉
			//9 目を閉じていて3回瞬き。口を開けていて、3回開閉
		
		// 1100~め 50未満が閉じてるのベース
		}else if (term == 1103){
			return [154,150,154,150, 150,152,150,152, 150,156,150,154] // 閉じるベース
		}else if (term == 1152){
			return [104,100,104,100, 100,102,100,102, 100,106,100,104]
		}else if (term == 1154){
			return [100,100,102,108, 152,150,152,150, 100,100,102,100]
		}else if (term == 1156){
			return [100,100,102,108, 152,150,152,150, 150,150,152,150]
		}else if (term == 1104){
			return [154,150,154,150, 150,152,150,152, 150,156,150,154]
		}else if (term == 1151){
			return [104,100,105,100, 100,105,100,104, 100,106,100,100] // 開けてるの中心。瞬きはそんなにしない
		}else if (term == 1158){//うろ
			return [103,100,102,100, 100,102,100,102, 100,106,100,104]
		}else if (term == 1160){//うろ
			return [103,100,102,100, 100,104,100,104, 150,156,150,154]
		}else if (term == 1105){
			return [154,150,155,150, 150,152,157,150, 150,150,150,155] // return [153,156,152,156, 150,152,157,150, 150,150,150,155]
		}else if (term == 1107){
			return [153,156,152,156, 150,152,150,152, 150,156,150,154]
		}else if (term == 1199){
		}else if (term == 1199){
		}else if (term == 1199){
		}else if (term == 1199){
		}else if (term == 1199){
		}else if (term == 1199){
		// 2100~くち　奇数が閉じてるのベース ido 3 52 54 56 4 51 58 4
		}else if (term == 2103){
			return [154,150,154,150, 153,152,153,152, 150,156,150,154]
		}else if (term == 2152){
			return [104,100,104,100, 100,102,100,152, 102,100,100,105]
		}else if (term == 2154){
			return [151,150,150,150, 152,150,152,150, 150,150,150,150]
		}else if (term == 2156){
			return [151,150,150,150, 152,150,152,150, 150,150,152,150] // ミミック調教、口開け　Aスライム調教目閉じ口開け
		}else if (term == 2104){
			return [100,101,104,100, 100,102,100,152, 102,100,100,105]
		}else if (term == 2151){
			return [150,156,157,150, 156,157,150,150, 150,150,150,150]
		}else if (term == 2158){
			return [100,100,106,106, 100,102,100,106, 102,100,100,105]
		}else if (term == 2160){
			return [102,100,106,106, 100,102,100,156, 102,100,100,105]
		}else if (term == 2105){
			return [150,156,157,150, 153,152,151,154, 150,150,150,155] // return [154,156,154,157, 153,152,151,154, 150,150,150,155]
		}else if (term == 2107){
			return [154,156,154,157, 100,102,100,152, 102,100,100,105]
		}else if (term == 2199){
		}else if (term == 2199){
		}else if (term == 2199){
		}else if (term == 2199){
		}else if (term == 2199){
		}else if (term == 2199){
		}else if (term >= 2000){
			return [150,156,157,150, 156,157,150,150, 150,150,150,150]
		}else if (term >= 1000){
			//return [152,152,152,152, 152,152,152,152, 152,152,152,152]
			return [100,100,102,108, 152,150,152,150, 100,100,102,100]
			//return [152,152,152,152, 152,152,152,152, 152,152,152,152]
		}
		return [7,7,7,7,7,7,7,7]
		
	};
	// キャラ番号メモ
	// うさぎ120マミー140 ミノタウロス150
	// シーマン200ねこまた210インプ220デーモン230
	// スライム赤300ゴブリン330ハーピー340
	// スライム緑400スライム青410オーガ420
	// ミミック500リス510はち520
	// ウロボロス600・・・
	// ひとくいそう700みの710
	// 1？、4怒、6くしゃくしゃ、
	// 2♪、3♡、7・・・、
	
	function get_data3(m_No,a_No,variableId) {
		if (a_No == -500){
			
			var commonEvent = $dataCommonEvents[47]
			if (commonEvent && commonEvent.list) {
				var textLines = []
				var j = -1
				for (var i = 0; i < commonEvent.list.length; i++) {
					var command = commonEvent.list[i]
					if (command.code === 105) {
						j++
						i++ // 最初の一行、種類とか描いてるのをを飛ばすための処理
						textLines[j] = []
					}else if (command.code === 405) {
						textLines[j].push(commonEvent.list[i].parameters[0])
					}
				}
			}
			//var a0 = textLines[m_No]
			//var r0 = Math.floor(Math.random() * a0.length)
			//var se0 = 'z' + String(m_No).padStart(2, '0')
			
			return textLines // 全データ返す
			
		}else if (a_No == -700){ // 喘ぎパターン
			var textsNo = 0
			var vos = [[0],[0],[0]]
			var heart = false
			if (m_No == 80){ // パターン番号[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24]
				// テスト用69,
				textsNo = 1
				vos = [[87,89,90,88,91],[59,69,74],[72,73,5,47]]
				heart = true
			}else if (m_No == 2){ // スライム調教前
				textsNo = 2
				vos = [[0,1,3,5,6,8,9,10,12,13,14,15,16,17,18,19,20,21,22,23],[34,35,37],[30,31,32,33]]
			}else if (m_No == 3){ // スライム調教後。抑え気味
				textsNo = 2
				heart = true
				vos = [[2,3,4,5,6,7,9,10,11,12,13,14,15,18,19,20,21,22,23,24,25,26,27,28,29],[34,35,36,37],[30,31,32,33]]
			}else if (m_No == 4){
				textsNo = 1
				vos = [[0,1,2,3,4,9,10,11,12,15,16,17],[6,13,14],[5,7,8]]
			}else if (m_No == 5){
				textsNo = 1
				vos = [[18,19,20,21,27,28,29,30,31,32,33,34],[23,25,35],[22,24,26,31]]
			}else if (m_No == 6){
				textsNo = 1
				vos = [[22,24,7,8,39,40,41,42,43,44,45,46,48,49],[36,37],[47,51,52,54]]
			}else if (m_No == 7){
				textsNo = 1
				vos = [[60,61,62,63,64,65,66,67,68,70,71],[53,58,59],[54,55,56,57]]
				heart = true
			}else if (m_No == 8){ // すごく感じてる系
				textsNo = 1
				vos = [[75,76,77,78,79,80,63,64,65,66,68,70],[59,69,74],[72,73,5,47]]
				heart = true
			}else if (m_No == 9){ // ガーディアンフェラ苦痛・呻き重視
				textsNo = 1
				vos = [[81,81,82,82,83,83,84,84,85,85,86,86,93,93,94,94, 1,3,10,12,44,16],[87,90,88],[54,55,56,57]]
			}else if (m_No == 10){ // 従順フェラ
				textsNo = 1
				vos = [[81,81,82,82,91,91,92,92,93,93,94,94, 80,63,64,30,32,34],[53,87,89],[47,51,52,54]]
				heart = true
			}else if (m_No == 11){ // パイズリなめ音を重視
				textsNo = 1
				vos = [[81,82,84,85,86,91,92,93, 10,44,12,72,61,30],[35],[55,7,8,22]]
			}else if (m_No == 12){ // 媚びパイズリ
				textsNo = 1
				vos = [[81,81,82,82,84,84,85,85,92,92, 62,63,64,78,34,30,56],[23,74],[54,7,8,24]]
				heart = true
			}else if (m_No == 13){ // デーモン１なめ
				textsNo = 1
				vos = [[81,82,83,85,86,91,92,93,79,63,64,66],[89,90],[7,54,55,56]]
				heart = true
			}else if (m_No == 14){ // デーモン２あえ
				textsNo = 1
				vos = [[81,82,91,93,61,62,63,80,64,66,79,30,31,32,33,34],[23,69],[54,55,72,73]]
				heart = true
			}else if (m_No == 1){ //  3,快楽に使う
				// テスト用
				textsNo = 0
				vos = [[70,71,72,73,74,75,76,77,78,79,80,81,82,83],[0],[0]]
				//heart = true
			}else if (m_No == 15){// ファミリア、オーガ 23-28喘ぎ声極小
				textsNo = 0
				vos = [[0,1,2,4,6,7,8,9,10,5,11,12,13,14,15,16,18,19],[52,53,54],[33,34,35,36]]
			}else if (m_No == 16){// ゴブリン、ミミック抑制
				textsNo = 0
				vos = [[2,3,4,6,7,8,9,10,11,12,13,15,16,17,18,19,21],[55,56,59],[37,38,39,40]]
			}else if (m_No == 17){ // 快楽抑え気味　ヘルバードインプ
				textsNo = 0 //
				vos = [[1,3,6,7,9,10,23,24,25,26,27,28,29,30],[58,57],[41,42,46,47]]
			}else if (m_No == 18){ // 33-吐息
				textsNo = 0
				vos = [[33,34,35,36,37,38,39,40,41,46,47],[48,51],[42,43,44,45]]
			}else if (m_No == 19){
				textsNo = 0
				vos = [[33,34,35,36,37,42,43,44,45],[49,50],[35,36,37,40]]
				if (variableId == 1){
					heart = true
				}
			}else if (m_No == 20){// 抑え気味　ゴブリン_55バード
				textsNo = 0
				vos = [[84,85,86,87,88,89,70,71,72,73,75,76,77,78,79,80,81],[91,93],[43,35,38,41]]
				heart = true
			}else if (m_No == 21){// ふつうオーガ_30インプ
				textsNo = 0
				vos = [[70,71,72,73,74,75,76,77,78,79,80,81,82,83],[90,92],[44,36,39,42]]
				heart = true
			}else if (m_No == 22){// 快楽露骨ファミリアミミック_32
				textsNo = 0
				vos = [[60,61,62,63,64,65,66,67,68,69,70,72,73,75,76,77,78,81,83],[94,95],[42,34,37,40]]
				heart = true
			}else if (m_No == 99){ // 90番台は例外にする。
				if ($gameVariables.value(12) == 2 || $gameVariables.value(12) == 3){ // ウロボロス苦痛
					textsNo = 2
					vos = [[56,57,58,59,60,61,62,63,64,65,47,49,51,66,67,68,69],[70,71,72],[30,31,32,33]]
				}else{ // ウロボロス快楽
					textsNo = 2
					heart = true
					vos = [[45,46,47,48,49,50,51,52,53,54,55,25,38,39,40,41,42,43,44],[73,74,75],[12,13,14,15,18,19,20,21,22,23,24]]
				}
			}
			return {"textsNo":textsNo,"heart":heart,"voNo":vos[variableId]}
		}else if (a_No == -100){ // a_Noがマイナスの時は、いろいろなデータについて
			//マイナス100はモンスターの初期データ109876543210 ,40,3666,21
			
			// m_Noは120、121、122などが混在してややこしいので、最後の一桁を0にして統一
			var m10 = Math.floor(m_No / 10)
			m_No = m10 * 10
			
			
			
			
			
			
			if (m_No == 0){ // 本番のは下一桁が0でないとダメなので注意
				//0[M番号(2桁),ソート番号,モンスター名前,ツクールと連動させる番号（共通固定データ基本）
				//1[モン女在庫数,（保留）モンスター数（共通変動データ、在庫や研究費系）
				//2[販売総数,売春総数,販売額売春額総額,販売数rank,売春数rank,総額rank（共通変動データ、販売数系）
				//3[獲得ONか,[調教前ONか,費用],[調教済みON,費用],[獲得判定,費用,スキルNo],
				//[獲得判定,費用,スキルNo],[獲得判定,費用,スキルNo,アイテムタイプ],[,ONか,使役費用],[費用,ON],[費用,ON]（研究系その他）
				//4[表情変化パターン,調教前後吹出番号,あえぎテキスト番号,初期売値,初期人気度％（種別固定データ基本）
				//5[確率動き人数,人気確率,金買う目的金額,売春数修正値（種別固定データ、見えないもの）
				//6[人気度％,販売数,販売額,売春数,売春額（種別変動データ、見るもの）
				//7[確率で動く客数,人気度で動く客数,金で動く客数（種別変動データ、見えないもの）
				//8[今回の販売数,今回の売春数,今回の販売額,％OFF,お相手の男は？,（種別変動データ、販売中のみのもの）
				//9[アニメ表示設定,文章表示設定,色相変更の数値（その他）
				//10[テキスト
				//
				//
				// なし0　？1　音符2　ハート3　怒り4　冷や汗5　くしゃくしゃ6　沈黙7　ひらめき8　ねむり9 huki
				// 
				//
				//
				// ido
				// メモ。研究資金。５階20万ぐらい。６階販売所拡張他消費で３５万ぐらい。毒の道稼げば１００万溜めることができるぐらい
				// 連続処理を数値でやりたいから配列にしたけど、クソややこしい。次回から配列とハッシュ組み合わせてできないか検討する。
			}else if (m_No == 400){ // true,18,5 これ、モンスター研究。18は18万G、5は5体
				var result = [[m10,10,'スライム',20,140],
				[0,0],
				[0,0,0,99,99,99],
				[true,[true,18,48],[false,20,12],[false,4,4,20],[false,10,4,48],[false,30,4,20,1],[false,30,4],[false,60,10],[false,90,10]],//3[(その他)
				[[1,1,2,3,200],[2,3,3,4,200]],
				[[0.35,5,1,0.35],[0.35,5,1,0.35]],
				[[200,0,0,0,0],[180,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 200){ // 
				var result = [[m10,11,'サハギン',13,110],
				[0,0],
				[0,0,0,99,99,99],
				[true,[true,12,17],[false,15,7],[false,5,4,50],[false,40,5,37],[false,20,4,13,1],[false,30,5],[false,60,11],[false,90,15]],//3[(その他)
				[[3,6,5,3,150],[3,3,7,4,140]],
				[[0.39,5.3,1,0.31],[0.39,5,1,0.31]],
				[[150,0,0,0,0],[140,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 220){
				var result = [[m10,12,'インプ',15,110],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,1,3],[false,20,10],[false,5,2,12],[false,15,2,47],[false,15,5,15,1],[false,30,8],[false,60,11],[false,90,15]],//3[(その他)
				[[54,7,17,3,150],[54,3,21,5,150]],
				[[ 0.35, 6, 1, 0.37],[ 0.35, 6, 2, 0.34]],
				[[150,0,0,0,0],[150,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 120){
				var result = [[m10,13,'狂いウサギ',10,110],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,2,3],[false,15,15],[false,20,5,30],[false,90,10,40],[false,20,5,10,1],[false,30,12],[false,60,10],[false,90,22]],//3[(その他)
				[[54,6,4,3,180],[54,7,6,4,120]],
				[[ 0.39, 5, 1, 0.35],[ 0.35, 5.5, 1, 0.32]],
				[[180,0,0,0,0],[120,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 500){
				var result = [[m10,14,'ミミック',23,90],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,2,4],[false,20,10],[false,7,4,13],[false,80,7,86],[false,30,10,26,99],[false,30,6],[false,60,6],[false,90,10]],//3[(その他)
				[[56,7,16,4,140],[52,3,22,5,140]],
				[[ 0.32, 5.2, 1, 0.39],[ 0.31, 5.5, 2, 0.39]],
				[[140,0,0,0,0],[140,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 700){
				var result = [[m10,15,'人喰い草',26,120],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,1,3],[false,20,6],[false,10,4,58],[false,30,5,63],[false,18,4,26,0],[false,30,4],[false,60,10],[false,90,12]],//3[(その他)
				[[3,1,11,4,170],[52,3,12,5,170]],
				[[ 0.39, 5, 1, 0.45],[ 0.35, 5, 2, 0.45]],
				[[170,0,0,0,0],[170,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 330){
				var result = [[m10,16,'ゴブリン',18,110],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,3,4],[false,20,6],[false,5,2,15],[false,12,3,42],[false,35,3,18,0],[false,30,6],[false,60,7],[false,90,8]],//3[(その他)
				[[52,7,16,4,150],[52,3,20,6,150]],
				[[ 0.5, 5, 1, 0.37],[ 0.5, 5, 2, 0.35]],
				[[150,0,0,0,0],[150,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 410){
				var result = [[m10,17,'アシッドスライム',21,135],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,3,4],[false,16,6],[false,5,3,16],[false,20,8,14],[false,50,10,21,1],[false,30,3],[false,60,6],[false,90,8]],//3[(その他)
				[[3,1,2,4,120],[4,3,3,5,120]],
				[[ 0.32, 5, 1, 0.35],[ 0.31, 5, 1, 0.35]],
				[[120,0,0,0,0],[120,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 420){
				var result = [[m10,18,'オーガ',22,90],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,3,3],[false,20,7],[false,50,8,43],[false,100,10,84],[false,50,4,22,0],[false,30,3],[false,60,6],[false,90,12]],//3[(その他)
				[[52,4,15,4,140],[54,3,21,6,190]],
				[[ 0.3, 5, 2, 0.37],[ 0.35, 4.4, 2, 0.41]],
				[[140,0,0,0,0],[190,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 520){
				var result = [[m10,19,'ホーネット',25,120],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,4,3],[false,20,5],[false,55,7,61],[false,20,6,83],[false,30,3,25,1],[false,30,6],[false,60,6],[false,90,6]],//3[(その他)
				[[54,4,5,4,160],[54,7,6,6,130]],
				[[ 0.35, 4, 1, 0.45],[ 0.35, 5, 2, 0.42]],
				[[160,0,0,0,0],[130,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 340){
				var result = [[m10,20,'ヘルバード',19,100],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,4,3],[false,20,5],[false,45,5,54],[false,30,4,53],[false,65,5,19,1],[false,30,6],[false,60,5],[false,90,4]],//3[(その他)
				[[51,6,17,4,150],[3,3,20,6,140]],
				[[ 0.3, 4.5, 1, 0.36],[ 0.35, 4.4, 2, 0.36]],
				[[150,0,0,0,0],[140,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 140){
				var result = [[m10,21,'マミー',11,100],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,4,2],[false,25,5],[false,10,4,32],[false,30,6,34],[false,55,9,11,1],[false,30,6],[false,60,7],[false,90,9]],//3[(その他)
				[[1,9,18,5,130],[2,7,19,7,140]],
				[[ 0.3, 4.1, 1, 0.45],[ 0.3, 4.1, 1, 0.35]],
				[[130,0,0,0,0],[140,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 210){
				var result = [[m10,22,'ケット・シー',14,100],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,4,4],[false,28,5],[false,20,5,33],[false,50,6,56],[false,30,8,14,1],[false,30,5],[false,60,6],[false,90,8]],//3[(その他)
				[[54,6,4,5,180],[54,3,8,7,180]],
				[[ 0.55, 4.2, 1, 0.39],[ 0.5, 4.1, 2, 0.37]],
				[[180,0,0,0,0],[180,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 230){
				var result = [[m10,23,'デーモン',16,110],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,5,3],[false,30,4],[false,100,6,36],[false,100,7,88],[false,90,4,28,99],[false,30,1],[false,60,4],[false,90,6]],//3[(その他)
				[[54,3,13,6,160],[52,2,14,8,170]],
				[[ 0.31, 5.7, 2, 0.53],[ 0.32, 5.8, 2, 0.52]],
				[[160,0,0,0,0],[170,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 300){
				var result = [[m10,24,'ブラッドスライム',17,100],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,5,5],[false,25,5],[false,7,4,52],[false,10,2,38],[false,100,9,17,1],[false,30,2],[false,60,5],[false,90,5]],//3[(その他)
				[[54,1,2,6,120],[52,2,3,9,120]],
				[[ 0.45, 5.1, 2, 0.45],[ 0.44, 5.6, 2, 0.46]],
				[[120,0,0,0,0],[120,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 150){
				var result = [[m10,25,'ミノタウロス',12,90],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,5,4],[false,25,4],[false,50,6,39],[false,100,12,87],[false,100,8,12,1],[false,30,2],[false,60,3],[false,90,7]],//3[(その他)
				[[52,4,5,7,130],[54,3,7,8,140]],
				[[ 0.3, 5.2, 3, 0.44],[ 0.33, 5, 2, 0.47]],
				[[130,0,0,0,0],[140,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 510){
				var result = [[m10,26,'ファミリア',24,110],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,4,3],[false,25,7],[false,60,4,59],[false,50,8,89],[false,90,10,24,0],[false,30,4],[false,60,6],[false,90,8]],//3[(その他)
				[[54,6,15,5,150],[54,3,22,7,150]],
				[[ 0.51, 4.4, 2, 0.37],[ 0.52, 4.3, 3, 0.36]],
				[[150,0,0,0,0],[150,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 710){
				var result = [[m10,27,'ガーディアン',27,90],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,3,3],[false,30,8],[false,4,1,22],[false,100,8,85],[false,110,15,27,0],[false,30,3],[false,60,4],[false,90,6]],//3[(その他)
				[[3,7,9,5,150],[52,3,10,7,170]],
				[[ 0.37, 5, 2, 0.41],[ 0.35, 5.6, 3, 0.39]],
				[[150,0,0,0,0],[170,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 800){
				var result = [[m10,27,'モンスター１',28],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,9,0],[false,17,0],[false,0,0,0],[false,0,0,10],[false,0,0,10,1],[false,0,0],[false,0,0],[false,0,0]],//3[(その他)
				[[1,1,1,5,150],[2,2,2,7,170]],
				[[ 0.37, 5, 1, 0.38],[ 0.35, 5.6, 1, 0.39]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 810){
				var result = [[m10,27,'モンスター２',29],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,9,0],[false,17,0],[false,0,0,0],[false,0,0,10],[false,15,10,1,0],[false,0,0],[false,0,0],[false,0,0]],//3[(その他)
				[[1,1,1,5,150],[2,2,2,7,170]],
				[[ 0.37, 5, 1, 0.38],[ 0.35, 5.6, 1, 0.39]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 820){
				var result = [[m10,27,'モンスター３',30],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,9,0],[false,17,0],[false,0,0,0],[false,0,0,0],[false,15,10,1,0],[false,0,0],[false,0,0],[false,0,0]],//3[(その他)
				[[1,1,1,5,150],[2,2,2,7,170]],
				[[ 0.37, 5, 1, 0.38],[ 0.35, 5.6, 1, 0.39]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 600){ // 理由忘れたけど、ソート番号39 ido
				var result = [[m10,39,'ウロボロス',90,110],
				[1,0],
				[0,0,0,99,99,99],
				[false,[false,44,0],[false,44,0],[false,10,0],[false,20,0],[false,30,0],[false,50,0,67],[false,50,0,68],[false,50,0,69],[false,200,0,70],[false,20,0,71]],
				[[1,0,99,2,150],[2,0,99,7,140]],
				[[ 0.35, 5, 1, 0.35],[ 0.35, 5, 1, 0.35]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 900){
				var result = [[m10,39,'ウロボロス？',91],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,66,0],[false,66,0],[false,0,0,0],[false,0,0,0],[false,15,10,1,0],[false,1,0],[false,2,0],[false,3]],//3[(その他)
				[[1,1,1,2,150],[2,2,2,7,140]],
				[[ 0.35, 5, 1, 0.35],[ 0.35, 5, 1, 0.35]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 910){
				var result = [[m10,39,'ウロボロス！',92],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,66,0],[false,66,0],[false,0,0,0],[false,8,11,0],[false,15,10,1,0],[false,1,0],[false,2,0],[false,3]],//3[(その他)
				[[1,1,1,2,150],[2,2,2,7,140]],
				[[ 0.35, 5, 1, 0.35],[ 0.35, 5, 1, 0.35]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
			}else if (m_No == 0){
				var result = [[m10,99,'a',99],
				[0,0],
				[0,0,0,99,99,99],
				[false,[false,2,0],[false,12,0],[false,0,0,0],[false,8,11,0],[false,15,10,1,0],[false,1,0],[false,2,0],[false,3]],//3[(その他)
				[[1,1,1,2,150],[2,2,2,7,140]],
				[[ 0.35, 5, 1, 0.35],[ 0.35, 5, 1, 0.35]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,0,0],[0,0,0,0,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 
				[['0','1','2','3','4'],['0','1','2','3','4']]
				]
				
			}else if (m_No == 161){
				result[0] = [m_No,112,1000,150]
				result[4] = [0.1,0.2,0.3]
				result[6] = ['冒険者',
				'冒険者',
				'冒険者',
				'奴隷商',
				'奴隷商',
				'奴隷商',
				'変態貴族',
				'変態貴族',
				'変態貴族','独白','独白','独白']
			}else if (m_No == 171){
				result[0] = [m_No,112,1000,150]
				result[4] = [0.1,0.2,0.3]
				result[6] = ['冒険者',
				'冒険者',
				'冒険者',
				'奴隷商',
				'奴隷商',
				'奴隷商',
				'変態貴族',
				'変態貴族',
				'変態貴族','独白','独白','独白']
			}else if (m_No == 181){
				result[0] = [m_No,112,1000,150]
				result[4] = [0.1,0.2,0.3]
				result[6] = ['冒険者',
				'冒険者',
				'冒険者',
				'奴隷商',
				'奴隷商',
				'奴隷商',
				'変態貴族',
				'変態貴族',
				'変態貴族','独白','独白','独白']
			}else if (m_No == 999){
				
			}else if (m_No == 999){
			}else if (m_No == 999){
			}else if (m_No == 999){
			}else if (m_No == 999){
			}else if (m_No == 999){
			}else{
				// ここから
				var result = [[m10,66,'ようじょめがみー',11],
				[200,0],
				[0,0,0,99,99,99],
				[true,[true,1,0],[false,5,0],[false,5,10,0],[false,8,11,0],[false,15,10,1,0],[false,1,0],[false,2,0],[false,3]],//3[(その他)
				[[1,1,1,2,180],[2,2,2,9,180]],
				[[0.35,5,3,0.35],[0.35,5,3,0.35]],
				[[100,0,0,0,0],[100,0,0,0,0]], // 6
				[[0,0,0],[0,0,0]],
				[[0,0,0,40,0],[0,0,0,90,0]],
				[[1,1,0,[0,127,200]],[0,0,0,[0,127,200]]], // 9 ミミックの色相変化200、マミーの100はいい感じ
				[['【幼女神】は気持ち悪いおじさんに犯されている…','「だめっ！だめなのぉ！人間ごときが女神に欲情するなんて許されないのぉっ！」「うるせえ！クソ女神！人間オスチンポのザーメンで孕んで反省しろ！」','「クソガキバカ女神の割にはいい具合だぜ。\n腰が止まんねえ」\n「あっあっあっ…許してなの…\n誰か助けてなの…」','「ふー濃っゆぅいのがたくさん出たぜ」\n「ううう…汚いの…穢されたの…」\n「汚いだと！神聖なおじさんザーメンに何て言い草だ！もっと犯してやる！」\n「いやっ！やなの！もう許して…いやあぁぁぁ…」','へ','ん','な','こ'],['0','1','2','3','4']]
				]
				// ここまで
			}
			return result
			
		}else if (a_No == -200){
var a0 = []
switch (m_No) { // 便宜上m_Noとの表記だが、実際は喘ぎ越えの識別ナンバー
	case 0: // "抑制型あえぎ声" #ほとんど吐息
	a0 = ["ぐっ", "あっ", "うっ", "あ", "は", "ぁ", "ぁぁ", "うぁ", "ふっ", "くっ", "あくっ"];
	break;
	case 1: // "あえぎ声"
	a0 = ["はんっ", "ん…", "ん", "あぁん", "ああんっ", "あん", "あっ", "ああっ", "ふあっ"];
	break;
	case 2: // "泣き声"
	a0 = ["ひぐっ", "ぐすっ", "うぐっ", "えぅ", "あぅ", "う", "うぅ"];
	break;
	case 3: // "下品あえぎ"
	a0 = ["んお", "はひっ", "ひぃん", "お゛っ", "ひっ", "あひぃ", "うっ！", "あへぇ", "ん゛っ"];
	break;
	case 4: // "オーソドックス激しいあえぎ"
	a0 = ["ひあっ", "おっ", "はっ", "あっ", "ふあっ", "あぁっ"];
	break;
	case 5: // "うめき、ぞんび"
	a0 = ["あ゛", "あ゛ー", "う゛", "うあ？", "あぎ？", "？？？", "が！", "うが"];
	break;
	case 6: // "さけび、ぞんび"
	a0 = ["うがー", "が", "ぐあー", "ぎゃー", "がぁっ", "おがっ", "ぐぎっ", "んが"];
	break;
	case 7: // "媚びてるかわいい系"
	a0 = ["やぁ", "ゃっ", "きゃ", "んっ", "もうっ", "んふっ", "んんぅ"];
	break;
	case 8: // "屈辱"
	a0 = ["くっ、ぐっ", "くそっ", "うぐ", "ぐぐ", "くっ", "…っ"];
	break;
	case 9: // "苦痛"
	a0 = ["ひゃっ", "はぐっ", "ぐあっ", "うぐ", "いぎっ", "がっ", "ぐっ", "あ゛っ"];
	break;
	case 10: // "無言
	a0 = ["...", "っ", "ん", "...", "...", "...", "...", "...", "...", "...", "...", "...", "...", "...", "...", "...", "..."];
	break;
	case 11: // "苦悶"出産時
	a0 = ["ぐっ", "うぐ", "いぎっ", "ひぃ", "あ゛", "くっ", "はくっ", "ハァ", "あがっ"];
	break;
	case 12: // 狂気笑い
	a0 = ["あは", "はは", "ふひ", "くは", "あはっ", "ひひ"];
	break;
	case 13: // いきあえぎ絶頂
	a0 = ["～～！", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 14: // 
	a0 = ["ひっ、だめ…、っつ！、あっ、あ゛っ、ひっ、あ゛あ゛、", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 15: //
	a0 = ["はひっ、はっ、あっ、あがっ、らめっ、やめ、やだっ、あぁっ", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 16: // あへ抵抗
	a0 = ["な゛っ、いやっ、やだ、うあ、い゛っ、う゛ぁ、やめ゛っ", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 17: // 
	a0 = ["～～！", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 18: // 呆然フィニッシュ
	a0 = ["ふう、んふぅ、あぁ、あっ、うっ、う、んっ、んぁ、ぁあ、", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 19: // 泣くフィニッシュ
	a0 = ["んっ、ううぅ、ううっ、えぅ、えぐ、", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 20: // あへフィニッシュ
	a0 = ["あ゛～、えう、んむ、んうぅ、えへ、いぐ、あ゛、ふぁ、はひ、あ゛あ゛、ぁあ", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
	case 21: // いきあえぎ絶頂
	a0 = ["～～！", "あぁっ", "っっ！", "ぁっ！", "あぁん", "あっ"];
	break;
}
var r0 = Math.floor(Math.random() * a0.length)
return a0[r0]
			
		}else{
			
		}
	}
	
	
	function get_data(m_No,a_No,variableId) {
		if (a_No == -999){
			// ソートによるモンスターデータの回収。書き込みはできない
			// なるべくソート番号に沿うように並べるが、条件に合わないものは除外
			// ソート番号獲得するデータの種類a_No:-999で使った
			
			// 1が通常で2が調教済みなんだけど、1だけで判別できる
			var arr0 = [120,140,150,200,210,220,230,300,330,340,400,410,420,500,510,520,700,710,600,800,810,820,900,910]
			var result = []
			
			if (variableId == 0){ //すべてのデータを集めて、ソートで並び替える予定
				//for (var i = 0; i < arr0.length; i++) {
				//	result.push($shopVariables.value(arr0[i]))
				//}
				for (var i = 0; i < arr0.length; i++) {
					if ($shopVariables.value(arr0[i])){
						result.push($shopVariables.value(arr0[i]))
					}
				}
			}else if (variableId == 1){ // 捕獲しただけ
				for (var i = 0; i < arr0.length; i++) {
					if ($shopVariables.value(arr0[i])[3][0]){ // ここ間違い。SHOPVALUEにする
						result.push($shopVariables.value(arr0[i]))
					}
				}
			}else if (variableId == 2){ // 通常の女体化しただけ
				for (var i = 0; i < arr0.length; i++) {
					if ($shopVariables.value(arr0[i])[3][1][0]){
						result.push($shopVariables.value(arr0[i]))
					}
				}
			}else if (variableId == 3){ // 女体化した上で、調教もしたものを集めた
				for (var i = 0; i < arr0.length; i++) {
					if ($shopVariables.value(arr0[i])[3][2][0]){
						result.push($shopVariables.value(arr0[i]))
					}
				}
			}else if (variableId == 99){ // arr0の内容を統一するため、別のところで使う時もここから引っ張ってくる
				return arr0
			}
			
			
			result.sort(function(a, b) {
				return a[0][1] - b[0][1];
			});
			return result
			
		}else if (a_No == -600){ // 男などが何か言ってるの
			
			var c0 = m_No
			if (c0 == 29){ // 特殊な処理。ウロボロスなど
				var c1 = 46
				c0 = 0
			}else{
				var c1 = 48 + Math.floor(c0 / 10)
				c0 = c0 % 10
			}
			
			var commonEvent = $dataCommonEvents[c1]
			if (commonEvent && commonEvent.list) {
				var textLines = []
				var j = -1
				for (var i = 0; i < commonEvent.list.length; i++) {
					var command = commonEvent.list[i]
					if (command.code === 105) {
						j++
						// ここに i++ を入れると最初の一行、種類とか描いてるのをを飛ばすための処理になる
						textLines[j] = []
					}else if (command.code === 405) {
						textLines[j].push(commonEvent.list[i].parameters[0])
					}
				}
			}
			//window.alert(c0)
			return textLines[c0]
			
		}else if (a_No !== -100){
			
			
			
			var result = [0,1,2,3,4,5,6,7,5,'zm000000']
			var number = Math.floor(variableId /100)
			var number2 = Math.floor((variableId % 100) / 10)
			var term = 0
			var n1 = Math.floor(m_No / 10) // monsternameこの４つの項目は、普通の時はこのままなので先に定める
			var n2 = m_No % 10             // もし違う時に、下のifで修正する
			var n3 = a_No
			var size0 = 1 // 0小さいサイズ　1大きいサイズ　v変数で変化させるので、後で設定
			var dv1 = Math.floor($gameVariables.value(71) /100) // Hのアニメの間隔について調整
			var dv2 = Math.floor($gameVariables.value(72) /100)
			var dv3 = Math.floor($gameVariables.value(73) /100)
			var dv4 = Math.floor($gameVariables.value(74) /100)
			var dv5 = Math.floor($gameVariables.value(75) /100)
			if (a_No == 2 || a_No == 3){ // 表情タイプ。目、口の変化に関係
				if (n1 == 60){ // 例外の絵の処理3 52 54 56 4 51  :58 58 5 5 58 58
					if ($gameVariables.value(12) == 0){
						var hyou = 58
					}else if ($gameVariables.value(12) == 1){
						var hyou = 60
					}else if ($gameVariables.value(12) == 2){
						var hyou = 7
					}else if ($gameVariables.value(12) == 3){
						var hyou = 58
					}else if ($gameVariables.value(12) == 4){
						var hyou = 56
					}else{
						var hyou = 58
					}
				}else{ // 例外じゃない普通の
					var hyou = get_data3(n1 * 10,-100,variableId)[4][n2 - 1][0]
				}
			}
			
			if (a_No >= 5 && a_No <= 8){ // 喘ぎ制御
				//result = [0,0,0,3,4,5,6,7,5,'zm00000']
				// get_data3(n1 * 10,-100,variableId):-100で素のデータを参照しているので、セーブしているデータを見ていないので注意
				var aegi = get_data3(n1 * 10,-100,variableId)[4][n2 - 1][2] // 喘ぎの種類番号
				var p0 = get_data3(n1 * 10,-100,variableId)[0][4]
				
				if (variableId >= 3500 && variableId < 3570){ // 経過時間によって喘ぎ変える
					var phase = 0
				//	result[8] = 3
				//}else if (number >= dv2 && number < dv3){ // フィニッシュ前
				//	var vodata = get_data3(aegi,-700,0)
				//	result[8] = 3
				}else if (variableId >= 3570 && variableId < 3590){
					var phase = 1
				}else if (variableId >= 3590){
					var phase = 2
					result[8] = 10
					//window.alert(m_No)
				}else{
					var phase = 0
					result[8] = 5
				}
				var vodata = get_data3(aegi,-700,phase)
				Object.assign(vodata,{"texts":get_data3(0,-500,0)[vodata.textsNo],"pitch":p0,"phase":phase});
				result[0] = vodata
				return result
			}else if (m_No == 100){ // アニメーション停止措置の場所。後で修正$gameSwitches.value(21)
				// この場合、土台の絵だけ使う
				result = [0,0,0,0,0,0,0,0,5,'zm000000']
				if (a_No == 1) {
					//var n1 = '00'//暫定処置
					var n2 = '0'
					var n3 = '0'
				}else{ 
					var n1 = '00'
					var n2 = '0'
					var n3 = '0'
				}
			
			}else if (number >= 0 && number < dv1){ // 待機状態huki
				// 使用する絵の処理。termで区分けしようとしたがやめる
				
				result = [0,0,0,0,0,0,0,0,10,'zm000000']
				if (a_No == 1) {
					//var n1 = '00'//暫定処置
					var n2 = '0'
					var n3 = '0' // エッチじゃない方の絵は番号0
					
				}else if (a_No == 2){ 
					var n1 = '00'
					var n2 = '0'
					var n3 = '0'
				}else if (a_No == 3){
					var huki = get_data3(n1*10,-100,variableId)[4][n2 - 1][1]
					if (huki != 0){
						result = [0,1,2,3,4,5,6,7,5,'zm000000']
					}
					var n1 = '00'
					var n2 = huki
					var n3 = '0'
					
				}
			}else if (number >= dv1 && number < dv2){ // 普通にやる
				// 使用する絵の処理。
				
				if (a_No == 1){ // 身体// エロ画像本体肉体、チンコや射精はタイプで変えない
					result = [0,1,2,3,4,5,6,7]// 抜き差し
					var n2 = '0'
				}else if (a_No == 2){ // 目 ido
					if (hyou < 50){
						term = renumber(number, 151, 159) // 101~107~109
					}else{
						term = renumber(number, 101, 109) // 101~107~109
					}
					result = get_data2(term,number2)
					
				}else if (a_No == 3){ // 口
					if (hyou % 2 == 1){
						term = renumber(number + 77, 151, 159) // 101~107~109
					}else{
						term = renumber(number + 77, 101, 109) // 101~107~109
					}
					result = get_data2(term,number2) // 開閉
					
				//}else if (a_No == 4){ // ちん
				//	var n2 = '0'
				//	 result = [0,1,2,3,4,5,6,7]// 抜き差し
				}
				result[8] = 5
			}else if (number >= dv2 && number < dv3){ // フィニッシュ前
				// 使用する絵の処理。
				//1 目を開けて２回瞬き。口を閉じていて、２回開閉
				//2 目を開けて２回瞬き。口を閉じていて、２回開閉
				//3 目を開けて２回瞬き。口を閉じていて、２回開閉
				//4 １かいだけまばたき
				//5 １かいだけまばたき
				//6 はんはん
				//7 はんはん
				//8 目を閉じていて２回瞬き。口を開けていて、２回開閉
				//9 目を閉じていて3回瞬き。口を開けていて、3回開閉

				if (a_No == 1){ // 身体
					result = [0,1,2,3,4,5,6,7]// 抜き差し
					var n2 = '0'
				}else if (a_No == 2){ // 目jump
					term = get_data2(hyou + 1100,number2)[renumber(number, 0, 3)]
					result = get_data2(term,number2)
				}else if (a_No == 3){ // 口
					term = get_data2(hyou + 2100,number2)[renumber(number, 0, 3)]
					result = get_data2(term,number2)
					
				//}else if (a_No == 4){ // ちん
				//	var n2 = '0'
				//	result = [0,1,2,3,4,5,6,7]
					
				}
				result[8] = 3 // 速く動く
			}else if (number >= dv3){ // 射精中 && number < 48
				// 使用する絵の処理。
				if (a_No == 2){ // 目
					if (number < dv4){
						term = get_data2(hyou + 1100,number2)[renumber(number, 4, 7)] // term = renumber(number, 107, 109)
					}else{
						term = get_data2(hyou + 1100,number2)[renumber(number, 8, 11)]
					}
					result = get_data2(term,number2)
					
				}else if (a_No == 3){ // 口
					if (number < dv4){
						term = get_data2(hyou + 2100,number2)[renumber(number, 4, 7)]
					}else{
						term = get_data2(hyou + 2100,number2)[renumber(number, 8, 11)]
					}
					result = get_data2(term,number2)
				}else if (a_No == 1) {
					var n2 = '0'
					
					if (number == dv3){
						term = 201
						var n3 = '1'
						result = get_data2(term,number2)
					}else if (number == dv3 + 1){
						term = 202
						var n3 = '5'
						result = get_data2(term,number2)
					}else if (number >= dv3 + 2 && number < dv4){
						term = 203
						var n3 = '5'
						result = get_data2(term,number2)
					}else if (number == dv4){
						term = 204
						var n3 = '5'
						result = get_data2(term,number2)
					}else if (number >= dv4 + 1){
						term = 205
						var n3 = '5'
						result = get_data2(term,number2)
					}else{
						term = 205
						var n3 = '5'
						result = get_data2(term,number2)
					}
					
				}
				result[8] = 5
			
			}
			
			// 例外の絵の処理
			if (n1 == 60){
				if ($gameVariables.value(12) == 4 || $gameVariables.value(12) == 5){
					n1 = 62
				}else if ($gameVariables.value(12) == 2 || $gameVariables.value(12) == 3){
					n1 = 61
				}else{
					n1 = 60
				}
			}
			result[9] = 'zm' + n1 + n2 + n3 + size0
			return result
		}
		
		
		
		
	}
	
	// モンスターのデータが入っているが、いろいろ。読み取りのみでセーブはできない
	//
	function mons_data() {
	    this.initialize.apply(this, arguments);
	}

	mons_data.prototype.initialize = function() {
	    this.clear();
	};

	mons_data.prototype.clear = function() {
	    this._data = [];
	};

	mons_data.prototype.value = function(m_No,a_No,variableId) {
		this._data[variableId] = get_data(m_No,a_No,variableId)//ここの配列変数が返ってくる
	    return this._data[variableId] || 0;
	};
	
	
// 店で売る事の出来るモンスターのデータが入っている場所。変更とセーブが可能
function Shop_Variables() {
    this.initialize.apply(this, arguments);
}

Shop_Variables.prototype.initialize = function() {
	this.clear()
}
Shop_Variables.prototype.clear = function() {
    this._data = [];
	var arr0 = get_data(-999,-999,99) //[120,140,150,200,210,220,230,300,330,340,400,410,420,500,510,520,700,710,600]
	// 念のため、0を後からプッシュする。バグ予防だが、逆にバグが見えにくくなるかもしれないから今は保留
	// arr0.push(0)
	for (var i = 0; i < arr0.length; i++) {
		this._data[arr0[i] / 10] = get_data3(Math.floor(arr0[i] / 10) * 10,-100,0)// a_Noが-100の時は、いろいろなデータについて
	}
//window.alert(this._data[12][6])
};

Shop_Variables.prototype.todayclear = function() { // 今回の～を全員全部ゼロにする。店頭に並んでなくも、入れ替えとかのがあるから全員0
    var arr0 = get_data(-999,-999,99)
	for (var i = 0; i < arr0.length; i++) {
		for (var j = 0; j <= 1; j++) {
			for (var i0 = 0; i0 <= 2; i0++) { // 3は割引％、4は相手男データなんで消さなくていいかな
				this._data[arr0[i] / 10][8][j][i0] = 0
			}
		}
	}
	
};
Shop_Variables.prototype.clearDB = function() { // 固定パラメータのを初期化
    var arr0 = get_data(-999,-999,99)
	for (var i = 0; i < arr0.length; i++) {
		// 0 3 4 5 9
		this._data[arr0[i] / 10][0] = get_data3(arr0[i],-100,0)[0]
		this._data[arr0[i] / 10][3] = get_data3(arr0[i],-100,0)[3]
		this._data[arr0[i] / 10][4] = get_data3(arr0[i],-100,0)[4]
		this._data[arr0[i] / 10][5] = get_data3(arr0[i],-100,0)[5]
		this._data[arr0[i] / 10][9] = get_data3(arr0[i],-100,0)[9]
	}
	
};
Shop_Variables.prototype.clearDB2 = function() { // 金関連だけ初期化
    var arr0 = get_data(-999,-999,99)
	for (var i = 0; i < arr0.length; i++) {
		if (arr0[i] == 600) continue; // ウロボロス除く
		var a0 = this._data[arr0[i] / 10][3][0]
		var a1 = []
		for (var j = 1; j <= 8; j++) {
			//window.alert(this._data[arr0[i] / 10][3][j])
			a1[j] = this._data[arr0[i] / 10][3][j][0]
		}
		this._data[arr0[i] / 10][3] = get_data3(arr0[i],-100,0)[3]
		this._data[arr0[i] / 10][3][0] = a0
		for (var j = 1; j <= 8; j++) {
			this._data[arr0[i] / 10][3][j][0] = a1[j]
		}
	}
	
};
Shop_Variables.prototype.clearDB3 = function(silv,num) { // silvどのレベルの使役か。678の数値のどれか。num変更額。使役モンスター必要額をへらす
    var arr0 = get_data(-999,-999,99)
	for (var i = 0; i < arr0.length; i++) {
		if (arr0[i] == 600) continue; // ウロボロス除く
		var a0 = this._data[arr0[i] / 10][3][0]
		var a1 = []
		this._data[arr0[i] / 10][3][silv][1] = num
	}
	
};
Shop_Variables.prototype.value = function(m_No) {
	
	var m1 = Math.floor(m_No / 10)
	var m2 = m_No % 10 > 0 ? (m_No % 10) - 1 : 0;
	//if (!(this._data[m1])) {
	//	this._data[m1] = get_data3(m1*10,-100,0)// a_Noが-100の時は、いろいろなデータについて
	//}
	var a0 = this._data[m1]
	var ex_data = [a0[0],a0[1],a0[2],a0[3],a0[4][m2],a0[5][m2],a0[6][m2],a0[7][m2],a0[8][m2],a0[9][m2],a0[10][m2]]
	
    return ex_data;
};

Shop_Variables.prototype.setValue = function(m_No, data0) {
	
	var m1 = Math.floor(m_No / 10)
	var m2 = m_No % 10 > 0 ? (m_No % 10) - 1 : 0;
	var a0 = this._data[m1]
	if (m_No > 0) {
		a0[0] = data0[0]
		a0[1] = data0[1]
		a0[2] = data0[2]
		a0[3] = data0[3]
		a0[4][m2] = data0[4][m2]
		a0[5][m2] = data0[5][m2]
		a0[6][m2] = data0[6][m2]
		a0[7][m2] = data0[7][m2]
		a0[8][m2] = data0[8][m2]
		a0[9][m2] = data0[9][m2]
		a0[10][m2] = data0[10][m2]
		this._data[m1] = a0;
        // 自分用は全部配列にした方が処理しやすいかと思ったけど、思ったより面倒だったので次回から変える
		// あと、ミニアニメ用のデータは、混ぜない形式にした方が良かったかも
		// ツクール本体のモンスター戦闘データも独立させてるし、
		// それぞれ別個でやってからやりやすいように連結すべきなので、次回はそういうやり方で
		
    }
	
};

Shop_Variables.prototype.m_uri = function(m_No,adjust) { // ここを動かすと、モンスターが売れる。adjustは複数のお店に出勤しているか
	var m1 = Math.floor(m_No / 10)
	//window.alert(m_No);
	if (this._data[m1][1][0] < 1){
		return // 在庫なしだと休業
	}
	if (this._data[m1][3][2][0]){
		this.zbaibai(m1,1)
		this.hukk(m1,1,adjust)
	}
	this.zbaibai(m1,0) // 在庫数の関係があるので、売れてほしい高額の調教済みを先に
	this.hukk(m1,0,adjust) // 順番は、人身売買、売春の順で行う
}
	
Shop_Variables.prototype.hukk = function(m1,t_type,adjust = 0) { // ここは売春。人身売買に間接的に影響する
	if (this._data[m1][6][t_type][0] < 2){
		this._data[m1][6][t_type][0] += Math.floor(Math.random() * 4)
		return // 人気度なしだと休業
	}else if (this._data[m1][6][t_type][0] < 50 && this._data[m1][6][t_type][0] > 5){
		if (this._data[m1][6][t_type][0] < Math.floor(Math.random() * 100)){
			return // 人気度低いと頻度減る
		}
	}
	var sale = Math.max(this._data[m1][8][t_type][3], 0) // 割引が売春に与える影響
	var ra0 = this._data[m1][2] // ランキングが売春に与える影響。多分、順位０とか変な数字を使ってるのはないはず・・・
	var ra1 = Math.max(4 - ra0[3], 0) + Math.max(4 - ra0[4], 0) + Math.max(4 - ra0[5], 0)
	// adjustは同一モンスターを複数店舗で売った場合の修正。露骨に影響が出るように派手に入れている
	var kn = Math.pow(this._data[m1][6][t_type][0] * (1 + sale / 300),0.66) * 3.636 * 2 + adjust * 30  + ra1 * 10// 適当な数で割ったりかけたりして修正する。
	var kd = this._data[m1][5][t_type][3] + ra1 / 50
	var total = 0
	for (var i = 0; i <= 2; i++) {
		// keisanでの分散の計算で作った数を求める
		// s1売春人数。計算の際、t_type調教済みは売春数少なくしておく
		var s1 = Math.max(this.keisan(kn,kd) / (1 + t_type * 2), 0) / 10 / 1 // 最後の1の数字は売春数の調整。ここをいろいろやることで、売れすぎを調整できるので保留している
		this._data[m1][7][t_type][i] = Math.floor(Math.min(this._data[m1][7][t_type][i] + s1 / 10 * (1 + sale / 100),30) * 1000) / 1000
		// 確率でもん娘を買う客、人気度で買う客、値段で買う客が今回増えた分を足す。売春の１０分の１が潜在的な奴隷購入層となる
		// ただ、数値上、あんまり１０分の１になっていないので、下を調整する
		total += s1 / 2 
	}
	this._data[m1][2][1] = this.ulit(this._data[m1][2][1] + total)
	this._data[m1][6][t_type][3] = this.ulit(this._data[m1][6][t_type][3] + total)
	this._data[m1][8][t_type][1] = this.ulit(this._data[m1][8][t_type][1] + total)
	this._data[m1][6][t_type][0] = this._data[m1][6][t_type][0] - ((total * 0.05 + 0.05) * (2 - t_type)) // 売春１回で人気度％を減らす。調教済みは人気度減りにくい
	this._data[m1][6][t_type][0] = Math.max(this._data[m1][6][t_type][0], 0)
}
	
Shop_Variables.prototype.zbaibai = function(m1,t_type) {
	if (this._data[m1][6][t_type][0] < 3){
		this._data[m1][6][t_type][0] += Math.floor(Math.random() * 3)
		return // 人気度なしだと休業
	}
	var sale = Math.max(this._data[m1][8][t_type][3] / 25, 0) // 割引が売り上げに与える影響
	var kn = this._data[m1][7][t_type] //客になるかもしれない人の数３種
	var kd = this._data[m1][5][t_type] //客になるかもしれない人の数値データ
	// 最初に、人気度％に影響を受ける人から。別のを先にすると人気度％の減少影響が出てしまう
	var mean = Math.min(this._data[m1][6][t_type][0] * (kd[1] / 2000), 0.7) //kd[1]はおよそ200で、少ないほど効果が強い
	var number = this.keisan(kn[1],mean)
	this.baibaiDatain(m1,t_type,number,1)
	number = this.keisan(kn[0],kd[0]) // 日常的に確率で買っていく人
	this.baibaiDatain(m1,t_type,number,0)
	// 割引時の計算、で、客の割合変動
	
	if (this._data[m1][8][t_type][3] == 0){
		number = 0 // 割引で買う客は、割引がない時は全く買わない
	}else{
		var a0 = ((100 - this._data[m1][8][t_type][3]) / 100) * this._data[m1][4][t_type][3]//値段。
		mean = kd[2] / a0  // 目的値段/現在値段
		number = this.keisan(kn[2],Math.min(0.7 * mean,0.7) * (1 + sale * 0.01))
		var a1 = this._data[m1][4][t_type][3] - kd[2] // 最大に売れそうな値。ゼロ徐さんにはならないはず
		var a2 = this._data[m1][4][t_type][3] - a0 // 現在の。a2/a1は売れそう割合
		if (a2/a1 <= Math.random()){
			number = 0
		}
	
	}
	
	this.baibaiDatain(m1,t_type,number,2)
}
	
Shop_Variables.prototype.keisan = function(number,mean) {
	var n = Math.floor(number / 3) // 元の数値より大きくなるパターンが多いので、ゲームバランス上調整
	var variance = 0.3; // 分散
	var total = n * mean;
	var stand = Math.sqrt(n * variance);
	var result = Math.round(total + stand * Math.sqrt(12) * (2 * Math.random() - 1))
	return (result >= 0 ? result : 0)
}
	
Shop_Variables.prototype.baibaiDatain = function(m_type,t_type,number,kyaku) { // モン種、調教タイプ、販売人数、
	
	if (Math.floor(this._data[m_type][1][0]) < number){
		number = Math.floor(this._data[m_type][1][0]) // 在庫チェック 
	}
	if (this._data[m_type][7][t_type][kyaku] < number){
		number = Math.floor(this._data[m_type][7][t_type][kyaku]) // 乱数が元の数を上回った場合や負の値になった場合
	} 
	if (number < 1){
		number = 0
	}else if (number >= 2){
		number = 1
	}
	this._data[m_type][1][0]         = this.ulit(this._data[m_type][1][0] - number) // 在庫減らす
	this._data[m_type][7][t_type][kyaku] = this.ulit(this._data[m_type][7][t_type][kyaku] - number) // 買う予定客層が買った客になるので、その分減らす
	this._data[m_type][2][0]         = this.ulit(this._data[m_type][2][0] + number) // 販売人数調教前後総数を加算
	this._data[m_type][6][t_type][1] = this.ulit(this._data[m_type][6][t_type][1] + number) // 販売人数総数を加算
	this._data[m_type][8][t_type][0] = this.ulit(this._data[m_type][8][t_type][0] + number) // 販売人数今回のみを加算
	var mouke = number * (100 - this._data[m_type][8][t_type][3]) / 100 * this._data[m_type][4][t_type][3]
	// ここに、支払われる報酬合計をｖ変数に入れたりする処理
	$gameVariables.setValue(7,$gameVariables.value(7) + mouke)
	
	this._data[m_type][2][2]         = this.ulit(this._data[m_type][2][2] + mouke)
	this._data[m_type][6][t_type][2] = this.ulit(this._data[m_type][6][t_type][2] + mouke)
	this._data[m_type][8][t_type][2] = this.ulit(this._data[m_type][8][t_type][2] + mouke)
	// 人気度％をここで減らす。微調整するかも
	var nani = $gameVariables.value(62)
	if (nani == 4) { // イージー、ベリーイージー
		nani = 5
	}
	this._data[m_type][6][t_type][0] = this._data[m_type][6][t_type][0] - number * (2.4 - 0.2 * nani)
	this._data[m_type][6][t_type][0] = Math.max(this._data[m_type][6][t_type][0], 0) //人気度の最低値。ゲームバランスで調整
}
	
	// 数値の上限を決める。さらに、小数点以下は２桁まで。お金以外を適用this.ulit()。後から見るとMath.minでよくね？
Shop_Variables.prototype.ulit = function(data) {//ULlimitだと長いので略称で
	var result = data
	if (data > 999999) {
		result = 999999
	}else if (data < -999999) {
		result = -999999
	}
	return Number(result.toFixed(2))
}
	// 整数を足して、範囲をはみ出ていたら修正する
	// これは特に+＝の整数のみ対応で３けた。-999から999まで。直接変数の内容を変える。
	//上限決める方法、こっちの方がやりやすいんだろうか。よくわからん
Shop_Variables.prototype.kasan = function(data,m_type,type0,type1,type2 = 99) { // type2 = 99は変数の次元が違うのも扱っているからその処理
	var result = data
	if (type2 == 99) {
		result += this._data[m_type][type0][type1]
	}else{
		result += this._data[m_type][type0][type1][type2]
	}
	
	// -999から999までをはみ出ていたら修正
	if (result > 999) {
		result = 999
	}else if (result < -999) {
		result = -999
	}
	
	if (type2 == 99) {
		this._data[m_type][type0][type1] = result 
	}else{
		this._data[m_type][type0][type1][type2] = result
	}
	
}
Shop_Variables.prototype.zaikoplus = function() {
	var arr0 = get_data(-999,-999,99)
	// 納品数調整。ｖ変数62難易度変更の時に代入して変化させる、納品数の数の変数
	if ($gameVariables.value(1) < 9){
		var r0 = Math.random() * 0.7
	}else if ($gameVariables.value(1) >= 13){
		var r0 = Math.random() * 0.4
	}else if ($gameVariables.value(1) >= 17){
		var r0 = Math.random() * 0.5
	}else if ($gameVariables.value(1) >= 23){
		var r0 = Math.random() * 0.6
	}else if ($gameVariables.value(1) >= 28){
		var r0 = Math.random() * 0.7
	}else{
		var r0 = Math.random() * 0.3
	}
	
	if ($gameVariables.value(62) == 0){
		var nouhin = 1 + r0
	}else{
		var nouhin = 0.8 + r0 + $gameVariables.value(62) * 0.47
	}
	for (var i = 0; i < arr0.length; i++) { // モンスター数１８予定だけど、増やした時に注意
		// これ、112・・・の並びの変数をそのまま使っているので、注意が必要
		var item_v = $gameParty.numItems($dataItems[30 + i])
		if (item_v > 0){
			this.kasan(Math.floor(item_v * nouhin * 100) / 100,arr0[i] / 10,1,0)
			this._data[arr0[i] / 10][3][0] = true
		}
		$gameParty.loseItem($dataItems[30 + i],99)
	}
	
}

Shop_Variables.prototype.ureta_on = function() {
	// その回に売れたモンスターがいるかどうかの判定
	for (var i = 1; i <= 8; i++) { 
		var n0 = $gameVariables.value(50 + i)
		if (!(n0) || n0 == 0){ // 存在しない場合は除外
			
		}else if (this._data[Math.floor(n0 / 10)][8][0][0] >= 1){
			$gameSwitches.setValue(40 + i,true)
		}
	}
	
}
Shop_Variables.prototype.saoyaku = function(m_No) { // アニメの竿役を決める。調教前後で竿役の数が違うと、バグになるので注意。
	var m1 = Math.floor(m_No / 10)
	// 竿役をランダム変更。今のところ竿役数が2
	var sa = Math.floor(Math.random() * 2)
	this._data[m1][8][1][4] = sa
	this._data[m1][8][0][4] = sa
	if (this._data[m1][8][1][0] > 0){ // 調教済みが売れた時は、販売所のモンスターと竿役は調教済み
		return (m1 * 10 + 2)
	}else{
		return (m1 * 10 + 1)
	}
	
}
	
	
	
	
	// テキストのデータ集まとめ
function Text_Datas() {
    //this.initialize.apply(this, arguments);
}
Text_Datas.prototype.data = function(number) {
	//モンスターNo2桁+(0:調教前、1:調教後)+テキスト番号2桁
	//もしくは、３桁の一般文章、２桁であえぎ声
	// 仕方なくマネージャーの方に描き込んでいるが、煩雑さを避けるため、テキストは別にする
	var result = ''
	if (number == 0) {
		result = '黒闇天'
    }else if (number == 999) {
    	result = 'んくっ…人間の…部分が…こんなに気持ちいいとは知らなかったのじゃ…。そこ、そこをもっと突いてほしいのじゃ。きもちいい。いい。ああっ。あっああっ。もっと激しくほしいのじゃ。んんっ奥をされると、もう。ああっ！'
    }
	return result
};
	





