//=============================================================================
// CGGalleryScroll.js
//=============================================================================

/*:
 * @plugindesc 回想実装プラグイン V1.0.1
 * @target MV
 * @author NJ
 *
 * @param StartMapID
 * @text タイトル起動時のマップID
 * @type number
 * @min 1
 * @default 1
 * @desc タイトル画面から回想を開いた際に使用するマップIDを指定します。
 *
 * @param ShowGalleryCommand
 * @text タイトルに回想表示
 * @type boolean
 * @default true
 * @desc タイトルコマンドに「回想」を追加するかどうかを設定します。
 *
 * @param ConfirmSE
 * @text 解放サムネイル決定音
 * @type file
 * @dir audio/se
 * @desc 解放済みサムネイルをクリックした時に鳴る効果音。

 * @param BuzzerSE
 * @text 未解放サムネイル禁止音
 * @type file
 * @dir audio/se
 * @desc 未解放サムネイルをクリックした時に鳴る効果音。

 * @param GalleryBGM
 * @text 回想BGM
 * @type file
 * @dir audio/bgm
 * @desc 回想中に流れるBGM。
 *
 * @param GalleryBGMVolume
 * @text 回想BGM音量
 * @type number
 * @default 90
 * @desc 回想BGMの音量を設定します。
 *
 * @param Columns
 * @text 表示する列数
 * @type number
 * @min 1
 * @default 3
 * @desc サムネイルを1ページに表示する列数

 * @param Rows
 * @text 表示する行数
 * @type number
 * @min 1
 * @default 3
 * @desc サムネイルを1ページに表示する行数
 *
 * @param ReturnSwitchID
 * @text 回想終了スイッチ ID
 * @type switch
 * @desc スイッチが ON の場合は元のゲーム画面に戻り、OFF の場合はタイトル画面に戻ります。
 * @default 0
 *
 * @param ExitButtonImage
 * @text 終了ボタン画像
 * @type file
 * @dir img/system
 * @desc 終了ボタンに使用する画像を指定します。指定がない場合はデフォルトのボタンを表示します。
 *
 * @param ExitButtonX
 * @text 終了ボタン X 座標
 * @type number
 * @min 0
 * @default 400
 * @desc 終了ボタンの X 座標を指定します。
 *
 * @param ExitButtonY
 * @text 終了ボタン Y 座標
 * @type number
 * @min 0
 * @default 500
 * @desc 終了ボタンの Y 座標を指定します。
 *
 * @param ThumbnailWidth
 * @text サムネイル幅
 * @type number
 * @min 1
 * @default 150
 *
 * @param ThumbnailHeight
 * @text サムネイル高さ
 * @type number
 * @min 1
 * @default 100
 *
 * @param StartX
 * @text サムネイル開始位置X
 * @type number
 * @min 0
 * @default 50
 *
 * @param StartY
 * @text サムネイル開始位置Y
 * @type number
 * @min 0
 * @default 50
 *
 * @param Spacing
 * @text サムネイル間のスキマ
 * @type number
 * @min 0
 * @default 20
 *
 * @param BackgroundImage
 * @text 背景画像
 * @type file
 * @dir img/pictures
 * @default
 *
 * @param LockedThumbnail
 * @text 未解放サムネイル画像
 * @type file
 * @dir img/pictures
 *
 * @param RecollectionList
 * @text 回想リスト
 * @type struct<Recollection>[]
 * 
 * @help
 * このプラグインはスクロールビューで回想ギャラリーを実装するためのシンプルなものです。
 *
 * プラグインコマンド:
 *   OpenGallery - 回想を開きます。
 *   ResetGallerySwitches - 保存しているスイッチを全てリセットします。
 *
 * バージョン
 * V1.0.1 データの保存場所を変更。
 * V1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~Recollection:
 * @param CommonEventID
 * @text コモンイベントID
 * @type common_event
 *
 * @param UnlockSwitchID
 * @text 解放スイッチID
 * @type switch
 *
 * @param Thumbnail
 * @text サムネイル画像
 * @type file
 * @dir img/pictures
 */

(function () {
    const parameters = PluginManager.parameters('CGGalleryScroll');
    const thumbnailWidth = Number(parameters['ThumbnailWidth'] || 150);
    const thumbnailHeight = Number(parameters['ThumbnailHeight'] || 100);
    const startX = Number(parameters['StartX'] || 50);
    const startY = Number(parameters['StartY'] || 50);
    const spacing = Number(parameters['Spacing'] || 20);
    const columns = Number(parameters['Columns'] || 3);
    const rows = Number(parameters['Rows'] || 3);
    const lockedThumbnail = parameters['LockedThumbnail'] || '';
    const recollectionList = JSON.parse(parameters['RecollectionList'] || '[]').map(item => JSON.parse(item));
    const backgroundImage = parameters['BackgroundImage'] || '';
	const returnSwitchID = Number(parameters['ReturnSwitchID'] || 0);
    const exitButtonImage = parameters['ExitButtonImage'] || '';
    const exitButtonX = Number(parameters['ExitButtonX'] || 400);
    const exitButtonY = Number(parameters['ExitButtonY'] || 500);
	const startMapID = Number(parameters['StartMapID'] || 1);
    const confirmSE = parameters['ConfirmSE'] || '';
    const buzzerSE = parameters['BuzzerSE'] || '';
    const galleryBGM = parameters['GalleryBGM'] || '';
    const galleryBGMVolume = Number(parameters['GalleryBGMVolume'] || 90);
    const showGalleryCommand = parameters['ShowGalleryCommand'] === 'true';
	let _savedBgmInfo = null;

    let isCGModeEnabled = false;
    let isReturningToGallery = false;

    const CG_UNLOCK_SWITCH_KEY = 'CGGalleryUnlockSwitches';


    const _Game_Switches_setValue = Game_Switches.prototype.setValue;
    Game_Switches.prototype.setValue = function (switchId, value) {
        _Game_Switches_setValue.call(this, switchId, value);

        if (isGalleryUnlockSwitch(switchId)) {
            saveUnlockSwitches();
        }
    };

    function isGalleryUnlockSwitch(switchId) {
        return recollectionList.some(item => Number(item.UnlockSwitchID || 0) === switchId);
    }

    function loadUnlockSwitches() {
        const fs = require('fs');
        const path = getCGGallerySavePath();
        if (fs.existsSync(path)) {
            try {
                const data = fs.readFileSync(path, 'utf8');
                return JSON.parse(data);
            } catch (e) {
                return {};
            }
        } else {
            return {};
        }
    }

    function saveUnlockSwitches() {
        const fs = require('fs');
        const path = getCGGallerySavePath();
        const switchesToSave = {};

        recollectionList.forEach(item => {
            const switchId = Number(item.UnlockSwitchID || 0);
            if (switchId > 0) {
                switchesToSave[switchId] = $gameSwitches.value(switchId);
            }
        });

        try {
            fs.writeFileSync(path, JSON.stringify(switchesToSave));
        } catch (e) {}
    }

    function getCGGallerySavePath() {
        const fs = require('fs');
        const path = require('path');
        const base = path.resolve(StorageManager.localFilePath(1), '..');
        return path.join(base, 'CGGalleryUnlockSwitches.json');
    }

    function enableCGMode() {
        if (!isCGModeEnabled) {
            isCGModeEnabled = true;
            restoreUnlockSwitches();
        }
    }

    function restoreUnlockSwitches() {
        const savedSwitches = loadUnlockSwitches();
        Object.keys(savedSwitches).forEach(switchId => {
        $gameSwitches.setValue(Number(switchId), savedSwitches[switchId]);
        });
    }

    function disableCGMode() {
        if (isCGModeEnabled) {
            isCGModeEnabled = false;
            saveUnlockSwitches();
        }
    }

    function Scene_CGGallery() {
        this.initialize.apply(this, arguments);
    }

    Scene_CGGallery.prototype = Object.create(Scene_Base.prototype);
    Scene_CGGallery.prototype.constructor = Scene_CGGallery;

    Scene_CGGallery.prototype.initialize = function () {
        Scene_Base.prototype.initialize.call(this);
        this._scrollY = 0;
        this._thumbnails = [];
		this._scrollBar = null;
	    this._isDraggingBar = false;
        this._dragStartY = 0;
        this._scrollStartY = 0;
        enableCGMode(); 
    };

    Scene_CGGallery.prototype.create = function () {
        Scene_Base.prototype.create.call(this);
        this.createBackground();
        this.createThumbnails();
        this.createExitButton();
		this.createScrollBar();

		if (galleryBGM) {
            AudioManager.playBgm({ name: galleryBGM, volume: galleryBGMVolume, pitch: 100, pan: 0 });
        }
    };

    Scene_CGGallery.prototype.createScrollBar = function () {
        this._scrollBarBase = new Sprite();
        this._scrollBarBase.bitmap = new Bitmap(12, Graphics.height - startY * 2);
        this._scrollBarBase.bitmap.fillAll('gray');
        this._scrollBarBase.x = Graphics.width - 30;
        this._scrollBarBase.y = startY;
        this.addChild(this._scrollBarBase);

        this._scrollBar = new Sprite();
        this._scrollBar.bitmap = new Bitmap(12, 100);
        this._scrollBar.bitmap.fillAll('white'); 
        this._scrollBar.x = this._scrollBarBase.x; 
        this._scrollBar.y = startY;
        this.addChild(this._scrollBar);
    };

    Scene_CGGallery.prototype.createExitButton = function () {
        this._exitButton = new Sprite_Button();

        if (exitButtonImage) {
            this._exitButton.bitmap = ImageManager.loadSystem(exitButtonImage);
        } else {
            this._exitButton.bitmap = new Bitmap(120, 40);
            this._exitButton.bitmap.fillAll('red');
            this._exitButton.bitmap.drawText('終了', 0, 0, 120, 40, 'center');
        }

        this._exitButton.x = exitButtonX;
        this._exitButton.y = exitButtonY;
        this._exitButton.setClickHandler(this.onExitButtonClicked.bind(this));
        this.addChild(this._exitButton);
    };

    Scene_CGGallery.prototype.onExitButtonClicked = function () {
        $gameTemp._returnToGallery = false;

        if (returnSwitchID > 0 && $gameSwitches.value(returnSwitchID)) {
            SceneManager.pop();
        } else {
           disableCGMode();
           SceneManager.goto(Scene_Title);
        }
    };

    Scene_CGGallery.prototype.terminate = function () {
        Scene_Base.prototype.terminate.call(this);

        if (_savedBgmInfo) {
            AudioManager.stopBgm();
            setTimeout(() => { 
                AudioManager.playBgm({
                    name: _savedBgmInfo.name,
                    volume: _savedBgmInfo.volume,
                    pitch: _savedBgmInfo.pitch,
                    pan: _savedBgmInfo.pan
                });
            }, 100);
        }

        if (!isReturningToGallery) {
            disableCGMode();
        }

         resetScreenEffects();
    };

    Scene_CGGallery.prototype.createBackground = function () {
        if (backgroundImage) {
            this._backgroundSprite = new Sprite();
            this._backgroundSprite.bitmap = ImageManager.loadPicture(backgroundImage);
            this.addChild(this._backgroundSprite);
        } else {
            this._backgroundSprite = new Sprite();
            this._backgroundSprite.bitmap = new Bitmap(Graphics.width, Graphics.height);
            this._backgroundSprite.bitmap.fillAll('#000000');
            this.addChild(this._backgroundSprite);
        }
    };

    Scene_CGGallery.prototype.createThumbnails = function () {
        this._thumbnails = [];

        recollectionList.forEach((item, index) => {
            const switchId = Number(item.UnlockSwitchID || 0);
            const isUnlocked = $gameSwitches.value(switchId);

            const x = startX + (index % columns) * (thumbnailWidth + spacing);
            const y = startY + Math.floor(index / columns) * (thumbnailHeight + spacing);

            const thumbnailImage = isUnlocked ? item.Thumbnail : lockedThumbnail;
            const sprite = new Sprite_CGThumbnail(thumbnailImage, x, y, item);
            this.addChild(sprite);
            this._thumbnails.push(sprite);
        });
    };

    Scene_CGGallery.prototype.update = function () {
        Scene_Base.prototype.update.call(this);
        this.updateScroll();
        this.updateBarDragging();

        if (TouchInput.isTriggered()) {
            const x = TouchInput.x;
           const y = TouchInput.y;

            this._thumbnails.forEach(sprite => {
                if (sprite.hitTest(x, y)) {
                    sprite.activate();
                }
            });
        }
    };

    Scene_CGGallery.prototype.updateBarDragging = function () {
        const bar = this._scrollBar;

        if (bar) {
            if (TouchInput.isPressed()) {
                if (this._isDraggingBar) {
                    const deltaY = TouchInput.y - this._dragStartY;
                    const maxScroll = Math.max(0, this.calculateTotalHeight() - Graphics.height + thumbnailHeight + spacing);
                    const maxBarMove = Graphics.height - bar.height - startY * 2;

                    const scrollRatio = deltaY / maxBarMove;
                    this._scrollY = Math.max(0, Math.min(this._scrollStartY + scrollRatio * maxScroll, maxScroll));
                    this.updateScroll();
                } else if (this.isTouchingBar(TouchInput.x, TouchInput.y)) {
                    this._isDraggingBar = true;
                    this._dragStartY = TouchInput.y;
                    this._scrollStartY = this._scrollY;
                }
            } else {
                this._isDraggingBar = false;
            }
        }
    };

    Scene_CGGallery.prototype.isTouchingBar = function (x, y) {
        const bar = this._scrollBar;
        return (
            x >= bar.x &&
            x <= bar.x + bar.width &&
            y >= bar.y &&
            y <= bar.y + bar.height
        );
    };

    Scene_CGGallery.prototype.updateScroll = function () {
        const maxScroll = Math.max(0, this.calculateTotalHeight() - Graphics.height + thumbnailHeight + spacing);

        if (TouchInput.wheelY > 0) {
                this._scrollY = Math.min(this._scrollY + 20, maxScroll);
            } else if (TouchInput.wheelY < 0) {
                this._scrollY = Math.max(this._scrollY - 20, 0);
            }

        this._thumbnails.forEach(sprite => {
            sprite.y = sprite._originalY - this._scrollY;
        });

        this.updateScrollBar(maxScroll);
    };

    Scene_CGGallery.prototype.updateScrollBar = function (maxScroll) {
        if (this._scrollBar) {
            const visibleHeight = Graphics.height - startY * 2;
            const totalHeight = this.calculateTotalHeight();

            const barHeight = Math.max(visibleHeight * (visibleHeight / totalHeight), 20);
            this._scrollBar.bitmap.clear();
            this._scrollBar.bitmap = new Bitmap(12, barHeight); 
            this._scrollBar.bitmap.fillAll('white'); 

            const scrollRatio = this._scrollY / maxScroll;
            const maxBarMove = visibleHeight - barHeight;
            this._scrollBar.y = startY + scrollRatio * maxBarMove;
        }
    };

    Scene_CGGallery.prototype.calculateTotalHeight = function () {
        const totalRows = Math.ceil(recollectionList.length / columns);
        return totalRows * (thumbnailHeight + spacing);
    };

    function Sprite_CGThumbnail(imageName, x, y, item) {
        this.initialize.apply(this, arguments);
    }

    Sprite_CGThumbnail.prototype = Object.create(Sprite.prototype);
    Sprite_CGThumbnail.prototype.constructor = Sprite_CGThumbnail;

    Sprite_CGThumbnail.prototype.initialize = function (imageName, x, y, item) {
        Sprite.prototype.initialize.call(this);
        this.bitmap = ImageManager.loadPicture(imageName);
        this.x = x;
        this.y = y;
        this._originalY = y;
        this._commonEventId = Number(item.CommonEventID || 0);
        this._unlockSwitchId = Number(item.UnlockSwitchID || 0);

        this.bitmap.addLoadListener(() => {
            this.scale.x = thumbnailWidth / this.bitmap.width;
            this.scale.y = thumbnailHeight / this.bitmap.height;
        });
    };

    Sprite_CGThumbnail.prototype.hitTest = function (x, y) {
        return (
            x >= this.x &&
            x <= this.x + thumbnailWidth &&
            y >= this.y &&
            y <= this.y + thumbnailHeight
        );
    };

    Sprite_CGThumbnail.prototype.activate = function () {
        if (!isCGModeEnabled) {
            return;
        }

        if (!$gameSwitches.value(this._unlockSwitchId)) {
	        if (buzzerSE) {
                AudioManager.playSe({ name: buzzerSE, volume: 90, pitch: 100, pan: 0 });
            } else {
                SoundManager.playBuzzer();
			}
            return;
        }

        if (this._commonEventId > 0) {
            AudioManager.stopBgm();
			if (confirmSE) {
                AudioManager.playSe({ name: confirmSE, volume: 90, pitch: 100, pan: 0 });
            } else {
				SoundManager.playOk();
			}

            $gameTemp.reserveCommonEvent(this._commonEventId);

            if ($gameTemp._isFromTitle) {
                $gameTemp._isFromTitle = false;
                DataManager.setupNewGame();
            }

            $gameTemp._returnToGallery = true;
            isReturningToGallery = true;

            isCGModeEnabled = true;

            SceneManager.goto(Scene_Map);
        }
    };

    const _Game_Interpreter_setup = Game_Interpreter.prototype.setup;
    Game_Interpreter.prototype.setup = function (list, eventId) {
        _Game_Interpreter_setup.call(this, list, eventId);

        if (isCGModeEnabled && this._list) {
            this._cgStartIndex = null;
            this._cgEndIndex = null;

            this._list.forEach((command, index) => {
                if (command.code === 356 && command.parameters[0]) {
                    const args = command.parameters[0].split(' ');
                    if (args[0] === 'CG' && args[1] === 'Start') {
                        this._cgStartIndex = index + 1;
                    }
                    if (args[0] === 'CG' && args[1] === 'End') {
                        this._cgEndIndex = index;
                    }
                }
            });

            if (this._cgStartIndex === null || this._cgEndIndex === null) {
            } else {
            }
        }
    };

    const _Game_Interpreter_executeCommand = Game_Interpreter.prototype.executeCommand;
    Game_Interpreter.prototype.executeCommand = function () {
        if (isCGModeEnabled && this._cgStartIndex !== null && this._cgEndIndex !== null) {
            if (this._index < this._cgStartIndex) {
                this._index++;
                return true;
            }
            if (this._index >= this._cgEndIndex) {
                this.terminateCGPlayback();
                return false;
            }

            if (this._index === this._cgEndIndex) {
                this.terminateCGPlayback();
                return false;
            }
        }

        return _Game_Interpreter_executeCommand.call(this);
    };

    Game_Interpreter.prototype.terminateCGPlayback = function () {
        this._index = this._list.length;

        $gameTemp._returnToGallery = true;
        isReturningToGallery = true;
		resetScreenEffects();
        SceneManager.push(Scene_CGGallery);
        this._cgStartIndex = null;
        this._cgEndIndex = null;
        isReturningToGallery = false;
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);

        if ($gameTemp._returnToGallery && !$gameMap.isEventRunning()) {
            $gameTemp._returnToGallery = false;
            SceneManager.push(Scene_CGGallery);
			if (galleryBGM) {
                AudioManager.playBgm({ name: galleryBGM, volume: galleryBGMVolume, pitch: 100, pan: 0 });
            }
            isReturningToGallery = false;
        }
    };

    const _Window_TitleCommand_makeCommandList = Window_TitleCommand.prototype.makeCommandList;
    Window_TitleCommand.prototype.makeCommandList = function () {
        _Window_TitleCommand_makeCommandList.call(this);
		if (showGalleryCommand) {
            this.addCommand('回想', 'gallery');
		}
    };

    const _Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function () {
        _Scene_Title_createCommandWindow.call(this);
        this._commandWindow.setHandler('gallery', this.commandGallery.bind(this));
    };

    Scene_Title.prototype.commandGallery = function () {
        SceneManager._scene.startFadeOut(30, false); 
		DataManager.setupNewGame();
        $gamePlayer.reserveTransfer(startMapID, 0, 0, 0, 0);
		SceneManager.goto(Scene_Map); 
        SceneManager.push(Scene_CGGallery);
    };

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === 'OpenGallery') {
			enableCGMode();
            if (AudioManager._currentBgm) {
                _savedBgmInfo = {
                    name: AudioManager._currentBgm.name,
                    volume: AudioManager._currentBgm.volume,
                    pitch: AudioManager._currentBgm.pitch,
                    pan: AudioManager._currentBgm.pan
                };
            }
            SceneManager.push(Scene_CGGallery);
        }
		
        if (command === 'ResetGallerySwitches') {
            const fs = require('fs');
            const path = getCGGallerySavePath();
            try {
                if (fs.existsSync(path)) fs.unlinkSync(path);
            } catch (e) {
                console.error("回想スイッチ削除失敗:", e);
            }
        }
    };

    function resetScreenEffects() {
        $gameScreen.clearTone();
        $gameScreen.changeWeather(0, 0, 0);
        $gameScreen.clearPictures();
        $gameScreen.startFadeIn(30);
        AudioManager.stopBgm();
        AudioManager.stopBgs();
        AudioManager.stopMe();
    }

})();
