//=============================================================================
// CustomScroll.js
//=============================================================================

/*:
 * @plugindesc スクロールバーを表示プラグイン v1.1.0
 * @author NJ
 *
 * @param MaxValueVarID
 * @text 最大値の変数ID
 * @type variable
 * @desc スクロールの最大値を格納する変数ID
 * @default 1
 *
 * @param MinValueVarID
 * @text 最小値の変数ID
 * @type variable
 * @desc スクロールの最小値を格納する変数ID
 * @default 2
 *
 * @param ScrollPositionVarID
 * @text スクロール位置保存用変数ID
 * @type variable
 * @desc 現在のスクロール位置を保存する変数ID
 * @default 3
 *
 * @param SmoothScroll
 * @text なめらかスクロール
 * @type boolean
 * @desc ON: なめらかスクロール / OFF: 区分ごとに移動
 * @default true
 *
 * @param ScrollScripts
 * @text スクロール時スクリプト一覧
 * @type struct<ScrollScript>[]
 * @desc 条件別に実行するスクリプトの設定
 * @default []
 *
 * @help
 * 使用例:
 * プラグインコマンド:
 *   CustomScroll show x y height
 *   CustomScroll hide
 *
 * イベントスクリプト:
 *   CustomScroll.show(x, y, height);
 *   CustomScroll.hide();
 *
 * バージョン
 * v1.0.0 初回
 * v1.1.0 複数のスクリプトを登録可能にしました。
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~ScrollScript:
 * @param Condition
 * @text 実行条件
 * @desc "value" を使った式や $gameSwitches.value(10) === true など自由に。
 * @default value >= 50
 *
 * @param Script
 * @text 実行スクリプト
 * @type note
 * @desc 条件を満たした時に実行するJSコード。例: console.log("～");
 * @default []
 */

(function() {
    const parameters = PluginManager.parameters('CustomScroll');
    const maxVarID = Number(parameters['MaxValueVarID'] || 1);
    const minVarID = Number(parameters['MinValueVarID'] || 2);
    const scrollVarID = Number(parameters['ScrollPositionVarID'] || 3);
    const smoothScroll = parameters['SmoothScroll'] === "true";

    const scrollScripts = (() => {
        try {
            const rawData = parameters['ScrollScripts'] || '[]';
            const dataArray = JSON.parse(rawData);

            return dataArray.map(data => {
                try {
                    const parsed = JSON.parse(data);
                    parsed.Script = parsed.Script
                        .replace(/\\n/g, '\n')
                        .replace(/\\"/g, '"')
                        .replace(/\\\\/g, '\\');

                    return {
                        Condition: parsed.Condition,
                        Script: parsed.Script
                    };
                } catch (e) {
                    console.error("【CustomScroll】個別データのパースエラー:", e);
                    return null;
                }
            }).filter(Boolean);
        } catch (e) {
            console.error("【CustomScroll】ScrollScripts全体の読み込みエラー:", e);
            return [];
        }
    })();

    class Sprite_Scrollbar extends Sprite {
        constructor(x, y, height) {
            super();
            this.x = x;
            this.y = y;
            this._height = height;
            this._dragging = false;
            this._scrollRatio = 0;

            this.createScrollbar();
            this.updateThumbSize();
            this.updateBarPosition();
        }

        createScrollbar() {
            this.bitmap = new Bitmap(10, this._height);
            this.bitmap.fillAll("rgba(0, 0, 0, 0.5)");
            this._bar = new Sprite();
            this.addChild(this._bar);
        }

        updateThumbSize() {
            const maxVal = $gameVariables.value(maxVarID);
            const minVal = $gameVariables.value(minVarID);
            const range = Math.max(1, maxVal - minVal);
            const viewSize = this._height;

            let newHeight = Math.floor(viewSize * (1 / (range + 1)));
            if (isNaN(newHeight) || newHeight <= 0 || newHeight > viewSize) {
                newHeight = Math.max(20, viewSize / 3);
            }
            this._bar.bitmap = new Bitmap(10, newHeight);
            this._bar.bitmap.fillAll("rgba(255, 255, 255, 0.8)");
            this._barHeight = newHeight;
        }

        update() {
            super.update();
            this.updateDrag();
            this.updateWheel();
            this.updateThumbSize();
        }

        updateDrag() {
            if (TouchInput.isPressed()) {
                if (!this._dragging && this.isTouchingBar(TouchInput.x, TouchInput.y)) {
                    this._dragging = true;
                    this._dragOffset = TouchInput.y - (this._bar.y + this.y);
                }
                if (this._dragging) {
                    const newY = TouchInput.y - this._dragOffset - this.y;
                    const newRatio = Math.max(0, Math.min(1, newY / (this._height - this._barHeight)));
                    this.setScrollRatio(newRatio);
                }
            } else {
                this._dragging = false;
            }
        }

        updateWheel() {
            if (!this._dragging) {
                const maxVal = $gameVariables.value(maxVarID);
                const minVal = $gameVariables.value(minVarID);
                const range = Math.max(1, maxVal - minVal);
                const scrollStep = 1 / range;
                const delta = Math.sign(TouchInput.wheelY);

                if (delta !== 0) {
                    let newRatio = this._scrollRatio + delta * scrollStep;
                    newRatio = Math.max(0, Math.min(1, newRatio));
                    this.setScrollRatio(newRatio);
                    TouchInput._wheelY = 0;
                }
            }
        }

        isTouchingBar(x, y) {
            const realX = this.x + this._bar.x;
            const realY = this.y + this._bar.y;
            return (
                x >= realX && x <= realX + this._bar.width &&
                y >= realY && y <= realY + this._bar.height
            );
        }

        setScrollRatio(newRatio) {
            this._scrollRatio = newRatio;
            const maxVal = $gameVariables.value(maxVarID);
            const minVal = $gameVariables.value(minVarID);
            const value = Math.round(minVal + newRatio * (maxVal - minVal));

            $gameVariables.setValue(scrollVarID, value);

            this.updateBarPosition();

            if (this._lastValue !== value) {
                this.executeScripts(value);
                this._lastValue = value;
            }
        }

        executeScripts(currentValue) {
            scrollScripts.forEach(scriptData => {
                try {
                    const value = currentValue; 
                    const condition = scriptData.Condition;
                    let script = scriptData.Script;

                    if (!!eval(condition)) {
                        script = script
                            .replace(/^\"|\"$/g, '')
                            .replace(/\\n/g, '\n')
                            .replace(/\\"/g, '"')
                            .replace(/\\\\/g, '\\');
                        eval(script);
                    }
                } catch (e) {
                    console.error("【CustomScroll】スクリプト実行エラー:", e);
                    console.error("条件:", scriptData.Condition);
                    console.error("スクリプト:", scriptData.Script);
                }
            });
        }

        updateBarPosition() {
            this._bar.y = this._scrollRatio * (this._height - this._barHeight);
        }
    }

    const CustomScroll = {
        _scrollbar: null,
        _visible: false,

        show(x, y, height) {
            if (this._scrollbar) {
                this.hide();
            }
            this._scrollbar = new Sprite_Scrollbar(x, y, height);
            this._visible = true;
            this.refreshLayer();
        },

        hide() {
            if (this._scrollbar) {
                const pictureLayer = SceneManager._scene._spriteset._pictureContainer;

                if (pictureLayer && pictureLayer.children.includes(this._scrollbar)) {
                    pictureLayer.removeChild(this._scrollbar);
                } else {
                    console.warn("【CustomScroll】スクロールバーが見つかりませんでした。");
                }

                this._scrollbar = null;
                this._visible = false;
            }
        },

        refreshLayer() {
            if (this._scrollbar && this._visible) {
                const pictureLayer = SceneManager._scene._spriteset._pictureContainer;
                if (pictureLayer && !pictureLayer.children.includes(this._scrollbar)) {
                    pictureLayer.addChild(this._scrollbar);
                }
            }
        },

        update() {
            if (this._scrollbar) {
                this._scrollbar.update();
            }
        }
    };

    window.CustomScroll = CustomScroll;

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        CustomScroll.update();
        CustomScroll.refreshLayer();
    };

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === "CustomScroll") {
            if (args[0] === "show") {
                const x = Number(args[1]) || 0;
                const y = Number(args[2]) || 0;
                const height = Number(args[3]) || 200;
                CustomScroll.show(x, y, height);
            }
            if (args[0] === "hide") {
                CustomScroll.hide();
            }
        }
    };
})();
