//=============================================================================
// MoveActionManager.js
//=============================================================================

/*:
 * @plugindesc イベント行動管理プラグイン v1.5.1 - JSONエスケープ解除対応
 * @target MZ, MV
 * 
 * @param Actions
 * @text 行動リスト
 * @type struct<Action>[]
 * @desc イベントの行動パターンを定義するリストです。
 * @default []
 * 
 * @help
 * このプラグインは、プラグインパラメータで定義した行動を
 * 自律移動の「移動ルートの設定」から呼び出せるようにします。
 * 
 * === 使用方法 ===
 * 1. プラグインパラメータ「行動リスト」に行動を登録する。
 * 2. 移動ルートのスクリプトコマンドで次のように呼び出す:
 *      MoveAction: プレイヤーを追尾
 * 
 * === 例 ===
 * if (characterRef.posX() === $gamePlayer.x && characterRef.posY() === $gamePlayer.y) {
 *     $gameSwitches.setValue(1, true);
 * } else {
 *     characterRef.moveTowardPlayer();
 * }
 * 
 * 利用規約:
 * このプラグインは自由に使用、改変、再配布可能です。
 */

/*~struct~Action:
 * @param Name
 * @text 行動名
 * @desc この行動の名前を設定します (例: プレイヤーを追尾)
 * 
 * @param Script
 * @text 実行スクリプト
 * @type note
 * @desc 行動として実行されるスクリプト (例: this.moveTowardPlayer();)
 * @default ""
 */

(() => {
    const pluginName = 'MoveActionManager';
    const parameters = PluginManager.parameters(pluginName);

    /**
     * JSONエスケープ文字を解除する関数
     * @param {string} text - エスケープ解除する文字列
     * @returns {string} - エスケープ解除後の文字列
     */
    function parseScript(text) {
        return text
            .replace(/^"|"$/g, '')   // 文字列の両端のダブルクォートを削除
            .replace(/\\n/g, '\n')    // 改行を正常な改行文字に変換
            .replace(/\\"/g, '"')     // エスケープされたクォートを元に戻す
            .replace(/\\\\/g, '\\');  // エスケープされたバックスラッシュを元に戻す
    }

    const actions = JSON.parse(parameters['Actions'] || '[]').map(data => {
        try {
            const action = JSON.parse(data);
            return {
                Name: action.Name,
                Script: parseScript(action.Script)
            };
        } catch (e) {
            console.error(`【${pluginName}】行動リストの読み込みエラー:`, e);
            return null;
        }
    }).filter(Boolean);

    class MoveActionManager {
        static initialize() {
            this._actions = {};
            actions.forEach(action => {
                this._actions[action.Name] = action.Script;
            });
        }

        static execute(actionName, character) {
            if (!this._actions[actionName]) {
                console.warn(`【${pluginName}】「${actionName}」という行動は定義されていません。`);
                return;
            }

            const script = this._actions[actionName];
            try {
                const characterRef = character;
                character._moveActionScript = script;  // 実行するスクリプトを保存
            } catch (e) {
                console.error(`【${pluginName}】「${actionName}」の実行中にエラーが発生しました:`, e);
            }
        }
    }

    MoveActionManager.initialize();
    window.MoveActionManager = MoveActionManager;

    // 自律移動のスクリプトコマンドをフック
    const _Game_Character_processMoveCommand = Game_Character.prototype.processMoveCommand;
    Game_Character.prototype.processMoveCommand = function(command) {
        if (command.code === Game_Character.ROUTE_SCRIPT) {
            const script = command.parameters[0];
            if (script.startsWith("MoveAction: ")) {
                const actionName = script.replace("MoveAction: ", "").trim();
                MoveActionManager.execute(actionName, this);
                return;
            }
        }
        _Game_Character_processMoveCommand.call(this, command);
    };

    // 自律移動の更新処理でスクリプトを実行する
    const _Game_Character_updateRoutineMove = Game_Character.prototype.updateRoutineMove;
    Game_Character.prototype.updateRoutineMove = function() {
        if (this._moveActionScript) {
            try {
                const characterRef = this;
                eval(this._moveActionScript);  // 自律移動の処理をフレームごとに実行
            } catch (e) {
                console.error(`【${pluginName}】スクリプト実行エラー:`, e);
            }
        }
        _Game_Character_updateRoutineMove.call(this);
    };
})();
