//=============================================================================
// NMessageSkip.js
//=============================================================================

/*:
 * @plugindesc メッセージスキップ専用
 * @author NJ
 *
 * @param EnableSkipSwitch
 * @desc 指定したスイッチがONのときのみスキップ機能が動作します。
 * 0の場合、常に有効になります。
 * @default 0
 * @type switch
 *
 * @param SkipSwitch
 * @desc 指定したスイッチがONのときは強制スキップ
 * @default 0
 * @type switch
 *
 * @param SkipKey
 * @desc スキップを有効にするキー (デフォルト: control)
 * @default control
 * @type select
 * @option shift
 * @option control
 * @option tab
 *
 * @param ToggleSkipKey
 * @desc スキップのON/OFFを切り替えるキー (デフォルト: S)
 * @default S
 * @type select
 * @option shift
 * @option control
 * @option tab
 * @option S
 *
 * @param SkipButton
 * @desc 画面上にスキップボタンを表示するか
 * @default true
 * @type boolean
 *
 * @param SkipButtonX
 * @desc スキップボタンのX座標
 * @default 700
 * @type number
 *
 * @param SkipButtonY
 * @desc スキップボタンのY座標
 * @default 500
 * @type number
 *
 * @param SkipSpeedMultiplier
 * @desc スキップ中のイベント進行速度 (1:等倍, 2:2倍, ... 5まで推奨)
 * @default 2
 * @type number
 * @min 1
 * @max 5
 *
 * @help
 * 【機能一覧】
 * - Controlキーを押しっぱなしでメッセージスキップ
 * - SキーでスキップのON/OFF切り替え
 * - スキップ中はメッセージウィンドウの透明度を変更
 * - クリックやOKキーでスキップ解除
 * - スキップボタンでクリック操作にも対応
 * - 指定スイッチでスキップを強制ONにできる
 * - スキップの進行速度を倍速調整可能
 * - 指定スイッチがONのときのみスキップ機能を有効化
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('NMessageSkip');
    var enableSkipSwitchId = Number(parameters['EnableSkipSwitch'] || 0);
    var skipSwitchId = Number(parameters['SkipSwitch'] || 0);
    var skipKey = String(parameters['SkipKey'] || 'control').toLowerCase();
    var toggleSkipKey = String(parameters['ToggleSkipKey'] || 'S').toLowerCase();
    var showSkipButton = parameters['SkipButton'] === 'true';
    var skipButtonX = Number(parameters['SkipButtonX'] || 700);
    var skipButtonY = Number(parameters['SkipButtonY'] || 500);
    var skipSpeedMultiplier = Math.max(1, Number(parameters['SkipSpeedMultiplier'] || 2));

    var isSkipMode = false;
    var isFastSkipping = false;

    function canUseSkip() {
        return enableSkipSwitchId === 0 || $gameSwitches.value(enableSkipSwitchId);
    }

    function toggleSkip() {
        if (!canUseSkip()) return;
        isSkipMode = !isSkipMode;
        //console.log("[DEBUG] スキップモード: " + (isSkipMode ? "ON" : "OFF"));
    }

    function stopSkip() {
        isSkipMode = false;
        isFastSkipping = false;
        //console.log("[DEBUG] スキップ解除");
    }

    function checkSkipCondition() {
        return canUseSkip() && (Input.isPressed(skipKey) || isSkipMode || (skipSwitchId > 0 && $gameSwitches.value(skipSwitchId)));
    }

    var _Window_Message_update = Window_Message.prototype.update;
    Window_Message.prototype.update = function() {
        _Window_Message_update.call(this);
        if (!canUseSkip()) return;

        var prevFastSkipping = isFastSkipping;
        isFastSkipping = checkSkipCondition();

        if (!isFastSkipping && prevFastSkipping) {
            //console.log("[DEBUG] スキップ解除");
        }

        if (Input.isTriggered(toggleSkipKey)) {
            toggleSkip();
        }

        if (Input.isTriggered("ok") || TouchInput.isTriggered()) {
            stopSkip();
        }

    };

    var _Window_Message_updateInput = Window_Message.prototype.updateInput;
    Window_Message.prototype.updateInput = function() {
        if (!canUseSkip()) return _Window_Message_updateInput.call(this);

        if (isFastSkipping) {
            if (this.isAnySubWindowActive()) return false;
            if (this.pause) {
                this.pause = false;
                this.terminateMessage();
            }
            return this.isTriggered();
        }
        return _Window_Message_updateInput.call(this);
    };

    var _Scene_Map_updateMain = Scene_Map.prototype.updateMain;
    Scene_Map.prototype.updateMain = function() {
        if (!canUseSkip()) {
            _Scene_Map_updateMain.call(this);
            return;
        }

        var loops = isFastSkipping ? skipSpeedMultiplier : 1;
        for (var i = 0; i < loops; i++) {
            _Scene_Map_updateMain.call(this);
        }
    };

    if (showSkipButton) {
        var _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
        Scene_Map.prototype.createAllWindows = function() {
            _Scene_Map_createAllWindows.call(this);
            this.createSkipButton();
        };

        Scene_Map.prototype.createSkipButton = function() {
            this._skipButton = new Sprite_Button();
            this._skipButton.bitmap = new Bitmap(100, 40);
            this._skipButton.bitmap.fillRect(0, 0, 100, 40, "#000000");
            this._skipButton.bitmap.drawText("スキップ", 0, 0, 100, 40, "center");
            this._skipButton.x = skipButtonX;
            this._skipButton.y = skipButtonY;
            this._skipButton.setClickHandler(toggleSkip);
            SceneManager._scene.addChild(this._skipButton);
        };
    }
})();

