//=============================================================================
// NovelGamePlus.js
//=============================================================================

/*:
 * @plugindesc ログ追加プラグイン V1.1.0
 * @author NJ
 * 
 * @help
 * このプラグインはログ機能を追加します。
 * 
 *
 * 使用例:
 * 【プラグインコマンド】
 * LogStart          : ログ記録を開始します。
 * LogStop           : ログ記録を停止します。
 * LogClear          : ログを削除します。
 * ShowLog           : メッセージログを開きます。
 * SetLogName [名前]  : 次のメッセージブロックの名前を設定します。
 *
 * 【スクリプトコマンド】
 * this.pluginCommand("LogStart", []);
 * - ログ記録開始。
 * 
 * this.pluginCommand("LogStop", []);
 * - ログ記録停止。
 *
 * this.pluginCommand("LogClear", []);
 * - ログクリア。
 * 
 * SceneManager.push(Scene_MessageLog);
 * - ログ呼び出し。
 *
 * バージョン
 * v1.0.0 初回
 * v1.1.0 MPP_MessageEX の制御文字に対応
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    let isLogging = false;
    let messageLogBuffer = [];
    let currentMessageBlock = { name: "", messages: [], voice: null };

    /**
     * プラグインコマンドの追加
     */
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        switch (command) {
            case "LogStart":
                isLogging = true;
                break;

            case "LogStop":
                isLogging = false;
                break;

            case "LogClear":
                $gameMessage.clearMessageLog();
                break;

            case "ShowLog":
                SceneManager.push(Scene_MessageLog);
                break;

            case "SetLogName":
                if (isLogging) {
                    const name = args.join(" ");
                    currentMessageBlock.name = `【${name}】`;
                }
                break;
        }
    };

    /**
     * メッセージウィンドウ終了時の処理
     */
    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function () {
        if (isLogging) {
            if (currentMessageBlock.messages.length > 0 || currentMessageBlock.name || currentMessageBlock.voice) {
                messageLogBuffer.push(currentMessageBlock);
            }
            currentMessageBlock = { name: "", messages: [], voice: null };
        }
        _Window_Message_terminateMessage.call(this);
    };

    /**
     * メッセージの追加処理
     */
    const _Game_Message_add = Game_Message.prototype.add;
    Game_Message.prototype.add = function (text) {
        if (isLogging) {
            const processedText = processControlCharacters(text);
            currentMessageBlock.messages.push(processedText);

            if (currentMessageBlock.name === "" && processedText.includes("【")) {
                const nameMatch = processedText.match(/【.+?】/);
                if (nameMatch) {
                    currentMessageBlock.name = nameMatch[0];
                }
            }

            if (!currentMessageBlock.voice && processedText.includes("\\SV[")) {
                const voiceMatch = processedText.match(/\\SV\[(.+?)\]/);
                if (voiceMatch) {
                    currentMessageBlock.voice = voiceMatch[1];
                }
            }
        }
        _Game_Message_add.call(this, text);
    };

    /**
     * 再帰的に制御文字を変換する関数
     */
    function processControlCharacters(text) {
        text = text.replace(/\\SV\[((?:\\[vV]\[(\d+)\]|\\[nN]\[(\d+)\]|[^\\\]]+)+)\]/g, (_, content) => {
            let resolved = content;

            resolved = resolved.replace(/\\[vV]\[(\d+)\]/g, (_, varId) => {
                return String($gameVariables.value(Number(varId)));
            });

            resolved = resolved.replace(/\\[nN]\[(\d+)\]/g, (_, actorId) => {
                const actor = $gameActors.actor(Number(actorId));
                return actor ? actor.name() : "";
            });

            if (isLogging && resolved) {
                currentMessageBlock.voice = resolved;
            }
            return "";
        });

        text = text.replace(/\\n\[(\d+)\]/g, (_, actorId) => {
            const actor = $gameActors.actor(Number(actorId));
            return actor ? actor.name() : "";
        });

        text = text.replace(/\\N\[(\d+)\]/g, (_, actorId) => {
            const actor = $gameActors.actor(Number(actorId));
            return actor ? actor.name() : "";
        });

        text = text.replace(/\\SV\[(.+?)\]/g, (_, rawContent) => {
            const resolvedContent = processControlCharacters(rawContent);
            if (isLogging) currentMessageBlock.voice = resolvedContent;
            return "";
        });

        text = text.replace(/\\NW\[(.+?)\]/g, (_, content) => {
            if (isLogging) currentMessageBlock.name = `【${content.trim()}】`;
            return "";
        });

        text = text.replace(/\\SET\[\d+\]/g, () => {
            if (isLogging) {
                currentMessageBlock.customSet = "";
            }
            return "";
        });

        const controlPatterns = [
            /\\SP\[\d+\]/g, 
            /\\AT\[\d+\]/g, 
            /\\CO\[.+?\]/g, 
            /\\RB\[.+?,.+?\]/g, 
            /\\MX\[-?\d+\]/g, 
            /\\MY\[-?\d+\]/g, 
            /\\PX\[-?\d+\]/g, 
            /\\PY\[-?\d+\]/g, 
            /\\SW\[\d+\]/g, 
            /\\WE/g, 
            /\\A/g, 
            /\\ES/g, 
            /\\DF/g, 
            /\\SV/g, 
            /\\LD/g, 
            /\\FO\[\d+\]/g, 
        ];

        controlPatterns.forEach(pattern => {
            text = text.replace(pattern, "");
        });

        text = text.replace(/\\C\[(\d+)\]/g, "");
        text = text.replace(/\\C\[\d+,\d+,\d+\]/g, "");
        text = text.replace(/\\C\[\d+,\d+,\d+,\d+\]/g, "");
        text = text.replace(/\\FS\[\d+\]/g, "");
        text = text.replace(/\\FB/g, "");
        text = text.replace(/\\FI/g, "");
        text = text.replace(/\\OP\[\d+\]/g, "");
        text = text.replace(/\\OC\[\d+\]/g, "");
        text = text.replace(/\\OW\[\d+\]/g, "");
        text = text.replace(/\\RC\[\d+\]/g, "");
        text = text.replace(/\\RC\[\d+,\d+,\d+\]/g, "");
        text = text.replace(/\\RC\[\d+,\d+,\d+,\d+\]/g, "");

        text = text.replace(/\\SN\[(\d+)\]/g, (_, skillId) => {
            const skill = $dataSkills[Number(skillId)];
            return skill ? skill.name : "";
        });
        text = text.replace(/\\SIN\[(\d+)\]/g, (_, skillId) => {
            const skill = $dataSkills[Number(skillId)];
            return skill ? `\\I[${skill.iconIndex}]${skill.name}` : "";
        });
        text = text.replace(/\\IN\[(\d+)\]/g, (_, itemId) => {
            const item = $dataItems[Number(itemId)];
            return item ? item.name : "";
        });
        text = text.replace(/\\IIN\[(\d+)\]/g, (_, itemId) => {
            const item = $dataItems[Number(itemId)];
            return item ? `\\I[${item.iconIndex}]${item.name}` : "";
        });
        text = text.replace(/\\WN\[(\d+)\]/g, (_, weaponId) => {
            const weapon = $dataWeapons[Number(weaponId)];
            return weapon ? weapon.name : "";
        });
        text = text.replace(/\\WIN\[(\d+)\]/g, (_, weaponId) => {
            const weapon = $dataWeapons[Number(weaponId)];
            return weapon ? `\\I[${weapon.iconIndex}]${weapon.name}` : "";
        });
        text = text.replace(/\\AN\[(\d+)\]/g, (_, armorId) => {
            const armor = $dataArmors[Number(armorId)];
            return armor ? armor.name : "";
        });
        text = text.replace(/\\AIN\[(\d+)\]/g, (_, armorId) => {
            const armor = $dataArmors[Number(armorId)];
            return armor ? `\\I[${armor.iconIndex}]${armor.name}` : "";
        });

        return text;
    }

    /**
     * メッセージログの取得・クリア
     */
    Game_Message.prototype.getMessageLog = function () {
        return messageLogBuffer;
    };

    Game_Message.prototype.clearMessageLog = function () {
        messageLogBuffer = [];
    };

    /**
     * メッセージログ表示シーン
     */
    function Scene_MessageLog() {
        this.initialize(...arguments);
    }

    Scene_MessageLog.prototype = Object.create(Scene_Base.prototype);
    Scene_MessageLog.prototype.constructor = Scene_MessageLog;

    Scene_MessageLog.prototype.initialize = function () {
        Scene_Base.prototype.initialize.call(this);
    };

    Scene_MessageLog.prototype.create = function () {
        Scene_Base.prototype.create.call(this);
        this.createBackground();
        this.createLogWindow();
        this.createReturnButton();
		this._logWindow.scrollToBottom();
    };

    Scene_MessageLog.prototype.createBackground = function () {
        this._backgroundSprite = new Sprite();
        this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
        this._backgroundSprite.opacity = 192;
        this.addChild(this._backgroundSprite);
    };

    Scene_MessageLog.prototype.createLogWindow = function () {
        this._logWindow = new Window_MessageLog();
        this.addChild(this._logWindow);
    };

    Scene_MessageLog.prototype.createReturnButton = function () {
        this._returnButton = new Sprite_Button();
        this._returnButton.bitmap = new Bitmap(200, 50);
        this._returnButton.bitmap.fillRect(0, 0, 200, 50, "#000000");
        this._returnButton.bitmap.drawText("戻る", 0, 0, 200, 50, "center");
        this._returnButton.x = Graphics.boxWidth / 2 - 100;
        this._returnButton.y = Graphics.boxHeight - 80;
        this._returnButton.setClickHandler(() => SceneManager.pop());
        this.addChild(this._returnButton);
    };

    /**
     * メッセージログウィンドウ
     */
    function Window_MessageLog() {
        this.initialize(...arguments);
    }

    Window_MessageLog.prototype = Object.create(Window_Base.prototype);
    Window_MessageLog.prototype.constructor = Window_MessageLog;

    Window_MessageLog.prototype.initialize = function () {
        const width = Graphics.boxWidth - 50;
        const height = Graphics.boxHeight - 150;
        Window_Base.prototype.initialize.call(this, 25, 25, width, height);
        this._scrollY = 0;
        this._interactiveButtons = [];
        this.refresh();
    };

    Window_MessageLog.prototype.refresh = function () {
        this.contents.clear();
        this._interactiveButtons = [];
        const messages = $gameMessage.getMessageLog();
        this.contents.fontSize = 22;

        const leftWidth = 200;
        const textStartX = leftWidth + 20;
        let y = -this._scrollY;

        messages.forEach((messageBlock) => {
            let name = messageBlock.name || "";
            let messageLines = messageBlock.messages.map(processControlCharacters);

            if (name) {
                if (y + this.lineHeight() > 0 && y < this.height) {
                    this.contents.fontSize = 26;
                    this.changeTextColor(this.textColor(6));
                    this.drawText(name, 0, y, leftWidth, "center");
                    this.resetTextColor();

                    if (messageBlock.voice) {
                        y += this.lineHeight();
                        this.changeTextColor(this.textColor(14));
                        this.drawText("▶ 再生", 0, y, leftWidth, "center");
                        this.resetTextColor();

                        const buttonRect = { x: 0, y: y, width: leftWidth, height: this.lineHeight() };
                        this.addInteractiveButton(buttonRect, () => {
                            AudioManager.playVoice({
                                name: messageBlock.voice,
                                volume: 100,
                                pitch: 100,
                                pan: 0,
                            });
                        });
                    }
                }
                y += this.lineHeight();
            }

            messageLines.forEach((line) => {
                if (y + this.lineHeight() > 0 && y < this.height) {
                    const autoBrokenLine = applyAutoLineBreak(line);
                    this.drawTextEx(autoBrokenLine, textStartX, y);
                }
                y += this.lineHeight();
            });

            y += this.lineHeight();
        });

        this._contentHeight = Math.max(y + this._scrollY, this.height);
    };

    Window_MessageLog.prototype.scrollUp = function (amount) {
        this._scrollY = Math.max(0, this._scrollY - amount);
        this.refresh();
    };

    Window_MessageLog.prototype.scrollDown = function (amount) {
        const maxScroll = Math.max(0, this._contentHeight - this.contentsHeight());
        this._scrollY = Math.min(maxScroll, this._scrollY + amount);
        this.refresh();
    };

    Window_MessageLog.prototype.addInteractiveButton = function (rect, callback) {
        this._interactiveButtons.push({ rect, callback });
    };

    Window_MessageLog.prototype.update = function () {
        Window_Base.prototype.update.call(this);

        if (TouchInput.isTriggered()) {
            const x = TouchInput.x - this.x;
            const y = TouchInput.y - this.y;
            for (const button of this._interactiveButtons) {
                const { rect, callback } = button;
                if (x >= rect.x && x < rect.x + rect.width && y >= rect.y && y < rect.y + rect.height) {
                    callback();
                    break;
                }
            }
        }

        const scrollAmount = 40;
        if (TouchInput.wheelY > 0) {
            this.scrollDown(scrollAmount);
        } else if (TouchInput.wheelY < 0) {
            this.scrollUp(scrollAmount);
        }
    };

    function applyAutoLineBreak(text) {
        const parameters = PluginManager.parameters('AutoLineBreak');
        const maxChars = Number(parameters['MaxCharsPerLine'] || 20);
        const insertAutoBreakSpace = String(parameters['InsertAutoBreakSpace']) === 'true';
        const nwActive = /\\[Nn][Ww]\[/.test(text);

        let count = 0;
        let result = '';
        for (let i = 0; i < text.length; i++) {
            const c = text[i];
            if (c === '\n') {
                count = 0;
                result += c;
                continue;
            }
            result += c;
            count++;
            if (count >= maxChars) {
                result += '\n';
                if (insertAutoBreakSpace && nwActive) {
                    result += '　';
                }
                count = 0;
            }
        }
        return result;
    }

	Window_MessageLog.prototype.scrollToBottom = function () {
        const maxScroll = Math.max(0, this._contentHeight - this.contentsHeight());
        this._scrollY = maxScroll;
        this.refresh();
    };

    window.Scene_MessageLog = Scene_MessageLog;

})();
