//=============================================================================
// NrEasyNumber.js
//=============================================================================

/*:
 * @plugindesc 数値スプライト表示 v1.0.0
 * @author NJ
 *
 * @param StatusDisplays
 * @type struct<StatusDisplay>[]
 * @desc 表示設定
 * @default []
 *
 * バージョン履歴：
 * v1.0.0 初版公開
 *
 * 利用規約：
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~StatusDisplay:
 * @param label
 * @type string
 *
 * @param variableId
 * @type variable
 *
 * @param x
 * @type number
 *
 * @param y
 * @type number
 *
 * @param fontSize
 * @type number
 * @default 20
 *
 * @param align
 * @type select
 * @option left
 * @option center
 * @option right
 * @default left
 *
 * @param maxDigits
 * @type number
 * @default 3
 *
 * @param visibleSwitchId
 * @type switch
 * @default 0
 *
 * @param disableSwitches
 * @type switch[]
 * @default []
 *
 * @param variableConditions
 * @type struct<VarCondition>[]
 * @default []
 */

/*~struct~VarCondition:
 * @param variableId
 * @type variable
 *
 * @param operator
 * @type select
 * @option ==
 * @option >=
 * @option <=
 * @option <
 * @default ==
 *
 * @param value
 * @type string
 * @default ""
 */

(function() {
    const pluginName = "NrEasyNumber";

    function parseStructArray(param) {
        const raw = PluginManager.parameters(pluginName)[param];
        if (!raw) return [];
        try {
            return JSON.parse(raw).map(str => JSON.parse(str));
        } catch (e) {
            console.error(`[${pluginName}] Failed to parse struct array "${param}":`, e);
            return [];
        }
    }

    const configs = parseStructArray("StatusDisplays");

    class StatusDisplaySprite extends Sprite {
        constructor(cfg) {
            super(new Bitmap(200, 40));
            this._cfg = cfg;
            this._label = cfg.label || "";
            this._variableId = Number(cfg.variableId);
            this._fontSize = Number(cfg.fontSize || 20);
            this._align = cfg.align || "left";
            this._maxDigits = Number(cfg.maxDigits || 3);
            this._visibleSwitchId = Number(cfg.visibleSwitchId || 0);

            try {
                const raw = cfg.disableSwitches;
                if (typeof raw === "string") {
                    this._disableSwitches = JSON.parse(raw).map(Number);
                } else if (Array.isArray(raw)) {
                    this._disableSwitches = raw.map(Number);
                } else {
                    this._disableSwitches = [];
                }
            } catch (e) {
                console.warn(`[${pluginName}] 無効スイッチ parse error:`, e);
                this._disableSwitches = [];
            }

            try {
                const raw = cfg.variableConditions;
                const arr = typeof raw === "string" ? JSON.parse(raw) : raw;
                this._varConds = Array.isArray(arr) ? arr.map(item => {
                    const obj = typeof item === "string" ? JSON.parse(item) : item;
                    return {
                        variableId: Number(obj.variableId),
                        operator: obj.operator,
                        value: obj.value
                    };
                }) : [];
            } catch (e) {
                console.error(`[${pluginName}] variableConditions parse failed`, e);
                this._varConds = [];
            }

            this.x = Number(cfg.x || 0);
            this.y = Number(cfg.y || 0);
            this.visible = false;
            this._lastValue = null;
        }

        isEnabled() {
            if (this._visibleSwitchId > 0 && !$gameSwitches.value(this._visibleSwitchId)) return false;
            if (this._disableSwitches.some(id => $gameSwitches.value(id))) return false;

            if (this._varConds.length > 0) {
                return this._varConds.every(c => {
                    const actual = $gameVariables.value(c.variableId);
                    const expected = c.value;

                    if (c.operator === "==") {
                        const actualNum = Number(actual);
                        const expectedNum = Number(expected);
                        if (!isNaN(actualNum) && !isNaN(expectedNum)) {
                            return actualNum === expectedNum;
                        } else {
                            return String(actual) === String(expected);
                        }
                    } else {
                        const a = Number(actual);
                        const b = Number(expected);
                        if (isNaN(a) || isNaN(b)) return false;

                        return c.operator === ">=" ? a >= b :
                               c.operator === "<=" ? a <= b :
                               c.operator === "<"  ? a < b : false;
                    }
                });
            }

            return true;
        }

        logConditionStatus() {
            //console.group(`[NrEasyNumber] 表示スプライト: ${this._label}`);
            const sw = this._visibleSwitchId;

            if (this._disableSwitches.length > 0) {
                this._disableSwitches.forEach(id => {
                    //console.log(`無効スイッチ [${id}]: ${$gameSwitches.value(id) ? 'ON' : 'OFF'}`);
                });
            } else {
                //console.log("無効スイッチ: 指定なし");
            }

            if (this._varConds.length > 0) {
                this._varConds.forEach(c => {
                    const actual = $gameVariables.value(c.variableId);
                    const actualStr = String(actual);
                    const expectedStr = String(c.value);
                    const result = actualStr === expectedStr;
                    //console.log(`変数[${c.variableId}] "${actualStr}" ${c.operator} "${expectedStr}" → ${result ? '✔' : '✘'}`);
                });
            } else {
                //console.log("変数条件: 指定なし");
            }

            console.groupEnd();
        }

        update() {
            super.update();
            const visibleNow = this.isEnabled();

            if (!this.visible && visibleNow) {
                this.logConditionStatus();
            }

            this.visible = visibleNow;

            if (!visibleNow) return;

            const val = $gameVariables.value(this._variableId);
            if (val !== this._lastValue) {
                this.updateText();
            }
        }

        updateText(force = false) {
            const val = $gameVariables.value(this._variableId);
            const str = isNaN(Number(val))
                ? String(val)
                : String(Math.min(Number(val), Math.pow(10, this._maxDigits) - 1));

            this.bitmap.clear();
            this.bitmap.fontSize = this._fontSize;
            this.bitmap.textColor = "#ffffff";

            const width = this.bitmap.width;
            const tw = this.bitmap.measureTextWidth(str);
            const x = this._align === "left" ? 0 :
                      this._align === "center" ? (width - tw) / 2 :
                      (width - tw);

            this.bitmap.drawText(str, x, 0, width, this.bitmap.height, "left");
            this._lastValue = val;
        }
    }

    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function() {
        _Scene_Map_onMapLoaded.call(this);
        this._easySprites = configs.map(cfg => {
            const sp = new StatusDisplaySprite(cfg);
            this.addChild(sp);

            return sp;
        });
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (this._easySprites) {
            this._easySprites.forEach(sp => sp.update());
        }
    };

    window.NrEasyNumber = {
        setPosition(label, x, y) {
            const scene = SceneManager._scene;
            if (scene && scene._easySprites) {
                const target = scene._easySprites.find(s => s._label === label);
                if (target) {
                    target.x = x;
                    target.y = y;
                }
            }
        }
    };
})();
