//=============================================================================
// NrFakeLoading.js
//=============================================================================

/*:
 * @plugindesc タイトル前にフェイクロード画面を表示するプラグイン v1.0.0
 * @author NJ
 *
 * @param gaugeFrameImage
 * @text ゲージ枠画像
 * @desc ゲージの枠画像ファイル名。空なら枠なし。
 * @default 
 *
 * @param gaugeBarImage
 * @text ゲージ本体画像
 * @desc ゲージバー画像ファイル名。空なら白矩形。
 * @default 
 *
 * @param gaugeDirection
 * @text ゲージ方向
 * @desc "up/down/left/right"
 * @default right
 *
 * @param gaugeX
 * @text ゲージX座標
 * @default 100
 * @param gaugeY
 * @text ゲージY座標
 * @default 300
 *
 * @param text
 * @text 表示テキスト
 * @default 読み込み中...
 * @param textSize
 * @text テキストサイズ(px)
 * @default 20
 * @param textColor
 * @text テキストカラー
 * @default #ffffff
 * @param textX
 * @text テキストX座標
 * @default 100
 * @param textY
 * @text テキストY座標
 * @default 270
 *
 * @param waitTime
 * @text フェイク時間(秒) or "min-max"
 * @desc 数値単一 or "1-3"形式
 * @default 2
 *
 * @param useAnimation
 * @text アニメーション使用
 * @type boolean
 * @default true
 * @param animationId
 * @text アニメーションID
 * @type number
 * @default 3
 * @param animationX
 * @text アニメーションX座標
 * @default 80
 * @param animationY
 * @text アニメーションY座標
 * @default 320
 *
 * @param showPercent
 * @text パーセント表示
 * @type boolean
 * @default true
 * @param percentX
 * @text パーセントX座標
 * @default 200
 * @param percentY
 * @text パーセントY座標
 * @default 300
 *
 * @param preloadImages
 * @text プリロード画像
 * @type struct<PreloadSet>[]
 * @desc フェイクローディング中に読み込んでおく画像
 * @default []
 *
 * @help
 * タイトル画面に行く前に指定のフェイクローディング演出を挿入します。
 *
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~PreloadSet:
 * @param folder
 * @text フォルダ（img/以降）
 * @desc 例: pictures, faces, characters, system
 * @default pictures
 *
 * @param files
 * @text 画像ファイル名配列
 * @type string[]
 * @desc 拡張子不要。例: ["Actor1", "Enemy1"]
 * @default []
 */

Graphics._paintUpperCanvas = function() {};
Graphics._showLoading = function() {};
Graphics._hideLoading = function() {};
Graphics._loadingImage = new Image();

(function() {

    var params = PluginManager.parameters('NrFakeLoading');
    var cfg = {
        gaugeFrameImage: String(params['gaugeFrameImage'] || ''),
        gaugeBarImage: String(params['gaugeBarImage'] || ''),
        gaugeDirection: String(params['gaugeDirection'] || 'right'),
        gaugeX: Number(params['gaugeX'] || 100),
        gaugeY: Number(params['gaugeY'] || 300),
        text: String(params['text'] || '読み込み中...'),
        textSize: Number(params['textSize'] || 20),
        textColor: String(params['textColor'] || '#ffffff'),
        textX: Number(params['textX'] || 100),
        textY: Number(params['textY'] || 270),
        waitTime: String(params['waitTime'] || '2'),
        useAnimation: params['useAnimation'] === 'true',
        animationId: Number(params['animationId'] || 3),
        animationX: Number(params['animationX'] || 80),
        animationY: Number(params['animationY'] || 320),
        showPercent: params['showPercent'] === 'true',
        percentX: Number(params['percentX'] || 200),
        percentY: Number(params['percentY'] || 300)
    };

    function Scene_NrFakeLoading() {
        this.initialize.apply(this, arguments);
    }
    Scene_NrFakeLoading.prototype = Object.create(Scene_Base.prototype);
    Scene_NrFakeLoading.prototype.constructor = Scene_NrFakeLoading;

    Scene_NrFakeLoading.prototype.initialize = function() {
        Scene_Base.prototype.initialize.call(this);
        this._counter = 0;
        var wt = cfg.waitTime;
        if (wt.match(/^\d+$/)) {
            this._waitMax = Number(wt) * 60;
        } else {
            var sp = wt.split('-');
            var mn = Number(sp[0]), mx = Number(sp[1]);
            this._waitMax = (mn + Math.random() * (mx - mn)) * 60;
        }
        this._fakeProgress = 0;
        this._preloadDone = false;
    };

    Scene_NrFakeLoading.prototype.create = function() {
        Scene_Base.prototype.create.call(this);

        var gaugeGroup = new Sprite();
        gaugeGroup.x = cfg.gaugeX;
        gaugeGroup.y = cfg.gaugeY;
        this.addChild(gaugeGroup);

        var barBitmap = cfg.gaugeBarImage ? ImageManager.loadPicture(cfg.gaugeBarImage) : null;
        var barW = barBitmap ? barBitmap.width : 200;
        var barH = barBitmap ? barBitmap.height : 15;

        this._bar = barBitmap ? new Sprite(barBitmap) : new Sprite(new Bitmap(barW, barH));
        this._bar.x = 0;
        this._bar.y = 0;
        gaugeGroup.addChild(this._bar);

        if (cfg.gaugeFrameImage) {
            this._frame = new Sprite(ImageManager.loadPicture(cfg.gaugeFrameImage));
            this._frame.x = 0;
            this._frame.y = 0;
            gaugeGroup.addChild(this._frame);
        }

        this._text = new Sprite(new Bitmap(200, cfg.textSize + 4));
        this._text.bitmap.fontSize = cfg.textSize;
        this._text.bitmap.textColor = cfg.textColor;
        this._text.x = cfg.textX;
        this._text.y = cfg.textY;
        this.addChild(this._text);

        if (cfg.useAnimation) {
            this._animTarget = new Sprite_Base(); 
            this._animTarget.x = cfg.animationX;
            this._animTarget.y = cfg.animationY;
            this.addChild(this._animTarget);
        }

        this._percent = new Sprite(new Bitmap(100, cfg.textSize + 4));
        this._percent.x = cfg.percentX;
        this._percent.y = cfg.percentY;
        this.addChild(this._percent);
    };

    Scene_NrFakeLoading.prototype.update = function() {
        Scene_Base.prototype.update.call(this);
        this._counter++;

        if (!this._preloadDone) {
            this._preloadDone = true;

            var preloadList = JSON.parse(params['preloadImages'] || '[]').map(function(entry) {
                var set = JSON.parse(entry);
                return {
                    folder: set.folder,
                    files: JSON.parse(set.files)
                };
            });

            function preloadImage(folder, name) {
                var path = 'img/' + folder + '/';
                ImageManager.loadBitmap(path, name);
            }

            preloadList.forEach(function(set) {
                set.files.forEach(function(name) {
                    preloadImage(set.folder, name);
                });
            });
        }

        if (this._fakeProgress < 1) {
            const remain = Math.max(this._waitMax - this._counter, 1);
            const restProgress = 1.0 - this._fakeProgress;
            const minSpeed = restProgress / remain;
            const randomSpeed = minSpeed + Math.random() * minSpeed;
            this._fakeProgress += randomSpeed;
            this._fakeProgress = Math.min(this._fakeProgress, 1);
        }

        const rate = this._fakeProgress;

        if (!cfg.gaugeBarImage) {
            this._bar.bitmap.clear();
            const w = this._bar.bitmap.width;
            const h = this._bar.bitmap.height;
            if (cfg.gaugeDirection === 'right') {
                this._bar.bitmap.fillRect(0, 0, w * rate, h, '#ffffff');
            } else if (cfg.gaugeDirection === 'left') {
                this._bar.bitmap.fillRect(w * (1 - rate), 0, w * rate, h, '#ffffff');
            } else if (cfg.gaugeDirection === 'down') {
                this._bar.bitmap.fillRect(0, 0, w, h * rate, '#ffffff');
            } else if (cfg.gaugeDirection === 'up') {
                this._bar.bitmap.fillRect(0, h * (1 - rate), w, h * rate, '#ffffff');
            }
        } else {
            const w = this._bar.bitmap.width;
            const h = this._bar.bitmap.height;
            this._bar.setFrame(0, 0, w * rate, h);
        }

        this._text.bitmap.clear();
        this._text.bitmap.drawText(cfg.text, 0, 0, 200, cfg.textSize + 4, 'left');

        if (cfg.showPercent) {
            this._percent.bitmap.clear();
            this._percent.bitmap.drawText(Math.floor(rate * 100) + '%', 0, 0,
                100, cfg.textSize + 4, 'right');
        }

        if (cfg.useAnimation && this._animTarget) {
            if (!this._animTarget._animationSprites || this._animTarget._animationSprites.length === 0) {
                this._animTarget.startAnimation($dataAnimations[cfg.animationId], false, 0);
            }
        }

        if (this._counter >= this._waitMax && rate >= 1) {
            SceneManager.goto(Scene_Title);
        }
    };

    var _SceneManager_goto = SceneManager.goto;
    SceneManager.goto = function(sceneClass) {
        if (sceneClass === Scene_Title && !this._afterSplashHandled) {
            this._afterSplashHandled = true;
            sceneClass = Scene_NrFakeLoading;
        }
        _SceneManager_goto.call(this, sceneClass);
    };

})();
